# -*- coding: utf-8 -*-
#
# Author: Guillermo Gonzalez <guillermo.gonzalez@canonical.com>
#
# Copyright 2009 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Tests for the Volume Manager."""

from __future__ import with_statement

import collections
import logging
import os
import uuid

from ubuntuone.storageprotocol.client import ListShares
from ubuntuone.storageprotocol.sharersp import (
    NotifyShareHolder,
    ShareResponse,
)
from ubuntuone.storageprotocol import volumes, request
from contrib.testing.testcase import (
    FakeMain,
    BaseTwistedTestCase,
    environ,
    MementoHandler,
)
from ubuntuone.syncdaemon import config
from ubuntuone.syncdaemon.volume_manager import (
    Share,
    Shared,
    UDF,
    Root,
    _Share,
    _UDF,
    allow_writes,
    VolumeManager,
    LegacyShareFileShelf,
    MetadataUpgrader,
    VMFileShelf,
    VolumeDoesNotExist,
)
from twisted.internet import defer, reactor

# grab the metadata version before tests fiddle with it
CURRENT_METADATA_VERSION = VolumeManager.METADATA_VERSION


class BaseVolumeManagerTests(BaseTwistedTestCase):
    """ Bas TestCase for Volume Manager tests """

    def setUp(self):
        """ setup the test """
        BaseTwistedTestCase.setUp(self)
        self.log = logging.getLogger("ubuntuone.SyncDaemon.TEST")
        self.log.info("starting test %s.%s", self.__class__.__name__,
                      self._testMethodName)
        self.home_dir = self.mktemp('ubuntuonehacker')
        self.root_dir = self.mktemp(os.path.join('ubuntuonehacker', 'root_dir'))
        self.data_dir = self.mktemp('data_dir')
        self.shares_dir = self.mktemp('shares_dir')
        self.partials_dir = self.mktemp('partials_dir')
        self.main = FakeMain(self.root_dir, self.shares_dir,
                             self.data_dir, self.partials_dir)
        self.vm = self.main.vm
        self.handler = MementoHandler()
        self.handler.setLevel(logging.INFO)
        self.vm.log.addHandler(self.handler)

    def tearDown(self):
        """ cleanup main and remove the temp dir """
        self.vm.log.removeHandler(self.handler)
        self.main.shutdown()
        self.rmtree(self.home_dir)
        self.rmtree(self.data_dir)
        self.rmtree(self.shares_dir)
        self.log.info("finished test %s.%s", self.__class__.__name__,
                      self._testMethodName)
        VolumeManager.METADATA_VERSION = CURRENT_METADATA_VERSION
        return BaseTwistedTestCase.tearDown(self)

    def _listen_for(self, event, callback, count=1, collect=False):
        """Setup a EQ listener for the especified event."""
        event_q = self.main.event_q
        class Listener(object):
            """A basic listener to handle the pushed event."""

            def __init__(self):
                self.hits = 0
                self.events = []

            def _handle_event(self, *args, **kwargs):
                self.hits += 1
                if collect:
                    self.events.append((args, kwargs))
                if self.hits == count:
                    event_q.unsubscribe(self)
                    if collect:
                        callback(self.events)
                    elif kwargs:
                        callback((args, kwargs))
                    else:
                        callback(args)

        listener = Listener()
        setattr(listener, 'handle_'+event, listener._handle_event)
        event_q.subscribe(listener)
        return listener


class VolumeManagerTests(BaseVolumeManagerTests):
    """ Tests for Volume Manager internal API. """

    @defer.inlineCallbacks
    def test__got_root_ok_first(self):
        """Test _got_root method first time."""
        d = defer.Deferred()
        self._listen_for('SYS_ROOT_RECEIVED', d.callback)
        self.vm._got_root('root_uuid')

        res = yield d
        mdobj = self.main.fs.get_by_path(self.root_dir)
        self.assertEqual(res, ('root_uuid', mdobj.mdid))

    @defer.inlineCallbacks
    def test__got_root_ok_twice(self):
        """Test _got_root method twice."""
        d = defer.Deferred()
        # first time
        self.vm._got_root('root_uuid')

        # now listen and receive it again
        self._listen_for('SYS_ROOT_RECEIVED', d.callback)
        self.vm._got_root('root_uuid')

        res = yield d
        mdobj = self.main.fs.get_by_path(self.root_dir)
        self.assertEqual(res, ('root_uuid', mdobj.mdid))

    @defer.inlineCallbacks
    def test__got_root_mismatch(self):
        """Test for _got_root with different root node_id."""
        self.vm._got_root('root_uuid')
        d = defer.Deferred()
        self._listen_for('SYS_ROOT_MISMATCH', d.callback)
        self.vm._got_root('other_root_uuid')
        yield d

    def test_add_share(self):
        """ test the add_share method. """
        # initialize the the root
        self.vm._got_root('root_uuid')
        share_path = os.path.join(self.shares_dir, 'fake_share')
        share = Share(path=share_path, volume_id='share_id')
        self.vm.add_share(share)
        self.assertIn(share.volume_id, self.vm.shares)

    def test_share_deleted(self):
        """ Check that a share is deleted from the share mapping. """
        # initialize the the root
        self.vm._got_root('root_uuid')
        share_path = os.path.join(self.shares_dir, 'fake_share')
        share = Share(path=share_path, volume_id='share_id')
        self.vm.add_share(share)
        self.assertIn(share.volume_id, self.vm.shares)
        self.vm.share_deleted(share.volume_id)
        self.assertNotIn(share.volume_id, self.vm.shares)

    def test_share_deleted_with_content(self):
        """ Check that a share is deleted from the share mapping. """
        # initialize the the root
        self.vm._got_root('root_uuid')
        share_path = os.path.join(self.shares_dir, 'fake_share')
        share = Share(path=share_path, volume_id='share_id',
                      node_id='share_node_id', access_level='Modify',
                      accepted=True)
        self.vm.add_share(share)
        # create a few files and directories
        dirs = ['dir', 'dir/subdir', 'dir/empty_dir']
        for i, dir in enumerate(dirs):
            path = os.path.join(share.path, dir)
            with allow_writes(os.path.split(share.path)[0]):
                if not os.path.exists(path):
                    os.makedirs(path)
            self.main.fs.create(path, share.volume_id, is_dir=True)
            self.main.fs.set_node_id(path, 'dir_node_id'+str(i))
            self.main.event_q.inotify_add_watch(path)
        files = ['a_file', 'dir/file', 'dir/subdir/file']
        for i, file in enumerate(files):
            path = os.path.join(share.path, file)
            self.main.fs.create(path, share.volume_id)
            self.main.fs.set_node_id(path, 'file_node_id'+str(i))

        paths = list(self.main.fs.get_paths_starting_with(share.path))
        for path, is_dir in paths:
            self.assertTrue(self.main.fs.get_by_path(path))
            if is_dir:
                self.assertTrue(self.main.event_q.inotify_has_watch(path), path)
        self.assertIn(share.volume_id, self.vm.shares)
        self.vm.share_deleted(share.volume_id)
        self.assertNotIn(share.volume_id, self.vm.shares)
        for path, is_dir in paths:
            self.assertRaises(KeyError, self.main.fs.get_by_path, path)
            if is_dir:
                self.assertFalse(self.main.event_q.inotify_has_watch(path), path)

    def test_share_changed(self):
        """Check that VM.share_changed updates the access_level."""
        share_holder = NotifyShareHolder.from_params('share_id', None,
                                                     'fake_share',
                                                     'test_username',
                                                     'visible_name', 'Modify')
        # initialize the the root
        self.vm._got_root('root_uuid')
        share_path = os.path.join(self.shares_dir, share_holder.share_name)
        share = Share(path=share_path, volume_id=share_holder.share_id,
                      access_level='View')
        self.vm.add_share(share)
        self.vm.share_changed(share_holder)
        self.assertEqual('Modify',
                         self.vm.shares[share.volume_id].access_level)

    def test_share_changed_no_share(self):
        """Test share_changed for a share we don't have."""
        called = []
        self.vm.refresh_volumes = lambda: called.append(True)
        holder = collections.namedtuple('Holder', 'share_id')('not such id')
        self.vm.share_changed(holder)
        self.assertTrue(called)
        self.assertTrue(self.handler.check_warning("don't have the share"))

    def test_handle_AQ_SHARES_LIST(self):
        """Test the handling of the AQ_SHARE_LIST event."""
        share_id = uuid.uuid4()
        share_response = ShareResponse.from_params(share_id, 'to_me',
                                                   'fake_share_uuid',
                                                   'fake_share', 'username',
                                                   'visible_username', 'yes',
                                                   'View')
        # initialize the the root
        self.vm._got_root('root_uuid')
        response = ListShares(None)
        response.shares = [share_response]
        self.vm.handle_AQ_SHARES_LIST(response)
        self.assertEqual(2, len(self.vm.shares)) # the new shares and root
        # check that the share is in the shares dict
        self.assertIn(str(share_id), self.vm.shares)
        share = self.vm.shares[str(share_id)]
        self.assertEqual('fake_share', share.name)
        self.assertEqual('fake_share_uuid', share.node_id)

    def test_handle_SV_SHARE_CHANGED(self):
        """ test the handling of the AQ_SHARE_LIST event. """
        share_id = uuid.uuid4()
        share_holder = NotifyShareHolder.from_params(share_id, None,
                                                     'fake_share',
                                                     'test_username',
                                                     'visible_name', 'Modify')
        # initialize the the root
        self.vm._got_root('root_uuid')
        # create a share
        share_path = os.path.join(self.shares_dir, share_holder.share_name)
        share = Share(path=share_path, volume_id=str(share_holder.share_id),
                      access_level='View')
        self.vm.add_share(share)
        self.vm.handle_SV_SHARE_CHANGED(info=share_holder)
        self.assertEquals('Modify', self.vm.shares[str(share_id)].access_level)
        self.vm.handle_SV_SHARE_DELETED(share_holder.share_id)
        self.assertNotIn('share_id', self.vm.shares)

    def test_persistence(self):
        """ Test that the persistence of shares works as expected. """
        # create the folders layout
        share_path = os.path.join(self.shares_dir, 'my_share')
        share = Share(path=share_path, volume_id='a_share_id', access_level='View')
        self.vm.add_share(share)
        other_vm = VolumeManager(self.main)
        for key in self.vm.shares:
            self.assertEquals(self.vm.shares[key].__dict__,
                              other_vm.shares[key].__dict__)

    def test_handle_AQ_SHARES_LIST_shared(self):
        """test the handling of the AQ_SHARE_LIST event, with a shared dir."""
        share_id = uuid.uuid4()
        share_response = ShareResponse.from_params(share_id, 'to_me',
                                                   'fake_share_uuid',
                                                   'fake_share', 'username',
                                                   'visible_username', 'yes',
                                                   'View')
        shared_id = uuid.uuid4()
        shared_response = ShareResponse.from_params(shared_id, 'from_me',
                                                   'shared_uuid',
                                                   'fake_shared', 'myname',
                                                   'my_visible_name', 'yes',
                                                   'Modify')
        # initialize the the root
        self.vm._got_root('root_uuid')
        shared_dir = os.path.join(self.root_dir, 'shared_dir')
        self.main.fs.create(path=shared_dir, share_id="", is_dir=True)
        self.main.fs.set_node_id(shared_dir, shared_response.subtree)
        response = ListShares(None)
        response.shares = [share_response, shared_response]
        self.vm.handle_AQ_SHARES_LIST(response)
        self.assertEquals(2, len(self.vm.shares)) # the new shares and root
        self.assertEquals(1, len(self.vm.shared)) # the new shares and root
        shared = self.vm.shared[str(shared_id)]
        self.assertEquals('fake_shared', shared.name)
        # check that the uuid is stored in fs
        mdobj = self.main.fs.get_by_path(shared.path)
        self.assertEquals(shared.node_id, mdobj.node_id)

    def test_add_shared(self):
        """ Test VolumeManager.add_shared """
        path = os.path.join(self.vm.root.path, 'shared_path')
        self.main.fs.create(path, "")
        self.main.fs.set_node_id(path, 'node_id')
        self.main.fs.get_by_node_id("", 'node_id')

        # helper method, pylint: disable-msg=C0111
        def fake_create_share(node_id, user, name, access_level, marker):
            self.assertIn(marker, self.vm.marker_share_map)
            self.vm.handle_AQ_CREATE_SHARE_OK(share_id='share_id',
                                              marker=marker)
        self.main.action_q.create_share = fake_create_share
        self.vm.create_share(path, 'fake_user', 'shared_name', 'View')

        self.assertTrue(self.vm.shared.get('share_id') is not None)
        share = self.vm.shared.get('share_id')
        self.assertEqual('fake_user', share.other_username)
        self.assertEqual('shared_name', share.name)
        self.assertEqual('View', share.access_level)
        self.assertEqual('node_id', share.node_id)
        self.assertEqual('share_id', share.volume_id)

    def test_create_share(self):
        """ Test VolumeManager.create_share """
        path = os.path.join(self.vm.root.path, 'shared_path')
        self.main.fs.create(path, "")
        self.main.fs.set_node_id(path, 'node_id')
        self.main.fs.get_by_node_id("", 'node_id')

        # helper method, pylint: disable-msg=C0111
        def fake_create_share(node_id, user, name, access_level, marker):
            self.assertEquals('node_id', node_id)
            self.assertEquals('fake_user', user)
            self.assertEquals('shared_name', name)
            self.assertEquals('View', access_level)
            self.assertTrue(marker is not None)
            share = self.vm.marker_share_map[marker]
            self.assertEquals(path, share.path)
            self.assertEquals('View', share.access_level)
            self.assertEquals(marker, share.volume_id)
            self.assertEquals('fake_user', share.other_username)
            self.assertEquals('node_id', share.node_id)

        self.main.action_q.create_share = fake_create_share
        self.vm.create_share(path, 'fake_user', 'shared_name', 'View')

    def test_create_share_error(self):
        """ Test VolumeManager.create_share """
        path = os.path.join(self.vm.root.path, 'shared_path')
        self.main.fs.create(path, "")
        self.main.fs.set_node_id(path, 'node_id')
        self.main.fs.get_by_node_id("", 'node_id')

        # helper method, pylint: disable-msg=C0111
        def fake_create_share(node_id, user, name, access_level, marker):
            self.vm.handle_AQ_CREATE_SHARE_ERROR(marker, 'a fake error')

        self.main.action_q.create_share = fake_create_share
        self.vm.create_share(path, 'fake_user', 'shared_name', 'View')

    @defer.inlineCallbacks
    def test_create_share_missing_node_id(self):
        """Test VolumeManager.create_share in the case of missing node_id."""
        path = os.path.join(self.vm.root.path, 'shared_path')
        self.main.fs.create(path, "")
        expected_node_id = uuid.uuid4()
        expected_share_id = uuid.uuid4()

        # helper method, pylint: disable-msg=C0111
        def fake_create_share(node_id, user, name, access_level, marker):
            # some sanity checks
            share = self.vm.marker_share_map[marker]
            self.assertEquals(path, share.path)
            self.assertEquals('View', share.access_level)
            self.assertEquals(marker, share.volume_id)
            self.assertEquals('fake_user', share.other_username)
            self.assertEquals(marker, share.node_id)
            # fake a node_id demark and set the node_id
            self.main.fs.set_node_id(path, str(expected_node_id))
            self.main.event_q.push("AQ_CREATE_SHARE_OK",
                                   share_id=expected_share_id, marker=marker)

        d = defer.Deferred()
        self._listen_for('AQ_CREATE_SHARE_OK', d.callback)
        self.main.action_q.create_share = fake_create_share
        self.vm.create_share(path, 'fake_user', 'shared_name', 'View')
        yield d
        share = self.vm.shared[str(expected_share_id)]
        self.assertEquals(str(expected_node_id), share.node_id)

    @defer.inlineCallbacks
    def test_delete_shared(self):
        """Test VolumeManager.delete_share."""
        path = os.path.join(self.vm.root.path, 'shared_path')
        self.main.fs.create(path, "")
        self.main.fs.set_node_id(path, 'node_id')
        share = Shared(path=path, volume_id='share_id', node_id="node_id")
        self.vm.add_shared(share)

        def fake_delete_share(share_id):
            """Fake delete_share."""
            self.assertEqual(share_id, share.volume_id)
            self.main.event_q.push('AQ_DELETE_SHARE_OK', share_id)

        self.patch(self.main.action_q, 'delete_share', fake_delete_share)
        d = defer.Deferred()
        self._listen_for('VM_SHARE_DELETED', d.callback, 1, collect=True)
        self.vm.delete_share(share.volume_id)
        events = yield d
        event = events[0]
        self.assertEqual(event[0], (share,))

    @defer.inlineCallbacks
    def test_delete_shared_error(self):
        """Test VolumeManager.delete_share."""
        path = os.path.join(self.vm.root.path, 'shared_path')
        self.main.fs.create(path, "")
        self.main.fs.set_node_id(path, 'node_id')
        share = Shared(path=path, volume_id='share_id', node_id="node_id")
        self.vm.add_shared(share)

        def fake_delete_share(share_id):
            """Fake delete_share that always fails."""
            self.main.event_q.push('AQ_DELETE_SHARE_ERROR',
                                   share_id, 'a fake error')

        self.patch(self.main.action_q, 'delete_share', fake_delete_share)
        d = defer.Deferred()
        self._listen_for('VM_SHARE_DELETE_ERROR', d.callback, 1, collect=True)
        self.vm.delete_share(share.volume_id)
        events = yield d
        event = events[0]
        self.assertEqual(event[0], (share.volume_id, 'a fake error'))

    @defer.inlineCallbacks
    def test_delete_shared_error_missing_share(self):
        """Test VolumeManager.delete_share with a non-existent share."""
        path = os.path.join(self.vm.root.path, 'shared_path')
        self.main.fs.create(path, "")
        self.main.fs.set_node_id(path, 'node_id')
        d = defer.Deferred()
        self._listen_for('VM_SHARE_DELETE_ERROR', d.callback, 1, collect=True)
        self.vm.delete_share('fake_share_id')
        events = yield d
        event = events[0]
        self.assertEqual(event[0], ('fake_share_id', 'DOES_NOT_EXIST'))

    def test_accept_share(self):
        """ Test the accept_share method. """
        d = defer.Deferred()
        self.vm._got_root('root_uuid')
        share_path = os.path.join(self.shares_dir, 'fake_share')
        share = Share(path=share_path, volume_id='share_id', node_id="node_id")
        self.vm.add_share(share)
        self.assertIn(share.volume_id, self.vm.shares)
        self.assertEquals(False, share.accepted)
        # helper method, pylint: disable-msg=C0111
        def answer_share(share_id, answer):
            reactor.callLater(0.2, d.callback, (share_id, answer))
            return d
        self.main.action_q.answer_share = answer_share
        def callback(result):
            share_id, answer = result
            self.assertEquals(share.volume_id, share_id)
            self.assertEquals('Yes', answer)
        d.addCallback(callback)
        self.vm.accept_share(share.volume_id, True)
        return d

    def test_handle_AQ_SHARES_LIST_shared_missing_md(self):
        """test the handling of the AQ_SHARE_LIST event, when the md
        isn't there yet.
        """
        shared_response = ShareResponse.from_params('shared_id', 'from_me',
                                                   'shared_uuid',
                                                   'fake_shared', 'myname',
                                                   'my_visible_name', 'yes',
                                                   'Modify')
        # initialize the the root
        self.vm._got_root('root_uuid')
        response = ListShares(None)
        response.shares = [shared_response]
        self.vm.handle_AQ_SHARES_LIST(response)
        self.assertEquals(1, len(self.vm.shared)) # the new shares and root
        shared = self.vm.shared['shared_id']
        self.assertEquals('fake_shared', shared.name)
        # check that the uuid is stored in fs
        self.assertEquals(shared_response.subtree, shared.node_id)
        self.assertEquals(None, shared.path)

    def test_handle_SV_SHARE_ANSWERED(self):
        """ test the handling of the AQ_SHARE_ANSWERED. """
        path = os.path.join(self.vm.root.path, 'shared_path')
        self.main.fs.create(path, "")
        self.main.fs.set_node_id(path, 'node_id')
        self.main.fs.get_by_node_id("", 'node_id')
        # initialize the the root
        self.vm._got_root('root_uuid')
        # add the shared folder
        share = Share(path=path, volume_id='share_id', access_level='View')
        self.vm.add_shared(share)
        self.assertEquals(False, self.vm.shared['share_id'].accepted)
        # check that a answer notify of a missing share don't blowup
        self.vm.handle_SV_SHARE_ANSWERED('share_id', 'Yes')
        self.assertEquals(True, self.vm.shared['share_id'].accepted)

    def test_handle_SV_SHARE_ANSWERED_missing(self):
        """ test the handling of the AQ_SHARE_ANSWERED. """
        path = os.path.join(self.vm.root.path, 'shared_path')
        self.main.fs.create(path, "")
        self.main.fs.set_node_id(path, 'node_id')
        self.main.fs.get_by_node_id("", 'node_id')
        # initialize the the root
        self.vm._got_root('root_uuid')
        self.assertNotIn('share_id', self.vm.shared)
        # check that a answer notify of a missing share don't blowup
        self.vm.handle_SV_SHARE_ANSWERED('share_id', 'Yes')
        self.assertNotIn('share_id', self.vm.shared)

    def test_delete_share(self):
        """Test the deletion of a share and if it's removed from the MD."""
        share_response = ShareResponse.from_params('share_id', 'to_me',
                                                   'fake_share_uuid',
                                                   'fake_share', 'username',
                                                   'visible_username', False,
                                                   'View')
        share_response_1 = ShareResponse.from_params('share_id_1', 'to_me',
                                                   'fake_share_uuid_1',
                                                   'fake_share_1', 'username',
                                                   'visible_username', False,
                                                   'View')
        # initialize the the root
        self.vm._got_root('root_uuid')
        response = ListShares(None)
        response.shares = [share_response, share_response_1]
        self.vm.handle_AQ_SHARES_LIST(response)
        self.assertEquals(3, len(self.vm.shares)) # the new shares and root
        # check that the share is in the shares dict
        self.assertIn('share_id', self.vm.shares)
        self.assertIn('share_id_1', self.vm.shares)
        share = self.vm.shares['share_id']
        self.assertEquals('fake_share', share.name)
        self.assertEquals('fake_share_uuid', share.node_id)
        share = self.vm.shares['share_id_1']
        self.assertEquals('fake_share_1', share.name)
        self.assertEquals('fake_share_uuid_1', share.node_id)
        # inject a new ListShares response
        new_response = ListShares(None)
        new_response.shares = [share_response]
        self.vm.handle_AQ_SHARES_LIST(new_response)
        # check that the missing share was removed
        self.assertIn('share_id', self.vm.shares)
        self.assertFalse('share_id_1' in self.vm.shares)
        share = self.vm.shares['share_id']
        self.assertEquals('fake_share', share.name)
        self.assertEquals('fake_share_uuid', share.node_id)
        self.assertEquals(None, self.vm.shares.get('share_id_1'))

    def test_remove_watch(self):
        """Test for VolumeManager._remove_watch"""
        path = os.path.join(self.root_dir, 'dir')
        os.makedirs(path)
        self.main.event_q.inotify_add_watch(path)
        self.assertTrue(self.main.event_q.inotify_has_watch(path))
        self.vm._remove_watch(path)
        self.assertFalse(self.main.event_q.inotify_has_watch(path))

    def test_remove_watches(self):
        """Test for VolumeManager._remove_watches"""
        dirs = ['dir', 'dir/subdir', 'emptydir']
        paths = [os.path.join(self.root_dir, dir) for dir in dirs]
        # create metadata and add watches
        for i, path in enumerate(paths):
            if not os.path.exists(path):
                os.makedirs(path)
            self.main.fs.create(path, "", is_dir=True)
            self.main.fs.set_node_id(path, 'dir_node_id'+str(i))
            self.main.event_q.inotify_add_watch(path)
        # insert the root_dir in the list
        self.main.event_q.inotify_add_watch(self.root_dir)
        paths.insert(0, self.root_dir)
        for path in paths:
            self.assertTrue(self.main.event_q.inotify_has_watch(path))
        # remove the watches
        self.vm._remove_watches(self.root_dir)
        for path in paths:
            self.assertFalse(self.main.event_q.inotify_has_watch(path), path)

    def test_remove_watches_after_dir_rename(self):
        """Test for VolumeManager._remove_watches after dir rename."""
        path = os.path.join(self.root_dir, 'testit')
        os.mkdir(path)
        self.main.fs.create(path, "", is_dir=True)
        self.main.fs.set_node_id(path, 'dir_node_id')
        self.main.event_q.inotify_add_watch(path)

        os.rename(path, path+'.old')
        # remove the watches
        self.vm._remove_watches(self.root_dir)

        self.assertFalse(self.main.event_q.inotify_has_watch(path),
                         'watch should noy be present')

    def test_delete_fsm_object(self):
        """Test for VolumeManager._delete_fsm_object"""
        path = os.path.join(self.root_dir, 'dir')
        os.makedirs(path)
        self.main.fs.create(path, "", is_dir=True)
        self.main.fs.set_node_id(path, 'dir_node_id')
        self.main.event_q.inotify_add_watch(path)
        self.assertTrue(self.main.event_q.inotify_has_watch(path), path)
        self.assertTrue(self.main.fs.get_by_path(path), path)
        # remove the watch
        self.vm._delete_fsm_object(path)
        self.assertRaises(KeyError, self.main.fs.get_by_path, path)
        self.assertTrue(self.main.event_q.inotify_has_watch(path), path)

    def test_create_fsm_object(self):
        """Test for VolumeManager._create_fsm_object"""
        path = os.path.join(self.root_dir, 'node')
        self.assertRaises(KeyError, self.main.fs.get_by_path, path)
        self.vm._create_fsm_object(path, "", "node_id")
        self.assertTrue(self.main.fs.get_by_path(path), path)

    @defer.inlineCallbacks
    def test_root_mismatch(self):
        """Test that SYS_ROOT_MISMATCH is pushed."""
        self.vm._got_root('root_node_id')
        d = defer.Deferred()
        self._listen_for('SYS_ROOT_MISMATCH', d.callback)
        self.vm._got_root('root_id')
        current_id, new_id = yield d
        self.assertEquals('root_node_id', current_id)
        self.assertEquals('root_id', new_id)

    def test_handle_SYS_QUOTA_EXCEEDED_is_called(self):
        """Test that we handle the event."""
        # set up to record the call
        called = []
        self.vm.handle_SYS_QUOTA_EXCEEDED = lambda **k: called.append(k)

        # send the event
        data = dict(volume_id=request.ROOT, free_bytes=123987)
        self.main.event_q.push('SYS_QUOTA_EXCEEDED', **data)

        # check that we handled it
        self.assertEqual(called, [data])

    def test_handle_SYS_QUOTA_EXCEEDED_root(self):
        """Test that it updates the free space when error is on root."""
        self.vm.handle_SYS_QUOTA_EXCEEDED(request.ROOT, 11221)
        self.assertEqual(self.vm.get_volume(request.ROOT).free_bytes, 11221)

    def test_handle_SYS_QUOTA_EXCEEDED_udf(self):
        """Test that it updates the free space when error is on an UDF."""
        # create the udf
        with environ('HOME', self.home_dir):
            path = self.vm._build_udf_path(u"~/UDF")
        volume_id = str(uuid.uuid4())
        volume = volumes.UDFVolume(volume_id, 'udf_node_id', None, 0, u"~/UDF")
        udf = UDF.from_udf_volume(volume, path)
        self.vm.add_udf(udf)

        # call the handler
        self.vm.handle_SYS_QUOTA_EXCEEDED(volume_id, 1122)

        # but check the free bytes from root, that is who stores this info
        self.assertEqual(self.vm.get_volume(request.ROOT).free_bytes, 1122)

    def test_handle_SYS_QUOTA_EXCEEDED_share(self):
        """Test that it updates the free space when error is on a share."""
        # build the share
        share_id = str(uuid.uuid4())
        share_volume = volumes.ShareVolume(share_id, 'fake_share_uuid', None,
                                           10, 'to_me', 'fake_share', 'usern',
                                           'visible_username', True, 'Modify')
        dir_name = self.vm._build_share_path(share_volume.share_name,
                                             share_volume.other_visible_name)
        share_path = os.path.join(self.main.shares_dir, dir_name)
        share = Share.from_share_volume(share_volume, share_path)
        self.vm.add_share(share)

        # call and check
        self.vm.handle_SYS_QUOTA_EXCEEDED(share_id, 1122)
        self.assertEqual(self.vm.get_volume(share_id).free_bytes, 1122)

    @defer.inlineCallbacks
    def test_handle_AQ_ANSWER_SHARE_OK(self):
        """Test for handle_AQ_ANSWER_SHARE_OK."""
        share_id = uuid.uuid4()
        share_volume = volumes.ShareVolume(share_id, 'fake_share_uuid', None,
                                           10, 'to_me', 'fake_share', 'usern',
                                           'visible_username', False, 'View')
        dir_name = self.vm._build_share_path(share_volume.share_name,
                                          share_volume.other_visible_name)
        share_path = os.path.join(self.main.shares_dir, dir_name)
        share = Share.from_share_volume(share_volume, share_path)

        scratch_d = defer.Deferred()
        def fake_rescan_from_scratch(volume_id):
            """A fake get_delta that check the arguments."""
            self.assertEquals(share.volume_id, volume_id)
            scratch_d.callback(None)
        self.main.action_q.rescan_from_scratch = fake_rescan_from_scratch

        self.vm.add_share(share)
        self.vm.handle_AQ_ANSWER_SHARE_OK(share.volume_id, 'Yes')
        yield scratch_d
        share = self.vm.get_volume(share.volume_id)
        self.assertTrue(share.accepted, 'accepted != True')
        self.assertTrue(self.main.fs.get_by_path(share.path),
                        'No metadata for share root node.')
        self.assertTrue(os.path.exists(share.path),
                        'share path missing on disk!')

    @defer.inlineCallbacks
    def test_handle_AQ_DELETE_SHARE_OK(self):
        """Test for handle_AQ_DELETE_SHARE_OK."""
        path = os.path.join(self.vm.root.path, 'shared_path')
        self.main.fs.create(path, "")
        self.main.fs.set_node_id(path, 'node_id')
        share = Shared(path=path, volume_id='share_id', node_id="node_id")
        self.vm.add_shared(share)
        d = defer.Deferred()
        self._listen_for('VM_SHARE_DELETED', d.callback, 1, collect=True)
        self.main.event_q.push('AQ_DELETE_SHARE_OK', 'share_id')
        events = yield d
        event = events[0]
        self.assertEqual(event[0], (share,))

    @defer.inlineCallbacks
    def test_handle_AQ_DELETE_SHARE_ERROR(self):
        """Test for handle_AQ_DELETE_SHARE_ERROR."""
        path = os.path.join(self.vm.root.path, 'shared_path')
        self.main.fs.create(path, "")
        self.main.fs.set_node_id(path, 'node_id')
        share = Shared(path=path, volume_id='share_id', node_id="node_id")
        self.vm.add_shared(share)
        d = defer.Deferred()
        self._listen_for('VM_SHARE_DELETE_ERROR', d.callback, 1, collect=True)
        self.main.event_q.push('AQ_DELETE_SHARE_ERROR',
                               'share_id', 'a fake error')
        events = yield d
        event = events[0]
        self.assertEqual(event[0], (share.volume_id, 'a fake error'))


class VolumeManagerUnicodeTests(BaseVolumeManagerTests):
    """Tests for Volume Manager unicode capabilities."""

    def test_handle_SHARES_sharename(self):
        """test the handling of AQ_SHARE_LIST with non-ascii share name."""
        share_response = ShareResponse.from_params('share_id', 'to_me',
                                                   'fake_share_uuid',
                                                   u'montón', 'username',
                                                   'visible', 'yes',
                                                   'View')
        # initialize the the root
        self.vm._got_root('root_uuid')
        response = ListShares(None)
        response.shares = [share_response]
        self.vm.handle_AQ_SHARES_LIST(response)

        # check
        share = self.vm.shares['share_id']
        shouldbe_dir = os.path.join(self.shares_dir,
                                    u"montón".encode("utf8") + " from visible")
        self.assertEquals(shouldbe_dir, share.path)

    def test_handle_SHARES_visible_username(self):
        """test the handling of AQ_SHARE_LIST with non-ascii visible uname."""
        share_response = ShareResponse.from_params('share_id', 'to_me',
                                                   'fake_share_uuid',
                                                   'sharename', 'username',
                                                   u'Darío Toño', 'yes',
                                                   'View')
        # initialize the the root
        self.vm._got_root('root_uuid')
        response = ListShares(None)
        response.shares = [share_response]
        self.vm.handle_AQ_SHARES_LIST(response)

        # check
        share = self.vm.shares['share_id']
        shouldbe_dir = os.path.join(self.shares_dir,
                            "sharename from " + u"Darío Toño".encode("utf8"))
        self.assertEquals(shouldbe_dir, share.path)

    def test_handle_SV_SHARE_CHANGED_sharename(self):
        """test the handling of SV_SHARE_CHANGED for non-ascii share name."""
        share_holder = NotifyShareHolder.from_params('share_id', None,
                                                     u'año',
                                                     'test_username',
                                                     'visible', 'Modify')
        self.vm._got_root('root_uuid')
        self.vm.handle_SV_SHARE_CHANGED(info=share_holder)
        shouldbe_dir = os.path.join(self.shares_dir,
                                    u"año".encode("utf8") + " from visible")
        self.assertEquals(shouldbe_dir, self.vm.shares['share_id'].path)

    def test_handle_SV_SHARE_CHANGED_visible(self):
        """test the handling of SV_SHARE_CHANGED for non-ascii visible name."""
        share_holder = NotifyShareHolder.from_params('share_id', None,
                                                     'share',
                                                     'test_username',
                                                     u'Ramón', 'Modify')
        self.vm._got_root('root_uuid')
        self.vm.handle_SV_SHARE_CHANGED(info=share_holder)
        shouldbe_dir = os.path.join(self.shares_dir,
                                    "share from " + u"Ramón".encode("utf8"))
        self.assertEquals(shouldbe_dir, self.vm.shares['share_id'].path)


class VolumeManagerVolumesTests(BaseVolumeManagerTests):
    """Test UDF/Volumes bits of the VolumeManager."""

    def setUp(self):
        """Setup the test."""
        BaseVolumeManagerTests.setUp(self)
        self.old_home = os.environ['HOME']
        os.environ['HOME'] = self.home_dir

    def tearDown(self):
        """Tear down."""
        os.environ['HOME'] = self.old_home
        return BaseVolumeManagerTests.tearDown(self)

    def _create_udf(self, id, node_id, suggested_path, subscribed=True,
                    generation=None, free_bytes=100):
        """Create an UDF and returns it and the volume"""
        with environ('HOME', self.home_dir):
            path = self.vm._build_udf_path(suggested_path)
        # make sure suggested_path is unicode
        if isinstance(suggested_path, str):
            suggested_path = suggested_path.decode('utf-8')
        volume = volumes.UDFVolume(id, node_id, generation,
                                   free_bytes, suggested_path)
        udf = UDF.from_udf_volume(volume, path)
        udf.subscribed = subscribed
        udf.generation = generation
        return udf, volume

    def test_build_udf_path(self):
        """Test for VolumeManager._build_udf_path."""
        suggested_path = u"suggested_path"
        with environ('HOME', self.home_dir):
            udf_path = self.vm._build_udf_path(u"~/" + suggested_path)
        self.assertEquals(os.path.join(self.home_dir,
                                       suggested_path.encode('utf-8')),
                          udf_path)

    def test_udf_ancestors(self):
        """UDF's ancestors are correctly returned."""
        suggested_path = u'~/Documents/Reading/Books/PDFs'
        expected = [u'~', u'~/Documents', u'~/Documents/Reading',
                    u'~/Documents/Reading/Books']
        with environ('HOME', self.home_dir):
            udf, volume = self._create_udf('uid', 'nid', suggested_path)
            self.assertEquals(map(os.path.expanduser, expected), udf.ancestors)

    @defer.inlineCallbacks
    def test_add_udf(self):
        """Test for VolumeManager.add_udf."""
        suggested_path = "suggested_path"
        udf, volume = self._create_udf(uuid.uuid4(), 'udf_node_id',
                                       '~/' + suggested_path, subscribed=False)
        yield self.vm.add_udf(udf)
        self.assertEquals(os.path.join(self.home_dir, suggested_path),
                          udf.path)
        self.assertEquals(1, len(self.vm.udfs))
        # check that the UDF is in the fsm metadata
        mdobj = self.main.fs.get_by_path(udf.path)
        self.assertEquals(mdobj.node_id, 'udf_node_id')
        self.assertEquals(mdobj.share_id, udf.volume_id)
        # check that there isn't a watch in the UDF (we aren't
        # subscribed to it)
        self.assertFalse(self.main.event_q.inotify_has_watch(udf.path))
        # remove the udf
        self.vm.udf_deleted(udf.volume_id)
        # add it again, but this time with subscribed = True
        udf.subscribed = True
        yield self.vm.add_udf(udf)
        self.assertEquals(os.path.join(self.home_dir, suggested_path),
                          udf.path)
        self.assertEquals(1, len(self.vm.udfs))
        # check that the UDF is in the fsm metadata
        mdobj = self.main.fs.get_by_path(udf.path)
        self.assertEquals(mdobj.node_id, 'udf_node_id')
        self.assertEquals(mdobj.share_id, udf.volume_id)
        # check that there is a watch in the UDF
        self.assertTrue(self.main.event_q.inotify_has_watch(udf.path))

    @defer.inlineCallbacks
    def test_add_udf_calls_AQ(self):
        """Test that VolumeManager.add_udf calls AQ.rescan_from_scratch."""
        suggested_path = "suggested_path"
        udf, volume = self._create_udf(uuid.uuid4(), 'udf_node_id',
                                       '~/' + suggested_path, subscribed=True)
        scratch_d = defer.Deferred()
        def fake_rescan_from_scratch(volume_id):
            """A fake rescan_from_scratch that check the arguments."""
            self.assertEquals(udf.volume_id, volume_id)
            scratch_d.callback(None)
        self.main.action_q.rescan_from_scratch = fake_rescan_from_scratch

        yield self.vm.add_udf(udf)
        yield scratch_d
        self.assertEquals(os.path.join(self.home_dir, suggested_path),
                          udf.path)
        self.assertEquals(1, len(self.vm.udfs))
        # check that the UDF is in the fsm metadata
        mdobj = self.main.fs.get_by_path(udf.path)
        self.assertEquals(mdobj.node_id, 'udf_node_id')
        self.assertEquals(mdobj.share_id, udf.volume_id)
        self.assertTrue(self.main.event_q.inotify_has_watch(udf.path))

    @defer.inlineCallbacks
    def test_udf_deleted(self):
        """Test for VolumeManager.udf_deleted."""
        suggested_path = "suggested_path"
        udf, volume = self._create_udf(uuid.uuid4(), 'udf_node_id',
                                       '~/' + suggested_path)
        yield self.vm.add_udf(udf)
        self.assertEquals(1, len(self.vm.udfs))
        self.vm.udf_deleted(udf.volume_id)
        self.assertEquals(0, len(self.vm.udfs))
        # check that the UDF isn't in the fsm metadata
        self.assertRaises(KeyError, self.main.fs.get_by_path, udf.path)
        # check that there isn't a watch in the UDF
        self.assertFalse(self.main.event_q.inotify_has_watch(udf.path))

    @defer.inlineCallbacks
    def test_udf_deleted_with_content(self):
        """
        Test for VolumeManager.udf_deleted when the UDF
        contains files and directories.

        """
        suggested_path = "suggested_path"
        udf, volume = self._create_udf(uuid.uuid4(), 'udf_node_id',
                                       '~/' + suggested_path, subscribed=True)
        yield self.vm.add_udf(udf)
        self.assertEquals(1, len(self.vm.udfs))
        # create a few files and directories
        dirs = ['dir', 'dir/subdir', 'dir/empty_dir']
        for i, dir in enumerate(dirs):
            path = os.path.join(udf.path, dir)
            if not os.path.exists(path):
                os.makedirs(path)
            self.main.fs.create(path, udf.volume_id, is_dir=True)
            self.main.fs.set_node_id(path, 'dir_node_id'+str(i))
            # add a inotify watch to the dir
            self.main.event_q.inotify_add_watch(path)
        files = ['a_file', 'dir/file', 'dir/subdir/file']
        for i, file in enumerate(files):
            path = os.path.join(udf.path, file)
            self.main.fs.create(path, udf.volume_id)
            self.main.fs.set_node_id(path, 'file_node_id'+str(i))
        paths = self.main.fs.get_paths_starting_with(udf.path)
        self.assertEquals(len(paths), len(dirs+files)+1)
        self.vm.udf_deleted(udf.volume_id)
        self.assertEquals(0, len(self.vm.udfs))
        # check that the UDF isn't in the fsm metadata
        self.assertRaises(KeyError, self.main.fs.get_by_path, udf.path)
        # check that there isn't a watch in the UDF
        self.assertFalse(self.main.event_q.inotify_has_watch(udf.path))
        # check that there isn't any udf childs around
        for path, _ in paths:
            self.assertRaises(KeyError, self.main.fs.get_by_path, path)
        # get the childs (should be an empty list)
        paths = self.main.fs.get_paths_starting_with(udf.path)
        self.assertEquals(0, len(paths))

    @defer.inlineCallbacks
    def test_get_volume(self):
        """Test for VolumeManager.get_volume."""
        # create a Share
        share_id = uuid.uuid4()
        share_volume = volumes.ShareVolume(share_id, 'fake_share_uuid', None,
                                           10, 'to_me', 'fake_share',
                                           'username', 'visible_username',
                                           True, 'View')
        dir_name = self.vm._build_share_path(share_volume.share_name,
                                          share_volume.other_visible_name)
        share_path = os.path.join(self.main.shares_dir, dir_name)
        share = Share.from_share_volume(share_volume, share_path)
        # create a UDF
        udf_id = uuid.uuid4()
        udf, volume = self._create_udf(udf_id, 'udf_node_id', '~/UDF')
        yield self.vm.add_udf(udf)
        self.vm.add_share(share)
        self.assertEqual(1, len(self.vm.udfs))
        self.assertEqual(2, len(self.vm.shares))
        self.assertEqual(udf.volume_id,
                         self.vm.get_volume(str(udf_id)).id)
        self.assertEqual(udf.path, self.vm.get_volume(str(udf_id)).path)
        self.assertEqual(share.volume_id, self.vm.get_volume(str(share_id)).id)

    @defer.inlineCallbacks
    def test_handle_AQ_LIST_VOLUMES(self):
        """Test the handling of the AQ_LIST_VOLUMES event."""
        share_id = uuid.uuid4()
        share_volume = volumes.ShareVolume(share_id, 'fake_share_uuid', None,
                                           10, 'to_me', 'fake_share',
                                           'username', 'visible_username',
                                           True, 'View')
        udf_id = uuid.uuid4()
        udf_volume = volumes.UDFVolume(udf_id, 'udf_uuid', None, 100, u'~/UDF')
        root_volume = volumes.RootVolume(uuid.uuid4(), 17, 10)
        response = [share_volume, udf_volume, root_volume]
        d1 = defer.Deferred()
        d2 = defer.Deferred()
        self.vm.refresh_volumes = lambda: d1.errback('refresh_volumes called!')
        self._listen_for('VM_UDF_CREATED', d1.callback)
        self._listen_for('SV_VOLUME_NEW_GENERATION', d2.callback)
        # use a custom home
        with environ('HOME', self.home_dir):
            self.vm.handle_AQ_LIST_VOLUMES(response)
        yield d1
        yield d2
        self.assertEqual(2, len(self.vm.shares)) # the new share and root
        self.assertEqual(1, len(self.vm.udfs)) # the new udf
        # check that the share is in the shares dict
        self.assertIn(str(share_id), self.vm.shares)
        self.assertIn(str(udf_id), self.vm.udfs)
        share = self.vm.shares[str(share_id)]
        self.assertEqual('fake_share', share.name)
        self.assertEqual('fake_share_uuid', share.node_id)
        udf = self.vm.udfs[str(udf_id)]
        self.assertEqual('udf_uuid', udf.node_id)
        self.assertEqual(os.path.join(self.home_dir, 'UDF'), udf.path)
        # check that the root it's there, have right node_id and generation
        self.assertIn(request.ROOT, self.vm.shares)
        root = self.vm.shares[request.ROOT]
        self.assertEqual(root.node_id, str(root_volume.node_id))
        # now send the same list again and check
        # use a custom home
        with environ('HOME', self.home_dir):
            self.vm.handle_AQ_LIST_VOLUMES(response)
        self.assertEqual(2, len(self.vm.shares)) # the share and root
        self.assertEqual(1, len(self.vm.udfs)) # one udf
        # check that the udf is the same.
        new_udf = self.vm.udfs[str(udf_id)]
        self.assertEqual(udf.__dict__, new_udf.__dict__)

    def test_handle_AQ_LIST_VOLUMES_ERROR(self):
        """Test the handling of the AQ_LIST_VOLUMES_ERROR event."""
        # patch AQ.list_volumes
        class Helper(object):
            """Helper class to keep count of the retries"""
            retries = 0

            def list_volumes(self):
                """Fake list_volumes"""
                self.retries += 1

        helper = Helper()
        self.main.action_q = helper
        self.vm.handle_AQ_LIST_VOLUMES_ERROR("ERROR!")
        self.assertEquals(self.vm.list_volumes_retries, helper.retries)
        self.vm.handle_AQ_LIST_VOLUMES_ERROR("ERROR!")
        self.assertEquals(self.vm.list_volumes_retries, helper.retries)
        # reset the retry counter
        helper.retries = 0
        response = []
        self.vm.handle_AQ_LIST_VOLUMES(response)
        self.assertEquals(self.vm.list_volumes_retries, helper.retries)
        self.assertEquals(0, self.vm.list_volumes_retries)

    @defer.inlineCallbacks
    def test_handle_AQ_LIST_VOLUMES_unicode(self):
        """Test the handling of the AQ_LIST_VOLUMES event."""
        share_id = uuid.uuid4()
        share_volume = volumes.ShareVolume(share_id, 'fake_share_uuid', None,
                                           10, 'to_me', u'ñoño',
                                           'username', u'visible_username',
                                           True, 'View')
        udf_id = uuid.uuid4()
        udf_volume = volumes.UDFVolume(udf_id, 'udf_uuid', None, 10, u'~/ñoño')
        # initialize the the root
        self.vm._got_root('root_uuid')
        response = [share_volume, udf_volume]
        d = defer.Deferred()
        self._listen_for('VM_UDF_CREATED', d.callback)
        # use a custom home
        with environ('HOME', self.home_dir):
            self.vm.handle_AQ_LIST_VOLUMES(response)
        yield d
        self.assertEqual(2, len(self.vm.shares)) # the new shares and root
        self.assertEqual(1, len(self.vm.udfs)) # the new shares and root
        # check that the share is in the shares dict
        self.assertIn(str(share_id), self.vm.shares)
        self.assertIn(str(udf_id), self.vm.udfs)
        share = self.vm.shares[str(share_id)]
        self.assertEqual(u'ñoño', share.name)
        self.assertEqual('fake_share_uuid', share.node_id)
        udf = self.vm.udfs[str(udf_id)]
        self.assertEqual('udf_uuid', udf.node_id)
        self.assertEqual(os.path.join(self.home_dir, u'ñoño'.encode("utf8")),
                         udf.path)

    def test_handle_AQ_LIST_VOLUMES_root(self):
        """Test the handling of the AQ_LIST_VOLUMES event."""
        root_volume = volumes.RootVolume(uuid.uuid4(), None, 10)
        response = [root_volume]
        self.vm.refresh_volumes = lambda: self.fail('refresh_volumes called!')
        # use a custom home
        with environ('HOME', self.home_dir):
            self.vm.handle_AQ_LIST_VOLUMES(response)
        self.assertEqual(1, len(self.vm.shares)) # the new share and root
        # check that the root is in the shares dict
        self.assertIn(request.ROOT, self.vm.shares)
        self.assertEqual(self.vm.shares[request.ROOT].node_id,
                          str(root_volume.node_id))

    @defer.inlineCallbacks
    def test_handle_AQ_LIST_VOLUMES_accepted_share(self):
        """Test handle_AQ_LIST_VOLUMES event with an accepted share."""
        share_id = uuid.uuid4()
        share_response = ShareResponse.from_params(share_id, 'to_me',
                                                   'fake_share_uuid_to_me',
                                                   'fake_share_to_me',
                                                   'username_1',
                                                   'visible_username_1', False,
                                                   'View')
        shared_id = uuid.uuid4()
        shared_response = ShareResponse.from_params(shared_id, 'from_me',
                                                   'fake_share_uuid',
                                                   'fake_share', 'username',
                                                   'visible_username', True,
                                                   'View')
        shares_response = ListShares(None)
        shares_response.shares = [shared_response, share_response]
        self.vm.handle_AQ_SHARES_LIST(shares_response)
        # check
        self.assertIn(str(share_id), self.vm.shares)
        self.assertFalse(self.vm.shares[str(share_id)].active)
        self.assertIn(str(shared_id), self.vm.shared)
        udf_id = uuid.uuid4()
        udf_volume = volumes.UDFVolume(udf_id, 'udf_uuid', None, 100, u'~/UDF')
        root_volume = volumes.RootVolume(uuid.uuid4(), 17, 10)
        share_volume = volumes.ShareVolume(share_id, 'fake_share_uuid', 10,
                                           10, 'to_me', 'fake_share',
                                           'username', 'visible_username',
                                           True, 'View')
        response = [share_volume, udf_volume, root_volume]
        d1 = defer.Deferred()
        d2 = defer.Deferred()
        share_created_d = defer.Deferred()
        self.vm.refresh_volumes = lambda: d1.errback('refresh_volumes called!')
        self._listen_for('VM_UDF_CREATED', d1.callback)
        self._listen_for('SV_VOLUME_NEW_GENERATION', d2.callback, 2, collect=True)
        # listen for VM_SHARE_CREATED event for the new share
        self._listen_for('VM_SHARE_CREATED', share_created_d.callback)
        # use a custom home
        with environ('HOME', self.home_dir):
            self.vm.handle_AQ_LIST_VOLUMES(response)
        yield d1
        events = yield d2
        yield share_created_d
        events = [evt[1] for evt in events]
        expected_events = [{'generation': 10, 'volume_id': str(share_id)},
                           {'generation': 17, 'volume_id': ''}]
        self.assertEqual(events, expected_events)
        self.assertEqual(3, len(list(self.vm.get_volumes()))) # root, share and udf
        self.assertEqual(2, len(self.vm.shares)) # the share and the root
        self.assertEqual(1, len(self.vm.udfs)) # the new udf
        # check that the share is in the shares dict
        self.assertIn(str(share_id), self.vm.shares)
        self.assertIn(str(udf_id), self.vm.udfs)
        share = self.vm.shares[str(share_id)]
        self.assertEqual('fake_share_to_me', share.name)
        self.assertEqual('fake_share_uuid_to_me', share.node_id)
        self.assertTrue(share.accepted, "The share isn't accepted")
        udf = self.vm.udfs[str(udf_id)]
        self.assertEqual('udf_uuid', udf.node_id)
        self.assertEqual(os.path.join(self.home_dir, 'UDF'), udf.path)
        # check that the root it's there, have right node_id and generation
        self.assertIn(request.ROOT, self.vm.shares)
        root = self.vm.shares[request.ROOT]
        self.assertEqual(root.node_id, str(root_volume.node_id))
        # now send the same list again and check
        # use a custom home
        with environ('HOME', self.home_dir):
            self.vm.handle_AQ_LIST_VOLUMES(response)
        self.assertEqual(2, len(self.vm.shares)) # the share and root
        self.assertEqual(1, len(self.vm.udfs)) # one udf
        # check that the udf is the same.
        self.assertEqual(3, len(list(self.vm.get_volumes()))) # root, share and udf
        self.assertEqual(2, len(self.vm.shares)) # the share and the root
        self.assertEqual(1, len(self.vm.udfs)) # the new udf

    def test_get_udf_path_name(self):
        """Test for _get_udf_path_name."""
        home_dir = '/home/ubuntuonehacker'
        in_home = os.path.join(home_dir, 'foo')
        deep_in_home = os.path.join(home_dir, 'docs', 'foo', 'bar')
        outside_home = os.path.join('/', 'home', 'other', 'foo')
        relative_home = '../../home/foo/bar'
        with environ('HOME', home_dir):
            meth = self.vm._get_udf_path_name
            path, name = meth(in_home)
            self.assertEquals('~', path)
            self.assertEquals('foo', name)
            path, name = meth(deep_in_home)
            self.assertEquals('~/docs/foo', path)
            self.assertEquals('bar', name)
            self.assertRaises(ValueError, meth, outside_home)
            self.assertRaises(ValueError, meth, None)
            self.assertRaises(ValueError, meth, relative_home)

    def test_create_udf(self):
        """
        Test that VolumeManager.create_udf calls AQ.create_udf and
        AQ_CREATE_UDF_OK is correctly handled.

        """
        d = defer.Deferred()
        path = os.path.join(self.home_dir, "MyUDF")
        udf_id = uuid.uuid4()
        node_id = uuid.uuid4()
        # patch AQ.create_udf
        def create_udf(path, name, marker):
            """Fake create_udf"""
            self.main.event_q.push("AQ_CREATE_UDF_OK", **dict(volume_id=udf_id,
                                                       node_id=node_id,
                                                       marker=marker))
        self.main.action_q.create_udf = create_udf
        def check(udf):
            """Check the udf attributes."""
            udf = udf[0]
            self.assertEquals(udf.path, path)
            self.assertEquals(udf.volume_id, str(udf_id))
            self.assertEquals(udf.node_id, str(node_id))
            self.assertEquals(udf.suggested_path, u'~/MyUDF')
            self.assertTrue(isinstance(udf.suggested_path, unicode),
                            'suggested_path should be unicode')
            self.assertIn(udf.volume_id, self.vm.udfs)

        self._listen_for('VM_UDF_CREATED', d.callback)
        with environ('HOME', self.home_dir):
            self.vm.create_udf(path)
        d.addCallback(check)
        return d

    def test_create_udf_unicode(self):
        """Test VolumeManager.create_udf.

        Check that VM calls AQ.create_udf with unicode values.
        """
        d = defer.Deferred()
        path = os.path.join(self.home_dir, u"ñoño",
                            u"mirá que lindo mi udf").encode('utf-8')
        # patch AQ.create_udf
        def create_udf(path, name, marker):
            """Fake create_udf"""
            d.callback((path, name))
        self.main.action_q.create_udf = create_udf
        def check(info):
            """Check the values passed to AQ.create_udf"""
            path, name = info
            self.assertTrue(isinstance(name, unicode),
                            'name should be unicode but is: %s' % type(name))
            self.assertTrue(isinstance(path, unicode),
                            'path should be unicode but is: %s' % type(name))
        with environ('HOME', self.home_dir):
            self.vm.create_udf(path)
        d.addCallback(check)
        return d

    @defer.inlineCallbacks
    def test_delete_volume(self):
        """
        Test that VolumeManager.delete_volume calls AQ.delete_volume and
        AQ_DELETE_VOLUME_OK is correctly handled.

        """
        suggested_path = "suggested_path"
        udf, volume = self._create_udf(uuid.uuid4(), 'udf_node_id',
                                       '~/' + suggested_path, subscribed=False)
        yield self.vm.add_udf(udf)
        share_id = uuid.uuid4()
        share_volume = volumes.ShareVolume(share_id, 'fake_share_uuid', None,
                                           10, 'to_me', u'ñoño',
                                           'username', u'visible_username',
                                           True, 'View')
        dir_name = self.vm._build_share_path(share_volume.share_name,
                                          share_volume.other_visible_name)
        share_path = os.path.join(self.main.shares_dir, dir_name)
        share = Share.from_share_volume(share_volume, share_path)
        self.vm.add_share(share)
        d = defer.Deferred()
        # patch AQ.delete_volume
        def delete_volume(volume_id):
            """Fake delete_volume"""
            self.main.event_q.push("AQ_DELETE_VOLUME_OK", volume_id=volume_id)
        self.main.action_q.delete_volume = delete_volume
        def check_udf(info):
            """Check the udf attributes."""
            deleted_udf = info[0]
            self.assertEquals(deleted_udf.path, udf.path)
            self.assertEquals(deleted_udf.volume_id, udf.volume_id)
            self.assertEquals(deleted_udf.node_id, udf.node_id)
            self.assertEquals(deleted_udf.suggested_path, udf.suggested_path)
            self.assertNotIn(deleted_udf.volume_id, self.vm.udfs)
            d = defer.Deferred()
            self._listen_for('VM_VOLUME_DELETED', d.callback)
            with environ('HOME', self.home_dir):
                self.vm.delete_volume(share.volume_id)
            return d

        def check_share(info):
            """Check the share attributes."""
            deleted_share = info[0]
            self.assertEquals(deleted_share.path, share.path)
            self.assertEquals(deleted_share.volume_id, share.volume_id)
            self.assertEquals(deleted_share.node_id, share.node_id)
            self.assertNotIn(deleted_share.volume_id, self.vm.shares)

        self._listen_for('VM_VOLUME_DELETED', d.callback)
        d.addCallback(check_udf)
        d.addCallback(check_share)
        with environ('HOME', self.home_dir):
            self.vm.delete_volume(udf.volume_id)
        yield d

    @defer.inlineCallbacks
    def test_subscribe_udf(self):
        """Test VolumeManager.subscribe_udf method."""
        # create and add a UDF
        suggested_path = "suggested_path"
        udf, volume = self._create_udf(uuid.uuid4(), 'udf_node_id',
                                       '~/' + suggested_path, subscribed=False)
        yield self.vm.add_udf(udf)
        self.assertFalse(self.vm.udfs[udf.volume_id].subscribed)
        # subscribe to it
        yield self.vm.subscribe_udf(udf.volume_id)
        self.assertTrue(self.vm.udfs[udf.volume_id].subscribed)

    @defer.inlineCallbacks
    def test_subscribe_udf_missing_path(self):
        """Test VolumeManager.subscribe_udf with a missing path """
        # create and add a UDF
        suggested_path = "suggested_path"
        udf, volume = self._create_udf(uuid.uuid4(), 'udf_node_id',
                                       '~/' + suggested_path, subscribed=False)
        yield self.vm.add_udf(udf)
        self.assertFalse(os.path.exists(udf.path))
        self.assertFalse(self.vm.udfs[udf.id].subscribed)
        # subscribe to it
        yield self.vm.subscribe_udf(udf.id)
        self.assertTrue(self.vm.udfs[udf.id].subscribed)
        self.assertTrue(os.path.exists(udf.path))

    @defer.inlineCallbacks
    def test_subscribe_udf_missing_fsm_md(self):
        """Test VolumeManager.subscribe_udf with a missing node in fsm."""
        # create and add a UDF
        suggested_path = "suggested_path"
        udf, volume = self._create_udf(uuid.uuid4(), 'udf_node_id',
                                       '~/' + suggested_path, subscribed=False)
        yield self.vm.add_udf(udf)
        self.assertFalse(os.path.exists(udf.path))
        self.assertFalse(self.vm.udfs[udf.id].subscribed)
        yield self.vm.subscribe_udf(udf.id)
        yield self.vm.unsubscribe_udf(udf.id)
        # delete the fsm metadata
        self.main.fs.delete_metadata(udf.path)
        # subscribe to it and fail!
        try:
            yield self.vm.subscribe_udf(udf.id)
        except KeyError, e:
            self.assertIn(udf.path, e.args[0])
        else:
            self.fail('Must get a KeyError!')

    @defer.inlineCallbacks
    def test_subscribe_udf_missing_volume(self):
        """Test VolumeManager.subscribe_udf with a invalid volume_id."""
        # create and add a UDF
        try:
            yield self.vm.subscribe_udf('invalid_udf_id')
        except VolumeDoesNotExist, e:
            self.assertEquals('DOES_NOT_EXIST', e.args[0])
            self.assertEquals('invalid_udf_id', e.args[1])
        else:
            self.fail('Must get a VolumeDoesNotExist!')

    @defer.inlineCallbacks
    def _test_subscribe_udf_generations(self, udf):
        """Test subscribe_udf with a generation."""
        scratch_d = defer.Deferred()
        def fake_rescan_from_scratch(volume_id):
            """A fake rescan_from_scratch that check the arguments."""
            self.assertEquals(udf.volume_id, volume_id)
            scratch_d.callback(None)
        self.main.action_q.rescan_from_scratch = fake_rescan_from_scratch
        # subscribe to it
        yield self.vm.subscribe_udf(udf.volume_id)
        yield scratch_d
        self.assertTrue(self.vm.udfs[udf.volume_id].subscribed)

    @defer.inlineCallbacks
    def test_subscribe_udf_valid_generation(self):
        """Test subscribe_udf with a valid generation."""
        suggested_path = "suggested_path"
        udf, _ = self._create_udf(uuid.uuid4(), 'udf_node_id',
                                       '~/' + suggested_path,
                                       subscribed=False)
        yield self.vm.add_udf(udf)
        self.assertFalse(self.vm.udfs[udf.volume_id].subscribed)
        # update udf generation
        self.vm.update_generation(udf.volume_id, 0)
        yield self._test_subscribe_udf_generations(udf)

    @defer.inlineCallbacks
    def test_subscribe_udf_without_generation(self):
        """Test subscribe_udf without a valid generation."""
        # create and add a UDF
        suggested_path = "suggested_path"
        udf, _ = self._create_udf(uuid.uuid4(), 'udf_node_id',
                                       '~/' + suggested_path,
                                       subscribed=False)
        yield self.vm.add_udf(udf)
        self.assertFalse(self.vm.udfs[udf.volume_id].subscribed)
        # update udf generation
        self.vm.update_generation(udf.volume_id, None)
        yield self._test_subscribe_udf_generations(udf)

    @defer.inlineCallbacks
    def test_unsubscribe_udf(self):
        """Test VolumeManager.unsubscribe_udf method."""
        # create and add a UDF
        suggested_path = "suggested_path"
        udf, volume = self._create_udf(uuid.uuid4(), 'udf_node_id',
                                       '~/' + suggested_path, subscribed=True)
        yield self.vm.add_udf(udf)
        self.assertTrue(self.vm.udfs[udf.volume_id].subscribed)
        # unsubscribe from it
        self.vm.unsubscribe_udf(udf.volume_id)
        self.assertFalse(self.vm.udfs[udf.volume_id].subscribed)

    @defer.inlineCallbacks
    def test_unsubscribe_udf_with_content(self):
        """Test VolumeManager.unsubscribe_udf method in a UDF with content."""
        # create and add a UDF
        suggested_path = "suggested_path"
        udf, volume = self._create_udf(uuid.uuid4(), 'udf_node_id',
                                       '~/' + suggested_path, subscribed=True)
        yield self.vm.add_udf(udf)
        self.assertTrue(self.vm.udfs[udf.volume_id].subscribed)
        # create a few files and directories
        dirs = ['dir', 'dir/subdir', 'dir/empty_dir']
        for i, dir in enumerate(dirs):
            path = os.path.join(udf.path, dir)
            if not os.path.exists(path):
                os.makedirs(path)
            self.main.fs.create(path, udf.volume_id, is_dir=True)
            self.main.fs.set_node_id(path, 'dir_node_id'+str(i))
            # add a inotify watch to the dir
            self.main.event_q.inotify_add_watch(path)
        files = ['a_file', 'dir/file', 'dir/subdir/file']
        for i, file in enumerate(files):
            path = os.path.join(udf.path, file)
            open(path, 'w').close()
            self.main.fs.create(path, udf.volume_id)
            self.main.fs.set_node_id(path, 'file_node_id'+str(i))
        paths = self.main.fs.get_paths_starting_with(udf.path)
        self.assertEquals(len(paths), len(dirs+files)+1)

        # unsubscribe from it
        self.vm.unsubscribe_udf(udf.volume_id)

        self.assertEquals(1, len(self.vm.udfs))
        self.assertFalse(self.vm.udfs[udf.volume_id].subscribed)
        # check that the UDF is in the fsm metadata
        self.assertTrue(self.main.fs.get_by_path(udf.path))
        # get the childs (should be an empty list)
        paths = list(self.main.fs.get_paths_starting_with(udf.path))
        self.assertEquals(len(dirs+files)+1, len(paths))
        # check that there isn't a watch in the UDF
        self.assertFalse(self.main.event_q.inotify_has_watch(udf.path))
        # check that the childs don't have a watch
        for path, is_dir in paths:
            if is_dir:
                self.assertFalse(self.main.event_q.inotify_has_watch(path))

    @defer.inlineCallbacks
    def test_unsubscribe_subscribe_udf_with_content(self):
        """Test for re-subscribing to a UDF."""
        # create and add a UDF
        suggested_path = "suggested_path"
        udf, volume = self._create_udf(uuid.uuid4(), 'udf_node_id',
                                       '~/' + suggested_path, subscribed=True)
        yield self.vm.add_udf(udf)
        self.main.event_q.inotify_rm_watch(udf.path)
        self.assertTrue(self.vm.udfs[udf.volume_id].subscribed)
        # create a few files and directories
        dirs = ['dir', 'dir/subdir', 'dir/empty_dir']
        for i, path in enumerate(dirs):
            path = os.path.join(udf.path, path)
            if not os.path.exists(path):
                os.makedirs(path)
            self.main.fs.create(path, udf.volume_id, is_dir=True)
            self.main.fs.set_node_id(path, 'dir_node_id'+str(i))
        files = ['a_file', 'dir/file', 'dir/subdir/file']
        for i, path in enumerate(files):
            path = os.path.join(udf.path, path)
            open(path, 'w').close()
            self.main.fs.create(path, udf.volume_id)
            self.main.fs.set_node_id(path, 'file_node_id'+str(i))
        paths = list(self.main.fs.get_paths_starting_with(udf.path))
        # add a inotify watch to the dirs
        for path, is_dir in paths:
            if is_dir:
                self.main.event_q.inotify_add_watch(path)
        self.assertEquals(len(paths), len(dirs+files)+1, paths)

        # unsubscribe from it
        self.vm.unsubscribe_udf(udf.volume_id)

        self.assertEquals(1, len(self.vm.udfs))
        self.assertFalse(self.vm.udfs[udf.volume_id].subscribed)
        # check that the UDF is in the fsm metadata
        self.assertTrue(self.main.fs.get_by_path(udf.path))
        # check that there isn't a watch in the UDF
        self.assertFalse(self.main.event_q.inotify_has_watch(udf.path))
        # check that the childs don't have a watch
        for path, is_dir in paths:
            if is_dir:
                self.assertFalse(self.main.event_q.inotify_has_watch(path))
        # check the childs
        paths = self.main.fs.get_paths_starting_with(udf.path)
        self.assertEquals(len(dirs+files)+1, len(paths))
        # resubscribe to it
        yield self.vm.subscribe_udf(udf.volume_id)
        paths = list(self.main.fs.get_paths_starting_with(udf.path))
        # we should only have the dirs, as the files metadata is
        # delete by local rescan (both hashes are '')
        self.assertEquals(len(dirs)+1, len(paths))
        # check that there is a watch in the UDF
        self.assertTrue(self.main.event_q.inotify_has_watch(udf.path))
        # check that the child dirs have a watch
        for path, is_dir in paths:
            if is_dir:
                self.assertTrue(self.main.event_q.inotify_has_watch(path))
                self.vm._remove_watch(path)

    @defer.inlineCallbacks
    def test_cleanup_volumes(self):
        """Test for VolumeManager._cleanup_volumes"""
        share_path = os.path.join(self.shares_dir, 'fake_share')
        share = Share(path=share_path, volume_id='share_id')
        self.vm.add_share(share)
        suggested_path = "suggested_path"
        udf, volume = self._create_udf(uuid.uuid4(), 'udf_node_id',
                                       '~/' + suggested_path, subscribed=True)
        yield self.vm.add_udf(udf)
        self.assertIn(share.volume_id, self.vm.shares)
        self.assertIn(udf.volume_id, self.vm.udfs)
        self.vm._cleanup_volumes(shares=[])
        self.assertNotIn(share.volume_id, self.vm.shares)
        self.assertIn(udf.volume_id, self.vm.udfs)
        self.vm._cleanup_volumes(udfs=[])
        self.assertNotIn(udf.volume_id, self.vm.udfs)
        # all at once
        self.vm.add_share(share)
        yield self.vm.add_udf(udf)
        self.assertIn(share.volume_id, self.vm.shares)
        self.assertIn(udf.volume_id, self.vm.udfs)
        self.vm._cleanup_volumes(shares=[], udfs=[])
        self.assertNotIn(share.volume_id, self.vm.shares)
        self.assertNotIn(udf.volume_id, self.vm.udfs)

    def test_cleanup_shared(self):
        """Test for VolumeManager._cleanup_shared"""
        shared_path = os.path.join(self.root_dir, 'fake_shared')
        shared = Share(path=shared_path, volume_id='shared_id')
        self.vm.add_shared(shared)
        self.assertIn(shared.volume_id, self.vm.shared)
        self.vm._cleanup_shared([shared.volume_id])
        self.assertIn(shared.volume_id, self.vm.shared)
        self.vm._cleanup_shared([])
        self.assertNotIn(shared.volume_id, self.vm.shared)

    def test_cleanup_shares(self):
        """Test for VolumeManager._cleanup_shares"""
        share_path = os.path.join(self.shares_dir, 'fake_share')
        share = Share(path=share_path, volume_id='share_id')
        share_2_path = os.path.join(self.root_dir, 'fake_share_2')
        share_2 = Share(path=share_2_path, volume_id='share_2_id')
        self.vm.add_share(share)
        self.vm.add_share(share_2)
        self.assertIn(share.volume_id, self.vm.shares)
        self.assertIn(share_2.volume_id, self.vm.shares)
        self.vm._cleanup_shares([])
        self.assertNotIn(share.volume_id, self.vm.shares)
        self.assertNotIn(share_2.volume_id, self.vm.shares)
        self.vm.add_share(share)
        self.vm.add_share(share_2)
        self.vm._cleanup_shares([share.volume_id])
        self.assertIn(share.volume_id, self.vm.shares)
        self.assertNotIn(share_2.volume_id, self.vm.shares)
        self.vm.add_share(share)
        self.vm.add_share(share_2)
        self.vm._cleanup_shares([share.volume_id, share_2.volume_id])
        self.assertIn(share.volume_id, self.vm.shares)
        self.assertIn(share_2.volume_id, self.vm.shares)

    def _test_handle_AQ_CREATE_UDF_OK(self, auto_subscribe):
        """Test for handle_AQ_CREATE_UDF_OK."""
        user_conf = config.get_user_config()
        user_conf.set_udf_autosubscribe(auto_subscribe)
        d = defer.Deferred()
        path = os.path.join(self.home_dir, u'ñoño'.encode("utf8"))
        udf_id = uuid.uuid4()
        node_id = uuid.uuid4()
        # patch AQ.create_udf
        def create_udf(path, name, marker):
            """Fake create_udf"""
            self.main.event_q.push("AQ_CREATE_UDF_OK", **dict(volume_id=udf_id,
                                                       node_id=node_id,
                                                       marker=marker))
        self.main.action_q.create_udf = create_udf
        self._listen_for('VM_UDF_CREATED', d.callback)
        # fake VM state, call create_udf
        with environ('HOME', self.home_dir):
            self.vm.create_udf(path)
        def check(info):
            """The callback"""
            udf = info[0]
            self.assertEquals(udf.path, path)
            self.assertEquals(udf.volume_id, str(udf_id))
            self.assertEquals(udf.node_id, str(node_id))
            self.assertEquals(0, len(self.vm.marker_udf_map))
            self.assertTrue(self.vm.udfs[str(udf_id)])
            if auto_subscribe:
                self.assertTrue(udf.subscribed)
                self.assertTrue(os.path.exists(udf.path))
            else:
                self.assertFalse(udf.subscribed)
                self.assertFalse(os.path.exists(udf.path))
        d.addCallback(check)
        return d

    def test_handle_AQ_CREATE_UDF_OK_subscribe(self):
        """Test AQ_CREATE_UDF_OK with udf auto_subscribe """
        return self._test_handle_AQ_CREATE_UDF_OK(True)

    def test_handle_AQ_CREATE_UDF_OK_no_subscribe(self):
        """Test AQ_CREATE_UDF_OK without udf auto_subscribe """
        return self._test_handle_AQ_CREATE_UDF_OK(False)

    def test_handle_AQ_CREATE_UDF_ERROR(self):
        """Test for handle_AQ_CREATE_UDF_ERROR."""
        d = defer.Deferred()
        path = os.path.join(self.home_dir, u'ñoño'.encode("utf8"))
        # patch AQ.create_udf
        def create_udf(path, name, marker):
            """Fake create_udf"""
            self.main.event_q.push("AQ_CREATE_UDF_ERROR", **dict(marker=marker,
                                                            error="ERROR!"))
        self.main.action_q.create_udf = create_udf
        self._listen_for('VM_UDF_CREATE_ERROR', d.callback)
        # fake VM state, call create_udf
        with environ('HOME', self.home_dir):
            self.vm.create_udf(path)
        def check(info):
            """The callback"""
            udf_path, error = info
            self.assertEquals(udf_path, path)
            self.assertEquals(error, "ERROR!")
            self.assertEquals(0, len(self.vm.marker_udf_map))
        d.addCallback(check)
        return d

    @defer.inlineCallbacks
    def test_handle_AQ_DELETE_VOLUME_OK(self):
        """Test for handle_AQ_DELETE_VOLUME_OK."""
        suggested_path = "suggested_path"
        udf, volume = self._create_udf(uuid.uuid4(), 'udf_node_id',
                                       '~/' + suggested_path, subscribed=False)
        yield self.vm.add_udf(udf)
        d = defer.Deferred()
        # patch AQ.delete_volume
        def delete_volume(path):
            """Fake delete_volume"""
            self.main.event_q.push("AQ_DELETE_VOLUME_OK", volume_id=udf.volume_id)
        self.main.action_q.delete_volume = delete_volume
        def check(info):
            """Check the udf attributes."""
            deleted_udf = info[0]
            self.assertEquals(deleted_udf.path, udf.path)
            self.assertEquals(deleted_udf.volume_id, udf.volume_id)
            self.assertEquals(deleted_udf.node_id, udf.node_id)
            self.assertEquals(deleted_udf.suggested_path, udf.suggested_path)
            self.assertNotIn(deleted_udf.volume_id, self.vm.udfs)
        self._listen_for('VM_VOLUME_DELETED', d.callback)
        d.addCallback(check)
        with environ('HOME', self.home_dir):
            self.vm.delete_volume(udf.volume_id)
        yield d

    @defer.inlineCallbacks
    def test_handle_AQ_DELETE_VOLUME_ERROR(self):
        """Test for handle_AQ_DELETE_VOLUME_ERROR."""
        suggested_path = "suggested_path"
        udf, volume = self._create_udf(uuid.uuid4(), 'udf_node_id',
                                       '~/' + suggested_path, subscribed=False)
        yield self.vm.add_udf(udf)
        d = defer.Deferred()
        # patch AQ.delete_volume
        def delete_volume(path):
            """Fake delete_volume"""
            self.main.event_q.push("AQ_DELETE_VOLUME_ERROR",
                                   volume_id=udf.volume_id, error="ERROR!")
        self.main.action_q.delete_volume = delete_volume
        def check(info):
            """Check the udf attributes."""
            deleted_udf, error = info
            self.assertEquals(deleted_udf, udf.volume_id)
            self.assertIn(deleted_udf, self.vm.udfs)
            self.assertEquals(error, 'ERROR!')
        self._listen_for('VM_VOLUME_DELETE_ERROR', d.callback)
        d.addCallback(check)
        with environ('HOME', self.home_dir):
            self.vm.delete_volume(udf.volume_id)
        yield d

    def test_handle_AQ_DELETE_VOLUME_ERROR_missing_volume(self):
        """Test for handle_AQ_DELETE_VOLUME_ERROR for a missing volume."""
        called = []
        self.vm.refresh_volumes = lambda: called.append(True)
        self.vm.handle_AQ_DELETE_VOLUME_ERROR('unknown vol id', 'error')
        self.assertTrue(called)
        self.assertTrue(self.handler.check_warning("missing volume id"))

    @defer.inlineCallbacks
    def _test_handle_SV_VOLUME_CREATED(self, auto_subscribe):
        """Test for handle_SV_VOLUME_CREATED."""
        user_conf = config.get_user_config()
        user_conf.set_udf_autosubscribe(auto_subscribe)
        # start the test
        udf_id = uuid.uuid4()
        udf_volume = volumes.UDFVolume(udf_id, 'udf_uuid', None, 10, u'~/ñoño')
        share_id = uuid.uuid4()
        share_volume = volumes.ShareVolume(share_id, 'fake_share_uuid', None,
                                           10, 'to_me', u'ñoño',
                                           'username', u'visible_username',
                                           True, 'View')
        # initialize the the root
        self.vm._got_root('root_uuid')

        d = defer.Deferred()
        self._listen_for('VM_UDF_CREATED', d.callback)
        rescan_cb = defer.Deferred()
        self.patch(self.main.action_q, 'rescan_from_scratch', rescan_cb.callback)

        with environ('HOME', self.home_dir):
            self.vm.handle_SV_VOLUME_CREATED(udf_volume)
        info = yield d
        udf = info[0]
        self.assertEquals(udf.volume_id, str(udf_id))
        self.assertIn(str(udf_id), self.vm.udfs)
        if auto_subscribe:
            self.assertTrue(udf.subscribed)
            self.assertTrue(os.path.exists(udf.path))
            # check that rescan_from_scratch is called
            vol_id = yield rescan_cb
            self.assertEqual(vol_id, udf.volume_id)
        else:
            self.assertFalse(udf.subscribed)
            self.assertFalse(os.path.exists(udf.path))
        # subscribe a new listener
        d = defer.Deferred()
        self._listen_for('VM_SHARE_CREATED', d.callback)
        rescan_cb = defer.Deferred()
        self.patch(self.main.action_q, 'rescan_from_scratch', rescan_cb.callback)
        # fire SV_VOLUME_CREATED with a share
        with environ('HOME', self.home_dir):
            self.vm.handle_SV_VOLUME_CREATED(share_volume)
        info = yield d
        share_id = info[0]
        share = self.vm.get_volume(share_id)
        self.assertEquals(share.volume_id, str(share_id))
        self.assertIn(str(share_id), self.vm.shares)
        # check that rescan_from_scratch is called
        vol_id = yield rescan_cb
        self.assertEqual(vol_id, share.volume_id)

    def test_handle_SV_VOLUME_CREATED_subscribe(self):
        """Test SV_VOLUME_CREATED with udf auto_subscribe """
        return self._test_handle_SV_VOLUME_CREATED(True)

    def test_handle_SV_VOLUME_CREATED_no_subscribe(self):
        """Test SV_VOLUME_CREATED without udf auto_subscribe """
        return self._test_handle_SV_VOLUME_CREATED(False)

    @defer.inlineCallbacks
    def test_handle_SV_VOLUME_DELETED(self):
        """Test for handle_SV_VOLUME_DELETED."""
        share_id = uuid.uuid4()
        share_volume = volumes.ShareVolume(share_id, 'fake_share_uuid', None,
                                           10, 'to_me', u'ñoño',
                                           'username', u'visible_username',
                                           True, 'View')
        dir_name = self.vm._build_share_path(share_volume.share_name,
                                          share_volume.other_visible_name)
        share_path = os.path.join(self.main.shares_dir, dir_name)
        share = Share.from_share_volume(share_volume, share_path)
        # create a UDF
        udf_id = uuid.uuid4()
        udf, udf_volume = self._create_udf(udf_id, 'udf_uuid', u'~/ñoño')
        yield self.vm.add_udf(udf)
        self.vm.add_share(share)
        # initialize the the root
        self.vm._got_root('root_uuid')
        d = defer.Deferred()
        self._listen_for('VM_VOLUME_DELETED', d.callback)
        with environ('HOME', self.home_dir):
            self.vm.handle_SV_VOLUME_DELETED(udf_volume.volume_id)
        info = yield d

        udf = info[0]
        self.assertEquals(udf.volume_id, str(udf_id))
        self.assertNotIn(str(udf_id), self.vm.udfs)
        # subscribe a new listener, for deleting a share.
        share_deferred = defer.Deferred()
        self._listen_for('VM_VOLUME_DELETED', share_deferred.callback)
        # fire SV_VOLUME_DELETED with a share
        with environ('HOME', self.home_dir):
            self.vm.handle_SV_VOLUME_DELETED(share_volume.volume_id)
        share_info = yield share_deferred
        share = share_info[0]
        self.assertEquals(share.volume_id, str(share_id))
        self.assertNotIn(str(share.volume_id), self.vm.shares)

    @defer.inlineCallbacks
    def test_get_volumes(self):
        """Tests for VolumeManager.get_volumes."""
        share_path = os.path.join(self.shares_dir, 'fake_share')
        share_modify = Share(path=share_path, volume_id='share_id',
                             node_id=str(uuid.uuid4()),
                             access_level='Modify', accepted=True)
        share_no_accepted_path = os.path.join(self.shares_dir, 'fake_share')
        share_no_accepted = Share(path=share_no_accepted_path,
                               node_id=str(uuid.uuid4()),
                               volume_id='accepted_share_id',
                               access_level='Modify', accepted=False)
        share_view = Share(path=share_path, volume_id='share_id_view',
                           access_level='View', accepted=True)
        self.vm.add_share(share_modify)
        self.vm.add_share(share_view)
        self.vm.add_share(share_no_accepted)
        shared_path = os.path.join(self.root_dir, 'fake_shared')
        shared = Share(path=shared_path, volume_id='shared_id')
        self.vm.add_shared(shared)
        suggested_path = "suggested_path"
        udf_subscribed, _ = self._create_udf(uuid.uuid4(), 'udf_node_id',
                                       '~/' + suggested_path, subscribed=True)
        udf_unsubscribed, _ = self._create_udf(uuid.uuid4(), 'udf_node_id',
                                       '~/' + suggested_path, subscribed=False)
        yield self.vm.add_udf(udf_subscribed)
        yield self.vm.add_udf(udf_unsubscribed)
        volumes = list(self.vm.get_volumes())
        volumes_ids = [v.id for v in volumes]
        self.assertNotIn(share_no_accepted.id, volumes_ids)
        self.assertIn(share_modify.id, volumes_ids)
        self.assertIn(share_view.id, volumes_ids)
        self.assertNotIn(shared.id, volumes_ids)
        self.assertIn(udf_subscribed.id, volumes_ids)
        self.assertNotIn(udf_unsubscribed.id, volumes_ids)
        all_volumes = list(self.vm.get_volumes(all_volumes=True))
        all_volumes_ids = [v.id for v in all_volumes]
        self.assertIn(share_no_accepted.id, all_volumes_ids)
        self.assertIn(share_modify.id, all_volumes_ids)
        self.assertIn(share_view.id, all_volumes_ids)
        self.assertNotIn(shared.id, all_volumes_ids)
        self.assertIn(udf_subscribed.id, all_volumes_ids)
        self.assertIn(udf_unsubscribed.id, all_volumes_ids)

    def test_udf_can_write(self):
        """Test that UDF class match Share.can_write() API."""
        suggested_path = "suggested_path"
        udf, volume = self._create_udf(uuid.uuid4(), 'udf_node_id',
                                       '~/' + suggested_path, subscribed=False)
        self.assertTrue(udf.can_write())

    def test_udf_from_udf_volume(self):
        """Test for UDF.from_udf_volume."""
        suggested_path = u'~/foo/bar'
        with environ('HOME', self.home_dir):
            path = self.vm._build_udf_path(suggested_path)
        volume = volumes.UDFVolume(uuid.uuid4(), uuid.uuid4(), None,
                                   10, suggested_path)
        udf = UDF.from_udf_volume(volume, path)
        self.assertTrue(isinstance(udf.id, basestring))
        self.assertTrue(isinstance(udf.node_id, basestring))

    def test_share_from_share_volume(self):
        """Test for Share.from_share_volume."""
        volume = volumes.ShareVolume(uuid.uuid4(), uuid.uuid4(), None, 10,
                                     "to_me", u"share_name", u"other_username",
                                     u"other_visible_name", True, 'Modify')
        dir_name = self.vm._build_share_path(volume.share_name,
                                          volume.other_visible_name)
        path = os.path.join(self.main.shares_dir, dir_name)
        share = Share.from_share_volume(volume, path)
        self.assertTrue(isinstance(share.id, basestring))
        self.assertTrue(isinstance(share.node_id, basestring))

    @defer.inlineCallbacks
    def test_volumes_list_args_as_AQ_wants(self):
        """Test that handle_AQ_LIST_VOLUMES match the kwargs used by AQ."""
        root_volume = volumes.RootVolume(uuid.uuid4(), None, 10)
        d = defer.Deferred()
        self.vm._got_root = lambda node_id, free_bytes: d.callback(
                                                        (node_id, free_bytes))
        self.main.event_q.push('AQ_LIST_VOLUMES', volumes=[root_volume])
        root_node_id, free_bytes = yield d
        self.assertEquals(str(root_volume.node_id), root_node_id)
        self.assertEquals(root_volume.free_bytes, free_bytes)

    @defer.inlineCallbacks
    def test_no_UDFs_inside_root(self):
        """Test that a UDF can't be created inside the root"""
        # initialize the root
        self.vm._got_root('root_uuid')
        udf_path = os.path.join(self.root_dir, 'udf_inside_root')
        # patch FakeAQ
        def create_udf(path, name, marker):
            """Fake create_udf"""
            d = dict(volume_id=uuid.uuid4(), node_id=uuid.uuid4(),
                     marker=marker)
            self.main.event_q.push("AQ_CREATE_UDF_OK", **d)

        self.main.action_q.create_udf = create_udf
        d = defer.Deferred()
        self._listen_for('VM_UDF_CREATE_ERROR', d.callback)
        self._listen_for('VM_UDF_CREATED', lambda r: d.errback(Exception(r)))
        self.vm.create_udf(udf_path)
        result = yield d
        self.assertEquals(0, len(list(self.vm.udfs.keys())))
        self.assertEquals(result[0], udf_path)
        self.assertEquals(result[1], "UDFs can not be nested")

    @defer.inlineCallbacks
    def test_no_UDFs_inside_udf(self):
        """Test that a UDF can't be created inside a UDF."""
        udf, _ = self._create_udf(uuid.uuid4(), uuid.uuid4(),
                               '~/a/b/c', subscribed=True)
        udf_child = os.path.join(udf.path, 'd')
        yield self.vm.add_udf(udf)
        # patch FakeAQ
        def create_udf(path, name, marker):
            """Fake create_udf"""
            self.main.event_q.push("AQ_CREATE_UDF_OK", **dict(volume_id=uuid.uuid4(),
                                                       node_id=uuid.uuid4(),
                                                       marker=marker))
        self.main.action_q.create_udf = create_udf
        d = defer.Deferred()
        self._listen_for('VM_UDF_CREATE_ERROR', d.callback)
        self._listen_for('VM_UDF_CREATED', lambda r: d.errback(Exception(r)))
        self.vm.create_udf(udf_child)
        result = yield d
        self.assertEquals(1, len(list(self.vm.udfs.keys())))
        self.assertEquals(result[0], udf_child)
        self.assertEquals(result[1], "UDFs can not be nested")

    @defer.inlineCallbacks
    def test_no_UDFs_as_UDF_parent(self):
        """Test that a UDF can't be created if there is a UDF inside."""
        udf, _ = self._create_udf(uuid.uuid4(), uuid.uuid4(),
                               '~/a/b/c', subscribed=True)
        yield self.vm.add_udf(udf)
        udf_parent_path = os.path.dirname(udf.path)
        # patch FakeAQ
        def create_udf(path, name, marker):
            """Fake create_udf"""
            self.main.event_q.push("AQ_CREATE_UDF_OK", **dict(volume_id=uuid.uuid4(),
                                                       node_id=uuid.uuid4(),
                                                       marker=marker))
        self.main.action_q.create_udf = create_udf
        d = defer.Deferred()
        self._listen_for('VM_UDF_CREATE_ERROR', d.callback)
        self._listen_for('VM_UDF_CREATED', lambda r: d.errback(Exception(r)))
        self.vm.create_udf(udf_parent_path)
        result = yield d
        self.assertEquals(1, len(list(self.vm.udfs.keys())))
        self.assertEquals(result[0], udf_parent_path)
        self.assertEquals(result[1], "UDFs can not be nested")


    @defer.inlineCallbacks
    def test_dont_delete_volumes_on_handle_AQ_SHARES_LIST(self):
        """Test that VM don't delete shares when handling AQ_SHARE_LIST."""
        share_id = uuid.uuid4()
        share_volume = volumes.ShareVolume(share_id, 'fake_share_uuid', None,
                                           10, 'to_me', 'fake_share', 'username',
                                           'visible_username', True, 'View')
        udf_id = uuid.uuid4()
        udf_volume = volumes.UDFVolume(udf_id, 'udf_uuid', None, 10, u'~/UDF')
        root_volume = volumes.RootVolume(uuid.uuid4(), None, 10)
        response = [share_volume, udf_volume, root_volume]
        d = defer.Deferred()
        self.vm.refresh_volumes = lambda: d.errback('refresh_volumes called!')
        self._listen_for('VM_UDF_CREATED', d.callback)
        # use a custom home
        with environ('HOME', self.home_dir):
            self.vm.handle_AQ_LIST_VOLUMES(response)
        yield d
        self.assertEquals(2, len(self.vm.shares)) # the new share and root
        self.assertEquals(1, len(self.vm.udfs)) # the new udf
        shared_id = uuid.uuid4()
        shared_response = ShareResponse.from_params(shared_id, 'from_me',
                                                   'fake_share_uuid',
                                                   'fake_share', 'username',
                                                   'visible_username', 'yes',
                                                   'View')
        shares_response = ListShares(None)
        shares_response.shares = [shared_response]
        self.vm.handle_AQ_SHARES_LIST(shares_response)
        # check that all the shares are still there
        self.assertEquals(2, len(self.vm.shares)) # the new share and root

    @defer.inlineCallbacks
    def test_handle_SV_FREE_SPACE(self):
        """Test for VolumeManager.handle_SV_FREE_SPACE."""
        # create a Share
        share_id = uuid.uuid4()
        share_volume = volumes.ShareVolume(share_id, 'fake_share_uuid', None,
                                           None, 'to_me', 'fake_share', 'username',
                                           'visible_username', True, 'View')
        dir_name = self.vm._build_share_path(share_volume.share_name,
                                          share_volume.other_visible_name)
        share_path = os.path.join(self.main.shares_dir, dir_name)
        share = Share.from_share_volume(share_volume, share_path)
        # get the root
        root = self.vm.get_volume(request.ROOT)
        self.vm._got_root('root_uuid')
        # create a UDF
        udf_id = uuid.uuid4()
        udf, volume = self._create_udf(udf_id, 'udf_node_id', '~/UDF')
        self.vm.add_share(share)
        yield self.vm.add_udf(udf)
        # override AQ.check_conditions
        d = defer.Deferred()
        def check_conditions():
            """Fake check_conditions that just keep count of calls."""
            if d.called:
                d.addCallback(lambda _: defer.succeed(d.result + 1))
            else:
                d.callback(1)
        self.main.action_q.check_conditions = check_conditions
        # now start playing
        assert root.free_bytes is None, 'root free_bytes should be None'
        assert share.free_bytes is None, 'share free_bytes should be None'
        self.vm.handle_SV_FREE_SPACE(root.volume_id, 10)
        self.assertEquals(10, self.vm.get_free_space(root.volume_id))
        self.vm.handle_SV_FREE_SPACE(share.volume_id, 20)
        self.assertEquals(20, self.vm.get_free_space(share.volume_id))
        self.vm.handle_SV_FREE_SPACE(udf.volume_id, 50)
        self.assertEquals(50, self.vm.get_free_space(udf.volume_id))
        # udf free space is root free space, check it's the same
        self.assertEquals(50, self.vm.get_free_space(root.volume_id))
        counter = yield d
        # check that check_conditions was called 3 times
        self.assertEquals(3, counter)

    @defer.inlineCallbacks
    def test_update_and_get_free_space(self):
        """Test for VolumeManager.update_free_space and get_free_space."""
        share_id = uuid.uuid4()
        share_volume = volumes.ShareVolume(share_id, 'fake_share_uuid', None,
                                           None, 'to_me', 'fake_share', 'username',
                                           'visible_username', True, 'View')
        dir_name = self.vm._build_share_path(share_volume.share_name,
                                          share_volume.other_visible_name)
        share_path = os.path.join(self.main.shares_dir, dir_name)
        share = Share.from_share_volume(share_volume, share_path)
        # get the root
        root = self.vm.get_volume(request.ROOT)
        self.vm._got_root('root_node_id')
        # create a UDF
        udf_id = uuid.uuid4()
        udf, volume = self._create_udf(udf_id, 'udf_node_id', '~/UDF')
        self.vm.add_share(share)
        yield self.vm.add_udf(udf)
        self.assertEquals(None, self.vm.get_free_space(share.volume_id))
        self.assertEquals(None, self.vm.get_free_space(udf.volume_id))
        self.assertEquals(None, self.vm.get_free_space(root.volume_id))
        self.vm.update_free_space(share.volume_id, 10)
        self.vm.update_free_space(udf.volume_id, 20)
        self.vm.update_free_space('missing_id', 20)
        self.assertEquals(10, self.vm.get_free_space(share.volume_id))
        self.assertEquals(20, self.vm.get_free_space(udf.volume_id))
        self.assertEquals(20, self.vm.get_free_space(root.volume_id))
        self.assertEquals(0, self.vm.get_free_space('missing_id'))
        self.vm.update_free_space(root.volume_id, 30)
        self.assertEquals(30, self.vm.get_free_space(udf.volume_id))
        self.assertEquals(30, self.vm.get_free_space(root.volume_id))

    def test_get_free_space_no_volume(self):
        """Test get_free_space for a volume we don't have."""
        called = []
        self.vm.refresh_volumes = lambda: called.append(True)
        self.vm.get_free_space('unknown vol id')
        self.assertTrue(called)
        self.assertTrue(self.handler.check_warning("there is no such volume"))

    def test_update_free_space_no_volume(self):
        """Test update_free_space for a volume we don't have."""
        called = []
        self.vm.refresh_volumes = lambda: called.append(True)
        self.vm.update_free_space('unknown vol id', 123)
        self.assertTrue(called)
        self.assertTrue(self.handler.check_warning("no such volume_id"))

    @defer.inlineCallbacks
    def test_UDF_cant_be_a_symlink(self):
        """Test that a UDF can't be a symlink."""
        # initialize the root
        self.vm._got_root('root_uuid')
        real_udf_path = os.path.join(self.home_dir, "my_udf")
        udf_path = os.path.join(self.home_dir, "MyUDF")
        # patch FakeAQ
        def create_udf(path, name, marker):
            """Fake create_udf"""
            self.main.event_q.push("AQ_CREATE_UDF_OK", **dict(volume_id=uuid.uuid4(),
                                                       node_id=uuid.uuid4(),
                                                       marker=marker))
        self.main.action_q.create_udf = create_udf
        # create the symlink
        os.makedirs(real_udf_path)
        os.symlink(real_udf_path, udf_path)
        d = defer.Deferred()
        self._listen_for('VM_UDF_CREATE_ERROR', d.callback)
        self._listen_for('VM_UDF_CREATED', lambda r: d.errback(Exception(r)))
        self.vm.create_udf(udf_path)
        result = yield d
        self.assertEquals(0, len(list(self.vm.udfs.keys())))
        self.assertEquals(result[0], udf_path)
        self.assertEquals(result[1], "UDFs can not be a symlink")

    @defer.inlineCallbacks
    def test_UDF_cant_be_inside_symlink(self):
        """Test that a UDF can't be inside a symlink."""
        # initialize the root
        self.vm._got_root('root_uuid')
        real_udf_path = os.path.join(self.home_dir, "udf_parent", "my_udf")
        udf_path = os.path.join(self.home_dir, "MyUDF")
        # patch FakeAQ
        def create_udf(path, name, marker):
            """Fake create_udf"""
            self.main.event_q.push("AQ_CREATE_UDF_OK", **dict(volume_id=uuid.uuid4(),
                                                       node_id=uuid.uuid4(),
                                                       marker=marker))
        self.main.action_q.create_udf = create_udf
        # create the symlink
        os.makedirs(real_udf_path)
        os.symlink(real_udf_path, udf_path)
        d = defer.Deferred()
        self._listen_for('VM_UDF_CREATE_ERROR', d.callback)
        self._listen_for('VM_UDF_CREATED', lambda r: d.errback(Exception(r)))
        self.vm.create_udf(udf_path)
        result = yield d
        self.assertEquals(0, len(list(self.vm.udfs.keys())))
        self.assertEquals(result[0], udf_path)
        self.assertEquals(result[1], "UDFs can not be a symlink")

    @defer.inlineCallbacks
    def test_server_rescan(self):
        """Test the server_rescan method."""
        share_id = uuid.uuid4()
        share_volume = volumes.ShareVolume(share_id, 'fake_share_uuid', 1, 100,
                                           'to_me', 'fake_share', 'username',
                                           'visible_username', True, 'View')
        udf_id = uuid.uuid4()
        udf_volume = volumes.UDFVolume(udf_id, 'udf_node_id', 1, 200, u'~/UDF')
        root_id = uuid.uuid4()
        root_volume = volumes.RootVolume(root_id, 1, 500)
        response = [share_volume, udf_volume, root_volume]

        # patch fake action queue
        self.main.action_q.query_volumes = lambda: defer.succeed(response)
        vol_rescan_d = defer.Deferred()
        self._listen_for('SV_VOLUME_NEW_GENERATION',
                         vol_rescan_d.callback, 2, True)
        server_rescan_d = defer.Deferred()
        self._listen_for('SYS_SERVER_RESCAN_DONE', server_rescan_d.callback)
        with environ('HOME', self.home_dir):
            yield self.vm.server_rescan()
        yield server_rescan_d
        events = yield vol_rescan_d
        events_dict = dict((event[1]['volume_id'], event[1]['generation']) \
                           for event in events)
        self.assertIn(str(share_id), events_dict)
        # new udfs server rescan is triggered after local rescan.
        self.assertNotIn(str(udf_id), events_dict)
        self.assertIn(request.ROOT, events_dict)
        self.assertEquals(1, events_dict[str(share_id)])
        self.assertEquals(1, events_dict[request.ROOT])

    @defer.inlineCallbacks
    def test_server_rescan_error(self):
        """Test the server_rescan method."""
        # patch fake action queue
        self.main.action_q.query_volumes = lambda: defer.fail(Exception('foo bar'))
        d = defer.Deferred()
        self._listen_for('SYS_SERVER_RESCAN_ERROR', d.callback)
        with environ('HOME', self.home_dir):
            yield self.vm.server_rescan()
        yield d
        # now when _volumes_rescan_cb fails
        # patch fake action queue
        self.main.action_q.query_volumes = lambda: defer.succeed([])
        # patch volume manager
        def broken_volumes_rescan_cb(_):
            raise ValueError('die!')
        self.vm._volumes_rescan_cb = broken_volumes_rescan_cb
        d = defer.Deferred()
        self._listen_for('SYS_SERVER_RESCAN_ERROR', d.callback)
        with environ('HOME', self.home_dir):
            yield self.vm.server_rescan()
        yield d

    @defer.inlineCallbacks
    def test_refresh_shares_called_after_server_rescan(self):
        """Test that refresh_shares is called after server_rescan."""
        root_id = uuid.uuid4()
        root_volume = volumes.RootVolume(root_id, 1, 500)
        response = [root_volume]

        # patch fake action queue
        self.main.action_q.query_volumes = lambda: defer.succeed(response)
        d = defer.Deferred()
        self.vm.refresh_shares = lambda: d.callback(True)
        with environ('HOME', self.home_dir):
            yield self.vm.server_rescan()
        called = yield d
        self.assertTrue(called)
    test_refresh_shares_called_after_server_rescan.timeout  = 1

    @defer.inlineCallbacks
    def test_server_rescan_clean_dead_udf(self):
        """Test cleanup of dead volumes after server_rescan method."""
        share_id = uuid.uuid4()
        share_volume = volumes.ShareVolume(share_id, 'fake_share_uuid', 1, 100,
                                           'to_me', 'fake_share', 'username',
                                           'visible_username', True, 'View')
        udf_id = uuid.uuid4()
        udf_volume = volumes.UDFVolume(udf_id, 'udf_node_id', 1, 200, u'~/UDF')
        root_id = uuid.uuid4()
        root_volume = volumes.RootVolume(root_id, 1, 500)
        response = [share_volume, udf_volume, root_volume]

        # patch fake action queue
        self.main.action_q.query_volumes = lambda: defer.succeed(response)
        server_rescan_d = defer.Deferred()
        self._listen_for('SYS_SERVER_RESCAN_DONE', server_rescan_d.callback)
        with environ('HOME', self.home_dir):
            yield self.vm.server_rescan()
        yield server_rescan_d
        self.assertIn(request.ROOT, self.vm.shares)
        self.assertIn(str(share_volume.volume_id), self.vm.shares)
        self.assertEquals(1, len(self.vm.udfs))
        self.assertEquals(2, len(self.vm.shares))
        # remove the udf from the response list
        response = [share_volume, root_volume]
        server_rescan_d = defer.Deferred()
        self._listen_for('SYS_SERVER_RESCAN_DONE', server_rescan_d.callback)
        with environ('HOME', self.home_dir):
            yield self.vm.server_rescan()
        yield server_rescan_d
        self.assertIn(request.ROOT, self.vm.shares)
        self.assertIn(str(share_volume.volume_id), self.vm.shares)
        self.assertEquals(0, len(self.vm.udfs))
        self.assertEquals(2, len(self.vm.shares))

    @defer.inlineCallbacks
    def test_server_rescan_clean_dead_shares(self):
        """Test cleanup of dead volumes after server_rescan method."""
        share_id = uuid.uuid4()
        share_volume = volumes.ShareVolume(share_id, 'fake_share_uuid', 1, 100,
                                           'to_me', 'fake_share', 'username',
                                           'visible_username', True, 'View')
        udf_id = uuid.uuid4()
        udf_volume = volumes.UDFVolume(udf_id, 'udf_node_id', 1, 200, u'~/UDF')
        root_id = uuid.uuid4()
        root_volume = volumes.RootVolume(root_id, 1, 500)
        response = [share_volume, udf_volume, root_volume]
        # patch fake action queue
        self.main.action_q.query_volumes = lambda: defer.succeed(response)
        server_rescan_d = defer.Deferred()
        self._listen_for('SYS_SERVER_RESCAN_DONE', server_rescan_d.callback)
        with environ('HOME', self.home_dir):
            yield self.vm.server_rescan()
        yield server_rescan_d
        self.assertIn(request.ROOT, self.vm.shares)
        self.assertIn(str(udf_volume.volume_id), self.vm.udfs)
        self.assertEquals(1, len(self.vm.udfs))
        self.assertEquals(2, len(self.vm.shares))
        # remove the share from the response list
        response = [udf_volume, root_volume]
        server_rescan_d = defer.Deferred()
        self._listen_for('SYS_SERVER_RESCAN_DONE', server_rescan_d.callback)
        with environ('HOME', self.home_dir):
            yield self.vm.server_rescan()
        yield server_rescan_d
        self.assertIn(request.ROOT, self.vm.shares)
        self.assertIn(str(udf_volume.volume_id), self.vm.udfs)
        self.assertEquals(1, len(self.vm.udfs))
        self.assertEquals(1, len(self.vm.shares))

    @defer.inlineCallbacks
    def test_volumes_rescan_cb(self):
        """Test for _volumes_rescan_cb."""
        share_id = uuid.uuid4()
        share_volume = volumes.ShareVolume(share_id, 'fake_share_uuid', 1, 100,
                                           'to_me', 'fake_share', 'username',
                                           'visible_username', True, 'View')
        udf_id = uuid.uuid4()
        udf_volume = volumes.UDFVolume(udf_id, 'udf_node_id', 1, 200, u'~/UDF')
        root_id = uuid.uuid4()
        root_volume = volumes.RootVolume(root_id, 1, 500)
        response = [share_volume, udf_volume, root_volume]
        d = defer.Deferred()
        self._listen_for('SV_VOLUME_NEW_GENERATION', d.callback, 2, True)
        udf_d = defer.Deferred()
        self._listen_for('VM_UDF_CREATED', udf_d.callback)
        with environ('HOME', self.home_dir):
            shares, udfs = yield self.vm._volumes_rescan_cb(response)
        events = yield d
        # check the returned shares and udfs
        self.assertEqual(shares, [str(share_id), ''])
        self.assertEqual(udfs, [str(udf_id)])
        # wait for the UDF local and server rescan
        yield udf_d
        events_dict = dict((event[1]['volume_id'], event[1]['generation']) \
                           for event in events)
        self.assertIn(str(share_id), events_dict)
        # new udfs server rescan is triggered after local rescan.
        self.assertNotIn(str(udf_id), events_dict)
        self.assertIn(request.ROOT, events_dict)
        self.assertEqual(1, events_dict[str(share_id)])
        self.assertEqual(1, events_dict[request.ROOT])
        # set the local metadata generation to new value
        share = self.vm.shares[str(share_id)]
        share.generation = share_volume.generation
        self.vm.shares[str(share_id)] = share
        udf = self.vm.udfs[str(udf_id)]
        udf.generation = udf_volume.generation
        self.vm.udfs[str(udf_id)] = udf
        root = self.vm.root
        root.generation = root_volume.generation
        self.vm.shares[request.ROOT] = root
        # now that we have the volumes in metadata, try with a higher  value.
        share_volume.generation = 10
        udf_volume.generation = 5
        root_volume.generation = 1
        response = [share_volume, udf_volume, root_volume]
        d = defer.Deferred()
        self._listen_for('SV_VOLUME_NEW_GENERATION', d.callback, 2, True)
        with environ('HOME', self.home_dir):
            self.vm._volumes_rescan_cb(response)
        events = yield d
        events_dict = dict((event[1]['volume_id'], event[1]['generation']) \
                           for event in events)
        self.assertIn(str(share_id), events_dict)
        self.assertIn(str(udf_id), events_dict)
        self.assertNotIn(request.ROOT, events_dict) # same gen as metadata
        self.assertEqual(10, events_dict[str(share_id)])
        self.assertEqual(5, events_dict[str(udf_id)])
        # now only change the root volume generation
        share_volume.generation = 1
        udf_volume.generation = 1
        root_volume.generation = 100
        response = [share_volume, udf_volume, root_volume]
        d = defer.Deferred()
        self._listen_for('SV_VOLUME_NEW_GENERATION', d.callback, 1, True)
        with environ('HOME', self.home_dir):
            self.vm._volumes_rescan_cb(response)
        events = yield d
        events_dict = dict((event[1]['volume_id'], event[1]['generation']) \
                           for event in events)
        self.assertNotIn(str(share_id), events_dict)
        self.assertNotIn(str(udf_id), events_dict)
        self.assertIn(request.ROOT, events_dict) # same generation as metadata
        self.assertEqual(100, events_dict[request.ROOT])

    @defer.inlineCallbacks
    def test_volumes_rescan_cb_handle_root_node_id(self):
        """Test _volumes_rescan_cb handling of the root node id."""
        root_id = uuid.uuid4()
        root_volume = volumes.RootVolume(root_id, 1, 500)
        response = [root_volume]
        self.assertEquals(None, self.vm.root.node_id)
        d = defer.Deferred()
        self._listen_for('SV_VOLUME_NEW_GENERATION', d.callback, 1, True)
        with environ('HOME', self.home_dir):
            self.vm._volumes_rescan_cb(response)
        events = yield d
        events_dict = dict((event[1]['volume_id'], event[1]['generation']) \
                           for event in events)
        self.assertIn(request.ROOT, events_dict)
        self.assertEquals(str(root_id), self.vm.root.node_id)
        self.assertTrue(self.main.fs.get_by_node_id(request.ROOT,
                                                    str(root_id)))

    @defer.inlineCallbacks
    def test_volumes_rescan_cb_root_node_id_not_in_fsm(self):
        """Test _volumes_rescan_cb with the root node id missing from fsm."""
        root_id = uuid.uuid4()
        root_volume = volumes.RootVolume(root_id, 1, 500)
        response = [root_volume]
        # set the node_id
        root = self.vm.root
        root.node_id = str(root_id)
        self.vm.shares[request.ROOT] = root
        self.assertEquals(str(root_id), self.vm.root.node_id)
        d = defer.Deferred()
        self._listen_for('SV_VOLUME_NEW_GENERATION', d.callback, 1, True)
        with environ('HOME', self.home_dir):
            self.vm._volumes_rescan_cb(response)
        events = yield d
        events_dict = dict((event[1]['volume_id'], event[1]['generation']) \
                           for event in events)
        self.assertIn(request.ROOT, events_dict)
        self.assertEquals(str(root_id), self.vm.root.node_id)
        self.assertTrue(self.main.fs.get_by_node_id(request.ROOT,
                                                    str(root_id)))

    @defer.inlineCallbacks
    def test_volumes_rescan_cb_inactive_volume(self):
        """Test _volumes_rescan_cb with inactive volume."""
        udf_id = uuid.uuid4()
        udf, udf_volume = self._create_udf(udf_id, uuid.uuid4(), '~/UDF')
        udf_volume.generation = 10
        root_id = uuid.uuid4()
        root_volume = volumes.RootVolume(root_id, 1, 500)
        response = [udf_volume, root_volume]
        yield self.vm.add_udf(udf)
        # unsubscribe the udf
        self.vm.unsubscribe_udf(udf.volume_id)
        d = defer.Deferred()
        self._listen_for('SV_VOLUME_NEW_GENERATION', d.callback, 1)
        with environ('HOME', self.home_dir):
            self.vm._volumes_rescan_cb(response)
        event = yield d
        self.assertEqual(len(event), 2)
        events_dict = {event[1]['volume_id']:event[1]['generation']}
        self.assertNotIn(udf.volume_id, events_dict)
        self.assertIn(request.ROOT, events_dict)

    @defer.inlineCallbacks
    def test_volumes_rescan_cb_missing_fsm_md(self):
        """Test _volumes_rescan_cb with a missing fsm node."""
        udf_id = uuid.uuid4()
        udf, udf_volume = self._create_udf(udf_id, uuid.uuid4(), '~/UDF')
        udf_volume.generation = 10
        root_id = uuid.uuid4()
        root_volume = volumes.RootVolume(root_id, 1, 500)
        response = [udf_volume, root_volume]
        yield self.vm.add_udf(udf)
        # delete the fsm metadata
        self.main.fs.delete_metadata(udf.path)
        d = defer.Deferred()
        self._listen_for('SV_VOLUME_NEW_GENERATION', d.callback, 2, collect=True)
        with environ('HOME', self.home_dir):
            self.vm._volumes_rescan_cb(response)
        events = yield d
        self.assertEqual(len(events), 2)
        events_dict = dict((evt[1]['volume_id'], evt[1]['generation']) \
                           for evt in events)
        self.assertIn(udf.volume_id, events_dict)
        self.assertIn(request.ROOT, events_dict)
        # check that the fsm metadata is there
        mdobj = self.main.fs.get_by_path(udf.path)
        self.assertEquals(udf.node_id, mdobj.node_id)
        self.assertEquals(udf.id, mdobj.share_id)
        self.assertEquals(udf.path, mdobj.path)

    @defer.inlineCallbacks
    def test_update_generation(self):
        """Test for the update_generation method."""
        share_id = uuid.uuid4()
        share_volume = volumes.ShareVolume(share_id, 'fake_share_uuid', None,
                                           10, 'to_me', 'fake_share', 'username',
                                           'visible_username', True, 'View')
        dir_name = self.vm._build_share_path(share_volume.share_name,
                                          share_volume.other_visible_name)
        share_path = os.path.join(self.main.shares_dir, dir_name)
        share = Share.from_share_volume(share_volume, share_path)
        # get the root
        root = self.vm.get_volume(request.ROOT)
        self.vm._got_root('root_node_id')
        # create a UDF
        udf_id = uuid.uuid4()
        udf, _ = self._create_udf(udf_id, 'udf_node_id', '~/UDF')
        self.vm.add_share(share)
        yield self.vm.add_udf(udf)
        self.assertEqual(None, udf.generation)
        self.assertEqual(None, share.generation)
        self.assertEqual(None, root.generation)
        self.vm.update_generation(udf.volume_id, 1)
        self.vm.update_generation(share.volume_id, 2)
        self.vm.update_generation(root.volume_id, 3)
        udf = self.vm.get_volume(udf.volume_id)
        share = self.vm.get_volume(share.volume_id)
        root = self.vm.root
        self.assertEqual(1, udf.generation)
        self.assertEqual(2, share.generation)
        self.assertEqual(3, root.generation)
        # try to update the generation of non-existing volume_id
        self.assertRaises(VolumeDoesNotExist, self.vm.update_generation,
                          str(uuid.uuid4()), 1)

    @defer.inlineCallbacks
    def test_handle_SV_VOLUME_NEW_GENERATION_udf(self):
        """Test handle_SV_VOLUME_NEW_GENERATION for udf."""
        # create a UDF
        udf_id = uuid.uuid4()
        udf, udf_volume = self._create_udf(udf_id, 'udf_node_id', '~/UDF')
        yield self.vm.add_udf(udf)
        self.vm.update_generation(udf.volume_id, 10)
        d = defer.Deferred()
        self.patch(self.main.action_q, 'get_delta', lambda v, g: d.callback((v, g)))
        self.main.event_q.push('SV_VOLUME_NEW_GENERATION', udf.volume_id, 100)
        vol_id, gen = yield d
        vol = self.vm.get_volume(vol_id)
        self.assertEqual(vol_id, vol.volume_id)
        self.assertEqual(gen, vol.generation)

    @defer.inlineCallbacks
    def test_handle_SV_VOLUME_NEW_GENERATION_udf_from_scratch(self):
        """Test handle_SV_VOLUME_NEW_GENERATION for udf."""
        # create a UDF
        udf_id = uuid.uuid4()
        udf, udf_volume = self._create_udf(udf_id, 'udf_node_id', '~/UDF')
        yield self.vm.add_udf(udf)
        self.vm.update_generation(udf.volume_id, None)
        d = defer.Deferred()
        self.patch(self.main.action_q, 'rescan_from_scratch', d.callback)
        self.main.event_q.push('SV_VOLUME_NEW_GENERATION', udf.volume_id, 100)
        vol_id = yield d
        self.assertEquals(vol_id, udf.volume_id)

    @defer.inlineCallbacks
    def test_handle_SV_VOLUME_NEW_GENERATION_udf_eq(self):
        """Test handle_SV_VOLUME_NEW_GENERATION for udf."""
        # get the root
        udf_id = uuid.uuid4()
        udf, udf_volume = self._create_udf(udf_id, 'udf_node_id', '~/UDF')
        yield self.vm.add_udf(udf)
        self.vm.update_generation(udf.volume_id, 100)
        self.main.event_q.push('SV_VOLUME_NEW_GENERATION', udf.volume_id, 100)
        self.assertEqual(1, len(self.handler.records))
        msg = 'Got SV_VOLUME_NEW_GENERATION(%r, %r) but volume' + \
                ' is at generation: %r'
        self.assertEqual(msg % (udf.volume_id, 100, 100),
                         self.handler.records[0].message)

    @defer.inlineCallbacks
    def test_handle_SV_VOLUME_NEW_GENERATION_root(self):
        """Test handle_SV_VOLUME_NEW_GENERATION for root share."""
        # get the root
        root = self.vm.get_volume(request.ROOT)
        self.vm._got_root('root_node_id')
        self.vm.update_generation(root.volume_id, 10)
        d = defer.Deferred()
        self.patch(self.main.action_q, 'get_delta', lambda v, g: d.callback((v, g)))
        self.main.event_q.push('SV_VOLUME_NEW_GENERATION', root.volume_id, 100)
        vol_id, gen = yield d
        vol = self.vm.get_volume(vol_id)
        self.assertEqual(vol_id, vol.volume_id)
        self.assertEqual(gen, vol.generation)

    @defer.inlineCallbacks
    def test_handle_SV_VOLUME_NEW_GENERATION_root_from_scratch(self):
        """Test handle_SV_VOLUME_NEW_GENERATION for root share."""
        # get the root
        root = self.vm.get_volume(request.ROOT)
        self.vm._got_root('root_node_id')
        self.vm.update_generation(root.volume_id, None)
        d = defer.Deferred()
        self.patch(self.main.action_q, 'rescan_from_scratch', d.callback)
        self.main.event_q.push('SV_VOLUME_NEW_GENERATION', root.volume_id, 100)
        vol_id = yield d
        self.assertEquals(vol_id, root.volume_id)

    def test_handle_SV_VOLUME_NEW_GENERATION_root_eq(self):
        """Test handle_SV_VOLUME_NEW_GENERATION for root share."""
        # get the root
        root = self.vm.get_volume(request.ROOT)
        self.vm._got_root('root_node_id')
        self.vm.update_generation(root.volume_id, 100)
        self.main.event_q.push('SV_VOLUME_NEW_GENERATION', root.volume_id, 100)
        self.assertEqual(1, len(self.handler.records))
        msg = 'Got SV_VOLUME_NEW_GENERATION(%r, %r) but volume' + \
                ' is at generation: %r'
        self.assertEqual(msg % (root.volume_id, 100, 100),
                         self.handler.records[0].message)

    @defer.inlineCallbacks
    def test_handle_SV_VOLUME_NEW_GENERATION_share(self):
        """Test handle_SV_VOLUME_NEW_GENERATION for a share."""
        share_id = uuid.uuid4()
        share_volume = volumes.ShareVolume(share_id, 'fake_share_uuid', None,
                                           10, 'to_me', 'fake_share', 'username',
                                           'visible_username', True, 'View')
        dir_name = self.vm._build_share_path(share_volume.share_name,
                                          share_volume.other_visible_name)
        share_path = os.path.join(self.main.shares_dir, dir_name)
        share = Share.from_share_volume(share_volume, share_path)
        self.vm.add_share(share)
        self.vm.update_generation(share.volume_id, 10)
        d = defer.Deferred()
        self.patch(self.main.action_q, 'get_delta', lambda v, g: d.callback((v, g)))
        self.main.event_q.push('SV_VOLUME_NEW_GENERATION', share.volume_id, 100)
        vol_id, gen = yield d
        vol = self.vm.get_volume(vol_id)
        self.assertEqual(vol_id, vol.volume_id)
        self.assertEqual(gen, vol.generation)

    @defer.inlineCallbacks
    def test_handle_SV_VOLUME_NEW_GENERATION_share_from_scratch(self):
        """Test handle_SV_VOLUME_NEW_GENERATION for a share."""
        share_id = uuid.uuid4()
        share_volume = volumes.ShareVolume(share_id, 'fake_share_uuid', None,
                                           10, 'to_me', 'fake_share', 'username',
                                           'visible_username', True, 'View')
        dir_name = self.vm._build_share_path(share_volume.share_name,
                                          share_volume.other_visible_name)
        share_path = os.path.join(self.main.shares_dir, dir_name)
        share = Share.from_share_volume(share_volume, share_path)
        self.vm.add_share(share)
        self.vm.update_generation(share.volume_id, None)
        d = defer.Deferred()
        self.patch(self.main.action_q, 'rescan_from_scratch', d.callback)
        self.main.event_q.push('SV_VOLUME_NEW_GENERATION', share.volume_id, 100)
        vol_id = yield d
        self.assertEquals(vol_id, share.volume_id)

    def test_handle_SV_VOLUME_NEW_GENERATION_share_eq(self):
        """Test handle_SV_VOLUME_NEW_GENERATION for a share."""
        share_id = uuid.uuid4()
        share_volume = volumes.ShareVolume(share_id, 'fake_share_uuid', None,
                                           10, 'to_me', 'fake_share', 'username',
                                           'visible_username', True, 'View')
        dir_name = self.vm._build_share_path(share_volume.share_name,
                                          share_volume.other_visible_name)
        share_path = os.path.join(self.main.shares_dir, dir_name)
        share = Share.from_share_volume(share_volume, share_path)
        self.vm.add_share(share)
        self.vm.update_generation(share.volume_id, 100)
        self.main.event_q.push('SV_VOLUME_NEW_GENERATION', share.volume_id, 100)
        self.assertTrue(self.handler.check_info(
                        'Got SV_VOLUME_NEW_GENERATION(%r, %r) but volume '
                        'is at generation: %r' % (share.volume_id, 100, 100)))

    def test_handle_SV_VOLUME_NEW_GENERATION_no_volume(self):
        """Test handle_SV_VOLUME_NEW_GENERATION for a volume we don't have."""
        called = []
        self.vm.refresh_volumes = lambda: called.append(True)
        self.vm.handle_SV_VOLUME_NEW_GENERATION('unknown vol id', 123)
        self.assertTrue(called)
        self.assertTrue(self.handler.check_warning("missing volume"))

    @defer.inlineCallbacks
    def test_handle_AQ_DELTA_NOT_POSSIBLE(self):
        """Test for handle_AQ_DELTA_NOT_POSSIBLE."""
        share_id = uuid.uuid4()
        share_volume = volumes.ShareVolume(share_id, 'fake_share_uuid', None,
                                           10, 'to_me', 'fake_share', 'username',
                                           'visible_username', True, 'View')
        dir_name = self.vm._build_share_path(share_volume.share_name,
                                          share_volume.other_visible_name)
        share_path = os.path.join(self.main.shares_dir, dir_name)
        share = Share.from_share_volume(share_volume, share_path)
        # get the root
        root = self.vm.get_volume(request.ROOT)
        self.vm._got_root('root_node_id')
        # create a UDF
        udf_id = uuid.uuid4()
        udf, _ = self._create_udf(udf_id, 'udf_node_id', '~/UDF')
        self.vm.add_share(share)
        yield self.vm.add_udf(udf)
        # patch AQ.rescan_from_scratch
        calls = []
        self.patch(self.main.action_q, 'rescan_from_scratch', calls.append)
        self.vm.handle_AQ_DELTA_NOT_POSSIBLE(udf_id)
        self.vm.handle_AQ_DELTA_NOT_POSSIBLE(share_id)
        self.vm.handle_AQ_DELTA_NOT_POSSIBLE(root.volume_id)
        for i, vol in enumerate([udf, share, root]):
            self.assertEquals(calls[i], str(vol.volume_id))

    def test_handle_AQ_DELTA_NOT_POSSIBLE_missing_volume(self):
        """Test for handle_AQ_DELTA_NOT_POSSIBLE with an missing volume."""
        called = []
        self.vm.refresh_volumes = lambda: called.append(True)
        self.vm.handle_AQ_DELTA_NOT_POSSIBLE(uuid.uuid4())
        self.assertTrue(called)
        self.assertTrue(self.handler.check_warning(
                        'Got a AQ_DELTA_NOT_POSSIBLE for a missing volume'))

    @defer.inlineCallbacks
    def test_handle_AQ_SHARES_LIST_shared_in_UDF(self):
        """Test the handling of the AQ_SHARE_LIST event.

        This tests the case of a receiving shared directory inside a UDF.
        """
        udf_id = uuid.uuid4()
        udf, _ = self._create_udf(udf_id, str(uuid.uuid4()), '~/UDF')
        yield self.vm.add_udf(udf)
        share_id = uuid.uuid4()
        share_response = ShareResponse.from_params(
            share_id, 'from_me', uuid.UUID(udf.node_id), 'fake_share',
            'username', 'visible_username', True, 'View',
            uuid.UUID(udf.volume_id))
        # initialize the the root
        self.vm._got_root('root_uuid')
        response = ListShares(None)
        response.shares = [share_response]
        self.vm.handle_AQ_SHARES_LIST(response)
        self.assertEquals(1, len(self.vm.shared)) # the new shares and root
        # check that the share is in the shares dict
        self.assertIn(str(share_id), self.vm.shared)
        shared = self.vm.shared[str(share_id)]
        # check that path is correctly set
        self.assertEqual(udf.path, shared.path)
        self.assertEquals('fake_share', shared.name)
        self.assertEquals(udf.node_id, shared.node_id)

class MetadataTestCase(BaseTwistedTestCase):
    md_version_None = False
    main = None
    data_dir = None
    share_md_dir = None
    shared_md_dir = None
    partials_dir = None
    u1_dir = None
    root_dir = None
    shares_dir = None
    shares_dir_link = None

    def setUp(self):
        """Create some directories."""
        BaseTwistedTestCase.setUp(self)
        self.data_dir = os.path.join(self.tmpdir, 'data_dir')
        self.vm_data_dir = os.path.join(self.tmpdir, 'data_dir', 'vm')
        self.partials_dir = self.mktemp('partials')
        self.u1_dir = os.path.join(self.tmpdir, 'Ubuntu One')
        self.version_file = os.path.join(self.vm_data_dir, '.version')

    def tearDown(self):
        """Cleanup all the cruft."""
        for path in [self.data_dir, self.partials_dir, self.root_dir,
                     self.shares_dir]:
            if path and os.path.exists(path):
                self.rmtree(path)
        if self.main:
            self.main.shutdown()
        VolumeManager.METADATA_VERSION = CURRENT_METADATA_VERSION
        BaseTwistedTestCase.tearDown(self)

    def check_version(self):
        """Check if the current version in the .version file is the last one."""
        with open(self.version_file, 'r') as fd:
            self.assertEquals(CURRENT_METADATA_VERSION, fd.read().strip())

    def set_md_version(self, md_version):
        """Write md_version to the .version file."""
        if not os.path.exists(self.vm_data_dir):
            os.makedirs(self.vm_data_dir)
        with open(self.version_file, 'w') as fd:
            fd.write(md_version)


class MetadataOldLayoutTests(MetadataTestCase):
    """Tests for 'old' layouts and metadata upgrade"""

    def setUp(self):
        MetadataTestCase.setUp(self)
        self.root_dir = os.path.join(self.u1_dir, 'My Files')
        self.shares_dir = os.path.join(self.u1_dir, 'Shared With Me')
        self.new_root_dir = self.u1_dir
        self.new_shares_dir = self.mktemp('shares_dir')

    def tearDown(self):
        """Cleanup all the cruft."""
        for path in [self.u1_dir, self.new_shares_dir]:
            if path and os.path.exists(path):
                self.rmtree(path)
        MetadataTestCase.tearDown(self)

    def _build_layout_version_0(self):
        """Build the dir structure to mimic md v.0/None."""
        self.share_md_dir = os.path.join(self.tmpdir, 'data_dir', 'vm')
        os.makedirs(self.share_md_dir)
        os.makedirs(self.root_dir)
        os.makedirs(self.shares_dir)

    def _build_layout_version_1(self):
        """Build the dir structure to mimic md v.1"""
        self.share_md_dir = os.path.join(self.vm_data_dir, 'shares')
        self.shared_md_dir = os.path.join(self.vm_data_dir, 'shared')
        os.makedirs(self.share_md_dir)
        os.makedirs(self.shared_md_dir)
        os.makedirs(self.root_dir)
        os.makedirs(self.shares_dir)

    def _set_permissions(self):
        """Set the RO perms in the root and the shares directory."""
        os.chmod(self.shares_dir, 0500)
        os.chmod(self.u1_dir, 0500)

    def test_upgrade_0(self):
        """Test the upgrade from the first shelf layout version."""
        self._build_layout_version_0()
        old_shelf = LegacyShareFileShelf(self.share_md_dir)
        # add the root_uuid key
        root_share = _Share(path=self.root_dir)
        root_share.access_level = 'Modify'
        old_shelf[request.ROOT] = root_share
        for idx in range(1, 10):
            sid = str(uuid.uuid4())
            old_shelf[sid] =  _Share(path=os.path.join(self.shares_dir, str(idx)),
                                     share_id=sid)
        # ShareFileShelf.keys returns a generator
        old_keys = [key for key in old_shelf.keys()]
        self.assertEquals(10, len(old_keys))
        if self.md_version_None:
            self.set_md_version('')
        # set the ro permissions
        self._set_permissions()
        # we want to keep a refernece to main in order to shutdown
        # pylint: disable-msg=W0201
        self.main = FakeMain(self.new_root_dir, self.new_shares_dir,
                             self.data_dir, self.partials_dir)
        new_keys = [new_key for new_key in self.main.vm.shares.keys()]
        self.assertEquals(10, len(new_keys))
        for new_key in new_keys:
            self.assertIn(new_key, old_keys)
        # check the old data is still there (in the backup)
        bkp_dir = os.path.join(os.path.dirname(self.vm_data_dir), '5.bkp', '0.bkp')
        backup_shelf = LegacyShareFileShelf(bkp_dir)
        backup_keys = [key for key in backup_shelf.keys()]
        for old_key in old_keys:
            self.assertIn(old_key, backup_keys)
        for new_key in new_keys:
            self.assertIn(new_key, backup_keys)
        self.check_version()

    def test_upgrade_1(self):
        """ Test the upgrade from v.1"""
        self._build_layout_version_1()
        # write the .version file with v.1
        self.set_md_version('1')

        share_file = os.path.join(self.share_md_dir,
                                  '0/6/6/0664f050-9254-45c5-9f31-3482858709e4')
        os.makedirs(os.path.dirname(share_file))
        # this is the str of a version 2 pickle
        share_value = "\x80\x02ccanonical.ubuntuone.storage.syncdaemon." + \
                "volume_manager\nShare\nq\x01)\x81q\x02}q\x03(U\x04nameq" + \
                "\x04U\tfakeshareq\x05U\x0eother_usernameq\x06U\x08fakeu" + \
                "serq\x07U\x07subtreeq\x08U$beb0c48c-6755-4fbd-938f-3d20" + \
                "fa7b102bq\tU\x12other_visible_nameq\nU\tfake userq\x0bU" + \
                "\x0caccess_levelq\x0cU\x04Viewq\rU\x04pathq\x0eU=/home/" + \
                "auser/Ubuntu One/Shared With Me/fakeshare from fakeuser" + \
                "q\x0fU\x08acceptedq\x10\x88U\x02idq\x11U$0664f050-9254-" + \
                "45c5-9f31-3482858709e4q\x12ub."
        with open(share_file, 'w') as fd:
            fd.write(share_value)

        # try to load the shelf
        old_shelf = LegacyShareFileShelf(self.share_md_dir)
        share = old_shelf['0664f050-9254-45c5-9f31-3482858709e4']
        self.assertTrue(share is not None)
        if self.md_version_None:
            self.set_md_version('')

        self._set_permissions()
        # now use the real VolumeManager
        # we want to keep a refernece to main in order to shutdown
        # pylint: disable-msg=W0201
        self.main = FakeMain(self.new_root_dir, self.new_shares_dir,
                             self.data_dir, self.partials_dir)
        new_keys = [new_key for new_key in self.main.vm.shares.keys()]
        self.assertEquals(2, len(new_keys)) # the fake share plus root
        for key in [request.ROOT, share.id]:
            self.assertIn(key, new_keys)
        self.check_version()

    def test_upgrade_2(self):
        """Test the upgrade from v.2."""
        self._build_layout_version_1()
        self.set_md_version('2')
        open(self.root_dir + '/foo.conflict', 'w').close()
        open(self.root_dir + '/foo.conflict.23', 'w').close()
        open(self.shares_dir + '/bar.partial', 'w').close()
        os.mkdir(self.shares_dir + '/baz/')
        open(self.shares_dir + '/baz/baz.conflict', 'w').close()
        os.chmod(self.shares_dir + '/baz/', 0500)
        if self.md_version_None:
            self.set_md_version('')
        self._set_permissions()
        self.main = FakeMain(self.new_root_dir, self.new_shares_dir,
                             self.data_dir, self.partials_dir)
        self.assertTrue(os.path.exists(self.new_root_dir + '/foo.u1conflict'))
        self.assertTrue(os.path.exists(self.new_root_dir + '/foo.u1conflict.23'))
        self.assertTrue(os.path.exists(self.new_shares_dir + '/.u1partial.bar'))
        self.assertTrue(os.path.exists(self.new_shares_dir + '/baz/baz.u1conflict'))
        self.check_version()

    def test_upgrade_2_more(self):
        """Test the upgrade from v.2 some more."""
        self._build_layout_version_1()
        self.set_md_version('2')

        expected = []

        for dirname, new_dirname in [(self.root_dir, self.new_root_dir),
                                          (self.shares_dir, self.new_shares_dir)]:
            # a plain .conflict...
            # ...on a file
            open(dirname + '/1a.conflict', 'w').close()
            expected.append(new_dirname + '/1a.u1conflict')
            # ...on an empty directory
            os.mkdir(dirname + '/1b.conflict')
            expected.append(new_dirname + '/1b.u1conflict')
            # ...on a directory with content
            os.mkdir(dirname + '/1c.conflict')
            os.mkdir(dirname + '/1c.conflict/1c')
            expected.append(new_dirname + '/1c.u1conflict/1c')
            # ...in a readonly directory
            os.mkdir(dirname + '/1d')
            os.mkdir(dirname + '/1d/1d.conflict')
            os.chmod(dirname + '/1d', 0500)
            expected.append(new_dirname + '/1d/1d.u1conflict')
            # ...in a directory that is also a .conflict
            os.mkdir(dirname + '/1e.conflict')
            os.mkdir(dirname + '/1e.conflict/1e.conflict')
            expected.append(new_dirname + '/1e.u1conflict/1e.u1conflict')

            # a numbered .conflict...
            # ...on a file
            open(dirname + '/2a.conflict.2', 'w').close()
            expected.append(new_dirname + '/2a.u1conflict.2')
            # ...on an empty directory
            os.mkdir(dirname + '/2b.conflict.3')
            expected.append(new_dirname + '/2b.u1conflict.3')
            # ...on a directory with content
            os.mkdir(dirname + '/2c.conflict.4')
            os.mkdir(dirname + '/2c.conflict.4/2c')
            expected.append(new_dirname + '/2c.u1conflict.4/2c')
            # ...in a readonly directory
            os.mkdir(dirname + '/2d')
            os.mkdir(dirname + '/2d/2d.conflict.5')
            os.chmod(dirname + '/2d', 0500)
            expected.append(new_dirname + '/2d/2d.u1conflict.5')
            # ...in a directory that is also a .conflict
            os.mkdir(dirname + '/2e.conflict')
            os.mkdir(dirname + '/2e.conflict/2e.conflict.6')
            expected.append(new_dirname + '/2e.u1conflict/2e.u1conflict.6')

            # a plain .conflict of which there already exists a .u1conflict...
            # ...on a file
            open(dirname + '/3a.conflict', 'w').close()
            open(dirname + '/3a.u1conflict', 'w').close()
            expected.append(new_dirname + '/3a.u1conflict')
            expected.append(new_dirname + '/3a.u1conflict.1')
            # ...on an empty directory
            os.mkdir(dirname + '/3b.conflict')
            os.mkdir(dirname + '/3b.u1conflict')
            expected.append(new_dirname + '/3b.u1conflict')
            expected.append(new_dirname + '/3b.u1conflict.1')
            # ...on a directory with content
            os.mkdir(dirname + '/3c.conflict')
            os.mkdir(dirname + '/3c.conflict/3c')
            os.mkdir(dirname + '/3c.u1conflict')
            os.mkdir(dirname + '/3c.u1conflict/3c2')
            expected.append(new_dirname + '/3c.u1conflict.1/3c')
            expected.append(new_dirname + '/3c.u1conflict/3c2')
            # ...in a readonly directory
            os.mkdir(dirname + '/3d')
            os.mkdir(dirname + '/3d/3d.conflict')
            os.mkdir(dirname + '/3d/3d.u1conflict')
            os.mkdir(dirname + '/3d/3d.u1conflict/3d')
            os.chmod(dirname + '/3d', 0500)
            expected.append(new_dirname + '/3d/3d.u1conflict/3d')
            expected.append(new_dirname + '/3d/3d.u1conflict.1')
            # ...in a directory that is also a .conflict
            os.mkdir(dirname + '/3e.conflict')
            os.mkdir(dirname + '/3e.conflict/3e.conflict')
            os.mkdir(dirname + '/3e.conflict/3e.u1conflict')
            os.mkdir(dirname + '/3e.conflict/3e.u1conflict/3e')
            expected.append(new_dirname + '/3e.u1conflict/3e.u1conflict/3e')
            expected.append(new_dirname + '/3e.u1conflict/3e.u1conflict.1')

            # a numbered .conflict of which there already exists a .u1conflict...
            # ...on a file
            open(dirname + '/4a.conflict.1', 'w').close()
            open(dirname + '/4a.u1conflict.1', 'w').close()
            expected.append(new_dirname + '/4a.u1conflict.1')
            expected.append(new_dirname + '/4a.u1conflict.2')
            # ...on an empty directory
            os.mkdir(dirname + '/4b.conflict.2')
            os.mkdir(dirname + '/4b.u1conflict.2')
            expected.append(new_dirname + '/4b.u1conflict.2')
            expected.append(new_dirname + '/4b.u1conflict.3')
            # ...on a directory with content
            os.mkdir(dirname + '/4c.conflict.3')
            os.mkdir(dirname + '/4c.conflict.3/4c')
            os.mkdir(dirname + '/4c.u1conflict.3')
            expected.append(new_dirname + '/4c.u1conflict.4/4c')
            expected.append(new_dirname + '/4c.u1conflict.3')
            # ...in a readonly directory
            os.mkdir(dirname + '/4d')
            os.mkdir(dirname + '/4d/4d.conflict.4')
            os.mkdir(dirname + '/4d/4d.u1conflict.4')
            os.chmod(dirname + '/4d', 0500)
            expected.append(new_dirname + '/4d/4d.u1conflict.4')
            expected.append(new_dirname + '/4d/4d.u1conflict.5')
            # ...in a directory that is also a .conflict
            os.mkdir(dirname + '/4e.conflict')
            os.mkdir(dirname + '/4e.conflict/4e.conflict.5')
            os.mkdir(dirname + '/4e.conflict/4e.u1conflict.5')
            expected.append(new_dirname + '/4e.u1conflict/4e.u1conflict.5')
            expected.append(new_dirname + '/4e.u1conflict/4e.u1conflict.6')

            # a plain .partial...
            # ...of a file
            open(dirname + '/5a.partial', 'w').close()
            expected.append(new_dirname + '/.u1partial.5a')
            # ...of a directory
            os.mkdir(dirname + '/5b')
            open(dirname + '/5b/.partial', 'w').close()
            expected.append(new_dirname + '/5b/.u1partial')
            # ...of a readonly directory
            os.mkdir(dirname + '/5c')
            open(dirname + '/5c/.partial', 'w').close()
            os.chmod(dirname + '/5c', 0500)
            expected.append(new_dirname + '/5c/.u1partial')

            # a plain .partial of which there already exists a .u1partial...
            # ...of a file
            open(dirname + '/6a.partial', 'w').close()
            open(dirname + '/.u1partial.6a', 'w').close()
            expected.append(new_dirname + '/.u1partial.6a')
            expected.append(new_dirname + '/.u1partial.6a.1')
            # ...of a directory
            os.mkdir(dirname + '/6b')
            open(dirname + '/6b/.partial', 'w').close()
            open(dirname + '/6b/.u1partial', 'w').close()
            expected.append(new_dirname + '/6b/.u1partial')
            expected.append(new_dirname + '/6b/.u1partial.1')
            # ...of a readonly directory
            os.mkdir(dirname + '/6c')
            open(dirname + '/6c/.partial', 'w').close()
            open(dirname + '/6c/.u1partial', 'w').close()
            os.chmod(dirname + '/6c', 0500)
            expected.append(new_dirname + '/6c/.u1partial')
            expected.append(new_dirname + '/6c/.u1partial.1')

        self._set_permissions()
        self.main = FakeMain(self.new_root_dir, self.new_shares_dir,
                             self.data_dir, self.partials_dir)

        for path in expected:
            self.assertTrue(os.path.exists(path), 'missing ' + path)
        self.check_version()

    def test_missing_version_file_with_version_non_0(self):
        """Test the upgrade from the first shelf layout version
        while the metadata sould be in v3 or greater format.

        """
        self._build_layout_version_1()
        maybe_old_shelf = LegacyShareFileShelf(self.share_md_dir)
        # add the root_uuid key
        root_share = _Share(path=self.root_dir)
        root_share.access_level = 'Modify'
        maybe_old_shelf[request.ROOT] = root_share
        for idx in range(1, 10):
            share_id = str(uuid.uuid4())
            maybe_old_shelf[share_id] = \
                    _Share(share_id=share_id,
                           path=os.path.join(self.shares_dir, str(idx)))
        # ShareFileShelf.keys returns a generator
        maybe_old_keys = [key for key in maybe_old_shelf.keys()]
        self.assertEquals(10, len(maybe_old_keys))
        if self.md_version_None:
            self.set_md_version('')
        # we want to keep a refernece to main in order to shutdown
        # pylint: disable-msg=W0201
        self.main = FakeMain(self.new_root_dir, self.new_shares_dir,
                             self.data_dir, self.partials_dir)
        new_keys = [new_key for new_key in self.main.vm.shares.keys()]
        self.assertEquals(10, len(new_keys))
        for new_key in new_keys:
            self.assertIn(new_key, maybe_old_keys)
        # as we didn't actually upgrade the shelf, just the .version file
        # check the empty 0.bkp
        # check the old data is still there (in the backup)
        backup_shelf = LegacyShareFileShelf(os.path.join(self.vm_data_dir, '0.bkp'))
        backup_keys = [key for key in backup_shelf.keys()]
        self.assertEquals(0, len(backup_keys))
        self.check_version()

    def test_upgrade_3(self):
        """Test upgrade from version 3."""
        self._build_layout_version_1()
        self.set_md_version('3')
        # create a dir in the root
        os.makedirs(os.path.join(self.root_dir, 'test_dir'))
        # create a file in the root
        open(os.path.join(self.root_dir, 'test_file'), 'w').close()
        # create a file in the new root
        open(os.path.join(self.new_root_dir, 'test_file'), 'w').close()
        share_path = os.path.join(self.shares_dir, 'Bla from Foo')
        os.makedirs(share_path)
        os.makedirs(os.path.join(share_path, 'test_dir'))
        open(os.path.join(share_path, 'test_file'), 'w').close()
        # fix permissions
        self._set_permissions()
        if self.md_version_None:
            self.set_md_version('')
        # migrate the data
        self.main = FakeMain(self.new_root_dir, self.new_shares_dir,
                             self.data_dir, self.partials_dir)
        self.assertFalse(os.path.exists(self.root_dir))
        self.assertTrue(os.path.exists(self.shares_dir))
        self.assertTrue(os.path.islink(self.shares_dir), self.shares_dir)
        self.assertEquals(self.shares_dir, self.main.shares_dir_link)
        self.assertTrue(os.path.exists(os.path.join(self.new_root_dir,
                                                    'test_dir')))
        self.assertTrue(os.path.exists(os.path.join(self.new_root_dir,
                                                    'test_file')))
        self.assertTrue(os.path.exists(os.path.join(self.new_root_dir,
                                                    'test_file.u1conflict')))
        self.assertTrue(os.path.exists(share_path))
        self.assertTrue(os.path.exists(os.path.join(share_path, 'test_dir')))
        self.assertTrue(os.path.exists(os.path.join(share_path, 'test_file')))
        self.check_version()

    def test_upgrade_3_with_symlink_in_myfiles(self):
        """Test upgrade from version 3 with symlink in 'My Files'."""
        self._build_layout_version_1()
        self.set_md_version('3')
        # build the old layout
        os.makedirs(os.path.join(self.root_dir, 'test_dir'))
        open(os.path.join(self.root_dir, 'test_file'), 'w').close()
        # create a file in the root
        open(os.path.join(self.new_root_dir, 'test_file'), 'w').close()
        share_path = os.path.join(self.shares_dir, 'Bla from Foo')
        os.makedirs(share_path)
        os.makedirs(os.path.join(share_path, 'test_dir'))
        open(os.path.join(share_path, 'test_file'), 'w').close()
        # create the Shared with Me symlink in My Files
        os.symlink(self.shares_dir, os.path.join(self.root_dir,
                                                 "Shared With Me"))
        # fix permissions
        self._set_permissions()
        if self.md_version_None:
            self.set_md_version('')
        # migrate the data
        self.main = FakeMain(self.new_root_dir, self.new_shares_dir,
                             self.data_dir, self.partials_dir)
        self.assertFalse(os.path.exists(self.root_dir))
        self.assertTrue(os.path.exists(self.shares_dir))
        self.assertTrue(os.path.islink(self.shares_dir))
        self.assertEquals(self.shares_dir, self.main.shares_dir_link)
        self.assertTrue(os.path.exists(os.path.join(self.new_root_dir,
                                                    'test_dir')))
        self.assertTrue(os.path.exists(os.path.join(self.new_root_dir,
                                                    'test_file')))
        self.assertTrue(os.path.exists(os.path.join(self.new_root_dir,
                                                    'test_file.u1conflict')))
        self.assertTrue(os.path.exists(share_path))
        self.assertTrue(os.path.exists(os.path.join(share_path, 'test_dir')))
        self.assertTrue(os.path.exists(os.path.join(share_path, 'test_file')))
        self.assertEquals(self.main.shares_dir,
                          os.readlink(self.main.shares_dir_link))
        self.check_version()


class MetadataNewLayoutTests(MetadataTestCase):
    """Test for 'new' layout and metadata upgrade."""

    def setUp(self):
        MetadataTestCase.setUp(self)
        self.share_md_dir = os.path.join(self.vm_data_dir, 'shares')
        self.shared_md_dir = os.path.join(self.vm_data_dir, 'shared')
        self.home_dir = os.path.join(self.tmpdir, 'home', 'ubuntuonehacker')
        self.u1_dir = os.path.join(self.home_dir, os.path.split(self.u1_dir)[1])
        self.root_dir = self.u1_dir
        self.shares_dir = os.path.join(self.tmpdir, 'shares')
        self.shares_dir_link = os.path.join(self.u1_dir, 'Shared With Me')

    def _build_layout_version_4(self):
        """Build the directory structure to mimic md v.4/5."""
        os.makedirs(self.share_md_dir)
        os.makedirs(self.shared_md_dir)
        os.makedirs(self.root_dir)
        os.makedirs(self.shares_dir)
        os.symlink(self.shares_dir, self.shares_dir_link)

    def _fix_permissions(self):
        """Fix shares dir permissions, making it read-only."""
        os.chmod(self.shares_dir, 0500)

    def test_upgrade_None_to_last(self):
        """Upgrade from version 'None' (possibly a clean start)."""
        old_root = os.path.join(self.root_dir, 'My Files')
        old_shares = os.path.join(self.root_dir, 'Shared With Me')
        # start and check that everything is ok
        self.main = FakeMain(self.root_dir, self.shares_dir,
                             self.data_dir, self.partials_dir)
        self.assertFalse(os.path.exists(old_root))
        self.assertTrue(os.path.exists(self.root_dir))
        self.assertTrue(os.path.exists(old_shares))
        self.assertTrue(os.path.islink(old_shares))
        self.assertEquals(old_shares, self.main.shares_dir_link)
        self.check_version()

    def test_upgrade_None_to_last_phantom_share_path(self):
        """Upgrade from version 'None' (possibly a clean start) with a root
        with missing path.

        """
        old_root = os.path.join(self.root_dir, 'My Files')
        old_shares = os.path.join(self.root_dir, 'Shared With Me')
        self.main = FakeMain(self.root_dir, self.shares_dir,
                             self.data_dir, self.partials_dir)
        self.main.shutdown()
        self.rmtree(self.vm_data_dir)
        os.makedirs(self.vm_data_dir)
        self.set_md_version('')
        shares = LegacyShareFileShelf(self.share_md_dir)
        root_share = _Share(path=self.root_dir)
        root_share.access_level = 'Modify'
        # set None to the share path
        root_share.path = None
        shares[request.ROOT] = root_share

        if self.md_version_None:
            self.set_md_version('')
        # check that it's all OK
        self.main = FakeMain(self.root_dir, self.shares_dir,
                             self.data_dir, self.partials_dir)
        self.assertFalse(os.path.exists(old_root))
        self.assertTrue(os.path.exists(self.root_dir))
        self.assertTrue(os.path.exists(self.shares_dir))
        self.assertTrue(os.path.islink(old_shares))
        self.assertEquals(old_shares, self.main.shares_dir_link)
        self.check_version()

    def test_upgrade_4(self):
        """Test migration from 4 to 5 (broken symlink in the root)."""
        self._build_layout_version_4()
        self.set_md_version('4')
        # break the symlink
        if os.path.exists(self.shares_dir_link):
            os.unlink(self.shares_dir_link)
        os.symlink(self.shares_dir_link, self.shares_dir_link)

        if self.md_version_None:
            self.set_md_version('')
        self.main = FakeMain(self.root_dir, self.shares_dir,
                             self.data_dir, self.partials_dir)
        self.assertEquals(self.main.shares_dir,
                          os.readlink(self.main.shares_dir_link))
        self.check_version()

    def test_upgrade_5(self):
        """Test the migration from version 5."""
        # build a fake version 5 state
        self._build_layout_version_4()
        self.set_md_version('5')
        # create some old shares and shared metadata
        legacy_shares = LegacyShareFileShelf(self.share_md_dir)
        root_share = _Share(path=self.root_dir, share_id=request.ROOT,
                            access_level='Modify')
        legacy_shares[request.ROOT] = root_share
        for idx, name in enumerate(['share'] * 10):
            sid = str(uuid.uuid4())
            share_name = name + '_' + str(idx)
            share = _Share(path=os.path.join(self.shares_dir, share_name),
                           share_id=sid, name=share_name,
                           node_id=str(uuid.uuid4()),
                           other_username='username'+str(idx),
                           other_visible_name='visible name ' + str(idx))
            if idx % 2:
                share.access_level = 'Modify'
            else:
                share.access_level = 'View'
            legacy_shares[sid] = share

        # create shared shares
        legacy_shared = LegacyShareFileShelf(self.shared_md_dir)
        for idx, name in enumerate(['dir'] * 5):
            sid = str(uuid.uuid4())
            share_name = name + '_' + str(idx)
            share = _Share(path=os.path.join(self.root_dir, share_name),
                           share_id=sid, node_id=str(uuid.uuid4()),
                           name=share_name, other_username='hola',
                           other_visible_name='hola')
            if idx % 2:
                share.access_level = 'Modify'
            else:
                share.access_level = 'View'
            legacy_shared[sid] = share

        # keep a copy of the current shares and shared metadata to check
        # the upgrade went ok
        legacy_shares = dict(legacy_shares.items())
        legacy_shared = dict(legacy_shared.items())

        if self.md_version_None:
            self.set_md_version('')
        # upgrade it!
        self.main = FakeMain(self.root_dir, self.shares_dir,
                             self.data_dir, self.partials_dir)
        vm = self.main.vm
        def compare_share(share, old_share):
            """Compare two shares, new and old"""
            self.assertEquals(share.volume_id, old_share.id)
            self.assertEquals(share.path, old_share.path)
            self.assertEquals(share.node_id, old_share.subtree)
            if not isinstance(share, Root):
                self.assertEquals(share.name, old_share.name)
                self.assertEquals(share.other_username, old_share.other_username)
                self.assertEquals(share.other_visible_name, old_share.other_visible_name)
                self.assertEquals(share.access_level, old_share.access_level)

        for sid in vm.shares:
            old_share = legacy_shares[sid]
            share = vm.shares[sid]
            self.assertTrue(isinstance(share, Share) or isinstance(share, Root))
            compare_share(share, old_share)

        for sid in vm.shared:
            old_share = legacy_shared[sid]
            share = vm.shared[sid]
            self.assertTrue(isinstance(share, Shared))
            compare_share(share, old_share)

    def test_upgrade_5_with_udfs(self):
        """Test the migration from version 5 with old UDFs."""
        # build a fake version 5 state
        self._build_layout_version_4()
        self.set_md_version('5')
        self.udfs_md_dir = os.path.join(self.vm_data_dir, 'udfs')
        # create some old shares and shared metadata
        legacy_shares = LegacyShareFileShelf(self.share_md_dir)
        root_share = _Share(path=self.root_dir, share_id=request.ROOT,
                            access_level='Modify')
        legacy_shares[request.ROOT] = root_share
        for idx, name in enumerate(['share'] * 10):
            sid = str(uuid.uuid4())
            share_name = name + '_' + str(idx)
            share = _Share(path=os.path.join(self.shares_dir, share_name),
                           share_id=sid, name=share_name,
                           node_id=str(uuid.uuid4()),
                           other_username='username'+str(idx),
                           other_visible_name='visible name ' + str(idx))
            if idx % 2:
                share.access_level = 'Modify'
            else:
                share.access_level = 'View'
            legacy_shares[sid] = share

        # create shared shares
        legacy_shared = LegacyShareFileShelf(self.shared_md_dir)
        for idx, name in enumerate(['dir'] * 5):
            sid = str(uuid.uuid4())
            share_name = name + '_' + str(idx)
            share = _Share(path=os.path.join(self.root_dir, share_name),
                           share_id=sid, node_id=str(uuid.uuid4()),
                           name=share_name, other_username='hola',
                           other_visible_name='hola')
            if idx % 2:
                share.access_level = 'Modify'
            else:
                share.access_level = 'View'
            legacy_shared[sid] = share

        # create some udfs
        legacy_udfs = LegacyShareFileShelf(self.udfs_md_dir)
        for idx, name in enumerate(['dir'] * 5):
            udf_id = str(uuid.uuid4())
            udf_name = name + '_' + str(idx)
            udf = _UDF(udf_id, str(uuid.uuid4()), u'~/' + udf_name.decode('utf-8'),
                         os.path.join(self.home_dir, udf_name))
            if idx % 2:
                udf.subscribed = True
            else:
                udf.subscribed = False
            legacy_udfs[sid] = udf

        # keep a copy of the current shares and shared metadata to check
        # the upgrade went ok
        legacy_shares = dict(legacy_shares.items())
        legacy_shared = dict(legacy_shared.items())
        legacy_udfs = dict(legacy_udfs.items())

        if self.md_version_None:
            self.set_md_version('')
        # upgrade it!
        self.main = FakeMain(self.root_dir, self.shares_dir,
                             self.data_dir, self.partials_dir)
        vm = self.main.vm
        def compare_share(share, old_share):
            """Compare two shares, new and old"""
            self.assertEquals(share.volume_id, old_share.id)
            self.assertEquals(share.path, old_share.path)
            self.assertEquals(share.node_id, old_share.subtree)
            if not isinstance(share, Root):
                self.assertEquals(share.name, old_share.name)
                self.assertEquals(share.other_username, old_share.other_username)
                self.assertEquals(share.other_visible_name, old_share.other_visible_name)
                self.assertEquals(share.access_level, old_share.access_level)

        for sid in vm.shares:
            old_share = legacy_shares[sid]
            share = vm.shares[sid]
            self.assertTrue(isinstance(share, Share) or isinstance(share, Root))
            compare_share(share, old_share)

        for sid in vm.shared:
            old_share = legacy_shared[sid]
            share = vm.shared[sid]
            self.assertTrue(isinstance(share, Shared))
            compare_share(share, old_share)

        for udf_id in vm.udfs:
            old_udf = legacy_udfs[udf_id]
            udf = vm.udfs[udf_id]
            self.assertTrue(isinstance(udf, UDF))
            self.assertEquals(udf.volume_id, old_udf.id)
            self.assertEquals(udf.path, old_udf.path)
            self.assertEquals(udf.node_id, old_udf.node_id)
            self.assertEquals(udf.suggested_path, old_udf.suggested_path)
            self.assertEquals(type(udf.suggested_path),
                              type(old_udf.suggested_path))
            self.assertEquals(udf.subscribed, old_udf.subscribed)

    def test_upgrade_5_partial_upgrade(self):
        """Test migration from version 5 with upgrade to 6 unfinished."""
        # build a fake version 5 state
        self._build_layout_version_4()
        self.set_md_version('5')
        self.udfs_md_dir = os.path.join(self.vm_data_dir, 'udfs')
        # create some old shares and shared metadata
        legacy_shares = LegacyShareFileShelf(self.share_md_dir)
        root_share = _Share(path=self.root_dir, share_id=request.ROOT,
                            access_level='Modify', node_id=str(uuid.uuid4()))
        legacy_shares[request.ROOT] = root_share
        for idx, name in enumerate(['share'] * 3):
            sid = str(uuid.uuid4())
            share_name = name + '_' + str(idx)
            share = _Share(path=os.path.join(self.shares_dir, share_name),
                           share_id=sid, name=share_name,
                           node_id=str(uuid.uuid4()),
                           other_username='username'+str(idx),
                           other_visible_name='visible name ' + str(idx))
            if idx == 0:
                share.access_level = 'Modify'
                legacy_shares[sid] = share
            elif idx == 1:
                share.access_level = 'View'
                legacy_shares[sid] = share
            else:
                # add a 'new' Share dict to the shelf
                share.access_level = 'Modify'
                share = Share(path=share.path,
                               volume_id=share.id, name=share.name,
                               access_level=share.access_level,
                               other_username=share.other_username,
                               other_visible_name=share.other_visible_name,
                               node_id=share.subtree)
                legacy_shares[sid] = share.__dict__

        # create shared shares
        legacy_shared = LegacyShareFileShelf(self.shared_md_dir)
        for idx, name in enumerate(['dir'] * 3):
            sid = str(uuid.uuid4())
            share_name = name + '_' + str(idx)
            share = _Share(path=os.path.join(self.root_dir, share_name),
                           share_id=sid, node_id=str(uuid.uuid4()),
                           name=share_name, other_username='hola',
                           other_visible_name='hola')
            if idx == 0:
                share.access_level = 'Modify'
                legacy_shares[sid] = share
            elif idx == 1:
                share.access_level = 'View'
                legacy_shares[sid] = share
            else:
                # add a 'new' Shared dict to the shelf
                share.access_level = 'Modify'
                share = Shared(path=share.path,
                               volume_id=share.id, name=share.name,
                               access_level=share.access_level,
                               other_username=share.other_username,
                               other_visible_name=share.other_visible_name,
                               node_id=share.subtree)
                legacy_shares[sid] = share.__dict__

        # keep a copy of the current shares and shared metadata to check
        # the upgrade went ok
        legacy_shares = dict(legacy_shares.items())
        legacy_shared = dict(legacy_shared.items())

        if self.md_version_None:
            self.set_md_version('')
        # upgrade it!
        self.main = FakeMain(self.root_dir, self.shares_dir,
                             self.data_dir, self.partials_dir)
        vm = self.main.vm
        def compare_share(share, old_share):
            """Compare two shares, new and old"""
            old_id = getattr(old_share, 'id', None)
            if old_id is None:
                old_id = old_share['volume_id']
            self.assertEquals(share.volume_id, old_id)
            self.assertEquals(share.path,
                getattr(old_share, 'path', None) or old_share['path'])
            self.assertEquals(share.node_id,
                getattr(old_share, 'subtree', None) or old_share['node_id'])
            if not isinstance(share, Root):
                self.assertEquals(share.name,
                    getattr(old_share, 'name', None) or old_share['name'])
                self.assertEquals(share.other_username,
                    getattr(old_share, 'other_username', None) \
                                  or old_share['other_username'])
                self.assertEquals(share.other_visible_name,
                    getattr(old_share, 'other_visible_name', None) \
                                  or old_share['other_visible_name'])
                self.assertEquals(share.access_level,
                    getattr(old_share, 'access_level', None) \
                                  or old_share['access_level'])

        for sid in vm.shares:
            old_share = legacy_shares[sid]
            share = vm.shares[sid]
            self.assertTrue(isinstance(share, Share) or isinstance(share, Root))
            compare_share(share, old_share)

        for sid in vm.shared:
            old_share = legacy_shared[sid]
            share = vm.shared[sid]
            self.assertTrue(isinstance(share, Shared))
            compare_share(share, old_share)

    def test_upgrade_5_critical_error(self):
        """Test the migration from version 5 with a critical error."""
        # build a fake version 5 state
        self._build_layout_version_4()
        self.set_md_version('5')
        # create some old shares and shared metadata
        legacy_shares = LegacyShareFileShelf(self.share_md_dir)
        root_share = _Share(path=self.root_dir, share_id=request.ROOT,
                            access_level='Modify')
        legacy_shares[request.ROOT] = root_share
        for idx, name in enumerate(['share'] * 10):
            sid = str(uuid.uuid4())
            share_name = name + '_' + str(idx)
            share = _Share(path=os.path.join(self.shares_dir, share_name),
                           share_id=sid, name=share_name,
                           node_id=str(uuid.uuid4()),
                           other_username='username'+str(idx),
                           other_visible_name='visible name ' + str(idx))
            if idx % 2:
                share.access_level = 'Modify'
            else:
                share.access_level = 'View'
            legacy_shares[sid] = share
        # create shared shares
        legacy_shared = LegacyShareFileShelf(self.shared_md_dir)
        for idx, name in enumerate(['dir'] * 5):
            sid = str(uuid.uuid4())
            share_name = name + '_' + str(idx)
            share = _Share(path=os.path.join(self.root_dir, share_name),
                           share_id=sid, node_id=str(uuid.uuid4()),
                           name=share_name, other_username='hola',
                           other_visible_name='hola')
            if idx % 2:
                share.access_level = 'Modify'
            else:
                share.access_level = 'View'
            legacy_shared[sid] = share

        # keep a copy of the current shares and shared metadata to check
        # the upgrade went ok
        legacy_shares = dict(legacy_shares.items())
        legacy_shared = dict(legacy_shared.items())

        if self.md_version_None:
            self.set_md_version('')
        # upgrade it!
        old_upgrade_share_to_volume = MetadataUpgrader._upgrade_share_to_volume
        def upgrade_share_to_volume(share, shared=False):
            raise ValueError('FAIL!')
        MetadataUpgrader._upgrade_share_to_volume = upgrade_share_to_volume
        try:
            self.assertRaises(ValueError, FakeMain, self.root_dir, self.shares_dir,
                              self.data_dir, self.partials_dir)
        finally:
            MetadataUpgrader._upgrade_share_to_volume = old_upgrade_share_to_volume

        shares = LegacyShareFileShelf(self.share_md_dir)
        self.assertEquals(len(list(shares.keys())), len(legacy_shares.keys()))
        for sid, share in shares.iteritems():
            old_share = legacy_shares[sid]
            self.assertTrue(isinstance(share, _Share))
            self.assertTrue(isinstance(old_share, _Share))
        shared = LegacyShareFileShelf(self.shared_md_dir)
        self.assertEquals(len(list(shared.keys())), len(legacy_shared.keys()))
        for sid, share in shared.iteritems():
            old_share = legacy_shared[sid]
            self.assertTrue(isinstance(share, _Share))
            self.assertTrue(isinstance(old_share, _Share))

    def test_broken_symlink_latest_metadata(self):
        """Test vm startup with latest metadata and a broken symlink."""
        self._build_layout_version_4()
        os.unlink(self.shares_dir_link)
        # create a broken link
        os.symlink('foo', self.shares_dir_link)
        # we want to keep a refernece to main in order to shutdown
        # pylint: disable-msg=W0201
        self.main = FakeMain(self.root_dir, self.shares_dir,
                             self.data_dir, self.partials_dir)
        self.check_version()


class BrokenOldMDVersionUpgradeTests(MetadataOldLayoutTests):
    """MetadataOldLayoutTests with broken .version file."""
    md_version_None = True


class BrokenNewMDVersionUpgradeTests(MetadataNewLayoutTests):
    """MetadataNewLayoutTests with broken .version file."""
    md_version_None = True


class GenerationsMetadataTestCase(MetadataNewLayoutTests):
    """Tests for VM metadata with generations."""

    def test_vol_without_generation_is_None(self):
        """Test that volumes without generations, get gen = None by default."""
        self._build_layout_version_4()
        self.set_md_version('6')
        self.udfs_md_dir = os.path.join(self.vm_data_dir, 'udfs')
        legacy_shares = LegacyShareFileShelf(self.share_md_dir)
        legacy_udfs = LegacyShareFileShelf(self.udfs_md_dir)
        # add a 'new' Share dict to the shelf
        share_name = 'share_1'
        share = Share(path=os.path.join(self.shares_dir, share_name),
                      volume_id=str(uuid.uuid4()), name=share_name,
                      access_level='Modify',
                      other_username='other_username',
                      other_visible_name='other_visible_name',
                      node_id=None)
        share.__dict__.pop('generation')
        legacy_shares[share.volume_id] = share.__dict__
        root = Root(node_id='root_node_id')
        root.__dict__.pop('generation')
        legacy_shares[root.volume_id] = root.__dict__
        udf = UDF(volume_id=str(uuid.uuid4()), node_id='udf_node_id',
                  suggested_path='~/UDF', path='a/fake/UDF')
        udf.__dict__.pop('generation')
        legacy_udfs[udf.volume_id] = udf.__dict__
        shares = VMFileShelf(self.share_md_dir)
        udfs = VMFileShelf(self.udfs_md_dir)
        self.assertEquals(None, shares[share.volume_id].generation)
        self.assertEquals(None, shares[root.volume_id].generation)
        self.assertEquals(None, udfs[udf.volume_id].generation)
        # add a value to the generation attribute
        share = shares[share.volume_id]
        share.generation = 1
        shares[share.volume_id] = share
        root = shares[root.volume_id]
        root.generation = 2
        shares[root.volume_id] = root
        udf = udfs[udf.volume_id]
        udf.generation = 3
        udfs[udf.volume_id] = udf
        # cleanup the cache
        del shares._cache[share.volume_id]
        del shares._cache[root.volume_id]
        del udfs._cache[udf.volume_id]
        # check again
        self.assertEquals(1, shares[share.volume_id].generation)
        self.assertEquals(2, shares[root.volume_id].generation)
        self.assertEquals(3, udfs[udf.volume_id].generation)


class MetadataUpgraderTests(MetadataTestCase):
    """MetadataUpgrader tests."""

    def setUp(self):
        """Create the MetadataUpgrader instance."""
        MetadataTestCase.setUp(self)
        self.share_md_dir = os.path.join(self.vm_data_dir, 'shares')
        self.shared_md_dir = os.path.join(self.vm_data_dir, 'shared')
        self.udfs_md_dir = os.path.join(self.vm_data_dir, 'udfs')
        self.home_dir = os.path.join(self.tmpdir, 'home', 'ubuntuonehacker')
        self.u1_dir = os.path.join(self.home_dir, os.path.split(self.u1_dir)[1])
        self.root_dir = self.u1_dir
        self.shares_dir = os.path.join(self.tmpdir, 'shares')
        self.shares_dir_link = os.path.join(self.u1_dir, 'Shared With Me')
        for path in [self.share_md_dir, self.shared_md_dir,
                     self.root_dir, self.shares_dir]:
            if not os.path.exists(path):
                os.makedirs(path)
        os.symlink(self.shares_dir, self.shares_dir_link)
        self.old_get_md_version = MetadataUpgrader._get_md_version
        MetadataUpgrader._get_md_version = lambda _: None
        self.md_upgrader = MetadataUpgrader(self.vm_data_dir, self.share_md_dir,
                                            self.shared_md_dir,
                                            self.udfs_md_dir, self.root_dir,
                                            self.shares_dir,
                                            self.shares_dir_link)
    def tearDown(self):
        """Restorre _get_md_version"""
        MetadataUpgrader._get_md_version = self.old_get_md_version
        MetadataTestCase.tearDown(self)

    def test_guess_metadata_version_None(self):
        """Test _guess_metadata_version method for pre-version."""
        # fake a version None layout
        if os.path.exists(self.version_file):
            os.unlink(self.version_file)
        for path in [self.share_md_dir, self.shared_md_dir,
                     self.root_dir, self.shares_dir]:
            if os.path.exists(path):
                self.rmtree(path)
        os.makedirs(os.path.join(self.root_dir, 'My Files'))
        shares_dir = os.path.join(self.root_dir, 'Shared With Me')
        os.makedirs(shares_dir)
        os.chmod(self.root_dir, 0500)
        os.chmod(shares_dir, 0500)
        version = self.md_upgrader._guess_metadata_version()
        self.assertEquals(None, version)

    def test_guess_metadata_version_1_or_2(self):
        """Test _guess_metadata_version method for version 1 or 2."""
        # fake a version 1 layout
        if os.path.exists(self.version_file):
            os.unlink(self.version_file)
        self.rmtree(self.root_dir)
        os.makedirs(os.path.join(self.root_dir, 'My Files'))
        shares_dir = os.path.join(self.root_dir, 'Shared With Me')
        os.makedirs(shares_dir)
        os.chmod(self.root_dir, 0500)
        os.chmod(shares_dir, 0500)
        self.rmtree(self.shares_dir)
        version = self.md_upgrader._guess_metadata_version()
        self.assertIn(version, ['1', '2'])

    def test_guess_metadata_version_4(self):
        """Test _guess_metadata_version method for version 4."""
        # fake a version 4 layout
        if os.path.exists(self.version_file):
            os.unlink(self.version_file)
        os.unlink(self.shares_dir_link)
        os.symlink(self.shares_dir_link, self.shares_dir_link)
        version = self.md_upgrader._guess_metadata_version()
        self.assertEquals(version, '4')

    def test_guess_metadata_version_5(self):
        """Test _guess_metadata_version method for version 5."""
        # fake a version 5 layout and metadata
        shelf = LegacyShareFileShelf(self.share_md_dir)
        shelf['foobar'] = _Share(path='/foo/bar', share_id='foobar')
        version = self.md_upgrader._guess_metadata_version()
        self.assertEquals(version, '5')

    def test_guess_metadata_version_6(self):
        """Test _guess_metadata_version method for version 6."""
        # fake a version 6 layout and metadata
        shelf = VMFileShelf(self.share_md_dir)
        shelf['foobar'] = Share(path='/foo/bar', volume_id='foobar')
        version = self.md_upgrader._guess_metadata_version()
        self.assertEquals(version, '6')

    def test_guess_mixed_metadata_5_and_6(self):
        """Test _guess_metadata_version method for mixed version 5 and 6."""
        # fake a version 6 layout and metadata
        shelf = LegacyShareFileShelf(self.share_md_dir)
        shelf['old_share'] = _Share(path='/foo/bar', share_id='old_share')
        shelf['new_share'] = Share(path='/bar/foo', volume_id='new_share').__dict__
        version = self.md_upgrader._guess_metadata_version()
        self.assertEquals(version, '5')
