#!/usr/bin/python
# Copyright 2009 Canonical Ltd.  All rights reserved.

"""
Dumps all the metadata.

Usage:

  dump_metadata.py <directory with vm and fsm subdirs>

"""

from __future__ import with_statement

import os
import sys
import user

from ubuntuone.syncdaemon import (
    filesystem_manager,
    volume_manager,
)
from xdg.BaseDirectory import xdg_cache_home

class FakeVM(object):
    """Some functionality needed from VM, without starting a new world."""

    def __init__(self, data_dir):
        # paths
        data_dir = os.path.join(data_dir, 'vm')
        version_file = os.path.join(data_dir, '.version')
        shares_dir = os.path.join(data_dir, 'shares')
        shared_dir = os.path.join(data_dir, 'shared')
        udfs_dir = os.path.join(data_dir, 'udfs')

        # check version
        with open(version_file) as fh:
            version_found = fh.read()
        version_should = volume_manager.VolumeManager.METADATA_VERSION
        if  version_found is None or version_found != version_should:
            print "The VolumeManager metadata version is too old (or too new?)"
            print "      it should be:", repr(version_should)
            print "      found:", repr(version_found)
            exit(-1)

        self.shares = volume_manager.VMFileShelf(shares_dir)
        self.shared = volume_manager.VMFileShelf(shared_dir)
        self.udfs =  volume_manager.VMFileShelf(udfs_dir)

    def get_volume(self, vol_id):
        """Gets the volume."""
        if vol_id in self.udfs:
            return self.udfs[vol_id]
        else:
            return self.shares[vol_id]

def main(data_dir):
    """Dump the metadata to stdout."""
    # start the players
    vm = FakeVM(data_dir)
    partials_dir = os.path.join(xdg_cache_home, 'ubuntuone', 'partials')
    fsm = filesystem_manager.FileSystemManager(data_dir, partials_dir, vm)

    shares = []
    root = None
    for share_id in vm.shares:
        share = vm.shares[share_id]
        if isinstance(share, volume_manager.Root):
            root = share
        else:
            shares.append(share)
    assert root is not None

    def show_data(volume_id):
        """Shows the info for the volume."""
        mdobjs = []
        for mdobj in fsm.get_mdobjs_by_share_id(volume_id):
            mdobjs.append((repr(mdobj.path), mdobj))
        mdobjs = [x[1] for x in sorted(mdobjs)]

        for mdobj in mdobjs:
            filedir = "DIR " if mdobj.is_dir else "FILE"
            print "  mdid=%r  node_id=%r  %s  %r" % (
                                mdobj.mdid, mdobj.node_id, filedir, mdobj.path)

    print "\nShowing Root: %r (id=%r)" % (root.path, root.id)
    show_data(root.id)

    for udf_id in vm.udfs:
        udf = vm.udfs[udf_id]
        print "\nShowing UDF: %r (id=%r)" % (udf.path, udf_id)
        show_data(udf_id)

    for share in shares:
        print "\nShowing Share: %r (id=%r)" % (share.path, share.id)
        show_data(share.id)

    print "\nShowing trash:"
    for (vol_id, node_id), (mdid, parent_id, path) in fsm.trash.iteritems():
        print "  mdid=%r  volume_id=%r  node_id=%r  parent_id=%r  path=%r" % (
            mdid, share_id, node_id, parent_id, path)


if __name__ == "__main__":
    if len(sys.argv) == 1:
        basedir = os.path.join(user.home,
                               '.local', 'share', 'ubuntuone', 'syncdaemon')
    elif len(sys.argv) == 2:
        basedir = sys.argv[1]
    else:
        print __doc__
        sys.exit()

    main(basedir)
