# test_main - tests for ubuntuone.oauthdesktop.main
#
# Author: Stuart Langridge <stuart.langridge@canonical.com>
#
# Copyright 2009 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
"""Tests for the OAuth client code for StorageFS."""

import os, StringIO
from contrib.mocker import MockerTestCase
from ubuntuone.oauthdesktop import config
from ubuntuone.oauthdesktop.main import LoginProcessor, BadRealmError

class Realm(MockerTestCase):
    """Test the realm handling finder"""

    def test_invalid_realm(self):
        """Are invalid realms rejected?"""
                
        login = LoginProcessor(None, use_libnotify=False)
        self.assertRaises(BadRealmError, login.login, "bad realm", "key")

    def test_realms(self):
        """Are realm URLs correctly retrieved from the config?"""
        
        # mock that tmp config file exists
        tmp_config_file = os.path.realpath(os.path.join(
          os.path.split(config.__file__)[0], "../../data/oauth_urls"
        ))
        osp = self.mocker.replace("os.path")
        osp.isfile(tmp_config_file)
        self.mocker.result(True)
        
        sio = StringIO.StringIO("""[default]
request_token_url = /rtu-default
user_authorisation_url = /uau-default
access_token_url = /atu-default
consumer_secret = foo-default

[http://localhost]
request_token_url = /rtu-localhost
user_authorisation_url = /uau-localhost
access_token_url = /atu-localhost
consumer_secret = foo-localhost

[http://ubuntuone.com]
request_token_url = /rtu-ubuntuone
user_authorisation_url = /uau-ubuntuone
access_token_url = /atu-ubuntuone
consumer_secret = foo-ubuntuone

""")
        mock_open = self.mocker.replace(open)
        mock_open(tmp_config_file)
        self.mocker.result(sio)
        
        self.mocker.replay()
        
        login = LoginProcessor(None, use_libnotify=False)
        
        # are localhost:XXXX URLs correctly fetched?
        (rtu, uau, atu, cs) = login.get_config_urls("http://localhost:9876")
        self.assertEqual(rtu, "http://localhost:9876/rtu-localhost")
        self.assertEqual(uau, "http://localhost:9876/uau-localhost")
        self.assertEqual(atu, "http://localhost:9876/atu-localhost")
        self.assertEqual(cs, "foo-localhost")

        # is a URL specifically in the config correctly fetched?
        (rtu, uau, atu, cs) = login.get_config_urls("http://ubuntuone.com")
        self.assertEqual(rtu, "http://ubuntuone.com/rtu-ubuntuone")
        self.assertEqual(uau, "http://ubuntuone.com/uau-ubuntuone")
        self.assertEqual(atu, "http://ubuntuone.com/atu-ubuntuone")
        self.assertEqual(cs, "foo-ubuntuone")

        # is a URL not in the config correctly fetched as default?
        (rtu, uau, atu, cs) = login.get_config_urls("http://other.example.net")
        self.assertEqual(rtu, "http://other.example.net/rtu-default")
        self.assertEqual(uau, "http://other.example.net/uau-default")
        self.assertEqual(atu, "http://other.example.net/atu-default")
        self.assertEqual(cs, "foo-default")


