# -*- coding: utf-8 -*-
#
# Copyright 2012 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# In addition, as a special exception, the copyright holders give
# permission to link the code of portions of this program with the
# OpenSSL library under certain conditions as described in each
# individual source file, and distribute linked combinations
# including the two.
# You must obey the GNU General Public License in all respects
# for all of the code used other than OpenSSL.  If you modify
# file(s) with this exception, you may extend this exception to your
# version of the file(s), but you are not obligated to do so.  If you
# do not wish to do so, delete this exception statement from your
# version.  If you delete this exception statement from all source
# files in the program, then also delete it here.
"""Tests for the runner helper module."""

from __future__ import unicode_literals

import os

from twisted.internet import defer

from ubuntu_sso.tests import TestCase
from ubuntu_sso.utils import runner


TEST_ME_DIR = 'test-me-more-♥'
TEST_ME_DIR_BYTES = TEST_ME_DIR.encode('utf-8')


class SpawnProgramTestCase(TestCase):
    """The test suite for the spawn_program method."""

    args = ('python', '-c',
            'import os; os.system("mkdir %s")' % TEST_ME_DIR)
    use_reactor = True
    timeout = 3

    @defer.inlineCallbacks
    def setUp(self):
        yield super(SpawnProgramTestCase, self).setUp()
        assert not os.path.exists(TEST_ME_DIR_BYTES)
        self.addCleanup(lambda: os.path.exists(TEST_ME_DIR_BYTES) and
                                os.rmdir(TEST_ME_DIR_BYTES))

    def spawn_fn(self, args):
        """The target function to test."""
        return runner.spawn_program(args, use_reactor=self.use_reactor)

    def assert_command_was_run(self):
        """The spawnned commnad was correctly run."""
        self.assertTrue(os.path.exists(TEST_ME_DIR_BYTES))
        self.assertTrue(os.path.isdir(TEST_ME_DIR_BYTES))

    @defer.inlineCallbacks
    def test_program_is_spawned(self):
        """The program is actually spawned."""
        yield self.spawn_fn(self.args)
        self.assert_command_was_run()

    @defer.inlineCallbacks
    def test_program_is_spawned_returned_code_non_zero(self):
        """The program is actually spawned."""
        status = yield self.spawn_fn(self.args)
        self.assertEqual(status, 0)

    @defer.inlineCallbacks
    def test_failed_to_start(self):
        """FailedToStartError is raised if the program does not start."""
        no_such_program = './myexecutablethatdoesnotexist'
        assert not os.path.exists(no_such_program)

        d = self.spawn_fn((no_such_program,))
        exc = yield self.assertFailure(d, runner.FailedToStartError)

        self.assertIn(no_such_program, exc.message)

    @defer.inlineCallbacks
    def test_other_error(self):
        """SpawnError is raised if the program does not start."""
        d = self.spawn_fn((None,))
        exc = yield self.assertFailure(d, runner.SpawnError)

        self.assertIn('Unspecified error', exc.message)
