# test_keyring - tests for ubuntu_sso.keyring
#
# Author: Alejandro J. Cura <alecu@canonical.com>
# Author: Natalia B. Bidart <natalia.bidart@canonical.com>
#
# Copyright 2010 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
"""Tests for the keyring.py module."""

import itertools
import socket
import urllib

import gnomekeyring
from twisted.trial.unittest import TestCase

from ubuntu_sso import keyring

APP_NAME = 'Yadda Yadda Doo'


def build_fake_gethostname(fake_hostname):
    """Return a fake hostname getter."""
    return lambda *a: fake_hostname


class MockKeyringItem(object):
    """A mock object that fakes an item found in a keyring search."""

    def __init__(self, item_id, keyring_name, attributes, secret):
        """Initialize this instance."""
        self.item_id = item_id
        self.keyring = keyring_name
        self.attributes = attributes
        self.secret = secret

    def matches(self, search_attr):
        """See if this item matches a given search."""
        for k, val in search_attr.items():
            if k not in self.attributes:
                return False
            if self.attributes[k] != val:
                return False
        return True


class MockGnomeKeyring(object):
    """A mock keyring that stores keys according to a given attr."""

    def __init__(self):
        """Initialize this instance."""
        self.id_counter = itertools.count()
        self.store = {}
        self.deleted = []

    def _get_next_id(self):
        """Return the next keyring id."""
        return self.id_counter.next()

    def item_create_sync(self, keyring_name, item_type, key_name, attr,
                         secret, update_if_exists):
        """Add a key to a keyring."""
        new_id = self._get_next_id()
        i = MockKeyringItem(new_id, keyring_name, attr, secret)
        self.store[new_id] = i

    def item_delete_sync(self, keyring_name, item_id):
        """Delete a key from a keyring, and keep a list of deleted keys."""
        item = self.store.pop(item_id)
        assert keyring_name == item.keyring
        self.deleted.append(item)

    def find_items_sync(self, item_type, attr):
        """Find all keys that match the given attributes."""
        items = [i for i in self.store.values() if i.matches(attr)]
        if len(items) == 0:
            raise gnomekeyring.NoMatchError()
        return items

    def is_available(self):
        """A very available keyring."""
        return True


class TestTokenNameBuilder(TestCase):
    """Test the method that builds the token name."""

    def test_get_simple_token_name(self):
        """A simple token name is built right."""
        sample_app_name = "UbuntuTwo"
        sample_hostname = "Darkstar"
        expected_result = "UbuntuTwo @ Darkstar"

        fake_gethostname = build_fake_gethostname(sample_hostname)
        self.patch(socket, "gethostname", fake_gethostname)
        result = keyring.get_token_name(sample_app_name)
        self.assertEqual(result, expected_result)

    def test_get_complex_token_name_for_app_name(self):
        """A complex token name is built right too."""
        sample_app_name = "Ubuntu @ Eleven"
        sample_hostname = "Mate+Cocido"
        expected_result = "Ubuntu @ Eleven @ Mate+Cocido"

        fake_gethostname = build_fake_gethostname(sample_hostname)
        self.patch(socket, "gethostname", fake_gethostname)
        result = keyring.get_token_name(sample_app_name)
        self.assertEqual(result, expected_result)

    def test_get_complex_token_name_for_hostname(self):
        """A complex token name is built right too."""
        sample_app_name = "Ubuntu Eleven"
        sample_hostname = "Mate @ Cocido"
        expected_result = "Ubuntu Eleven @ Mate AT Cocido"

        fake_gethostname = build_fake_gethostname(sample_hostname)
        self.patch(socket, "gethostname", fake_gethostname)
        result = keyring.get_token_name(sample_app_name)
        self.assertEqual(result, expected_result)


class TestKeyring(TestCase):
    """Test the gnome keyring related functions."""
    def setUp(self):
        """Initialize the mock used in these tests."""
        self.mgk = MockGnomeKeyring()
        self.patch(gnomekeyring, "item_create_sync", self.mgk.item_create_sync)
        self.patch(gnomekeyring, "is_available", self.mgk.is_available)
        self.patch(gnomekeyring, "find_items_sync", self.mgk.find_items_sync)
        self.patch(gnomekeyring, "item_delete_sync", self.mgk.item_delete_sync)
        fake_gethostname = build_fake_gethostname("darkstar")
        self.patch(socket, "gethostname", fake_gethostname)

    def test_set_ubuntusso(self):
        """Test that the set method does not erase previous keys."""
        sample_creds = {"name": "sample creds name"}
        sample_creds2 = {"name": "sample creds name 2"}
        keyring.Keyring("appname").set_ubuntusso_attr(sample_creds)
        keyring.Keyring("appname").set_ubuntusso_attr(sample_creds2)

        self.assertEqual(len(self.mgk.store), 2)
        self.assertEqual(len(self.mgk.deleted), 0)

    def test_delete_ubuntusso(self):
        """Test that a given key is deleted."""
        sample_creds = {"name": "sample creds name"}
        keyring.Keyring("appname").set_ubuntusso_attr(sample_creds)
        keyring.Keyring("appname").delete_ubuntusso_attr()

        self.assertEqual(len(self.mgk.store), 0)
        self.assertEqual(len(self.mgk.deleted), 1)

    def test_get_credentials(self):
        """Test that credentials are properly retrieved."""
        sample_creds = {"name": "sample creds name"}
        keyring.Keyring("appname").set_ubuntusso_attr(sample_creds)

        result = keyring.Keyring("appname").get_ubuntusso_attr()
        self.assertEqual(result, sample_creds)

    def test_get_credentials_migrating_token(self):
        """Test that credentials are properly retrieved and migrated."""
        sample_creds = {"name": "sample creds name"}
        obj = keyring.Keyring(APP_NAME)
        obj.token_name = keyring.get_old_token_name(APP_NAME)
        obj.set_ubuntusso_attr(sample_creds)

        result = keyring.Keyring(APP_NAME).get_ubuntusso_attr()
        self.assertEqual(result, sample_creds)

    def test_get_old_cred_found(self):
        """The method returns a new set of creds if old creds are found."""
        sample_oauth_token = "sample oauth token"
        sample_oauth_secret = "sample oauth secret"
        old_creds = {
            "oauth_token": sample_oauth_token,
            "oauth_token_secret": sample_oauth_secret,
        }
        secret = urllib.urlencode(old_creds)
        self.mgk.item_create_sync(None, None,
                                  keyring.U1_APP_NAME,
                                  keyring.U1_KEY_ATTR,
                                  secret, True)

        result = keyring.Keyring(keyring.U1_APP_NAME).get_ubuntusso_attr()

        self.assertIn("token", result)
        self.assertEqual(result["token"], sample_oauth_token)
        self.assertIn("token_secret", result)
        self.assertEqual(result["token_secret"], sample_oauth_secret)

    def test_get_old_cred_found_but_not_asked_for(self):
        """Returns None if old creds are present but the appname is not U1"""
        sample_oauth_token = "sample oauth token"
        sample_oauth_secret = "sample oauth secret"
        old_creds = {
            "oauth_token": sample_oauth_token,
            "oauth_token_secret": sample_oauth_secret,
        }
        secret = urllib.urlencode(old_creds)
        self.mgk.item_create_sync(None, None,
                                  keyring.U1_APP_NAME,
                                  keyring.U1_KEY_ATTR,
                                  secret, True)

        result = keyring.Keyring("Software Center").get_ubuntusso_attr()
        self.assertEqual(result, None)

    def test_get_old_cred_not_found(self):
        """The method returns None if no old nor new credentials found."""
        result = keyring.Keyring(keyring.U1_APP_NAME).get_ubuntusso_attr()
        self.assertEqual(result, None)
