/*
 * Fast index for tag data, based on integer indexes
 *
 * Copyright (C) 2006  Enrico Zini <enrico@debian.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <tests/test-utils.h>
#include <tagcoll/diskindex/mmap.h>

namespace tut {
using namespace std;
using namespace tagcoll::tests;
using namespace tagcoll::diskindex;

// Very simple test index

class TestIndex : public MMap
{
public:
	TestIndex(MasterMMap& master, int idx) : MMap(master, idx) {}

	const char* get() const { return m_buf; }
	unsigned int size() const { return m_size; }
};

class TestIndexer : public MMapIndexer
{
protected:
	std::string data;

public:
	TestIndexer(const std::string& data) : data(data) {}
	virtual ~TestIndexer() {}

	/// Return the size of the encoded index data (in bytes)
	virtual int encodedSize() const { return MMap::align(data.size() + 1); }

	/// Write the index data in the given buffer, which should be at least
	/// encodedSize bytes
	virtual void encode(char* buf) const
	{
		memcpy(buf, data.c_str(), data.size() + 1);
		int tail = data.size() + 1;
		int total = encodedSize();
		for ( ; tail < total; ++tail)
			buf[tail] = 0;
	}
};

static const char* fname = "tagcoll_mmapindex.tmp";

struct tagcoll_diskindex_mmap_shar {
	tagcoll_diskindex_mmap_shar() {
		TestIndexer index1("pippo");
		TestIndexer index2("pluto");
		TestIndexer index3("paperino");
		MasterMMapIndexer master(fname);
		master.append(index1);
		master.append(index2);
		master.append(index3);
		master.commit();
	}
	~tagcoll_diskindex_mmap_shar() {
		// Delete the test index
		unlink(fname);
	}
};
TESTGRP(tagcoll_diskindex_mmap);

// Check to see if the data read with the index is the same as encoded
template<> template<>
void to::test<1>()
{
	MasterMMap master(fname);

	TestIndex index1(master, 0);
	ensure_equals(index1.size(), MMap::align(6u));
	ensure_equals(string(index1.get()), string("pippo"));

	TestIndex index2(master, 1);
	ensure_equals(index2.size(), MMap::align(6u));
	ensure_equals(string(index2.get()), string("pluto"));

	TestIndex index3(master, 2);
	ensure_equals(index3.size(), MMap::align(9u));
	ensure_equals(string(index3.get()), string("paperino"));
}

}

// vim:set ts=4 sw=4:
