#ifndef TAGCOLL_CONSUMER_H
#define TAGCOLL_CONSUMER_H

/** \file
 * Consumer interface for a stream of tagged items
 */

/*
 * Copyright (C) 2003,2004,2005  Enrico Zini <enrico@debian.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#include <tagcoll/OpSet.h>

namespace Tagcoll
{

/**
 * Interface for classes interested in accepting a stream of tagged items.
 */
template< typename ITEM, typename TAG >
class Consumer
{
protected:
	/*
	 * Implementation note: to avoid problems with classes implementing only
	 * some of the virtual methods, they are given different names.  The common
	 * 'comsume' methods are just inlined calls to the right virtual functions,
	 * and are a way of keeping the unoverridden methods from being hidden.
	 */
	
	/// Process an untagged item
	virtual void consumeItemUntagged(const ITEM& item) = 0;

	/// Process a tagged item, with its tags
	virtual void consumeItem(const ITEM& item, const OpSet<TAG>& tags) = 0;

	/// Process a set of items, all with no tags
	virtual void consumeItemsUntagged(const OpSet<ITEM>& items)
	{
		// Default implementation for consumer that have no special handling of
		// item groups
		for (typename OpSet<ITEM>::const_iterator i = items.begin();
				i != items.end(); i++)
			consume(*i);
	}

	/// Process a set of items identically tagged, with their tags
	virtual void consumeItems(const OpSet<ITEM>& items, const OpSet<TAG>& tags)
	{
		// Default implementation for consumer that have no special handling of
		// item groups
		for (typename OpSet<ITEM>::const_iterator i = items.begin();
				i != items.end(); i++)
			consume(*i, tags);
	}

public:
	virtual ~Consumer() {}

	/// Process an untagged item
	void consume(const ITEM& item) { consumeItemUntagged(item); }

	/// Process a tagged item, with its tags
	void consume(const ITEM& item, const OpSet<TAG>& tags) { consumeItem(item, tags); }

	/// Process a set of items, all with no tags
	void consume(const OpSet<ITEM>& items) { consumeItemsUntagged(items); }

	/// Process a set of items identically tagged, with their tags
	void consume(const OpSet<ITEM>& items, const OpSet<TAG>& tags) { consumeItems(items, tags); }
};


/**
 * Consumer that discards its input
 */
template<class ITEM, class TAG>
class Sink : public Consumer<ITEM, TAG>
{
protected:
	virtual void consumeItemUntagged(const ITEM& item) {}
	virtual void consumeItem(const ITEM& item, const OpSet<TAG>& tags) {}
	virtual void consumeItemsUntagged(const OpSet<ITEM>& items) {}
	virtual void consumeItems(const OpSet<ITEM>& items, const OpSet<TAG>& tags) {}
public:
};

};

// vim:set ts=4 sw=4:
#endif
