#!/usr/bin/env python3
#
# ssh-import-id - Authorize SSH public keys from trusted online identities.
#
# Copyright (c) 2013 Casey Marshall <casey.marshall@gmail.com>
# Copyright (c) 2013 Dustin Kirkland <dustin.kirkland@gmail.com>
#
# ssh-import-id is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, version 3.
#
# ssh-import-id is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with ssh-import-id.  If not, see <http://www.gnu.org/licenses/>.

import os
import json
import subprocess
import sys
import requests
from urllib.parse import quote_plus


X_RATELIMIT_REMAINING = 'x-ratelimit-remaining'
HELP_URL = 'https://developer.github.com/v3/#rate-limiting'


if __name__ == '__main__':
    if not sys.argv[:1]:
        sys.stderr.write("Usage: %s <github-id>\n" % (sys.argv[0]))
        os._exit(1)
    for ghid in sys.argv[1:]:
        try:
            url = "https://api.github.com/users/%s/keys" % (quote_plus(ghid))
            resp = requests.get(url, verify=True)
            text = resp.text
            data = json.loads(text)
            
            if resp.status_code == 404:
                print('Username "%s" not found at GitHub API' % ghid)
                os._exit(1)

            if X_RATELIMIT_REMAINING in resp.headers and int(resp.headers[X_RATELIMIT_REMAINING]) == 0:
                print('GitHub REST API rate-limited this IP address. See %s' % HELP_URL)
                os._exit(1)              

            for keyobj in data:
                sys.stdout.write("%s %s@github/%s\n" % (keyobj['key'], ghid, keyobj['id']))

            sys.stdout.flush()
        except (Exception,):
            e = sys.exc_info()[1]
            sys.stderr.write("ERROR: %s\n" % (str(e)))
            os._exit(1)
        sys.stdout.write("\n")
    os._exit(0)
    