/* This file is part of the SpeedCrunch project
   Copyright (C) 2005 Ariya Hidayat <ariya@kde.org>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include "deletevardlg.h"
#include "evaluator.h"
#include "hmath.h"
#include "settings.h"

#include <qdialog.h>
#include <qlayout.h>
#include <qlistview.h>
#include <qpushbutton.h>
#include <qtimer.h>

class DeleteVariableDlg::Private
{
public:
  Evaluator* eval;
  QListView* varList;
  QPushButton* deleteButton;
  QPushButton* closeButton;
};

DeleteVariableDlg::DeleteVariableDlg( Evaluator* e, QWidget* parent, const char* name ):
QDialog( parent, name )
{
  d = new Private;
  d->eval = e;
  
  QGridLayout* layout = new QGridLayout( this, 2, 3, 6, 6 );

  d->varList = new QListView( this );
  d->varList->addColumn( tr("Name") );
  d->varList->addColumn( tr("Value") );
  d->varList->setResizeMode( QListView::LastColumn );
  d->varList->setSelectionMode( QListView::Multi );
  d->varList->setAllColumnsShowFocus( true );
  d->varList->setColumnAlignment( 1, Qt::AlignRight );
    
  d->deleteButton = new QPushButton( tr("&Delete"), this );
  d->closeButton = new QPushButton( tr("Close"), this );

  layout->addMultiCellWidget( d->varList, 0, 0, 1, 3 );  
  layout->addItem( new QSpacerItem( 20, 20, QSizePolicy::Expanding ), 1, 1 );  
  layout->addWidget( d->deleteButton, 1, 2 );
  layout->addWidget( d->closeButton, 1, 3 );
  
  d->deleteButton->setDefault( true );
  
  connect( d->deleteButton, SIGNAL( clicked() ), this, SLOT( deleteVar() ) );
  connect( d->closeButton, SIGNAL( clicked() ), this, SLOT( reject() ) );
  
  QTimer::singleShot( 0, this, SLOT( initUI() ) );  
  setCaption( tr("Delete Variable") );
}

DeleteVariableDlg::~DeleteVariableDlg()
{
  delete d;
}

void DeleteVariableDlg::updateList()
{
  d->varList->clear();
  QValueVector<Variable> vars = d->eval->variables();
  for( unsigned i = 0; i < vars.count(); i++ )
  {
    QString var = vars[i].name;
    if( var.upper() == QString("ANS") ) continue;
    if( var.upper() == QString("PI") ) continue;
    char* ss = HMath::format( vars[i].value, 'g' );
    new QListViewItem( d->varList, vars[i].name, QString( ss ) );
    free( ss );  
  }
  
  d->varList->setFocus();
}

void DeleteVariableDlg::initUI()
{
  updateList();
  adjustSize();
}

void DeleteVariableDlg::deleteVar()
{
  QListViewItem* item;
  QListViewItem* nextItem;
  for( item = d->varList->firstChild(); item; item = nextItem )
  {
    nextItem = item->nextSibling();
    if( item->isSelected() )
    {
      d->eval->remove( item->text(0) );
      delete item;
    }  
  }
}
