/* This file is part of the SpeedCrunch project
   Copyright (C) 2004 Ariya Hidayat <ariya@kde.org>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include "crunch.h"
#include "evaluator.h"
#include "hmath.h"
#include "editor.h"
#include "functions.h"
#include "result.h"
#include "settings.h"
#include "insertfunctiondlg.h"
#include "insertvardlg.h"
#include "deletevardlg.h"
#include "configdlg.h"
#include "aboutbox.h"

#include <qaction.h>
#include <qapplication.h>
#include <qclipboard.h>
#include <qfile.h>
#include <qfiledialog.h>
#include <qhbox.h>
#include <qmainwindow.h>
#include <qmenubar.h>
#include <qmessagebox.h>
#include <qpopupmenu.h>
#include <qpushbutton.h>
#include <qradiobutton.h>
#include <qtextstream.h>
#include <qtooltip.h>
#include <qtimer.h>
#include <qvbox.h>
#include <qwidget.h>

#include <stdio.h>

class CrunchActions
{
public:
  QAction* sessionSave;
  QAction* sessionQuit;
  QAction* editPaste;
  QAction* editCopy;
  QAction* editCopyResult;
  QAction* insertFunction;
  QAction* insertVariable;
  QAction* deleteVariable;
  QAction* clearInput;
  QAction* clearDisplay;
  QAction* clearHistory;
  QAction* clearVariables;
  QAction* viewGeneral;
  QAction* viewFixed;
  QAction* viewExponential;
  QAction* digitsAuto;
  QAction* digits2;
  QAction* digits3;
  QAction* digits8;
  QAction* digits15;
  QAction* digits50;
  QAction* showClearButton;
  QAction* showEvalButton;
  QAction* configure;
  QAction* helpAboutQt;
  QAction* helpAbout;
};


class Crunch::Private
{
public:
  CrunchActions* actions;  
  Evaluator* eval;
  Editor *editor;
  Result* result;
  QPushButton* clearInputButton;
  QPushButton* evalButton;
  QRadioButton* degButton;
  QRadioButton* radButton;
  bool autoAns;
  
  ConfigDlg* configDlg;
  InsertFunctionDlg* insertFunctionDlg;
  InsertVariableDlg* insertVariableDlg;
  DeleteVariableDlg* deleteVariableDlg;
};

Crunch::Crunch(): QMainWindow()
{
  d = new Private;
  d->actions = new CrunchActions;
  
  d->eval = new Evaluator;
  d->autoAns = false;
  
  QVBox *box = new QVBox( this );
  box->setMargin( 5 );
  box->setSpacing( 5 );
  setCentralWidget( box );
  
  QHBox* topbox = new QHBox( box );
  
  QWidget* spacer = new QWidget( topbox );
  spacer->setMinimumWidth( 50 );
  spacer->setSizePolicy( QSizePolicy::Expanding, QSizePolicy::Minimum );
  d->degButton = new QRadioButton( tr( "&Degrees" ), topbox );
  d->radButton = new QRadioButton( tr( "&Radians" ), topbox );
  d->degButton->setFocusPolicy( ClickFocus );
  d->radButton->setFocusPolicy( ClickFocus );
  connect( d->degButton, SIGNAL( toggled( bool ) ), SLOT( angleModeChanged() ) );
  connect( d->radButton, SIGNAL( toggled( bool ) ), SLOT( angleModeChanged() ) );

  d->result = new Result( box );

  QHBox* inputBox = new QHBox( box );
  inputBox->setSpacing( 5 );
  
  d->clearInputButton = new QPushButton( inputBox );
  d->clearInputButton->setMaximumWidth( 25 );
  d->clearInputButton->setFlat( true );
  d->clearInputButton->setIconSet( QPixmap::fromMimeSource("clearinput.png") );  
  d->clearInputButton->hide();
  QToolTip::add( d->clearInputButton, tr("Clear input line") );
  
  d->editor = new Editor( d->eval, inputBox );
  d->editor->setFocus();
  
  d->evalButton = new QPushButton( inputBox );
  d->evalButton->setText( tr("Evaluate" ) );
  d->evalButton->hide();
  
  d->clearInputButton->setMaximumHeight( d->editor->sizeHint().height() );
  d->clearInputButton->setMaximumHeight( d->editor->sizeHint().height() );

  connect( d->clearInputButton, SIGNAL( clicked() ), SLOT( clearInput() ) );
  connect( d->evalButton, SIGNAL( clicked() ), SLOT( returnPressed() ) );
  connect( d->editor, SIGNAL( returnPressed() ), SLOT( returnPressed() ) );
  connect( d->editor, SIGNAL( textChanged() ), SLOT( textChanged() ) );

  d->configDlg = 0;
  d->insertFunctionDlg = 0;
  d->insertVariableDlg = 0;
  d->deleteVariableDlg = 0;
        
  setCaption( tr( "SpeedCrunch" ) );
  d->degButton->setChecked( true );
  createUI();  
  QTimer::singleShot( 0, this, SLOT( applySettings() ) );  
}

Crunch::~Crunch()
{
  delete d->actions;
  delete d->eval;
  delete d;
}

void Crunch::createUI()
{
  // create all the actions
  d->actions->sessionSave = new QAction( tr("&Save..."), CTRL + Key_S, this );
  d->actions->sessionQuit = new QAction( tr("&Quit"), 0, this );
  d->actions->editCopy = new QAction( tr("&Copy"), CTRL + Key_C, this );
  d->actions->editPaste = new QAction( tr("&Paste"), CTRL + Key_V, this );
  d->actions->editCopyResult = new QAction( tr("Copy &Result"), CTRL + Key_R, this );
  d->actions->insertFunction = new QAction( tr("Insert &Function..."), CTRL + Key_F, this );
  d->actions->insertVariable = new QAction( tr("Insert &Variable..."), CTRL + Key_I, this );
  d->actions->deleteVariable = new QAction( tr("D&elete Variable..."), CTRL + Key_D, this );
  d->actions->clearInput = new QAction( tr("Clear &Input" ), 0, this );
  d->actions->clearDisplay = new QAction( tr("Clear &Display" ), 0, this );
  d->actions->clearHistory = new QAction( tr("Clear &History" ), 0, this );
  d->actions->clearVariables = new QAction( tr("Clear V&ariables" ), 0, this );
  QActionGroup *formatGroup = new QActionGroup( this );
  d->actions->viewGeneral = new QAction( tr("&General"), 0, formatGroup );
  d->actions->viewFixed = new QAction( tr("&Fixed Decimal"), 0, formatGroup );
  d->actions->viewExponential = new QAction( tr("&Exponential"), 0, formatGroup );
  d->actions->viewGeneral->setToggleAction( true );
  d->actions->viewFixed->setToggleAction( true );
  d->actions->viewExponential->setToggleAction( true );
  QActionGroup *digitsGroup = new QActionGroup( this );
  d->actions->digitsAuto = new QAction( tr("&Automatic Precision"), 0, digitsGroup );
  d->actions->digits2 = new QAction( tr("&2 Decimal Digits"), 0, digitsGroup );
  d->actions->digits3 = new QAction( tr("&3 Decimal Digits"), 0, digitsGroup );
  d->actions->digits8 = new QAction( tr("&8 Decimal Digits"), 0, digitsGroup );
  d->actions->digits15 = new QAction( tr("&15 Decimal Digits"), 0, digitsGroup );
  d->actions->digits50 = new QAction( tr("&50 Decimal Digits"), 0, digitsGroup );
  d->actions->digitsAuto->setToggleAction( true );
  d->actions->digits2->setToggleAction( true );
  d->actions->digits3->setToggleAction( true );
  d->actions->digits8->setToggleAction( true );
  d->actions->digits15->setToggleAction( true );
  d->actions->digits50->setToggleAction( true );
  d->actions->showClearButton = new QAction( tr("&Show Clear Button"), 0, this );
  d->actions->showEvalButton = new QAction( tr("Show &Evaluate Button"), 0, this );
  d->actions->showClearButton->setToggleAction( true );
  d->actions->showEvalButton->setToggleAction( true );
  d->actions->configure = new QAction( tr("&Configure..."), 0, this );
  d->actions->helpAbout = new QAction( tr("&About"), 0, this );
  d->actions->helpAboutQt = new QAction( tr("About &Qt"), 0, this );
  
  // signal/slot
  connect( d->actions->sessionSave, SIGNAL( activated() ), this, SLOT( saveSession() ) );  
  connect( d->actions->sessionQuit, SIGNAL( activated() ), this, SLOT( close() ) );  
  connect( d->actions->editPaste, SIGNAL( activated() ), d->editor, SLOT( paste() ) );  
  connect( d->actions->editCopy, SIGNAL( activated() ), d->editor, SLOT( copy() ) );  
  connect( d->actions->editCopyResult, SIGNAL( activated() ), this, SLOT( copyResult() ) );  
  connect( d->actions->clearInput, SIGNAL( activated() ), this, SLOT( clearInput() ) );  
  connect( d->actions->clearDisplay, SIGNAL( activated() ), d->result, SLOT( clear() ) );  
  connect( d->actions->clearHistory, SIGNAL( activated() ), d->editor, SLOT( clearHistory() ) );  
  connect( d->actions->clearVariables, SIGNAL( activated() ), this, SLOT( clearVariables() ) );  
  connect( d->actions->insertFunction, SIGNAL( activated() ), this, SLOT( insertFunction() ) );  
  connect( d->actions->insertVariable, SIGNAL( activated() ), this, SLOT( insertVariable() ) );  
  connect( d->actions->deleteVariable, SIGNAL( activated() ), this, SLOT( deleteVariable() ) );  
  connect( d->actions->viewGeneral, SIGNAL( activated() ), this, SLOT( viewGeneral() ) );  
  connect( d->actions->viewFixed, SIGNAL( activated() ), this, SLOT( viewFixed() ) );  
  connect( d->actions->viewExponential, SIGNAL( activated() ), this, SLOT( viewExponential() ) );  
  connect( d->actions->digitsAuto, SIGNAL( activated() ), this, SLOT( digitsAuto() ) );  
  connect( d->actions->digits2, SIGNAL( activated() ), this, SLOT( digits2() ) );  
  connect( d->actions->digits3, SIGNAL( activated() ), this, SLOT( digits3() ) );  
  connect( d->actions->digits8, SIGNAL( activated() ), this, SLOT( digits8() ) );  
  connect( d->actions->digits15, SIGNAL( activated() ), this, SLOT( digits15() ) );  
  connect( d->actions->digits50, SIGNAL( activated() ), this, SLOT( digits50() ) );  
  connect( d->actions->showClearButton, SIGNAL( activated() ), this, SLOT( showClearButton() ) );  
  connect( d->actions->showEvalButton, SIGNAL( activated() ), this, SLOT( showEvalButton() ) );  
  connect( d->actions->configure, SIGNAL( activated() ), this, SLOT( configure() ) );  
  connect( d->actions->helpAbout, SIGNAL( activated() ), this, SLOT( about() ) );  
  connect( d->actions->helpAboutQt, SIGNAL( activated() ), this, SLOT( aboutQt() ) );  
  
  // construct the menu

  QPopupMenu *sessionMenu = new QPopupMenu( this );
  menuBar()->insertItem( tr("&Session"), sessionMenu );
  d->actions->sessionSave->addTo( sessionMenu );
  d->actions->sessionQuit->addTo( sessionMenu );

  QPopupMenu *editMenu = new QPopupMenu( this );
  menuBar()->insertItem( tr("&Edit"), editMenu );
  d->actions->editCopy->addTo( editMenu );
  d->actions->editCopyResult->addTo( editMenu );
  d->actions->editPaste->addTo( editMenu );
  editMenu->insertSeparator();
  d->actions->insertFunction->addTo( editMenu );
  d->actions->insertVariable->addTo( editMenu );
  editMenu->insertSeparator();
  d->actions->deleteVariable->addTo( editMenu );
  editMenu->insertSeparator();
  d->actions->clearInput->addTo( editMenu );
  d->actions->clearDisplay->addTo( editMenu );
  d->actions->clearHistory->addTo( editMenu );
  d->actions->clearVariables->addTo( editMenu );

  QPopupMenu *viewMenu = new QPopupMenu( this );
  menuBar()->insertItem( tr("&View"), viewMenu );
  d->actions->viewGeneral->addTo( viewMenu );
  d->actions->viewFixed->addTo( viewMenu );
  d->actions->viewExponential->addTo( viewMenu );
  viewMenu->insertSeparator();
  d->actions->digitsAuto->addTo( viewMenu );
  d->actions->digits2->addTo( viewMenu );
  d->actions->digits3->addTo( viewMenu );
  d->actions->digits8->addTo( viewMenu );
  d->actions->digits15->addTo( viewMenu );
  d->actions->digits50->addTo( viewMenu );

  QPopupMenu *settingsMenu = new QPopupMenu( this );
  d->actions->showClearButton->addTo( settingsMenu );
  d->actions->showEvalButton->addTo( settingsMenu );
  menuBar()->insertItem( tr("Se&ttings"), settingsMenu );
  settingsMenu->insertSeparator();
  d->actions->configure->addTo( settingsMenu );

  QPopupMenu *helpMenu = new QPopupMenu( this );
  menuBar()->insertItem( tr("&Help"), helpMenu );
  d->actions->helpAbout->addTo( helpMenu );
  d->actions->helpAboutQt->addTo( helpMenu );
  
  setIcon( QPixmap::fromMimeSource( "crunch.png" ) );
  
  Settings::self()->load();
}

void Crunch::applySettings()
{
  Settings* settings = Settings::self();
  settings->load();
  
  d->editor->setAutoCompleteEnabled( settings->autoComplete );
  d->editor->setAutoCalcEnabled( settings->autoCalc );
  d->editor->setSyntaxHighlight( settings->enableSyntaxHighlight );
  d->editor->setHighlightColor( Editor::Number, settings->highlightNumberColor );
  d->editor->setHighlightColor( Editor::FunctionName, settings->highlightFunctionColor );
  d->editor->setHighlightColor( Editor::Variable, settings->highlightVariableColor );
  d->editor->setHighlightColor( Editor::MatchedPar, settings->matchedParenthesisColor );
  
  if( settings->angleMode == "degree" )
  {
    d->eval->setAngleMode( Evaluator::Degree );
    d->degButton->setChecked( true );
    d->radButton->setChecked( false );
  }
  
  if( settings->angleMode == "radian" )
  {
    d->eval->setAngleMode( Evaluator::Radian );
    d->degButton->setChecked( false );
    d->radButton->setChecked( true );
  }
  
  if( settings->saveHistory )
  if( settings->history.count() )
    d->editor->setHistory( settings->history );
  
  if( settings->saveVariables )
  {
    for( unsigned k=0; k<settings->variables.count(); k++ )
    {
      d->eval->setExpression( settings->variables[k] );
      d->eval->eval();
    }
  }
  
  d->result->setFormat( settings->format );
  d->result->setDecimalDigits( settings->decimalDigits );
  d->editor->setFormat( settings->format );
  d->editor->setDecimalDigits( settings->decimalDigits );

  if( settings->customAppearance )
  {
    d->result->setFont( settings->customFont );
    d->editor->setFont( settings->customFont );
  }
  else
  {
    d->result->setFont( QApplication::font( d->result ) );
    d->editor->setFont( QApplication::font( d->editor ) );  
  }
  
  d->result->setCustomAppearance( settings->customAppearance );
  d->result->setCustomTextColor( settings->customTextColor );
  d->result->setCustomBackgroundColor( settings->customBackgroundColor1, 
    settings->customBackgroundColor2 );
    
  if( settings->format == 'g' ) d->actions->viewGeneral->setOn( true );
  if( settings->format == 'f' ) d->actions->viewFixed->setOn( true );
  if( settings->format == 'e' ) d->actions->viewExponential->setOn( true );
  
  if( settings->decimalDigits < 0 ) d->actions->digitsAuto->setOn( true );
  if( settings->decimalDigits == 2 ) d->actions->digits2->setOn( true );
  if( settings->decimalDigits == 3 ) d->actions->digits3->setOn( true );
  if( settings->decimalDigits == 8 ) d->actions->digits8->setOn( true );
  if( settings->decimalDigits == 15 ) d->actions->digits15->setOn( true );
  if( settings->decimalDigits == 50 ) d->actions->digits50->setOn( true );
  
  if( settings->showClearInputButton )
    d->clearInputButton->show();
  else
    d->clearInputButton->hide();
  d->actions->showClearButton->setOn( settings->showClearInputButton );
    
  if( settings->showEvaluateButton )
    d->evalButton->show();
  else
    d->evalButton->hide();
  d->actions->showEvalButton->setOn( settings->showEvaluateButton );
}

void Crunch::closeEvent( QCloseEvent* e )
{
  saveSettings();
  QMainWindow::closeEvent( e ); 
}

void Crunch::saveSession()
{
  QString filters = tr( "Text Files (*.txt);; All Files (*.*)" );
  QString fname = QFileDialog::getSaveFileName( QString::null, filters, this, 0,
    tr("Save Session") );
  if( fname.isEmpty() ) return;
  
  QFile file( fname );
  if( !file.open( IO_WriteOnly ) )
  {
    QMessageBox::critical( this, tr("Error"), tr("Can't write to file %1").arg( fname ) );
    return;
  }
  
  QTextStream stream( &file );
  stream << d->result->asText();
  
  file.close();
}

void Crunch::saveSettings()
{
  Settings* settings = Settings::self();
  
  if( d->eval->angleMode()== Evaluator::Degree )
    settings->angleMode = "degree";
  if( d->eval->angleMode()== Evaluator::Radian )
    settings->angleMode = "radian";
  
  if( settings->saveHistory )
    settings->history = d->editor->history();
  
  if( settings->saveVariables )
  {
    settings->variables.clear();
    QValueVector<Variable> vars = d->eval->variables();
    for( unsigned i=0; i<vars.count(); i++ )
      if( vars[i].name.lower() != "pi" )
      {
        char* str = HMath::formatFixed( vars[i].value, 100 );
        settings->variables.append( QString("%1=%2").arg( vars[i].name ).
          arg( QString( str ) ) );
        free( str );  
      }
  }
  
  settings->save();  
}

void Crunch::angleModeChanged()
{
  const QObject* s = sender();
  if( !s ) return;
  if( !s->isA( "QRadioButton" ) ) return;
  
  blockSignals( true );
  if( s == static_cast<QObject*>( d->degButton ) )
    d->radButton->setChecked( !d->degButton->isChecked() );
  if( s == static_cast<QObject*>( d->radButton ) )
    d->degButton->setChecked( !d->radButton->isChecked() );
  blockSignals( false );
  
  if( d->degButton->isChecked() ) 
    d->eval->setAngleMode( Evaluator::Degree );
  if( d->radButton->isChecked() ) 
    d->eval->setAngleMode( Evaluator::Radian );
  
}

void Crunch::returnPressed()
{
  QString str = Evaluator::autoFix( d->editor->text() );
  if( str.isEmpty() ) return;
  
  d->eval->setExpression( str );
  d->editor->appendHistory( str );

  HNumber result = d->eval->eval();
  if( !d->eval->error().isEmpty() )
    d->result->appendError( str, tr( "Error: %1" ).arg( d->eval->error() )  );
  else
    d->result->append( str, result );
  
  d->editor->setText( str );  
  d->editor->selectAll();
  d->autoAns = true;
}

void Crunch::textChanged()
{
  if( d->autoAns )
  {
    QString expr = Evaluator::autoFix( d->editor->text() );
    if( expr.isEmpty() ) return;  
    Tokens tokens = Evaluator::scan( expr );
    if( tokens.count() == 1 )
    if( ( tokens[0].asOperator() == Token::Plus ) ||
        ( tokens[0].asOperator() == Token::Minus ) ||
        ( tokens[0].asOperator() == Token::Asterisk ) ||
        ( tokens[0].asOperator() == Token::Slash ) ||
        ( tokens[0].asOperator() == Token::Caret ) )
     {
       d->autoAns = false;
       expr.prepend( "ans" );
       d->editor->setText( expr );
       d->editor->setCursorPosition( 0, expr.length() );
     }
  }
}


void Crunch::copyResult()
{
  QClipboard *cb = QApplication::clipboard();
  char *ss = HMath::formatFixed( d->eval->get("ans") );
  cb->setText( QString(ss), QClipboard::Clipboard );  
  free( ss );
}

void Crunch::clearInput()
{
  d->editor->clear();
}

void Crunch::clearVariables()
{
  d->eval->clearVariables();
}

void Crunch::insertFunction()
{
  if( !d->insertFunctionDlg )
    d->insertFunctionDlg = new InsertFunctionDlg( this );
  //else    d->insertFunctionDlg->updateList();  
  
  if( d->insertFunctionDlg->exec() == InsertFunctionDlg::Accepted )
  {
    QString fname = d->insertFunctionDlg->functionName();
    if( !fname.isEmpty() )
      d->editor->insert( fname );  
  }  
}

void Crunch::insertVariable()
{
  if( !d->insertVariableDlg )
    d->insertVariableDlg = new InsertVariableDlg( d->eval, this );
  else     
    d->insertVariableDlg->updateList();  
  
  if( d->insertVariableDlg->exec() == InsertVariableDlg::Accepted )
  {
    QString varname = d->insertVariableDlg->variableName();
    if( !varname.isEmpty() )
      d->editor->insert( varname );
  }
}

void Crunch::deleteVariable()
{
  if( !d->deleteVariableDlg )
    d->deleteVariableDlg = new DeleteVariableDlg( d->eval, this );
  else     
    d->deleteVariableDlg->updateList();  
  
  d->deleteVariableDlg->exec();
}

void Crunch::viewGeneral()
{
  Settings* settings = Settings::self();
  settings->format = 'g';
  saveSettings();
  applySettings();
}

void Crunch::viewFixed()
{
  Settings* settings = Settings::self();
  settings->format = 'f';
  saveSettings();
  applySettings();
}

void Crunch::viewExponential()
{
  Settings* settings = Settings::self();
  settings->format = 'e';
  saveSettings();
  applySettings();
}

void Crunch::digitsAuto()
{
  Settings* settings = Settings::self();
  settings->decimalDigits = -1;
  saveSettings();
  applySettings();
}

void Crunch::digits2()
{
  Settings* settings = Settings::self();
  settings->decimalDigits = 2;
  saveSettings();
  applySettings();
}

void Crunch::digits3()
{
  Settings* settings = Settings::self();
  settings->decimalDigits = 3;
  saveSettings();
  applySettings();
}

void Crunch::digits8()
{
  Settings* settings = Settings::self();
  settings->decimalDigits = 8;
  saveSettings();
  applySettings();
}

void Crunch::digits15()
{
  Settings* settings = Settings::self();
  settings->decimalDigits = 15;
  saveSettings();
  applySettings();
}

void Crunch::digits50()
{
  Settings* settings = Settings::self();
  settings->decimalDigits = 50;
  saveSettings();
  applySettings();
}

void Crunch::showClearButton()
{
  Settings* settings = Settings::self();
  settings->showClearInputButton = !settings->showClearInputButton;
  saveSettings();
  applySettings();
}

void Crunch::showEvalButton()
{
  Settings* settings = Settings::self();
  settings->showEvaluateButton = !settings->showEvaluateButton;
  saveSettings();
  applySettings();
}

void Crunch::configure()
{
  saveSettings();
  if( !d->configDlg )
  {
    d->configDlg = new ConfigDlg( this );
    connect( d->configDlg, SIGNAL( settingsChanged() ), SLOT( applySettings() ) );
  }
  d->configDlg->exec();
}

void Crunch::about()
{
  AboutBox* aboutBox = new AboutBox( this );
  aboutBox->setCaption( tr("About SpeedCrunch" ) );
  aboutBox->exec();
  delete aboutBox;
}

void Crunch::aboutQt()
{
  QMessageBox::aboutQt( this, tr("About Qt") );
}
