#!/usr/bin/python
#
# Copyright 2009-2013 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
"""Setup.py: build, distribute, clean."""

import platform
import glob
import re
from subprocess import call
import sys

from DistUtilsExtra.auto import setup

# The VERSION of software-center
VERSION = '5.6.0'

# Get the distribution information for various functions.
(distro, release, codename) = platform.dist()


def merge_authors_into_about_dialog():
    fname = "./data/ui/gtk3/SoftwareCenter.ui"
    authors = open("AUTHORS").read()
    gtkbuilder = open(fname).read()
    gtkbuilder = re.sub(r'<property name="authors">.*?</property>',
                        r'<property name="authors">%s</property>' % authors,
                        gtkbuilder, flags=re.DOTALL)
    open(fname, "w").write(gtkbuilder)


def merge_extras_ubuntu_com_channel_file():
    # update ubuntu-extras.list.in (this will not be part of debian as
    # its killed of in debian/rules on a non-ubuntu build)
    channelfile = "data/channels/Ubuntu/ubuntu-extras.list"
    s = open(channelfile + ".in").read()
    open(channelfile, "w").write(s.replace("#DISTROSERIES#", codename))


# update version.py
def update_version():
    # this comes from the build host
    open("softwarecenter/version.py", "w").write("""
VERSION = '%s'
CODENAME = '%s'
DISTRO = '%s'
RELEASE = '%s'
""" % (VERSION, codename, distro, release))
    # double check that we have the latest distro series and fail if not
    if distro == "Ubuntu":
        import softwarecenter.distro.ubuntu
        if not codename in softwarecenter.distro.ubuntu.Ubuntu.DISTROSERIES:
            raise ValueError("Could not find '%s' in ubuntu distro class "
                             "please add it to DISTROSERIES", codename)


# update po4a
if sys.argv[1] == "build":
    update_version()
    merge_authors_into_about_dialog()
    merge_extras_ubuntu_com_channel_file()
    call(["po4a", "po/help/po4a.conf"])


# real setup
setup(
    name="software-center",
    version=VERSION,
    scripts=[
        "software-center",
        "software-center-dbus",
        # gtk3
        "utils/submit_review_gtk3.py",
        "utils/report_review_gtk3.py",
        "utils/submit_usefulness_gtk3.py",
        "utils/delete_review_gtk3.py",
        "utils/modify_review_gtk3.py",
        # db helpers
        "utils/update-software-center",
        "utils/update-software-center-channels",
        "utils/update-software-center-agent",
        # generic helpers
        "utils/expunge-cache.py",
    ] + glob.glob("utils/piston-helpers/*.py"),
    packages=[
        'softwarecenter',
        'softwarecenter.backend',
        'softwarecenter.backend.installbackend_impl',
        'softwarecenter.backend.channel_impl',
        'softwarecenter.backend.oneconfhandler',
        'softwarecenter.backend.login_impl',
        'softwarecenter.backend.piston',
        'softwarecenter.backend.reviews',
        'softwarecenter.db',
        'softwarecenter.db.pkginfo_impl',
        'softwarecenter.db.history_impl',
        'softwarecenter.distro',
        'softwarecenter.plugins',
        'softwarecenter.sso',
        'softwarecenter.ui',
        'softwarecenter.ui.gtk3',
        'softwarecenter.ui.gtk3.dialogs',
        'softwarecenter.ui.gtk3.models',
        'softwarecenter.ui.gtk3.panes',
        'softwarecenter.ui.gtk3.session',
        'softwarecenter.ui.gtk3.views',
        'softwarecenter.ui.gtk3.widgets',
        'softwarecenter.ui.qml',
    ],
    data_files=[
        # gtk3
        ('share/software-center/ui/gtk3/', glob.glob("data/ui/gtk3/*.ui")),
        ('share/software-center/ui/gtk3/css/',
         glob.glob("data/ui/gtk3/css/*.css")),
        ('share/software-center/ui/gtk3/art/',
         glob.glob("data/ui/gtk3/art/*.png")),
        ('share/software-center/ui/gtk3/art/icons',
         glob.glob("data/ui/gtk3/art/icons/*.png")),
        ('share/software-center/default_banner',
         glob.glob("data/default_banner/*")),
        # sso
        ('share/software-center/ui/sso/', glob.glob("data/ui/sso/*.ui")),
        # dbus
        ('../etc/dbus-1/system.d/',
         ["data/dbus/com.ubuntu.SoftwareCenter.conf"]),
        ('share/dbus-1/services',
         ["data/dbus/com.ubuntu.SoftwareCenterDataProvider.service"]),
        # images
        ('share/software-center/images/',
         glob.glob("data/images/*.png") + glob.glob("data/images/*.gif")),
        ('share/software-center/icons/', glob.glob("data/emblems/*.png")),
        # xapian
        ('share/apt-xapian-index/plugins',
         glob.glob("apt_xapian_index_plugin/*.py")),
        # apport
        ('share/apport/package-hooks/', ['debian/source_software-center.py']),
        # extra software channels (can be distro specific)
        ('share/app-install/channels/',
         glob.glob("data/channels/%s/*" % distro)),
        ('lib/ubuntu-sso-client', ['software-center-sso-gtk']),
    ],
)
