/* NVClock 0.8 - Linux overclocker for NVIDIA cards
 *
 * site: http://nvclock.sourceforge.net
 *
 * Copyright(C) 2001-2004 Roderick Colenbrander
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

/* This source file uses some clock calculation code from nvidia's xfree86 driver.
   To keep Nvidia happy I have added their copyright. The way they interpret it (see linux kernel riva_hw.h)
   is that you need to add the disclaimer and copyright and when that's done
   you can basicly do what you want.
*/

 /***************************************************************************\
|*                                                                           *|
|*       Copyright 1993-2003 NVIDIA, Corporation.  All rights reserved.      *|
|*                                                                           *|
|*     NOTICE TO USER:   The source code  is copyrighted under  U.S. and     *|
|*     international laws.  Users and possessors of this source code are     *|
|*     hereby granted a nonexclusive,  royalty-free copyright license to     *|
|*     use this code in individual and commercial software.                  *|
|*                                                                           *|
|*     Any use of this source code must include,  in the user documenta-     *|
|*     tion and  internal comments to the code,  notices to the end user     *|
|*     as follows:                                                           *|
|*                                                                           *|
|*       Copyright 1993-2003 NVIDIA, Corporation.  All rights reserved.      *|
|*                                                                           *|
|*     NVIDIA, CORPORATION MAKES NO REPRESENTATION ABOUT THE SUITABILITY     *|
|*     OF  THIS SOURCE  CODE  FOR ANY PURPOSE.  IT IS  PROVIDED  "AS IS"     *|
|*     WITHOUT EXPRESS OR IMPLIED WARRANTY OF ANY KIND.  NVIDIA, CORPOR-     *|
|*     ATION DISCLAIMS ALL WARRANTIES  WITH REGARD  TO THIS SOURCE CODE,     *|
|*     INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY, NONINFRINGE-     *|
|*     MENT,  AND FITNESS  FOR A PARTICULAR PURPOSE.   IN NO EVENT SHALL     *|
|*     NVIDIA, CORPORATION  BE LIABLE FOR ANY SPECIAL,  INDIRECT,  INCI-     *|
|*     DENTAL, OR CONSEQUENTIAL DAMAGES,  OR ANY DAMAGES  WHATSOEVER RE-     *|
|*     SULTING FROM LOSS OF USE,  DATA OR PROFITS,  WHETHER IN AN ACTION     *|
|*     OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,  ARISING OUT OF     *|
|*     OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOURCE CODE.     *|
|*                                                                           *|
|*     U.S. Government  End  Users.   This source code  is a "commercial     *|
|*     item,"  as that  term is  defined at  48 C.F.R. 2.101 (OCT 1995),     *|
|*     consisting  of "commercial  computer  software"  and  "commercial     *|
|*     computer  software  documentation,"  as such  terms  are  used in     *|
|*     48 C.F.R. 12.212 (SEPT 1995)  and is provided to the U.S. Govern-     *|
|*     ment only as  a commercial end item.   Consistent with  48 C.F.R.     *|
|*     12.212 and  48 C.F.R. 227.7202-1 through  227.7202-4 (JUNE 1995),     *|
|*     all U.S. Government End Users  acquire the source code  with only     *|
|*     those rights set forth herein.                                        *|
|*                                                                           *|
 \***************************************************************************/

#include <stdio.h>
#include "nvclock.h"
#include "backend.h"

int nv40_get_pixel_pipelines(unsigned char *mask)
{
    unsigned char pipe_cfg = nv_card.PMC[0x1540/4] & 0xf;
    int i, pipelines=0;

    /* The number of enabled pixel pipelines is stored in the first 4 (or more?) bits.
    /  In case of 6800 hardware a single bit corresponds to 4 pipelines and on 6200/6600
    /  hardware a block corresponds to 2 pipelines
    */
    for(i=0; i<8; i++)
	if((pipe_cfg >> i) & 0x1)
	    pipelines++;

    *mask = pipe_cfg;

    /* If I'm correct some Geforce 6200/6600 boards use a different core for which one bit corresponds to 2 pixel pipelines instead of 4; this needs to be figured out soon */
    return pipelines*4;
}

int nv40_get_vertex_pipelines(unsigned char *mask)
{
    unsigned char pipe_cfg = (nv_card.PMC[0x1540/4]  >> 8) & 0x3f;
    int i, pipelines=0;

    /* The number of enabled vertex pipelines is stored in the second byte.
    /  A a single bit corresponds to 1 vertex pipeline.
    */
    for(i=0; i<8; i++)
	if((pipe_cfg >> i) & 0x1)
	    pipelines++;

    *mask = pipe_cfg;
    
    return pipelines;
}

void nv40_set_pixel_pipelines(unsigned char mask)
{
    int pipe_cfg = nv_card.PMC[0x1540/4];
    nv_card.PMC[0x1540/4] = ~(~pipe_cfg | 0xff) | mask;
}

void nv40_set_vertex_pipelines(unsigned char mask)
{
    int pipe_cfg = nv_card.PMC[0x1540/4];
    nv_card.PMC[0x1540/4] = ~(~pipe_cfg | 0xff00) | (mask<<8);
}

/* Fanspeed code for Geforce6800 hardware */
float nv40_get_fanspeed()
{
    /* Bit 26-16 of register 0x10f0 control the voltage for the pwm signal generator in 0.1%
    /  that is connected to the fan. By changing the value in the register the duty cycle
    /  can be controlled so that the fan turns slower or faster.
    /  The value stored in the registers needs to be inverted, so a value of 10% means 90% and so on.
    */
    float fanspeed = (1000 - ((nv_card.PMC[0x10f0/4] >> 16) & 0x3ff))/10;
    return fanspeed;
}

void nv40_set_fanspeed(float speed)
{
    int value;

    /* For safety reasons we should never disable the fan by not putting it below 10%; further negative values don't exist ;)  */
    if(speed < 10 || speed > 100)
	return;
    
    value = (nv_card.PMC[0x10f0/4] & 0xfc000000) + (((int)(1000 - speed * 10) & 0x3ff)<<16) + (nv_card.PMC[0x10f0/4] & 0xffff);
    nv_card.PMC[0x10f0/4] = value;
}

/* Fanspeed code for Geforce6600 hardware (does this work for 6200 cards too??)*/
float nv43_get_fanspeed()
{
    /* The first 8bits of register 0x15f4 control the voltage for the pwm signal generator in case
    /  of Geforce6600(GT) hardware. By changing the value in the register the duty cycle of the pwm signal
    /  can be controlled so that the fan turns slower or faster.
    /  The value stored in the registers needs to be inverted, so a value of 10% means 90% and so on. (0xff means off, 0 means on)
    */
    float fanspeed = (0xff - (nv_card.PMC[0x15f4/4] & 0xff)) * 100/0xff;
    return fanspeed;
}

void nv43_set_fanspeed(float speed)
{
    int value;

    /* For safety reasons we should never disable the fan by not putting it below 10%; further negative values don't exist ;) */
    if(speed < 10 || speed > 100)
        return;

    value = 0x80000000 + (int)((100 - speed) * 0xff/100);
    nv_card.PMC[0x15f4/4] = value;
}

float GetClock_nv40(int base_freq, unsigned int pll, unsigned int pll2)
{
    int m1, m2, n1, n2, p;

    /* mpll at 0x4020 and 0x4024; nvpll at 0x4000 and 0x4004 */	
    p = (pll >> 16) & 0x03;
    m1 = pll2 & 0xFF;
    n1 = (pll2 >> 8) & 0xFF;
    m2 = (pll2 >> 16) & 0xFF;
    n2 = (pll2 >> 24) & 0xFF;

    if(nv_card.debug)
	printf("m1=%d m2=%d n1=%d n2=%d p=%d\n", m1, m2, n1, n2, p);

    return (float)CalcSpeed_nv30(base_freq, m1, m2, n1, n2, p)/1000;
}

void ClockSelect_nv40(int clockIn, int p_current, unsigned int *pllOut, unsigned int *pllBOut)
{
    unsigned diff, diffOld;
    unsigned VClk, Freq;
    unsigned m, m2, n, n2, p = 0;
    int base_freq = 27000;
    unsigned plow = 0;

    diffOld = 0xFFFFFFFF;

    plow = p_current;
	
    VClk = (unsigned)clockIn;

    for (p = plow; p <= plow; p++) {
        Freq = VClk;
        if ((Freq >= 150000) && (Freq <= 1200000)) {
    	    for(m = 4; m <= 4; m++)
	    {
                for (m2 = 1; m2 <= 1; m2++)
		{
	    	    for(n = 1; n <= 31; n++)
		    {
            		n2 = (int)((float)((VClk << p) * m * m2) / (float)(base_freq * n)+.5);

//			printf("m1=%d m2=%d n1=%d n2=%d p=%d\n", m, m2, n, n2, p);
			/* For now this line seems to create "good" speeds, it makes sure that n/m isn't much bigger than n2/m2  */
//            		if((((n/m >= 1) && (n/m <= 8)) && ((n2/m2 >= 1) && (n2/m2 <= 8)))  && n < 31)
			if((n2 < 24) && (n > n2) && (n/m *27 < 200))
			{
                	    Freq = ((base_freq * n * n2) / (m * m2)) >> p;
                	    if (Freq > VClk)
                    		diff = Freq - VClk;
                	    else
		    		diff = VClk - Freq;

                	    /* When the difference is 0 or less than .5% accept the speed */
                	    if(((diff == 0) || ((float)diff/(float)clockIn <= 0.001)))
                	    {
				/* What do the 0x1c and 0xe mean? further there is some bit in pllOut that is sometimes 1 */
				*pllOut = 0x1c + (p << 16) + (0xe << 24);
				*pllBOut = m + (n<<8) + (m2<<16) + (n2 << 24);
                    		return;
                	    }
                	    if (diff < diffOld)
			    {
				*pllOut = 0x1c + (p << 16) + (0xe << 24);
				*pllBOut = m + (n<<8) + (m2<<16) + (n2 << 24);
                    		diffOld  = diff;
                	    }
            		}
        	    }
		}
	    }
        }
    }
}

float nv40_get_gpu_speed()
{
    int pll = nv_card.PMC[0x4000/4];
    int pll2 = nv_card.PMC[0x4004/4];
    if(nv_card.debug == 1)
    {
	printf("NVPLL_COEFF=%08x\n", pll);
	printf("NVPLL2_COEFF=%08x\n", pll2);
    }

    return (float)GetClock_nv40(nv_card.base_freq, pll, pll2);
}

void nv40_set_gpu_speed(unsigned int nvclk)
{
    unsigned int PLL, PLL2;
    int p = (nv_card.PMC[0x4000/4] >> 16) & 0x03;
    nvclk *= 1000;
	    
    ClockSelect_nv40(nvclk, p, &PLL, &PLL2);
	
    /* When no speed is found, don't change the PLL */
    /* The algorithm doesn't allow too low speeds */
    if(PLL)
    {
        if(nv_card.debug)
        {
            printf("NVPLL_COEFF: %08x\n", PLL);
            printf("NVPLL2_COEFF: %08x\n", PLL2);
        }

	/* By locking 'p' we avoid touching the first pll as I don't know much about the first one */
//        nv_card.PMC[0x4000/4] = PLL;
        nv_card.PMC[0x4004/4] = PLL2;
    }
}

float nv40_get_memory_speed()
{
    int pll = nv_card.PMC[0x4020/4];
    int pll2 = nv_card.PMC[0x4024/4];
    if(nv_card.debug == 1)
    {
	printf("MPLL_COEFF=%08x\n", pll);
	printf("MPLL2_COEFF=%08x\n", pll2);
    }

    return (float)GetClock_nv40(nv_card.base_freq, pll, pll2);
}

void nv40_set_memory_speed(unsigned int memclk)
{
    unsigned int PLL, PLL2;
    int p = (nv_card.PMC[0x4020/4] >> 16) & 0x03;
    memclk *= 1000;
	    
    ClockSelect_nv40(memclk, p, &PLL, &PLL2);
	
    /* When no speed is found, don't change the PLL */
    /* The algorithm doesn't allow too low speeds */
    if(PLL)
    {
        if(nv_card.debug)
        {
            printf("MPLL_COEFF: %08x\n", PLL);
            printf("MPLL2_COEFF: %08x\n", PLL2);
        }

//        nv_card.PMC[0x4020/4] = PLL;
        nv_card.PMC[0x4024/4] = PLL2;
    }
}

void nv40_reset_gpu_speed()
{
    /* For now only reset PLL2 as we don't touch PLL */
    /* Set the gpu speed */
    nv_card.PMC[0x4004/4] = nv_card.nvpll2;
}

void nv40_reset_memory_speed()
{
    /* For now only reset PLL2 as we don't touch PLL */
    /* Set the memory speed */
    nv_card.PMC[0x4024/4] = nv_card.mpll2;
}
