/*
 * This file is part of the GNOME-keyring signond extension
 *
 * Copyright (C) 2012 Canonical Ltd.
 *
 * Contact: Alberto Mardegan <alberto.mardegan@canonical.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 */

#include "mocked-gnome-keyring.h"
#include <gnome-keyring.h>

#define DEFAULT_KEYRING "mock-keyring"

static GHashTable *keyrings = NULL;
static guint32 last_item_id = 123;

typedef struct {
    guint32 id;
    GnomeKeyringItemType type;
    char *display_name;
    GnomeKeyringAttributeList *attributes;
    char *secret;
} MockedItem;

typedef struct {
    GList *items;
} MockedKeyring;

static MockedItem *
mocked_item_new(GnomeKeyringItemType type,
                const char *display_name,
                GnomeKeyringAttributeList *attributes,
                const char *secret)
{
    MockedItem *item = g_slice_new0(MockedItem);
    item->id = last_item_id++;
    item->type = type;
    item->display_name = g_strdup(display_name);
    item->attributes = gnome_keyring_attribute_list_copy(attributes);
    item->secret = g_strdup(secret);
    return item;
}

static gboolean
attributes_are_equal (GnomeKeyringAttribute *a1,
                      GnomeKeyringAttribute *a2)
{
    if (g_strcmp0(a1->name, a2->name) != 0) return FALSE;
    if (a1->type != a2->type) return FALSE;
    if (a1->type == GNOME_KEYRING_ATTRIBUTE_TYPE_STRING) {
        return g_strcmp0(a1->value.string, a2->value.string) == 0;
    } else {
        return a1->value.integer == a2->value.integer;
    }
}

static gboolean
mocked_item_has_attributes(MockedItem *item,
                           GnomeKeyringAttributeList *attributes)
{
    guint i, j;

    for (i = 0; i < attributes->len; i++) {
        gboolean found = FALSE;
        GnomeKeyringAttribute *required =
            &g_array_index(attributes, GnomeKeyringAttribute, i);

        for (j = 0; j < item->attributes->len; j++) {
            GnomeKeyringAttribute *attribute =
                &g_array_index(item->attributes, GnomeKeyringAttribute, j);
            if (attributes_are_equal(attribute, required)) {
                found = TRUE;
                break;
            }
        }

        if (!found) return FALSE;
    }

    return TRUE;
}

static void
mocked_item_free(MockedItem *item)
{
    g_free(item->display_name);
    gnome_keyring_attribute_list_free(item->attributes);
    g_free(item->secret);
    g_slice_free(MockedItem, item);
}

static MockedItem *
mocked_keyring_get_item(MockedKeyring *keyring, guint32 item_id)
{
    GList *list;

    for (list = keyring->items; list != NULL; list = g_list_next(list)) {
        MockedItem *item = list->data;
        if (item->id == item_id) return item;
    }

    return NULL;
}

static MockedKeyring *
mocked_keyring_new()
{
    return g_slice_new0(MockedKeyring);
}

void
mocked_gnome_keyring_init()
{
    keyrings = g_hash_table_new(g_str_hash, g_str_equal);
    g_hash_table_insert(keyrings, DEFAULT_KEYRING, mocked_keyring_new());
    /* create the keyring for use with the unit tests */
    g_hash_table_insert(keyrings, TEST_KEYRING, mocked_keyring_new());
}

GnomeKeyringResult
gnome_keyring_get_default_keyring_sync(char **keyring)
{
    *keyring = g_strdup(DEFAULT_KEYRING);
    return GNOME_KEYRING_RESULT_OK;
}

GnomeKeyringResult
gnome_keyring_item_create_sync(const char *keyring,
                               GnomeKeyringItemType type,
                               const char *display_name,
                               GnomeKeyringAttributeList *attributes,
                               const char *secret,
                               gboolean update_if_exists,
                               guint32 *item_id)
{
    GList *list, *found = NULL;
    GnomeKeyringFound *existing_result = NULL;
    MockedKeyring *mocked_keyring;

    g_return_val_if_fail (keyring != NULL,
                          GNOME_KEYRING_RESULT_BAD_ARGUMENTS);
    g_return_val_if_fail (type < GNOME_KEYRING_ITEM_LAST_TYPE,
                          GNOME_KEYRING_RESULT_BAD_ARGUMENTS);

    mocked_keyring = g_hash_table_lookup (keyrings, keyring);
    if (G_UNLIKELY (mocked_keyring == NULL))
        return GNOME_KEYRING_RESULT_NO_SUCH_KEYRING;

    gnome_keyring_find_items_sync(type, attributes, &found);
    for (list = found; list != NULL; list = g_list_next(list)) {
        GnomeKeyringFound *result = list->data;
        if (g_strcmp0(result->keyring, keyring) == 0) {
            existing_result = result;
            break;
        }
    }
    gnome_keyring_found_list_free(found);

    if (existing_result != NULL) {
        if (!update_if_exists) return GNOME_KEYRING_RESULT_ALREADY_EXISTS;

        /* Update the existing item */
        MockedItem *item = mocked_keyring_get_item(mocked_keyring,
                                                   existing_result->item_id);
        g_assert(item != NULL);

        g_free(item->display_name);
        item->display_name = g_strdup(display_name);

        g_free(item->secret);
        item->secret = g_strdup(secret);

        if (item_id != NULL)
            *item_id = item->id;
    } else {
        MockedItem *item =
            mocked_item_new(type, display_name, attributes, secret);
        if (item_id != NULL)
            *item_id = item->id;

        mocked_keyring->items = g_list_prepend(mocked_keyring->items,
                                               item);
    }
    return GNOME_KEYRING_RESULT_OK;
}

GnomeKeyringResult
gnome_keyring_find_items_sync(GnomeKeyringItemType type,
                              GnomeKeyringAttributeList *attributes,
                              GList **found)
{
    GHashTableIter iter;
    const gchar *keyring_name;
    MockedKeyring *keyring;
    GList *results = NULL;

    g_return_val_if_fail (type < GNOME_KEYRING_ITEM_LAST_TYPE,
                          GNOME_KEYRING_RESULT_BAD_ARGUMENTS);

    g_hash_table_iter_init(&iter, keyrings);
    while (g_hash_table_iter_next(&iter,
                                  (gpointer)&keyring_name,
                                  (gpointer)&keyring)) {
        GList *list;

        for (list = keyring->items; list != NULL; list = g_list_next(list)) {
            GnomeKeyringFound *found;
            MockedItem *item = list->data;
            if (item->type != type) continue;

            if (!mocked_item_has_attributes(item, attributes)) continue;

            found = g_slice_new0(GnomeKeyringFound);
            found->keyring = g_strdup(keyring_name);
            found->item_id = item->id;
            found->attributes =
                gnome_keyring_attribute_list_copy(item->attributes);
            found->secret = g_strdup(item->secret);
            results = g_list_prepend(results, found);
        }
    }

    *found = results;
    return results != NULL ?
        GNOME_KEYRING_RESULT_OK : GNOME_KEYRING_RESULT_NO_MATCH;
}

void
gnome_keyring_found_free(GnomeKeyringFound *found)
{
    g_free(found->keyring);
    gnome_keyring_attribute_list_free(found->attributes);
    g_free(found->secret);
    g_slice_free(GnomeKeyringFound, found);
}

GnomeKeyringResult
gnome_keyring_item_get_info_sync(const char *keyring,
                                 guint32 id,
                                 GnomeKeyringItemInfo **info)
{
    GnomeKeyringItemInfo *result;
    MockedKeyring *mocked_keyring;
    MockedItem *item;

    g_return_val_if_fail (keyring != NULL,
                          GNOME_KEYRING_RESULT_BAD_ARGUMENTS);

    mocked_keyring = g_hash_table_lookup (keyrings, keyring);
    if (G_UNLIKELY (mocked_keyring == NULL))
        return GNOME_KEYRING_RESULT_NO_SUCH_KEYRING;

    item = mocked_keyring_get_item(mocked_keyring, id);
    if (item == NULL) return GNOME_KEYRING_RESULT_NO_MATCH;

    result = gnome_keyring_item_info_new();
    gnome_keyring_item_info_set_type(result, item->type);
    gnome_keyring_item_info_set_display_name(result, item->display_name);
    gnome_keyring_item_info_set_secret(result, item->secret);
    *info = result;
    return GNOME_KEYRING_RESULT_OK;
}

GnomeKeyringResult
gnome_keyring_item_delete_sync(const char *keyring,
                               guint32 item_id)
{
    MockedKeyring *mocked_keyring;
    MockedItem *item;
    GList *list;

    g_return_val_if_fail (keyring != NULL,
                          GNOME_KEYRING_RESULT_BAD_ARGUMENTS);

    mocked_keyring = g_hash_table_lookup (keyrings, keyring);
    if (G_UNLIKELY (mocked_keyring == NULL))
        return GNOME_KEYRING_RESULT_NO_SUCH_KEYRING;

    for (list = mocked_keyring->items;
         list != NULL;
         list = g_list_next(list)) {
        item = list->data;
        if (item->id == item_id) break;
    }

    if (list == NULL) return GNOME_KEYRING_RESULT_NO_MATCH;
    mocked_item_free(item);
    mocked_keyring->items = g_list_delete_link(mocked_keyring->items, list);
    return GNOME_KEYRING_RESULT_OK;
}
