/* -*- Mode: C++; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 * This file is part of the GNOME-keyring signond extension
 *
 * Copyright (C) 2011 Canonical Ltd.
 *
 * Contact: Alberto Mardegan <alberto.mardegan@canonical.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 */

#include "debug.h"
#include "secrets-storage.h"

#include <QDataStream>
#include <gnome-keyring.h>

using namespace SignOn;

SecretsStorage::SecretsStorage(QObject *parent) :
    AbstractSecretsStorage(parent),
    m_keyringName()
{
    TRACE() << "Constructed";
}

SecretsStorage::~SecretsStorage()
{
    TRACE() << "Destroyed";
}

bool SecretsStorage::initialize(const QVariantMap &configuration)
{
    if (configuration.contains(QLatin1String("KeyringName"))) {
        m_keyringName = configuration.value(QLatin1String("KeyringName")).toByteArray();
    } else {
        GnomeKeyringResult ret;
        gchar *name;

        ret = gnome_keyring_get_default_keyring_sync(&name);
        if (ret != GNOME_KEYRING_RESULT_OK) {
            BLAME() << "Couldn't get default keyring name:" << ret;
            return false;
        }

        m_keyringName = QByteArray(name);
        g_free(name);
    }
    TRACE() << "Using keyring:" << m_keyringName;

    setIsOpen(true);
    return true;
}

bool SecretsStorage::clear()
{
    if (!removeSecrets(Password, 0, 0, TypeField))
        return false;

    if (!removeSecrets(Username, 0, 0, TypeField))
        return false;

    if (!removeSecrets(Data, 0, 0, TypeField))
        return false;

    return true;
}

bool SecretsStorage::updateCredentials(const quint32 id,
                                       const QString &username,
                                       const QString &password)
{
    if (!username.isEmpty() &&
        !storeSecret(Username, id, 0, username.toUtf8()))
        return false;

    if (!password.isEmpty() &&
        !storeSecret(Password, id, 0, password.toUtf8()))
        return false;

    return true;
}

bool SecretsStorage::removeCredentials(const quint32 id)
{
    return removeSecrets(NoType, id, 0, IdField);
}

bool SecretsStorage::loadCredentials(const quint32 id,
                                     QString &username,
                                     QString &password)
{
    QByteArray baUsername, baPassword;
    bool found = false;

    if (loadSecret(Username, id, 0, baUsername)) {
        username = QString::fromUtf8(baUsername.constData());
        found = true;
    } else {
        username = QString();
    }

    if (loadSecret(Password, id, 0, baPassword)) {
        password = QString::fromUtf8(baPassword.constData());
        found = true;
    } else {
        password = QString();
    }

    return found;
}

QVariantMap SecretsStorage::loadData(quint32 id, quint32 method)
{
    QByteArray serializedData;
    bool ret;

    ret = loadSecret(Data, id, method, serializedData);
    if (!ret) return QVariantMap();

    QDataStream stream(QByteArray::fromPercentEncoding(serializedData));
    QVariantMap data;
    stream >> data;
    return data;
}

bool SecretsStorage::storeData(quint32 id, quint32 method,
                               const QVariantMap &data)
{
    QByteArray serializedData;
    QDataStream stream(&serializedData, QIODevice::WriteOnly);
    stream << data;
    return storeSecret(Data, id, method, serializedData.toPercentEncoding());
}

bool SecretsStorage::removeData(quint32 id, quint32 method)
{
    return removeSecrets(Data, id, method, IdField | MethodField | TypeField);
}

bool SecretsStorage::storeSecret(SignonSecretType type,
                                 quint32 id,
                                 quint32 method,
                                 const QByteArray &secret)
{
    GnomeKeyringResult ret;
    guint32 createdItemId;

    TRACE() << "Storing secret:" << id <<
        "type:" << type <<
        "method:" << method;

    GnomeKeyringAttributeList *attributes = gnome_keyring_attribute_list_new();
    gnome_keyring_attribute_list_append_uint32(attributes, "signon-type", type);
    gnome_keyring_attribute_list_append_uint32(attributes, "signon-id", id);
    if (type == Data) {
        gnome_keyring_attribute_list_append_uint32(attributes,
                                                   "signon-method", method);
    }

    QString displayName =
        QString::fromLatin1("Ubuntu Web Account: id %1-%2").arg(id).arg(type);
    QByteArray ba = displayName.toUtf8();

    ret = gnome_keyring_item_create_sync(keyring(),
                                         GNOME_KEYRING_ITEM_GENERIC_SECRET,
                                         ba.constData(),
                                         attributes,
                                         secret.constData(),
                                         TRUE,
                                         &createdItemId);
    gnome_keyring_attribute_list_free(attributes);
    if (ret != GNOME_KEYRING_RESULT_OK) {
        TRACE() << "Got error from GNOME keyring:" << ret;
        return false;
    }

    Q_UNUSED(createdItemId);
    return true;
}

bool SecretsStorage::loadSecret(SignonSecretType type,
                                quint32 id,
                                quint32 method,
                                QByteArray &secret)
{
    GList *found = 0;
    GnomeKeyringResult ret;

    TRACE() << "id:" << id << "type:" << type << "method:" << method;
    GnomeKeyringAttributeList *attributes = gnome_keyring_attribute_list_new();
    gnome_keyring_attribute_list_append_uint32(attributes, "signon-type", type);
    gnome_keyring_attribute_list_append_uint32(attributes, "signon-id", id);
    if (type == Data) {
        gnome_keyring_attribute_list_append_uint32(attributes, "signon-method",
                                                   method);
    }

    ret = gnome_keyring_find_items_sync(GNOME_KEYRING_ITEM_GENERIC_SECRET,
                                        attributes,
                                        &found);
    gnome_keyring_attribute_list_free(attributes);
    if (ret != GNOME_KEYRING_RESULT_OK) {
        TRACE() << "Credentials loading failed:" << ret;
        return false;
    }

    for (GList *list = found; list != 0; list = list->next) {
        GnomeKeyringFound *result = (GnomeKeyringFound *)list->data;

        if (!isActiveKeyring(result->keyring))
            continue;

        GnomeKeyringItemInfo *info;
        ret = gnome_keyring_item_get_info_sync(result->keyring,
                                               result->item_id,
                                               &info);
        if (ret != GNOME_KEYRING_RESULT_OK) {
            TRACE() << "Error loading credentials:" << ret;
            break;
        }

        gchar *data = gnome_keyring_item_info_get_secret(info);
        secret = QByteArray(data);
        g_free(data);
        gnome_keyring_item_info_free(info);
    }
    gnome_keyring_found_list_free(found);

    return ret == GNOME_KEYRING_RESULT_OK;
}

bool SecretsStorage::removeSecrets(SignonSecretType type,
                                   quint32 id,
                                   quint32 method,
                                   QueryFields fields)
{
    GList *found = 0;
    GnomeKeyringResult ret;

    GnomeKeyringAttributeList *attributes = gnome_keyring_attribute_list_new();
    if (fields & IdField) {
        gnome_keyring_attribute_list_append_uint32(attributes,
                                                   "signon-id", id);
    }
    if (fields & MethodField) {
        gnome_keyring_attribute_list_append_uint32(attributes,
                                                   "signon-method", method);
    }
    if (fields & TypeField) {
        gnome_keyring_attribute_list_append_uint32(attributes,
                                                   "signon-type", type);
    }

    ret = gnome_keyring_find_items_sync(GNOME_KEYRING_ITEM_GENERIC_SECRET,
                                        attributes,
                                        &found);
    if (ret != GNOME_KEYRING_RESULT_OK) {
        if (ret == GNOME_KEYRING_RESULT_NO_MATCH) {
            return true;
        }
        TRACE() << "Credentials search failed:" << ret;
        return false;
    }

    for (GList *list = found; list != 0; list = list->next) {
        GnomeKeyringFound *result = (GnomeKeyringFound *)list->data;

        if (!isActiveKeyring(result->keyring))
            continue;

        ret = gnome_keyring_item_delete_sync(result->keyring, result->item_id);
        if (ret != GNOME_KEYRING_RESULT_OK) {
            TRACE() << "Error deleting credentials:" << ret;
            break;
        }
    }
    gnome_keyring_found_list_free(found);

    return ret == GNOME_KEYRING_RESULT_OK;
}

const char *SecretsStorage::keyring() const
{
    return m_keyringName.isEmpty() ? 0 : m_keyringName.constData();
}

bool SecretsStorage::isActiveKeyring(const char *keyringName) const
{
    /* This method is needed not to apply the same hack in different parts of
     * the code.
     */
    const gchar *activeKeyringName = keyring();
    if (qstrcmp(keyringName, activeKeyringName) == 0)
        return true;

    /* Unfortunately GNOME keyring doesn't return the proper name of the
     * default keyring (it returns NULL). This means that if the active
     * keyring is the default keyring the above string comparison would fail.
     * In other words, if the current keyring is NULL, we have no clue on what
     * to return from this method.
     * https://bugzilla.gnome.org/show_bug.cgi?id=664454
     *
     * If the current keyring is NULL, we just return true.
     */
    if (activeKeyringName == 0) {
        return true;
    } else {
        return false;
    }
}

