/*  Screem:  screem-dtd_db.c
 *
 *  Copyright (C) 2003 David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 */

#include "config.h"

#include <libgnomevfs/gnome-vfs-utils.h>

#include <gconf/gconf-client.h>

#include <glib/ghash.h>
#include <glib/gstring.h>
#include <glib/gunicode.h>

#include <gtk/gtkcombo.h>

#include <libxml/tree.h>

#include <dirent.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>

#include "screem-dtd-db.h"
#include "fileops.h"
#include "support.h"

static void screem_dtd_load_catalog( ScreemDTDDB *db,
		const gchar *pathname );
static void screem_dtd_load_catalogs( ScreemDTDDB *db );
static gint screem_dtd_db_sort_compare( const gchar *a, const gchar *b);
static void dtds_insert( gpointer key, gpointer value, GObject *widget);
static void screem_dtd_db_offline_notify( GConfClient *client,
					guint cnxn_id,
					GConfEntry *entry,
					gpointer data );

static xmlNodePtr screem_dtd_db_lookup_override( ScreemDTDDB *db,
						const gchar *publicid );
static gchar *screem_dtd_db_override_get_root( ScreemDTDDB *db,
						const gchar *publicid );
static gchar *screem_dtd_db_override_get_system_id( ScreemDTDDB *db,
						const gchar *publicid );
static ScreemDTD *screem_dtd_db_new_dtd( ScreemDTDDB *db,
					 const gchar *publicid,
					 const gchar *systemid );
static gboolean screem_dtd_db_load_internal( ScreemDTDDB *db,
					const gchar *publicid,
					const gchar *systemid );

static void screem_dtd_db_set_hidden( ScreemDTDDB *db, 
		const gchar *publicid, gboolean hide );
static void screem_dtd_db_row_changed( GtkTreeModel *model,
		GtkTreePath *path, GtkTreeIter *iter,
		ScreemDTDDB *db );


struct ScreemDTDDBPrivate {
	GHashTable *loaded_dtds;
	GHashTable *dtd_types;

	xmlDocPtr  overrides;
	gchar *overridepath;

	GtkListStore *store;

	gboolean offline;
	guint notify;
};

G_DEFINE_TYPE( ScreemDTDDB, screem_dtd_db, G_TYPE_OBJECT )

static void screem_dtd_db_finalize( GObject *object );
static void screem_dtd_db_set_prop( GObject *object, guint prop_id,
				const GValue *value, GParamSpec *spec );
static void screem_dtd_db_get_prop( GObject *object, guint prop_id,
				GValue *value, GParamSpec *spec );

static void screem_dtd_db_class_init( ScreemDTDDBClass *klass )
{
	GObjectClass *obj_class;
	
	obj_class = G_OBJECT_CLASS( klass );
	obj_class->finalize = screem_dtd_db_finalize;
	obj_class->get_property = screem_dtd_db_get_prop;
	obj_class->set_property = screem_dtd_db_set_prop;
}


static void screem_dtd_db_init( ScreemDTDDB *dtd_db )
{
	ScreemDTDDBPrivate *priv;
	GConfClient *client;
	gchar *tmp;
	xmlNodePtr node;
	xmlDtdPtr dtd;
	
	priv = dtd_db->priv = g_new0( ScreemDTDDBPrivate, 1 );

	client = gconf_client_get_default();

	priv->notify = gconf_client_notify_add( client,
					 "/apps/screem/general/work_offline", 
					 screem_dtd_db_offline_notify,
					 dtd_db, NULL, NULL );
	priv->offline = gconf_client_get_bool( client,
					"/apps/screem/general/work_offline",
					NULL );

	priv->dtd_types = g_hash_table_new( g_str_hash, g_str_equal );
	priv->loaded_dtds = g_hash_table_new( g_str_hash, g_str_equal );

	tmp = screem_get_dot_dir();
	priv->overridepath = g_build_filename( tmp, "doctypes.xml",
			NULL );
	g_free( tmp );

	priv->overrides = NULL;
	if( uri_exists( priv->overridepath, NULL ) ) {
		priv->overrides = xmlParseFile( priv->overridepath );
	}
	if( ! priv->overrides && uri_exists( DATADIR"/screem/doctypes.xml", NULL ) ) {
		/* no user override file, load system overrides */
		priv->overrides = xmlParseFile( DATADIR"/screem/doctypes.xml" );
		if( priv->overrides ) {
			xmlSaveFile( priv->overridepath, 
					priv->overrides );
		}
	}
	if( ! priv->overrides ) {
		/* no overrides file, create a new doc */

		g_print( "No system or user doctypes file, all dtds will be hidden by default\n" );
		
		priv->overrides = xmlNewDoc( (const xmlChar*)XML_DEFAULT_VERSION );
		dtd = xmlCreateIntSubset( priv->overrides,
				(const xmlChar*)"doctypes",
				(const xmlChar*)"-//Screem/DTD doctypes 1.0",
				(const xmlChar*)"http://www.screem.org/dtd/screem-doctypes.dtd" );
		node = xmlNewDocNode( priv->overrides,
				NULL, /* ns */ (const xmlChar*)"doctypes", NULL );
		xmlDocSetRootElement( priv->overrides, node );
		xmlSaveFile( priv->overridepath, priv->overrides );
	}

	screem_dtd_load_catalogs( dtd_db );

	priv->store = gtk_list_store_new( SCREEM_DTD_DB_MAX_COL, 
			G_TYPE_STRING, /* public */
			G_TYPE_STRING, /* system */
			G_TYPE_STRING, /* root */
			G_TYPE_BOOLEAN /* active */ );
	g_signal_connect( G_OBJECT( priv->store ), "row_changed",
			G_CALLBACK( screem_dtd_db_row_changed ),
			dtd_db );
	screem_dtd_db_fill_list( dtd_db, priv->store );
}

static gboolean screem_dtd_db_finalize_loaded( gpointer key,
						gpointer value,
						gpointer data )
{
	g_free( key );
	g_object_unref( G_OBJECT( value ) );

	return FALSE;
}

static gboolean screem_dtd_db_finalize_dtd_types( gpointer key,
						gpointer value,
						gpointer data )
{
	g_free( key );
	g_free( value );

	return FALSE;
}

static void screem_dtd_db_finalize( GObject *object )
{
	ScreemDTDDB *dtd_db;
	ScreemDTDDBPrivate *priv;
	GConfClient *client;
			
	g_return_if_fail( object != NULL );
	g_return_if_fail( SCREEM_IS_DTD_DB( object ) );

	dtd_db = SCREEM_DTD_DB( object );

	priv = dtd_db->priv;

	g_free( priv->overridepath );
	xmlFreeDoc( priv->overrides );
	g_object_unref( priv->store );

	g_hash_table_foreach( priv->loaded_dtds,
			(GHFunc)screem_dtd_db_finalize_loaded,
			NULL );
	g_hash_table_destroy( priv->loaded_dtds );

	g_hash_table_foreach( priv->dtd_types,
			(GHFunc)screem_dtd_db_finalize_dtd_types,
			NULL );
	g_hash_table_destroy( priv->dtd_types );

	
	client = gconf_client_get_default();
	gconf_client_notify_remove( client, priv->notify );
	g_object_unref( client );

	g_free( priv );
	
	G_OBJECT_CLASS( screem_dtd_db_parent_class )->finalize( object );
}

static void screem_dtd_db_set_prop( GObject *object, guint prop_id,
				const GValue *value, GParamSpec *spec )
{
	ScreemDTDDB *dtd_db;
	ScreemDTDDBPrivate *priv;
	
	dtd_db = SCREEM_DTD_DB( object );
	priv = dtd_db->priv;

	switch( prop_id ) {
		default:
			break;
	}
}

static void screem_dtd_db_get_prop( GObject *object, guint prop_id,
				GValue *value, GParamSpec *spec )
{
	ScreemDTDDB *dtd_db;
	ScreemDTDDBPrivate *priv;
	
	dtd_db = SCREEM_DTD_DB( object );
	priv = dtd_db->priv;

	switch( prop_id ) {
		default:
			break;
	}
}

/* static stuff */

static void screem_dtd_load_catalog( ScreemDTDDB *db,
		const gchar *pathname )
{
	gchar buffer[ BUFSIZ ];
	FILE *file;

	gchar id[ BUFSIZ ];
	gchar path[ BUFSIZ ];

	gchar *tid;
	gchar *tpath;
	gchar *temp;
	gint ret;

	/* we have a catalog file */
	file = fopen( pathname, "r" );
	if( ! file )
		return;

	while( fgets( buffer, BUFSIZ, file ) ) {
		g_strchug( buffer );
		if( ! strncmp( "CATALOG ", buffer, strlen( "CATALOG " ) ) ) {
			/* another catalog file to check */
			buffer[ strlen( buffer) - 1 ] = '\0';
			screem_dtd_load_catalog( db, buffer + strlen( "CATALOG " ) );
		} else if( ! strncmp( "PUBLIC ", buffer, strlen( "PUBLIC " ) ) ) {
			ret = sscanf( buffer, "PUBLIC \"%[^\"]\" \"%[^\"]\"", 
				      id, path );
			if( ! ret ) {
				break;
			} else if( ret == 1 ) {
				/* ok try a different match */
				if( sscanf( buffer, "PUBLIC \"%[^\"]\" %s", 
					    id, path ) != 2 )
					break;
			}
		
			/* is id already in the hash table? */
			if( ! g_hash_table_lookup( db->priv->dtd_types, id ) ) {
				/* nope */
				gchar *tmp;
			
				tid = g_strdup( id );
				tpath = gnome_vfs_escape_host_and_path_string( path );
				if( ! g_path_is_absolute( path ) ) {
					tmp = g_path_get_dirname( pathname );
					temp = g_strconcat( "file://", 
							tmp,
							G_DIR_SEPARATOR_S,
							tpath, NULL );
					g_free( tmp );
				} else {
					temp = g_strconcat( "file://", 
							tpath, NULL );
				}
				g_free( tpath );
				tpath = temp;
				g_hash_table_insert( db->priv->dtd_types, tid, tpath );
			}
		}
	}
	fclose( file );
}

static void screem_dtd_load_catalogs( ScreemDTDDB *db )
{
	gchar *paths[] = {
		SCREEM_DTD_PATH,
		"/etc/sgml",
		"/usr/lib/sgml",                      /* Debian, RedHat */
		"/usr/share/sgml",                    /* SuSE */
		"/usr/share/lib/sgml/locale/C/dtds",  /* Solaris, use the 
							 current locale at 
							 runtime? */
		NULL,                                 /* replaced with the
							 users .screem dir at
							 runtime */
		NULL
	};

	gint i;
	gchar *file;

	DIR *dir;
	struct dirent *entry;

	gchar *tmp;
	
	tmp = screem_get_dot_dir();
	paths[ 4 ] = g_strconcat( tmp, G_DIR_SEPARATOR_S,
				  "dtds", NULL );
	g_free( tmp );

	/* make dtds dir if needed */
	mkdir_recursive( paths[ 4 ], GNOME_VFS_PERM_USER_ALL,
			NULL, NULL );

	for( i = 0; paths[ i ]; ++ i ) {
		if( ! ( dir = opendir( paths[ i ] ) ) ) {
			continue;
		}

		/* get next file */
		while( ( entry = readdir( dir ) ) ) {
			if( ( ! g_strcasecmp( "catalog", entry->d_name ) ) ||
			    ( ! g_strncasecmp( "catalog.", 
					       entry->d_name, 
					       strlen( "catalog." ) ) ) ) {
				file = g_strconcat( paths[ i ],
						G_DIR_SEPARATOR_S,
						entry->d_name,
						NULL );
				screem_dtd_load_catalog( db, file );
				g_free( file );
			}
		}
		closedir( dir );
	}
	
	g_free( paths[ 4 ] );
}

static gint screem_dtd_db_sort_compare( const gchar *a, const gchar *b )
{
	gint ret;
		
	ret = 0;

	if( a && ! b ) {
		ret = -1;
	} else if( b && ! a ) {
		ret = 1;
	} else {
		const gchar *htmla;
		const gchar *htmlb;
		
		htmla = strstr( a, "HTML" );
		htmlb = strstr( b, "HTML" );
		if( ! htmla ) {
			htmla = strstr( a, "html" );
		}
		if( ! htmlb ) {
			htmlb = strstr( b, "html" );
		}
		if( htmla && ! htmlb ) {
			ret = -1;
		} else if( htmlb && ! htmla ) {
			ret = 1;
		} else {
			ret = strcmp( a, b );
		}
	}

	return ret;
}

static void dtds_insert( gpointer key, gpointer value, GObject *widget )
{
	if( ! strstr( (gchar*)key, "ENTITIES" ) ) {
		GList *list;

		list = (GList*)g_object_get_data( G_OBJECT( widget ), "list" );

		list = g_list_prepend( list, key );

		g_object_set_data( G_OBJECT( widget ), "list", list );
	}
}

static void screem_dtd_db_offline_notify( GConfClient *client,
					guint cnxn_id,
					GConfEntry *entry,
					gpointer data )
{
	ScreemDTDDB *db;

	db = SCREEM_DTD_DB( data );

	if( entry->value && entry->value->type == GCONF_VALUE_BOOL ) {
		gboolean state;

		state = gconf_value_get_bool( entry->value );
		db->priv->offline = state;
	}
}

static xmlNodePtr screem_dtd_db_lookup_override( ScreemDTDDB *db,
					  const gchar *publicid )
{
	xmlNodePtr ret;
	xmlChar *val;
	
	g_return_val_if_fail( SCREEM_IS_DTD_DB( db ), NULL );

	ret = xmlDocGetRootElement( db->priv->overrides );

	if( ret ) {
		for( ret = ret->children; ret; ret = ret->next ) {
			val = NULL;
			if( ! strcmp( "doctype", (gchar*)ret->name ) ) {
				val = xmlGetProp( ret, 
						(const xmlChar*)"public" ); 
			}
			if( val && ! strcmp( (gchar*)val, publicid ) ) {
				xmlFree( val );
				break;
			}
			xmlFree( val );
		}
	}

	return ret;
}

static gchar *screem_dtd_db_override_get_root( ScreemDTDDB *db,
						const gchar *publicid )
{
	gchar *ret;
	xmlNodePtr override;

	g_return_val_if_fail( SCREEM_IS_DTD_DB( db ), NULL );
	g_return_val_if_fail( publicid != NULL, NULL );

	ret = NULL;
	override = screem_dtd_db_lookup_override( db, publicid );
	if( override ) {
		ret = (gchar*)xmlGetProp( override, (const xmlChar*)"root" );
	}

	return ret;
}

static gchar *screem_dtd_db_override_get_system_id( ScreemDTDDB *db,
						const gchar *publicid )
{
	gchar *ret;
	xmlNodePtr override;

	g_return_val_if_fail( SCREEM_IS_DTD_DB( db ), NULL );
	g_return_val_if_fail( publicid != NULL, NULL );

	ret = NULL;
	override = screem_dtd_db_lookup_override( db, publicid );
	if( override ) {
		ret = (gchar*)xmlGetProp( override, (const xmlChar*)"system" );
	}

	return ret;
}

static gboolean screem_dtd_db_get_hidden( ScreemDTDDB *db, 
		const gchar *publicid )
{
	xmlChar *attr;
	xmlNodePtr override;
	gboolean ret;

	g_return_val_if_fail( SCREEM_IS_DTD_DB( db ), TRUE );
	g_return_val_if_fail( publicid != NULL, TRUE );

	ret = TRUE;
	override = screem_dtd_db_lookup_override( db, publicid );
	if( override ) {
		attr = xmlGetProp( override, (const xmlChar*)"active" );
		if( attr ) {
			ret = (*attr != '1');
			xmlFree( attr );
		}
	}

	return ret;

}

static ScreemDTD *screem_dtd_db_new_dtd( ScreemDTDDB *db,
					 const gchar *publicid,
					 const gchar *systemid )
{
	ScreemDTDDBPrivate *priv;
	ScreemDTD *dtd;
	gchar *data;
	ScreemDTDParse parse;

	g_return_val_if_fail( SCREEM_IS_DTD_DB( db ), NULL );
	g_return_val_if_fail( publicid != NULL ||
			      systemid != NULL, NULL );

	priv = db->priv;

	/* check we aren't creating a new dtd for one
	   we already have */
	dtd = g_hash_table_lookup( priv->loaded_dtds, systemid );

	if( dtd ) {
		g_warning( "DTD %s already loaded\n", systemid );
		return dtd;
	}

	dtd = NULL;
	data = screem_dtd_db_resolve_entity( db, publicid, systemid );
	if( data ) {
		dtd = screem_dtd_new( publicid, systemid );

		/* parse data, adding to dtd */
		parse.dtd = dtd;
		parse.resolve_entity = screem_dtd_db_resolve_entity;
		parse.userdata = db;
		
		if( ! screem_dtd_parse( dtd, &parse, data ) ) {
			g_object_unref( dtd );
			dtd = NULL;
		} else {
			g_hash_table_insert( priv->loaded_dtds,
				     g_strdup( systemid ),
				     dtd );
		}

		g_free( data );
	}

	return dtd;
}

gchar *screem_dtd_db_resolve_entity( void *db,
				  const gchar *publicid,
				  const gchar *systemid )
{
	ScreemDTDDB *dtddb;
	ScreemDTDDBPrivate *priv;
	const gchar *localuri;
	gchar *ret;
	GnomeVFSURI *uri;
	gboolean local;

	g_return_val_if_fail( SCREEM_IS_DTD_DB( db ), NULL );

	dtddb = SCREEM_DTD_DB( db );

	priv = dtddb->priv;
	localuri = NULL;	
	if( publicid ) {
		localuri = g_hash_table_lookup( priv->dtd_types, 
						publicid );

		if( ! localuri ) {
			/* no local copy of publicid, or we haven't
			   loaded a catalog with it in */

			if( screem_dtd_db_load_internal( dtddb, 
						publicid, systemid ) ) {
				localuri = g_hash_table_lookup( priv->dtd_types,
								publicid );
			}
		}
	} else if( systemid ) {
		/* there was no public identifier, there was
		   a system id, so just load that */
		uri = gnome_vfs_uri_new( systemid );
		if( uri ) {
			local = gnome_vfs_uri_is_local( uri );
			if( ( ! priv->offline ) || local ) {
				localuri = systemid;
			}
			gnome_vfs_uri_unref( uri );
		}
	}

	ret = NULL;

	if( localuri ) {
		ret = load_file( localuri, NULL, NULL, NULL );
	}

	return ret;
}

static gboolean screem_dtd_db_load_internal( ScreemDTDDB *db,
					const gchar *publicid,
					const gchar *systemid )
{
	ScreemDTDDBPrivate *priv;
	GnomeVFSURI *uri;
	gchar *file;
	gchar *catfile;
	gboolean local;
	gchar *filename;
	gchar *tmp;
	gchar *path;
	gchar *catalog;
	gchar *sysid;
	gboolean ret;

	g_return_val_if_fail( SCREEM_IS_DTD_DB( db ), FALSE );

	if( ! systemid ) {
		return FALSE;
	}
	
	ret = FALSE;

	priv = db->priv;

	file = NULL;
	local = FALSE;
	uri = NULL;
	if( systemid ) {
		uri = gnome_vfs_uri_new( systemid );
	}
	if( uri ) {
		local = gnome_vfs_uri_is_local( uri );
		if( ( ! priv->offline ) || local ) {
			file = load_file( systemid, NULL, NULL, NULL );
		}
		gnome_vfs_uri_unref( uri );
	}
	
	if( file && publicid ) {
		filename = gnome_vfs_escape_slashes( systemid );
		tmp = screem_get_dot_dir();
		path = g_build_filename( tmp, "dtds", NULL );
		g_free( tmp );
		tmp = g_build_filename( path, filename, NULL );
		sysid = filename;

		filename = gnome_vfs_escape_host_and_path_string( tmp );
		g_free( tmp );
		
		if( save_file( filename, file,
				GNOME_VFS_PERM_USER_READ |
				GNOME_VFS_PERM_USER_WRITE,
				FALSE, FALSE,
				NULL ) ) {
			catalog = g_build_filename( path, "catalog",
						    NULL );
			catfile = load_file( catalog, NULL, NULL, NULL );
			
			if( catfile ) {
				tmp = g_strdup_printf( "%s\nPUBLIC \"%s\" \"%s\"\n",
						catfile, publicid, 
						sysid );
					
			} else {
				tmp = g_strdup_printf( "PUBLIC \"%s\" \"%s\"\n",
						publicid, sysid );
			}
			save_file( catalog, tmp,
				   GNOME_VFS_PERM_USER_READ |
				   GNOME_VFS_PERM_USER_WRITE, 
				   FALSE, FALSE,
				   NULL );
			g_free( catfile );
			g_free( tmp );
			g_free( catalog );
		}
		g_free( path );
		g_free( sysid );

		g_hash_table_insert( priv->dtd_types,
				     g_strdup( publicid ),
				     filename );
		ret = TRUE;
	}
	g_free( file );

	return ret;
}

static void screem_dtd_db_set_hidden( ScreemDTDDB *db, 
		const gchar *publicid, gboolean hide )
{
	xmlNodePtr droot;
	xmlNodePtr override;
	
	g_return_if_fail( SCREEM_IS_DTD_DB( db ) );
	g_return_if_fail( publicid != NULL );

	override = screem_dtd_db_lookup_override( db, publicid );
	if( ! override ) {
		override = xmlNewNode( NULL, (const xmlChar*)"doctype" );
		xmlSetProp( override, (const xmlChar*)"public", 
				(xmlChar*)publicid );
		droot = xmlDocGetRootElement( db->priv->overrides );
		xmlAddChild( droot, override );
	}

	if( ! hide ) {
		xmlSetProp( override, (const xmlChar*)"active", 
				(const xmlChar*)"1" );
	} else {
		xmlUnsetProp( override, (const xmlChar*)"active" );
	}

	xmlSaveFile( db->priv->overridepath, db->priv->overrides );
}

static void screem_dtd_db_row_changed( GtkTreeModel *model,
		GtkTreePath *path, GtkTreeIter *iter,
		ScreemDTDDB *db )
{
	gchar *public;
	gboolean active;

	gtk_tree_model_get( model, iter,
			SCREEM_DTD_DB_PUBLIC_COL, &public,
			SCREEM_DTD_DB_ACTIVE_COL, &active,
			-1 );
	if( public ) {
		if( active == screem_dtd_db_get_hidden( db, public ) ) {
			screem_dtd_db_set_hidden( db, public, ! active );
		}
		g_free( public );
	}
}

/* public stuff */

ScreemDTDDB *screem_dtd_db_new( void )
{
	ScreemDTDDB *dtd_db;

	dtd_db = g_object_new( SCREEM_TYPE_DTD_DB, NULL );

	return dtd_db;
}

GtkListStore *screem_dtd_db_get_store( ScreemDTDDB *db )
{
	g_return_val_if_fail( SCREEM_IS_DTD_DB( db ), NULL );

	return db->priv->store;
}

ScreemDTD *screem_dtd_db_get_default_dtd( ScreemDTDDB *db,
		const gchar *mime, const gchar *root )
{
	ScreemDTD *dtd;
	gchar *ddtd;

	if( ! mime ) {
		mime = "*";
	}

	ddtd = screem_dtd_db_lookup_mime_type( db, mime, root );
	if( ! ddtd ) {
		ddtd = screem_dtd_db_lookup_mime_type( db, "*", root );
	}

	if( ! ddtd ) {
		ddtd = g_strdup( "-//W3C//DTD HTML 4.01 Transitional//EN" );
	}
	
	dtd = screem_dtd_db_get_dtd( db, ddtd, NULL ); 

	g_free( ddtd );
		
	return dtd;
}

void screem_dtd_db_fill_list( ScreemDTDDB *db, GtkListStore *store )
{
	GList *list;
	GList *tmp;
	GtkTreeIter it;
	gchar *sysid;
	gint i;
	gboolean active;

	gtk_list_store_clear( store );
	g_object_set_data( G_OBJECT( store ), "list", NULL );
	g_hash_table_foreach( db->priv->dtd_types, 
			      (GHFunc)dtds_insert, store );
	list = g_object_get_data( G_OBJECT( store ), "list" );
	list = g_list_sort( list, (GCompareFunc)screem_dtd_db_sort_compare );
	for( i = 0, tmp = list; tmp; tmp = tmp->next, ++ i ) {
		sysid = screem_dtd_db_get_system_id( db,
				tmp->data, NULL );
		active = ! screem_dtd_db_get_hidden( db, tmp->data );
		gtk_list_store_insert_with_values( store, &it, i,
				SCREEM_DTD_DB_PUBLIC_COL, tmp->data,
				SCREEM_DTD_DB_SYSTEM_COL, sysid,
				SCREEM_DTD_DB_ACTIVE_COL, active,
				-1 );

		g_free( sysid );
	}
	g_list_free( list );
}

void screem_dtd_db_override_root( ScreemDTDDB *db,
				  const gchar *publicid,
				  const gchar *root )
{
	xmlNodePtr droot;
	xmlNodePtr override;
	
	g_return_if_fail( SCREEM_IS_DTD_DB( db ) );
	g_return_if_fail( publicid != NULL );

	override = screem_dtd_db_lookup_override( db, publicid );
	if( ! override ) {
		override = xmlNewNode( NULL, (const xmlChar*)"doctype" );
		xmlSetProp( override, (const xmlChar*)"public", 
				(xmlChar*)publicid );
		droot = xmlDocGetRootElement( db->priv->overrides );
		xmlAddChild( droot, override );
	}
	if( root ) {
		xmlSetProp( override, (const xmlChar*)"root", 
				(xmlChar*)root );
	} else {
		xmlUnsetProp( override, (const xmlChar*)"root" );
	}

	xmlSaveFile( db->priv->overridepath, db->priv->overrides );
}

void screem_dtd_db_override_systemid( ScreemDTDDB *db,
				 const gchar *publicid,
				 const gchar *uri )
{
	xmlNodePtr droot;
	xmlNodePtr override;
	
	g_return_if_fail( SCREEM_IS_DTD_DB( db ) );
	g_return_if_fail( publicid != NULL );

	override = screem_dtd_db_lookup_override( db, publicid );
	if( ! override ) {
		override = xmlNewNode( NULL, (const xmlChar*)"doctype" );
		xmlSetProp( override, (const xmlChar*)"public", 
				(xmlChar*)publicid );
		droot = xmlDocGetRootElement( db->priv->overrides );
		xmlAddChild( droot, override );
	}
	if( uri ) {
		xmlSetProp( override, (const xmlChar*)"system", 
				(xmlChar*)uri );
	} else {
		xmlUnsetProp( override, (const xmlChar*)"system" );
	}
	
	xmlSaveFile( db->priv->overridepath, db->priv->overrides );
}

gchar *screem_dtd_db_get_root( ScreemDTDDB *db, const gchar *publicid,
				gboolean *override )
{
	ScreemDTD *dtd;
	gchar *ret;
	
	g_return_val_if_fail( SCREEM_IS_DTD_DB( db ), NULL );
	g_return_val_if_fail( publicid != NULL, NULL );

	dtd = screem_dtd_db_get_dtd( db, publicid, NULL );

	ret = screem_dtd_db_override_get_root( db, publicid );
	if( override ) {
		*override = ( ret != NULL );
	}
	if( dtd && ! ret ) {
		ret = (gchar*)screem_dtd_get_root_name( dtd );
		if( ret ) {
			ret = g_strdup( ret );
		}
	}

	return ret;
}

gchar* screem_dtd_db_get_system_id( ScreemDTDDB *db,
				const gchar *publicid,
				gboolean *override )
{
	GnomeVFSURI *uri;
	gchar *systemid;
	gchar *tmp;
	gchar *base;

	g_return_val_if_fail( SCREEM_IS_DTD_DB( db ), NULL );
	g_return_val_if_fail( publicid != NULL, NULL );

	systemid = screem_dtd_db_override_get_system_id( db,
						publicid );
	if( override ) {
		*override = ( systemid != NULL );
	}
	if( ! systemid ) {
		systemid = g_hash_table_lookup( db->priv->dtd_types, 
						publicid ); 

		systemid = g_strdup( systemid );
		tmp = g_path_get_basename( systemid );
		base = gnome_vfs_unescape_string( tmp, "" );
		g_free( tmp );
		tmp = gnome_vfs_unescape_string( base, "" );
		uri = gnome_vfs_uri_new( tmp );
		if( uri ) {
			if( ! gnome_vfs_uri_is_local( uri ) ) {
				g_free( systemid );
				systemid = g_strdup( tmp );
				/*gnome_vfs_unescape_string( base, "" );*/
			}
			gnome_vfs_uri_unref( uri );
		}
		g_free( tmp );
		g_free( base );
	}
	
	return systemid;
}

gchar *screem_dtd_db_get_mime_type( ScreemDTDDB *db,
		const gchar *publicid )
{
	xmlNodePtr override;
	gchar *ret;
	
	g_return_val_if_fail( SCREEM_IS_DTD_DB( db ), NULL );
	g_return_val_if_fail( publicid != NULL, NULL );

	override = screem_dtd_db_lookup_override( db, publicid );
	ret = NULL;
	if( override ) {
		ret = (gchar*)xmlGetProp( override, 
				(const xmlChar*)"mime" );
	} 	
	return ret;
}

gboolean screem_dtd_db_get_match_root( ScreemDTDDB *db,
		const gchar *publicid )
{
	xmlNodePtr override;
	xmlChar *match;
	gboolean ret;
	
	g_return_val_if_fail( SCREEM_IS_DTD_DB( db ), FALSE );
	g_return_val_if_fail( publicid != NULL, FALSE );

	override = screem_dtd_db_lookup_override( db, publicid );
	ret = FALSE;
	if( override ) {
		match = xmlGetProp( override, 
				(const xmlChar*)"rootmatch" );
		if( match ) {
			ret = ( *match == '1' );
			xmlFree( match );
		}
	} 	

	return ret;
}

void screem_dtd_db_set_mime_type( ScreemDTDDB *db,
		const gchar *publicid, const gchar *mime )
{
	ScreemDTDDBPrivate *priv;
	xmlNodePtr override;
	xmlNodePtr droot;

	g_return_if_fail( SCREEM_IS_DTD_DB( db ) );
	g_return_if_fail( publicid != NULL );

	priv = db->priv;

	override = screem_dtd_db_lookup_override( db, publicid );
	if( ! override ) {
		override = xmlNewNode( NULL, (const xmlChar*)"doctype" );
		xmlSetProp( override, (const xmlChar*)"public", 
				(xmlChar*)publicid );
		droot = xmlDocGetRootElement( db->priv->overrides );
		xmlAddChild( droot, override );
	}
	if( ! mime ) {
		xmlUnsetProp( override, (const xmlChar*)"mime" );
		/* can't rootmatch without mime */
		xmlUnsetProp( override, (const xmlChar*)"rootmatch" );
	} else if( strcmp( "*", mime ) ) {
		/* we want this doctype as the main default */
		xmlSetProp( override, (const xmlChar*)"mime", 
				(xmlChar*)mime );
	} else {
		/* default for specific type */
		xmlSetProp( override, (const xmlChar*)"mime", 
				(xmlChar*)mime );
	}

	/* could be possible conflicts for handling,
	 * resolve them */

	xmlSaveFile( priv->overridepath, priv->overrides );
}

void screem_dtd_db_set_match_root( ScreemDTDDB *db,
		const gchar *publicid, gboolean match )
{
	ScreemDTDDBPrivate *priv;
	xmlNodePtr override;
	xmlNodePtr droot;
	
	g_return_if_fail( SCREEM_IS_DTD_DB( db ) );
	g_return_if_fail( publicid != NULL );

	priv = db->priv;

	override = screem_dtd_db_lookup_override( db, publicid );
	if( ! override ) {
		override = xmlNewNode( NULL, (const xmlChar*)"doctype" );
		xmlSetProp( override, (const xmlChar*)"public", 
				(xmlChar*)publicid );
		droot = xmlDocGetRootElement( db->priv->overrides );
		xmlAddChild( droot, override );
	}
	if( ! match ) {
		xmlUnsetProp( override, (const xmlChar*)"rootmatch" );
	} else {
		xmlSetProp( override, (const xmlChar*)"rootmatch", 
				(const xmlChar*)"1" );
	}

	/* could be possible conflicts for handling,
	 * resolve them */
	
	xmlSaveFile( priv->overridepath, priv->overrides );
}



const gchar *screem_dtd_db_parse_doctype( ScreemDTDDB *db,
		const gchar *doctype,
		gchar **publicid, gchar **systemid, gchar **root )
{
	GString *tmp;
	gunichar c;
	gunichar term;
	gboolean istag;
	guint len;
	
	g_return_val_if_fail( SCREEM_IS_DTD_DB( db ), NULL );
	g_return_val_if_fail( publicid != NULL, NULL );
	g_return_val_if_fail( systemid != NULL, NULL );

	istag = ! g_strncasecmp( "<!DOCTYPE ", doctype,
				 strlen( "<!DOCTYPE" ) );

	*publicid = NULL;
	*systemid = NULL;
	
	if( root ) {
		*root = NULL;
	}

	c = '\0';
	term = '\0';

	/* max size we need for strings, just alloc this much
	 * to avoid lots of reallocs */
	len = strlen( doctype );
	tmp = g_string_sized_new( len );
	
	if( istag ) {
		doctype += strlen( "<!DOCTYPE " );
		doctype = screem_utf8_skip_space( doctype );
		c = g_utf8_get_char( doctype );

		/* get root name */
		while( *doctype && 
			( g_unichar_isalnum( c ) ||
			  c == '.' || c == '_' || c == '-' ) ) {
			g_string_append_unichar( tmp, c );
			doctype = g_utf8_next_char( doctype );
			c = g_utf8_get_char( doctype );
		}
		*root = g_strdup( tmp->str );
		g_string_truncate( tmp, 0 );
		
		/* incase we didn't hit whitespace, skip to it */
		while( ! g_unichar_isspace( c ) ) {
			doctype = g_utf8_next_char( doctype );
			c = g_utf8_get_char( doctype );
		}
	}
	doctype = screem_utf8_skip_space( doctype );
	if( ( ! strncmp( "PUBLIC", doctype, strlen( "PUBLIC" ) ) ) ||
	    ( ! strncmp( "public", doctype, strlen( "public" ) ) ) ) {
		/* public id found */
		doctype += strlen( "PUBLIC" );

		doctype = screem_utf8_skip_space( doctype );
	
		term = g_utf8_get_char( doctype );
		if( term == '"' || term == '\'' ) {
			do {
				c = g_utf8_get_char( doctype );
				if( c != term ) {
					g_string_append_unichar( tmp,
								c );
				}
				doctype = g_utf8_next_char( doctype );
			} while( *doctype && *doctype != term );
		
			/* doctype should be the char after the " */
			*publicid = g_strdup( tmp->str );
			g_string_truncate( tmp, 0 );
		
			if( *doctype == term ) {
				doctype = g_utf8_next_char( doctype );
			}
			doctype = screem_utf8_skip_space( doctype );
			term = g_utf8_get_char( doctype );
		}
		if( term == '"' || term == '\'' ) {
			do {
				c = g_utf8_get_char( doctype );
				if( c != term ) {
					g_string_append_unichar( tmp,
								c );
				}
				doctype = g_utf8_next_char( doctype );
			} while( *doctype && *doctype != term );
		
			/* doctype should be the char after the " */
			*systemid = g_strdup( tmp->str );
			g_string_truncate( tmp, 0 );

			if( *doctype == term ) {
				doctype = g_utf8_next_char( doctype );
			}			
			doctype = screem_utf8_skip_space( doctype );
			term = g_utf8_get_char( doctype );
		}	
	} else if( ( ! strncmp( "SYSTEM", doctype, strlen( "SYSTEM" ) ) ||
		   ( ! strncmp( "system", doctype, strlen( "system" ) ) ) ) ) {
		/* system id only */
	
		doctype += strlen( "SYSTEM" );

		doctype = screem_utf8_skip_space( doctype );
	
		term = g_utf8_get_char( doctype );
		if( term == '"' || term == '\'' ) {
			do {
				c = g_utf8_get_char( doctype );
				if( c != term ) {
					g_string_append_unichar( tmp,
								c );
				}
				doctype = g_utf8_next_char( doctype );
			} while( *doctype && *doctype != term );
		
			/* doctype should be the char after the " */
			*systemid = g_strdup( tmp->str );
			g_string_truncate( tmp, 0 );
		
			if( *doctype == term ) {
				doctype = g_utf8_next_char( doctype );
			}			
			doctype = screem_utf8_skip_space( doctype );
			term = g_utf8_get_char( doctype );
		}
	}	
	if( term != '[' || ! istag ) {
		doctype = NULL;
	}
			
	g_string_free( tmp, TRUE );

	return doctype;
}

ScreemDTD *screem_dtd_db_get_dtd_from_doctype( ScreemDTDDB *db,
						const gchar *doctype )
{
	ScreemDTD *dtd;
	gchar *publicid;
	gchar *systemid;
	gchar *root;

	gchar *temp;
	gchar *temp2;
	gchar *data;
	ScreemDTDParse parse;

	gboolean success;
	
	doctype = screem_dtd_db_parse_doctype( db, doctype, 
			&publicid, &systemid, &root );
	dtd = NULL;

	/* only give a dtd back if we have a root element and
	 * a public or system identifier
	 *
	 * check if the doctype contains any entities, if so we
	 * always want a new ScreemDTD object which contains them
	 *
	 */
	if( ( publicid || systemid ) && root ) {
		if( doctype ) {
			doctype = g_utf8_next_char( doctype );
			temp = strstr( doctype, "]>" );
			if( temp ) {
				doctype = g_strndup( doctype,
						temp - doctype );
				temp = (gchar*)doctype;
			} else {
				doctype = NULL;
			}
		}
		if( ! doctype ) {
			/* standard doctype */
			dtd = screem_dtd_db_get_dtd( db, publicid, systemid );
		} else {
			success = FALSE;
			
			/* document contains entities */
			data = screem_dtd_db_resolve_entity( db, 
					publicid, systemid );
			if( data ) {
				dtd = screem_dtd_new( publicid, 
						systemid );

				/* parse data, adding to dtd */
				parse.dtd = dtd;
				parse.resolve_entity = screem_dtd_db_resolve_entity;
				parse.userdata = db;
				temp2 = g_strconcat( doctype, data,
						NULL );
				g_free( data );
				data = temp2;
				success = screem_dtd_parse( dtd, 
						&parse, data );
				g_free( data );
			}
			g_free( temp );
			if( success ) {
				g_object_set_data( G_OBJECT( dtd ),
						"NO_CACHE",
						GINT_TO_POINTER( 1 ) );
			} else if( dtd ) {
				g_object_unref( dtd );
				dtd = NULL;
			}
		}
	}

	g_free( publicid );
	g_free( systemid );
	g_free( root );

	return dtd;
}

ScreemDTD *screem_dtd_db_get_dtd( ScreemDTDDB *db, 
				  const gchar *publicid,
				  const gchar *systemid )
{
	ScreemDTDDBPrivate *priv;
	ScreemDTD *dtd;
	const gchar *localid;

	g_return_val_if_fail( SCREEM_IS_DTD_DB( db ), NULL );
	g_return_val_if_fail( publicid != NULL ||
			      systemid != NULL, NULL );

	priv = db->priv;
	localid = NULL;
	if( publicid ) {
		localid = g_hash_table_lookup( priv->dtd_types,
						publicid );
	}
	
	dtd = NULL;
	if( localid ) {
		systemid = localid;
		dtd = g_hash_table_lookup( priv->loaded_dtds,
					   localid );
	}
	if( systemid && ! dtd ) {
		dtd = screem_dtd_db_new_dtd( db,
					     publicid, 
					     systemid );
	}
	
	return dtd;
}

GSList *screem_dtd_db_get_default_conflicts( ScreemDTDDB *db,
	const gchar *publicid, const gchar *mime, gboolean rootmatch )
{
	GSList *ret;
	ScreemDTDDBPrivate *priv;
	gchar *root;
	xmlNodePtr node;
	
	g_return_val_if_fail( SCREEM_IS_DTD_DB( db ), NULL );
	g_return_val_if_fail( publicid != NULL, NULL );
	g_return_val_if_fail( mime != NULL, NULL );

	ret = NULL;
	
	priv = db->priv;

	root = screem_dtd_db_get_root( db, publicid, NULL );

	node = xmlDocGetRootElement( priv->overrides );
	node = node->children;
	while( node ) {
		xmlChar *cpublicid;
		xmlChar *cmime;
		xmlChar *crootmatch;
		gboolean crootmatchflag;
		gchar *croot;
		
		cpublicid = xmlGetProp( node, (const xmlChar*)"public" );
		cmime = xmlGetProp( node, (const xmlChar*)"mime" );
		crootmatch = xmlGetProp( node, 
				(const xmlChar*)"rootmatch" );

		crootmatchflag = ( crootmatch && *crootmatch == '1' );
		
		/* should probably assert cpublicid != NULL, 
		   although the dtd says public is optional as we 
		   can have system instead there is no way for that
		   to work properly at the moment 
		
		   only check different public id's + those with
		   a matching mime type set, if they have no mime type they
		   are not the default for anything.   
		*/
		if( ( cpublicid && strcmp( publicid, (gchar*)cpublicid ) ) && 
		    ( cmime && ! strcmp( mime, (gchar*)cmime ) ) ) {
			
			if( ( ! rootmatch ) && ( ! crootmatchflag ) ) {
				/* conflict */
				ret = g_slist_prepend( ret, g_strdup( (gchar*)cpublicid ) );
			} else if( rootmatch && crootmatchflag ) {
				croot = screem_dtd_db_get_root( db, (gchar*)cpublicid, NULL );
				if( root && croot && ! strcmp( root, (gchar*)croot ) ) {
					/* conflict */
					ret = g_slist_prepend( ret, g_strdup( (gchar*)cpublicid ) );
				}
				g_free( croot );
			}
		}

		if( cpublicid ) {
			xmlFree( cpublicid );		
		}
		if( cmime ) {
			xmlFree( cmime );	
		}
		if( crootmatch ) {
			xmlFree( crootmatch );		
		}
		
		node = node->next;
	}
	
	g_free( root );
	
	return ret;
}

gchar *screem_dtd_db_lookup_mime_type( ScreemDTDDB *db,
		const gchar *mime, const gchar *root )
{
	ScreemDTDDBPrivate *priv;
	gchar *ret;
	xmlNodePtr node;

	gchar *fallback;

	priv = db->priv;

	if( ! mime ) {
		mime = "*";
		g_warning( "NULL mime type sent to screem_dtd_db_lookup_mime_type\n" );
	}

	ret = NULL;
		
	node = xmlDocGetRootElement( priv->overrides );
	node = node->children;

	fallback = NULL;
	while( node && ! ret ) {
		xmlChar *cpublic;
		xmlChar *cmime;
		xmlChar *crootmatch;
		gboolean crootmatchflag;
		gchar *croot;

		cpublic = xmlGetProp( node, (const xmlChar*)"public" );
		cmime = xmlGetProp( node, (const xmlChar*)"mime" );
		crootmatch = xmlGetProp( node, 
				(const xmlChar*)"rootmatch" );
		crootmatchflag = ( crootmatch && *crootmatch == '1' );
		
		if( mime && cmime && ! strcmp( mime, (gchar*)cmime ) ) {

			if( ! crootmatchflag ) {
				/* match */
				if( ! fallback ) {
					fallback = g_strdup( (gchar*)cpublic );				
				}
			} else if( root ) {
				croot = screem_dtd_db_get_root( db,
						(gchar*)cpublic, NULL );
				if( croot && ! strcmp( root, (gchar*)croot ) ) {
					ret = g_strdup( (gchar*)cpublic );
				}
				g_free( croot );
			}
		}		
		if( cpublic ) {
			xmlFree( cpublic );
		}
		if( cmime ) {
			xmlFree( cmime );
		}
		if( crootmatch ) {
			xmlFree( crootmatch );
		}
		
		node = node->next;
	}
	
	if( fallback && ! ret ) {
		ret = g_strdup( fallback );
	}	
	g_free( fallback );

	if( ( ! ret ) && mime[ 0 ] != '*' && mime[ 1 ] != '\0' ) {
		ret = screem_dtd_db_lookup_mime_type( db, "*", root );
	}	
	return ret;
}


