/*  Screem:  screem-doctype-manager.c
 *
 *  not a GObject like the other screem-*-manager.c files as the
 *  doctypes are part of the main preferences. code split out to here
 *  as it is getting rather large and is easier to manage this way
 *
 *  Copyright (C) 2005 David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */
#include <config.h>

#include <gtk/gtk.h>
#include <glib/gi18n.h>
#include <libgnomevfs/gnome-vfs-mime-handlers.h>

#include <string.h>

#include "screem-doctype-manager.h"
#include "screem-dtd-db.h"
#include "support.h"

/* Private API */

static void screem_preferences_dtd_select( GtkTreeSelection *sel,
					   ScreemApplication *app );
static void screem_preferences_dtd_root_toggle( GtkToggleButton *button,
						gpointer data );
static void 
screem_preferences_dtd_system_id_toggle( GtkToggleButton *button,
						gpointer data );


static gboolean
screem_preferences_dtd_set_system_id( GtkWidget *entry, 
					GdkEventFocus *event,
					gpointer data );
static gboolean
screem_preferences_dtd_set_root( GtkWidget *entry, 
				GdkEventFocus *focus,
				gpointer data );
static void apply_as_default_changed( GtkComboBox *box, 
		ScreemDTDDB *db );
static void apply_match_root( GtkToggleButton *button, 
		ScreemDTDDB *db );


static void
screem_preferences_doctype_toggle( GtkCellRendererToggle *toggle,
		const gchar *path, gpointer data )
{
	GtkTreeModel *model;
	GtkTreeIter it;
	GtkTreeIter cit;
	gboolean active;
	
	model = GTK_TREE_MODEL( data );

	active = gtk_cell_renderer_toggle_get_active( toggle );
	
	if( gtk_tree_model_get_iter_from_string( model, &it, path ) ) {

		active = ! active;
		
		gtk_tree_model_filter_convert_iter_to_child_iter( GTK_TREE_MODEL_FILTER( model ), &cit, &it );
		
		model = gtk_tree_model_filter_get_model( GTK_TREE_MODEL_FILTER( model ) );

		gtk_list_store_set( GTK_LIST_STORE( model ), &cit,
				SCREEM_DTD_DB_ACTIVE_COL, active, -1 );
	}
}

static gboolean
screem_preferences_dtd_filter( GtkTreeModel *model, GtkTreeIter *iter,
		GladeXML *xml )
{
	GtkWidget *all;
	gboolean show;
	
	all = glade_xml_get_widget( xml, "show_all_doctypes" );
	show = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( all ) );

	if( ! show ) {
		gtk_tree_model_get( model, iter,
				SCREEM_DTD_DB_ACTIVE_COL, &show, -1 );
	}

	return show;
}

/* key == value for this hash table */
typedef enum {
	DOCTYPE_MIME_DESC,
	DOCTYPE_MIME_MIMETYPE,
	DOCTYPE_MIME_ROWTYPE,
	DOCTYPE_MIME_MAX
} DoctypeMimeCols;
typedef enum {
	DOCTYPE_MIME_ROW_NONE = 0,
	DOCTYPE_MIME_ROW_ALL = 1,
	DOCTYPE_MIME_ROW_TOP_SEP = 2,
	DOCTYPE_MIME_ROW_NORMAL  = 3,
	DOCTYPE_MIME_ROW_BOTTOM_SEP = 4,
	DOCTYPE_MIME_ROW_OTHER = 5
} DoctypeMimeRowType;

static gint screem_preferences_doctype_mime_sort( GtkTreeModel *model,
		GtkTreeIter *a, GtkTreeIter *b, gpointer data )
{
	gint atype;
	gint btype;
	gchar *adesc;
	gchar *bdesc;

	gint ret;
	
	gtk_tree_model_get( model, a,
			DOCTYPE_MIME_DESC, &adesc,
			DOCTYPE_MIME_ROWTYPE, &atype,
			-1 );
	gtk_tree_model_get( model, b,
			DOCTYPE_MIME_DESC, &bdesc,
			DOCTYPE_MIME_ROWTYPE, &btype,
			-1 );

	ret = atype - btype;
	if( ret == 0 ) {
		if( adesc && bdesc ) {
			ret = strcmp( adesc, bdesc );
		} else {
			ret = ( adesc - bdesc );
		}
	}

	return ret;
}

static void screem_preferences_doctype_mime_build( const gchar *key,
		gpointer value, GtkListStore *model )
{
	GtkTreeIter iter;
	const gchar *desc;
	
	desc = gnome_vfs_mime_get_description( key );

	if( ! desc ) {
		desc = key;
	}
	
	gtk_list_store_insert_with_values( GTK_LIST_STORE( model ),
			&iter, G_MAXINT, 
			DOCTYPE_MIME_DESC, desc,
			DOCTYPE_MIME_MIMETYPE, key, 
			DOCTYPE_MIME_ROWTYPE, DOCTYPE_MIME_ROW_NORMAL,
			-1 );
}
static gboolean screem_preferences_doctype_sep_row( GtkTreeModel *model,
		GtkTreeIter *iter, gpointer data )
{
	gint type;
	gboolean ret;
	
	type = -1;
	gtk_tree_model_get( model, iter, 2, &type, -1 );

	ret = ( type == DOCTYPE_MIME_ROW_TOP_SEP ||
		type == DOCTYPE_MIME_ROW_BOTTOM_SEP );

	return ret;
}

static void screem_preferences_dtd_select( GtkTreeSelection *sel,
		ScreemApplication *app )
{
	ScreemDTDDB *db;
	GtkTreeView *view;
	GtkTreeModel *model;
	GtkTreeIter it;
	GtkWidget *widget;
	GladeXML *xml;

	gchar *public;
	gchar *system;
	gchar *root;
	ScreemDTD *dtd;
	gboolean rover;
	gboolean sover;

	gchar *mime;
	gboolean rootmatch;
	
	g_object_set_data( G_OBJECT( app ), "prefblock",
			   GINT_TO_POINTER( 1 ) );

	db = screem_application_get_dtd_db( app );
       	view = gtk_tree_selection_get_tree_view( sel );

	xml = glade_get_widget_tree( GTK_WIDGET( view ) );

	widget = glade_xml_get_widget( xml, "doctype_mime" );
	g_signal_handlers_block_by_func( G_OBJECT( widget ),
			G_CALLBACK( apply_as_default_changed ),
			db );
	widget = glade_xml_get_widget( xml, "doctype_check_root" );
	g_signal_handlers_block_by_func( G_OBJECT( widget ),
			G_CALLBACK( apply_match_root ),
			db );

	dtd = NULL;
	public = system = root = NULL;
	if( gtk_tree_selection_get_selected( sel, &model, &it ) ) {
		gtk_tree_model_get( model, &it,
				SCREEM_DTD_DB_PUBLIC_COL, &public, -1 );

		dtd = screem_dtd_db_get_dtd( db, public, NULL ); 
	}
	
	if( dtd ) {
		root = screem_dtd_db_get_root( db, public, &rover );
		if( ! root ) {
			root = g_strdup( "unknown" );
		}
		system = screem_dtd_db_get_system_id( db, public,
							&sover );

		widget = glade_xml_get_widget( xml, "root" );
		gtk_entry_set_text( GTK_ENTRY( widget ), root );

		widget = glade_xml_get_widget( xml, "system" );
		gtk_entry_set_text( GTK_ENTRY( widget ), system );

		widget = glade_xml_get_widget( xml, "rootoverride" );
		gtk_widget_set_sensitive( widget, TRUE );
		gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ), rover );
		widget = glade_xml_get_widget( xml, "systemoverride" );
		gtk_widget_set_sensitive( widget, TRUE );
		gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ), sover );

		widget = glade_xml_get_widget( xml, "doctype_mime" );
		gtk_widget_set_sensitive( widget, TRUE );
		mime = screem_dtd_db_get_mime_type( db, public );
		model = gtk_combo_box_get_model( GTK_COMBO_BOX( widget ) );
		if( screem_gtk_list_store_find_string( GTK_LIST_STORE( model ), &it, DOCTYPE_MIME_MIMETYPE, mime ) ) {
			gtk_combo_box_set_active_iter( GTK_COMBO_BOX( widget ), &it );
			widget = glade_xml_get_widget( xml, "doctype_check_root" );
			gtk_widget_set_sensitive( widget, mime != NULL );
			rootmatch = screem_dtd_db_get_match_root( db, public );
		} else {
			gtk_combo_box_set_active( GTK_COMBO_BOX( widget ), 0 );
			widget = glade_xml_get_widget( xml, "doctype_check_root" );
			gtk_widget_set_sensitive( widget, FALSE );
			rootmatch = FALSE;
		}
		gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ),
				rootmatch );
		g_free( mime );
				
		g_free( root );
		g_free( system );

		widget = glade_xml_get_widget( xml, "doctype_properties" );
		gtk_widget_set_sensitive( widget, TRUE );

		widget = glade_xml_get_widget( xml, "doctypes" );
		g_object_set_data_full( G_OBJECT( widget ),
				"public", public, g_free );
	} else {
		widget = glade_xml_get_widget( xml, "root" );
		gtk_entry_set_text( GTK_ENTRY( widget ), "" );

		widget = glade_xml_get_widget( xml, "system" );
		gtk_entry_set_text( GTK_ENTRY( widget ), "" );
	
		widget = glade_xml_get_widget( xml, "rootoverride" );
		gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ), FALSE );
		gtk_widget_set_sensitive( widget, FALSE );
		widget = glade_xml_get_widget( xml, "systemoverride" );
		gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ), FALSE );
		gtk_widget_set_sensitive( widget, FALSE );

		widget = glade_xml_get_widget( xml, "doctype_mime" );
		gtk_widget_set_sensitive( widget, FALSE );
		widget = glade_xml_get_widget( xml, "doctype_check_root" );
		gtk_widget_set_sensitive( widget, FALSE );
	
		gtk_tree_selection_unselect_iter( sel, &it );
		
		widget = glade_xml_get_widget( xml, "doctype_properties" );
		gtk_widget_set_sensitive( widget, FALSE );
		
		widget = glade_xml_get_widget( xml, "doctypes" );
		g_object_set_data_full( G_OBJECT( widget ),
				"public", NULL, NULL );
		g_free( public );
	}

	widget = glade_xml_get_widget( xml, "doctype_mime" );
	g_signal_handlers_unblock_by_func( G_OBJECT( widget ),
			G_CALLBACK( apply_as_default_changed ),
			db );
	widget = glade_xml_get_widget( xml, "doctype_check_root" );
	g_signal_handlers_unblock_by_func( G_OBJECT( widget ),
			G_CALLBACK( apply_match_root ),
			db );
	
	g_object_set_data( G_OBJECT( app ), "prefblock",
			   GINT_TO_POINTER( 0 ) );
	
}

static void screem_preferences_dtd_root_toggle( GtkToggleButton *button,
		gpointer data )
{
	ScreemApplication *app;
	ScreemDTDDB *db;
	GladeXML *xml;
	GtkWidget *widget;
	const gchar *root;
	const gchar *publicid;
	
	app = SCREEM_APPLICATION( data );

	if( ! g_object_get_data( G_OBJECT( app ), "prefblock" ) ) {
		db = screem_application_get_dtd_db( app );
		xml = glade_get_widget_tree( GTK_WIDGET( button ) );

		widget = glade_xml_get_widget( xml, "doctypes" );
		publicid = g_object_get_data( G_OBJECT( widget ),
				"public" );
				
		root = NULL;
		if( gtk_toggle_button_get_active( button ) ) {
			widget = glade_xml_get_widget( xml, "root" );
			root = gtk_entry_get_text( GTK_ENTRY(widget) );
		}
		screem_dtd_db_override_root( db, publicid, root );
	}
}

static void 
screem_preferences_dtd_system_id_toggle( GtkToggleButton *button,
		gpointer data )
{
	ScreemApplication *app;
	ScreemDTDDB *db;
	GladeXML *xml;
	GtkWidget *widget;
	const gchar *systemid;
	const gchar *publicid;
	gchar *id;
	GtkTreeSelection *sel;
	GtkTreeModel *model;
	GtkTreeIter it;
	GtkTreeIter cit;
	
	app = SCREEM_APPLICATION( data );

	if( ! g_object_get_data( G_OBJECT( app ), "prefblock" ) ) {
		db = screem_application_get_dtd_db( app );
		xml = glade_get_widget_tree( GTK_WIDGET( button ) );

		widget = glade_xml_get_widget( xml, "doctypes" );
		publicid = g_object_get_data( G_OBJECT( widget ),
				"public" );
		sel = gtk_tree_view_get_selection( GTK_TREE_VIEW( widget ) );
		/* selection must be present or this toggle
		 * wouldn't be togglable */
		gtk_tree_selection_get_selected( sel, &model, &it );
		
		systemid = NULL;
		if( gtk_toggle_button_get_active( button ) ) {
			widget = glade_xml_get_widget( xml,"system" );
			systemid = gtk_entry_get_text( GTK_ENTRY(widget) );
		}
		screem_dtd_db_override_systemid( db, publicid,
				systemid );

		widget = glade_xml_get_widget( xml, "doctypes" );
		id = screem_dtd_db_get_system_id( db, publicid, NULL );
		
		gtk_tree_model_filter_convert_iter_to_child_iter( GTK_TREE_MODEL_FILTER( model ), &cit, &it );
		model = gtk_tree_model_filter_get_model( GTK_TREE_MODEL_FILTER( model ) );
	
		gtk_list_store_set( GTK_LIST_STORE( model ), 
				&cit, 
				SCREEM_DTD_DB_SYSTEM_COL, id,
				-1 );
		
		g_free( id );
	}
}

static gboolean
screem_preferences_dtd_set_root( GtkWidget *entry, GdkEventFocus *event,
		gpointer data )
{
	ScreemApplication *app;
	ScreemDTDDB *db;
	GladeXML *xml;
	GtkWidget *widget;
	const gchar *root;
	const gchar *publicid;
	
	app = SCREEM_APPLICATION( data );

	if( ! g_object_get_data( G_OBJECT( app ), "prefblock" ) ) {
		db = screem_application_get_dtd_db( app );
		xml = glade_get_widget_tree( entry );
	
		widget = glade_xml_get_widget( xml, "doctypes" );
		publicid = g_object_get_data( G_OBJECT( widget ),
				"public" );

		widget = glade_xml_get_widget( xml, "rootoverride" );
		if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
			root = gtk_entry_get_text( GTK_ENTRY( entry ) );
			screem_dtd_db_override_root( db, publicid,
							root );
		}
	}

	return FALSE;
}

static gboolean
screem_preferences_dtd_set_system_id( GtkWidget *entry, 
		GdkEventFocus *event, gpointer data )
{
	ScreemApplication *app;
	ScreemDTDDB *db;
	GladeXML *xml;
	GtkWidget *widget;
	const gchar *systemid;
	const gchar *publicid;
	GtkTreeSelection *sel;
	GtkTreeModel *model;
	GtkTreeIter it;
	GtkTreeIter cit;

	app = SCREEM_APPLICATION( data );

	if( ! g_object_get_data( G_OBJECT( app ), "prefblock" ) ) {
		db = screem_application_get_dtd_db( app );
		xml = glade_get_widget_tree( entry );
	
		widget = glade_xml_get_widget( xml, "doctypes" );
		publicid = g_object_get_data( G_OBJECT( widget ),
				"public" );

		widget = glade_xml_get_widget( xml, "systemoverride" );
		if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
			systemid = gtk_entry_get_text( GTK_ENTRY(entry) );
			screem_dtd_db_override_systemid( db, publicid,
							 systemid );

			widget = glade_xml_get_widget( xml, "doctypes" );
			sel = gtk_tree_view_get_selection( GTK_TREE_VIEW( widget ) );
			if( gtk_tree_selection_get_selected( sel, &model, &it ) ) {
				gtk_tree_model_filter_convert_iter_to_child_iter( GTK_TREE_MODEL_FILTER( model ), &cit, &it );
				model = gtk_tree_model_filter_get_model( GTK_TREE_MODEL_FILTER( model ) );

				gtk_list_store_set( GTK_LIST_STORE( model ), &cit, SCREEM_DTD_DB_SYSTEM_COL, systemid, -1 );
			}
		}
	}

	return FALSE;
}

static gboolean resolve_conflicts( ScreemDTDDB *db,
		const gchar *publicid, const gchar *mime,
		gboolean rootmatch )
{
	GSList *conflicts;
	GSList *tmp;
	GladeXML *cxml;
	GtkListStore *store;
	gboolean apply;
	GtkCellRenderer *rend;
	GtkTreeViewColumn *col;
	GtkWidget *widget;
	GtkTreeIter it;

	apply = TRUE;
	conflicts = screem_dtd_db_get_default_conflicts( db,
		publicid, mime, rootmatch );
	if( conflicts ) {
		apply = FALSE;
		cxml = glade_xml_new( GLADE_PATH"/screem.glade",
				"doctype_conflict", NULL );
			
		widget = glade_xml_get_widget( cxml, "conflict_tree" );
		rend = gtk_cell_renderer_text_new();
		col = gtk_tree_view_column_new();
		gtk_tree_view_column_pack_start( col, rend, TRUE );
		gtk_tree_view_column_set_resizable( col, TRUE );
		gtk_tree_view_append_column( GTK_TREE_VIEW( widget ), 
				col );
		gtk_tree_view_column_set_attributes( col, rend,
				"text", 0, NULL ); 
		
		store = gtk_list_store_new( 1, G_TYPE_STRING );
		for( tmp = conflicts; tmp; tmp = tmp->next ) {
			gtk_list_store_insert_with_values( store, &it, G_MAXINT,
				0, tmp->data, -1 );
		}
		gtk_tree_view_set_model( GTK_TREE_VIEW( widget ),
				GTK_TREE_MODEL( store ) );
		g_object_unref( G_OBJECT( store ) );
			
		widget = glade_xml_get_widget( cxml, "doctype_conflict" );
		if( gtk_dialog_run( GTK_DIALOG( widget ) ) == GTK_RESPONSE_APPLY ) {
			apply = TRUE;
			for( tmp = conflicts; tmp; tmp = tmp->next ) {
				screem_dtd_db_set_mime_type( db, tmp->data, NULL );
			}
		}
		gtk_widget_destroy( widget );
	
		g_slist_foreach( conflicts, (GFunc)g_free, NULL );
		g_slist_free( conflicts );
				
		g_object_unref( cxml );
	}

	return apply;
}

static void apply_as_default_changed( GtkComboBox *box, ScreemDTDDB *db )
{
	GladeXML *xml;
	GtkWidget *widget;
	const gchar *publicid;
	GtkTreeIter it;
	GtkTreeModel *model;
	gint type;
	gchar *mime;
	gboolean rootmatch;
	GtkTreeSelection *sel;
	ScreemApplication *app;
	
	xml = glade_get_widget_tree( GTK_WIDGET( box ) );
	widget = glade_xml_get_widget( xml, "doctypes" );
	publicid = g_object_get_data( G_OBJECT( widget ), "public" );
	widget = glade_xml_get_widget( xml, "doctype_check_root" );
	rootmatch = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) );

	if( gtk_combo_box_get_active( box ) == 0 ) {
		gtk_widget_set_sensitive( widget, FALSE );
		gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ), FALSE );	
	} else {
		gtk_widget_set_sensitive( widget, TRUE );
	}
	
	if( gtk_combo_box_get_active_iter( box, &it ) ) {
		model = gtk_combo_box_get_model( box );
		
		type = -1;
		mime = NULL;
		gtk_tree_model_get( model, &it, 
				DOCTYPE_MIME_ROWTYPE, &type,
				DOCTYPE_MIME_MIMETYPE, &mime,
				-1 );
						
		if( resolve_conflicts( db, publicid, mime, rootmatch ) ) {
			screem_dtd_db_set_mime_type( db, publicid, mime );
		} else {
			/* revert to previous value, cheat, just
			 * reselect the row */
			widget = glade_xml_get_widget( xml, "doctypes" );
			sel = gtk_tree_view_get_selection( GTK_TREE_VIEW( widget ) );
			app = g_object_get_data( G_OBJECT( widget ),
					"app" );
			screem_preferences_dtd_select( sel, app );
		}
				
		g_free( mime );
	} else {
		/* this shouldn't occur */
		g_warning( "doctype mime type selection has no active item!" );
		gtk_combo_box_set_active( box, 0 );	
	}
}

static void apply_match_root( GtkToggleButton *button, ScreemDTDDB *db )
{
	GladeXML *xml;
	GtkWidget *widget;
	GtkWidget *box;
	const gchar *publicid;
	GtkTreeIter it;
	GtkTreeModel *model;
	gint type;
	gchar *mime;
	gboolean rootmatch;
	GtkTreeSelection *sel;
	ScreemApplication *app;

	if( ! GTK_WIDGET_SENSITIVE( GTK_WIDGET( button ) ) ) {
		return;	
	}
	
	xml = glade_get_widget_tree( GTK_WIDGET( button ) );
	widget = glade_xml_get_widget( xml, "doctypes" );
	publicid = g_object_get_data( G_OBJECT( widget ), "public" );
	rootmatch = gtk_toggle_button_get_active( button );
	box = glade_xml_get_widget( xml, "doctype_mime" );
	
	if( gtk_combo_box_get_active_iter( GTK_COMBO_BOX( box ), &it ) ) {
		model = gtk_combo_box_get_model( GTK_COMBO_BOX( box ) );
		
		type = -1;
		mime = NULL;
		gtk_tree_model_get( model, &it, 
				DOCTYPE_MIME_ROWTYPE, &type,
				DOCTYPE_MIME_MIMETYPE, &mime,
				-1 );
		if( resolve_conflicts( db, publicid, mime, rootmatch ) ) {
			screem_dtd_db_set_match_root( db, publicid, 
				rootmatch );
		} else {
			/* revert to previous value, cheat, just
			 * reselect the row */
			widget = glade_xml_get_widget( xml, "doctypes" );
			sel = gtk_tree_view_get_selection( GTK_TREE_VIEW( widget ) );
			app = g_object_get_data( G_OBJECT( widget ),
					"app" );
			screem_preferences_dtd_select( sel, app );
		}
	
		g_free( mime );
	}
}


/* Public API */

void screem_doctype_manager_init( ScreemApplication *application, GladeXML *xml )
{
	ScreemDTDDB *db;
	GtkCellRenderer *rend;
	GtkTreeViewColumn *col;
	GtkListStore *store;
	GtkTreeSelection *sel;
	GtkWidget *widget;

	GtkTreeModel *filter;

	GtkTreeModel *mime;
	GHashTable *features;
	GtkTreeIter it;
	
	db = screem_application_get_dtd_db( application );
	store = screem_dtd_db_get_store( db );

	filter = gtk_tree_model_filter_new( GTK_TREE_MODEL( store ),
			NULL );
	gtk_tree_model_filter_set_visible_func( GTK_TREE_MODEL_FILTER( filter ), (GtkTreeModelFilterVisibleFunc)screem_preferences_dtd_filter,
			xml, NULL );

	widget = glade_xml_get_widget( xml, "show_all_doctypes" );
	g_signal_connect_swapped( G_OBJECT( widget ), "toggled",
			G_CALLBACK( gtk_tree_model_filter_refilter ),
			filter );
	
	widget = glade_xml_get_widget( xml, "doctypes" );
	g_object_set_data( G_OBJECT( widget ), "app", application );
	
	rend = gtk_cell_renderer_toggle_new();
	g_signal_connect( G_OBJECT( rend ), "toggled",
			  G_CALLBACK( screem_preferences_doctype_toggle ),
			  filter );
	col = gtk_tree_view_column_new();
	gtk_tree_view_column_pack_start( col, rend, TRUE );
	gtk_tree_view_column_set_resizable( col, TRUE );
	gtk_tree_view_append_column( GTK_TREE_VIEW( widget ), col );
	gtk_tree_view_column_set_attributes( col, rend,
			"active", SCREEM_DTD_DB_ACTIVE_COL, NULL ); 
	gtk_tree_view_column_set_title( col, "Show" );
	
	rend = gtk_cell_renderer_text_new();
	col = gtk_tree_view_column_new();
	gtk_tree_view_column_pack_start( col, rend, TRUE );
	gtk_tree_view_column_set_resizable( col, TRUE );
	gtk_tree_view_append_column( GTK_TREE_VIEW( widget ), col );
	gtk_tree_view_column_set_attributes( col, rend,
			"text", SCREEM_DTD_DB_PUBLIC_COL, NULL ); 
	gtk_tree_view_column_set_title( col, "PUBLIC" );

/*	rend = gtk_cell_renderer_text_new();
	col = gtk_tree_view_column_new();
	gtk_tree_view_column_pack_start( col, rend, TRUE );
	gtk_tree_view_column_set_resizable( col, TRUE );
	gtk_tree_view_append_column( GTK_TREE_VIEW( widget ), col );
	gtk_tree_view_column_set_attributes( col, rend,
			"text", SCREEM_DTD_DB_SYSTEM_COL, NULL ); 
	gtk_tree_view_column_set_title( col, "SYSTEM" ); */

	gtk_tree_view_set_model( GTK_TREE_VIEW( widget ), filter ); 
	g_object_unref( G_OBJECT( filter ) );
		
	sel = gtk_tree_view_get_selection( GTK_TREE_VIEW( widget ) );
	g_signal_connect( G_OBJECT( sel ), "changed",
			  G_CALLBACK( screem_preferences_dtd_select ),
			  application );

	widget = glade_xml_get_widget( xml, "root" );
	gtk_entry_set_text( GTK_ENTRY( widget ), "" );
	g_signal_connect( G_OBJECT( widget ), "focus_out_event",
			  G_CALLBACK( screem_preferences_dtd_set_root ),
			  application );

	widget = glade_xml_get_widget( xml, "system" );
	gtk_entry_set_text( GTK_ENTRY( widget ), "" );
	g_signal_connect( G_OBJECT( widget ), "focus_out_event",
			  G_CALLBACK( screem_preferences_dtd_set_system_id ),
			  application );
	
	widget = glade_xml_get_widget( xml, "rootoverride" );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ), 
					FALSE );
	g_signal_connect( G_OBJECT( widget ), "toggled",
			  G_CALLBACK( screem_preferences_dtd_root_toggle ),
			  application );
	gtk_widget_set_sensitive( widget, FALSE );
	widget = glade_xml_get_widget( xml, "systemoverride" );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ), 
					FALSE );
	g_signal_connect( G_OBJECT( widget ), "toggled",
			  G_CALLBACK( screem_preferences_dtd_system_id_toggle ),
			  application );
	gtk_widget_set_sensitive( widget, FALSE );


	mime = GTK_TREE_MODEL( gtk_list_store_new( 3, 
		G_TYPE_STRING, G_TYPE_STRING, G_TYPE_INT ) );
	widget = glade_xml_get_widget( xml, "doctype_mime" );
	features = screem_application_get_features( application, 
			"markup" );
	/* None */
	gtk_list_store_insert_with_values( GTK_LIST_STORE( mime ), &it,
			G_MAXINT, 
			DOCTYPE_MIME_DESC, _( "No document types" ),
			DOCTYPE_MIME_ROWTYPE, DOCTYPE_MIME_ROW_NONE,
			-1 );

	/* All */
	gtk_list_store_insert_with_values( GTK_LIST_STORE( mime ), &it,
			G_MAXINT, 
			DOCTYPE_MIME_DESC, _( "Unmatched documents" ),
			DOCTYPE_MIME_MIMETYPE, "*",
			DOCTYPE_MIME_ROWTYPE, DOCTYPE_MIME_ROW_ALL,
			-1 );

	
	/* separator */
	gtk_list_store_insert_with_values( GTK_LIST_STORE( mime ), &it,
			G_MAXINT, DOCTYPE_MIME_ROWTYPE,
			DOCTYPE_MIME_ROW_TOP_SEP, -1 );
	/* types */
	g_hash_table_foreach( features, 
			(GHFunc)screem_preferences_doctype_mime_build,
			mime );
	/* separator */
	gtk_list_store_insert_with_values( GTK_LIST_STORE( mime ), &it,
			G_MAXINT, DOCTYPE_MIME_ROWTYPE,
			DOCTYPE_MIME_ROW_BOTTOM_SEP, -1 );

	/* other */
/*	gtk_list_store_insert_with_values( GTK_LIST_STORE( mime ), &it,
			G_MAXINT, 
			DOCTYPE_MIME_DESC, _( "Other type..." ),
			DOCTYPE_MIME_ROWTYPE, DOCTYPE_MIME_ROW_OTHER,
			-1 );*/
	
	gtk_tree_sortable_set_sort_func( GTK_TREE_SORTABLE( mime ),
			DOCTYPE_MIME_DESC, 
			screem_preferences_doctype_mime_sort, 
			NULL, NULL );
	gtk_tree_sortable_set_sort_column_id( GTK_TREE_SORTABLE( mime ),
			DOCTYPE_MIME_DESC, GTK_SORT_ASCENDING );

	gtk_cell_layout_clear( GTK_CELL_LAYOUT( widget ) );
	rend = gtk_cell_renderer_text_new();
	gtk_cell_layout_pack_start( GTK_CELL_LAYOUT( widget ), rend,
			TRUE );
	gtk_cell_layout_set_attributes( GTK_CELL_LAYOUT( widget ),
			rend, "text", DOCTYPE_MIME_DESC, NULL );
	
	gtk_combo_box_set_model( GTK_COMBO_BOX( widget ), mime ); 
	gtk_combo_box_set_row_separator_func( GTK_COMBO_BOX( widget ),
			screem_preferences_doctype_sep_row, NULL,
			NULL );
	gtk_combo_box_set_active( GTK_COMBO_BOX( widget ), 0 );

	g_signal_connect( G_OBJECT( widget ), "changed",
		G_CALLBACK( apply_as_default_changed ),	db );

	widget = glade_xml_get_widget( xml, "doctype_check_root" );
	g_signal_connect( G_OBJECT( widget ), "toggled",
		G_CALLBACK( apply_match_root ), db );
	

	g_object_unref( G_OBJECT( mime ) );
}
