/*  Screem:  skel-plugin.c
 *
 *  Copyright (C) 2004 David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 */
#include <config.h>

#include <glib/gi18n.h>
#include <gtk/gtk.h>
#include <glade/glade.h>

#include "screem-skel-plugin.h"

/* setup your plugin here */

#include "support.h"

/* plugin name should only include a-zA-Z */
static const gchar *plugin_name = "ScreemTablewizard";

static const gchar *authors[] = {
	"David A Knight <david@screem.org>",
	NULL
};
static const gchar *displayed_name = N_( "Screem Table Wizard" );
static const gchar *description = N_( "A wizard for inserting an HTML table" );
static const gchar *version = "2.0.0";

/* add any per instance data items here */
struct ScreemSkelPluginPrivate {
	GladeXML *xml;
	GtkWidget *dialog;
};

static void table_wizard_clicked( GtkWidget *w, gint button,
		gpointer data );
static void table_wizard_display( GtkAction *action, gpointer user_data );
static void init_dialog( ScreemPlugin *plugin );

/**
 * setup:
 * 
 * this function will be called once for each window,
 * you should add any actions / ui here, eg.
 * 
 * screem_plugin_add_action( plugin, name, label, tip, stock_id,
 * 				callback, error );
 * screem_plugin_add_menu( plugin, path, action, error );
 * screem_plugin_add_toolbar( plugin, path, action, error );
 *
 *
 * to insert text into the current page being edited your callbacks
 * should make use of
 * screem_plugin_get_cursor_position( plugin )
 * screem_plugin_set_cursor_position( plugin, pos )
 * screem_plugin_insert( plugin, pos, text, length, indent )
 * 
 **/
static gboolean setup( ScreemPlugin *plugin )
{
	GError *error;
	gboolean ret;

	error = NULL;
	
	ret = screem_plugin_add_interface( plugin, "TableWizard",
				_( "Table Wizard" ),
				_( "Insert a Table" ),
				"Screem_Table",
				G_CALLBACK( table_wizard_display ),
				&error  );
	if( ! ret ) {
		g_print( "Add interface error: %s\n", error->message );
		g_error_free( error );
	} 	
	return ret;
}

/**
 * cleanup:
 *
 * this function will be called once for each window when
 * it is closed, you should cleanup any data items you
 * have in ScreemSkelPluginPrivate here
 **/
static void cleanup( ScreemSkelPluginPrivate *priv )
{
	gtk_widget_destroy( priv->dialog );
	g_object_unref( priv->xml );
}

static void table_wizard_clicked( GtkWidget *w, gint button,
		gpointer data )
{
	ScreemPlugin *plugin;
	ScreemSkelPluginPrivate *priv;
	gint c;
	gint pos;
	GtkWidget *widget;
	GtkWidget *widget2;
	gint border = 0;
	GString *text;
	gint cellSpacing;
	gint cellPadding;
	gint width_value;
	gint rows;
	gint cols;
	gboolean caption;

	GladeXML *xml;
	
	plugin = SCREEM_PLUGIN( data );
	priv = SCREEM_SKEL_PLUGIN( plugin )->priv;
	
	xml = priv->xml;

    	if( button == GTK_RESPONSE_APPLY || button == GTK_RESPONSE_OK ) {
		widget = glade_xml_get_widget( 	xml, "caption" );
		caption = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) );
		widget = glade_xml_get_widget( xml, "border" );
		if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON( widget ))) {
			widget = glade_xml_get_widget( xml, 
					"border_width" );
			border = gtk_spin_button_get_value_as_int( GTK_SPIN_BUTTON ( widget ) );
		}

		text = g_string_new( "" );
		g_string_sprintf( text, "<table border=\"%d\"", border );

		widget = glade_xml_get_widget( xml, "left" );
		widget2 = glade_xml_get_widget( xml, "centre" );

		if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON(widget) ) )
			g_string_append( text, " align=\"left\"" );
		else if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget2) ) )
			g_string_append( text, " align=\"center\"" );
		else
			g_string_append( text, " align=\"right\"" );

		widget = glade_xml_get_widget( xml, "spacing" );
	        cellSpacing = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON(widget));
		widget = glade_xml_get_widget( xml, "padding" );
	        cellPadding = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON(widget));
		g_string_append_printf( text, 
				   " cellspacing=\"%d\" cellpadding=\"%d\"",
				   cellSpacing, cellPadding );
	
		widget = glade_xml_get_widget( xml, "rows" );
	        rows = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON(widget));
		widget = glade_xml_get_widget( xml, "cols" );
	        cols = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON(widget));

		widget = glade_xml_get_widget( xml, "width" );
		if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
			widget = glade_xml_get_widget( xml, "table_width" );
			width_value = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(widget));
			g_string_append_printf( text, " width=\"%d",
					   width_value );
					
			widget = glade_xml_get_widget( xml, "percent" );
			if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) )
				g_string_append_c( text, '%' );
			g_string_append_c( text, '"' );
		}

		widget = glade_xml_get_widget( xml, "use_colour" );
		if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
			GdkColor c;
			gchar *tmp;
			
			widget = glade_xml_get_widget( xml, "colour" );
			gtk_color_button_get_color( GTK_COLOR_BUTTON( widget ), &c );
			tmp = screem_gdk_color_to_string( &c, TRUE );
			g_string_append_printf( text, 
					   " bgcolor=\"%s\"", tmp );
			g_free( tmp );
		}

		widget = glade_xml_get_widget( xml, "use_image" );
		if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
			gchar *uri;
			widget = glade_xml_get_widget( xml, 
						       "image_entry" );
			uri = gtk_file_chooser_get_uri( GTK_FILE_CHOOSER( widget ) );
			if( ! uri ) {
				uri = g_strdup( "" );
			}
			g_string_append_printf( text, " background=\"%s\"",
					uri );
			g_free( uri );
		}

		widget = glade_xml_get_widget( xml, "summary" );
		if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
			widget = glade_xml_get_widget( xml,
					"summary_entry" );
			widget = GTK_BIN( widget )->child;
			g_string_append_printf( text, " summary=\"%s\"",
					   gtk_entry_get_text( GTK_ENTRY( widget ) ) );
		}

		g_string_append_c( text, '>' );

		widget = glade_xml_get_widget( xml, "above" );
		if( caption &&
		    gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget)) ) {
			g_string_append( text,  "\n<caption> </caption>" );
			caption = FALSE;
		}

		for( ; rows; rows -- ) {
			g_string_append( text, "\n<tr>" );
			for( c = 0; c < cols; c ++ )
				g_string_append( text, "\n<td> </td>" );
			g_string_append( text, "\n</tr>" );
		}

		if( caption )
			g_string_append( text,  "\n<caption> </caption>" );

		g_string_append( text, "\n</table>" );

		pos = screem_plugin_get_cursor_position( plugin );
		screem_plugin_insert( plugin, pos, text->str, text->len,
				TRUE );
		pos += text->len;
		screem_plugin_set_cursor_position( plugin, pos );

		g_string_free( text, TRUE );
	}

	screem_plugin_store_in_session( plugin, priv->dialog );
	
	if( button == GTK_RESPONSE_OK || button == GTK_RESPONSE_CLOSE ){
		gtk_widget_hide( priv->dialog );
	}
}

static void table_wizard_display( GtkAction *action, gpointer user_data )
{
	ScreemPlugin *plugin;
	ScreemSkelPluginPrivate *priv;
	ScreemPage *page;
		
	plugin = SCREEM_PLUGIN( user_data );
	priv = SCREEM_SKEL_PLUGIN( plugin )->priv;

	page = screem_plugin_get_current_document( plugin );
        
	if( page ) {
		init_dialog( plugin );
		if( ! GTK_WIDGET_VISIBLE( priv->dialog ) ) {
			screem_plugin_restore_from_session( plugin,
					priv->dialog );
		}
		gtk_widget_show_all( priv->dialog );
		gdk_window_raise( priv->dialog->window );
	}
}

static void init_dialog( ScreemPlugin *plugin )
{
	ScreemSkelPluginPrivate *priv;
	GtkWidget *widget;
	
	priv = SCREEM_SKEL_PLUGIN( plugin )->priv;
	if( ! priv->xml ) {
		priv->xml = glade_xml_new( GLADE_PATH"/tablewizard.glade",
					"tablewizard", NULL );
			priv->dialog = glade_xml_get_widget( priv->xml,
					"tablewizard" );
			g_signal_connect( G_OBJECT( priv->dialog ), 
					"response",
					G_CALLBACK( table_wizard_clicked ),
					plugin );
			widget = glade_xml_get_widget( priv->xml,
					"summary_entry" );
			screem_gtk_add_history( widget );

			glade_xml_signal_autoconnect( priv->xml );
	}
}

/* There should be no need to change any code below here */
enum {
	ARG_0
};

static void screem_skel_plugin_class_init( ScreemSkelPluginClass *klass );
static void screem_skel_plugin_init( ScreemSkelPlugin *skel_plugin );
static void screem_skel_plugin_finalize( GObject *object );

/* G Object stuff */
#define PARENT_TYPE SCREEM_TYPE_PLUGIN

static gpointer parent_class;

static void screem_skel_plugin_class_init( ScreemSkelPluginClass *klass )
{
	GObjectClass *object_class;

	object_class = G_OBJECT_CLASS( klass );

	object_class->finalize = screem_skel_plugin_finalize;
	parent_class = g_type_class_peek_parent( klass );
}

static void screem_skel_plugin_init( ScreemSkelPlugin *skel_plugin )
{
	skel_plugin->priv = g_new0( ScreemSkelPluginPrivate, 1 );
	SCREEM_PLUGIN( skel_plugin )->setup = setup;
}

static void screem_skel_plugin_finalize( GObject *object )
{
	ScreemSkelPlugin *skel_plugin;
	ScreemSkelPluginPrivate *priv;
	
	skel_plugin = SCREEM_SKEL_PLUGIN( object );
	priv = skel_plugin->priv;

	cleanup( priv );
	
	g_free( priv );
	
	G_OBJECT_CLASS( parent_class )->finalize( object );
}

static GType screem_skel_plugin_get_type()
{
	static GType type = 0;
	
	if( ! type ) {
		static const GTypeInfo info = {
			sizeof( ScreemSkelPluginClass ),
			NULL, /* base init */
			NULL, /* base finalise */
			(GClassInitFunc)screem_skel_plugin_class_init,
			NULL, /* class finalise */
			NULL, /* class data */
			sizeof( ScreemSkelPlugin ),
			0, /* n_preallocs */
			(GInstanceInitFunc)screem_skel_plugin_init
		};

		type = g_type_register_static( PARENT_TYPE,
					       plugin_name,
					       &info, 0 );
	}

	return type;
}

static ScreemSkelPlugin *screem_skel_plugin_new( void )
{
	ScreemSkelPlugin *skel_plugin;

	skel_plugin = SCREEM_SKEL_PLUGIN( g_object_new( SCREEM_TYPE_SKEL_PLUGIN, 
				"name", plugin_name,
				NULL ) );

	return skel_plugin;
}

G_MODULE_EXPORT void get_details( ScreemPluginDetails **ret )
{
	ScreemPluginDetails *details;

	details = g_new0( ScreemPluginDetails, 1 );
	details->name = plugin_name;
	details->displayed_name = displayed_name;
	details->authors = authors;
	details->description = description;
	details->version = version;
	details->create = screem_skel_plugin_new;
	details->api_version = SCREEM_PLUGIN_REQUIRED_VERSION;

	*ret = details;
}

