/*  Screem:  screem-tagtree.c
 *
 *  the tag tree object
 *
 *  Copyright (C) 2001 David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */
#include <config.h>

#include <unistd.h>
#include <string.h>
#include <stdlib.h>

#include <glade/glade.h>

#include <gtk/gtk.h>

#include <gtksourceview/gtksourceview.h>

#include <glib/gi18n.h>

#include <libgnomevfs/gnome-vfs-mime-handlers.h>
#include <libgnomevfs/gnome-vfs-mime-utils.h>
#include <libgnomevfs/gnome-vfs-uri.h>
#include <libgnomevfs/gnome-vfs-utils.h>

#include <libxml/tree.h>

#include "screem-tagtree.h"

#include "screem-window.h"
#include "screem-window-private.h"
#include "screem-page.h"
#include "screem-dtd.h"

#include "fileops.h"
#include "support.h"

#include "eggtreemodelunion.h"

#include "screem-tagfile.h"

#include "screem-top-level-model.h"

static GSList *trees = NULL;
static GSList *treefiles = NULL;
static GtkListStore *filesstore = NULL;
static GStaticMutex mutex = G_STATIC_MUTEX_INIT;

static xmlDocPtr config = NULL;
static gchar *configpath = NULL;

struct ScreemTagTreePrivate {
	GtkWidget *box;

	GtkWidget *view;

	GtkTreeModel *umodel;
	GtkTreeModel *sorted;

	GHashTable *paths;

	ScreemWindow *window;

	GtkWidget *toolbar;

	ScreemDTD *dtd;
	ScreemPage *page;

	guint timeout;

	GSList *dtdcache;
	guint dtdcachelen;

	ScreemTagFile *dtdbranch;

	const gchar *search_pattern;
	gboolean regexp;
	GtkTreePath *search_result;
};

enum {
	ARG_0,
	ARG_WINDOW,
	ARG_PAGE
};
static void screem_tag_tree_add_dtd( ScreemPage *page, GParamSpec *spec,
				     ScreemTagTree *tree );

static void screem_tagtree_sel_change( GtkTreeSelection *selection,
					ScreemTagTree *tree );
static gboolean screem_tag_tree_set_model( ScreemTagTree *tree ); 
static gpointer screem_tag_tree_create( gpointer data );
static void screem_tag_tree_add_file( ScreemTagTree *tree,
					ScreemTagFile *file,
					gboolean prepend );
static void screem_tag_tree_remove_file( ScreemTagTree *tree,
					ScreemTagFile *file );

static void screem_tag_file_active_set( ScreemTagFile *file,
					GParamSpec *spec,
					gpointer data );

static void screem_tag_tree_save_config( void );
static xmlNodePtr screem_tag_tree_get_file_node( const gchar *url );
static gboolean screem_tag_tree_is_active( const gchar *url,
					gchar **name,
					gchar **desc );
static void screem_tag_tree_search( GtkAction *action,
				gpointer data );
static gboolean screem_tag_tree_search_func( GtkTreeModel *model,
					GtkTreePath *path,
					GtkTreeIter *iter,
					gpointer data );
static gint screem_tag_tree_sort( GtkTreeModel *model,
		GtkTreeIter *a, GtkTreeIter *b,
		gpointer data );
static gboolean screem_tag_tree_add_current( ScreemTagTree *tree );

static void screem_tag_tree_class_init( ScreemTagTreeClass *klass );
static void screem_tag_tree_init( ScreemTagTree *tag_tree );
static void screem_tag_tree_finalize( GObject *object );
static void screem_tag_tree_set_prop( GObject *object, guint property_id,
				    const GValue *value, GParamSpec *pspec );
static void screem_tag_tree_get_prop( GObject *object, guint property_id,
				      GValue *value, GParamSpec *pspec);
static void screem_tag_tree_size_request( GtkWidget *widget,
                           	  	  GtkRequisition *requisition );
static void screem_tag_tree_size_allocate( GtkWidget *widget,
                              		   GtkAllocation *allocation );

ScreemTagTree *screem_tag_tree_new()
{
	ScreemTagTree *tree;
	GType type;

	type = screem_tag_tree_get_type();
	
	tree = SCREEM_TAG_TREE( g_object_new( type, NULL ) );

	return tree;
}

gboolean screem_tag_tree_load_trees( void )
{
	GConfClient *client;
	gchar *dot;

	g_static_mutex_lock( &mutex );
	if( ! filesstore ) {
		filesstore = gtk_list_store_new( MAX_TAG_FILE_COLS,
						G_TYPE_STRING,
						G_TYPE_STRING,
						G_TYPE_STRING,
						G_TYPE_BOOLEAN,
						G_TYPE_POINTER );
		dot = screem_get_dot_dir();
		configpath = g_build_path( G_DIR_SEPARATOR_S,
				dot, "tagtreeconfig.xml",
				NULL );
		g_free( dot );
		config = NULL;
		if( uri_exists( configpath, NULL ) ) {
			config = xmlParseFile( configpath );
		}
		if( ! config ) {
			config = xmlNewDoc( (const xmlChar*)XML_DEFAULT_VERSION );
			xmlDocSetRootElement( config,
					xmlNewDocNode( config, NULL,
						(const xmlChar*)"refconfig", NULL ) );
		}
	}
	g_static_mutex_unlock( &mutex );

#if 1
	
	client = gconf_client_get_default();
	if( ! gconf_client_get_bool( client,
				"/apps/screem/general/threaded_tree_load",
				NULL ) ) {
		g_idle_add( (GSourceFunc)screem_tag_tree_create, NULL );
	} else {
		g_thread_create( screem_tag_tree_create, NULL, 
				FALSE, NULL );
	}
	g_object_unref( client );
#endif	
	return FALSE;
}

GtkListStore *screem_tag_tree_get_file_model( void )
{
	return filesstore;
}

static gint screem_tag_tree_find_tag_file( gconstpointer a,
					gconstpointer b )
{
	const gchar *uri;
	gint match;
	
	uri = screem_tag_file_get_uri( SCREEM_TAGFILE( a ) );
	
	match = ! gnome_vfs_uris_match( uri, (const gchar*)b );
	
	return match;
}

gboolean screem_tag_tree_add_uri( const gchar *uri,
				  ScreemTagFileFormat format )
{
	ScreemTagFile *tfile;
	gboolean active;
	gchar *name;
	gchar *desc;
	GtkTreeIter it;
	xmlNodePtr node;
	xmlChar *enable;

	/* don't add already existing uris */
	if( g_slist_find_custom( treefiles, uri,
				(GCompareFunc)screem_tag_tree_find_tag_file ) ) {
		return TRUE;
	}

	tfile = screem_tag_file_new();
	screem_tag_file_set_format( tfile, format );

	screem_tag_file_set_uri( tfile, uri );
	g_static_mutex_lock( &mutex );
	active = screem_tag_tree_is_active( uri, &name, &desc );
	if( name ) {
		screem_tag_file_set_name( tfile, name );
	}
	if( desc ) {
		screem_tag_file_set_desc( tfile, desc );
	}

	treefiles = g_slist_prepend( treefiles, tfile );
	gdk_threads_enter();
	/* use large pos to ensure append */
	gtk_list_store_insert_with_values( filesstore, &it,
			G_MAXINT32,
			TAG_FILE_NAME_COL, name,
			TAG_FILE_URI_COL, uri,
			TAG_FILE_DESC_COL, desc,
			TAG_FILE_OBJ_COL, tfile,
			-1 );	
	g_static_mutex_unlock( &mutex );
	g_free( name );
	g_free( desc );
	gdk_threads_leave();
	
	g_signal_connect( G_OBJECT( tfile ),
			"notify::active",
			G_CALLBACK( screem_tag_file_active_set ),
			NULL );
	node = screem_tag_tree_get_file_node( uri );
	if( node ) {
		enable = xmlGetProp( node, (const xmlChar*)"enable" );
		if( enable && *enable == '1' ) {
			screem_tag_file_load( tfile, 
					SCREEM_TAG_UNKNOWN_FORMAT );
		}
		if( enable ) {
			xmlFree( enable );
		}
	}
	
	return TRUE;
}

void screem_tag_tree_remove_uri( const gchar *uri )
{
	GSList *tmp;
	const gchar *furi;
	ScreemTagFile *tfile;
	GtkTreeIter it;
	gboolean active;
	xmlNodePtr node;
	gboolean found;
	
	g_static_mutex_lock( &mutex );

	tmp = g_slist_find_custom( treefiles, uri,
			(GCompareFunc)screem_tag_tree_find_tag_file );

	if( tmp ) {
		tfile = SCREEM_TAGFILE( tmp->data );
		furi = screem_tag_file_get_uri( tfile );

		found = screem_gtk_list_store_find_string( filesstore, 
				&it, TAG_FILE_URI_COL, furi );

		g_assert( found );

		gtk_list_store_remove( filesstore, &it );

		treefiles = g_slist_remove( treefiles, tfile );

		g_object_get( G_OBJECT( tfile ),
				"active", &active, NULL );
		if( active ) {
			for( tmp = trees; tmp; tmp = tmp->next ) {
				screem_tag_tree_remove_file( tmp->data,
							tfile );
			}
		}
	
		if( ( node = screem_tag_tree_get_file_node( uri ) ) ) {
			xmlUnlinkNode( node );
			xmlFreeNode( node );
			screem_tag_tree_save_config();
		}
		g_object_unref( tfile );
	}
	g_static_mutex_unlock( &mutex );
}

GSList *screem_tag_tree_autocomplete( const gchar *mime_type,
				const gchar *prefix,
				gboolean fortip )
{
	GSList *ret;
	GSList *tmp;
	GSList *tret;
	ScreemTagFile *tfile;
	const gchar *type;
	
	g_return_val_if_fail( mime_type != NULL, NULL );
	g_return_val_if_fail( prefix != NULL, NULL );

	ret = NULL;

	g_static_mutex_lock( &mutex );
	
	for( tmp = treefiles; tmp; tmp = tmp->next ) {
		tfile = SCREEM_TAGFILE( tmp->data );

		type = screem_tag_file_for_type( tfile );
		if( type ) {
			if( ( ! strcmp( mime_type, type ) ) || 
				( gnome_vfs_mime_type_get_equivalence( mime_type, (gchar*)type ) == GNOME_VFS_MIME_PARENT ) ) {

				tret = screem_tag_file_autocomplete( tfile,
						prefix, fortip );
				ret = g_slist_concat( ret, tret );
			}
		}
	}
	
	g_static_mutex_unlock( &mutex );
	
	return ret;
}

/* static stuff */

static const gchar *screem_tag_tree_dtd_cache( ScreemTagTree *tree,
					ScreemDTD *dtd )
{
	const gchar *ret;
	GSList *tmp;
	const GSList *elements;
	GString *str;
	gchar *data;
	gchar *escaped;
	gboolean cache;
	
	g_return_val_if_fail( SCREEM_IS_DTD( dtd ), NULL );
	
	ret = g_object_get_data( G_OBJECT( dtd ),
				"tagtree_string" );
	if( ! ret ) {
		const gchar *pub;
		gboolean html;
		const gchar *closing;
		
		cache = ( g_object_get_data( G_OBJECT( dtd ),
				"NO_CACHE" ) == NULL );
		
		elements = screem_dtd_get_elements( dtd );
	
		html = FALSE;
		pub = screem_dtd_get_public_id( dtd );
		if( pub ) {
			html = ( strstr( pub, " HTML " ) != NULL );
			html |= ( strstr( pub, " html " ) != NULL );
			escaped = g_markup_escape_text( pub, strlen( pub ) );
		} else {
			pub = screem_dtd_get_system_id( dtd );
			if( pub ) {
				escaped = g_markup_escape_text( pub, strlen( pub ) );
			} else {
				escaped = g_strdup( "" );
			}
		}
		
		if( html ) {
			closing = "&gt;";
		} else {
			closing = " /&gt;";
		}
		
		str = g_string_new( "<?xml version=\"1.0\"?>" );
		g_string_append_printf( str, "<refs><ref name=\"%s\" description=\"%s:\n%s\">",
					_( "Current Doctype" ),
				_( "Elements for the current doctype" ),
				escaped );
		g_free( escaped );

		while( elements ) {
			const gchar *name;
			const gchar *desc;
			ScreemDTDTagExistance close;
			const GSList *attrs;
			const gchar *adesc;

			name = screem_dtd_element_get_name( (ScreemDTDElement*)elements->data );
			desc = screem_dtd_element_get_description( (ScreemDTDElement*)elements->data );		
			close = screem_dtd_element_close_state( dtd, name );
			if( close != SCREEM_DTD_MUST_NOT ) {
				g_string_append_printf(str, 
						"<entry name=\"%s\"><insert open=\"&lt;%s&gt;\"",
						name, name );
				g_string_append_printf( str, " close=\"&lt;/%s&gt;\"",
							name );
			} else {
				g_string_append_printf(str, 
						"<entry name=\"%s\"><insert open=\"&lt;%s%s\"",
						name, name, closing );

			}
			g_string_append( str, "/>");
			if( desc ) {
				escaped = g_markup_escape_text( desc, strlen( desc ) );
				g_string_append_printf( str, 
						"<description>%s</description>",
						escaped );
				g_free( escaped );
			}
			
			attrs = screem_dtd_element_get_attrs( (ScreemDTDElement*)elements->data );
			while( attrs ) {
				const ScreemDTDAttribute *attr;

				attr = (const ScreemDTDAttribute*)attrs->data;
				adesc = screem_dtd_attribute_get_description( attr );
				if( ! adesc ) {
					adesc = "";
				} 
				escaped = g_markup_escape_text( adesc, strlen( adesc ) );
				g_string_append_printf( str,
						"<property name=\"%s\" required=\"%i\" vallist=\"0\" default=\"%s\">%s</property>",
						screem_dtd_attribute_get_name( attr ),
						screem_dtd_attribute_get_required( attr ),
						"",
						escaped
						);
				g_free( escaped );
				attrs = attrs->next;
			}
		
			g_string_append( str, "</entry>" );
	
			elements = elements->next;
		}

		g_string_append( str, "</ref></refs>" );

		ret = str->str;

		
		if( cache ) {
			tmp = g_slist_append( tree->private->dtdcache,
					dtd );
			tree->private->dtdcache = tmp;
			g_object_set_data( G_OBJECT( dtd ), 
					"tagtree_string",
				   (gpointer)ret );
		} else {
			g_object_set_data_full( G_OBJECT( dtd ), 
					"tagtree_string",
					(gpointer)ret, g_free );
		}
		
		g_string_free( str, FALSE );
		
		if( cache &&
			( ++ tree->private->dtdcachelen ) > 4 ) {
			tmp = tree->private->dtdcache;
			tree->private->dtdcache = tmp->next;
			tmp->next = NULL;
			dtd = tmp->data;
			data = g_object_get_data( G_OBJECT( dtd ),
						  "tagtree_string" );
			g_free( data );
			g_object_set_data( G_OBJECT( dtd ), "tagtree_string",
					   NULL );
			g_slist_free( tmp );
			tree->private->dtdcachelen --;
		}
	}
	return ret;
}

static void screem_tag_tree_add_dtd( ScreemPage *page, GParamSpec *spec,
				     ScreemTagTree *tag_tree )
{
	ScreemTagTreePrivate *priv;
	ScreemDTD *dtd;
	const gchar *str;
	
	priv = tag_tree->private;
	
	if( spec ) {
		dtd = screem_page_get_dtd( page );
		if( dtd == priv->dtd ) {
			return;
		}
		if( dtd ) {
			g_object_ref( dtd );
		}
		if( priv->dtd ) {
			g_object_unref( priv->dtd );
		}
		priv->dtd = dtd;
	}
	
	dtd = priv->dtd;
	if( dtd ) {
		str = screem_tag_tree_dtd_cache( tag_tree, dtd );
		screem_tag_file_set_from_string( priv->dtdbranch,
						str, 0 );
	}
}

static gboolean screem_tag_tree_click( GtkTreeView *tree_view,
				       GtkTreePath *path,
				       GtkTreeViewColumn *column,
				       gpointer data )
{
	GtkTreeIter iter;
	GtkTreeModel *model;
	GtkTreeSelection *sel;
	ScreemTagTree *tag_tree;
	guint pos;

	tag_tree = SCREEM_TAG_TREE( data );

	sel = gtk_tree_view_get_selection( GTK_TREE_VIEW( tree_view ) );

	if( sel ) {
		NodeInfo *info;
		ScreemWindow *window;
		ScreemView *editor;

		window = SCREEM_WINDOW( tag_tree->private->window );

		editor = SCREEM_VIEW( window->details->editor );

		gtk_tree_selection_get_selected( sel, &model, &iter );
		gtk_tree_model_get( model, &iter, 
				SCREEM_TAG_FILE_DATA, &info, -1 );

		if( info && ( ! info->text ) && info->file ) {
			GnomeVFSURI *vuri;
			
			vuri = gnome_vfs_uri_new( info->file );
			if( ! vuri ) {
				screem_window_show_error( window, _( "Invalid URI for requested resource" ), FALSE );
			} else if( gnome_vfs_uri_is_local( vuri ) ||
				  ! window->details->offline ) {
				/*  set info->text to be the file 
				 *  contents, this way we don't
				 *  refetch files we already have */
				info->text = load_file( info->file, 
						NULL, NULL, NULL );
			} else {
				screem_window_show_message( window, _( "Remote resources are not available in offline mode" ), FALSE );	
			}

			if( vuri ) {	
				gnome_vfs_uri_unref( vuri );
			}
		} else if( info && info->open ) {
			screem_view_insert_markup( SCREEM_VIEW( editor ),
					info->open, info->close );
		}
		/* we do info->text here so you can have something like
		 * <insert open="&lt;title&gt;" close="&lt;title&gt;">
		 *      page title
		 * </insert>
		 */
		if( info && info->text ) {
			screem_view_insert( SCREEM_VIEW( editor ),
					-1, info->text );
			if( g_str_has_suffix( info->text, ")" ) ) {
				pos = screem_view_get_pos( SCREEM_VIEW( editor ) );
				screem_view_set_pos( SCREEM_VIEW( editor ), pos - 1 );
			}
		}
	} else {
		/* hmm, no iterator */
		
	}

	return (sel != NULL);
}


static gboolean screem_tag_tree_press( GtkWidget *widget, 
				       GdkEventButton *event,
				       ScreemTagTree *tag_tree )
{
	
	return FALSE;
}

static void screem_tagtree_sel_change( GtkTreeSelection *selection,
					ScreemTagTree *tree )
{
	ScreemTagTreePrivate *priv;
	ScreemWindow *window;
	gint rows;
	GtkTreeModel *model;
	GtkTreeIter iter;
	NodeInfo *info;
	
	priv = tree->private;
	
	window = SCREEM_WINDOW( priv->window );
	rows = gtk_tree_selection_count_selected_rows( selection );
	if( rows > 0 ) {
		gtk_tree_selection_get_selected( selection, &model, &iter );
		gtk_tree_model_get( model, &iter, 
				SCREEM_TAG_FILE_DATA, &info, -1 );
		if( info && info->description ) {
			screem_window_show_message( window, info->description, FALSE );
		}
	}
	gtk_widget_set_sensitive( priv->toolbar, ( rows > 0 ) );
}

static gboolean screem_tag_tree_set_model( ScreemTagTree *tree )
{
	ScreemTagTreePrivate *priv;
	GtkTreeModel *model;
	
	g_return_val_if_fail( SCREEM_IS_TAG_TREE( tree ), FALSE );
	
	priv = tree->private;

	priv->umodel = egg_tree_model_union_new( SCREEM_TAG_FILE_COLS,
			GDK_TYPE_PIXBUF, G_TYPE_STRING, 
			G_TYPE_POINTER );

	priv->dtdbranch = screem_tag_file_new();

	model = gtk_tree_model_sort_new_with_model( priv->umodel );
	gtk_tree_sortable_set_sort_func( GTK_TREE_SORTABLE( model ),
			0, screem_tag_tree_sort, tree, NULL );
	gtk_tree_sortable_set_sort_column_id( GTK_TREE_SORTABLE( model ),
			SCREEM_TAG_FILE_NAME, GTK_SORT_ASCENDING );
	priv->sorted = model;

	screem_tag_tree_add_file( tree, priv->dtdbranch, TRUE );

	gdk_threads_enter();
	gtk_tree_view_set_model( GTK_TREE_VIEW( tree->private->view ), 
			model );
	gdk_threads_leave();

	g_object_unref( model );

	return FALSE;
}

static gint tagtree_filter( const GnomeVFSFileInfo *info )
{
	const gchar *mime_type;
	gchar *type;
	gint ret;
	
	mime_type = info->mime_type;

	type = screem_get_mime_type_override( info->name, mime_type ); 

	ret = -1;
	if( type ) {
		ret = strcmp( "application/x-screem-tag-tree", 
				type );
		g_free( type );
	}

	return ret;
}
#if 0
static gint weml_filter( const GnomeVFSFileInfo *info )
{
	const gchar *mime_type;
	gchar *type;
	gint ret;
	
	mime_type = info->mime_type;

	type = screem_get_mime_type_override( info->name, mime_type ); 

	ret = -1;
	if( type ) {
		ret = strcmp( "application/x-weml+xml", 
				type );
		g_free( type );
	}

	return ret;

}
#endif
static gboolean screem_tag_tree_add_screem_format( gpointer data )
{
	screem_tag_tree_add_uri( data, SCREEM_TAG_TREE_FORMAT );

	return FALSE;
}

static gpointer screem_tag_tree_create( gpointer data )
{
	GSList *files;
	GSList *tmp;
	
	gint i;
	gchar *treepaths[] = {
		DATADIR"/screem/tagtrees",
		NULL,
		NULL
	};
	xmlNodePtr node;
	xmlNodePtr dead;
	xmlChar *file;
	GnomeVFSResult res;

	/* FIXME: windows should show a busy-interactive cursor
	   while this function is running */

	/* load tag trees */
	treepaths[ 1 ] = screem_get_dot_dir();

	for( i = 0; treepaths[ i ]; ++ i ) {
		files = screem_vfs_scandir( treepaths[ i ],
				tagtree_filter, NULL, FALSE );
		for( tmp = files; tmp; tmp = tmp->next ) {
			screem_tag_tree_add_screem_format( tmp->data );
			g_free( tmp->data );
		}
		g_slist_free( files );
	}
	g_free( (gchar*)treepaths[ 1 ] );

	/* now load any additional trees that may be present in the
	 * tagtreeconfig.xml */
	g_static_mutex_lock( &mutex );
	files = NULL;
	node = xmlDocGetRootElement( config );
	node = node->children;
	while( node ) {
		file = xmlGetProp( node, (const xmlChar*)"src" );

		if( file && uri_exists( (const gchar*)file, &res ) ) {
			files = g_slist_prepend( files, file );
			node = node->next;
		} else if( file && res == GNOME_VFS_ERROR_NOT_FOUND ){
			/* remove if not found, we will
			 * allow other errors to keep the uri
			 * in the tag config */
			dead = node;
			node = node->next;
			xmlUnlinkNode( dead );
			xmlFreeNode( dead );
			xmlFree( file );
		}
	}
	screem_tag_tree_save_config();
	g_static_mutex_unlock( &mutex );
	for( tmp = files; tmp; tmp = tmp->next ) {
		screem_tag_tree_add_screem_format( tmp->data );
		g_free( tmp->data );
	}
	g_slist_free( files );

	return NULL;
}

static void add_widget( GtkUIManager *merge, GtkWidget *widget, ScreemTagTree *tree )
{
	ScreemTagTreePrivate *priv;
	GtkWidget *box;
	
	priv = tree->private;
	box = priv->box;
	
	if( GTK_IS_TOOLBAR( widget ) ) {
		gtk_toolbar_set_show_arrow( GTK_TOOLBAR( widget ), TRUE );
		gtk_toolbar_set_tooltips( GTK_TOOLBAR( widget ), TRUE );
		
		gtk_box_pack_start( GTK_BOX( box ), widget,
				    FALSE, TRUE, 0 );
		gtk_widget_set_sensitive( widget, FALSE );
		priv->toolbar = widget;
	}
}

static void screem_tag_tree_tag_help( GtkAction *action, ScreemTagTree *tree )
{
	ScreemTagTreePrivate *priv;
	GtkTreeView *view;
	GtkTreeSelection *selection;
	GtkTreeModel *model;
	GtkTreeIter it;
	gint rows;
	NodeInfo *info;
	
	priv = tree->private;

	view = GTK_TREE_VIEW( priv->view );
	selection = gtk_tree_view_get_selection( view );
	
	rows = gtk_tree_selection_count_selected_rows( selection );
	if( rows > 0 ) {
		gtk_tree_selection_get_selected( selection, &model, &it );
		gtk_tree_model_get( model, &it, 
				SCREEM_TAG_FILE_DATA, &info, -1 );
		if( info ) {
			GtkWidget *dialog;
			GtkWidget *box;
			GtkWidget *widget;
			GtkWidget *notebook;
			gchar *title;
			ScreemWindow *window;
			ScreemView *editor;
			GtkWidget *sw;
			GtkTextBuffer *buffer;
			gchar *txt;
			GtkSizeGroup *group;
			GSList *attrs;
			NodeAttribute *attr;
			GtkWidget *attrbox;
			gchar *attrtxt;
			
			g_object_get( G_OBJECT( tree ), "window", &window, NULL );
			editor = SCREEM_VIEW( window->details->editor );
			
			box = gtk_vbox_new( FALSE, 0 );
			gtk_box_set_spacing( GTK_BOX( box ), 6 );
			gtk_container_set_border_width( GTK_CONTAINER( box ), 12 );
		
			if( ! info->isfunc ) {
				title = g_strconcat( "<span size=\"xx-large\">",
							info->name,
							"</span>",
							NULL );
			} else {
				title = g_strconcat( "<span size=\"xx-large\">",
							info->name,
							"()</span>",
							NULL );
			}
			widget = gtk_label_new( title );
			gtk_label_set_use_markup( GTK_LABEL( widget ), TRUE );
			g_free( title );
			gtk_label_set_justify( GTK_LABEL( widget ), GTK_JUSTIFY_LEFT );
			gtk_misc_set_alignment( GTK_MISC( widget ), 0.0, 0.0 );
			gtk_box_pack_start( GTK_BOX( box ), widget, FALSE, TRUE, 0 );
			gtk_label_set_line_wrap( GTK_LABEL( widget ), TRUE );
			gtk_label_set_selectable( GTK_LABEL( widget ), TRUE );
	

			widget = gtk_label_new( info->description );
			gtk_label_set_justify( GTK_LABEL( widget ), GTK_JUSTIFY_LEFT );
			gtk_misc_set_alignment( GTK_MISC( widget ), 0.0, 0.0 );
			gtk_label_set_line_wrap( GTK_LABEL( widget ), TRUE );
			gtk_label_set_selectable( GTK_LABEL( widget ), TRUE );
			gtk_box_pack_start( GTK_BOX( box ), widget, FALSE, TRUE, 0 );

			txt = NULL;
			if( ( ! info->text ) && info->file ) {
				GnomeVFSURI *vuri;
			
				vuri = gnome_vfs_uri_new( info->file );
				if( ! vuri ) {
					txt = g_strdup( _( "Invalid URI for requested resource" ) );
				} else if( gnome_vfs_uri_is_local( vuri ) ||
					  ! window->details->offline ) {
					/* now set info->text to be the
					 * file contents, this way we
					 * don't refetch files we
					 * already have */
					info->text = load_file( info->file, 
							NULL, NULL, 
							NULL );
				} else {
					txt = g_strdup( _( "Remote resources are not available in offline mode" ) );	
				}

				if( vuri ) {	
					gnome_vfs_uri_unref( vuri );
				}
			} 
			if( info->open ) {
				txt = g_strconcat( info->open, info->close, NULL );
			}
			if( info->text ) {
				g_free( txt );
				txt = g_strdup( info->text );
			}
			
			notebook = gtk_notebook_new();
			gtk_notebook_set_show_border( GTK_NOTEBOOK( notebook ),
							FALSE );
			gtk_notebook_set_tab_pos( GTK_NOTEBOOK( notebook ),
						GTK_POS_BOTTOM );
			gtk_box_pack_start( GTK_BOX( box ), notebook, TRUE, TRUE, 0 );
	

			if( info->attributes ) {
				sw = gtk_scrolled_window_new( NULL, NULL );
				gtk_scrolled_window_set_policy( GTK_SCROLLED_WINDOW( sw ),
								GTK_POLICY_NEVER,
								GTK_POLICY_AUTOMATIC );
				gtk_scrolled_window_set_shadow_type( GTK_SCROLLED_WINDOW( sw ),
								GTK_SHADOW_IN );

				group = gtk_size_group_new( GTK_SIZE_GROUP_HORIZONTAL );

				attrbox = gtk_vbox_new( FALSE, 0 );

				for( attrs = info->attributes; attrs; attrs = attrs->next ) {
					attr = (NodeAttribute*)attrs->data;

					if( attr->type ) {
						attrtxt = g_strconcat( "<b><i>", attr->name, "(",
									attr->type,
									") - </i></b>",
									attr->description,
									NULL );
					} else {
						attrtxt = g_strconcat( "<b><i>", attr->name,
									" - </i></b>",
									attr->description,
									NULL );
					}
					widget = gtk_label_new( attrtxt );
					gtk_label_set_use_markup( GTK_LABEL( widget ), TRUE );
					g_free( attrtxt );
					gtk_size_group_add_widget( group, widget );
				
					gtk_label_set_justify( GTK_LABEL( widget ), GTK_JUSTIFY_LEFT );
					gtk_misc_set_alignment( GTK_MISC( widget ), 0.0, 0.0 );
					gtk_label_set_line_wrap( GTK_LABEL( widget ), TRUE );
					gtk_label_set_selectable( GTK_LABEL( widget ), TRUE );

					gtk_box_pack_start( GTK_BOX( attrbox ), widget, FALSE, FALSE, 0 );
					gtk_box_reorder_child( GTK_BOX( attrbox ), widget, 0 );
				}
				gtk_container_set_border_width( GTK_CONTAINER( attrbox ), 6 );
				gtk_box_set_spacing( GTK_BOX( attrbox ), 6 );
				gtk_scrolled_window_add_with_viewport( GTK_SCROLLED_WINDOW( sw ),
								attrbox );
				
				if( info->isfunc ) {
					widget = gtk_label_new( _( "Parameters" ) );
				} else {
					widget = gtk_label_new( _( "Attributes" ) );
				}
				
				gtk_notebook_append_page( GTK_NOTEBOOK( notebook ),
							  sw, widget );
			}
			
			if( txt ) {
				sw = gtk_scrolled_window_new( NULL, NULL );
				gtk_scrolled_window_set_policy( GTK_SCROLLED_WINDOW( sw ),
								GTK_POLICY_AUTOMATIC,
								GTK_POLICY_AUTOMATIC );
				gtk_scrolled_window_set_shadow_type( GTK_SCROLLED_WINDOW( sw ),
								GTK_SHADOW_IN );
				widget = gtk_source_view_new();
				gtk_text_view_set_editable( GTK_TEXT_VIEW( widget ), FALSE );
				gtk_text_view_set_wrap_mode( GTK_TEXT_VIEW( widget ), GTK_WRAP_WORD );
				buffer = gtk_text_view_get_buffer( GTK_TEXT_VIEW( widget ) );
				gtk_text_buffer_set_text( buffer, txt, -1 );
				gtk_container_add( GTK_CONTAINER( sw ), widget );

				widget = gtk_label_new( _( "Text to Insert" ) );
				gtk_notebook_append_page( GTK_NOTEBOOK( notebook ),
							 sw, widget );
				dialog = gtk_dialog_new_with_buttons( _( "Tag Help - Screem" ),
								GTK_WINDOW( window ),
								GTK_DIALOG_MODAL | 
								GTK_DIALOG_NO_SEPARATOR,
								GTK_STOCK_CLOSE,
								GTK_RESPONSE_CLOSE,
								GTK_STOCK_ADD,
								GTK_RESPONSE_APPLY,
								NULL);
			} else {
				dialog = gtk_dialog_new_with_buttons( _( "Tag Help - Screem" ),
								GTK_WINDOW( window ),
								GTK_DIALOG_MODAL | 
								GTK_DIALOG_NO_SEPARATOR,
								GTK_STOCK_CLOSE,
								GTK_RESPONSE_CLOSE,
								NULL );
			}
			gtk_widget_show_all( box );
			
			gtk_box_pack_start( GTK_BOX( GTK_DIALOG( dialog )->vbox ), box, TRUE, TRUE, 0 );

			if( gtk_dialog_run( GTK_DIALOG( dialog ) ) == GTK_RESPONSE_APPLY ) {
				/* insert txt into the editor */
				if( info->open ) {
					screem_view_insert_markup( SCREEM_VIEW( editor ), info->open, info->close );
				} else {
					screem_view_insert( SCREEM_VIEW( editor ), -1, txt );
				}
			}
			g_free( txt );
			
			
			gtk_widget_destroy( dialog );
		}
	} else {
		g_warning( "no item selected for info\n" );
	}
}

static gboolean screem_tag_tree_page_set( gpointer data )
{
	ScreemTagTree *tree;
	ScreemPage *page;
	ScreemDTD *dtd;
	
	tree = SCREEM_TAG_TREE( data );
	page = tree->private->page;

	if( ! tree->private->dtdbranch ) {
		return TRUE;
	}
	
	if( page ) {
		g_signal_connect(G_OBJECT( page ), "notify::dtd",
				 G_CALLBACK( screem_tag_tree_add_dtd ),
				 tree );
		dtd = screem_page_get_dtd( page );
		if( dtd != tree->private->dtd ) {
			if( dtd ) {
				g_object_ref( dtd );
			}
			if( tree->private->dtd ) {
				g_object_unref( tree->private->dtd );
			}
			tree->private->dtd = dtd;
			gdk_threads_enter();
			screem_tag_tree_add_dtd( page, NULL, tree );
			gdk_threads_leave();
		}
	} else {
		tree->private->dtd = NULL;
	}

	tree->private->timeout = 0;
	
	return FALSE;
}

static void screem_tag_tree_add_file( ScreemTagTree *tree,
					ScreemTagFile *file,
					gboolean prepend )
{
	GtkTreeModel *model;
	
	g_return_if_fail( SCREEM_IS_TAG_TREE( tree ) );
	g_return_if_fail( SCREEM_IS_TAGFILE( file ) );

	model = screem_tag_file_get_model( file );

	if( egg_tree_model_union_contains( EGG_TREE_MODEL_UNION( tree->private->umodel ), model ) ) {
		g_warning( "attempted to add tag file twice\n" );
		return;
	}
	
	gdk_threads_enter();
	
	if( prepend ) {
		egg_tree_model_union_prepend( EGG_TREE_MODEL_UNION( tree->private->umodel ), model );
	} else {
		egg_tree_model_union_append( EGG_TREE_MODEL_UNION( tree->private->umodel ), model );

	}

	gdk_threads_leave();
}

static void screem_tag_tree_remove_file( ScreemTagTree *tree,
					ScreemTagFile *file )
{
	GtkTreeModel *model;
	
	g_return_if_fail( SCREEM_IS_TAG_TREE( tree ) );
	g_return_if_fail( SCREEM_IS_TAGFILE( file ) );

	model = screem_tag_file_get_model( file );

	if( egg_tree_model_union_contains( EGG_TREE_MODEL_UNION( tree->private->umodel ), model ) ) {
		egg_tree_model_union_remove( EGG_TREE_MODEL_UNION( tree->private->umodel ), model );
	}
}

static gboolean active_set_cb( ScreemTagFile *file )
{
	gboolean active;
	const gchar *uri;
	GtkTreeIter it;
	GSList *tmp;
	xmlNodePtr node;
	
	const gchar *name;
	const gchar *desc;

	gboolean found;
	
	g_static_mutex_lock( &mutex );

	uri = screem_tag_file_get_uri( file );
	g_object_get( G_OBJECT( file ), 
			"active", &active, 
			NULL );
	
	found = screem_gtk_list_store_find_string( filesstore, 
				&it, TAG_FILE_URI_COL, uri );
	g_assert( found );

	name = screem_tag_file_get_name( file );
	desc = screem_tag_file_get_desc( file );
		
	node = screem_tag_tree_get_file_node( uri );
	if( ! node ) {
		/* new node being added, add to config */
		node = xmlDocGetRootElement( config );
		node = xmlNewChild( node, NULL, 
				(const xmlChar*)"tagfile", NULL );
	
		xmlSetProp( node, (const xmlChar*)"src", 
				(xmlChar*)uri );
		if( name ) {
			xmlSetProp( node, (const xmlChar*)"name", 
					(xmlChar*)name );
		}
		if( desc ) {
			xmlSetProp( node, 
					(const xmlChar*)"description",
					(xmlChar*)desc );
		}
	}
	
	gdk_threads_enter();
	gtk_list_store_set( filesstore, &it,
			TAG_FILE_NAME_COL, name,
			TAG_FILE_DESC_COL, desc,
			TAG_FILE_ACTIVE_COL, active, -1 );
	gdk_threads_leave();

	if( active ) {
		for( tmp = trees; tmp; tmp = tmp->next ) {
			screem_tag_tree_add_file( tmp->data,
						file, FALSE );
		}
	} else {
		for( tmp = trees; tmp; tmp = tmp->next ) {
			screem_tag_tree_remove_file( tmp->data,
							file );
		}
	}

	xmlSetProp( node, (const xmlChar*)"enable", 
			active ? (const xmlChar*)"1" : 
			(const xmlChar*) "0" );
	
	screem_tag_tree_save_config();
	
	g_static_mutex_unlock( &mutex );

	return FALSE;
}

static void screem_tag_file_active_set( ScreemTagFile *file,
					GParamSpec *spec,
					gpointer data )
{
	g_idle_add( (GSourceFunc)active_set_cb, file );
}

/* NOTE: do not call without the mutex locked */
static void screem_tag_tree_save_config( void )
{
	xmlNodePtr node;
	GSList *tmp;
	const gchar *uri;
	const gchar *name;
	const gchar *desc;
	
	for( tmp = treefiles; tmp; tmp = tmp->next ) {
		uri = screem_tag_file_get_uri( SCREEM_TAGFILE( tmp->data ) );
		node = screem_tag_tree_get_file_node( uri );

		/* possible we don't find a node for new tag
		 * files that are still loading, waiting to be loaded */
		if( node ) {
			name = screem_tag_file_get_name( SCREEM_TAGFILE( tmp->data ) );
			desc = screem_tag_file_get_desc( SCREEM_TAGFILE( tmp->data ) );
			xmlSetProp( node, (const xmlChar*)"name", 
					(xmlChar*)name );
			xmlSetProp( node, (const xmlChar*)"description",
					(xmlChar*)desc );
		}
	}
	
	xmlSaveFile( configpath, config );
}

/* NOTE: do not call without the mutex locked */
static xmlNodePtr screem_tag_tree_get_file_node( const gchar *url )
{
	xmlNodePtr ret;
	xmlChar *src;

	ret = NULL;
	
	ret = xmlDocGetRootElement( config );

	g_return_val_if_fail( ret != NULL, NULL );
	
	ret = ret->children;
	while( ret ) {
		src = xmlGetProp( ret, (const xmlChar*)"src" );
		if( src ) {
			if( ! strcmp( (const gchar*)src, url ) ) {
				xmlFree( src );
				break;
			}
			xmlFree( src );
		}
		ret = ret->next;
	}
	
	return ret;
}
/* NOTE: do not call without the mutex locked */
static gboolean screem_tag_tree_is_active( const gchar *url,
					gchar **name,
					gchar **desc )
{
	gboolean ret;
	xmlNodePtr node;
	xmlChar *enable;
	
	g_assert( name != NULL );
	g_assert( desc != NULL );
	
	ret = TRUE;

	*name = NULL;
	*desc = NULL;

	node = screem_tag_tree_get_file_node( url );
	if( node ) {
		enable = xmlGetProp( node, (const xmlChar*)"enable" );
		if( enable ) {
			ret = ( *enable == '1' );
			xmlFree( enable );
		}
		*name = (gchar*)xmlGetProp( node, (const xmlChar*)"name" );
		*desc = (gchar*)xmlGetProp( node, (const xmlChar*)"description" );
	} 	

	return ret;
}

static void screem_tag_tree_search( GtkAction *action,
				gpointer data )
{
	ScreemTagTree *tree;
	ScreemTagTreePrivate *priv;
	GtkWidget *widget;
	GtkTreeModel *model;
	GladeXML *xml;
	GtkCellRenderer *rend;
	GtkCellRenderer *prend;
	GtkTreeViewColumn *col;
	GtkTreeSelection *selection;
	GtkTreeIter it;
	GtkTreePath *path;
	gint *indices;
	gint i;
	NodeInfo *info;
	
	tree = SCREEM_TAG_TREE( data );
	priv = tree->private;
	
	selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( priv->view ) );
	i = -1;
	if( gtk_tree_selection_get_selected( selection, &model, &it ) ) {
		path = gtk_tree_model_get_path( model, &it );	
		indices = gtk_tree_path_get_indices( path );
		i = indices[ 0 ];
		gtk_tree_path_free( path );
	}
	
	xml = glade_xml_new( GLADE_PATH"/screem.glade", 
			"find_tagtree", NULL );
	
	widget = glade_xml_get_widget( xml, "tagtrees" );
	model = screem_top_level_model_new();
	screem_top_level_model_set_model( SCREEM_TOP_LEVEL_MODEL( model ), priv->sorted );
	
	selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( widget ) );

	prend = gtk_cell_renderer_pixbuf_new();
	rend = gtk_cell_renderer_text_new();
	col = gtk_tree_view_column_new();
	
	gtk_tree_view_column_pack_start( col, prend, FALSE );
	gtk_tree_view_column_pack_start( col, rend, TRUE );
	gtk_tree_view_column_set_resizable( col, TRUE );
	gtk_tree_view_append_column( GTK_TREE_VIEW( widget ), col );
	
	gtk_tree_view_column_set_title( col, _( "Resources" ) );
	gtk_tree_view_column_set_attributes( col, rend,
			"text", SCREEM_TAG_FILE_NAME, NULL );
	gtk_tree_view_column_set_attributes( col, prend,
			"pixbuf", SCREEM_TAG_FILE_ICON, NULL );

	/* HIG says 1 column lists should have no header */
	gtk_tree_view_set_headers_visible( GTK_TREE_VIEW( widget ), 
					FALSE ); 

	
	gtk_tree_view_set_model( GTK_TREE_VIEW( widget ), model );
	g_object_unref( model );
	
	if( i != -1 ) {
		path = gtk_tree_path_new_from_indices( i, -1 );
		gtk_tree_selection_select_path( selection, path );
	
		gtk_tree_view_scroll_to_cell( GTK_TREE_VIEW( widget ), 
				path, col, TRUE, 0.0, 0.5 );
		gtk_tree_path_free( path );
	}

	widget = glade_xml_get_widget( xml, "tag_tree_find_pattern" );
	screem_gtk_add_history( widget );
	
	widget = glade_xml_get_widget( xml, "find_tagtree" );
	gtk_widget_show_all( widget );
	
	if( gtk_dialog_run( GTK_DIALOG( widget ) ) == GTK_RESPONSE_OK ) {
		widget = glade_xml_get_widget( xml, "tag_tree_find_pattern" );
		widget = GTK_BIN( widget )->child;
		priv->search_pattern = gtk_entry_get_text( GTK_ENTRY( widget ) );
		widget = glade_xml_get_widget( xml, "regexp" );
		priv->regexp = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) );
		if( gtk_tree_selection_get_selected( selection, &model, &it ) ) {
			path = gtk_tree_model_get_path( model, &it );
			gtk_tree_model_get( model, &it,
					SCREEM_TAG_FILE_DATA, &info,
					-1 );
			if( info ) {
				model = screem_tag_file_get_model( SCREEM_TAGFILE( info->tfile ) );
				gtk_tree_model_foreach( model, 
					screem_tag_tree_search_func,
					tree );
			}

			if( priv->search_result ) {
				/* modify the path based on its position
				 * in the union model */
				gtk_tree_path_get_indices( priv->search_result )[ 0 ] = gtk_tree_path_get_indices( path )[ 0 ];

				selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( priv->view ) );
				gtk_tree_view_expand_to_path( GTK_TREE_VIEW( priv->view ), priv->search_result );	
				gtk_tree_selection_select_path( selection, priv->search_result );
				col = gtk_tree_view_get_column( GTK_TREE_VIEW( priv->view ), 0 );
				gtk_tree_view_scroll_to_cell( GTK_TREE_VIEW( priv->view ), priv->search_result, col, TRUE, 0.0, 0.5 );
				gtk_tree_path_free( priv->search_result );
				priv->search_result = NULL;
			} else {
				/* not found */
			}

			gtk_tree_path_free( path );
		}
	}
	
	widget = glade_xml_get_widget( xml, "find_tagtree" );
	gtk_widget_destroy( widget );	
	g_object_unref( G_OBJECT( xml ) );
}

static gboolean screem_tag_tree_search_func( GtkTreeModel *model,
					GtkTreePath *path,
					GtkTreeIter *iter,
					gpointer data )
{
	ScreemTagTree *tree;
	ScreemTagTreePrivate *priv;
	gchar *name;
	guint len;

	tree = SCREEM_TAG_TREE( data );
	priv = tree->private;

	gtk_tree_model_get( model, iter,
			SCREEM_TAG_FILE_NAME, &name, -1 );

	priv->search_result = NULL;

	if( ! screem_search_static( name, priv->search_pattern,
				priv->regexp, 0, &len, NULL, NULL ) ) {
		priv->search_result = gtk_tree_path_copy( path );
	}

	g_free( name );
	
	return ( priv->search_result != NULL );
}

static gint screem_tag_tree_sort( GtkTreeModel *model,
		GtkTreeIter *a, GtkTreeIter *b,
		gpointer data )
{
	ScreemTagTree *tree;
	gchar *aname;
	gchar *bname;
	gint ret;
	
	tree = SCREEM_TAG_TREE( data );
	
	gtk_tree_model_get( model, a, SCREEM_TAG_FILE_NAME, &aname,
			-1 );
	gtk_tree_model_get( model, b, SCREEM_TAG_FILE_NAME, &bname,
			-1 );

	ret = 0;
	if( aname && ! bname ) {
		ret = 1;
	} else if( bname && ! aname ) {
		ret = -1;
	} else {
		ret = strcmp( aname, bname );
	}

	return ret;
}

static gboolean screem_tag_tree_add_current( ScreemTagTree *tree )
{
	GSList *tmp;
	
	g_static_mutex_lock( &mutex );
	trees = g_slist_prepend( trees, tree );

	for( tmp = treefiles; tmp; tmp = tmp->next ) {
		screem_tag_tree_add_file( tree, 
				SCREEM_TAGFILE( tmp->data ), FALSE );
	}
	g_static_mutex_unlock( &mutex );

	return FALSE;
}



/* G Object stuff */
G_DEFINE_TYPE( ScreemTagTree, screem_tag_tree, GTK_TYPE_BIN )

static void screem_tag_tree_class_init( ScreemTagTreeClass *klass )
{
	GObjectClass *object_class;
	GtkWidgetClass *widget_class;

	GParamSpec *pspec;

	object_class = G_OBJECT_CLASS( klass );
	widget_class = (GtkWidgetClass *)klass;

	object_class->finalize = screem_tag_tree_finalize;

	object_class->get_property = screem_tag_tree_get_prop;
	object_class->set_property = screem_tag_tree_set_prop;

	pspec = g_param_spec_pointer( "window", "window",
				      "window object",
				      G_PARAM_READABLE |
				      G_PARAM_WRITABLE );
	g_object_class_install_property( object_class,
					 ARG_WINDOW, pspec );

	pspec = g_param_spec_object( "page", "page",
				     "page",
				     SCREEM_TYPE_PAGE,
				     G_PARAM_WRITABLE );
	g_object_class_install_property( G_OBJECT_CLASS( object_class ),
					 ARG_PAGE,
					 pspec );

	
	widget_class->size_request = screem_tag_tree_size_request;
	widget_class->size_allocate = screem_tag_tree_size_allocate;
}

static void screem_tag_tree_init( ScreemTagTree *tag_tree )
{
	GtkWidget *sw;
	GtkCellRenderer *rend;
	GtkCellRenderer *prend;
	GtkTreeViewColumn *col;
	GtkTreeSelection *selection;
	gchar *dot;

	static const GtkActionEntry action_entries[] = {
		{
		   "SearchTagTree", GTK_STOCK_FIND,
		   N_( "Find in Tag Tree..." ),
		   "", N_( "Search the tag tree" ),
		   G_CALLBACK( screem_tag_tree_search )
		},
		{ 
		   "Tag Help", GTK_STOCK_DIALOG_INFO, N_( "Tag Help" ),
		   "", N_( "Help" ),
	  	   G_CALLBACK( screem_tag_tree_tag_help ) 
		}
	};
	static const guint action_n_entries=G_N_ELEMENTS( action_entries );
	GtkActionGroup *group;
	GtkUIManager *merge;
	GError *error;
	
	ScreemTagTreePrivate *priv;

	g_static_mutex_lock( &mutex );
	if( ! filesstore ) {
		filesstore = gtk_list_store_new( MAX_TAG_FILE_COLS,
						G_TYPE_STRING,
						G_TYPE_STRING,
						G_TYPE_STRING,
						G_TYPE_BOOLEAN,
						G_TYPE_POINTER );
		dot = screem_get_dot_dir();
		configpath = g_build_path( G_DIR_SEPARATOR_S,
				dot, "tagtreeconfig.xml",
				NULL );
		g_free( dot );
		config = NULL;
		if( uri_exists( configpath, NULL ) ) {
			config = xmlParseFile( configpath );
		}
		if( ! config ) {
			config = xmlNewDoc( (const xmlChar*)XML_DEFAULT_VERSION );
			xmlDocSetRootElement( config,
					xmlNewDocNode( config, NULL,
						(const xmlChar*)"refconfig", NULL ) );
		}
	}
	g_static_mutex_unlock( &mutex );
	priv = tag_tree->private = g_new0( ScreemTagTreePrivate, 1 );

	priv->box = gtk_vbox_new( FALSE, 0 );
	gtk_container_add( GTK_CONTAINER( tag_tree ), priv->box );
		
	sw = gtk_scrolled_window_new( NULL, NULL );
	gtk_scrolled_window_set_policy( GTK_SCROLLED_WINDOW( sw ),
					GTK_POLICY_AUTOMATIC,
                                        GTK_POLICY_AUTOMATIC );
	gtk_scrolled_window_set_shadow_type( GTK_SCROLLED_WINDOW( sw ),
						GTK_SHADOW_IN );
	priv->view = gtk_tree_view_new();
	
        gtk_tree_view_set_rules_hint( GTK_TREE_VIEW( priv->view ), TRUE );

	selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( priv->view ) );
	g_signal_connect( G_OBJECT( selection ), "changed",
			  G_CALLBACK( screem_tagtree_sel_change ), tag_tree );

	prend = gtk_cell_renderer_pixbuf_new();
	rend = gtk_cell_renderer_text_new();
	col = gtk_tree_view_column_new();
	
	gtk_tree_view_column_pack_start( col, prend, FALSE );
	gtk_tree_view_column_pack_start( col, rend, TRUE );
	gtk_tree_view_column_set_resizable( col, TRUE );
/*	gtk_tree_view_column_set_sizing( col, GTK_TREE_VIEW_COLUMN_FIXED );*/
	gtk_tree_view_append_column( GTK_TREE_VIEW( priv->view ), col );
	
	gtk_tree_view_column_set_title( col, _( "Resources" ) );
	gtk_tree_view_column_set_attributes( col, rend,
			"text", SCREEM_TAG_FILE_NAME, NULL );
	gtk_tree_view_column_set_attributes( col, prend,
			"pixbuf", SCREEM_TAG_FILE_ICON, NULL );

	/* HIG says 1 column lists should have no header */
	gtk_tree_view_set_headers_visible( GTK_TREE_VIEW( priv->view ), FALSE ); 

/*	horrible behaviour do not enable
 *	g_object_set( G_OBJECT( priv->view ), "fixed-height-mode", TRUE,
			NULL );*/
	
	gtk_widget_show( priv->view );

	g_signal_connect( G_OBJECT( priv->view ), "row_activated",
			  G_CALLBACK( screem_tag_tree_click ),
			  tag_tree );

	g_signal_connect( G_OBJECT( priv->view ), "button_press_event",
			  G_CALLBACK( screem_tag_tree_press ),
			  tag_tree );

	gtk_container_add( GTK_CONTAINER( sw ), priv->view );

	gtk_box_pack_start( GTK_BOX( priv->box ), sw, TRUE, TRUE, 0 );


	group = gtk_action_group_new( "ScreemTagTreeActions" );
	gtk_action_group_set_translation_domain( group, 
			GETTEXT_PACKAGE );	

	gtk_action_group_add_actions( GTK_ACTION_GROUP( group ),
					action_entries,
					action_n_entries,
					tag_tree );
	merge = gtk_ui_manager_new ();
	gtk_ui_manager_insert_action_group( merge, 
					    GTK_ACTION_GROUP( group ), 
					    0 );
	g_signal_connect( merge, "add_widget", G_CALLBACK( add_widget ), tag_tree );
  	if( ! gtk_ui_manager_add_ui_from_file( merge, 
					 UIDATADIR"/screem-tag-tree-bar.xml", 
					 &error ) ) {
		g_message( "Tag Tree ui file error = %s", 
				error->message );
		g_error_free( error );
	}
	
	gtk_widget_show_all( GTK_WIDGET( tag_tree ) );

	g_idle_add( (GSourceFunc)screem_tag_tree_set_model,
			tag_tree );

	g_idle_add( (GSourceFunc)screem_tag_tree_add_current,
			tag_tree );
}

static void screem_tag_tree_finalize( GObject *object )
{
	ScreemTagTree *tree;
	ScreemTagTreePrivate *priv;
	gboolean last;
	GSList *tmp;
	gboolean active;
	
	tree = SCREEM_TAG_TREE( object );
	priv = tree->private;

	if( priv->timeout != 0 ) {
		g_source_remove( priv->timeout );
	}

	/* clear and unref union model */
	g_static_mutex_lock( &mutex );
	
	for( tmp = treefiles; tmp; tmp = tmp->next ) {
		g_object_get( G_OBJECT( tmp->data ),
				"active", &active, NULL );
		if( active ) {
			screem_tag_tree_remove_file( tree,
					SCREEM_TAGFILE( tmp->data ) );
		}
	}
	screem_tag_tree_remove_file( tree, priv->dtdbranch );
	
	g_object_unref( priv->umodel );

	/* destroy dtd branch ScreemTagFile */
	g_object_unref( priv->dtdbranch );

	g_free( priv );
	
	G_OBJECT_CLASS( screem_tag_tree_parent_class )->finalize( object );

	trees = g_slist_remove( trees, tree );
	last = ( trees == NULL );
	if( last ) {
		gtk_list_store_clear( filesstore );
		g_object_unref( filesstore );
		for( tmp = treefiles; tmp; tmp = tmp->next ) {
			g_object_unref( G_OBJECT( tmp->data ) );
		}
	}
	
	g_static_mutex_unlock( &mutex );

	if( last ) {
		g_free( configpath );
		xmlFreeDoc( config );
	}
}

static void screem_tag_tree_set_prop( GObject *object, guint property_id,
				    const GValue *value, GParamSpec *pspec )
{
	ScreemTagTree *tree;
	ScreemPage *page;
	
	tree = SCREEM_TAG_TREE( object );

	switch( property_id ) {
	case ARG_WINDOW:
		tree->private->window = g_value_get_pointer( value );
		break;
	case ARG_PAGE:
		page = g_value_get_object( value );
		if( page != tree->private->page ) {
			if( tree->private->page ) {
				g_signal_handlers_disconnect_matched( G_OBJECT( tree->private->page ),
								      G_SIGNAL_MATCH_DATA,
								      0, 0, NULL, NULL, tree );
			}

			tree->private->page = page;
			if( tree->private->timeout ) {
				g_source_remove( tree->private->timeout );
				tree->private->timeout = 0;
			}
			tree->private->timeout = g_timeout_add( 2000, 
					(GSourceFunc)screem_tag_tree_page_set, 
					tree );
		}
		break;
	}
}

static void screem_tag_tree_get_prop( GObject *object, guint property_id,
				    GValue *value, GParamSpec *pspec)
{
	switch( property_id ) {
	case ARG_WINDOW:
		g_value_set_pointer( value,
				     (gpointer)
				     SCREEM_TAG_TREE( object )->private->window );
		break;
	}
}

static void screem_tag_tree_size_request( GtkWidget *widget,
                           	  	  GtkRequisition *requisition )
{
	GtkBin *bin;

	bin = GTK_BIN (widget);

	requisition->width = GTK_CONTAINER( widget )->border_width * 2;
	requisition->height = GTK_CONTAINER( widget )->border_width * 2;

	if( bin->child && GTK_WIDGET_VISIBLE( bin->child ) ) {
		GtkRequisition child_requisition;

		gtk_widget_size_request( bin->child, &child_requisition );
		requisition->width += child_requisition.width;
		requisition->height += child_requisition.height;
	}
}

static void screem_tag_tree_size_allocate( GtkWidget *widget,
                              		   GtkAllocation *allocation )
{
	GtkBin *bin;
	GtkAllocation child_allocation;

	bin = GTK_BIN( widget );
	widget->allocation = *allocation;
	
	if( bin->child ) {
	child_allocation.x = allocation->x + 
				GTK_CONTAINER( widget )->border_width; 
	child_allocation.y = allocation->y + 
				GTK_CONTAINER (widget)->border_width;
	child_allocation.width = MAX( allocation->width - 
				      GTK_CONTAINER( widget )->border_width * 2,
					0);
	child_allocation.height = MAX( allocation->height - 
				       GTK_CONTAINER (widget)->border_width * 2,
					0);
	gtk_widget_size_allocate( bin->child, &child_allocation );
	}
}

