/*  Screem:  screem-page-druid.c,
 *  A page druid for guiding users through the process of creating a page
 *
 *  Copyright (C) 2002  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>

#include <unistd.h>
#include <pwd.h>
#include <sys/types.h>

#include <gconf/gconf-client.h>

#include <glade/glade.h>

#include <glib/gi18n.h>
#include <libgnomeui/gnome-druid.h>

#include <libgnomevfs/gnome-vfs-uri.h>
#include <libgnomevfs/gnome-vfs-utils.h>

#include <gtk/gtk.h>

#include <string.h>

#include "screem-application.h"
#include "screem-window.h"
#include "screem-page-druid.h"
#include "screem-preview.h"
#include "screem-site.h"
#include "screem-page.h"
#include "screem-markup.h"
#include "screem-dtd.h"

#include "fileops.h"

#include "support.h"

static void charset_changed( GtkWidget *entry, GtkWidget *combo );
static void screem_page_druid_create_page( ScreemPage *page, GladeXML *xml,
					   ScreemWindow *window );

void screem_page_druid_begin( ScreemApplication *application,
			      ScreemWindow *window )
{
	ScreemSite *site;
	ScreemPage *page;
	GladeXML *xml;
	GtkWidget *druid;
	ScreemSession *session;
	
	session = screem_application_get_session( application );

	site = screem_window_get_current( window );

	xml = glade_xml_new( GLADE_PATH"/screem.glade", 
			"new_page_druid", NULL );

	druid = glade_xml_get_widget( xml, "new_page_druid" );
	page = screem_page_new( G_OBJECT( application ) );

	g_object_set_data( G_OBJECT( druid ), "page", page );
	g_object_set_data( G_OBJECT( druid ), "application", application );
	g_object_set_data( G_OBJECT( druid ), "window", window );
	g_object_set_data( G_OBJECT( druid ), "site", site );

	glade_xml_signal_autoconnect( xml );

	gtk_window_set_transient_for( GTK_WINDOW( druid ),
				      GTK_WINDOW( window ) );

	screem_session_restore_dialog( session, druid );

	gtk_widget_show_all( druid );
}

void screem_page_druid_finish( GnomeDruidPage *druid_page,
		GtkWidget *widget )
{
	ScreemSite *site;
	ScreemPage *page;
	const gchar *pathname;
	ScreemWindow *window;
	GladeXML *xml;
	ScreemApplication *application;
	ScreemSession *session;
	GError *error;
	
	xml = glade_get_widget_tree( GTK_WIDGET( druid_page ) );
	widget = glade_xml_get_widget( xml, "new_page_druid" );

	site = SCREEM_SITE( g_object_get_data( G_OBJECT( widget ), "site" ) );
	page = SCREEM_PAGE( g_object_get_data( G_OBJECT( widget ), "page" ) );
	window = SCREEM_WINDOW( g_object_get_data( G_OBJECT( widget ), 
						   "window" ) );
	application = SCREEM_APPLICATION( g_object_get_data( G_OBJECT( widget ),
							     "application" ) );

	pathname = screem_page_get_pathname( page );
	screem_page_set_changed( page, TRUE );
	error = NULL;
	if( ( ! pathname ) ||
		   ( overwrite( pathname ) && 
		     screem_page_save( page, &error ) ) ) {
		if( ! screem_site_add_page( site, page ) ) {
			/* already got a page in the site with the
			   same pathname, FIXME: what now? */
		}
		
		if( site == screem_window_get_current( window ) ) {
			screem_window_set_document( window, page );
			
		}
	} else {
		/* failed to save the page */
		gchar *primary;
		const gchar *secondary;

		primary = g_strdup_printf( _( "Failed to create: %s" ),
				pathname );
		secondary = "";
		if( error ) {
			secondary = error->message;
		}
		screem_hig_alert( GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE,
				primary, secondary,
				GTK_WIDGET( window ) );
		if( error ) {
			g_error_free( error );
		}
	}
	g_object_unref( page );
	session = screem_application_get_session( application );
	screem_session_store_dialog( session, widget );

	gtk_widget_destroy( widget );
}


void screem_page_druid_cancel( GtkWidget *widget )
{
	ScreemPage *page;
	GladeXML *xml;
	ScreemApplication *app;
	ScreemSession *session;

	xml = glade_get_widget_tree( widget );
	widget = glade_xml_get_widget( xml, "new_page_druid" );

	page = SCREEM_PAGE( g_object_get_data( G_OBJECT( widget ), "page" ) );
	g_object_unref( page );

	app = SCREEM_APPLICATION( g_object_get_data( G_OBJECT( widget ),
				"application" ) );
	session = screem_application_get_session( app );
	screem_session_store_dialog( session, widget );

	gtk_widget_destroy( widget );
}

static gboolean
screem_page_druid_dtd_filter( GtkTreeModel *model, GtkTreeIter *iter,
		GladeXML *xml )
{
	GtkWidget *all;
	gboolean show;
	
	all = glade_xml_get_widget( xml, "show_all_doctypes" );
	show = FALSE;
	if( all ) {
		show = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( all ) );
	}

	if( ! show ) {
		gtk_tree_model_get( model, iter,
				SCREEM_DTD_DB_ACTIVE_COL, &show, -1 );
	}

	return show;
}


void screem_page_druid_setup_prepare( GnomeDruidPage *druid_page,
		GtkWidget *druid )
{
	GladeXML *xml;
	GtkWidget *widget;
	ScreemSite *site;
	const gchar *sitepath;
	ScreemApplication *application;
	ScreemDTDDB *db;
	GtkListStore *list;
	gchar *default_dtd;
	GtkWidget *entry;
	gchar *default_charset;

	GtkTreeModel *filter;
	
	xml = glade_get_widget_tree( GTK_WIDGET( druid_page ) );

	druid = glade_xml_get_widget( xml, "new_page_druid" );

	if( g_object_get_data( G_OBJECT( druid ), "setup_prepared" ) ) {
		return;
	}
	g_object_set_data( G_OBJECT( druid ), "setup_prepared",
			GINT_TO_POINTER( 1 ) );

	site = SCREEM_SITE( g_object_get_data( G_OBJECT( druid ),
				"site" ) );
	sitepath = screem_site_get_pathname( site );
	if( sitepath ) {
		widget = glade_xml_get_widget( xml, "page_path" );
		gtk_file_chooser_set_current_folder_uri( GTK_FILE_CHOOSER( widget ), sitepath );
	}
					   
	widget = glade_xml_get_widget( xml, "doctype" );
		
	application = g_object_get_data( G_OBJECT( druid ), 
			"application" );
	db = screem_application_get_dtd_db( application );

	list = screem_dtd_db_get_store( db );
	filter = gtk_tree_model_filter_new( GTK_TREE_MODEL( list ),
			NULL );
	gtk_tree_model_filter_set_visible_func( GTK_TREE_MODEL_FILTER( filter ), (GtkTreeModelFilterVisibleFunc)screem_page_druid_dtd_filter,
			xml, NULL );
	gtk_combo_box_set_model( GTK_COMBO_BOX( widget ), filter );
	g_object_unref( G_OBJECT( filter ) );

	gtk_combo_box_entry_set_text_column( GTK_COMBO_BOX_ENTRY( widget ),
			SCREEM_DTD_DB_PUBLIC_COL );
	
	default_dtd = screem_dtd_db_lookup_mime_type( db, "*", NULL );
	if( ! default_dtd ) {
		default_dtd = g_strdup( "-//W3C//DTD HTML 4.01 Transitional//EN" );
	}
	widget = GTK_BIN( widget )->child;
	gtk_entry_set_text( GTK_ENTRY( widget ), default_dtd );
		
	g_free( default_dtd );
		
	widget = glade_xml_get_widget( xml, "page_charset" );
	entry = GTK_BIN( widget )->child;
	list = GTK_LIST_STORE( screem_application_get_encodings( application ) );
	gtk_combo_box_set_model( GTK_COMBO_BOX( widget ), 
			GTK_TREE_MODEL( list ) ); 
	gtk_combo_box_entry_set_text_column( GTK_COMBO_BOX_ENTRY( widget ),
				0 );

	default_charset = gconf_client_get_string( application->client,
			"/apps/screem/editor/default_charset", NULL );
	if( ! default_charset ) {
		g_get_charset( (const gchar**)&default_charset );
		default_charset = g_strdup( default_charset );
	}
	g_signal_connect( G_OBJECT( entry ), "changed",
			G_CALLBACK( charset_changed ),
			widget );
	gtk_entry_set_text( GTK_ENTRY( entry ), default_charset );
	g_free( default_charset );
}

static void charset_changed( GtkWidget *entry, GtkWidget *combo )
{
	GtkTreeModel *model;
	GtkTreeIter it;
	const gchar *value;
	gboolean found;
	gchar *colval;
	gchar *disval;
	
	value = gtk_entry_get_text( GTK_ENTRY( entry ) );
	model = gtk_combo_box_get_model( GTK_COMBO_BOX( combo ) );
	found = screem_gtk_list_store_find_string( GTK_LIST_STORE( model ), &it, 1, value );
	if( ! found ) {
		found = screem_gtk_list_store_find_string( GTK_LIST_STORE( model ), &it, 0, value );
	}
	if( found ) {
		gtk_tree_model_get( model, &it,	
				0, &disval,
				1, &colval, -1 );
		if( ! strcmp( value, colval ) || 
		    ! strcmp( value, disval ) ) {
			gtk_combo_box_set_active_iter( GTK_COMBO_BOX( combo ), &it );
		}
		g_free( disval );
		g_free( colval );
	}

}

void screem_page_druid_details_prepare( GnomeDruidPage *druid_page,
		GtkWidget *druid )
{
	GladeXML *xml;
	GtkWidget *widget;

	xml = glade_get_widget_tree( GTK_WIDGET( druid_page ) );
	druid = glade_xml_get_widget( xml, "new_page_druid" );
	
	if( g_object_get_data( G_OBJECT( druid ), "details_prepared" ) ) {
		return;
	}
	g_object_set_data( G_OBJECT( druid ), "details_prepared",
			GINT_TO_POINTER( 1 ) );

	widget = glade_xml_get_widget( xml, "page_author" );
	screem_gtk_add_history( widget );
	widget = GTK_BIN( widget )->child;
	gtk_entry_set_text( GTK_ENTRY( widget ), g_get_real_name() );

	widget = glade_xml_get_widget( xml, "page_title" );
	screem_gtk_add_history( widget );

	widget = glade_xml_get_widget( xml, "page_description" );
	screem_gtk_add_history( widget );

	widget = glade_xml_get_widget( xml, "page_keywords" );
	screem_gtk_add_history( widget );
}

void screem_page_druid_template_prepare( GnomeDruidPage *druid_page,
		GtkWidget *druid )
{
	GtkListStore *store;
	GtkTreeView *view;
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *col;
	GladeXML *xml;
	ScreemSite *site;
	const gchar *template;
	const gchar *sitepath;
	GtkWidget *widget;

	xml = glade_get_widget_tree( GTK_WIDGET( druid_page ) );
	widget = glade_xml_get_widget( xml, "new_page_druid" );

	if( g_object_get_data( G_OBJECT( widget ),
				"template_prepared" ) ) {
		return;
	}
	g_object_set_data( G_OBJECT( widget ), "template_prepared",
			GINT_TO_POINTER( 1 ) );

	site = SCREEM_SITE( g_object_get_data( G_OBJECT( widget ),
				"site" ) );
		
	view = GTK_TREE_VIEW( glade_xml_get_widget( xml, 
				"templatelist" ) );

	store = gtk_list_store_new( 2, G_TYPE_STRING, G_TYPE_STRING,
			NULL );
	gtk_tree_view_set_model( view, GTK_TREE_MODEL( store ) );
	g_object_unref( store );
		
	renderer = gtk_cell_renderer_text_new();
	col = gtk_tree_view_column_new();
	gtk_tree_view_column_set_title( col, "Name" );
	gtk_tree_view_column_pack_start( col, renderer, TRUE );
	gtk_tree_view_append_column( view, col );
	gtk_tree_view_column_set_attributes( col, renderer, 
			"text", 0, NULL );
		
	renderer = gtk_cell_renderer_text_new();
	col = gtk_tree_view_column_new();
	gtk_tree_view_column_set_title( col, "Path" );
	gtk_tree_view_column_pack_start( col, renderer, TRUE );
	gtk_tree_view_append_column( view, col );
	gtk_tree_view_column_set_attributes( col, renderer, 
			"text", 1, NULL );

	widget = glade_xml_get_widget( xml, "custom_template" );
	sitepath = screem_site_get_pathname( site );
	if( sitepath ) {
		gtk_file_chooser_set_current_folder_uri( GTK_FILE_CHOOSER( widget ), sitepath );
	}
		
	gtk_widget_set_sensitive( widget, FALSE );
	template = screem_site_get_template_path( site );
	if( template ) {
		gtk_file_chooser_set_uri( GTK_FILE_CHOOSER( widget ),
				template );
		widget = glade_xml_get_widget( xml, "use_custom" );
		gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ), TRUE );
	}

}

void screem_page_druid_appearance_prepare( GnomeDruidPage *druid_page,
		GtkWidget *druid )
{
	GladeXML *xml;
	GtkWidget *widget;
	gboolean template;
	ScreemSite *site;
	const gchar *sitepath;

	xml = glade_get_widget_tree( GTK_WIDGET( druid_page ) );
	druid = glade_xml_get_widget( xml, "new_page_druid" );

	if( g_object_get_data( G_OBJECT( druid ), 
				"appearance_prepared" ) ) {
		return;
	}
	g_object_set_data( G_OBJECT( druid ), "appearance_prepared",
			GINT_TO_POINTER( 1 ) );

	widget = glade_xml_get_widget( xml, "notemplate" );
	template = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) );
	widget = glade_xml_get_widget( xml, "override" );

	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ),
					      template );
	gtk_widget_set_sensitive( widget, ! template );

	site = SCREEM_SITE( g_object_get_data( G_OBJECT( druid ),
				"site" ) );
	sitepath = screem_site_get_pathname( site );
	if( sitepath ) {
		widget = glade_xml_get_widget( xml, "page_stylesheet" );
		gtk_file_chooser_set_current_folder_uri( GTK_FILE_CHOOSER( widget ), sitepath );
	}

}

void screem_page_druid_preview_prepare( GnomeDruidPage *druid_page,
		GtkWidget *druid )
{
	GladeXML *xml;
	ScreemPreview *preview;
	ScreemPage *page;
	GList *children;
	ScreemWindow *window;
	GtkWidget *widget;

	xml = glade_get_widget_tree( GTK_WIDGET( druid_page ) );

	druid = glade_xml_get_widget( xml, "new_page_druid" );

	window = SCREEM_WINDOW( g_object_get_data( G_OBJECT( druid ),
				"window" ) );

	widget = glade_xml_get_widget( xml, "page_preview" );
	children = gtk_container_get_children( GTK_CONTAINER( widget ) );
	children = children->next;

	if( ! children ) {
		preview = screem_preview_new();
		gtk_box_pack_start( GTK_BOX( widget ),
				    GTK_WIDGET( preview ),
				    TRUE, TRUE, 0 );
	} else {
		preview = SCREEM_PREVIEW( children->data );
	}
	gtk_widget_show_all( GTK_WIDGET( preview ) );

	/* now take the page template, colours, stylesheet etc
	   and construct a page */
	widget = glade_xml_get_widget( xml, "new_page_druid" );
	page = SCREEM_PAGE( g_object_get_data( G_OBJECT( widget ),
				"page" ) );

	g_return_if_fail( page != NULL );
	screem_page_druid_create_page( page, xml, window );
	screem_page_set_mime_type( page, "text/html" );

	g_object_set( G_OBJECT( preview ), "page", page, 
			"online", FALSE, NULL );
	screem_view_display( SCREEM_VIEW( preview ) );
}

static void screem_page_druid_create_page( ScreemPage *page, 
		GladeXML *xml, ScreemWindow *window )
{
	GtkWidget *widget;

	gchar *pathname;
	const gchar *ext;
	gchar *uri;
	
	gchar *stylesheet;
	const gchar *doctype;
	
	const gchar *title;
	const gchar *author;
	gchar *charset;
	const gchar *keywords;
	const gchar *description;

	gchar *template;

	gboolean override;
	gboolean frameset;

	GString *newtext;
	gchar *text;
	guint pos;
	guint start;
	guint end;
	gchar *name;
	gchar *tag;

	gboolean xhtml;
	GCompareFunc compare;

	gboolean setauthor;
	gboolean setgenerator;
	gboolean setkeywords;
	gboolean setdescription;
	gboolean setcontent;
	gboolean setscript;
	gboolean setstyle;

	ScreemApplication *app;
	ScreemDTDDB *db;
	ScreemDTD *dtd;
	ScreemSite *site;
	const gchar *sitepath;
	gchar *spath;

	GConfClient *client;
	gfloat tabwidth;
	gchar *spaces;
	gchar *tmp;
	
	app = window->application;
	db = screem_application_get_dtd_db( app );

	site = screem_window_get_current( window );
	
	widget = glade_xml_get_widget( xml, "page_path" );
	tmp = gtk_file_chooser_get_uri( GTK_FILE_CHOOSER( widget ) );
	widget = glade_xml_get_widget( xml, "page_name" );
	/* safe cast as we null or use it to make a copy */
	pathname = (gchar*)gtk_entry_get_text( GTK_ENTRY( widget ) );
	if( pathname && *pathname != '\0' ) {
		pathname = g_build_filename( tmp, pathname, NULL );
	} else {
		pathname = NULL;
	}
	
	widget = glade_xml_get_widget( xml, "page_stylesheet" );
	stylesheet = gtk_file_chooser_get_uri( GTK_FILE_CHOOSER( widget ) ); 
	if( stylesheet ) {
		if( ! screem_site_get_fake_flag( site ) ) {
			spath = relative_path( stylesheet,
					screem_site_get_pathname( site ) );
			g_free( stylesheet );
			stylesheet = spath;
			spath = NULL;
		} else if( pathname ) {
			tmp = g_path_get_dirname( pathname );
			if( tmp &&
			! strncmp( tmp, stylesheet, strlen( tmp ) ) ) {
				spath = relative_path( stylesheet,
						tmp );
				g_free( stylesheet );
				stylesheet = spath;
				spath = NULL;
			}
			g_free( tmp );
		}
	}
	
	widget = glade_xml_get_widget( xml, "doctype" );
	widget = GTK_BIN( widget )->child;
	doctype = gtk_entry_get_text( GTK_ENTRY( widget ) );

	dtd = screem_dtd_db_get_dtd( db, doctype, NULL );
	if( dtd ) {
		g_object_get( G_OBJECT( dtd ), "compare", &compare, 
				NULL );	
	} else {
		/* default to case sensitive if we failed to
		 * load the dtd */
		compare = (GCompareFunc)strcmp;
	}
	/* bit of a hack */
	xhtml = ( strstr( doctype, " XHTML " ) != NULL );
	if( pathname && ! xhtml ) {
		ext = strrchr( pathname, '.' );
		if( ext ) {
			xhtml = ! strcmp( ".xml", ext );
			xhtml |= ! strcmp( ".xhtml", ext );
		}
	}

	widget = glade_xml_get_widget( xml, "page_title" );
	widget = GTK_BIN( widget )->child;
	title = gtk_entry_get_text( GTK_ENTRY( widget ) );

	widget = glade_xml_get_widget( xml, "page_author" );
	widget = GTK_BIN( widget )->child;
	author = gtk_entry_get_text( GTK_ENTRY( widget ) );

	widget = glade_xml_get_widget( xml, "page_charset" );
	widget = GTK_BIN( widget )->child;
	charset = (gchar*)gtk_entry_get_text( GTK_ENTRY( widget ) );
	if( ! ( name = g_utf8_strchr( charset, -1, '(' ) ) ) {
		charset = g_strdup( charset );	
	} else {
		charset = g_utf8_next_char( name );
		name = g_utf8_strchr( charset, -1, ')' );
		if( ! name ) {
			charset = g_strdup( charset );
		} else {
			charset = g_strndup( charset,
					name - charset );
		}
	}

	widget = glade_xml_get_widget( xml, "page_description" );
	widget = GTK_BIN( widget )->child;
	description = gtk_entry_get_text( GTK_ENTRY( widget ) );

	widget = glade_xml_get_widget( xml, "page_keywords" );
	widget = GTK_BIN( widget )->child;
	keywords = gtk_entry_get_text( GTK_ENTRY( widget ) );

	template = NULL;
	widget = glade_xml_get_widget( xml, "use_custom" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
		widget = glade_xml_get_widget( xml, "custom_template" );

		template = gtk_file_chooser_get_uri( GTK_FILE_CHOOSER( widget ) ); 
	}
	widget = glade_xml_get_widget( xml, "use_standard" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
		widget = glade_xml_get_widget( xml, "druid" );
		template = g_object_get_data( G_OBJECT( widget ),
					      "template" );
		template = g_strdup( template );
	}

	
	widget = glade_xml_get_widget( xml, "override" );
	override = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) );

	widget = glade_xml_get_widget( xml, "frameset" );
	frameset = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) );
	
	screem_application_set_cursor( app, GDK_WATCH );

	/* if we are using a template load it */
	if( template ) {
		screem_page_set_pathname( page, template );
		if( screem_page_load( page, NULL ) ) {
			text = screem_page_get_data( page );
		} else {
			/* FIXME: report error */
	

			/* fallback to no template */
			text = screem_markup_basic_html( db, doctype,
				frameset, xhtml );
			screem_page_set_data( page, text );

			g_free( template );
			template = NULL;
		}
		screem_page_set_pathname( page, NULL );
	} else {
		/* create a basic page */

		text = screem_markup_basic_html( db, doctype,
				frameset, xhtml );
		screem_page_set_data( page, text );
	}

	newtext = g_string_new( NULL );
	pos = start = end = 0;

	setauthor = setgenerator = setdescription = 
		setkeywords = setcontent = setscript = setstyle = FALSE;

	while( ( tag = screem_markup_next_tag( text, pos, &start, &end, &name ) ) ) {
		GSList *attrs;
		GSList *list;
		gboolean processed;

		g_assert( name );
		
		attrs = screem_markup_build_attributes_list( tag, NULL );
		attrs = g_slist_reverse( attrs );

		if( start > pos ) {
			g_string_append_len( newtext, text + pos, start - pos);
		}

		pos = end;
		
		processed = FALSE;

		if( ! compare( "/head", name ) ) {
			/* so we can append unset meta elements, and
			   a link to a stylesheet if specified */
			if( ! setauthor ) {
				g_string_append( newtext,
						 "\t<meta name=\"author\" content=\"" );
				g_string_append( newtext, author );
				if( ! xhtml ) {
					g_string_append( newtext, "\" >\n\t" );
				} else {
					g_string_append( newtext, "\" />\n\t" );
				}
				setauthor = TRUE;
			}
			if( ! setgenerator ) {
				g_string_append( newtext,
						 "\t<meta name=\"generator\" content=\"" );
				g_string_append( newtext, PACKAGE_STRING );
				if( ! xhtml ) {
					g_string_append( newtext, "\" >\n\t" );
				} else {
					g_string_append( newtext, "\" />\n\t" );
				}
				setgenerator = TRUE;
			}
			if( ! setdescription ) {
				g_string_append( newtext,
						 "\t<meta name=\"description\" content=\"" );
				g_string_append( newtext, description );
				if( ! xhtml ) {
					g_string_append( newtext, "\" >\n\t" );
				} else {
					g_string_append( newtext, "\" />\n\t" );
				}
				setdescription = TRUE;

			}
			if( ! setkeywords ) {
				g_string_append( newtext,
						 "\t<meta name=\"keywords\" content=\"" );
				g_string_append( newtext, keywords );
				if( ! xhtml ) {
					g_string_append( newtext, "\" >\n\t" );
				} else {
					g_string_append( newtext, "\" />\n\t" );
				}
				setkeywords = TRUE;
			}
			if( ! setcontent && ! template ) {
				g_string_append( newtext,
						 "\t<meta http-equiv=\"content-type\" content=\"text/html; charset=" );
				g_string_append( newtext, charset );
				if( ! xhtml ) {
					g_string_append( newtext, "\" >\n\t" );
				} else {
					g_string_append( newtext, "\" />\n\t" );
				}
				setcontent = TRUE;
			}
			if( ! setscript ) {
				g_string_append( newtext,
						 "\t<meta http-equiv=\"Content-Script-Type\" content=\"text/javascript\" " );
				if( ! xhtml ) {
					g_string_append( newtext, ">\n\t" );
				} else {
					g_string_append( newtext, "/>\n\t" );
				}
				setscript = TRUE;
			}
			if( ! setstyle ) {
				g_string_append( newtext,
						 "\t<meta http-equiv=\"Content-Style-Type\" content=\"text/css\" " );
				if( ! xhtml ) {
					g_string_append( newtext, ">\n\t" );
				} else {
					g_string_append( newtext, "/>\n\t" );
				}
				setstyle = TRUE;
			}
			if( stylesheet && *stylesheet != '\0' ) {
				g_string_append( newtext,
						 "\t<link rel=\"stylesheet\" type=\"text/css\" href=\"" );
				g_string_append( newtext, stylesheet );
				if( ! xhtml ) {
					g_string_append( newtext, "\" >\n\t" );
				} else {
					g_string_append( newtext, "\" />\n\t" );
				}
			}
		} else if( ! compare( "title", name ) && *title != '\0' ) {
			processed = TRUE;
			g_string_append_len( newtext,
					text + start, end - start );
			g_string_append( newtext, title );
			/* shift to start of closing tag for <title> */
			while( text[ pos ] != '<' && text[ pos ] != '\0' ) {
				pos ++;
			}
		} else if( ! compare( "meta", name ) ) {
			/* we are interested in:
			   name="author"
			   name="generator"
			   name="keywords"
			   name="description"

			   and http-equiv="content-type" partly
			*/
			GSList *nme;
			GSList *content;
			nme = content = NULL;
			for( list = attrs; list; list = list->next ) {
				GSList *attr;

				attr = list;
				list = list->next;
				if( ! compare( "name", attr->data ) &&
				    ( ! g_strcasecmp( "author", list->data ) ||
				      ! g_strcasecmp( "generator", list->data ) ||
				      ! g_strcasecmp( "description", list->data ) ||
				      ! g_strcasecmp( "keywords", list->data ) ) ) {
					nme = list;
				} else if( ! compare( "http-equiv", attr->data ) &&
					   ( ! g_strcasecmp( "content-type",
							     list->data ) ||
					     ! g_strcasecmp( "Content-Style-Type",
							     list->data ) ||
					     ! g_strcasecmp( "Content-Script-Type",
							     list->data ) ) ) {
					nme = list;
				}
				if( ! compare( "content", attr->data ) ) {
					content = list;
				}
			}
			if( nme ) {
				const gchar *type;
				gchar *value;

				value = NULL;
				type = (const gchar*)nme->data;

				processed = TRUE;
				g_string_append_c( newtext, '<' );
				g_string_append( newtext, name );
				if( ! g_strcasecmp( "author", type ) ) {
					value = g_strdup( author );
				} else if( ! g_strcasecmp( "generator", 
							   type ) ) {
					value = g_strdup( PACKAGE_STRING );
				} else if( ! g_strcasecmp( "description",
							type ) ) {
					value = g_strdup( description );
				} else if( ! g_strcasecmp( "keywords",
							   type ) ) {
					value = g_strdup( keywords );
				} else if( content ) {
					value = g_strdup( content->data );
				}
					
				if( content ) {
					content->data = value;
				} else {
					attrs = g_slist_prepend( attrs, value );
					attrs = g_slist_prepend( attrs, 
								 g_strdup( type ) );
				}
					   
				g_string_append_c( newtext, '<' );
				g_string_append( newtext, name );

				for( list = attrs; list; list = list->next ) {
					g_string_append_c( newtext, ' ' );
					if( ! g_strcasecmp( "author", list->data ) ) {
						setauthor = TRUE;
					} else if( ! g_strcasecmp( "generator",
								   list->data ) ) {
						setgenerator = TRUE;
					} else if( ! g_strcasecmp( "description",
								list->data ) ) {
						setdescription = TRUE;
					} else if( ! g_strcasecmp( "keywords", 
								   list->data ) ) {
						setkeywords = TRUE;
					} else if( ! g_strcasecmp( "content-type",
								   list->data ) ) {
						setcontent = TRUE;
					} else if( ! g_strcasecmp( "Content-Style-Type",
								   list->data ) ) {
						setstyle = TRUE;
					} else if( ! g_strcasecmp( "Content-Script-Type",
								   list->data ) ) {
						setscript = TRUE;
					}

					g_string_append( newtext, list->data );
					g_string_append( newtext, "=\"" );
					if( ! list->next->data ) {
						g_string_append( newtext, list->data );
						list = list->next;
					} else {
						list = list->next;
						g_string_append( newtext, list->data );
					}
					g_string_append_c( newtext, '"' );
				}

			} else {
				processed = FALSE;
			}
		} else if( ! compare( "body", name ) && override ) {
			GSList *bgcolor;
			GSList *text;
			GSList *link;
			GSList *vlink;
			GSList *alink;
			GSList *background;
			GdkColor color;
			gchar *value;

			processed = TRUE;

			g_string_append_c( newtext, '<' );
			g_string_append( newtext, name );

			/* find attributes in attrs */
			bgcolor = text = link = vlink = alink = background = NULL;
			for( list = attrs; list; list = list->next ) {
				if( ! compare( "bgcolor", list->data ) ) {
					bgcolor = list->next;
				} else if( ! compare( "text", list->data ) ) {
					text = list->next;
				} else if( ! compare( "link", list->data ) ) {
					link = list->next;
				} else if( ! compare( "vlink", list->data ) ) {
					vlink = list->next;
				} else if( ! compare( "alink", list->data ) ) {
					alink = list->next;
				} else if( ! compare( "background", list->data ) ) {
					background = list->next;
				}
				list = list->next;
			}

			widget = glade_xml_get_widget( xml, "bgcolor" );
			if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
				widget = glade_xml_get_widget( xml, "bgcolorpicker" );
				gtk_color_button_get_color( GTK_COLOR_BUTTON( widget ), &color );
				value = screem_gdk_color_to_string( &color, TRUE );
				if( bgcolor ) {
					g_free( bgcolor->data );
					bgcolor->data = value;
				} else {
					attrs = g_slist_prepend( attrs, value );
					attrs = g_slist_prepend( attrs, 
								 g_strdup( "bgcolor" ) );
				}
			}
			widget = glade_xml_get_widget( xml, "text" );
			if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
				widget = glade_xml_get_widget( xml, "textpicker" );
				gtk_color_button_get_color( GTK_COLOR_BUTTON( widget ), &color );
				value = screem_gdk_color_to_string( &color, TRUE );
				if( text ) {
					g_free( text->data );
					text->data = value;
				} else {
					attrs = g_slist_prepend( attrs, value );
					attrs = g_slist_prepend( attrs, 
								 g_strdup( "text" ) );
				}
			}
			widget = glade_xml_get_widget( xml, "link" );
			if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
				widget = glade_xml_get_widget( xml, "linkpicker" );
				gtk_color_button_get_color( GTK_COLOR_BUTTON( widget ), &color );
				value = screem_gdk_color_to_string( &color, TRUE );
				if( link ) {
					g_free( link->data );
					link->data = value;
				} else {
					attrs = g_slist_prepend( attrs, value );
					attrs = g_slist_prepend( attrs, 
								 g_strdup( "link" ) );
				}
			}
			widget = glade_xml_get_widget( xml, "vlink" );
			if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
				widget = glade_xml_get_widget( xml, "vlinkpicker" );
				gtk_color_button_get_color( GTK_COLOR_BUTTON( widget ), &color );
				value = screem_gdk_color_to_string( &color, TRUE );
				if( vlink ) {
					g_free( vlink->data );
					vlink->data = value;
				} else {
					attrs = g_slist_prepend( attrs, value );
					attrs = g_slist_prepend( attrs, 
								 g_strdup( "vlink" ) );
				}
			}
			widget = glade_xml_get_widget( xml, "alink" );
			if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
				widget = glade_xml_get_widget( xml, "alinkpicker" );
				gtk_color_button_get_color( GTK_COLOR_BUTTON( widget ), &color );
				value = screem_gdk_color_to_string( &color, TRUE );
				if( alink ) {
					g_free( alink->data );
					alink->data = value;
				} else {
					attrs = g_slist_prepend( attrs, value );
					attrs = g_slist_prepend( attrs, 
								 g_strdup( "alink" ) );
				}
			}
			for( list = attrs; list; list = list->next ) {
				g_string_append_c( newtext, ' ' );
				g_string_append( newtext, list->data );
				g_string_append( newtext, "=\"" );
				if( ! list->next->data ) {
					g_string_append( newtext, list->data );
					list = list->next;
				} else {
					list = list->next;
					g_string_append( newtext, list->data );
				}
				g_string_append_c( newtext, '"' );
			}
			g_string_append_c( newtext, '>' );

		} 

		if( ! processed ) {
			g_string_append_len( newtext, text + start, end - start );
		}

		g_slist_foreach( attrs, (GFunc)g_free, NULL );
		g_slist_free( attrs );
		g_free( tag );
	}
	g_string_append( newtext, text + end );
	g_free( text );

	client = gconf_client_get_default();
	tabwidth = gconf_client_get_float( client,
			"/apps/screem/editor/tabwidth",
			NULL );
	if( tabwidth == 0.0 ) {
		tabwidth = 8.0;
	}
	if( gconf_client_get_bool( client,
				"/apps/screem/editor/spaces_instead_of_tabs",
				NULL ) ) {
		spaces = g_new0( gchar, (gint)(tabwidth + 1) );
		memset( spaces, ' ', tabwidth );
		text = find_text( newtext->str, "\t", spaces, NULL );
 		g_free( spaces );
	} else {
		text = g_strdup( newtext->str );
	}
	g_object_unref( client );
	
	g_string_free( newtext, TRUE );
	screem_page_set_data( page, text );
	
	g_free( text );

	spath = NULL;
	uri = NULL;
	if( pathname ) {
		if( screem_site_get_fake_flag( site ) &&
			! g_path_is_absolute( pathname ) ) {
			spath = g_get_current_dir();
		} else {
			sitepath = screem_site_get_pathname( site );
			spath = g_strdup( sitepath );
		}
	}
	if( spath ) {	
		uri = relative_to_full( pathname, spath );	
		g_free( spath );
	} else if( pathname ) {
		uri = g_strdup( pathname );
	}

	if( uri ) {
		screem_page_set_pathname( page, uri );
		g_free( uri );
	}
	g_free( pathname );
	g_free( stylesheet );
	g_free( template );
	g_free( charset );
		
	screem_application_set_cursor( app, GDK_LAST_CURSOR );
}

