/*  Screem:  screem-todo.c
 *
 *  Copyright (C) 2002 David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <gconf/gconf-client.h>

#include <glade/glade.h>

#include <gtk/gtkdialog.h>
#include <gtk/gtkliststore.h>
#include <gtk/gtktreeview.h>
#include <gtk/gtkcellrenderer.h>
#include <gtk/gtkcelleditable.h>
#include <gtk/gtkcellrenderertext.h>
#include <gtk/gtktreeselection.h>


#include "screem-todo.h"
#include "screem-site.h"

static void screem_todo_class_init( ScreemTodoClass *klass );
static void screem_todo_init( ScreemTodo *todo );
static void screem_todo_finalize( GObject *object );

void screem_todo_response( GtkDialog *dialog, gint response_id );

static void screem_todo_edited( GtkCellRendererText *cell,
				const gchar *path,
				const gchar *new_text,
				ScreemTodo *todo );

struct ScreemTodoPrivate {
	GladeXML *xml;
	GList *tasks;
	GtkWidget *dialog;

	GtkListStore *store;
	ScreemSite *site;
};


ScreemTodo* screem_todo_new( ScreemSite *site )
{
	ScreemTodo *todo;
	GType type;
	
	type = screem_todo_get_type();

	todo = SCREEM_TODO( g_object_new( type, NULL ) );

	todo->private->site = site;

	return todo;
}

void screem_todo_clear( ScreemTodo *todo )
{
	if( todo->private->tasks ) {
		g_list_foreach( todo->private->tasks,
				(GFunc)screem_todo_item_free,
				NULL );
		g_list_free( todo->private->tasks );
		todo->private->tasks = NULL;
	}
}

GList* screem_todo_get_tasks( ScreemTodo *todo )
{
	return todo->private->tasks;
}


void screem_todo_add_new_item( GtkWidget *widget )
{
	GladeXML *xml;
	ScreemTodo *todo;
	
	xml = glade_get_widget_tree( widget );
	widget = glade_xml_get_widget( xml, "todo_dialog" );
	todo = SCREEM_TODO( g_object_get_data( G_OBJECT( widget ), "todo" ) );

     	screem_todo_add_todo_item_with_values( todo,
					       "Name here",
					       "Not Assigned",
					       "Low",
					       "Nothing",
					       "Description here",
					       FALSE );
}

void screem_todo_remove_item( GtkWidget *widget )
{
	GladeXML *xml;
	ScreemTodo *todo;
	GtkTreeSelection *selection;
	GtkTreeModel *model;

	xml = glade_get_widget_tree( widget );
	widget = glade_xml_get_widget( xml, "todo_dialog" );
	todo = SCREEM_TODO( g_object_get_data( G_OBJECT( widget ), "todo" ) );
	widget = glade_xml_get_widget( xml, "tasklist" );

	model = gtk_tree_view_get_model( GTK_TREE_VIEW( widget ) );
	selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( widget ) );
	if( selection ) {
		GtkTreeIter it;
		ScreemTodoItem *item;
		GValue value = {0};
		
		gtk_tree_selection_get_selected( selection, &model, &it );

		gtk_tree_model_get_value( model, &it, 5, &value );
		item = g_value_get_pointer( &value );

		todo->private->tasks = g_list_remove( todo->private->tasks,
						      item );

		gtk_list_store_remove( GTK_LIST_STORE( todo->private->store ),
				       &it );
	}
}

void screem_todo_complete_item( GtkWidget *widget )
{
	GladeXML *xml;
	ScreemTodo *todo;
	GtkTreeSelection *selection;
	GtkTreeModel *model;

	xml = glade_get_widget_tree( widget );
	widget = glade_xml_get_widget( xml, "todo_dialog" );
	todo = SCREEM_TODO( g_object_get_data( G_OBJECT( widget ), "todo" ) );
	widget = glade_xml_get_widget( xml, "tasklist" );

	model = gtk_tree_view_get_model( GTK_TREE_VIEW( widget ) );
	selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( widget ) );
	if( selection ) {
		GtkTreeIter it;
		ScreemTodoItem *item;
		GValue value = {0};
		
		gtk_tree_selection_get_selected( selection, &model, &it );

		gtk_tree_model_get_value( model, &it, 5, &value );

		item = g_value_get_pointer( &value );

		gtk_list_store_set( GTK_LIST_STORE( todo->private->store ),
				    &it, 
				    6, FALSE,
				    7, TRUE,
				    -1 );
		item->complete = TRUE;
	}
}

void screem_todo_view( ScreemTodo *todo )
{
	ScreemTodoPrivate *private;
	ScreemSite *site;
	GConfClient *client;
	gchar *gladepath;

	private = todo->private;

	site = private->site;

	if( ! private->dialog ) {
		GtkWidget *tree;
		GtkCellRenderer *renderer;
		GtkTreeViewColumn *col;
		gint i;
		const gchar *columns[] = {
			"Name", "Assigned To", "Priority", "Linked To",
			"Description", NULL
		};

		client = gconf_client_get_default();
		gladepath = gconf_client_get_string( client,
						     "/apps/screem/general/glade_path",
						     NULL );
		
		private->xml = glade_xml_new( gladepath, "todo_dialog", NULL );
		
		g_free( gladepath );
		g_object_unref( client );
		
		private->dialog = glade_xml_get_widget( private->xml,
							"todo_dialog" );

		{
			gchar *title;

			title = g_strdup_printf( "%s Todo List - Screem",
						 screem_site_get_name( site ) );

			gtk_window_set_title( GTK_WINDOW( private->dialog ),
					      title );
			g_free( title );
		}

		g_object_set_data( G_OBJECT( private->dialog ),
				   "todo", todo );

		/* now create tbe tree view columns / rendereres */
		tree = glade_xml_get_widget( private->xml, "tasklist" );

		for( i = 0; columns[ i ]; ++ i ) {
			renderer = gtk_cell_renderer_text_new();
			col = gtk_tree_view_column_new();
			gtk_tree_view_column_set_title( col, columns[ i ] );
			gtk_tree_view_column_pack_start( col, renderer, TRUE );
			gtk_tree_view_append_column( GTK_TREE_VIEW( tree ),
						     col );
			gtk_tree_view_column_set_attributes( col, renderer,
							     "text", i,
							     "editable", 6,
							     "strikethrough",
							     7,
							     NULL );
			g_object_set_data( G_OBJECT( renderer ), "col",
					   GINT_TO_POINTER( i ) );
			g_signal_connect( G_OBJECT( renderer ), "edited",
					  G_CALLBACK( screem_todo_edited ),
					  todo );
		}
		gtk_tree_view_set_model( GTK_TREE_VIEW( tree ),
					 GTK_TREE_MODEL( private->store ) );

		glade_xml_signal_autoconnect( private->xml );
	}
		
	gtk_dialog_run( GTK_DIALOG( private->dialog ) );
}

ScreemTodoItem* screem_todo_item_new()
{
	return g_new0( ScreemTodoItem, 1 );
}
ScreemTodoItem* screem_todo_item_new_with_values( const gchar *name,
						  const  gchar *assigned,
						  const gchar *priority, 
						  const gchar *linkedto,
						  const gchar *description,
						  gboolean complete)
{
	ScreemTodoItem *item;

	item = screem_todo_item_new();

	item->task = g_strdup( name );
	item->assigned = g_strdup( assigned );
	item->priority = g_strdup( priority );
	item->linked_to = g_strdup( linkedto );
	item->description = g_strdup( description );
	item->complete = complete;

	return item;
}

void screem_todo_item_free( ScreemTodoItem *item )
{
	if( item->task ) {
		g_free( item->task );
	}
	if( item->assigned ) {
		g_free( item->assigned );
	}
	if( item->priority ) {
		g_free( item->priority );
	}
	if( item->linked_to ) {
		g_free( item->linked_to );
	}
	if( item->description ) {
		g_free( item->description );
	}

	g_free( item );
}

void screem_todo_add_todo_item( ScreemTodo *todo, ScreemTodoItem *item )
{
	GtkTreeIter it;

	todo->private->tasks = g_list_prepend( todo->private->tasks,
					       item );

	/* put it in the list store as well */
	gtk_list_store_append( GTK_LIST_STORE( todo->private->store ),
			       &it );
	gtk_list_store_set( GTK_LIST_STORE( todo->private->store ),
			    &it, 
			    0, item->task,
			    1, item->assigned,
			    2, item->priority,
			    3, item->linked_to,
			    4, item->description,
			    5, item,
			    6, (! item->complete),
			    7, item->complete,
			    -1 );
}

void screem_todo_add_todo_item_with_values( ScreemTodo *todo,
					    const gchar *name, 
					    const gchar *assigned,
					    const gchar *priority, 
					    const gchar *linkedto,
					    const gchar *description,
					    gboolean complete )
{
	ScreemTodoItem *item;

	item = screem_todo_item_new_with_values( name, assigned,
						 priority, linkedto,
						 description, complete );
	screem_todo_add_todo_item( todo, item );
}

void screem_todo_response( GtkDialog *dialog, gint response_id )
{
	if( response_id < 0 ) {
		gtk_widget_hide( GTK_WIDGET( dialog ) );
	}
}


static void screem_todo_edited( GtkCellRendererText *cell,
				const gchar *path,
				const gchar *new_text,
				ScreemTodo *todo )
{
	GtkTreeModel *model;
	GtkTreePath *tpath;
	GtkTreeIter it;
	gint col;

	col = GPOINTER_TO_INT( g_object_get_data( G_OBJECT( cell ),
						  "col" ) );
	
	model = GTK_TREE_MODEL( todo->private->store );
	tpath = gtk_tree_path_new_from_string( path );
	if( gtk_tree_model_get_iter( model, &it, tpath ) ) {
		ScreemTodoItem *item;
		GValue value = { 0 };
		
		gtk_tree_model_get_value( model, &it, 5, &value );
		item = g_value_get_pointer( &value );
		g_value_unset( &value );

		switch( col ) {
		case 0:
			/* task */
			g_free( item->task );
			item->task = g_strdup( new_text );
			break;
		case 1:
			/* assigned */
			g_free( item->assigned );
			item->assigned = g_strdup( new_text );
			break;
		case 2:
			/* priority */
			g_free( item->priority );
			item->priority = g_strdup( new_text );
			break;
		case 3:
			/* linked_to */
			g_free( item->linked_to );
			item->linked_to = g_strdup( new_text );
			break;
		case 4:
			/* description */
			g_free( item->description );
			item->description = g_strdup( new_text );
			break;
		}

		gtk_list_store_set( GTK_LIST_STORE( model ), &it,
				    col, new_text, -1 );
	}
	gtk_tree_path_free( tpath );
}

/* G Object stuff */
#define PARENT_TYPE G_TYPE_OBJECT

static gpointer parent_class;

static void screem_todo_class_init( ScreemTodoClass *klass )
{
	GObjectClass *object_class;

	object_class = G_OBJECT_CLASS( klass );

	parent_class = g_type_class_peek_parent( klass );

	object_class->finalize = screem_todo_finalize;
}

static void screem_todo_init( ScreemTodo *todo )
{
	todo->private = g_new0( ScreemTodoPrivate, 1 );


	todo->private->store = gtk_list_store_new( 8, 
						   G_TYPE_STRING, 
						   G_TYPE_STRING,
						   G_TYPE_STRING, 
						   G_TYPE_STRING, 
						   G_TYPE_STRING,
						   G_TYPE_POINTER,
						   G_TYPE_BOOLEAN,
						   G_TYPE_BOOLEAN,
						   NULL );
}

static void screem_todo_finalize( GObject *object )
{
	ScreemTodo *todo;

	todo = SCREEM_TODO( object );

	screem_todo_clear( todo );

	gtk_list_store_clear( todo->private->store );
	g_object_unref( todo->private->store );

	g_free( todo->private );

	g_assert( G_IS_OBJECT_CLASS( parent_class ) );

	G_OBJECT_CLASS( parent_class )->finalize( object );
}

GType screem_todo_get_type()
{
	static GType type = 0;

	if( ! type ) {
		static const GTypeInfo info = {
			sizeof( ScreemTodoClass ),
			NULL, /* base init */
			NULL, /* base finalise */
			(GClassInitFunc)screem_todo_class_init,
			NULL, /* class finalise */
			NULL, /* class data */
			sizeof( ScreemTodo ),
			0, /* n_preallocs */
			(GInstanceInitFunc)screem_todo_init
		};
		
		type = g_type_register_static( PARENT_TYPE,
					       "ScreemTodo",
					       &info, 0 );
	}

	return type;
}
