#include <config.h>

#include <glib/gmarkup.h>
#include <glib/gstrfuncs.h>
#include <string.h>

#include "fileops.h"

#include "screem-site.h"
#include "screem-todo.h"

static void screem_site_parse_start_element( GMarkupParseContext *context,
						const gchar *element,
						const gchar **attributes,
						const gchar **values,
						gpointer data,
						GError **error );
static void screem_site_parse_end_element( GMarkupParseContext *context,
						const gchar *element,
						gpointer data,
						GError **error );
static void screem_site_parse_text( GMarkupParseContext *context,
					const gchar *text,
					gsize len,
					gpointer data,
					GError **error );
static void screem_site_parse_error( GMarkupParseContext *context,
					GError *error,
					gpointer data );


static void screem_site_parse_remote_tag( ScreemSite *site,
					  const gchar **attributes,
					  const gchar **values );
static void screem_site_parse_preview_tag( ScreemSite *site,
					  const gchar **attributes,
					  const gchar **values );
static void screem_site_parse_options_tag( ScreemSite *site,
					  const gchar **attributes,
					  const gchar **values );
static void screem_site_parse_ctags_tag( ScreemSite *site,
					  const gchar **attributes,
					  const gchar **values );
static void screem_site_parse_cvs_tag( ScreemSite *site,
					  const gchar **attributes,
					  const gchar **values );
static void screem_site_parse_task_tag( ScreemSite *site,
					  const gchar **attributes,
					  const gchar **values );



static GMarkupParser parser_cbs = {
	screem_site_parse_start_element,
	screem_site_parse_end_element,
	screem_site_parse_text,
	NULL,
	screem_site_parse_error
};

static const gchar *project_tags[] = {
	"",
	"SCREEM_PROJECT",
	"screem:title",
	"screem:remote",
	"screem:preview",
	"screem:options",
	"screem:http",
	"screem:ctags",
	"screem:cvs",
	"screem:template",
	"screem:tasks",
	"screem:task",
	"screem:excludes",
	"screem:exclude",
	"screem:ignores",
	"screem:ignore",
	"screem:asciis",
	"screem:ascii",
	"screem:open_files",
	"screem:file"
};

typedef enum {
	PROJECT_STATE_NONE = 0,
	PROJECT_STATE_ROOT,
	PROJECT_STATE_TITLE,
	PROJECT_STATE_REMOTE,
	PROJECT_STATE_PREVIEW,
	PROJECT_STATE_OPTIONS,
	PROJECT_STATE_HTTP,
	PROJECT_STATE_CTAGS,
	PROJECT_STATE_CVS,
	PROJECT_STATE_TEMPLATE,
	PROJECT_STATE_TASKS,
	PROJECT_STATE_TASK,
	PROJECT_STATE_EXCLUDES,
	PROJECT_STATE_EXCLUDE,
	PROJECT_STATE_IGNORES,
	PROJECT_STATE_IGNORE,
	PROJECT_STATE_ASCIIS,
	PROJECT_STATE_ASCII,
	PROJECT_STATE_FILES,
	PROJECT_STATE_FILE
} ScreemSiteProjectState;

typedef struct {
	ScreemSite *site;
	GSList *states;
} ScreemSiteProjectParser;


gboolean screem_site_parse_project_file( ScreemSite *site,
					 const gchar *filename )
{
	GMarkupParseContext *ctx;
	ScreemSiteProjectParser *data;
	gboolean ret;

	g_return_val_if_fail( SCREEM_IS_SITE( site ), FALSE );
	g_return_val_if_fail( filename != NULL, FALSE );

	ret = FALSE;

	data = g_new0( ScreemSiteProjectParser, 1 );
	data->site = site;
	data->states = g_slist_prepend( data->states,
					GINT_TO_POINTER( PROJECT_STATE_NONE ) );
	ctx = g_markup_parse_context_new( &parser_cbs, 0, 
					  data, NULL );
	if( ctx ) {
		GString *file;

		/* the lock will be held as we should only
		 * be called from screem_site_load(), which
		 * should only be called from
		 * screem_site_open_with_filename(), which
		 * should have the lock claimed */
		gdk_threads_leave();
		file = load_file( filename );
gdk_threads_enter();

		if( file ) {
			ret = g_markup_parse_context_parse( ctx, file->str,
								file->len,
								NULL );
			g_string_free( file, TRUE );
		}
		g_markup_parse_context_free( ctx );
	}
	g_slist_free( data->states );
	g_free( data );

	return ret;
}

static void screem_site_parse_start_element( GMarkupParseContext *context,
						const gchar *element,
						const gchar **attributes,
						const gchar **values,
						gpointer data,
						GError **error )
{
	ScreemSiteProjectParser *parser;
	ScreemSiteProjectState state;
	ScreemSite *site;

	parser = (ScreemSiteProjectParser*)data;
	state = GPOINTER_TO_INT( parser->states->data );
	site = parser->site;
	
	switch( state ) {
		case PROJECT_STATE_NONE:
			if( ! strcmp( "SCREEM_PROJECT", element ) ) {
				/* got a screem project */
				state = PROJECT_STATE_ROOT;
			} else {
				/* invalid project file */
			}
			break;
		case PROJECT_STATE_ROOT:
			/* allowed tags:  title, remote, preview, options,
					  http, cvs, template, tasks,
					  excludes, ignores, asciis, open_files
			*/
			if( ! strcmp( "screem:title", element ) ) {
				state = PROJECT_STATE_TITLE;
			} else if( ! strcmp( "screem:remote", element ) ) {
				state = PROJECT_STATE_REMOTE;
				screem_site_parse_remote_tag( site, 
							 	attributes,
								values );
			} else if( ! strcmp( "screem:preview", element ) ) {
				state = PROJECT_STATE_PREVIEW;
				screem_site_parse_preview_tag( site,
								attributes,
								values );
			} else if( ! strcmp( "screem:options", element ) ) {
				state = PROJECT_STATE_OPTIONS;
				screem_site_parse_options_tag( site,
								attributes,
								values );
			} else if( ! strcmp( "screem:http", element ) ) {
				state = PROJECT_STATE_HTTP;
			} else if( ! strcmp( "screem:ctags", element ) ) {
				state = PROJECT_STATE_CTAGS;
				screem_site_parse_ctags_tag( site,
							attributes,
							values );
			} else if( ! strcmp( "screem:cvs", element ) ) {
				state = PROJECT_STATE_CVS;
				screem_site_parse_cvs_tag( site,
							attributes,
							values );
			} else if( ! strcmp( "screem:template", element ) ) {
				state = PROJECT_STATE_TEMPLATE;
			} else if( ! strcmp( "screem:tasks", element ) ) {
				state = PROJECT_STATE_TASKS;
			} else if( ! strcmp( "screem:excludes", element ) ) {
				state = PROJECT_STATE_EXCLUDES;
			} else if( ! strcmp( "screem:ignores", element ) ) {
				state = PROJECT_STATE_IGNORES;
			} else if( ! strcmp( "screem:asciis", element ) ) {
				state = PROJECT_STATE_ASCIIS;
			} else if( ! strcmp( "screem:open_files", element ) ) {
				state = PROJECT_STATE_FILES;
			} else {
				state = PROJECT_STATE_NONE;
			}
			break;
		case PROJECT_STATE_TASKS:
			if( ! strcmp( "screem:task", element ) ) {
				state = PROJECT_STATE_TASK;
				screem_site_parse_task_tag( site,
								attributes,
								values );
			} else {
				state = PROJECT_STATE_NONE;
			}
			break;
		case PROJECT_STATE_EXCLUDES:
			if( ! strcmp( "screem:exclude", element ) ) {
				state = PROJECT_STATE_EXCLUDE;
			} else {
				state = PROJECT_STATE_NONE;
			}
			break;
		case PROJECT_STATE_IGNORES:
			if( ! strcmp( "screem:ignore", element ) ) {
				state = PROJECT_STATE_IGNORE;
			} else {
				state = PROJECT_STATE_NONE;
			}
			break;
		case PROJECT_STATE_ASCIIS:
			if( ! strcmp( "screem:ascii", element ) ) {
				state = PROJECT_STATE_ASCII;
			} else {
				state = PROJECT_STATE_NONE;
			}
			break;
		case PROJECT_STATE_FILES:
			if( ! strcmp( "screem:file", element ) ) {
				state = PROJECT_STATE_FILE;
			} else {
				state = PROJECT_STATE_NONE;
			}
			break;
		case PROJECT_STATE_TITLE:
		case PROJECT_STATE_REMOTE:
		case PROJECT_STATE_PREVIEW:
		case PROJECT_STATE_OPTIONS:
		case PROJECT_STATE_HTTP:
		case PROJECT_STATE_CTAGS:
		case PROJECT_STATE_CVS:
		case PROJECT_STATE_TEMPLATE:
		case PROJECT_STATE_TASK:
		case PROJECT_STATE_EXCLUDE:
		case PROJECT_STATE_IGNORE:
		case PROJECT_STATE_ASCII:
			/* states should not occur */
			state = PROJECT_STATE_NONE;
			break;
		default:
			/* unknown state, shouldn't happen */
			state = PROJECT_STATE_NONE;
			break;	
	}

	if( state != PROJECT_STATE_NONE ) {
		parser->states = g_slist_prepend( parser->states,
						  GINT_TO_POINTER( state ) );
	}
}

static void screem_site_parse_end_element( GMarkupParseContext *context,
						const gchar *element,
						gpointer data,
						GError **error )
{
	ScreemSiteProjectParser *parser;
	ScreemSiteProjectState state;
	GSList *top;

	parser = (ScreemSiteProjectParser*)data;
	state = GPOINTER_TO_INT( parser->states->data );
	top = parser->states;

	switch( state ) {
		default:
			if( strcmp( project_tags[ state ], element ) ) {
				state = PROJECT_STATE_NONE;
			}
			break;
		case PROJECT_STATE_NONE:
			/* shouldn't happen */
			state = PROJECT_STATE_NONE;
			break;
	}

	if( state != PROJECT_STATE_NONE ) {
		parser->states = top->next;
		top->next = NULL;
		g_slist_free( top );
	}
}

static void screem_site_parse_text( GMarkupParseContext *context,
					const gchar *text,
					gsize len,
					gpointer data,
					GError **error )
{
	ScreemSiteProjectParser *parser;
	ScreemSiteProjectState state;
	ScreemSite *site;

	parser = (ScreemSiteProjectParser*)data;
	state = GPOINTER_TO_INT( parser->states->data );
	site = parser->site;

	switch( state ) {
		case PROJECT_STATE_TITLE:
			screem_site_set_name( site, text );
			break;
		case PROJECT_STATE_REMOTE:
			screem_site_set_remote_url( site, text );
		case PROJECT_STATE_HTTP:
			screem_site_set_http_url( site, text );
			break;
		case PROJECT_STATE_CVS:
			screem_site_set_cvs_root( site, text );
			break;
		case PROJECT_STATE_TEMPLATE:
			screem_site_set_template_path( site, text );
			break;
		case PROJECT_STATE_EXCLUDE:
			screem_site_add_exclude( site, text );
			break;
		case PROJECT_STATE_IGNORE:
			screem_site_add_ignore( site, text );
			break;
		case PROJECT_STATE_ASCII:
			screem_site_add_ascii( site, text );
			break;
		case PROJECT_STATE_FILE:
			screem_site_add_auto_open( site, text );
			break;
		default:
			/* shouldn't happen */
			break;
	}
}

static void screem_site_parse_error( GMarkupParseContext *context,
					GError *error,
					gpointer data )
{
    
}


static void screem_site_parse_remote_tag( ScreemSite *site,
					  const gchar **attributes,
					  const gchar **values )
{
	gint i;
	const gchar *attr;
	const gchar *val;

	for( i = 0; attributes[ i ]; ++ i ) {
		attr = attributes[ i ];
		val = values[ i ];
		if( ! strcmp( "method", attr ) ) {
			if( ! strcmp( "ftp", val ) ) {
				screem_site_set_remote_method( site, FTP );
			} else if( ! strcmp( "webdav", val) ) {
				screem_site_set_remote_method( site, WEBDAV );
			} else if( ! strcmp( "rsh", val ) ) {
				screem_site_set_remote_method( site, RSH );
			} else if( ! strcmp( "ssh", val ) ) {
				screem_site_set_remote_method( site, SSH );
			} else {
				screem_site_set_remote_method( site, LOCAL );
			}
		} else if( ! strcmp( "path", attr ) ) {
			screem_site_set_remote_path( site, val );	
		} else if( ! strcmp( "username", attr ) ) {
			screem_site_set_remote_user( site, val );
		} else if( ! strcmp( "password", attr ) ) {
			screem_site_set_remote_pass( site, val );
		}
	}
}

static void screem_site_parse_preview_tag( ScreemSite *site,
					  const gchar **attributes,
					  const gchar **values )
{
	gint i;
	const gchar *attr;
	const gchar *val;
	gboolean state;
	
	for( i = 0; attributes[ i ]; ++ i ) {
		attr = attributes[ i ];
		val = values[ i ];

		state = ( ! g_strcasecmp( "TRUE", val ) );
		
		if( ! strcmp( "static_publish", attr ) ) {
			screem_site_set_preview_static( site, state );
		} else if( ! strcmp( "dynamic_publish", attr ) ) {
			screem_site_set_preview_dynamic( site, state );
		}
	}		
}

static void screem_site_parse_options_tag( ScreemSite *site,
					  const gchar **attributes,
					  const gchar **values )
{
	gint i;
	const gchar *attr;
	const gchar *val;
	gboolean state;
	
	for( i = 0; attributes[ i ]; ++ i ) {
		attr = attributes[ i ];
		val = values[ i ];

		state = ( ! g_strcasecmp( "TRUE", val ) );
		
		if( ! strcmp( "passive_ftp", attr ) ) {
			screem_site_set_passive_ftp( site, state );
		} else if( ! strcmp( "no_delete", attr ) ) {
			screem_site_set_no_delete( site, state );
		} else if( ! strcmp( "check_moved", attr ) ) {
			screem_site_set_check_moved( site, state );
		} else if( ! strcmp( "no_overwrite", attr ) ) {
			screem_site_set_no_overwrite( site, state );
		} else if( ! strcmp( "permissions", attr ) ) {
			if( ! strcmp( permission_strings[ PERMS_ALL ], val ) ) {
				screem_site_set_permissions( site, PERMS_ALL );
			} else if( ! strcmp( permission_strings[ PERMS_IGNORE ],
					    val ) ) {
				screem_site_set_permissions(site,PERMS_IGNORE);
			} else {
				screem_site_set_permissions( site,PERMS_EXEC );
			}
		} else if( ! strcmp( "symlinks", attr ) ) {
			if( ! strcmp( symlink_strings[ SYM_FOLLOW ], val ) ) {
				screem_site_set_symlinks( site, SYM_FOLLOW );
			} else if( ! strcmp( symlink_strings[ SYM_MAINTAIN ],
					     val) ) {
				screem_site_set_symlinks( site, SYM_MAINTAIN );
			} else {
				screem_site_set_symlinks( site, SYM_IGNORE );
			}
		}
	}		
}

static void screem_site_parse_ctags_tag( ScreemSite *site,
					const gchar **attributes,
					const gchar **values )
{
	gint i;
	const gchar *attr;
	const gchar *val;
	gboolean state;
	
	for( i = 0; attributes[ i ]; ++ i ) {
		attr = attributes[ i ];
		val = values[ i ];

		state = ( ! g_strcasecmp( "TRUE", val ) );
		
		if( ! strcmp( "use", attr ) ) {
			screem_site_set_use_ctags( site, state );
		}
	}		
}

static void screem_site_parse_cvs_tag( ScreemSite *site,
					const gchar **attributes,
					const gchar **values )
{
	gint i;
	const gchar *attr;
	const gchar *val;
	gboolean state;
	
	for( i = 0; attributes[ i ]; ++ i ) {
		attr = attributes[ i ];
		val = values[ i ];

		state = ( ! g_strcasecmp( "TRUE", val ) );
		
		if( ! strcmp( "auto_update", attr ) ) {
			screem_site_set_auto_update( site, state );
		} else if( ! strcmp( "ask", attr ) ) {
			screem_site_set_auto_update_ask( site, state );
		}
	}		
}

static void screem_site_parse_task_tag( ScreemSite *site,
					const gchar **attributes,
					const gchar **values )
{
	gint i;
	const gchar *attr;
	const gchar *val;
	ScreemTodoItem *todo;
	
	todo = screem_todo_item_new();
	
	for( i = 0; attributes[ i ]; ++ i ) {
		attr = attributes[ i ];
		val = values[ i ];

		if( ! strcmp( "name", attr ) ) {
			todo->task = g_strdup( val );
		} else if( ! strcmp( "assigned", attr ) ) {
			todo->assigned = g_strdup( val );
		} else if( ! strcmp( "priority", attr ) ) {
			todo->priority = g_strdup( val );
		} else if( ! strcmp( "linkedTo", attr ) ) {
			todo->linked_to = g_strdup( val );
		} else if( ! strcmp( "description", attr ) ) {
			todo->description = g_strdup( val );
		} else if( ! strcmp( "completed", attr ) ){
			todo->complete = ( ! g_strcasecmp( "TRUE", val ) );
		}
	}
	screem_todo_add_todo_item( SCREEM_TODO( screem_site_get_todo( site ) ),
				   todo );
}
