/*  Screem:  screem-session.c
 *
 *  The ScreemSession object, inspired by session.c from Epiphany
 *
 *  Copyright (C) 2003 David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include "config.h"

#include "screem-session.h"

#include "screem-page.h"
#include "pageUI.h"
#include "screem-site.h"
#include "screem-site-ui.h"
#include "screem-window.h"


#include "fileops.h"

#include <string.h>
#include <libgnome/gnome-macros.h>
#include <libgnomeui/gnome-client.h>

#include <libxml/tree.h>

static void screem_session_class_init( ScreemSessionClass *klass );
static void screem_session_instance_init( ScreemSession *session );
static void screem_session_finalize( GObject *object );
static void screem_session_set_prop( GObject *object, guint prop_id,
				const GValue *value, GParamSpec *spec );
static void screem_session_get_prop( GObject *object, guint prop_id,
				GValue *value, GParamSpec *spec );

static gchar *get_session_filename( const gchar *filename );
static gint
save_yourself( GnomeClient *client, gint phase, GnomeSaveStyle save_style,
               gint is_shutdown, GnomeInteractStyle interact_style,
               gint is_fast, ScreemSession *session );

enum {
	PROP_0,
	PROP_APP
};

struct ScreemSessionPrivate {
	ScreemApplication *app;
	GnomeClient *client;

	GString *output;
};

GNOME_CLASS_BOILERPLATE( ScreemSession, screem_session,
			  GObject,
			  G_TYPE_OBJECT )

static void screem_session_class_init( ScreemSessionClass *klass )
{
	GObjectClass *obj_class;

	obj_class = G_OBJECT_CLASS( klass );
	obj_class->finalize = screem_session_finalize;
	obj_class->get_property = screem_session_get_prop;
	obj_class->set_property = screem_session_set_prop;


	g_object_class_install_property(obj_class,
					PROP_APP,
					g_param_spec_object("app",
							    "Application",
							    "The Application",
							    G_TYPE_OBJECT,
							    G_PARAM_READWRITE |
							    G_PARAM_CONSTRUCT)
					);
}

static void screem_session_instance_init( ScreemSession *session )
{
	ScreemSessionPrivate *priv;
	GnomeClient *client;
	
	priv = session->priv = g_new0( ScreemSessionPrivate, 1 );

	client = priv->client = gnome_master_client ();
        g_signal_connect( G_OBJECT( client ), "save_yourself",
			  G_CALLBACK( save_yourself ), session );
        g_signal_connect( G_OBJECT( client ), "die",
			  G_CALLBACK( gtk_main_quit ), NULL );

}

static void screem_session_finalize( GObject *object )
{
	ScreemSession *session;
	ScreemSessionPrivate *priv;
	
	g_return_if_fail( object != NULL );
	g_return_if_fail( SCREEM_IS_SESSION( object ) );

	session = SCREEM_SESSION( object );

	priv = session->priv;
	
	g_free( priv );
	
	GNOME_CALL_PARENT( G_OBJECT_CLASS, finalize, (object) );
}

static void screem_session_set_prop( GObject *object, guint prop_id,
				const GValue *value, GParamSpec *spec )
{
	ScreemSession *session;
	ScreemSessionPrivate *priv;
	
	session = SCREEM_SESSION( object );
	priv = session->priv;

	switch( prop_id ) {
		case PROP_APP:
			priv->app = g_value_get_object( value );
			break;
		default:
			break;
	}
}

static void screem_session_get_prop( GObject *object, guint prop_id,
				GValue *value, GParamSpec *spec )
{
	ScreemSession *session;
	ScreemSessionPrivate *priv;
	
	session = SCREEM_SESSION( object );
	priv = session->priv;

	switch( prop_id ) {
		case PROP_APP:
			g_value_set_object( value, priv->app );
			break;
		default:
			break;
	}
}

/* static stuff */
static gchar *get_session_filename( const gchar *filename )
{
	gchar *ret;

	g_return_val_if_fail( filename != NULL, NULL );

	if( ! strcmp( filename, SESSION_LAST ) ) {
		gchar *tmp;

		tmp = screem_get_dot_dir();
		ret = g_build_filename( tmp, "session_last.xml", NULL );
		g_free( tmp );
	} else if( ! strcmp( filename, SESSION_GNOME ) ) {
		gchar *tmp;

		tmp = screem_get_dot_dir();
		ret = g_build_filename( tmp, "session_gnome-XXXXXX", NULL );
		g_free( tmp );
		
		tmp = screem_create_tmp_file( ret, "xml", FALSE );
		g_free( ret );
		ret = tmp;
	} else {
		ret = g_strdup( filename );
	}
	
	return ret;
}

static gint
save_yourself( GnomeClient *client, gint phase, GnomeSaveStyle save_style,
               gint is_shutdown, GnomeInteractStyle interact_style,
               gint is_fast, ScreemSession *session )
{
	gchar *argv[] = { "screem", "--load-session", NULL };
        gchar *discard_argv[] = { "rm", "-r", NULL, NULL };

	argv[ 2 ] = get_session_filename( SESSION_GNOME );
	gnome_client_set_restart_command( client, 3, argv );
	
	discard_argv[ 2 ] = argv[ 2 ];
        gnome_client_set_discard_command( client, 3, discard_argv );
	
	if( ! screem_session_save( session, argv[ 2 ] ) ) {
		g_warning( "Failed to save session\n" );
	}
	
        return TRUE;
}

static void screem_session_save_site( ScreemSite *site, ScreemSession *session )
{
	ScreemApplication *app;
	GList *windows;
	GList *tmp;
	GString *output;
	const gchar *pathname;
	gchar *etmp;
	
	g_return_if_fail( SCREEM_IS_SESSION( session ) );
	g_return_if_fail( SCREEM_IS_SITE( site ) );
	g_return_if_fail( session->priv->output != NULL );
	
	g_object_get( G_OBJECT( session ), "app", &app, NULL );
	
	output = session->priv->output;
	
	pathname = screem_site_get_pathname( site );
	if( ! pathname ) {
		pathname = "";
	}
	
	etmp = g_markup_escape_text( pathname, -1 );
	g_string_append_printf( output, 
				"<site url=\"%s\">", etmp );
	g_free( etmp );
	
	windows = screem_application_get_window_list( app );
	for( tmp = windows; tmp; tmp = tmp->next ) {
		ScreemWindow *window;
		const gchar *name;
		GList *pages;
		GList *ptmp;

		window = SCREEM_WINDOW( tmp->data );
		name = "fixme";
		pages = screem_site_get_documents( site, name );
		etmp = g_markup_escape_text( name, -1 );
		g_string_append_printf( output, "<pages name=\"%s\">",
					etmp );
		g_free( etmp );
		
		for( ptmp = pages; ptmp; ptmp = ptmp->next ) {
			ScreemPage *page;

			page = SCREEM_PAGE( ptmp->data );
			pathname = screem_page_get_pathname( page );
			if( ! pathname ) {
				pathname = "";
			}
			etmp = g_markup_escape_text( pathname, -1 );
			g_string_append_printf( output, "<page url=\"%s\"/>",
						etmp );
			g_free( etmp );
		}
		g_string_append_printf( output, "</pages>" );
	}
	
	g_string_append( output, "</site>" );
	g_object_unref( app );
}

static void screem_session_load_site( ScreemSession *session, 
					ScreemSite *site,
					xmlNodePtr node )
{
	ScreemApplication *app;
	gboolean fake;
	
	g_object_get( G_OBJECT( session ), "app", &app, NULL );
	
	fake = screem_site_get_fake_flag( site );
	
	while( node ) {
		if( ! strcmp( "pages", node->name ) ) {
			/* pages to be opened in window name="" */
			xmlChar *name;

			name = xmlGetProp( node, "name" );
			if( name && *name != '\0' ) {
				xmlNodePtr child;
				GList *docs;
				ScreemPage *page;
				
				child = node->children;
				for( docs = NULL; child; child = child->next ) {
					if( ! strcmp( "page", child->name ) ) {
						xmlChar *url;

						url = xmlGetProp( child, "url" );
						if( url ) {
							page = screem_site_locate_page( site, url );
							if( *url == '\0' && ! page ) {
								page = screem_page_new( G_OBJECT( app ) );
								screem_site_add_page( site, page );
							} else if( fake && ! page ) {
								page = screem_page_new( G_OBJECT( app ) );
								screem_page_set_pathname( page, url );
								screem_site_add_page( site, page );
							}
							xmlFree( url );
						} else {
							page = NULL;
						}
						if( page ) {
							docs = g_list_prepend( docs, page );
						}
					}
				}
				docs = g_list_reverse( docs );
				screem_site_set_documents( site, name, docs );
			} else if( name ) {
				xmlFree( name );
			}
		}
		node = node->next;
	}
	g_object_unref( app );
}

static void screem_session_save_windows( ScreemWindow *window, 
					 ScreemSession *session )
{
	GString *output;
	const gchar *name;
	gint x;
	gint y;
	gint w;
	gint h;
	
	ScreemApplication *app;
	ScreemSite *site;
	const gchar *sname;
	gchar *tmp;
	
	g_return_if_fail( SCREEM_IS_SESSION( session ) );
	g_return_if_fail( SCREEM_IS_WINDOW( window ) );
	g_return_if_fail( session->priv->output != NULL );
	
	output = session->priv->output;
	
	g_object_get( G_OBJECT( session ), "app", &app, NULL );

	name = "fixme";
	
	gtk_window_get_position( GTK_WINDOW( window ), &x, &y );
	gtk_window_get_size( GTK_WINDOW( window ), &w, &h );
	
	g_string_append_printf( output, 
				"<window name=\"%s\" x=\"%i\" y=\"%i\" w=\"%i\" h=\"%i\">",
				name, x, y, w, h );
	
	site = screem_window_get_current( window );
	sname = screem_site_get_name( site );

	if( site == screem_application_get_default_site( app ) ) {
		sname = "";
	}
	tmp = g_markup_escape_text( sname, -1 );
	g_string_append_printf( output, "<site name=\"%s\" />",
				tmp );
	g_free( tmp );
	
	g_string_append( output, "</window>" );

	g_object_unref( app );
}

static void screem_session_load_window( ScreemSession *session,
					ScreemWindow *window,
					xmlNodePtr node )
{
	ScreemApplication *app;
	ScreemSite *dsite;
	ScreemSite *site;
	
	g_object_get( G_OBJECT( session ), "app", &app, NULL );
	site = NULL;
				
	dsite = screem_application_get_default_site( app );

	while( node ) {
		if( ! strcmp( "site", node->name ) ) {
			xmlChar *tmp;

			tmp = xmlGetProp( node, "name" );
			if( ( ! tmp ) || *tmp == '\0' ) {
				site = dsite;
			} else {
				site = screem_application_get_site_by_name( app, tmp );
			}
			if( tmp ) {
				xmlFree( tmp );
			}
			if( site ) {
				screem_window_set_current( window, site );
				node = node->children;
			} else {
				node = node->next;
			}
		} else {
			node = node->next;
		}
	}
	/* if we failed to load any of the sites, set
	 * the default site as the current one */
	if( ! screem_window_get_current( window ) ) {
		screem_window_set_current( window, dsite );
	}
	g_object_unref( app );
}

static void screem_session_save_hint( ScreemSession *session )
{
	ScreemApplication *app;
	GtkWidget *hint;
	GString *output;
	gint x;
	gint y;
	
	g_return_if_fail( SCREEM_IS_SESSION( session ) );
	g_return_if_fail( session->priv->output != NULL );
	
	output = session->priv->output;
	
	g_object_get( G_OBJECT( session ), "app", &app, NULL );

	hint = app->hint;
	
	gtk_window_get_position( GTK_WINDOW( hint ), &x, &y );

	if( GTK_WIDGET_VISIBLE( hint ) ) {
		g_string_append_printf( output, "<hints x=\"%i\" y=\"%i\"/>",
				x, y );
	}

	g_object_unref( app );
}

static void screem_session_load_hint( ScreemSession *session, xmlNodePtr node )
{
	ScreemApplication *app;
	GtkWidget *hint;
	xmlChar *tmp;
	gint x;
	gint y;
				
	tmp = xmlGetProp( node, "x" );
	x = atoi( tmp );
	xmlFree( tmp );
	tmp = xmlGetProp( node, "y" );
	y = atoi( tmp );
	xmlFree( tmp );

	g_object_get( G_OBJECT( session ), "app", &app, NULL );

	hint = app->hint;
	gtk_window_set_position( GTK_WINDOW( hint ), 
				 GTK_WIN_POS_NONE );
	gtk_window_move( GTK_WINDOW( hint ), x, y );

	g_object_unref( app );
}

/* public stuff */

ScreemSession *screem_session_new( ScreemApplication *app )
{
	ScreemSession *session;

	session = g_object_new( SCREEM_TYPE_SESSION, "app", app, NULL );

	return session;
}

void screem_session_load( ScreemSession *session,
			  const gchar *filename )
{
	ScreemApplication *app;
	xmlDocPtr doc;
	gchar *file;

	g_object_get( G_OBJECT( session ), "app", &app, NULL );
	
	file = get_session_filename( filename );
	doc = NULL;
	if( uri_exists( file, NULL ) ) {
		doc = xmlParseFile( file );
	}
	g_free( file );

	if( doc ) {
		xmlNodePtr node;

		node = xmlDocGetRootElement( doc );
		/* make sure we are dealing with a screem session file */
		if( strcmp( "screemsession", node->name ) ) {
			node = NULL;
		} else {
			node = node->children;
		}
		
		while( node ) {
			if( ! strcmp( "site", node->name ) ) {
				ScreemSite *site;
				xmlChar *tmp;

				tmp = xmlGetProp( node, "url" );
				if( ( ! tmp ) || *tmp == '\0' ) {
					site = screem_application_get_default_site( app );
					g_assert( site );
				} else if( screem_site_open_with_filename( NULL, app, tmp ) ) {
					site = screem_application_get_site( app, tmp );
				} else {
					site = NULL;
				}
				if( tmp ) {
					xmlFree( tmp );
				}
				if( site ) {
					screem_session_load_site( session, site,
								  node->children );
				}
			} else if( ! strcmp( "window", node->name ) ) {
				ScreemWindow *win;
				xmlChar *tmp;
				gint x;
				gint y;
				gint w;
				gint h;
				
				tmp = xmlGetProp( node, "x" );
				x = atoi( tmp );
				xmlFree( tmp );
				tmp = xmlGetProp( node, "y" );
				y = atoi( tmp );
				xmlFree( tmp );
				tmp = xmlGetProp( node, "w" );
				w = atoi( tmp );
				xmlFree( tmp );
				tmp = xmlGetProp( node, "h" );
				h = atoi( tmp );
				xmlFree( tmp );

				win = screem_application_create_window( app );

				screem_session_load_window( session, win, 
							node->children );
				
				gtk_window_set_default_size( GTK_WINDOW( win ),
								w, h );
				gtk_widget_show( GTK_WIDGET( win ) );
				gtk_window_move( GTK_WINDOW( win ), x, y );
			} else if( ! strcmp( "hints", node->name ) ) {
				screem_session_load_hint( session, node );
			}

			node = node->next;
		}
		
		xmlFreeDoc( doc );
	}
	g_object_unref( app );
}

gboolean screem_session_save( ScreemSession *session, const gchar *filename )
{
	ScreemSessionPrivate *priv;
	gboolean ret;
	GList *list;
	gchar *file;	
	
	g_return_val_if_fail( SCREEM_IS_SESSION( session ), FALSE );

	priv = session->priv;

	/* ensure the the current site has a record of which pages are open
	 * in each window */
	list = screem_application_get_window_list( priv->app );
	while( list ) {
		ScreemSite *site;
		ScreemWindow *window;
		GList *docs;
		const gchar *name;
		
		window = SCREEM_WINDOW( list->data );
		name = "fixme";

		site = screem_window_get_current( window );
		g_assert( site );
		docs = screem_window_get_documents( window );
		screem_site_set_documents( site, name, docs );
		
		list = list->next;
	}
	
	ret = TRUE;

	priv->output = g_string_new( "<?xml version=\"1.0\" ?>\n<screemsession>" );

	list = screem_application_get_loaded_sites( priv->app );
	if( list ) {
		g_list_foreach( list,
				(GFunc)screem_session_save_site,
				session );
	} else {
		g_warning( "Default site missing!\n" );
	}
	
	list = screem_application_get_window_list( priv->app );
	if( list ) {
		g_list_foreach( list,
				(GFunc)screem_session_save_windows,
				session );
	} else {
		g_warning( "No windows!\n" );
	}

	screem_session_save_hint( session );
	
	g_string_append( priv->output, "</screemsession>" );
	
	file = get_session_filename( filename );
	ret = save_file( file, priv->output->str, 
			 GNOME_VFS_PERM_USER_READ |
			 GNOME_VFS_PERM_USER_WRITE );
	g_free( file );
	
	g_string_free( priv->output, TRUE );
	priv->output = NULL;
	
	return ret;
}


