/*  Screem:  screem-preferences.c
 *
 *  Copyright (C) 2002 David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <config.h>

#include <string.h>

#include <glade/glade.h>

#include <gconf/gconf-client.h>

#include <libgnome/gnome-i18n.h>

#include <libgnomeui/gnome-font-picker.h>
#include <libgnomeui/gnome-entry.h>
#include <libgnomeui/gnome-file-entry.h>
#include <libgnomeui/gnome-icon-entry.h>
#include <libgnomeui/gnome-color-picker.h>

#include <gtk/gtkentry.h>
#include <gtk/gtktogglebutton.h>
#include <gtk/gtkspinbutton.h>
#include <gtk/gtkdialog.h>
#include <gtk/gtkcombo.h>
#include <gtk/gtkliststore.h>
#include <gtk/gtktreeview.h>
#include <gtk/gtktreeselection.h>
#include <gtk/gtkcellrenderer.h>
#include <gtk/gtkcellrenderertext.h>
#include <gtk/gtkframe.h>
#include <gtk/gtklabel.h>
#include <gtk/gtkpaned.h>
#include <gtk/gtknotebook.h>

#include <gdk/gdkcolor.h>
#include <gdk/gdkkeysyms.h>

#include "screem-application.h"
#include "screem-preferences.h"
#include "screem-helper.h"
#include "screem-tagtree.h"
#include "screem-tagfile.h"

#include "fileops.h"
#include "support.h"

typedef enum {
	SCREEM_PREF_TOGGLE,
	SCREEM_PREF_RADIO,
	SCREEM_PREF_SPIN,
	SCREEM_PREF_COLOUR,
	SCREEM_PREF_STRING
} ScreemPrefNotifyType;

typedef struct {
	const gchar *widget;
	const gchar *key;
	ScreemPrefNotifyType type;
	gint value;
} ScreemPrefStruct;

static ScreemPrefStruct misc_prefs[] = {
	{ "default_filename_entry", "/apps/screem/default_filename",       SCREEM_PREF_STRING,	0 },
	{ "default_charset_combo",  "/apps/screem/editor/default_charset", SCREEM_PREF_STRING,	0 },
	{ "default_dtd_combo",      "/apps/screem/editor/default_dtd",     SCREEM_PREF_STRING,	0 },
	{ NULL, NULL, 0, 0 }
};

static ScreemPrefStruct editor_colours[] = {
	{ "fontpicker",		"/apps/screem/editor/font",		SCREEM_PREF_STRING,	0 },
	{ "editortextpicker", 	"/apps/screem/editor/text", 		SCREEM_PREF_COLOUR,	0 },
	{ "editorbgpicker", 	"/apps/screem/editor/back", 		SCREEM_PREF_COLOUR,	0 },
	{ "themefont",		"/apps/screem/editor/themefont",	SCREEM_PREF_TOGGLE,	0 },
	{ "themecolours",	"/apps/screem/editor/themecolours",	SCREEM_PREF_TOGGLE,	0 },
	{ NULL, NULL, 0, 0 }
};

static ScreemPrefStruct editor_features[] = {
	{ "highlight", 		  "/apps/screem/editor/highlight", 	 SCREEM_PREF_TOGGLE,	0 },
	{ "error_highlight", 	  "/apps/screem/editor/error_highlight", SCREEM_PREF_TOGGLE,	0 },
{ "tagging", 		  "/apps/screem/editor/inline_tagging",  SCREEM_PREF_TOGGLE,	0 },
	{ "indent", 		  "/apps/screem/editor/auto_indent", 	 SCREEM_PREF_TOGGLE,	0 },
	{ "tagclose", 		  "/apps/screem/editor/intelliclose",    SCREEM_PREF_TOGGLE,	0 },
	{ "wordwrap",		  "/apps/screem/editor/wordwrap",	 SCREEM_PREF_TOGGLE,	0 },
	{ "autocomplete",	  "/apps/screem/editor/autocomplete",    SCREEM_PREF_TOGGLE,    0 },
	{ "tabwidth", 	   	  "/apps/screem/editor/tabwidth",	 SCREEM_PREF_SPIN,	0 },
	{ "autoenton",		  "/apps/screem/editor/entityinsertion", SCREEM_PREF_RADIO,	0 },
	{ "autoentoff",		  "/apps/screem/editor/entityinsertion", SCREEM_PREF_RADIO,	1 },
	{ "autoentbychar",	  "/apps/screem/editor/entityinsertion", SCREEM_PREF_RADIO,	2 },
	{ NULL, NULL, 0, 0 }
};

static ScreemPrefStruct browser_prefs[] = {
	{ "browser1_static",  "/apps/screem/Browser1_static",  SCREEM_PREF_TOGGLE,	0 },
	{ "browser2_static",  "/apps/screem/Browser2_static",  SCREEM_PREF_TOGGLE,	0 },
	{ "browser3_static",  "/apps/screem/Browser3_static",  SCREEM_PREF_TOGGLE,	0 },
	{ "browser1_dynamic", "/apps/screem/Browser1_dynamic", SCREEM_PREF_TOGGLE,	0 },
	{ "browser2_dynamic", "/apps/screem/Browser2_dynamic", SCREEM_PREF_TOGGLE,	0 },
	{ "browser3_dynamic", "/apps/screem/Browser3_dynamic", SCREEM_PREF_TOGGLE,	0 },
	{ "browser1_path",    "/apps/screem/Browser1_path",    SCREEM_PREF_STRING,	0 },
	{ "browser2_path",    "/apps/screem/Browser2_path",    SCREEM_PREF_STRING,	0 },
	{ "browser3_path",    "/apps/screem/Browser3_path",    SCREEM_PREF_STRING,	0 },
	{ "browser1_icon",    "/apps/screem/Browser1_icon",    SCREEM_PREF_STRING,	0 },
	{ "browser2_icon",    "/apps/screem/Browser2_icon",    SCREEM_PREF_STRING,	0 },
	{ "browser3_icon",    "/apps/screem/Browser3_icon",    SCREEM_PREF_STRING,	0 },
	{ NULL, NULL, 0, 0 }
};

static const gchar *charsets[] = {
	"ISO-8859-1",
	"ISO-8859-2",
	"ISO-8859-5",
	"ISO-8859-7",
	"UTF-8",
	NULL
};

static void screem_preferences_helper_select( GtkTreeSelection *sel );

static void screem_preferences_macro_select( GtkTreeSelection *sel );


static void 
screem_preferences_edit_dtds( ScreemApplication *application,
				  GladeXML *xml );
static void screem_preferences_dtd_select( GtkTreeSelection *sel,
					   ScreemApplication *app );
static void screem_preferences_dtd_root_toggle( GtkToggleButton *button,
						gpointer data );
static void 
screem_preferences_dtd_system_id_toggle( GtkToggleButton *button,
						gpointer data );


static void
screem_preferences_dtd_set_system_id( GtkWidget *entry, 
					GdkEventFocus *event,
					gpointer data );
static void
screem_preferences_dtd_set_root( GtkWidget *entry, 
				GdkEventFocus *focus,
				gpointer data );


static void
screem_preferences_edit_tag_trees( ScreemApplication *application,
				GladeXML *xml );
static void screem_preferences_tag_tree_sel( GtkTreeSelection *sel,
					ScreemApplication *app );

static void screem_preferences_add_pref( GtkWidget *dialog,
				const gchar *key,
				ScreemPrefNotifyType type,
				gint value,
				GtkWidget *widget );


static void screem_preferences_add_page( GtkWidget *dialog,
					ScreemPrefStruct *pref )
{
	guint i;
	GtkWidget *widget;
	GladeXML *xml;

	xml = glade_get_widget_tree( dialog );
	for( i = 0; pref[ i ].widget; ++ i ) {
		widget = glade_xml_get_widget( xml, pref[ i ].widget );
		screem_preferences_add_pref( dialog,
					pref[ i ].key,
					pref[ i ].type,
					pref[ i ].value,
					widget );
	}
}

void screem_preferences_edit( ScreemApplication *application )
{
	ScreemDTDDB *db;
	GConfClient *client;
	gchar *gladepath;
	GladeXML *xml;
	GtkWidget *widget;
	gint i;
	static GtkWidget *dialog = NULL;
	GList *list;

	if( dialog ) {
		gtk_widget_show( dialog );
		gdk_window_raise( GDK_WINDOW( dialog->window ) );
		return;
	}

	db = screem_application_get_dtd_db( application );

	client = gconf_client_get_default();
	
	/* create dialog */
	gladepath = gconf_client_get_string( client, 
					     "/apps/screem/general/glade_path",
					     NULL );
	if( ! gladepath ) {
		gladepath = g_strconcat( GLADE_PATH, "/screem.glade",
					 NULL );
	}

	xml = glade_xml_new( gladepath, "preferences", NULL );
	dialog = glade_xml_get_widget( xml, "preferences" );
	glade_xml_signal_autoconnect( xml );
		
	widget = glade_xml_get_widget( xml, "default_dtd_combo" );
	screem_dtd_db_fill_combo( db, widget );
	
	widget = glade_xml_get_widget( xml, "default_charset_combo" );
	for( list = NULL, i = 0; charsets[ i ]; ++ i ) {
		list = g_list_prepend( list, (gpointer)charsets[ i ] );
	}
	list = g_list_reverse( list );
	gtk_combo_set_popdown_strings( GTK_COMBO( widget ), list );
	g_list_free( list );
	
	screem_preferences_edit_dtds( application, xml );

	screem_preferences_edit_tag_trees( application, xml );
	
	g_free( gladepath );

	g_signal_connect_swapped( G_OBJECT( dialog ), "delete_event",
			G_CALLBACK( g_nullify_pointer ),
			&dialog );
	g_signal_connect_swapped( G_OBJECT( dialog ), "destroy",
			G_CALLBACK( g_nullify_pointer ),
			&dialog );

	screem_preferences_add_page( dialog, misc_prefs );
	screem_preferences_add_page( dialog, editor_colours );
	screem_preferences_add_page( dialog, editor_features );
	screem_preferences_add_page( dialog, browser_prefs );
	
	gtk_widget_show( dialog );

	g_object_unref( client );
}

static void screem_preferences_toggle_notify( GConfClient *client,
					      guint cnxn_id,
					      GConfEntry *entry,
					      gpointer data )
{
	if( entry->value && entry->value->type == GCONF_VALUE_BOOL ) {
		GtkWidget *toggle;
		gboolean wstate;
		gboolean state;
		
		toggle = GTK_WIDGET( data );

		wstate = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(toggle));
		state = gconf_value_get_bool( entry->value );

		if( wstate != state ) {
			gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(toggle),
						      state );
		}
	}
}
static gboolean screem_preferences_toggle_changed( GtkWidget *toggle )
{
	GConfClient *client;
	gboolean state;
	const gchar *key;

	client = gconf_client_get_default();

	state = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( toggle ) );
	
	key = g_object_get_data( G_OBJECT( toggle ), "key" );
	gconf_client_set_bool( client, key, state, NULL );

	g_object_unref( client );
	
	return TRUE;
}

static void screem_preferences_spin_notify( GConfClient *client,
					    guint cnxn_id,
					    GConfEntry *entry,
					    gpointer data )
{
	if( entry->value && entry->value->type == GCONF_VALUE_INT ) {
		GtkWidget *spin;
		gint value;
		gint nvalue;
		
		spin = GTK_WIDGET( data );
		value = gtk_spin_button_get_value_as_int( GTK_SPIN_BUTTON( spin ) );
		nvalue = gconf_value_get_int( entry->value );

		if( value != nvalue ) {
			gtk_spin_button_set_value( GTK_SPIN_BUTTON( spin ),
						   nvalue );
		}
	}
}

static void screem_preferences_spin_changed( GtkWidget *spin )
{
	GConfClient *client;
	const gchar *key;
	gdouble value;

	client = gconf_client_get_default();
	key = g_object_get_data( G_OBJECT( spin ), "key" );
	
	value = gtk_spin_button_get_value( GTK_SPIN_BUTTON( spin ) );

	gconf_client_set_float( client, key, value, NULL );

	g_object_unref( client );
}

static void screem_preferences_colour_notify( GConfClient *client,
					      guint cnxn_id,
					      GConfEntry *entry,
					      gpointer data )
{
	if( entry->value && entry->value->type == GCONF_VALUE_STRING ) {
		const gchar *value; 
		GdkColor colour;
		GtkWidget *widget;
		GdkColor pcolour;
		gushort junk;
	
		value = gconf_value_get_string( entry->value );
		widget = GTK_WIDGET( data );
		gnome_color_picker_get_i16( GNOME_COLOR_PICKER( widget ),
				    &pcolour.red, &pcolour.green, 
				    &pcolour.blue, &junk );

		if( gdk_color_parse( value, &colour ) && 
		    ! gdk_color_equal( &colour, &pcolour ) ) {
			gnome_color_picker_set_i16( GNOME_COLOR_PICKER(widget),
						    colour.red, colour.green,
						    colour.blue, 0 );
		}
	}
}

static void screem_preferences_radio_notify( GConfClient *client,
					    guint cnxn_id,
					    GConfEntry *entry,
					    gpointer data )

{
	if( entry->value && entry->value->type == GCONF_VALUE_INT ) {
		GtkWidget *toggle;
		const gchar *name;
		gboolean wstate;
		gint gval;
		gint val;
		
		toggle = GTK_WIDGET( data );
		name = gtk_widget_get_name( toggle );

		wstate= gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(toggle));
		gval = gconf_value_get_int( entry->value );
		
		val = GPOINTER_TO_INT( g_object_get_data( G_OBJECT( toggle ),
							  "val" ) );

		if( val != gval && wstate ) {
			gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(toggle),
						     FALSE );
		} else if( val == gval && ! wstate ) {
			gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(toggle),
						     TRUE );
		}
	}
}

static void screem_preferences_string_notify( GConfClient *client,
					      guint cnxn_id,
					      GConfEntry *entry,
					      gpointer data )
{
	if( entry->value && entry->value->type == GCONF_VALUE_STRING ) {
		const gchar *value;
		GtkWidget *widget;
		GtkWidget *gtkentry;

		value = gconf_value_get_string( entry->value );
		widget = GTK_WIDGET( data );
		gtkentry = NULL;
		
		if( GNOME_IS_ENTRY( widget ) ) {
			gtkentry = gnome_entry_gtk_entry( GNOME_ENTRY(widget));
		} else if( GTK_IS_ENTRY( widget ) ) {
			gtkentry = widget;
		} else if( GTK_IS_COMBO( widget ) ) {
			gtkentry = GTK_COMBO( widget )->entry;
		} else if( GNOME_IS_FILE_ENTRY( widget ) ) {
			gtkentry = gnome_file_entry_gtk_entry( GNOME_FILE_ENTRY( widget ) );
		} else if( GNOME_IS_ICON_ENTRY( widget ) ) {
			const gchar *cvalue;
			cvalue = gnome_icon_entry_get_filename( GNOME_ICON_ENTRY( widget ) );
			if( ! cvalue || strcmp( cvalue, value ) )
				gnome_icon_entry_set_filename( GNOME_ICON_ENTRY( widget ),
							      value);
		} else if( GNOME_IS_FONT_PICKER( widget ) ) {
			const gchar *cvalue;
			cvalue = gnome_font_picker_get_font_name( GNOME_FONT_PICKER( widget ) );
			if( ! cvalue || strcmp( cvalue, value ) )
				gnome_font_picker_set_font_name( GNOME_FONT_PICKER( widget ), value );
			
		}
		if( gtkentry ) {
			const gchar *cvalue;
			cvalue = gtk_entry_get_text( GTK_ENTRY( gtkentry ) );
			if( value && strcmp( value, cvalue ) ) {
				gtk_entry_set_text( GTK_ENTRY( gtkentry ),
						    value );
			}
		}
	}
}

static gboolean screem_preferences_icon_changed( GtkWidget *widget )
{
	GConfClient *client;
	const gchar *key;
	gchar *value;

	client = gconf_client_get_default();
	key = g_object_get_data( G_OBJECT( widget ), "key" );
	
	value = gnome_icon_entry_get_filename( GNOME_ICON_ENTRY( widget ) );
	if( ! value )
		value = g_strdup( "" );

	gconf_client_set_string( client, key, value, NULL );

	g_free( value );

	g_object_unref( client );
	
	return TRUE;
}

static gboolean screem_preferences_string_set( GtkWidget *widget )
{
	GtkWidget *entry = NULL;
	const gchar *cvalue = NULL;
	const gchar *key;

	GConfClient *client;

	client = gconf_client_get_default();
	
	key = g_object_get_data( G_OBJECT( widget ), "key" );

	if( GNOME_IS_ENTRY( widget ) ) {
		entry = gnome_entry_gtk_entry( GNOME_ENTRY( widget ) );
		cvalue = gtk_entry_get_text( GTK_ENTRY( entry ) );
	} else if( GTK_IS_COMBO( widget ) ) {
		entry = GTK_COMBO( widget )->entry;
		cvalue = gtk_entry_get_text( GTK_ENTRY( entry ) );
	} else if( GNOME_IS_FILE_ENTRY( widget ) ) {
		entry = gnome_file_entry_gtk_entry( GNOME_FILE_ENTRY( widget ) );
		cvalue = gtk_entry_get_text( GTK_ENTRY( entry ) );	
	} else if( GTK_IS_ENTRY( widget ) ) {
		cvalue = gtk_entry_get_text( GTK_ENTRY( widget ) );
	}
	    
	if( cvalue ) {
		gconf_client_set_string( client, key, 
					 cvalue, NULL );
	}

	g_object_unref( client );
	
	return TRUE;
}


static void screem_preferences_font_set( GtkWidget *fp,
					  const gchar *font )
{
	GConfClient *client;
	const gchar *key;

	client = gconf_client_get_default();
	key = g_object_get_data( G_OBJECT( fp ), "key" );
	gconf_client_set_string( client, key, font, NULL );
	g_object_unref( client );
}

static void screem_preferences_colour_set( GnomeColorPicker *cp,
					   guint r, guint g, guint b,
					   guint a )
{
	GConfClient *client;
	const gchar *key;
	gchar *value;

	client = gconf_client_get_default();
	key = g_object_get_data( G_OBJECT( cp ), "key" );
	value = g_strdup_printf( "#%.4x%.4x%.4x", r, g, b );
	gconf_client_set_string( client, key, value, NULL );
	g_free( value );
	g_object_unref( client );
}

static gboolean screem_preferences_radio_changed( GtkWidget *toggle )
{
	GConfClient *client;
	gboolean state;
	gint val;
	const gchar *key;

	client = gconf_client_get_default();
	val = GPOINTER_TO_INT( g_object_get_data( G_OBJECT( toggle ),
						  "val" ) );
	state = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( toggle ) );
	
	key = g_object_get_data( G_OBJECT( toggle ), "key" );
	
	if( state ) {
		gconf_client_set_int( client, key, val, NULL );
	}
	
	g_object_unref( client );
	
	return TRUE;
}



void screem_preferences_edit_macros( ScreemApplication *application )
{
	GtkListStore *store;
	GList *macros;

	GtkWidget *view;
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *col;
	const gchar *titles[] = { "Key Combo", "Action/Text", NULL };
	gint i;
	GtkTreeSelection *selection;

	GConfClient *client;
	gchar *gladepath;
	GladeXML *xml;
	GtkWidget *widget;

	client = gconf_client_get_default();
	
	gladepath = gconf_client_get_string( client, 
					     "/apps/screem/general/glade_path",
					     NULL );
	if( ! gladepath ) {
		gladepath = g_strconcat( GLADE_PATH, "/screem.glade",
					 NULL );
	}

	xml = glade_xml_new( gladepath, "macros", NULL );
	glade_xml_signal_autoconnect( xml );	

	
	store = gtk_list_store_new( 3, G_TYPE_STRING, G_TYPE_STRING,
				    G_TYPE_POINTER, NULL );
	/* build model */
	for( macros = application->macros; macros; macros = macros->next ) {
		GtkTreeIter it;
		ScreemEditorMacro *macro;

		macro = (ScreemEditorMacro*)macros->data;

		gtk_list_store_append( store, &it );
		gtk_list_store_set( store, &it, 
				    0, macro->key_name,
				    1, macro->text,
				    2, macro, -1 );
	}

	/* create tree view */
	view = glade_xml_get_widget( xml, "macrolist" );
	g_object_set_data( G_OBJECT( view ) , "application", application );

	for( i = 0; titles[ i ]; ++ i ) {
		renderer = gtk_cell_renderer_text_new();
		col = gtk_tree_view_column_new();
		gtk_tree_view_column_set_title( col, titles[ i ] );
		gtk_tree_view_column_pack_start( col, renderer, TRUE );
		gtk_tree_view_append_column( GTK_TREE_VIEW( view ), col );
		gtk_tree_view_column_set_attributes( col, renderer, 
						     "text", i, NULL );
	}
	
	gtk_tree_view_set_model( GTK_TREE_VIEW( view ), 
				 GTK_TREE_MODEL( store ) );
	g_object_unref( store );

	selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( view ) );
	g_signal_connect( G_OBJECT( selection ), "changed",
			  G_CALLBACK( screem_preferences_macro_select ),
			  NULL );

	widget = glade_xml_get_widget( xml, "macros" );

	gtk_widget_show_all( widget );
	g_free( gladepath );
	g_object_unref( G_OBJECT( client ) );
}

/* also used for the update button as we perform a check to see
   if a helper already exists */
void screem_preferences_add_macro( GtkWidget *widget )
{
	GladeXML *xml;
	GtkWidget *view;
	ScreemEditorMacro *macro;
	ScreemApplication *application;
	GtkTreeModel *model;
	GtkTreeIter it;

	const gchar *combo;
	const gchar *action;

	xml = glade_get_widget_tree( widget );

	view = glade_xml_get_widget( xml, "macrolist" );
	model = gtk_tree_view_get_model( GTK_TREE_VIEW( view ) );

	application = SCREEM_APPLICATION( g_object_get_data( G_OBJECT( view ),
							     "application" ) );

	widget = glade_xml_get_widget( xml, "macrocombo" );
	combo = gtk_entry_get_text( GTK_ENTRY( widget ) );

	widget = glade_xml_get_widget( xml, "macroaction" );
	widget = GTK_COMBO( widget )->entry;
	action = gtk_entry_get_text( GTK_ENTRY( widget ) );

	macro = g_new0( ScreemEditorMacro, 1 );
	macro->key_name = g_strdup( combo );
	macro->text = g_strdup( action );

	screem_application_add_macro( application, macro );

	gtk_list_store_append( GTK_LIST_STORE( model ), &it );
	gtk_list_store_set( GTK_LIST_STORE( model ), &it, 
			    0, macro->key_name,
			    1, macro->text,
			    2, macro, -1 );
}

void screem_preferences_remove_macro( GtkWidget *widget )
{
	GladeXML *xml;
	GtkWidget *view;
	ScreemEditorMacro *macro;
	ScreemApplication *application;
	GtkTreeModel *model;
	GtkTreeIter it;
	GtkTreeSelection *selection;

	const gchar *combo;

	xml = glade_get_widget_tree( widget );

	view = glade_xml_get_widget( xml, "macrolist" );
	selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( view ) );

	application = SCREEM_APPLICATION( g_object_get_data( G_OBJECT( view ),
							     "application" ) );

	if( gtk_tree_selection_get_selected( selection, &model, &it ) ) {

		widget = glade_xml_get_widget( xml, "macrocombo" );
		combo = gtk_entry_get_text( GTK_ENTRY( widget ) );
		
		macro = g_new0( ScreemEditorMacro, 1 );
		macro->key_name = (gchar*)combo;
		screem_application_remove_macro( application, macro );
		g_free( macro );

		gtk_list_store_remove( GTK_LIST_STORE( model ), &it );
	}
}

static void screem_preferences_macro_select( GtkTreeSelection *sel )
{
	GtkTreeView *view;
	GtkTreeModel *model;
	GtkTreeIter it;
	GtkWidget *widget;
	GladeXML *xml;

    	view = gtk_tree_selection_get_tree_view( sel );

	xml = glade_get_widget_tree( GTK_WIDGET( view ) );

	if( gtk_tree_selection_get_selected( sel, &model, &it ) ) {
		GValue nvalue = {0};
		GValue pvalue = {0};
		GtkWidget *widget;

		gtk_tree_model_get_value( model, &it, 0, &nvalue );
		gtk_tree_model_get_value( model, &it, 1, &pvalue );
		
		widget = glade_xml_get_widget( xml, "macrocombo" );
		gtk_entry_set_text( GTK_ENTRY( widget ),
				    g_value_get_string( &nvalue ) );

		widget = glade_xml_get_widget( xml, "macroaction" );
		widget = GTK_COMBO( widget )->entry;
		gtk_entry_set_text( GTK_ENTRY( widget ),
				    g_value_get_string( &pvalue ) );

		g_value_unset( &nvalue );
		g_value_unset( &pvalue );

		widget = glade_xml_get_widget( xml, "remove" );
		gtk_widget_set_sensitive( widget, TRUE );
		widget = glade_xml_get_widget( xml, "update" );
		gtk_widget_set_sensitive( widget, TRUE );
	} else {
		widget = glade_xml_get_widget( xml, "macrocombo" );
		gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		widget = glade_xml_get_widget( xml, "macroaction" );
		widget = GTK_COMBO( widget )->entry;
		gtk_entry_set_text( GTK_ENTRY( widget ), "" );

		widget = glade_xml_get_widget( xml, "remove" );
		gtk_widget_set_sensitive( widget, FALSE );
		widget = glade_xml_get_widget( xml, "update" );
		gtk_widget_set_sensitive( widget, FALSE );
	}
}

gboolean screem_preferences_grab_key( GtkWidget *widget, GdkEventKey *event )
{
	gboolean ret;
	gchar *key;

	ret = FALSE;

	if( event->keyval != GDK_Tab &&
	    event->keyval != GDK_Up &&
	    event->keyval != GDK_Right &&
	    event->keyval != GDK_Down &&
	    event->keyval != GDK_Left ) {
		key = convert_keysym_state_to_string (event->keyval,
						      event->state);
		if( key ) {
			gtk_entry_set_text (GTK_ENTRY(widget), key );
			g_free( key );
		} else {
			gtk_entry_set_text (GTK_ENTRY(widget), "" );
		}
		ret = TRUE;
	}

	return ret;
}

void screem_preferences_edit_helpers( ScreemApplication *application  )
{
	GtkListStore *store;
	GList *helpers;
	GtkWidget *view;
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *col;
	GtkTreeSelection *selection;
	const gchar *titles[] = { "Name", "Path", "Icon", NULL };
	gint i;

	GConfClient *client;
	gchar *gladepath;
	GladeXML *xml;
	GtkWidget *widget;

	client = gconf_client_get_default();
	
	gladepath = gconf_client_get_string( client, 
					     "/apps/screem/general/glade_path",
					     NULL );
	if( ! gladepath ) {
		gladepath = g_strconcat( GLADE_PATH, "/screem.glade",
					 NULL );
	}

	xml = glade_xml_new( gladepath, "helpers", NULL );
	glade_xml_signal_autoconnect( xml );	


	store = gtk_list_store_new( 4, 
				    G_TYPE_STRING,
				    G_TYPE_STRING, G_TYPE_STRING,
				    G_TYPE_POINTER,
				    NULL );
	/* build model */
	for(helpers = application->helpers; helpers; helpers = helpers->next) {
		GtkTreeIter it;
		ScreemHelper *helper;
		const gchar *name;
		const gchar *pathname;
		const gchar *icon;

		helper = SCREEM_HELPER( helpers->data );

		g_object_get( G_OBJECT( helper ), 
			      "name", &name, "pathname", &pathname,
			      "iconpath", &icon, NULL );
		
		gtk_list_store_append( store, &it );

		gtk_list_store_set( store, &it,
				    0, name, 
				    1, pathname,
				    2, icon,
				    3, helper,
				    -1 );
	}

	/* create tree view */
	view = glade_xml_get_widget( xml, "helperlist" );
	g_object_set_data( G_OBJECT( view ) , "application", application );

	for( i = 0; titles[ i ]; ++ i ) {
		renderer = gtk_cell_renderer_text_new();
		col = gtk_tree_view_column_new();
		gtk_tree_view_column_set_title( col, titles[ i ] );
		gtk_tree_view_column_pack_start( col, renderer, TRUE );
		gtk_tree_view_append_column( GTK_TREE_VIEW( view ), col );
		gtk_tree_view_column_set_attributes( col, renderer, 
						     "text", i, NULL );
	}
	
	gtk_tree_view_set_model( GTK_TREE_VIEW( view ), 
				 GTK_TREE_MODEL( store ) );
	g_object_unref( store );

	selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( view ) );
	g_signal_connect( G_OBJECT( selection ), "changed",
			  G_CALLBACK( screem_preferences_helper_select ),
			  NULL );

	widget = glade_xml_get_widget( xml, "helpers" );

	gtk_widget_show_all( widget );
	g_free( gladepath );
	g_object_unref( G_OBJECT( client ) );
}

/* also used for the update button as we perform a check to see
   if a helper already exists */
void screem_preferences_add_helper( GtkWidget *widget )
{
	GladeXML *xml;
	GtkWidget *view;
	ScreemHelper *helper;
	ScreemApplication *application;
	GtkTreeModel *model;
	GtkTreeIter it;

	const gchar *name;
	const gchar *pathname;
	gchar *iconpath;

	HelperInputMode imode;
	HelperOutputMode omode;
	
	gboolean found;
	ScreemHelper *old;
	gchar *oname;

	xml = glade_get_widget_tree( widget );

	view = glade_xml_get_widget( xml, "helperlist" );
	model = gtk_tree_view_get_model( GTK_TREE_VIEW( view ) );

	application = SCREEM_APPLICATION( g_object_get_data( G_OBJECT( view ),
							     "application" ) );

	widget = glade_xml_get_widget( xml, "helpername" );
	widget = gnome_entry_gtk_entry( GNOME_ENTRY( widget ) );
	name = gtk_entry_get_text( GTK_ENTRY( widget ) );

	widget = glade_xml_get_widget( xml, "helperpath" );
	widget = gnome_file_entry_gtk_entry( GNOME_FILE_ENTRY( widget ) );
	pathname = gtk_entry_get_text( GTK_ENTRY( widget ) );

	widget = glade_xml_get_widget( xml, "helpericon" );
	iconpath = gnome_icon_entry_get_filename( GNOME_ICON_ENTRY( widget ) );

	imode = SCREEM_HELPER_STDIN_NONE;
	widget = glade_xml_get_widget( xml, "stdin_page" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
		imode = SCREEM_HELPER_STDIN_SELECTION;
	}
	widget = glade_xml_get_widget( xml, "stdin_user" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
		imode = SCREEM_HELPER_STDIN_USER;
	}
	
	omode = SCREEM_HELPER_STDOUT_NONE;
	widget = glade_xml_get_widget( xml, "stdout_page" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
		omode = SCREEM_HELPER_STDOUT_SELECTION;
	}
	widget = glade_xml_get_widget( xml, "stdout_insert" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
		omode = SCREEM_HELPER_STDOUT_INSERT;
	}
	
	helper = screem_helper_new( name, pathname, iconpath,
				    imode, omode );

	screem_application_remove_helper( application, helper );
	screem_application_add_helper( application, helper );

	found = FALSE;
	old = NULL;
	if( gtk_tree_model_get_iter_first( GTK_TREE_MODEL( model ),
					&it ) ) {
		do {
			old = NULL;
			oname = NULL;
			gtk_tree_model_get( GTK_TREE_MODEL( model ),
					&it, 3, &old, -1 );
			if( old ) {
				g_object_get( G_OBJECT( old ),
					"name", &oname,
					NULL );
			}
			found = ( oname && ! strcmp( name, oname ) );
		} while( ! found &&
			gtk_tree_model_iter_next( GTK_TREE_MODEL( model ),
						&it ) );
	}

	if( ! found ) {
		old = NULL;
		gtk_list_store_append( GTK_LIST_STORE( model ), &it );
	}
	gtk_list_store_set( GTK_LIST_STORE( model ), &it,
			    0, name, 
			    1, pathname,
			    2, iconpath,
			    3, helper,
			    -1 );

	g_free( iconpath );

	if( old ) {
		g_object_unref( old );
	}
}

void screem_preferences_remove_helper( GtkWidget *widget )
{
	GladeXML *xml;
	GtkWidget *view;
	ScreemHelper *helper;
	ScreemApplication *application;
	GtkTreeModel *model;
	GtkTreeIter it;
	GtkTreeSelection *selection;

	xml = glade_get_widget_tree( widget );

	view = glade_xml_get_widget( xml, "helperlist" );
	selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( view ) );

	application = SCREEM_APPLICATION( g_object_get_data( G_OBJECT( view ),
							     "application" ) );
	
	if( gtk_tree_selection_get_selected( selection, &model, &it ) ) {
		gtk_tree_model_get( GTK_TREE_MODEL( model ),
				&it, 3, &helper, -1 );
	
		screem_application_remove_helper( application, helper );

		gtk_list_store_remove( GTK_LIST_STORE( model ), &it );

		g_object_unref( G_OBJECT( helper ) );
	}
}

static void screem_preferences_helper_select( GtkTreeSelection *sel )
{
	GtkTreeView *view;
	GtkTreeModel *model;
	GtkTreeIter it;
	GtkWidget *widget;
	GladeXML *xml;
	const gchar *iwidgets[] = {
		"stdin_none", "stdin_page", "stdin_user"
	};
	const gchar *owidgets[] = {
		"stdout_msg", "stdout_page", "stdout_insert"
	};

       	view = gtk_tree_selection_get_tree_view( sel );

	xml = glade_get_widget_tree( GTK_WIDGET( view ) );

	if( gtk_tree_selection_get_selected( sel, &model, &it ) ) {
		GValue nvalue = { 0 };
		GValue pvalue = { 0 };
		GValue ivalue = { 0 };
		GValue hvalue = { 0 };
		GladeXML *xml;
		GtkWidget *widget;
		ScreemHelper *helper;
		gint imode;
		gint omode;
		
		gtk_tree_model_get_value( model, &it, 0, &nvalue );
		gtk_tree_model_get_value( model, &it, 1, &pvalue );
		gtk_tree_model_get_value( model, &it, 2, &ivalue );
		gtk_tree_model_get_value( model, &it, 3, &hvalue );
		
		xml = glade_get_widget_tree( GTK_WIDGET( view ) );
		
		widget = glade_xml_get_widget( xml, "helpername" );
		widget = gnome_entry_gtk_entry( GNOME_ENTRY( widget ) );
		gtk_entry_set_text( GTK_ENTRY( widget ),
				    g_value_get_string( &nvalue ) );

		widget = glade_xml_get_widget( xml, "helperpath" );
		widget = gnome_file_entry_gtk_entry(GNOME_FILE_ENTRY(widget));
		gtk_entry_set_text( GTK_ENTRY( widget ),
				    g_value_get_string( &pvalue ) );

		widget = glade_xml_get_widget( xml, "helpericon" );
		gnome_icon_entry_set_filename( GNOME_ICON_ENTRY( widget ),
					       g_value_get_string( &ivalue ) );

		helper = g_value_get_pointer( &hvalue );
		g_assert( helper );
		g_object_get( G_OBJECT( helper ), 
				"input", &imode, 
				"output", &omode, NULL );
		widget = glade_xml_get_widget( xml, iwidgets[ imode ] );
		gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ),
						TRUE );
		widget = glade_xml_get_widget( xml, owidgets[ omode ] );
		gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ),
						TRUE );

		widget = glade_xml_get_widget( xml, "helperremove" );
		gtk_widget_set_sensitive( widget, TRUE );
		widget = glade_xml_get_widget( xml, "helperupdate" );
		gtk_widget_set_sensitive( widget, TRUE );

		g_value_unset( &nvalue );
		g_value_unset( &pvalue );
		g_value_unset( &ivalue );
		g_value_unset( &hvalue );
	} else {
		widget = glade_xml_get_widget( xml, "helpername" );
		widget = gnome_entry_gtk_entry( GNOME_ENTRY( widget ) );
		gtk_entry_set_text( GTK_ENTRY( widget ), "" );

		widget = glade_xml_get_widget( xml, "helperpath" );
		widget = gnome_file_entry_gtk_entry(GNOME_FILE_ENTRY(widget));
		gtk_entry_set_text( GTK_ENTRY( widget ), "" );

		widget = glade_xml_get_widget( xml, "helpericon" );
		gnome_icon_entry_set_filename( GNOME_ICON_ENTRY( widget ), "");
		widget = glade_xml_get_widget( xml, "helperremove" );
		gtk_widget_set_sensitive( widget, FALSE );
		widget = glade_xml_get_widget( xml, "helperupdate" );
		gtk_widget_set_sensitive( widget, FALSE );
	
	}
}

static void 
screem_preferences_edit_dtds( ScreemApplication *application,
				  GladeXML *xml )
{
	ScreemDTDDB *db;
	GtkCellRenderer *rend;
	GtkTreeViewColumn *col;
	GtkListStore *store;
	GtkTreeSelection *sel;
	GtkWidget *widget;

	db = screem_application_get_dtd_db( application );
	store = screem_dtd_db_get_store( db );

	widget = glade_xml_get_widget( xml, "doctypes" );
	rend = gtk_cell_renderer_text_new();
	col = gtk_tree_view_column_new();
	gtk_tree_view_column_pack_start( col, rend, TRUE );
	gtk_tree_view_column_set_resizable( col, TRUE );
	gtk_tree_view_append_column( GTK_TREE_VIEW( widget ), col );
	gtk_tree_view_column_set_attributes( col, rend,
					     "text", 0, NULL ); 
	gtk_tree_view_column_set_title( col, "PUBLIC" );

	rend = gtk_cell_renderer_text_new();
	col = gtk_tree_view_column_new();
	gtk_tree_view_column_pack_start( col, rend, TRUE );
	gtk_tree_view_column_set_resizable( col, TRUE );
	gtk_tree_view_append_column( GTK_TREE_VIEW( widget ), col );
	gtk_tree_view_column_set_attributes( col, rend,
					     "text", 1, NULL ); 
	gtk_tree_view_column_set_title( col, "SYSTEM" );
	gtk_tree_view_set_model( GTK_TREE_VIEW( widget ), 
			GTK_TREE_MODEL( store ) );

	sel = gtk_tree_view_get_selection( GTK_TREE_VIEW( widget ) );
	g_signal_connect( G_OBJECT( sel ), "changed",
			  G_CALLBACK( screem_preferences_dtd_select ),
			  application );

	widget = glade_xml_get_widget( xml, "public" );
	gtk_label_set_text( GTK_LABEL( widget ), _( "None Selected" ) );
			
	widget = glade_xml_get_widget( xml, "root" );
	gtk_entry_set_text( GTK_ENTRY( widget ), "" );
	g_signal_connect( G_OBJECT( widget ), "focus_out_event",
			  G_CALLBACK( screem_preferences_dtd_set_root ),
			  application );

	widget = glade_xml_get_widget( xml, "system" );
	gtk_entry_set_text( GTK_ENTRY( widget ), "" );
	g_signal_connect( G_OBJECT( widget ), "focus_out_event",
			  G_CALLBACK( screem_preferences_dtd_set_system_id ),
			  application );
	
	widget = glade_xml_get_widget( xml, "rootoverride" );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ), 
					FALSE );
	g_signal_connect( G_OBJECT( widget ), "toggled",
			  G_CALLBACK( screem_preferences_dtd_root_toggle ),
			  application );
	gtk_widget_set_sensitive( widget, FALSE );
	widget = glade_xml_get_widget( xml, "systemoverride" );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ), 
					FALSE );
	g_signal_connect( G_OBJECT( widget ), "toggled",
			  G_CALLBACK( screem_preferences_dtd_system_id_toggle ),
			  application );
	gtk_widget_set_sensitive( widget, FALSE );
}

static void screem_preferences_dtd_select( GtkTreeSelection *sel,
					   ScreemApplication *app )
{
	ScreemDTDDB *db;
	GtkTreeView *view;
	GtkTreeModel *model;
	GtkTreeIter it;
	GtkWidget *widget;
	GladeXML *xml;

	gchar *public;
	gchar *system;
	gchar *root;
	ScreemDTD *dtd;
	gboolean rover;
	gboolean sover;

	g_object_set_data( G_OBJECT( app ), "prefblock",
			   GINT_TO_POINTER( 1 ) );

	db = screem_application_get_dtd_db( app );
       	view = gtk_tree_selection_get_tree_view( sel );

	xml = glade_get_widget_tree( GTK_WIDGET( view ) );

	dtd = NULL;
	public = system = root = NULL;
	if( gtk_tree_selection_get_selected( sel, &model, &it ) ) {
		gtk_tree_model_get( model, &it,
				    0, &public,
				    -1 );

		dtd = screem_dtd_db_get_dtd( db, public, NULL ); 
	}
	
	if( dtd ) {
		root = screem_dtd_db_get_root( db, public, &rover );
		if( ! root ) {
			root = g_strdup( "unknown" );
		}
		system = screem_dtd_db_get_system_id( db, public,
							&sover );

		widget = glade_xml_get_widget( xml, "public" );
		gtk_label_set_text( GTK_LABEL( widget ), public );

		widget = glade_xml_get_widget( xml, "root" );
		gtk_entry_set_text( GTK_ENTRY( widget ), root );

		widget = glade_xml_get_widget( xml, "system" );
		gtk_entry_set_text( GTK_ENTRY( widget ), system );

		widget = glade_xml_get_widget( xml, "rootoverride" );
		gtk_widget_set_sensitive( widget, TRUE );
		gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ), rover );
		widget = glade_xml_get_widget( xml, "systemoverride" );
		gtk_widget_set_sensitive( widget, TRUE );
		gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ), sover );
		g_free( root );
		g_free( system );
	} else {
		widget = glade_xml_get_widget( xml, "public" );
	
		if( public ) {
			gtk_label_set_text( GTK_LABEL( widget ),
					_( "Unable to load DTD" ) );
		} else {
			gtk_label_set_text( GTK_LABEL( widget ),
					_( "None Selected" ) );
		}
			
		widget = glade_xml_get_widget( xml, "root" );
		gtk_entry_set_text( GTK_ENTRY( widget ), "" );

		widget = glade_xml_get_widget( xml, "system" );
		gtk_entry_set_text( GTK_ENTRY( widget ), "" );
	
		widget = glade_xml_get_widget( xml, "rootoverride" );
		gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ), FALSE );
		gtk_widget_set_sensitive( widget, FALSE );
		widget = glade_xml_get_widget( xml, "systemoverride" );
		gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ), FALSE );
		gtk_widget_set_sensitive( widget, FALSE );

	}
	
	g_object_set_data( G_OBJECT( app ), "prefblock",
			   GINT_TO_POINTER( 0 ) );
	
	g_free( public );
}

static void screem_preferences_dtd_root_toggle( GtkToggleButton *button,
						gpointer data )
{
	ScreemApplication *app;
	ScreemDTDDB *db;
	GladeXML *xml;
	GtkWidget *widget;
	const gchar *root;
	const gchar *publicid;
	
	app = SCREEM_APPLICATION( data );

	if( ! g_object_get_data( G_OBJECT( app ), "prefblock" ) ) {
		db = screem_application_get_dtd_db( app );
		xml = glade_get_widget_tree( GTK_WIDGET( button ) );
	
		widget = glade_xml_get_widget( xml, "public" );
		publicid = gtk_label_get_label( GTK_LABEL( widget ) );

		root = NULL;
		if( gtk_toggle_button_get_active( button ) ) {
			widget = glade_xml_get_widget( xml, "root" );
			root = gtk_entry_get_text( GTK_ENTRY(widget) );
		}
		screem_dtd_db_override_root( db, publicid, root );
	}
}

static void 
screem_preferences_dtd_system_id_toggle( GtkToggleButton *button,
						gpointer data )
{
	ScreemApplication *app;
	ScreemDTDDB *db;
	GladeXML *xml;
	GtkWidget *widget;
	const gchar *systemid;
	const gchar *publicid;
	gchar *id;
	GtkTreeSelection *sel;
	GtkTreeModel *model;
	GtkTreeIter it;

	app = SCREEM_APPLICATION( data );

	if( ! g_object_get_data( G_OBJECT( app ), "prefblock" ) ) {
		db = screem_application_get_dtd_db( app );
		xml = glade_get_widget_tree( GTK_WIDGET( button ) );
	
		widget = glade_xml_get_widget( xml, "public" );
		publicid = gtk_label_get_label( GTK_LABEL( widget ) );

		systemid = NULL;
		if( gtk_toggle_button_get_active( button ) ) {
			widget = glade_xml_get_widget( xml,"system" );
			systemid = gtk_entry_get_text( GTK_ENTRY(widget) );
		}
		screem_dtd_db_override_systemid( db, publicid,
						 systemid );

		widget = glade_xml_get_widget( xml, "doctypes" );
		sel = gtk_tree_view_get_selection( GTK_TREE_VIEW( widget ) );
		id = screem_dtd_db_get_system_id( db, publicid, NULL );
		if( gtk_tree_selection_get_selected( sel, &model, &it ) ) {
			gtk_list_store_set( GTK_LIST_STORE( model ), &it, 1, id, -1 );
		}
		g_free( id );
	}
}

static void
screem_preferences_dtd_set_root( GtkWidget *entry, 
				GdkEventFocus *event,
				gpointer data )
{
	ScreemApplication *app;
	ScreemDTDDB *db;
	GladeXML *xml;
	GtkWidget *widget;
	const gchar *root;
	const gchar *publicid;
	
	app = SCREEM_APPLICATION( data );

	if( ! g_object_get_data( G_OBJECT( app ), "prefblock" ) ) {
		db = screem_application_get_dtd_db( app );
		xml = glade_get_widget_tree( entry );
	
		widget = glade_xml_get_widget( xml, "public" );
		publicid = gtk_label_get_label( GTK_LABEL( widget ) );

		widget = glade_xml_get_widget( xml, "rootoverride" );
		if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
			root = gtk_entry_get_text( GTK_ENTRY( entry ) );
			screem_dtd_db_override_root( db, publicid,
							root );
		}
	}
}

static void
screem_preferences_dtd_set_system_id( GtkWidget *entry, 
					GdkEventFocus *event,
					gpointer data )
{
	ScreemApplication *app;
	ScreemDTDDB *db;
	GladeXML *xml;
	GtkWidget *widget;
	const gchar *systemid;
	const gchar *publicid;
	GtkTreeSelection *sel;
	GtkTreeModel *model;
	GtkTreeIter it;
	
	app = SCREEM_APPLICATION( data );

	if( ! g_object_get_data( G_OBJECT( app ), "prefblock" ) ) {
		db = screem_application_get_dtd_db( app );
		xml = glade_get_widget_tree( entry );
	
		widget = glade_xml_get_widget( xml, "public" );
		publicid = gtk_label_get_label( GTK_LABEL( widget ) );

		widget = glade_xml_get_widget( xml, "systemoverride" );
		if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
			systemid = gtk_entry_get_text( GTK_ENTRY(entry) );
			screem_dtd_db_override_systemid( db, publicid,
							 systemid );

			widget = glade_xml_get_widget( xml, 
							"doctypes" );
			sel = gtk_tree_view_get_selection( GTK_TREE_VIEW( widget ) );
			if( gtk_tree_selection_get_selected( sel, &model, &it ) ) {
				gtk_list_store_set( GTK_LIST_STORE( model ), &it, 1, systemid, -1 );
			}
		}
	}
}

static void
screem_preferences_tag_tree_toggle( GtkCellRendererToggle *toggle,
				const gchar *path,
				gpointer data )
{
	GtkTreeModel *model;
	GtkTreeIter it;
	gboolean active;
	ScreemTagFile *tfile;
	
	model = GTK_TREE_MODEL( data );

	active = gtk_cell_renderer_toggle_get_active( toggle );
	
	if( gtk_tree_model_get_iter_from_string( model, &it, path ) ) {

		active = ! active;

		gtk_tree_model_get( model, &it,
				TAG_FILE_OBJ_COL, &tfile, -1 );
	
		g_object_set( G_OBJECT( tfile ), 
				"active", active, NULL );
	}
}

static void
screem_preferences_edit_tag_trees( ScreemApplication *application,
				GladeXML *xml )
{
	GtkListStore *model;
	GtkWidget *widget;
	GtkCellRenderer *rend;
	GtkTreeViewColumn *col;
	GtkTreeSelection *sel;

	widget = glade_xml_get_widget( xml, "tagtreeview" );
	model = screem_tag_tree_get_file_model();

	rend = gtk_cell_renderer_toggle_new();
	g_signal_connect( G_OBJECT( rend ), "toggled",
			  G_CALLBACK( screem_preferences_tag_tree_toggle ),
			  model );
	col = gtk_tree_view_column_new();
	gtk_tree_view_column_pack_start( col, rend, TRUE );
	gtk_tree_view_column_set_resizable( col, FALSE );
	gtk_tree_view_append_column( GTK_TREE_VIEW( widget ), col );
	gtk_tree_view_column_set_attributes( col, rend,
				"active", TAG_FILE_ACTIVE_COL, NULL );
	gtk_tree_view_column_set_title( col, _( "Active" ) );
	
	rend = gtk_cell_renderer_text_new();
	col = gtk_tree_view_column_new();
	gtk_tree_view_column_pack_start( col, rend, TRUE );
	gtk_tree_view_column_set_resizable( col, TRUE );
	gtk_tree_view_append_column( GTK_TREE_VIEW( widget ), col );
	gtk_tree_view_column_set_attributes( col, rend,
					     "text", 
					     TAG_FILE_NAME_COL, NULL ); 
	gtk_tree_view_column_set_title( col, _( "Name" ) );

	rend = gtk_cell_renderer_text_new();
	col = gtk_tree_view_column_new();
	gtk_tree_view_column_pack_start( col, rend, TRUE );
	gtk_tree_view_column_set_resizable( col, TRUE );
	gtk_tree_view_append_column( GTK_TREE_VIEW( widget ), col );
	gtk_tree_view_column_set_attributes( col, rend,
					     "text", 
					     TAG_FILE_URI_COL, NULL ); 
	gtk_tree_view_column_set_title( col, "URI" );

	gtk_tree_view_set_model( GTK_TREE_VIEW( widget ),
				 GTK_TREE_MODEL( model ) );

	sel = gtk_tree_view_get_selection( GTK_TREE_VIEW( widget ) );
	g_signal_connect( G_OBJECT( sel ), "changed",
			  G_CALLBACK( screem_preferences_tag_tree_sel ),
			  application );
}


static void screem_preferences_tag_tree_sel( GtkTreeSelection *sel,
					ScreemApplication *app )
{
	GtkWidget *widget;
	GladeXML *xml;
	GtkTreeModel *model;
	GtkTreeIter it;
	gboolean sens;
	ScreemTagFile *tfile;
	const gchar *uri;
	gchar *dot;
	
	widget = GTK_WIDGET( gtk_tree_selection_get_tree_view( sel ) );
	
	xml = glade_get_widget_tree( widget );
	sens = FALSE;
	uri = NULL;
	dot = screem_get_dot_dir();
	
	if( gtk_tree_selection_get_selected( sel, &model, &it ) ) {
		sens = TRUE;

		gtk_tree_model_get( model, &it,
				TAG_FILE_OBJ_COL, &tfile, -1 );
		uri = screem_tag_file_get_uri( tfile );
	}

	widget = glade_xml_get_widget( xml, "refreshtagfile" );
	gtk_widget_set_sensitive( widget, sens );

	/* we don't want to allow removal of system tag trees,
	 * or tag trees from ~/.screem */
	if( uri &&
	    ( ! strncmp( DATADIR"/screem/tagtrees", uri,
		    strlen( DATADIR"/screem/tagtrees" ) ) ||
	      ! strncmp( dot, uri, strlen( dot ) ) ) )  {
		
		sens = FALSE;
	}
	
	widget = glade_xml_get_widget( xml, "removetagfile" );
	gtk_widget_set_sensitive( widget, sens );

	g_free( dot );
}

void screem_preferences_refresh_tag_file( GtkWidget *widget )
{
	GladeXML *xml;
	GtkTreeSelection *sel;
	GtkTreeModel *model;
	GtkTreeIter it;
	ScreemTagFile *tfile;
	
	xml = glade_get_widget_tree( widget );
	widget = glade_xml_get_widget( xml, "tagtreeview" );
	sel = gtk_tree_view_get_selection( GTK_TREE_VIEW( widget ) );

	if( gtk_tree_selection_get_selected( sel, &model, &it ) ) {

		gtk_tree_model_get( model, &it,
				TAG_FILE_OBJ_COL, &tfile, -1 );

		screem_tag_file_load( tfile, SCREEM_TAG_TREE_FORMAT );
	}
}

void screem_preferences_remove_tag_file( GtkWidget *widget )
{
	GladeXML *xml;
	GtkTreeSelection *sel;
	GtkTreeModel *model;
	GtkTreeIter it;
	ScreemTagFile *tfile;
	const gchar *uri;
	
	xml = glade_get_widget_tree( widget );
	widget = glade_xml_get_widget( xml, "tagtreeview" );
	sel = gtk_tree_view_get_selection( GTK_TREE_VIEW( widget ) );

	if( gtk_tree_selection_get_selected( sel, &model, &it ) ) {

		gtk_tree_model_get( model, &it,
				TAG_FILE_OBJ_COL, &tfile, -1 );
		uri = screem_tag_file_get_uri( tfile );
		
		screem_tag_tree_remove_uri( uri );
	}

}

void screem_preferences_add_tag_file( GtkWidget *widget )
{
	GladeXML *xml;
	GladeXML *dxml;
	GConfClient *client;
	gchar *gladepath;
	const gchar *prompt;
	gchar *txt;
	gchar *title;
	const gchar *url;
	GtkTreeModel *model;
	GtkTreeIter *it;
	
	xml = glade_get_widget_tree( widget );
	widget = glade_xml_get_widget( xml, "tagtreeview" );
	model = gtk_tree_view_get_model( GTK_TREE_VIEW( widget ) );
	
	client = gconf_client_get_default();

	gladepath = gconf_client_get_string( client, 
					     "/apps/screem/general/glade_path",
					     NULL );
	if( ! gladepath ) {
		gladepath = g_strconcat( GLADE_PATH, "/screem.glade",
					 NULL );
	}
	
	dxml = glade_xml_new( gladepath, "open_location", NULL );

	widget = glade_xml_get_widget( dxml, "prompt" );
	prompt = _( "Enter the URL of the tag tree to add" );
	txt = g_strconcat( "<b>", prompt, "</b>", NULL );
	gtk_label_set_markup( GTK_LABEL( widget ), txt );
	g_free( txt );
	
	widget = glade_xml_get_widget( dxml, "open_location" );
	title = g_strdup_printf( "%s - Screem",
			_( "Add Tag Tree from Location" ) );
	gtk_window_set_title( GTK_WINDOW( widget ),
				title );
	g_free( title );

	if( gtk_dialog_run( GTK_DIALOG( widget ) ) == GTK_RESPONSE_OK ) {
		widget = glade_xml_get_widget( dxml, "url" );
		url = gtk_entry_get_text( GTK_ENTRY( widget ) );

		it = screem_support_find_in_list( GTK_LIST_STORE( model ), TAG_FILE_URI_COL, url );

		if( ! it ) {
			/* release lock, as screem_tag_tree_add_uri()
			 * is called from a thread on startup it
			 * needs to claim the lock to do anything */
			gdk_threads_leave();
			screem_tag_tree_add_uri( url, 
					SCREEM_TAG_TREE_FORMAT );	
gdk_threads_enter();
		} else {
			gtk_tree_iter_free( it );
			widget = gtk_message_dialog_new( NULL,
							 GTK_DIALOG_MODAL,
							 GTK_MESSAGE_WARNING,
							 GTK_BUTTONS_OK,
							 _( "A Tag Tree located at %s is already present.  Refusing to add it again." ),
							 url);
			gtk_dialog_run( GTK_DIALOG( widget ) );
			gtk_widget_destroy( widget );
		}
	}

	widget = glade_xml_get_widget( dxml, "open_location" );
	gtk_widget_destroy( widget );
	
	g_object_unref( client );
	g_object_unref( dxml );
}

void screem_preferences_page_setup( void )
{
	GConfClient *client;
	GladeXML *xml;
	gchar *gladepath;
	GtkWidget *widget;
	
	static const gchar *widgets[] = {
		"highlight", "/apps/screem/editor/print/highlight",
		"headers", "/apps/screem/editor/print/headers",
		"numbers", "/apps/screem/editor/print/lines",
		"wrap", "/apps/screem/editor/print/wrap_lines",
		"split", "/apps/screem/editor/print/split",
		NULL
	};
	guint i;
	const gchar *path;
	gboolean val;
	GSList *handles;
	guint handle;
	gdouble sval;
	
	client = gconf_client_get_default();
	
	gladepath = gconf_client_get_string( client, 
					     "/apps/screem/general/glade_path",
					     NULL );
	if( ! gladepath ) {
		gladepath = g_strconcat( GLADE_PATH, "/screem.glade",
					 NULL );
	}

	xml = glade_xml_new( gladepath, "page_setup", NULL );
	glade_xml_signal_autoconnect( xml );	

	for( handles = NULL, i = 0; widgets[ i ]; ++ i ) {
		
		widget = glade_xml_get_widget( xml, widgets[ i ] );
		path = widgets[ ++ i ];
		val = gconf_client_get_bool( client, path, NULL );
		gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ), val );
		g_object_set_data( G_OBJECT( widget ), "key", path );
		g_signal_connect( G_OBJECT( widget ), "toggled",
				G_CALLBACK( screem_preferences_toggle_changed ),
				NULL );
		handle = gconf_client_notify_add( client,
				path,
				screem_preferences_toggle_notify,
				widget, NULL, NULL );
		handles = g_slist_prepend( handles,
				GUINT_TO_POINTER( handle ) );
	}
	widget = glade_xml_get_widget( xml, "lines" ); 
	path = "/apps/screem/editor/print/nlines";
	sval = gconf_client_get_float( client, path, NULL );
	gtk_spin_button_set_value( GTK_SPIN_BUTTON( widget ), sval );
	g_object_set_data( G_OBJECT( widget ), "key", path );
	g_signal_connect( G_OBJECT( widget ), "changed",
			G_CALLBACK( screem_preferences_spin_changed ),
			NULL );
	handle = gconf_client_notify_add( client,
			path,
			screem_preferences_spin_notify,
			widget, NULL, NULL );
	handles = g_slist_prepend( handles,
			GUINT_TO_POINTER( handle ) );
	
	widget = glade_xml_get_widget( xml, "page_setup" );
	g_object_set_data( G_OBJECT( widget ), "handles", handles );

	gtk_widget_show( widget );

	g_free( gladepath );
	g_object_unref( client );
}

void screem_preferences_dialog_destroy( GtkWidget *widget )
{
	GConfClient *client;
	GSList *handles;
	GSList *tmp;
	GladeXML *xml;
	
	handles = g_object_get_data( G_OBJECT( widget ), "handles" );
	if( handles ) {
		client = gconf_client_get_default();
		for( tmp = handles; tmp; tmp = tmp->next ) {
			gconf_client_notify_remove( client,
					GPOINTER_TO_UINT( tmp->data ) );
		}
		g_slist_free( handles );
		g_object_unref( client );
	}

	xml = glade_get_widget_tree( widget );
	g_object_unref( G_OBJECT( xml ) );
	
	gtk_widget_destroy( widget );
}

void screem_preferences_dialog_response_check( GtkWidget *widget, gint response )
{
	if( response < 0 ) {
		screem_preferences_dialog_destroy( widget );
	}
}

static void screem_preferences_add_pref( GtkWidget *dialog,
				const gchar *key,
				ScreemPrefNotifyType type,
				gint value,
				GtkWidget *widget )
{
	GConfClient *client;
	GConfClientNotifyFunc func;
	guint handle;
	GSList *handles;

	gboolean tval;
	gint ival;
	gdouble spval;
	gchar *sval;
	GdkColor colour;
	GtkWidget *gtkentry;
	
	g_return_if_fail( key != NULL );
	g_return_if_fail( widget != NULL );
	
	client = gconf_client_get_default();
	
	handles = g_object_get_data( G_OBJECT( dialog ), "handles" );
	
	g_object_set_data( G_OBJECT( widget ), "key", (gchar*)key );
	
	switch( type ) {
		case SCREEM_PREF_TOGGLE:
			func = screem_preferences_toggle_notify;
			tval = gconf_client_get_bool( client, key, NULL );
			g_signal_connect( G_OBJECT( widget ), "toggled",
					  G_CALLBACK( screem_preferences_toggle_changed ),
					  NULL );
			gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ),
							tval );
			break;
		case SCREEM_PREF_RADIO:
			func = screem_preferences_radio_notify;
			g_object_set_data( G_OBJECT( widget ), "val", 
				   	   GINT_TO_POINTER( value ) );
			ival = gconf_client_get_int( client, key, NULL );
			if( value == ival ) {
				gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ),
								TRUE );
			}
			g_signal_connect( G_OBJECT( widget ), "toggled",
					  G_CALLBACK( screem_preferences_radio_changed ),
					  NULL );
			break;
		case SCREEM_PREF_SPIN:
			func = screem_preferences_spin_notify;
			spval = gconf_client_get_float( client, key, NULL );
			g_signal_connect( G_OBJECT( widget ), "value_changed",
					  G_CALLBACK( screem_preferences_spin_changed ),
					  NULL );
			gtk_spin_button_set_value( GTK_SPIN_BUTTON( widget ),
						   spval );
			break;
		case SCREEM_PREF_COLOUR:
			func = screem_preferences_colour_notify;
			sval = gconf_client_get_string( client, key, NULL );
			g_signal_connect( G_OBJECT( widget ), "color_set",
					  G_CALLBACK( screem_preferences_colour_set ),
					  NULL );
			if( gdk_color_parse( sval, &colour ) ) {
				gnome_color_picker_set_i16( GNOME_COLOR_PICKER(widget),
							    colour.red, colour.green,
							    colour.blue, 0 );
			}
			break;
		case SCREEM_PREF_STRING:
			func = screem_preferences_string_notify;
			sval = gconf_client_get_string( client, key, NULL );
			gtkentry = NULL;
			if( GNOME_IS_ENTRY( widget ) ) {
				g_signal_connect( G_OBJECT( widget ), "changed",
						  G_CALLBACK( screem_preferences_string_set ),
						  NULL );
				gtkentry = gnome_entry_gtk_entry( GNOME_ENTRY(widget));
			} else if( GTK_IS_ENTRY( widget ) ) {
				g_signal_connect( G_OBJECT( widget ), "changed",
						  G_CALLBACK( screem_preferences_string_set ),
						  NULL );
				gtkentry = widget;
			} else if( GTK_IS_COMBO( widget ) ) {
				gtkentry = GTK_COMBO( widget )->entry;
				g_object_set_data( G_OBJECT( gtkentry ), "key", (gchar*)key );
				g_signal_connect( G_OBJECT( gtkentry ), "changed",
						  G_CALLBACK( screem_preferences_string_set ),
						  NULL );
			} else if( GNOME_IS_FILE_ENTRY( widget ) ) {
				g_signal_connect( G_OBJECT( widget ), "changed",
						  G_CALLBACK( screem_preferences_string_set ),
						  NULL );
				gtkentry = gnome_file_entry_gtk_entry( GNOME_FILE_ENTRY( widget ) );
			} else if( GNOME_IS_ICON_ENTRY( widget ) ) {
				g_signal_connect( G_OBJECT( widget ), "changed",
						  G_CALLBACK( screem_preferences_icon_changed ),
						  NULL );
				gnome_icon_entry_set_filename( GNOME_ICON_ENTRY( widget ),
							       sval);
			} else if( GNOME_IS_FONT_PICKER( widget ) ) {
				g_signal_connect( G_OBJECT( widget ), "font_set",
						  G_CALLBACK( screem_preferences_font_set ),
						  NULL );
				gnome_font_picker_set_font_name( GNOME_FONT_PICKER( widget ), sval );
			}
			if( gtkentry && sval ) {
				gtk_entry_set_text( GTK_ENTRY( gtkentry ), sval );
			}
			g_free( sval );
			break;
		default:
			g_warning( "Invalid preference notify type\n" );
			return;
			break;
	}

	handle = gconf_client_notify_add( client,
					  key, func,
					  widget, NULL, NULL );
	handles = g_slist_prepend( handles,
			GUINT_TO_POINTER( handle ) );
	g_object_set_data( G_OBJECT( dialog ), "handles", handles );
	g_object_unref( client );
}
