/*  Screem:  screem-linkview.c
 *
 *  The link view widget
 *
 *  Copyright (C) 2001  Matt Colyer, David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */
#include <config.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <limits.h>
#include <unistd.h>
#include <string.h>
#include <math.h>

#include <gmodule.h>

#include <libgnome/gnome-i18n.h>

#include <libgnomeui/gnome-popup-menu.h>

#include <libgnomecanvas/gnome-canvas.h>
#include <libgnomecanvas/gnome-canvas-rect-ellipse.h>
#include <libgnomecanvas/gnome-canvas-text.h>
#include <libgnomecanvas/gnome-canvas-util.h>
#include <libgnomecanvas/gnome-canvas-line.h>

#include <libgnomevfs/gnome-vfs-mime-handlers.h>
#include <libgnomevfs/gnome-vfs-mime-utils.h>
#include <libgnomevfs/gnome-vfs-ops.h>
#include <libgnomevfs/gnome-vfs-job-limit.h>

#include <glade/glade.h>

#include <gdk/gdktypes.h>

#include <glib-object.h>

#include <gtk/gtkradiobutton.h>
#include <gtk/gtkspinbutton.h>
#include <gtk/gtkvbox.h>
#include <gtk/gtklabel.h>
#include <gtk/gtkadjustment.h>
#include <gtk/gtkmain.h>
#include <gtk/gtkimage.h>
#include <gtk/gtkscrolledwindow.h>
#include <gtk/gtkcheckbutton.h>

#include <gconf/gconf-client.h>

#include "screem-application.h"
#include "screem-window.h"
#include "screem-linkview.h"
#include "screem-plugin.h"

#include "fileops.h"

#include "screem-markup.h"

#include "screem-view.h"

#include "screem-linkview-util.h"
#include "screem-link-view-image.h"
#include "screem-link-view-html.h"

#include "libegg/menu/egg-menu.h"

#include "support.h"

static const gdouble half_mpi = M_PI / 2;
static const gdouble twice_mpi = M_PI * 2;

enum {
	PROP_0,
	PROP_UPLOAD_TABLE,
	PROP_MIN_X,
	PROP_MIN_Y,
	PROP_MAX_X,
	PROP_MAX_Y

};

struct ScreemLinkViewPrivate {
	GtkWidget *mainbox;
	
	GConfClient *client;

	gboolean check_external;
	gint check_external_notify;

	GHashTable *external_check_cache;
	gdouble max_x;
	gdouble max_y;
	gdouble min_x;
	gdouble min_y;

	gint zoom;
	gint depth;

	GNode *link_view_tree;
	
	GtkWidget *canvas;
	GObject *mainroot;
	GObject *root;

	GtkWidget *panel;
	GtkWidget *zoom_control;
	GtkWidget *depth_control;

	GtkWidget *check_links;

	gboolean show_links;

	GHashTable *checked_files;
	GHashTable *upload_table;

	ScreemSite *site;
	ScreemPage *page;
};

static void screem_link_view_display_view( ScreemView *view );

static void screem_link_view_display( ScreemLinkView *link_view, 
				      ScreemSite *site, 
				      ScreemPage *page,
				      gboolean online );

static gboolean traverse_for_remote(GNode *node, ScreemLinkView *link_view );

static GnomeCanvasGroup* 
screem_link_view_add_key( ScreemLinkView *link_view,
			  GnomeCanvas *legend, GnomeCanvasGroup *root, gdouble x, 
			  gdouble y );

static GNode* build_link_tree( ScreemLinkView *link_view,
			       ScreemWindow *window,
			       gboolean online,
			       const gchar *pathname, 
			       GNode *parent, gint level );

static void build_icons( ScreemLinkView *link_view,
			 GNode *node, GnomeCanvasGroup *root, gdouble root_x, 
			 gdouble root_y, gdouble root_angle, gint level, 
			 gint numOfNodesAtLevel, gint nthNode );

static gint background_event( GtkWidget *canvas, GdkEventButton *event, 
			      ScreemLinkView *link_view );

static void free_node_list( GNode* node, gpointer data );

static void remove_data (GNode* parent);
static gboolean screem_link_view_stop_check( GNode *node, ScreemLinkView *link_view );
static gboolean screem_link_view_check_remote_fill( ScreemLinkViewHtml *node );
static void screem_link_view_async_cb( GnomeVFSAsyncHandle *handle,
				       GnomeVFSResult result,
				       gpointer data );
static void screem_link_view_async_close_cb( GnomeVFSAsyncHandle *handle,
					     GnomeVFSResult result,
					     gpointer data );
static void* screem_link_view_check( GNode *node );

static gboolean link_view_toggle_labels(GNode *node, ScreemLinkView *link_view);

static void screem_link_view_sidebar_switch( GtkWidget *widget, gpointer data );
static void screem_link_view_sidebar_zoom( GtkWidget *widget );

static void screem_link_view_check_toggle( GtkWidget *widget );

void screem_link_view_depth( GtkWidget *widget );

static void screem_link_view_class_init( ScreemLinkViewClass *klass );
static void screem_link_view_init( ScreemLinkView *link_view );
static void screem_link_view_finalize( GObject *link_view );
static void screem_link_view_size_request( GtkWidget *widget, GtkRequisition *req );
static void screem_link_view_realize( GtkWidget *widget );
static void screem_link_view_hide( GtkWidget *widget );
static void screem_link_view_show( GtkWidget *widget );
static void screem_link_view_window_set( ScreemView *view );
static void screem_link_view_set_prop( GObject *object, guint property_id,
				       const GValue *value, GParamSpec *pspec );
static void screem_link_view_get_prop( GObject *object, guint property_id,
				       GValue *value, GParamSpec *pspec);


static void screem_link_view_check_external_notify( GConfClient *client,
						    guint cnxn_id,
						    GConfEntry *entry,
						    gpointer data );

ScreemLinkView *screem_link_view_new( ScreemWindow *window )
{
	ScreemLinkView *link_view;
	GType type;

	type = screem_link_view_get_type();

	link_view = SCREEM_LINK_VIEW( g_object_new( type, "window", window,
						    NULL ) );

	return link_view;
}

/* static stuff */

static void screem_link_view_display_view( ScreemView *view )
{
	ScreemSite *site;
	ScreemPage *page;
	gboolean online;

	g_object_get( G_OBJECT( view ),
		      "site", &site,
		      "page", &page,
		      "online", &online,
		      NULL );
	screem_link_view_display( SCREEM_LINK_VIEW( view ),
				  site, page, online );

	if( site ) {
		g_object_unref( site );
	}
	if( page ) {
		g_object_unref( page );
	}
}

static void screem_link_view_display( ScreemLinkView *link_view, 
				      ScreemSite *site, 
				      ScreemPage *page,
				      gboolean online )
{
	ScreemLinkViewPrivate *private;
	GtkWidget *canvas;
	GtkType type;
	ScreemApplication *application;
	GModule *uploadWiz;
	SiteSyncState screem_site_sync_status;
	SiteSyncState *status;
	ScreemWindow *window;

	private = link_view->private;

	canvas = private->canvas;

	private->checked_files = NULL;
	private->upload_table = NULL;

	uploadWiz = NULL;

	g_object_get( G_OBJECT( link_view ), 
		      "window", &window, 
		      NULL );
	
	application = window->application;
	uploadWiz = screem_plugin_get_sitecopy( application );
	
	if( private->link_view_tree ) {
		g_node_traverse(private->link_view_tree,
				G_IN_ORDER, G_TRAVERSE_ALL, -1,
				(GNodeTraverseFunc)screem_link_view_stop_check,
				link_view );
		remove_data ( private->link_view_tree );
		g_node_destroy( private->link_view_tree );
		private->link_view_tree = NULL;
		gtk_object_destroy( GTK_OBJECT( private->root ) );
		private->root = NULL;
	}

	/* we must have a page open */
	if( ! page ) {
		return;
	}

	private->site = site;
	private->page = page;

	if( uploadWiz ) {
		status = &screem_site_sync_status;
		g_module_symbol( uploadWiz, "screem_site_get_sync_status", 
				 (gpointer*)status );
		
		screem_site_sync_status( site, &private->upload_table );

		/* if failed table will be NULL, otherwise the status
		   of a file can be obtained via 
		   g_hash_table_lookup( table, filename );
		   the states are defined in screem-linkview-util.h */
	}

	gnome_canvas_set_pixels_per_unit( GNOME_CANVAS( canvas ),
					  private->zoom/100.0 );

	private->max_x = 0.0;
	private->max_y = 0.0;
	private->min_x = 0.0;
	private->min_y = 0.0;

	type = gnome_canvas_group_get_type();
	private->root = G_OBJECT( gnome_canvas_item_new( GNOME_CANVAS_GROUP( private->mainroot ),
							 type,
							 "x", 0.0,
							 "y", 0.0,
							 NULL ) );

	/* we now need to build up the tree of links */
	private->checked_files = g_hash_table_new( g_str_hash, g_str_equal );
	private->link_view_tree = build_link_tree( link_view,
						   window,
						   online,
						   screem_page_get_pathname( page ),
						   NULL, 0 );
	
	/* we can destroy the checked_files hash table now
	   as we no longer need it */
	g_hash_table_destroy( private->checked_files );

	/* GNode tree built, now create the icons */
	build_icons( link_view, private->link_view_tree,
		     GNOME_CANVAS_GROUP( private->root ),
		     0.0, 0.0, 0,0, 1, 0 );

	type = gnome_canvas_rect_get_type();
	
	gnome_canvas_item_raise_to_top( GNOME_CANVAS_ITEM( private->root ) );
	
	gnome_canvas_set_scroll_region( GNOME_CANVAS( canvas ),
					private->min_x - 20,
					private->min_y - 20,
					private->max_x + 20,
					private->max_y + 20 );

	/* if we aren't showing text labels hide them now */

	g_node_traverse( private->link_view_tree,G_IN_ORDER,G_TRAVERSE_ALL,-1,
			 (GNodeTraverseFunc)traverse_for_remote, link_view );
}

static gboolean traverse_for_remote( GNode *node, ScreemLinkView *link_view )
{
	ScreemLinkViewPrivate *private;
	ScreemLinkViewHtml *html;
	gboolean online;

	g_object_get( G_OBJECT( link_view ), "online", &online, NULL );

	private = link_view->private;
	html = SCREEM_LINK_VIEW_HTML( node->data );

	html->status = REMOTE_NOT_CHECKED;

	if( html->remoteAddr && ( ! online || ! private->check_external ) ) {
		gnome_canvas_item_set( html->icon, "fill_color", "grey", NULL );
	} else {
		gnome_canvas_item_set( html->icon, "fill_color", "orange",
				       NULL );
		/* release lock, as we call a function which
		 * is also called by an idle handler, so it
		 * claims the lock itself */
		gdk_threads_leave();
		screem_link_view_check( node );
		gdk_threads_enter();
	}

	return FALSE;
}


static GnomeCanvasGroup* 
screem_link_view_add_key( ScreemLinkView *link_view,
			  GnomeCanvas *legend, GnomeCanvasGroup *root, 
			  gdouble x, gdouble y )
{
	ScreemLinkViewPrivate *private;
	GnomeCanvasGroup *group;
	GnomeCanvasItem *image;
	GnomeCanvasItem *text;
	GtkType type;

	gdouble x2 = 0;
	gdouble y2;
	gdouble w;

	gchar *font;

	gint icons[] = {
		GOODNODE,
		EXTERNALNODE,
		BADNODE,
		NONODE,
		UPDATENODE,
		CHECKNODE,
		-1
	};
	gchar *names[] = {
		_( "Local" ),
		_( "External" ),
		_( "Broken" ),
		_( "Ignored" ),
		_( "Changed" ),
		_( "Checking" ),
		NULL
	};
	gint i;


	private = link_view->private;

	/* create the group */
	type = gnome_canvas_group_get_type();
	group = GNOME_CANVAS_GROUP( gnome_canvas_item_new( root, type,
							   "x", x,
							   "y", y,
							   NULL ) );
   
	/* create the image */
	y2 = 4.0;
	font = gconf_client_get_string( private->client,
					"/apps/screem/editor/font", NULL );
	w = 0;
	for( i = 0; icons[ i ] != -1; ++ i ) {
		image = node_draw( group, w, y2, icons[ i ], TRUE );
					       
		type = gnome_canvas_text_get_type();
		text = gnome_canvas_item_new( group, type, 
					      "x", w + 15.0,
					      "y", y2 + 4.0,
					      "anchor", GTK_ANCHOR_WEST,
					      "text", names[ i ],
					      "font", font,
					       "fill_color", "black",
					      NULL );
		
		w += ((GnomeCanvasText*)text)->max_width;
		w += 20.0;
		w += 10.0;
		if( w > x2 ) {
			x2 = w;
		}

		/* next */
		if( ( ( i + 1 ) % 6 ) == 0 ) {
			w = 0;
			y2 += 20.0;
		}
	}
	if( font ) {
		g_free( font );
	}

	/* y2 will be the height we want to set the legend canvas to */
	gtk_widget_set_size_request( GTK_WIDGET( legend ), x2, y2 + 4.0  );

	return group;
}

static GNode *build_link_tree( ScreemLinkView *view, 
			       ScreemWindow *window,
			       gboolean online,
			       const gchar *pathname,
			       GNode *parent, gint level )
{
	ScreemLinkViewPrivate *private;
	ScreemSite *site;
	const gchar *sitepath;
	GNode *ret = NULL;

	ScreemPage *page;
	gchar *mime_type;
	GNode *node = NULL;
	ScreemDTD *dtd;

	private = view->private;

	site = SCREEM_SITE( private->site );
	sitepath = screem_site_get_pathname( site );

	/* pathname will be NULL for a new blank document, hence
	   screem_site_locate_page() will fail */
	if( pathname ) {
		page = screem_site_locate_page( site, pathname );
	} else {
		/* as pathname is NULL we can safely assume that this
		   is the current document, as it can't be linked in
		   otherwise */
		page = screem_window_get_document( window );
	}

	dtd = NULL;
	mime_type = NULL;

	if( page ) {
		dtd = screem_page_get_dtd( page );
	} else if( FALSE && online ) {
		/* can lead to network access, therefore we don't want
		   to do this if we are not online */
		mime_type = screem_get_mime_type( pathname );
	}
	
	if( mime_type &&
	    ! strncmp( "image/", mime_type, strlen( "image/" ) ) ) {
		ScreemLinkViewHtml *html;	
		ScreemLinkViewImage *image;
		
		g_assert( parent );
		html = SCREEM_LINK_VIEW_HTML( parent->data );
		image = screem_link_view_image_new( pathname );
		
		image->next = html->images;
		html->images = image;
	} else {
		ScreemLinkViewHtml *html;
		html = screem_link_view_html_new( view, page, pathname );
		if( parent ) {
			node = g_node_append_data( parent, html );
		} else {
			node = g_node_new( html );
		}
	}
	if( mime_type ) {
		g_free( mime_type );
	}
	
	/* page is NULL for external links, non existant link or non text files,
	   node should null for images
	   
	   FIXME: we still need to handle non existant local links, even though
	   page is NULL */
	
	if( node && page ) {
		GSList *this_level = NULL;
		GSList *links = NULL;
		GSList *list;
		gchar *base_uri;
		gint processed;
		
		ret = node;
		if( pathname ) {
			base_uri = g_path_get_dirname( pathname );
			if( ! sitepath ) {
				sitepath = base_uri;
			}
		} else {
			base_uri = g_strdup( "/" );
			pathname = _( "Untitled" );
		}
		
		/* have we already processed this page before? */
		processed = GPOINTER_TO_INT( g_hash_table_lookup( private->checked_files, pathname ) );
		if( processed <= 0 ) {
			gchar *text;
			
			g_hash_table_replace( private->checked_files,
					      g_strdup( pathname ),
					      GINT_TO_POINTER( level + 1 ) );
			
			screem_page_load( page );
			text = screem_page_get_data( page );
			links = screem_markup_get_links( dtd, text );
			g_free( text );
		}
		for( list = links; list; list = list->next ) {
			GnomeVFSURI *uri;
			const gchar *path = (const gchar*) list->data;
			gchar *full;
			
			if( *path == '#' ) {
				path = pathname;
			}
			
			uri = gnome_vfs_uri_new( path );
			if( uri && path && *path == '/' ) {
				/* absolute link, strip fragment */
				full = gnome_vfs_uri_to_string( uri, GNOME_VFS_URI_HIDE_FRAGMENT_IDENTIFIER );
			} else if( uri ) {
				/* must be relative or include scheme */
				full = relative_to_full( path, 
							 base_uri );

				gnome_vfs_uri_unref( uri );
				uri = gnome_vfs_uri_new( full );
				g_free( full );
				full = gnome_vfs_uri_to_string( uri, GNOME_VFS_URI_HIDE_FRAGMENT_IDENTIFIER );
			}
			if( uri ) {
				gint processed;
				
				gnome_vfs_uri_unref( uri );
				
				processed = GPOINTER_TO_INT( g_hash_table_lookup( private->checked_files, full ) );
				
				if( *full && level < private->depth &&
				    ! processed ) {
					this_level = g_slist_prepend(this_level,
								     full );
					
					if( ! g_hash_table_lookup( private->checked_files, full ) ) {
						g_hash_table_insert( private->checked_files, full, GINT_TO_POINTER( -1 ) );
					}
				} else {
					g_free( full );
				}
			}
		}
		
		/* now try and build next level of links */
		for( list = this_level; list; list = list->next ) {
			build_link_tree( view, window, online,
					 list->data, node,
					 level + 1 );
			g_free( list->data );
		}
		g_slist_free( this_level );
		g_free( base_uri );
	}

	return ret;
}

static void build_icons( ScreemLinkView *link_view,
			 GNode *node, GnomeCanvasGroup *root, gdouble root_x, 
			 gdouble root_y, gdouble root_angle, gint level, 
			 gint numOfNodesAtLevel, gint nthNode )
{
	ScreemLinkViewPrivate *private;
	gint siblings;
	gint currentNode = 0;
	GNode *temp;

	GnomeCanvasPoints *points;
	GnomeCanvasItem *line;

	gdouble y;
	gdouble x = 10.0;
	static gdouble width = 0.0;

	GtkType type;
	
	gdouble temp_x;
	gdouble temp_y;
	gdouble temp_angle;
	
	gdouble angle = 0.0;

	private = link_view->private;

	g_assert( node );

	if( level == 0 ) {
		/* Base Node Case*/
		
		/* Count children*/
		siblings = 0;
		for( temp = g_node_first_child( node ); temp;
		     temp = g_node_next_sibling( temp ) ) {
			siblings++;     	
		}
		
		/* Build children */
		for( temp = g_node_first_child( node ); temp;
		     temp = g_node_next_sibling( temp ) ) {
		     							       
			build_icons( link_view, temp, root, root_x, root_y, 
				     - 0.785, 1, 
				     siblings ,currentNode);
			currentNode++;			
		}
		
		/* build the root icon */
		screem_link_view_html_build_icon(SCREEM_LINK_VIEW_HTML(node->data),
						 root_x, root_y, angle, &width, 
						 root );
	} else if( level <= private->depth ) {
		/* this is the links from the root icon, create an icon
		   then call build_icons again */
							   
		/* formulas to create a circle */
		if( level != 1 ) {
			angle = (half_mpi/numOfNodesAtLevel)*(nthNode+1)+root_angle;
	
			if( numOfNodesAtLevel > 4 ) {
				x = cos(angle)*30*numOfNodesAtLevel;
				y = sin(angle)*30*numOfNodesAtLevel;
			} else {
				x = cos(angle)*160;
				y = sin(angle)*160;
			}
		} else {
			angle = (twice_mpi/numOfNodesAtLevel)*(nthNode+1);

			x = cos(angle)*180;
			y = sin(angle)*180;
			type = gnome_canvas_line_get_type();
			points = gnome_canvas_points_new( 2 );
			points->coords[ 0 ] = (gdouble)0;
			points->coords[ 1 ] = (gdouble)0;
			points->coords[ 2 ] = (gdouble)x;
			points->coords[ 3 ] = (gdouble)y;		
			line = gnome_canvas_item_new( root, type,
						      "points", points,
						      "fill_color", "green",
						      NULL );
			gnome_canvas_points_unref(points);
			gnome_canvas_item_lower_to_bottom(line);
		}
			
		x = root_x + x;
		y = root_y + y;
							   
		/* build this icon */	
		screem_link_view_html_build_icon(SCREEM_LINK_VIEW_HTML(node->data),
						 x, y, 
						 angle, &width, 
						 root );
				
		/* build children */
		if( ( temp = g_node_first_child( node ) ) ) {		
			/**/
			gdouble tempa;
			gint mult;

			currentNode = siblings = 0;
			while( temp ) {
				siblings ++;
				temp = g_node_next_sibling( temp );
			}
			
			tempa = half_mpi / siblings;
			if( siblings > 5 ) {
				mult = 30 * siblings;
			} else {
				mult = 160;
			}

			for( temp = g_node_first_child( node ), angle -= 0.785; temp;
			     temp = g_node_next_sibling( temp ), ++ currentNode ) {
				build_icons( link_view, temp, root, 
					     x, y, 
					     angle, level +1,
					     siblings, currentNode );

				/* draw a line*/
				type = gnome_canvas_line_get_type();
				points = gnome_canvas_points_new( 2 );
				points->coords[ 0 ] = x;
				points->coords[ 1 ] = y;
				
				/* compute new x,y*/
				temp_angle = tempa + angle;
				temp_x = cos( temp_angle ) * mult;
				temp_y = sin( temp_angle ) * mult;
				
				tempa += tempa;

				points->coords[ 2 ] = (gdouble)temp_x + x;
				points->coords[ 3 ] = (gdouble)temp_y + y;
				line = gnome_canvas_item_new( root, type,
							      "points", points,
							      "fill_color", "green",
							      NULL );
				gnome_canvas_points_unref(points);
				gnome_canvas_item_lower_to_bottom(line);
			}
		}
	}
}

static gint background_event( GtkWidget *canvas, GdkEventButton *event, 
			      ScreemLinkView *link_view )
{
	ScreemLinkViewPrivate *private;
	GtkWidget *menu;
	GnomeCanvasItem *item;
	gdouble x;
	gdouble y;
	ScreemWindow *window;
	ScreemLinkViewHtml *html;

	g_object_get( G_OBJECT( link_view ), "window", &window, NULL );

	private = link_view->private;

	gnome_canvas_window_to_world( GNOME_CANVAS( canvas ),
				      event->x, event->y,
				      &x, &y );

	item = gnome_canvas_get_item_at( GNOME_CANVAS( canvas ),
					 x, y );
	if( item ) {
		html = g_object_get_data( G_OBJECT( item ), 
					  "ScreemLinkViewHtml" );
	} else {
		html = NULL;
	}

       	if( ( ! html ) && event->button == 3 ) {
		EggAction *action;
		
		g_signal_stop_emission_by_name( G_OBJECT( canvas ),
						"button_press_event" );
		menu = egg_menu_merge_get_widget( EGG_MENU_MERGE( window->merge ),
							"/popups/linkviewmenu" );
			
		action = egg_action_group_get_action( EGG_ACTION_GROUP( window->action_group ),
							"ExcludeFlag" );
		g_object_set( G_OBJECT( action ), "visible", FALSE, NULL );
		action = egg_action_group_get_action( EGG_ACTION_GROUP( window->action_group ),
							"IgnoreFlag" );
		g_object_set( G_OBJECT( action ), "visible", FALSE, NULL );
		action = egg_action_group_get_action( EGG_ACTION_GROUP( window->action_group ),
							"ASCIIFlag" );
		g_object_set( G_OBJECT( action ), "visible", FALSE, NULL );
		
		gnome_popup_menu_do_popup_modal( menu, 0, 0, event,
						 0, 0 );
			
		action = egg_action_group_get_action( EGG_ACTION_GROUP( window->action_group ),
							"ExcludeFlag" );
		g_object_set( G_OBJECT( action ), "visible", TRUE, NULL );
		action = egg_action_group_get_action( EGG_ACTION_GROUP( window->action_group ),
							"IgnoreFlag" );
		g_object_set( G_OBJECT( action ), "visible", TRUE, NULL );
		action = egg_action_group_get_action( EGG_ACTION_GROUP( window->action_group ),
							"ASCIIFlag" );
		g_object_set( G_OBJECT( action ), "visible", TRUE, NULL );
	} else {
		item = NULL;
	}

	return ( item != NULL );
}

static void remove_data (GNode* parent)
{
	g_node_traverse(parent,G_IN_ORDER,G_TRAVERSE_ALL,-1,
			(GNodeTraverseFunc)free_node_list,
			NULL);
}
static void free_node_list( GNode* node, gpointer data )
{
	g_object_unref( G_OBJECT( node->data ) );
}

static gboolean screem_link_view_stop_check( GNode *node,
					     ScreemLinkView *link_view )
{
	ScreemLinkViewPrivate *private;
	ScreemLinkViewHtml *html;
	
	private = link_view->private;
	html = SCREEM_LINK_VIEW_HTML( node->data );
	
	if( html->handle ) {
		gnome_vfs_async_cancel( html->handle );
	}
	
	return FALSE;
}

static gboolean screem_link_view_check_remote_fill( ScreemLinkViewHtml *node )
{
	g_return_val_if_fail( SCREEM_IS_LINK_VIEW_HTML( node ), FALSE );

	gdk_threads_enter();
	
	switch( node->status ) {
	case REMOTE_EXISTS:
		if( node->remoteAddr ) {
			gnome_canvas_item_set( node->icon,
					       "fill_color", "blue", NULL );
		} else if( node->uploadStatus == file_new ||
			   node->uploadStatus == file_changed ) {
			gnome_canvas_item_set( node->icon,
					       "fill_color", "yellow", NULL );
		} else {
			gnome_canvas_item_set( node->icon,
					       "fill_color", "green", NULL );
		}
		break;
	case REMOTE_BROKEN:
		gnome_canvas_item_set( node->icon, 
				       "fill_color", "red", NULL );
		break;
	case REMOTE_UNREACHABLE:
		/* not used yet */
		break;
	default:
		g_assert( FALSE );  /* should never happen */
		break;
	}

	gdk_threads_leave();
	
	return FALSE;
}

static void* screem_link_view_check( GNode *node )
{
	GnomeVFSAsyncHandle *handle;
	ScreemLinkViewHtml *data;
	const gchar *uri;

	data = (ScreemLinkViewHtml*)node->data;

	if( data->remoteAddr ) {
		uri = data->remoteAddr;
	} else {
		uri = data->filename;
	}

	if( data->page && ! screem_page_get_pathname( data->page ) ) {
		data->status = REMOTE_EXISTS;
		screem_link_view_check_remote_fill( data );
	} else {
	    	gnome_vfs_async_open( &handle, uri, GNOME_VFS_OPEN_READ,
				      -10, 
				      (GnomeVFSAsyncOpenCallback)screem_link_view_async_cb,
				      data );
		data->handle = handle;
	}
	
	return NULL;
}

static void screem_link_view_async_cb( GnomeVFSAsyncHandle *handle,
				       GnomeVFSResult result,
				       gpointer data )
{
	ScreemLinkViewHtml *fdata;

	g_assert( data );

	fdata = SCREEM_LINK_VIEW_HTML( data );

	switch( result ) {
	case GNOME_VFS_OK:
		/* exists */
		fdata->status = REMOTE_EXISTS;
		fdata->handle = NULL;
		gnome_vfs_async_close( handle,
				       screem_link_view_async_close_cb, 
				       NULL );
		break;
	case GNOME_VFS_ERROR_IS_DIRECTORY:
		fdata->status = REMOTE_EXISTS;
		fdata->handle = NULL;
		break;
	default:
		fdata->status = REMOTE_BROKEN;
		fdata->handle = NULL;
		break;
	}

	g_idle_add( (GSourceFunc)screem_link_view_check_remote_fill,
		    fdata );
}

static void screem_link_view_async_close_cb( GnomeVFSAsyncHandle *handle,
					     GnomeVFSResult result,
					     gpointer data )
{
}

static gboolean link_view_toggle_labels(GNode *node, ScreemLinkView *link_view)
{
	if( link_view->private->show_links ) {
		gnome_canvas_item_show(((ScreemLinkViewHtml*)node->data)->label);
	} else {
		gnome_canvas_item_hide(((ScreemLinkViewHtml*)node->data)->label);
		
	}
	return FALSE;
}



static void screem_link_view_sidebar_switch( GtkWidget *widget, gpointer data )
{
	ScreemLinkView *link_view;
	ScreemLinkViewPrivate *private;

	link_view = SCREEM_LINK_VIEW( widget );
	private = link_view->private;

	private->show_links =gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( data ) );
		
	g_node_traverse( private->link_view_tree, G_IN_ORDER, G_TRAVERSE_ALL, -1,
			 (GNodeTraverseFunc)link_view_toggle_labels,
			 link_view );
}

static void screem_link_view_sidebar_zoom( GtkWidget *widget )
{
	ScreemLinkView *link_view;
	ScreemLinkViewPrivate *private;

	link_view = SCREEM_LINK_VIEW( widget );
	private = link_view->private;

	private->zoom = gtk_spin_button_get_value_as_int( GTK_SPIN_BUTTON( private->zoom_control ) );
	gnome_canvas_set_pixels_per_unit( GNOME_CANVAS( private->canvas ), 
					  private->zoom/100.0 );
}

static void screem_link_view_check_toggle( GtkWidget *widget )
{
	ScreemLinkView *link_view;
	ScreemLinkViewPrivate *private;
	GConfClient *client;

	link_view = SCREEM_LINK_VIEW( widget );
	private = link_view->private;

	client = gconf_client_get_default();

	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( private->check_links ) ) ) {
		/* yes */
		private->check_external = TRUE;
		gconf_client_set_bool( client, "/apps/screem/linkview/check_external",
				       TRUE, NULL );
		g_node_traverse(private->link_view_tree,
				G_IN_ORDER, G_TRAVERSE_ALL, -1,
				(GNodeTraverseFunc)screem_link_view_stop_check,
				link_view );
		g_node_traverse( private->link_view_tree,G_IN_ORDER,G_TRAVERSE_ALL,-1,
				 (GNodeTraverseFunc)traverse_for_remote, link_view );
	} else {
		/* no */
		private->check_external = FALSE;
		g_node_traverse(private->link_view_tree,
				G_IN_ORDER, G_TRAVERSE_ALL, -1,
				(GNodeTraverseFunc)screem_link_view_stop_check,
				link_view );
		gconf_client_set_bool( client, "/apps/screem/linkview/check_external",
				       FALSE, NULL );
	}

	g_object_unref( client );
}

void screem_link_view_depth( GtkWidget *widget )
{
	ScreemLinkView *link_view;
	ScreemLinkViewPrivate *private;

	link_view = SCREEM_LINK_VIEW( widget );
	private = link_view->private;

	private->depth = gtk_spin_button_get_value_as_int( GTK_SPIN_BUTTON( private->depth_control ) );

	screem_view_display( SCREEM_VIEW( link_view ) );
}

static void screem_link_view_check_external_notify( GConfClient *client,
						    guint cnxn_id,
						    GConfEntry *entry,
						    gpointer data )
{
	ScreemLinkView *view;
	ScreemLinkViewPrivate *private;

	view = SCREEM_LINK_VIEW( data );
	private = view->private;

	if( entry->value->type == GCONF_VALUE_BOOL ) {
		gboolean state = private->check_external;

		private->check_external = gconf_value_get_bool( entry->value );

		if( private->check_external != state ) {
			GSList *list;
			GtkWidget *radio;

			list = gtk_radio_button_get_group( GTK_RADIO_BUTTON( private->check_links ) );
			radio = private->check_links;

			if( ! private->check_external ) {
				radio = GTK_WIDGET( list->data );
			}

			gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(radio),
						      TRUE );
		}
	}
}

/* G Object stuff */
#define PARENT_TYPE SCREEM_TYPE_VIEW

static gpointer parent_class;

static void screem_link_view_class_init( ScreemLinkViewClass *klass )
{
	GObjectClass *object_class;
	GtkWidgetClass *widget_class;

	GParamSpec *pspec;

	object_class = G_OBJECT_CLASS( klass );
	parent_class = g_type_class_peek_parent( klass );
	widget_class = (GtkWidgetClass *)klass;

	G_OBJECT_CLASS( object_class )->get_property = screem_link_view_get_prop;
	G_OBJECT_CLASS( object_class )->set_property = screem_link_view_set_prop;

	pspec = g_param_spec_pointer( "uploadtable", "uploadtable",
				      "uploadtable",
				      G_PARAM_READABLE );
	g_object_class_install_property( G_OBJECT_CLASS( object_class ),
					 PROP_UPLOAD_TABLE,
					 pspec );

	pspec = g_param_spec_double( "minx", "minx",
				  "minx",
				  -INT_MAX, INT_MAX, 0,
				  G_PARAM_READABLE | G_PARAM_WRITABLE );
	g_object_class_install_property( G_OBJECT_CLASS( object_class ),
					 PROP_MIN_X,
					 pspec );
	pspec = g_param_spec_double( "miny", "miny",
				  "miny",
				  -INT_MAX, INT_MAX, 0,
				  G_PARAM_READABLE | G_PARAM_WRITABLE );
	g_object_class_install_property( G_OBJECT_CLASS( object_class ),
					 PROP_MIN_Y,
					 pspec );
	pspec = g_param_spec_double( "maxx", "maxx",
				  "maxx",
				  -INT_MAX, INT_MAX, 0,
				  G_PARAM_READABLE | G_PARAM_WRITABLE );
	g_object_class_install_property( G_OBJECT_CLASS( object_class ),
					 PROP_MAX_X,
					 pspec );
	pspec = g_param_spec_double( "maxy", "maxy",
				  "maxy",
				  -INT_MAX, INT_MAX, 0,
				  G_PARAM_READABLE | G_PARAM_WRITABLE );
	g_object_class_install_property( G_OBJECT_CLASS( object_class ),
					 PROP_MAX_Y,
					 pspec );

	object_class->finalize = screem_link_view_finalize;

	widget_class->hide = screem_link_view_hide;
	widget_class->show = screem_link_view_show;
	widget_class->realize = screem_link_view_realize;
	widget_class->size_request = screem_link_view_size_request;
}

static void screem_link_view_init( ScreemLinkView *link_view )
{
	ScreemLinkViewPrivate *private;

	GtkWidget *label;
	GtkWidget *legend;
	GtkWidget *box;
	GtkAdjustment *adjustment;
	GnomeCanvasGroup *legend_root;
	GnomeCanvasItem *key;
	GdkColor back;
	GtkWidget *window;

	SCREEM_VIEW( link_view )->window_set = screem_link_view_window_set;
	SCREEM_VIEW( link_view )->display = screem_link_view_display_view;

	link_view->private = private = g_new0( ScreemLinkViewPrivate, 1 );
	
	link_view->private->mainbox = gtk_vbox_new( FALSE, 0 );
	
	private->client = gconf_client_get_default();

	gconf_client_add_dir( private->client, "/apps/screem/editor",
			      GCONF_CLIENT_PRELOAD_NONE, NULL );
	gconf_client_add_dir( private->client, "/apps/screem/linkview",
			      GCONF_CLIENT_PRELOAD_NONE, NULL );


       	private->zoom = 100;
	private->depth = 3;
	private->show_links = TRUE;

	window = gtk_scrolled_window_new( NULL, NULL );
	gtk_scrolled_window_set_policy( GTK_SCROLLED_WINDOW( window ),
					GTK_POLICY_AUTOMATIC,
					GTK_POLICY_AUTOMATIC );
	gtk_scrolled_window_set_shadow_type( GTK_SCROLLED_WINDOW( window ),
						GTK_SHADOW_IN );
	private->canvas = gnome_canvas_new(); /*_aa();*/

	back.red = 0xffff;
	back.green = 0xffff;
	back.blue = 0xffff;
	gtk_widget_modify_bg( GTK_WIDGET( private->canvas ),
			      GTK_STATE_NORMAL, &back );
	gtk_widget_show( private->canvas );
	private->mainroot = 
		G_OBJECT( gnome_canvas_root( GNOME_CANVAS( private->canvas ) ) );

	gtk_widget_add_events( private->canvas, 
			       GDK_BUTTON_PRESS_MASK );

	g_signal_connect( G_OBJECT( private->canvas ), "button_press_event",
			  G_CALLBACK( background_event ), 
			  link_view );

	/* should we be checking external links? */
	private->check_external = 
		gconf_client_get_bool( private->client, 
				       "/apps/screem/linkview/check_external",
				       NULL );
	
	private->check_external_notify = 
		gconf_client_notify_add( private->client,
					 "/apps/screem/linkview/check_external",
					 screem_link_view_check_external_notify,
					 link_view, NULL, NULL );


	/* create the side panel */
	private->panel = gtk_vbox_new( FALSE, 0 );

	legend = gnome_canvas_new();
	gnome_canvas_set_center_scroll_region( GNOME_CANVAS( legend ), FALSE );
	gtk_widget_show( legend );
	legend_root = gnome_canvas_root( GNOME_CANVAS( legend ) );
	key = (GnomeCanvasItem*)screem_link_view_add_key( link_view,
							  GNOME_CANVAS( legend ), 
							  legend_root, 6.0, 0 );
	gnome_canvas_item_raise_to_top( key );
	gtk_box_pack_start( GTK_BOX( private->panel ), legend, TRUE, TRUE, 0 );

	box = gtk_hbox_new( FALSE, 6 );
	gtk_box_pack_start( GTK_BOX( private->panel ), box, TRUE, TRUE, 0 );
	label = gtk_label_new( _( "Zoom:" ) );
	gtk_widget_show( label );
	gtk_box_pack_start( GTK_BOX( box ), label, FALSE, FALSE, 0 );

	adjustment = GTK_ADJUSTMENT( gtk_adjustment_new( 100.0, 20.0, 400.0, 20.0, 10.0, 10.0 ) );
	private->zoom_control = gtk_spin_button_new( adjustment,
						     1.0, /* climb rate */
						     0 /* digits */ );
	g_signal_connect_object( G_OBJECT( private->zoom_control ),
				 "changed",
				 G_CALLBACK( screem_link_view_sidebar_zoom ),
				 G_OBJECT( link_view ), G_CONNECT_SWAPPED );
	gtk_widget_show( private->zoom_control );
	gtk_box_pack_start( GTK_BOX( box ), private->zoom_control, 
			    FALSE, FALSE, 0 );

	label = gtk_label_new( _( "Depth:" ) );
	gtk_widget_show( label );
	gtk_box_pack_start( GTK_BOX( box ), label, FALSE, FALSE, 0 );

	adjustment = GTK_ADJUSTMENT( gtk_adjustment_new( 3.0, 1.0, 10.0, 1.0, 
							10.0, 10.0 ) );
	private->depth_control = gtk_spin_button_new( adjustment,
						      1.0, /* climb rate */
						      0 /* digits */ );
	g_signal_connect_object( G_OBJECT( private->depth_control ),
				 "changed",
				 G_CALLBACK( screem_link_view_depth ),
				 G_OBJECT( link_view ), G_CONNECT_SWAPPED );
	gtk_widget_show( private->depth_control );
	gtk_box_pack_start( GTK_BOX( box ), private->depth_control, 
			    FALSE, FALSE, 0 );

	gtk_spin_button_set_numeric( GTK_SPIN_BUTTON( private->depth_control ),
					TRUE );
	gtk_spin_button_set_numeric( GTK_SPIN_BUTTON( private->zoom_control ), 
					TRUE );

	label = gtk_check_button_new_with_label( _( "Show Labels" ) );
	gtk_widget_show( label );
	gtk_box_pack_start( GTK_BOX( box ), label, FALSE, FALSE, 0 );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( label ),
					private->show_links );
	
	g_signal_connect_object( G_OBJECT( label ), "clicked", 
				 G_CALLBACK( screem_link_view_sidebar_switch ),
				 G_OBJECT( link_view ), G_CONNECT_SWAPPED );

	label = gtk_check_button_new_with_label( _( "Check Links" ) );
	gtk_widget_show( label );
	gtk_box_pack_start( GTK_BOX( box), label, FALSE, FALSE, 0 );

	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( label ),
				      private->check_external );
	g_signal_connect_object( G_OBJECT( label ), "clicked", 
				 G_CALLBACK( screem_link_view_check_toggle ),
				 G_OBJECT( link_view ), G_CONNECT_SWAPPED );

	private->check_links = label;

	/* end of side panel */

	gtk_container_add( GTK_CONTAINER( window ), private->canvas );
	
	gtk_box_pack_start( GTK_BOX( private->mainbox ), window, TRUE, TRUE, 0 );
	gtk_box_pack_start( GTK_BOX( private->mainbox ), private->panel,
				FALSE, FALSE, 0 );
	gtk_container_add( GTK_CONTAINER( link_view ), private->mainbox );

	gnome_vfs_async_set_job_limit( 5 );
}

static void screem_link_view_finalize( GObject *link_view )
{
	ScreemLinkView *view;

	view = SCREEM_LINK_VIEW( link_view );

	g_free( view->private );

	G_OBJECT_CLASS( parent_class )->finalize( G_OBJECT( link_view ) );
}

static void screem_link_view_popup_zoom_callback( EggAction *action,
						  gpointer user_data )
{
	ScreemLinkView *view;
	gint zoom;
	gchar *verb;
	
	verb = action->name;
	verb += strlen( "LinkViewZoom" );
	zoom = atoi( verb );

	view = SCREEM_LINK_VIEW( user_data );

	gtk_spin_button_set_value(GTK_SPIN_BUTTON(view->private->zoom_control),
				  (gdouble)zoom );
}

static void screem_link_view_refresh_callback( EggAction *action,
					       gpointer user_data )
{
	screem_view_display( SCREEM_VIEW( user_data ) );
}

static void screem_link_view_window_set( ScreemView *view )
{
/*	GtkWidget *label;
	GtkWidget *image;*/
	ScreemWindow *window;

	static EggActionGroupEntry entries[] = {
		{ "LinkViewRefresh", N_( "Refresh" ),
	  	GTK_STOCK_REFRESH, NULL, NULL,
	  	G_CALLBACK( screem_link_view_refresh_callback ), NULL },
		
		{ "LinkViewZoom", N_( "Zoom" ),
	  	GTK_STOCK_ZOOM_FIT, NULL, NULL,
	  	G_CALLBACK( NULL ), NULL },
		{ "LinkViewZoom25", "25%",
	  	GTK_STOCK_ZOOM_OUT, NULL, NULL,
	  	G_CALLBACK( screem_link_view_popup_zoom_callback ), NULL },
		{ "LinkViewZoom50", "50%",
	  	GTK_STOCK_ZOOM_OUT, NULL, NULL,
	  	G_CALLBACK( screem_link_view_popup_zoom_callback  ), NULL },
		{ "LinkViewZoom100", "100%",
	  	GTK_STOCK_ZOOM_100, NULL, NULL,
	  	G_CALLBACK( screem_link_view_popup_zoom_callback  ), NULL },
		{ "LinkViewZoom200", "200%",
	  	GTK_STOCK_ZOOM_IN, NULL, NULL,
	  	G_CALLBACK( screem_link_view_popup_zoom_callback  ), NULL },
		{ "LinkViewZoom300", "300%",
	  	GTK_STOCK_ZOOM_IN, NULL, NULL,
	  	G_CALLBACK( screem_link_view_popup_zoom_callback  ), NULL },	
		{ "LinkViewZoom400", "400%",
	  	GTK_STOCK_ZOOM_IN, NULL, NULL,
	  	G_CALLBACK( screem_link_view_popup_zoom_callback  ), NULL },
	};
	static 
	guint n_entries=G_N_ELEMENTS(entries);
	gint i;
	
	g_object_get( G_OBJECT( view ), "window", &window, NULL );
	
/*	label = gtk_label_new( "Legend" );
	gtk_widget_show( label );
	image = gtk_image_new_from_stock("Screem_LinkView", 
					 GTK_ICON_SIZE_SMALL_TOOLBAR );
	gtk_widget_show( image );*/
	
	for( i = 0; i < n_entries; ++ i ) {
		entries[ i ].user_data = view;
	}
	egg_action_group_add_actions( EGG_ACTION_GROUP( window->action_group ),
					entries,
					n_entries );
}

static void screem_link_view_set_prop( GObject *object, guint property_id,
				       const GValue *value, GParamSpec *pspec )
{
	ScreemLinkView *linkview;
	ScreemLinkViewPrivate *private;

	linkview = SCREEM_LINK_VIEW( object );
	private = linkview->private;

	switch( property_id ) {
	case PROP_MIN_X:
		private->min_x = g_value_get_double( value );
		break;
	case PROP_MIN_Y:
		private->min_y = g_value_get_double( value );
		break;
	case PROP_MAX_X:
		private->max_x = g_value_get_double( value );
		break;
	case PROP_MAX_Y:
		private->max_y = g_value_get_double( value );
		break;
	}
}

static void screem_link_view_get_prop( GObject *object, guint property_id,
				       GValue *value, GParamSpec *pspec)
{
	ScreemLinkView *linkview;
	ScreemLinkViewPrivate *private;

	linkview = SCREEM_LINK_VIEW( object );
	private = linkview->private;
	
	switch( property_id ) {
	case PROP_UPLOAD_TABLE:
		g_value_set_pointer( value, private->upload_table );
		break;
	case PROP_MIN_X:
		g_value_set_double( value, private->min_x );
		break;
	case PROP_MIN_Y:
		g_value_set_double( value, private->min_y );
		break;
	case PROP_MAX_X:
		g_value_set_double( value, private->max_x );
		break;
	case PROP_MAX_Y:
		g_value_set_double( value, private->max_y );
		break;
	}
}


static void screem_link_view_size_request( GtkWidget *widget, GtkRequisition *req )
{
	GTK_WIDGET_CLASS( parent_class )->size_request( widget, req );
}

static void screem_link_view_realize( GtkWidget *widget )
{
	GTK_WIDGET_CLASS( parent_class )->realize( widget );
}

static void screem_link_view_hide( GtkWidget *widget )
{
	ScreemLinkView *view;

	view = SCREEM_LINK_VIEW( widget );

	gtk_widget_hide( view->private->panel );

	GTK_WIDGET_CLASS( parent_class )->hide( widget );
}

static void screem_link_view_show( GtkWidget *widget )
{
	ScreemLinkView *view;

	view = SCREEM_LINK_VIEW( widget );

	gtk_widget_show( view->private->panel );

	GTK_WIDGET_CLASS( parent_class )->show( widget );
}

GType screem_link_view_get_type()
{
	static guint type = 0;

	if( ! type ) {
		static const GTypeInfo info = {
			sizeof( ScreemLinkViewClass ),
			NULL, /* base init */
			NULL, /* base finalise */
			(GClassInitFunc)screem_link_view_class_init,
			NULL, /* class finalise */
			NULL, /* class data */
			sizeof( ScreemLinkView ),
			0, /* n_preallocs */
			(GInstanceInitFunc)screem_link_view_init
		};
		
		type = g_type_register_static( PARENT_TYPE,
					       "ScreemLinkView",
					       &info, 0 );
	}

	return type;
}
