/*  Screem:  screem-file-browser.c
 *
 *  a file browser widget, which follows the nautilus theme
 *
 *  Copyright (C) 2002  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>

#include <glib/gutils.h>
#include <glib/ghash.h>
#include <glib/gthread.h>

#include <libgnomevfs/gnome-vfs-mime-handlers.h>
#include <libgnomevfs/gnome-vfs-mime-utils.h>
#include <libgnomevfs/gnome-vfs-directory.h>
#include <libgnomevfs/gnome-vfs-file-info.h>
#include <libgnomevfs/gnome-vfs-ops.h>
#include <libgnomevfs/gnome-vfs-utils.h>

#ifdef HAVE_ICON_THEME
#include <libgnomeui/gnome-icon-theme.h>
#include <libgnomeui/gnome-icon-lookup.h>
#endif

#include <gtk/gtktreemodel.h>
#include <gtk/gtktreestore.h>
#include <gtk/gtktreemodelsort.h>

#include <gdk-pixbuf/gdk-pixbuf.h>

#include <string.h>

#include "screem-file-browser.h"

#include "screemmarshal.h"

#include "support.h"

#define ICON_X_SIZE 16
#define ICON_Y_SIZE 16

enum {
	ADDED = 0,
	REMOVED,
	ICON_CHANGE,
	THEME_CHANGE,
	LAST_SIGNAL
};

static guint screem_file_browser_signals[ LAST_SIGNAL ] = { 0 };


#ifdef HAVE_ICON_THEME
static GnomeIconTheme *gnome_theme = NULL;
#endif

/* static icon cache */
static GHashTable *icon_cache = NULL;
/* lock access to the icon cache */
static GMutex *cache_mutex = NULL;


typedef enum {
	BROWSER_NODE_DIRECTORY,
	BROWSER_NODE_FILE
} BrowserNodeType;

typedef struct {
	BrowserNodeType type;
	ScreemFileBrowser *browser;

	gchar *pathname;
	gchar *basename;

	gchar *current; /* only set if basename is ".." */

	GdkPixbuf *close;

	gchar *mime_type;

	GnomeVFSMonitorHandle *monitor;
} BrowserNode;

struct ScreemFileBrowserDetails {
	GtkTreeStore *model;

	GtkTreeModel *sorted;

	int mode;
};

static BrowserNode *browser_node_new( ScreemFileBrowser *browser, 
				gint depth,
				BrowserNodeType type, 
				const gchar *uri,
				const gchar *mime_type );
static void browser_node_free( BrowserNode *node );
static void scan_directory( ScreemFileBrowser *browser, 
			    GtkTreeIter *parent, 
			    const gchar *uri, 
			    const gchar *mime_type,
			    guint depth, int mode );
static void directory_monitor_cb( GnomeVFSMonitorHandle *handle,
				  const gchar *monitor_uri,
				  const gchar *info_uri,
				  GnomeVFSMonitorEventType type,
				  gpointer data );
static gboolean find_iter( GtkTreeModel *model, const gchar *uri,
			   GtkTreeIter *parent, GtkTreeIter *it );

static gchar* 
screem_file_browser_build_icon_name( const gchar *iconname );

static void screem_file_browser_set_icons( ScreemFileBrowser *browser,
					     GtkTreeIter *it );


#ifdef HAVE_ICON_THEME
static void screem_file_browser_theme_change( GnomeIconTheme *theme,
						gpointer data );
#endif

static gint screem_file_browser_compare_func( GtkTreeModel *model,
						GtkTreeIter *a,
						GtkTreeIter *b,
						gpointer data );

static GList *screem_file_browser_recurse( const gchar *pathname );
static gboolean screem_file_browser_clear( GtkTreeModel *model, 
					     GtkTreePath *path,
					     GtkTreeIter *it,
					     gpointer data );

static void screem_file_browser_class_init( ScreemFileBrowserClass *klass );
static void screem_file_browser_init( ScreemFileBrowser *file_browser );
static void screem_file_browser_finalize( GObject *object );
static void screem_file_browser_set_prop( GObject *object, guint prop_id, 
					    const GValue *value, GParamSpec *spec );
static void screem_file_browser_get_prop( GObject *object, guint prop_id, 
					    GValue *value, GParamSpec *spec );


ScreemFileBrowser *screem_file_browser_new()
{
	ScreemFileBrowser *browser;

	browser = SCREEM_FILE_BROWSER( g_object_new( screem_file_browser_get_type(),
						  NULL ) );


	return browser;
}

void screem_file_browser_set_mode( ScreemFileBrowser *browser,
				   ScreemFileBrowserMode mode )
{
	g_return_if_fail( SCREEM_IS_FILE_BROWSER( browser ) );

	browser->details->mode = mode;
}

GtkTreeModel *screem_file_browser_get_model( ScreemFileBrowser *browser )
{
        g_return_val_if_fail( SCREEM_IS_FILE_BROWSER( browser ), NULL );
                                                                                
        return GTK_TREE_MODEL( browser->details->sorted );
}


void screem_file_browser_scan_directory( ScreemFileBrowser *browser,
					 const gchar *uri, int mode )
{
	GtkTreeIter *parent;
	GtkTreeIter pit;
	GtkTreeIter it;
	GtkTreeModel *model;

	g_return_if_fail( SCREEM_IS_FILE_BROWSER( browser ) );
	g_return_if_fail( uri != NULL );

	model = GTK_TREE_MODEL( browser->details->model );

	/* try and find uri in the current model, and set
	   parent to the iter's parent if found */
	parent = NULL;
	if( find_iter( model, uri, NULL, &it ) ) {
		if( gtk_tree_model_iter_parent( model, &pit, &it ) ) {
			parent = &pit;
		}
	} 

	if( mode == -1 ) {
		mode = browser->details->mode;
	}

	scan_directory( browser, parent, uri, "x-directory/normal", 
			0, mode );

	if( mode & FILE_BROWSE_ROOT ) {
		/* replace first iter name with / */
		GtkTreeIter fit;

		if( gtk_tree_model_get_iter_first( model, &fit ) ) {
			gtk_tree_store_set( GTK_TREE_STORE( model ), &fit,
					    FILE_BROWSER_NAME_COL,
					    "/", 
					    -1 );
		}
	} else if( mode & FILE_BROWSE_FLAT ) {
		/* replace first iter with .. or remove completly
		   if at the top level */
		GtkTreeIter fit;

		if( gtk_tree_model_get_iter_first( model, &fit ) ) {
			BrowserNode *node;

			gtk_tree_model_get( model, &fit,
					  FILE_BROWSER_NODE_COL,
					  &node, -1 );
			
			if( node && ! strcmp( "/", node->pathname ) ) {
				/* root node, remove it,
				   we still want to monitor / though,
				   so we can't free the node so its
				   getting leaked currently */
				gtk_tree_store_remove( GTK_TREE_STORE( model ),
						       &fit );
			} else if( node ) {
				/* replace */
				gchar *temp;

				temp = g_dirname( node->pathname );
				g_free( node->pathname );
				node->pathname = temp;
				g_free( node->basename );
				node->basename = g_strdup( ".." );

				/* add monitor for temp */

				gtk_tree_store_set( GTK_TREE_STORE( model ),
						    &fit,
						    FILE_BROWSER_NAME_COL,
						    "..",
						    FILE_BROWSER_URI_COL,
						    node->pathname,
						    -1 );
			}
		}
	}
}

const gchar* screem_file_browser_scan_iter( ScreemFileBrowser *browser,
					    GtkTreeIter *iter,
					    int mode )
{
	GtkTreeModel *model;
	BrowserNode *node;
	gchar *uri;
	GtkTreeIter realit;
	GtkTreeIter *parent = NULL;
	GtkTreeIter pit;

	const gchar *current = NULL;

	g_return_val_if_fail( SCREEM_IS_FILE_BROWSER( browser ), NULL );
	g_return_val_if_fail( iter != NULL, NULL );

	model = GTK_TREE_MODEL( browser->details->model );

	/* convert iter to a child iter */
	gtk_tree_model_sort_convert_iter_to_child_iter( GTK_TREE_MODEL_SORT( browser->details->sorted ), &realit, iter );
	iter = &realit;

	gtk_tree_model_get( model, iter, 
			FILE_BROWSER_NODE_COL, &node, -1 );

	if( ( ! node ) || node->type != BROWSER_NODE_DIRECTORY ) {
		return NULL;
	}

	uri = g_strdup( node->pathname );

	if( gtk_tree_model_iter_parent( model, &pit, iter ) ) {
		parent = &pit;
	}

	if( ( ! ( mode & FILE_BROWSE_FLAT ) ) && ! parent ) {
		return NULL;
	}

	if( mode == -1 ) {
		mode = browser->details->mode;
	}

	scan_directory( browser, iter, uri,  "x-directory/normal",
			0, mode );

	current = "/";
	if( mode & FILE_BROWSE_FLAT ) {
		/* replace first iter with .. or remove completly
		   if at the top level */
		GtkTreeIter fit;

		if( gtk_tree_model_get_iter_first( model, &fit ) ) {
			BrowserNode *node;

			gtk_tree_model_get( model, &fit,
					  FILE_BROWSER_NODE_COL,
					  &node, -1 );
			
			if( node && ! strcmp( "/", node->pathname ) ) {
				/* root node, remove it */
				gtk_tree_store_remove( GTK_TREE_STORE( model ),
						       &fit );
			} else if( node ) {
				/* replace */
				gchar *temp;

				temp = g_dirname( node->pathname );
				current = node->current = node->pathname;
				node->pathname = temp;
				g_free( node->basename );
				node->basename = g_strdup( ".." );

				gtk_tree_store_set( GTK_TREE_STORE( model ),
						    &fit,
						    FILE_BROWSER_NAME_COL,
						    "..",
						    FILE_BROWSER_URI_COL,
						    node->pathname,
						    -1 );
			}
		}
	}

	g_free( uri );

	return current;
}

GList *screem_file_browser_get_pathnames( ScreemFileBrowser *browser,
					  GList *iters, gboolean recurse )
{
	GList *ret;
	GtkTreeModel *model;

	g_return_val_if_fail( SCREEM_IS_FILE_BROWSER( browser ), NULL );
	g_return_val_if_fail( iters != NULL, NULL );

	ret = NULL;

	model = GTK_TREE_MODEL( browser->details->model );

	while( iters ) {
		GtkTreeIter *iter;
		GtkTreeIter it;
		BrowserNode *node;

		iter = (GtkTreeIter*)iters->data;

		/* ideally we would check recursion here by seeing if iter has
		   children,
		   however if the file browser is in flat mode then there will
		   not be any children present
		   so we need to check flat mode later on */
		if( recurse && gtk_tree_model_iter_has_child( browser->details->sorted,
							      iter ) ) {
			GtkTreeIter child;

			if( gtk_tree_model_iter_children( browser->details->sorted,
							  &child, iter ) ) {
				GList *tmp;

				tmp = g_list_append( NULL, &child );
				ret = g_list_concat( ret, screem_file_browser_get_pathnames( browser, tmp, TRUE ) );
				g_list_free( tmp );
			}
		}
		
		gtk_tree_model_sort_convert_iter_to_child_iter( GTK_TREE_MODEL_SORT( browser->details->sorted ), &it, iter );

		gtk_tree_model_get( model, &it,
				  FILE_BROWSER_NODE_COL, &node, -1 );
		
		if( node ) {
			ret = g_list_append( ret, node->pathname );

			if( recurse && node->type == BROWSER_NODE_DIRECTORY && 
			    ( browser->details->mode & FILE_BROWSE_FLAT ) ) {
				/* we need to scan node->pathname */
				ret = g_list_concat( ret, screem_file_browser_recurse( node->pathname ) );
			}
		} else {
			gchar *uri;
	
			uri = NULL;
			gtk_tree_model_get( model, &it,
					    FILE_BROWSER_URI_COL,
					    &uri,
					    -1 );
			if( uri ) {
				ret = g_list_append( ret, uri );
			}
		}
		
		iters = iters->next;
	}

	return ret;
}

void screem_file_browser_set_sort_func( ScreemFileBrowser *browser,
					GtkTreeIterCompareFunc func,
					gpointer data )
{
	GtkTreeModel *model;

	model = screem_file_browser_get_model( browser );

	gtk_tree_sortable_set_sort_func( GTK_TREE_SORTABLE( model ),
					 0, func, data, NULL );

	gtk_tree_sortable_set_sort_column_id( GTK_TREE_SORTABLE( model ),
					      0, GTK_SORT_ASCENDING );
}

GdkPixbuf *screem_file_browser_get_icon( ScreemFileBrowser *browser,
					 const gchar *uri,
					 gint width,
					 gint height,
					gboolean ignore_cache,
					gchar **mtype )
{
	gchar *mime_type;
	GdkPixbuf *ret;
	gchar *file = NULL;
	GnomeVFSFileInfo *info;
	gchar *tmp;

	g_return_val_if_fail( SCREEM_IS_FILE_BROWSER( browser ), NULL );
	g_return_val_if_fail( uri != NULL, NULL );

	mime_type = NULL;
	if( mtype && *mtype ) {
		mime_type = g_strdup( *mtype );
	}
	
	/* Hack to avoid file access when we are accessing
	 * directories */
	if( uri[ strlen( uri ) -1 ] == '/' ) {
		mime_type = g_strdup( "x-directory/normal" );
	} else if( mime_type == NULL ){
		info = gnome_vfs_file_info_new();

		gnome_vfs_get_file_info( uri, info, 
				 GNOME_VFS_FILE_INFO_GET_MIME_TYPE |
				 GNOME_VFS_FILE_INFO_FORCE_FAST_MIME_TYPE |
				 GNOME_VFS_FILE_INFO_FOLLOW_LINKS );

		mime_type = g_strdup( info->mime_type );
		gnome_vfs_file_info_unref( info );
	}
	if( mtype && ( *mtype == NULL ) ) {
		*mtype = g_strdup( mime_type );
	}

	file = NULL;
#ifdef HAVE_ICON_THEME
	/* figure out icon filename */
	tmp = gnome_icon_lookup( gnome_theme, 
				  NULL,
				  uri,
				  NULL,
				  NULL,
				  mime_type,
				  0, NULL );
	if( tmp ) {
		file = screem_file_browser_build_icon_name( tmp );
		g_free( tmp );
	}
#endif
	g_free( mime_type );
	g_mutex_lock( cache_mutex );
	
	ret = NULL;
	if( file ) {
		GdkPixbuf *close = NULL;
		
		if( ! ignore_cache ) {
			ret = g_hash_table_lookup( icon_cache, file );
		}
		
		if( ! ret ) {
			close = gdk_pixbuf_new_from_file( file, NULL );
		} else {
			g_object_ref( ret );
		}
		if( close ) {
			gint w;
			gint h;
			gdouble f;

			w = gdk_pixbuf_get_width( close );
			h = gdk_pixbuf_get_height( close );

			if( h > w ) {
				f = h / (gdouble)height;
			} else {
				f = w / (gdouble)width;
			}
			h /= f;
			w /= f;

			ret = gdk_pixbuf_scale_simple( close,
						       w, h,
						       GDK_INTERP_BILINEAR );
			g_object_ref( ret );
			if( ! ignore_cache ) {
				g_hash_table_insert( icon_cache, 
							g_strdup( file ),
					     		ret );
			}
			g_object_unref( close );
		}

		g_free( file );
	}

	g_mutex_unlock( cache_mutex );

	return ret;
}

/* static stuff */
static BrowserNode *browser_node_new( ScreemFileBrowser *browser, 
		gint depth, BrowserNodeType type, const gchar *uri,
		const gchar *mime_type )
{
	BrowserNode *node;
	const gchar *lookup_uri;

	node = g_new0( BrowserNode, 1 );
	node->type = type;

	node->browser = browser;

	if( mime_type ) {
		node->mime_type = g_strdup( mime_type );
	}
	
	if( uri ) {
		gchar *temp;
		
		node->pathname = g_strdup( uri );

		temp = g_path_get_basename( node->pathname );
		if( temp ) {
			node->basename = gnome_vfs_unescape_string_for_display( temp );
			g_free( temp );
		}
		
		if( ! node->basename || *node->basename == '\0' ) {
			g_free( node->basename );
			node->basename = g_strdup( G_DIR_SEPARATOR_S );
		}

		lookup_uri = uri;
		if( type == BROWSER_NODE_DIRECTORY ) {
			lookup_uri = "/";
		}
		
		node->close = screem_file_browser_get_icon( browser, 
							lookup_uri,
							ICON_X_SIZE,
							ICON_Y_SIZE,
							FALSE,
							&node->mime_type );
	}

       	if( uri && type == BROWSER_NODE_DIRECTORY &&
	    ( ( ! ( browser->details->mode & FILE_BROWSE_FLAT ) ) ||
	      depth == 0 ) ) {
		/* don't monitor /dev, /proc, or /sys */
		gchar *monuri;

		monuri = gnome_vfs_get_local_path_from_uri( uri );
		if( ! monuri ) {
			monuri = g_strdup( uri );
		}
		if( strncmp( "/dev", monuri, strlen( "/dev" ) ) &&
		    strncmp( "/sys", monuri, strlen( "/sys" ) ) &&
		    strncmp( "/proc", monuri, strlen( "/proc" ) ) ) {
			gnome_vfs_monitor_add( &node->monitor, uri,
					       GNOME_VFS_MONITOR_DIRECTORY,
					       directory_monitor_cb, 
					       browser );
		}
		g_free( monuri );
	}

	return node;
}

static void browser_node_free( BrowserNode *node )
{
	g_return_if_fail( node != NULL );

	g_free( node->pathname );
	g_free( node->current );
	if( node->close ) {
		g_object_unref( node->close );
	}
	g_free( node->mime_type );
	g_free( node->basename );

	if( node->monitor ) {
		gnome_vfs_monitor_cancel( node->monitor );
	}

	g_free( node );
}

static void scan_directory( ScreemFileBrowser *browser, 
			    GtkTreeIter *parent,
			    const gchar *uri,
			    const gchar *mime_type,
			    guint depth, int mode )
{
	GnomeVFSResult result;
	GnomeVFSDirectoryHandle *handle;
	GnomeVFSFileInfo *info;

	GtkTreeIter clear;
	GtkTreeIter *root;

	gchar *parent_uri;

	GtkTreeModel *model;

	g_return_if_fail( SCREEM_IS_FILE_BROWSER( browser ) );

	model = GTK_TREE_MODEL( browser->details->model );

	g_return_if_fail( GTK_IS_TREE_MODEL( model ) );
	g_return_if_fail( uri != NULL );

	if( mode & FILE_BROWSE_FLAT ||
	    ( mode & FILE_BROWSE_ROOT && ( depth == 0 ) ) ) {
		parent = NULL;
		if( depth == 0 ) {
			gtk_tree_model_foreach( model, 
						screem_file_browser_clear,
						browser );
			gtk_tree_store_clear( GTK_TREE_STORE( model ) );
		}
	}

	root = parent;
	parent_uri = NULL;
	if( ! parent ) {
		if( gtk_tree_model_get_iter_root( model, &clear ) ) {
			root = &clear;
		}
	} else {
		BrowserNode *node;

		gtk_tree_model_get( model, parent,
				  FILE_BROWSER_NODE_COL, &node, -1 );
		if( node ) {
			parent_uri = g_strdup( node->pathname );
		}
	}

	if( depth == 0 && root && 
		gtk_tree_model_iter_has_child( model, root ) ) {
		/* scanning at depth 0, remove current children */
		GtkTreeIter cit;
		while( gtk_tree_model_iter_children( model, &cit, root ) ) {
			BrowserNode *node;

			gtk_tree_model_get( model, &cit,
					FILE_BROWSER_NODE_COL, &node,
					-1 );
			if( node ) {
				browser_node_free( node );
			}

			gtk_tree_store_remove( GTK_TREE_STORE( model ),
					       &cit );
		}
	}

	result = GNOME_VFS_ERROR_NOT_A_DIRECTORY;
	handle = NULL;
	info = NULL;
	if( mime_type && ! strcmp( "x-directory/normal", mime_type ) ) {
		result = GNOME_VFS_OK;
		if(  (mode & FILE_BROWSE_RECURSE) || ( depth == 0 ) ) {
			info = gnome_vfs_file_info_new();
			result = gnome_vfs_directory_open( &handle, uri,
							   GNOME_VFS_FILE_INFO_FOLLOW_LINKS | GNOME_VFS_FILE_INFO_GET_MIME_TYPE | GNOME_VFS_FILE_INFO_FORCE_FAST_MIME_TYPE );
		} 
	}
	if( result == GNOME_VFS_ERROR_NOT_A_DIRECTORY ) {
		/* normal file */
		GtkTreeIter it;
		BrowserNode *node;

		gtk_tree_store_append( GTK_TREE_STORE( model ), &it, parent );

		node = browser_node_new( browser, depth,
					 BROWSER_NODE_FILE, uri,
					 mime_type );
		gtk_tree_store_set( GTK_TREE_STORE( model ), &it,
				    FILE_BROWSER_NAME_COL, node->basename,
				    FILE_BROWSER_NODE_COL, node,
				    FILE_BROWSER_ICON_COL, node->close,
				    FILE_BROWSER_URI_COL, node->pathname,
				    -1 );

		g_signal_emit( G_OBJECT( browser ),
			       screem_file_browser_signals[ ADDED ], 0,
			       node->pathname, node->mime_type, &it );
	}  else if( result == GNOME_VFS_OK ){
		GtkTreeIter it;
		BrowserNode *node;

		/* append if parent uri != uri */
		if( ! parent_uri || strcmp( uri, parent_uri ) ) {
			gtk_tree_store_append( GTK_TREE_STORE( model ), &it, 
					       parent );
			
			node = browser_node_new( browser, depth,
						 BROWSER_NODE_DIRECTORY,
						 uri, mime_type );
	
			gtk_tree_store_set( GTK_TREE_STORE( model ), &it,
					    FILE_BROWSER_NAME_COL,
					    node->basename,
					    FILE_BROWSER_NODE_COL, node, 
					    FILE_BROWSER_ICON_COL,
					    node->close,
					    FILE_BROWSER_URI_COL,
					    node->pathname,
					    -1 );
			
			g_signal_emit( G_OBJECT( browser ),
				       screem_file_browser_signals[ ADDED ],
				       0,
				       node->pathname, node->mime_type, &it );
		} else if( parent_uri ) {
			it = *root;
		}

		if( ! handle ) {
			/* just add a dummy node */
			GtkTreeIter iter;

			gtk_tree_store_append( GTK_TREE_STORE( model ),
					&iter, &it );
		} else if( ( ! ( mode & FILE_BROWSE_FLAT ) ) ||
		    ( ( mode & FILE_BROWSE_FLAT ) && depth < 1 ) ) {
			const gchar *ccat;
			guint ndepth;

			ccat = "";
			if( uri[ strlen( uri ) - 1 ] != '/' ) {
				ccat = "/";
			}
			ndepth = depth + 1;
			
			do {
				result = gnome_vfs_directory_read_next( handle,
									info );
				if( result == GNOME_VFS_OK &&
				    *info->name != '.' ) {
					gchar *utfuri;
					gchar *nexturi;
					
					utfuri = screem_support_charset_convert( info->name );
			
					/* yuk */
					if( ! utfuri ) {
						continue;
					}
					
					nexturi = g_strconcat( uri,
							ccat,
							utfuri,
							NULL );
					g_free( utfuri );
					
					scan_directory( browser, &it,
							nexturi, 
							info->mime_type,
							ndepth, mode );
					
					g_free( nexturi );
				}
			} while( result == GNOME_VFS_OK ); 
		}
		if( handle ) {
			gnome_vfs_directory_close( handle );
			gnome_vfs_file_info_unref( info );
		}
	}
	g_free( parent_uri );

}


static void directory_monitor_cb( GnomeVFSMonitorHandle *handle,
				  const gchar *monitor_uri,
				  const gchar *info_uri,
				  GnomeVFSMonitorEventType type,
				  gpointer data )
{
	ScreemFileBrowser *browser;
	GtkTreeIter pit;
	GtkTreeIter it;
	GtkTreeIter iter;
	GtkTreeIter *parent;
	GtkTreePath *path;
	GtkTreeModel *model;
	GnomeVFSFileInfo *info;
	BrowserNodeType btype;
	gchar *base;
	gboolean hidden;
	int mode;
	BrowserNode *node;
	
	if( ! data ) {
		return;
	}

	browser = SCREEM_FILE_BROWSER( data );
	mode = browser->details->mode;
	
	base = g_path_get_basename( info_uri );
	hidden = FALSE;
	if( base ) {
		hidden = ( *base == '.' );
		g_free( base );
	}
	
	switch( type ) {
	case GNOME_VFS_MONITOR_EVENT_CHANGED:
		break;
	case GNOME_VFS_MONITOR_EVENT_DELETED:
		model = GTK_TREE_MODEL( browser->details->model );
		
		if( find_iter( model, info_uri, NULL, &it ) ) {
			gtk_tree_model_get( model, &it,
				FILE_BROWSER_NODE_COL, &node,
				-1 );

			g_signal_emit( G_OBJECT( browser ),
				       screem_file_browser_signals[ REMOVED ],
				       0, info_uri, &it );
		
			gtk_tree_store_remove( GTK_TREE_STORE( model ),
					       &it );
			browser_node_free( node );
		}
		break;
	case GNOME_VFS_MONITOR_EVENT_STARTEXECUTING:
		break;
	case GNOME_VFS_MONITOR_EVENT_STOPEXECUTING:
		break;
	case GNOME_VFS_MONITOR_EVENT_CREATED:
		model = GTK_TREE_MODEL( browser->details->model );
		parent = NULL;
		if( ( ! ( mode & FILE_BROWSE_FLAT ) ) &&
			find_iter( model, monitor_uri, NULL, &pit ) ) {
			/* got the parent */
			parent = &pit;
		}
			
		/* is info_uri a directory? */
		info = gnome_vfs_file_info_new();
		gnome_vfs_get_file_info( info_uri, info, 
				GNOME_VFS_FILE_INFO_GET_MIME_TYPE | 
				GNOME_VFS_FILE_INFO_FORCE_FAST_MIME_TYPE );
		if( info->type == GNOME_VFS_FILE_TYPE_DIRECTORY ) {
			btype = BROWSER_NODE_DIRECTORY;
		} else {
			btype = BROWSER_NODE_FILE;
		}
	
		if( btype == BROWSER_NODE_DIRECTORY &&
		    ( ! (mode & FILE_BROWSE_FLAT ) ) && ! hidden ) {
			if( ! (mode & FILE_BROWSE_RECURSE) ) {
				gtk_tree_store_append( GTK_TREE_STORE( model ), &iter, parent );
	
			} else {
				/* scan at depth 1 to
				 * avoid clearing */
				scan_directory( browser, 
						parent, 
						info_uri, 
						"x-directory/normal", 
							1, mode );
			}
		} else if( ! hidden ) {
			gtk_tree_store_append( GTK_TREE_STORE( model ),
					       &it, parent );
			
			/* we set a depth of 1 so that in flat mode we 
			 * don't try and monitor info_uri */
			node = browser_node_new( browser, 1, btype, 
					info_uri, info->mime_type );
			gtk_tree_store_set(GTK_TREE_STORE( model ),
					   &it,
					   FILE_BROWSER_NAME_COL,
					   node->basename,
					   FILE_BROWSER_NODE_COL, node,
					   FILE_BROWSER_ICON_COL,
				 	   node->close,
				 	   FILE_BROWSER_URI_COL,
				 	   node->pathname,
					   -1 );

			g_signal_emit( G_OBJECT( browser ),
				       screem_file_browser_signals[ ADDED ],
				       0,
				       info_uri, node->mime_type, &it );
		} else {
			g_signal_emit( G_OBJECT( browser ),
					screem_file_browser_signals[ ADDED ],
					0,
					info_uri, NULL, NULL );
		}
			
		gnome_vfs_file_info_unref( info );
		break;
	case GNOME_VFS_MONITOR_EVENT_METADATA_CHANGED:
		break;
	default:
		break;
	}
}

static gboolean find_iter( GtkTreeModel *model, const gchar *uri,
			   GtkTreeIter *parent, GtkTreeIter *it )
{
	gboolean got;
	gboolean found = FALSE;
	GnomeVFSURI *a;

	g_return_val_if_fail( GTK_IS_TREE_MODEL( model ), FALSE );
	g_return_val_if_fail( uri != NULL, FALSE );
	g_return_val_if_fail( it != NULL, FALSE );

	if( ! parent ) {
		got = gtk_tree_model_get_iter_first( model, it );
	} else {
		got = gtk_tree_model_iter_children( model, it, parent );
	}

	a = gnome_vfs_uri_new( uri );

	while( got && ! found ) {
		BrowserNode *node;
		GnomeVFSURI *b;

		gtk_tree_model_get( model, it, FILE_BROWSER_NODE_COL, 
				  &node, -1 );

		if( node && node->pathname ) {
			b = gnome_vfs_uri_new( node->pathname );
			found = gnome_vfs_uri_equal( a, b );
			gnome_vfs_uri_unref( b );
		}
		if( ! found ) {
			GtkTreeIter cit;
			if( gtk_tree_model_iter_has_child( model, it ) ) {
				if( find_iter( model, uri, it, &cit ) ) {
					/* found it */
					*it = cit;
					found = TRUE;
				}
			}
		}
		if( ! found ) {
			got = gtk_tree_model_iter_next( model, it );
		}
	}

	gnome_vfs_uri_unref( a );
	
	return found;
}

static gchar* 
screem_file_browser_build_icon_name( const gchar *iconname )
{
	gchar *name;


#ifdef HAVE_ICON_THEME

      	if( iconname[ 0 ] != '/' || ! g_file_test( iconname,
						   G_FILE_TEST_EXISTS ) ) {
	      	name = gnome_icon_theme_lookup_icon( gnome_theme, iconname,
						     24, NULL, NULL );
	} else {
		name = g_strdup( iconname );
	}
#else
	name = g_strdup( iconname );
#endif
	return name;
}

static void screem_file_browser_set_icons( ScreemFileBrowser *browser,
					     GtkTreeIter *it )
{
	GtkTreeModel *model;
	GtkTreeIter rit;

	g_return_if_fail( SCREEM_IS_FILE_BROWSER( browser ) );
     
	model = GTK_TREE_MODEL( browser->details->model );

	if( ! it ) {
		if( gtk_tree_model_get_iter_first( model, &rit ) ) {
			it = &rit;
		}
	}
	if( it ) {
		BrowserNode *node;
		GtkTreeIter cit;
		gchar *uri;
		GdkPixbuf *pixbuf;

		const gchar *pathname;
		const gchar *mimetype;
		
		gtk_tree_model_get( model, it,
				FILE_BROWSER_NODE_COL, &node,
				FILE_BROWSER_URI_COL, &uri,
				-1 );
		
		pathname = mimetype = NULL;
		if( node ) {
			pathname = node->pathname;
			mimetype = node->mime_type;
		}
		
		if( uri ) {
			if( node && node->close ) {
				g_object_unref( node->close );
			} 
			pixbuf = screem_file_browser_get_icon( browser,
								 uri,
								 ICON_X_SIZE,
								 ICON_Y_SIZE,
								FALSE,
								NULL );
			
			if( node ) {
				node->close = pixbuf;
			}

			gtk_tree_store_set( GTK_TREE_STORE( model ), it,
					    FILE_BROWSER_ICON_COL, 
					    pixbuf, -1 );
		}		
		g_signal_emit( G_OBJECT( browser ),
			       screem_file_browser_signals[ ICON_CHANGE ], 0,
			       pathname, mimetype, it );

		if( gtk_tree_model_iter_has_child( model, it ) ) {
			gtk_tree_model_iter_children( model, &cit, it);
			screem_file_browser_set_icons( browser, 
						       &cit );
		}
		if( gtk_tree_model_iter_next( model, it ) ) {
			screem_file_browser_set_icons( browser, it );
		}
		g_free( uri );
	}
}


#ifdef HAVE_ICON_THEME
static void screem_file_browser_theme_change( GnomeIconTheme *theme,
						gpointer data )
{
	ScreemFileBrowser *browser;

	browser = SCREEM_FILE_BROWSER( data );

	screem_file_browser_set_icons( browser, NULL );

	g_signal_emit( G_OBJECT( browser ),
		       screem_file_browser_signals[ THEME_CHANGE ], 0 );
}
#endif

static gint screem_file_browser_compare_func( GtkTreeModel *model,
					      GtkTreeIter *a,
					      GtkTreeIter *b,
					      gpointer data )
{
        BrowserNode *anode;
        BrowserNode *bnode;

        int ret = 0;

        gtk_tree_model_get( model, a, 
			FILE_BROWSER_NODE_COL, &anode, -1 );
        gtk_tree_model_get( model, b, 
			FILE_BROWSER_NODE_COL, &bnode, -1 );

	if( anode == bnode ) {
		return 0;
	} else if( anode && ! bnode ) {
		ret = 1;
	} else if( bnode && ! anode ) {
		ret = -1;
	} else if( ! anode && ! bnode ) {
		ret = 0;	
	} else	if( anode->type == BROWSER_NODE_DIRECTORY &&
		    bnode->type != BROWSER_NODE_DIRECTORY ) {
                ret = -1;
        } else if( bnode->type == BROWSER_NODE_DIRECTORY &&
		   anode->type != BROWSER_NODE_DIRECTORY ) {
                ret = 1;
	} else {
                ret = strcmp( anode->pathname, bnode->pathname );
	}

	if( ( anode && anode->basename && ! strcmp( "..", anode->basename ) ) ||
	    ( bnode &&  bnode->basename && ! strcmp( "..", bnode->basename ) ) ) {
		ret = 1;
	}

        return ret;
}

static GList *screem_file_browser_recurse( const gchar *pathname )
{
	GList *ret;
	GnomeVFSDirectoryHandle *handle;
	GnomeVFSResult result;

	ret = NULL;

	result = gnome_vfs_directory_open( &handle, pathname,
					   GNOME_VFS_FILE_INFO_FOLLOW_LINKS );

	if( result == GNOME_VFS_OK ) {
		GnomeVFSFileInfo *info;
		
		info = gnome_vfs_file_info_new();

		do {
			result = gnome_vfs_directory_read_next( handle, info );
			if( result == GNOME_VFS_OK && *info->name != '.' ) {
				gchar *temp;

				temp = g_strconcat( pathname, "/", info->name, NULL );

				ret = g_list_append( ret, temp );
				ret = g_list_concat( ret, screem_file_browser_recurse( temp ) );
			}
		} while( result == GNOME_VFS_OK );
		
		gnome_vfs_file_info_unref( info );
		
		gnome_vfs_directory_close( handle );
	}
	
	return ret;
}

static gboolean screem_file_browser_clear( GtkTreeModel *model, 
					   GtkTreePath *path,
					   GtkTreeIter *it,
					   gpointer data )
{
        BrowserNode *node;
	
        gtk_tree_model_get( model, it, 
				FILE_BROWSER_NODE_COL, &node, -1 );
        
	if( node ) {
		browser_node_free( node );
	}

	return FALSE;
}

/* G Object stuff */

#define PARENT_TYPE G_TYPE_OBJECT

static gpointer parent_class;

static void
screem_file_browser_class_init( ScreemFileBrowserClass *klass)
{
        GObjectClass *object_class = G_OBJECT_CLASS( klass );

	parent_class = g_type_class_peek_parent( klass );

	object_class->finalize = screem_file_browser_finalize;
	object_class->get_property = screem_file_browser_get_prop;
	object_class->set_property = screem_file_browser_set_prop;

	screem_file_browser_signals[ REMOVED ] = 
		g_signal_new( "removed",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( ScreemFileBrowserClass, removed ),
			      NULL, NULL,
			      screem_marshal_VOID__STRING_POINTER,
			      G_TYPE_NONE, 2,
			      G_TYPE_STRING,
			      G_TYPE_POINTER );
	screem_file_browser_signals[ ADDED ] = 
		g_signal_new( "added",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( ScreemFileBrowserClass, 
					       added ),
			      NULL, NULL,
			      screem_marshal_VOID__STRING_STRING_POINTER,
			      G_TYPE_NONE, 3,
			      G_TYPE_STRING,
			      G_TYPE_STRING,
			      G_TYPE_POINTER );
	screem_file_browser_signals[ ICON_CHANGE ] = 
		g_signal_new( "icon_change",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( ScreemFileBrowserClass, 
					       icon_change ),
			      NULL, NULL,
			      screem_marshal_VOID__STRING_STRING_POINTER,
			      G_TYPE_NONE, 3,
			      G_TYPE_STRING,
			      G_TYPE_STRING,
			      G_TYPE_POINTER );
	screem_file_browser_signals[ THEME_CHANGE ] = 
		g_signal_new( "theme_change",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( ScreemFileBrowserClass, 
					       theme_change ),
			      NULL, NULL,
			      screem_marshal_VOID__VOID,
			      G_TYPE_NONE, 0  );
}

static void
screem_file_browser_init( ScreemFileBrowser *browser )
{
	if( ! cache_mutex ) {
		cache_mutex = g_mutex_new();
	}
	if( ! icon_cache ) {
		icon_cache = g_hash_table_new( g_str_hash, g_str_equal );
	}

	browser->details = g_new0( ScreemFileBrowserDetails, 1 );

	browser->details->model = gtk_tree_store_new( FILE_BROWSER_MAX_COL,
						      G_TYPE_STRING,
						      GDK_TYPE_PIXBUF,
						      G_TYPE_POINTER,
						      G_TYPE_STRING,
						      G_TYPE_POINTER,
						      G_TYPE_STRING );

	browser->details->sorted = 
		GTK_TREE_MODEL( gtk_tree_model_sort_new_with_model( GTK_TREE_MODEL( browser->details->model ) ) );

	screem_file_browser_set_sort_func( browser, 
					   screem_file_browser_compare_func,
					   browser );


#ifdef HAVE_ICON_THEME
	if( ! gnome_theme ) {
		gnome_theme = gnome_icon_theme_new();
		gnome_icon_theme_set_allow_svg( gnome_theme, TRUE );
	}
	g_signal_connect( G_OBJECT( gnome_theme ),
			  "changed",
			  G_CALLBACK( screem_file_browser_theme_change ),
			  browser );
#endif
}

static void
screem_file_browser_set_prop( GObject *object, guint prop_id, 
				const GValue *value, GParamSpec *spec )
{
	ScreemFileBrowser *browser;

	browser = SCREEM_FILE_BROWSER( object );

	switch( prop_id ) {
	default:
		break;
	}
}

static void
screem_file_browser_get_prop( GObject *object, guint prop_id, 
			 GValue *value, GParamSpec *spec )
{
	ScreemFileBrowser *browser;

	browser = SCREEM_FILE_BROWSER( object );

	switch( prop_id ) {
	default:
		break;
	}
}

static void
screem_file_browser_finalize( GObject *object )
{
	ScreemFileBrowser *browser;

	browser = SCREEM_FILE_BROWSER( object );

#ifdef HAVE_ICON_THEME
	g_signal_handlers_disconnect_matched( G_OBJECT( gnome_theme ),
					      G_SIGNAL_MATCH_DATA,
					      0, 0, NULL,
					      NULL,
					      browser );
#endif

	g_object_unref( browser->details->sorted );

	gtk_tree_model_foreach( GTK_TREE_MODEL( browser->details->model ), 
				screem_file_browser_clear,
				browser );
	gtk_tree_store_clear( GTK_TREE_STORE( browser->details->model ) );

	g_object_unref( browser->details->model );

	g_free( browser->details );

	G_OBJECT_CLASS( parent_class )->finalize( object );
}

GType screem_file_browser_get_type()
{
	static GType type = 0;

	if( ! type ) {
		static const GTypeInfo info = {
			sizeof( ScreemFileBrowserClass ),
			NULL, /* base init */
			NULL, /* base finalise */
			(GClassInitFunc)screem_file_browser_class_init,
			NULL, /* class finalise */
			NULL, /* class data */
			sizeof( ScreemFileBrowser ),
			0, /* n_preallocs */
			(GInstanceInitFunc)screem_file_browser_init
		};

		type = g_type_register_static( PARENT_TYPE,
					       "ScreemFileBrowser",
					       &info, 0 );
	}

	return type;
}
