/*  Screem:  screem-cvs.c
 *
 *  Copyright (C) 2002 David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <glib/gutils.h>
#include <stdio.h>

#include <string.h>

#include <unistd.h>

#include <libgnomevfs/gnome-vfs-uri.h>

#include "screem-helper.h"
#include "screem-cvs.h"

static void screem_cvs_class_init( ScreemCVSClass *klass );
static void screem_cvs_init( ScreemCVS *cvs );
static void screem_cvs_finalize( GObject *object );

static void screem_cvs_set_property( GObject *object, guint prop_id,
				     const GValue *value, GParamSpec *pspec );
static void screem_cvs_get_property( GObject *object, guint prop_id,
				     GValue *value, GParamSpec *pspec );


static void screem_cvs_set_root( ScreemCVS *cvs, const gchar *root );
static void screem_cvs_set_base_path( ScreemCVS *cvs, const gchar *base );

static gchar *screem_cvs_get_root( ScreemCVS *cvs );

static void screem_cvs_execute( ScreemCVS *cvs, const gchar *command );

static gchar *screem_cvs_usable_pathname( const gchar *pathname );

enum {
	PROP_BOGUS,
	PROP_ROOT,
	PROP_BASE
};

struct ScreemCVSPrivate {
	ScreemWindow *window;

	gchar *root;
	gchar *base;
	gint len;
};

ScreemCVS* screem_cvs_new( ScreemWindow *window )
{
	ScreemCVS *cvs;
	GType type;
	
	type = screem_cvs_get_type();

	cvs = SCREEM_CVS( g_object_new( type,
					"input", SCREEM_HELPER_STDIN_NONE,
					"output", SCREEM_HELPER_STDOUT_NONE,
					NULL ) );
	cvs->private->window = window;

	return cvs;
}

void screem_cvs_commit( ScreemCVS *cvs, const gchar *pathname,
			const gchar *extra_flags, const gchar *log_text )
{
	gchar *root;
	gchar *command;

	root = screem_cvs_get_root( cvs );

	if( pathname ) {
		gchar *path;

		path = screem_cvs_usable_pathname( pathname );
		command = g_strconcat( "cvs", root, " commit ", 
				       extra_flags, " -m \"", 
				       log_text, "\" \"", 
				       path + cvs->private->len, "\"",
				       NULL );
		g_free( path );
	} else {
		command = g_strconcat( "cvs", root, " commit ", 
				       extra_flags, " -m \"", 
				       log_text, "\"",
				       NULL );
	}

	screem_cvs_execute( cvs, command );
	g_free( command );
	g_free( root );
}

void screem_cvs_update( ScreemCVS *cvs, const gchar *pathname,
			const gchar *extra_flags )
{
	gchar *root;
	gchar *command;

	root = screem_cvs_get_root( cvs );

	if( pathname ) {
		gchar *path;

		path = screem_cvs_usable_pathname( pathname );
		command = g_strconcat( "cvs", root, " update ",
				       extra_flags, " \"", 
				       path + cvs->private->len, "\"",
				       NULL );
		g_free( path );
	} else {
		command = g_strconcat( "cvs", root, " update ",
				       extra_flags, NULL );
	}
	screem_cvs_execute( cvs, command );
	g_free( command );
	g_free( root );
}

void screem_cvs_delete( ScreemCVS *cvs, const gchar *pathname,
			const gchar *extra_flags )
{
	gchar *root;
	gchar *command;
	gchar *path;

	root = screem_cvs_get_root( cvs );

	path = screem_cvs_usable_pathname( pathname );
	command = g_strconcat( "cvs", root, " delete ",
			       extra_flags, " \"", 
			       path + cvs->private->len, "\"",
			       NULL );
	g_free( path );
	screem_cvs_execute( cvs, command );
	g_free( command );
	g_free( root );
}

void screem_cvs_add( ScreemCVS *cvs, const gchar *pathname,
		     const gchar *extra_flags, const gchar *log_text )
{
	gchar *root;
	gchar *command;
	gchar *path;

	root = screem_cvs_get_root( cvs );

	path = screem_cvs_usable_pathname( pathname );
	command = g_strconcat( "cvs", root, " add ",
			       extra_flags, " -m \"", log_text,
			       "\" \"", 
			       path + cvs->private->len, "\"",
			       NULL );
	g_free( path );
	screem_cvs_execute( cvs, command );
	g_free( command );
	g_free( root );
}

void screem_cvs_checkout( ScreemCVS *cvs,
			  const gchar *extra_flags, const gchar *module )
{
	gchar *root;
	gchar *command;

	root = screem_cvs_get_root( cvs );

	command = g_strconcat( "cvs", root, " checkout ",
			       extra_flags, " \"", module, "\"",
			       NULL );
	screem_cvs_execute( cvs, command );
	g_free( command );
	g_free( root );
}

void screem_cvs_import( ScreemCVS *cvs,
			const gchar *extra_flags, const gchar *log_text,
			const gchar *module )
{
	gchar *root;
	gchar *command;

	root = screem_cvs_get_root( cvs );

	command = g_strconcat( "cvs", root, " import ",
			       extra_flags, " -m \"", log_text, 
			       "\" \"", module, "\" screem start",
			       NULL );
	screem_cvs_execute( cvs, command );
	
	g_free( command );
	g_free( root );
}

/* static stuff */
static void screem_cvs_set_root( ScreemCVS *cvs, const gchar *root )
{
	if( cvs->private->root ) {
		g_free( cvs->private->root );
	}

	if( root ) {
		cvs->private->root = g_strdup( root );
	} else {
		cvs->private->root = NULL;
	}
}

static void screem_cvs_set_base_path( ScreemCVS *cvs, const gchar *base )
{
	g_assert( base );

	if( cvs->private->base ) {
		g_free( cvs->private->base );
		cvs->private->base = NULL;
	}

	cvs->private->base = screem_cvs_usable_pathname( base );
	cvs->private->len = strlen( cvs->private->base );
}

static gchar *screem_cvs_get_root( ScreemCVS *cvs )
{
	gchar *root;

	if( ! cvs->private->root ) {
		screem_cvs_set_root( cvs, g_getenv( "CVSROOT" ) );
	}

	if( cvs->private->root ) {
		root = g_strconcat( " -d ", cvs->private->root, NULL );
	} else {
		root = g_strdup( "" );
	}

	return root;
}


static void screem_cvs_execute( ScreemCVS *cvs, const gchar *command )
{
	gchar *cwd;

	cwd = g_get_current_dir();
	if( ! cwd ) {
		cwd = g_strdup( "" );
	}

	g_assert( cvs->private->base != NULL );
	chdir( cvs->private->base );

	g_object_set( G_OBJECT( cvs ), "pathname", command, NULL );

	screem_helper_execute( SCREEM_HELPER( cvs ), cvs->private->window );

	if( cwd[ 0 ] != '\0' ) {
		chdir( cwd );
	}
	g_free( cwd );
}

static gchar *screem_cvs_usable_pathname( const gchar *pathname )
{
	GnomeVFSURI *uri;
	gchar *usable;

	uri = gnome_vfs_uri_new( pathname );
	usable = NULL;
	if( uri ) {
		usable = gnome_vfs_uri_to_string( uri,
						  GNOME_VFS_URI_HIDE_USER_NAME|
						  GNOME_VFS_URI_HIDE_PASSWORD |
						  GNOME_VFS_URI_HIDE_HOST_NAME|
						  GNOME_VFS_URI_HIDE_HOST_PORT|
						  GNOME_VFS_URI_HIDE_TOPLEVEL_METHOD |
						  GNOME_VFS_URI_HIDE_FRAGMENT_IDENTIFIER );
		gnome_vfs_uri_unref( uri );
	}

	return usable;
}

/* G Object stuff */

#define PARENT_TYPE SCREEM_TYPE_HELPER

static gpointer parent_class;
static void screem_cvs_class_init( ScreemCVSClass *klass )
{
	GObjectClass *object_class;

	object_class = G_OBJECT_CLASS( klass );
	parent_class = g_type_class_peek_parent( klass );

	object_class->finalize = screem_cvs_finalize;

	object_class->set_property = screem_cvs_set_property;
	object_class->get_property = screem_cvs_get_property;

	g_object_class_install_property( object_class,
					 PROP_ROOT,
					 g_param_spec_string("root",
							     "CVSROOT",
							     "The cvs repository to use",
							     "",
							     G_PARAM_READWRITE)
					 );
	g_object_class_install_property( object_class,
					 PROP_BASE,
					 g_param_spec_string("base",
							     "Base Path",
							     "The base path of the local copy to operate on",
							     "",
							     G_PARAM_READWRITE)
					 );
}

static void screem_cvs_init( ScreemCVS *cvs )
{
	cvs->private = g_new0( ScreemCVSPrivate, 1 );
}

static void screem_cvs_finalize( GObject *object )
{
	ScreemCVS *cvs;

	cvs = SCREEM_CVS( object );

	if( cvs->private->root )
		g_free( cvs->private->root );

	if( cvs->private->base )
		g_free( cvs->private->base );

	g_free( cvs->private );

	G_OBJECT_CLASS( parent_class )->finalize( object );
}

static void screem_cvs_set_property( GObject *object, guint prop_id,
				     const GValue *value, GParamSpec *pspec )
{
	ScreemCVS *cvs;
	ScreemCVSPrivate *private;
	const gchar *val;

	cvs = SCREEM_CVS( object );
	private = cvs->private;

	switch( prop_id ) {
	case PROP_ROOT:
		val = g_value_get_string( value );
		screem_cvs_set_root( cvs, val );
		break;
	case PROP_BASE:
		val = g_value_get_string( value );
		screem_cvs_set_base_path( cvs, val );
		break;
	default:
		break;
	}
}

static void screem_cvs_get_property( GObject *object, guint prop_id,
				     GValue *value, GParamSpec *pspec )
{
	ScreemCVS *cvs;
	ScreemCVSPrivate *private;
	const gchar *val;

	cvs = SCREEM_CVS( object );
	private = cvs->private;

	switch( prop_id ) {
	case PROP_ROOT:
		val = screem_cvs_get_root( cvs );
		if( ! val ) {
			val = "";
		}
		g_value_set_string( value, val );
		break;
	case PROP_BASE:
		val = private->base;
		if( ! val ) {
			val = "";
		} 
		g_value_set_string( value, val );
		break;
	default:
		break;
	}
}

GType screem_cvs_get_type()
{
	static GType type = 0;
	
	if( ! type ) {
		static const GTypeInfo info = {
			sizeof( ScreemCVSClass ),
			NULL, /* base init */
			NULL, /* base finalise */
			(GClassInitFunc)screem_cvs_class_init,
			NULL, /* class finalise */
			NULL, /* class data */
			sizeof( ScreemCVS ),
			0, /* n_preallocs */
			(GInstanceInitFunc)screem_cvs_init
		};

		type = g_type_register_static( PARENT_TYPE,
					       "ScreemCVS",
					       &info, 0 );
	}

	return type;
}
