/*  Screem:  pageUI.c,
 * 
 *  ScreemPage utility functions for gui stuff
 * 
 *  Copyright (C) 1999 - 2002  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>

#include <libgnome/gnome-i18n.h>
#include <libgnome/gnome-url.h>
#include <libgnomeui/gnome-messagebox.h>
#include <libgnomeui/gnome-stock-icons.h>
#include <libgnomeui/gnome-entry.h>

#include <libgnomevfs/gnome-vfs-uri.h>
#include <libgnomevfs/gnome-vfs-ops.h>
#include <libgnomevfs/gnome-vfs-file-info.h>
#include <libgnomevfs/gnome-vfs-utils.h>

#include <bonobo/bonobo-file-selector-util.h>

#include <gconf/gconf-client.h>

#include <errno.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>

#include <glade/glade.h>

#include <gtk/gtkinvisible.h>
#include <gtk/gtkdialog.h>
#include <gtk/gtkentry.h>
#include <gtk/gtkliststore.h>
#include <gtk/gtktreeview.h>
#include <gtk/gtktreemodel.h>
#include <gtk/gtkcellrenderer.h>
#include <gtk/gtkcellrenderertext.h>

#include "fileops.h"
#include "pageUI.h"
#include "support.h"

#include "screem-window.h"
#include "screem-site.h"
#include "screem-site-ui.h"
#include "screem-editor.h"
#include "screem-preview.h"
#include "screem-page.h"
#include "screem-search.h"

gboolean screem_page_open_with_filename( ScreemSite *site, 
					 ScreemWindow *window,
					 const gchar *filename )
{
	ScreemApplication *app;
	ScreemPage *page;
	gchar *path;
	GnomeVFSURI *uri;

	g_return_val_if_fail( filename != NULL, FALSE );

	uri = gnome_vfs_uri_new( filename );

	if( ! uri ) {
		return FALSE;
	}
	
	g_object_get( G_OBJECT( site ), "app", &app, NULL );
	
	/* is the filename local? */
	if( gnome_vfs_uri_is_local( uri ) ) {
		/* yes */
		/* HACK */
		if( *filename == G_DIR_SEPARATOR ) {
			path = g_strconcat( "file://", filename, NULL );
		} else {
			path = relative_to_full( filename, NULL );
		}
	} else {
		path = g_strdup( filename );
	}
	gnome_vfs_uri_unref( uri );
	

	page = screem_site_locate_page( site, path );

	if( screem_site_get_fake_flag( site ) && ! page ) {
		/* fake site, add to list of pages */
		page = screem_page_new( G_OBJECT( app ) );
		screem_page_set_pathname( page, path );
		
		if( ! screem_site_add_page( site, page ) ) {
			/* page is already loaded, FIXME: what now? */
		}
	}

	g_free( path );

	if( page ) {
		if( screem_window_get_current( window ) != site ) {
			screem_window_set_current( window, site );
		}
		screem_window_set_document( window, page );
	}

	g_object_unref( app );
	
	return ( page != NULL );
}


gint screem_page_save_as( ScreemPage *page, ScreemWindow *window )
{
	ScreemApplication *app;
	gchar *filename;
	gint status;
	ScreemSite *site;
	const gchar *pathname;
	gchar *spath;

	g_object_get( G_OBJECT( window ), "app", &app, NULL );
	
	site = screem_window_get_current( window );
	pathname = screem_site_get_pathname( site );
	spath = NULL;
	if( ! screem_site_get_fake_flag( site ) ) {
		GnomeVFSURI *uri;
		pathname = screem_site_get_pathname( site );
		uri = gnome_vfs_uri_new( pathname );
		if( ! uri ) {
			spath = NULL;
		} else if( gnome_vfs_uri_is_local( uri ) ) {
			spath = gnome_vfs_get_local_path_from_uri( pathname );
			gnome_vfs_uri_unref( uri );
		} else {
			spath = gnome_vfs_uri_to_string( uri, 0 );
			gnome_vfs_uri_unref( uri );
		}
	}

	filename = 
		bonobo_file_selector_save( GTK_WINDOW( window ), TRUE,
					   _( "Select filename to save as" ),
					   NULL, spath, NULL );

	/* we didn't select one */
	status = SAVE_AS_CANCELLED;
	if( filename ) {
		if( ! overwrite( filename ) ) {
			g_free( filename );
		} else {
			gchar *previous;
			gboolean hold;
			
			previous = (gchar*)screem_page_get_pathname( page );
			if( previous ) {
				previous = g_strdup( previous );
			}

			/* FIXME: if we are overwriting an already existing file
			 *
			 * 1) scan all windows, closing filename if it is open
			 * 2) remove filename from site
			 */
			
			screem_page_set_pathname( page, filename );
			
			/* allow save by setting page as changed */
			screem_page_set_changed( page, TRUE );
			status = SAVE_AS_FAILED;
			hold = screem_page_save( page );

			/* Hack so anything listening to notify::pathname
			 * will be able to get the mime type correctly
			 * as the file may not have existed before
			 * screem_page_save() was called */
			if( hold ) {
				status = SAVE_AS_OK;
				screem_page_set_pathname( page, filename );
			}

			/* if we aren't the fake site we now have a file
			 * for which there is no ScreemPage in the site,
			 * we need to add one */
			if( previous && ! screem_site_get_fake_flag( site ) &&
				! screem_site_locate_page( site, previous ) ) {
				ScreemPage *npage;

				npage = screem_page_new( G_OBJECT( app ) );
				screem_page_set_pathname( npage, previous );
				screem_site_add_page( site, npage );
			}
			g_free( previous );
		}
	}

	if( spath ) {
		g_free( spath );
	}

	g_object_unref( app );
	
	return status;
}

gint screem_page_save_confirmation( ScreemPage *page )
{
	GConfClient *client;
	gchar *gladepath;
	GladeXML *xml;
	GtkWidget *widget;
	const gchar *label;
	GString *str;
	const gchar *temp;
	gchar *tmp;
	guint pos;
	
	GtkWidget *box;
        gint button;
	const gchar *pathname;
	gchar *basename;
	
	if( ! screem_page_is_loaded( page ) ) {
		return GTK_RESPONSE_NO;
	} else if( ! screem_page_get_changed( page ) ) {
		return GTK_RESPONSE_NO;
	}

	client = gconf_client_get_default();
	gladepath = gconf_client_get_string( client, "/apps/screem/general/glade_path",
					     NULL );
	xml = glade_xml_new( gladepath, "save", NULL );
	g_free( gladepath );
	g_object_unref( client );
	
	widget = glade_xml_get_widget( xml, "text" );
	label = gtk_label_get_text( GTK_LABEL( widget ) );
	temp = strstr( label, "%s" );
	g_assert( temp );
	pos = (guint)(temp - label);
	str = g_string_new( "<span weight=\"bold\" size=\"larger\">" );
	pos += str->len;
	g_string_append( str, label );
	
	pathname = screem_page_get_pathname( page );

	if( ! pathname ) {
		pathname = _( "Untitled" );
	}
	basename = g_path_get_basename( pathname );
	temp = gnome_vfs_unescape_string_for_display( basename );
	g_free( basename );
	tmp = g_markup_escape_text( temp, -1 );
	g_free( (gchar*)temp );
	basename = (gchar*)tmp;

	g_string_erase( str, pos, 2 );
	g_string_insert( str, pos, basename );
	g_free( basename );
	
	temp = strstr( str->str, "\n\n" );
	g_assert( temp );
	pos = (guint)(temp - str->str);
	g_string_insert( str, pos, "</span>" );
	
	gtk_label_set_markup( GTK_LABEL( widget ), str->str );
	g_string_free( str, TRUE );

	box = glade_xml_get_widget( xml, "save" );
	gtk_window_set_wmclass( GTK_WINDOW( box ), "Screem",
				"save_confirm_prompt" );
        button = gtk_dialog_run( GTK_DIALOG( box ) );

	gtk_widget_destroy( box );
	g_object_unref( G_OBJECT( xml ) );

	if( button == GTK_RESPONSE_YES ) {
                screem_page_save( page );
	} else if( button == GTK_RESPONSE_NONE ) {
		button = GTK_RESPONSE_CANCEL;
	}

	return button;
}

gboolean screem_page_close_page( ScreemPage *page, ScreemWindow *window, 
			     gboolean confirm )
{
	gboolean ret;
	
	/* request save confirmation */
	ret = FALSE;
	if( ( ! confirm ) || 
		screem_page_save_confirmation( page ) != GTK_RESPONSE_CANCEL ) {
		screem_page_set_open( page, FALSE );
		screem_window_remove_document( window, page );
		ret = TRUE;
	}

	return ret;
}

gboolean screem_page_save_check( ScreemPage *page )
{
	const gchar *path;
	gint button = GTK_RESPONSE_NO;
	GtkWidget *prompt;
	GnomeVFSFileInfo *info;
	GnomeVFSURI *uri;
	GnomeVFSResult res;

	path = screem_page_get_pathname( page );

	/* Don't save if there is no file associated with this page yet */
	if( ! path ) {
		return FALSE;
	}

	uri = gnome_vfs_uri_new( path );

	if( uri ) {
		info = gnome_vfs_file_info_new();
		res = gnome_vfs_get_file_info_uri( uri, info,
						   GNOME_VFS_FILE_INFO_FOLLOW_LINKS );
		gnome_vfs_uri_unref( uri );
	} else {
		info = NULL;
		res = GNOME_VFS_ERROR_GENERIC;
	}

	if( res != GNOME_VFS_OK && res != GNOME_VFS_ERROR_NOT_FOUND ) {
		if( info ) {
			gnome_vfs_file_info_unref( info );
		}
		return FALSE;
	}

	/* file exists, check if it has been modified outside of screem */
	if( screem_page_get_modified( page ) ) {
		gchar *temp;
		temp = gnome_vfs_unescape_string_for_display( path );
		prompt = gtk_message_dialog_new( NULL, GTK_DIALOG_MODAL,
						 GTK_MESSAGE_WARNING,
						 GTK_BUTTONS_YES_NO,
						 _("%s has changed on disk\nReload instead of saving?"), temp );
		g_free( temp );
		gtk_window_set_wmclass( GTK_WINDOW( prompt ), "Screem",
					"revert_prompt" );

                button = gtk_dialog_run( GTK_DIALOG( prompt ) );

		gtk_widget_destroy( prompt );

		if( button == GTK_RESPONSE_YES ) {
			screem_page_revert( page );
		}
	}

	gnome_vfs_file_info_unref( info );

	return (button == GTK_RESPONSE_NO);
}

void external_view( ScreemPage *page )
{
        const gchar *pathname;
	gchar *uri;

        pathname = screem_page_get_pathname( page );

        g_return_if_fail( pathname != NULL );

        if( screem_page_save_confirmation( page ) != GTK_RESPONSE_CANCEL ) {
		uri = g_strconcat( "file://", pathname, NULL );
		
		gnome_url_show( uri, NULL );
		
		g_free( uri );
	}
}


void screem_page_edit_bookmarks( ScreemWindow *window, ScreemEditor *editor,
				 ScreemPage *page )
{
	GladeXML *xml;
	GConfClient *client;
	gchar *gladepath;

	GtkWidget *dialog;
	gint response;
	GtkWidget *bookmarkname;
	GtkWidget *bookmarkline;
	GtkWidget *jumpto;
	const gchar *name;
	gchar *title;
	gint line;
	gboolean allowjump;
	const gchar *pathname;
	gchar *temp;

	client = gconf_client_get_default();
	gladepath = gconf_client_get_string( client, "/apps/screem/general/glade_path",
					     NULL );
	xml = glade_xml_new( gladepath, "bookmarks", NULL );
	g_free( gladepath );

	dialog = glade_xml_get_widget( xml, "bookmarks" );
	bookmarkname = glade_xml_get_widget( xml, "bookmarkname" );
	bookmarkname = gnome_entry_gtk_entry( GNOME_ENTRY( bookmarkname ) );

	bookmarkline = glade_xml_get_widget( xml, "bookmarkline" );
	line = gtk_text_buffer_get_line_count( GTK_TEXT_BUFFER( page ) );
	gtk_spin_button_set_range( GTK_SPIN_BUTTON( bookmarkline ),
				   1.0, (gdouble)line );

	jumpto = glade_xml_get_widget( xml, "jumptobookmark" );
	allowjump = ( page == screem_window_get_document( window ) );
	gtk_widget_set_sensitive( jumpto, allowjump );

	{
		GtkWidget *treeview;
		GtkCellRenderer *renderer;
		GtkTreeViewColumn *col;
		GtkListStore *store;

		treeview = glade_xml_get_widget( xml, "bookmarkview" );
		g_object_set_data( G_OBJECT( treeview ), "page", page );
		
		renderer = gtk_cell_renderer_text_new();
		col = gtk_tree_view_column_new();
		gtk_tree_view_column_set_title( col, "Name" );
		gtk_tree_view_column_pack_start( col, renderer, TRUE );
		gtk_tree_view_append_column( GTK_TREE_VIEW( treeview ),
					     col );
		gtk_tree_view_column_set_attributes( col, renderer,
						     "text", 0, NULL );

		renderer = gtk_cell_renderer_text_new();
		col = gtk_tree_view_column_new();
		gtk_tree_view_column_set_title( col, "Line" );
		gtk_tree_view_column_pack_start( col, renderer, TRUE );
		gtk_tree_view_append_column( GTK_TREE_VIEW( treeview ),
					     col );
		gtk_tree_view_column_set_attributes( col, renderer,
						     "text", 1, NULL );

		store = screem_page_get_bookmark_model( page );
		gtk_tree_view_set_model( GTK_TREE_VIEW( treeview ),
					 GTK_TREE_MODEL( store ) );
	}


	pathname = screem_page_get_pathname( page );
	if( ! pathname ) {
		pathname = _( "Untitled" );
	}
	temp = gnome_vfs_unescape_string_for_display( pathname );

	title = g_strconcat( temp, "- Screem", NULL );
	gtk_window_set_title( GTK_WINDOW( dialog ), title );
	g_free( temp );
	g_free( title );

	glade_xml_signal_autoconnect( xml );

	while( dialog ) {
		response = gtk_dialog_run( GTK_DIALOG( dialog ) );
		
		name = gtk_entry_get_text( GTK_ENTRY( bookmarkname ) );
		line = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(bookmarkline));

		switch( response ) {
		case 0:
			/* add */
			screem_page_add_bookmark( page, line, name );
			break;
		case 1:
			/* remove */
			screem_page_remove_bookmark( page, name );
			break;
		case 2:
			/* jump to */
			line = screem_page_get_bookmark( page, name );
			if( line != -1 )
				screem_editor_goto_line( editor, line );
			break;
		default:
			break;
		}
		
		if( response != 0 && response != 1 ) {
			gtk_widget_destroy( dialog );
			dialog = NULL;
		}
	}
}

void screem_page_bookmark_activate( GtkTreeView *view,
				    GtkTreePath *path,
				    GtkTreeViewColumn *column,
				    gpointer data )
{
	GtkTreeSelection *selection;
	GtkListStore *model;
	GtkTreeModel *tmodel;
	GladeXML *xml;
	ScreemPage *page;

	xml = glade_get_widget_tree( GTK_WIDGET( view ) );
	selection = gtk_tree_view_get_selection( view );
	if( selection ) {
		GtkTreeIter it;
		GValue value = {0,};
		ScreemPageBookMark *bookmark;
		GtkWidget *name;
		GtkWidget *line;

		page = SCREEM_PAGE( g_object_get_data( G_OBJECT( view ), 
						       "page" ) );
		model = screem_page_get_bookmark_model( page );
		tmodel = GTK_TREE_MODEL( model );

		gtk_tree_selection_get_selected( selection, &tmodel, &it );
		gtk_tree_model_get_value( GTK_TREE_MODEL( model ),
					  &it, 2, &value );
		bookmark = g_value_get_pointer( &value );

		name = glade_xml_get_widget( xml, "bookmarkname" );
		line = glade_xml_get_widget( xml, "bookmarkline" );

		name = gnome_entry_gtk_entry( GNOME_ENTRY( name ) );
		gtk_entry_set_text( GTK_ENTRY( name ), 
				    bookmark->name->str );

		gtk_spin_button_set_value( GTK_SPIN_BUTTON( line ),
					   atof( bookmark->line_number ) );
	}
}


