/*  Screem:  linkWizard.c
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>

#include <string.h>

#include <libgnomeui/gnome-pixmap.h>
#include <libgnomeui/gnome-uidefs.h>
#include <libgnomeui/gnome-entry.h>
#include <libgnome/gnome-i18n.h>

#include <glade/glade.h>

#include <gtk/gtkentry.h>
#include <gtk/gtkdialog.h>
#include <gtk/gtkoptionmenu.h>
#include <gtk/gtkmenu.h>
#include <gtk/gtkmenuitem.h>
#include <gtk/gtktogglebutton.h>
#include <gtk/gtkspinbutton.h>


#include "screem-plugin.h"
#include "screem-window.h"
#include "screem-editor.h"

#include "libegg/menu/egg-menu.h"

static const gchar *protocols[] = {
	"http://", "ftp://",
	"mailto:", "irc://",
	"Other",
	NULL
};

static const gchar *relations[] = {
	"None",
	"Alternate",
	"Stylesheet",
	"Start",
	"Next",
	"Prev",
	"Contents",
	"Index",
	"Glossary",
	"Copyright",
	"Chapter",
	"Section",
	"Subsection",
	"Appendix",
	"Help",
	"Bookmark",
	NULL
};

/* per wizard struct */
typedef struct {
	ScreemWindow *window;
	ScreemEditor *editor;

	GtkWidget *dialog;
	GladeXML *xml;

	GtkWidget *pdialog;
	GladeXML *pxml;
	
	GSList *popup_attrs;
	guint popup_start;
	guint popup_len;
} LinkWizard;

/* keep track of how many link wizards we have */
static GList *wizards = NULL;

/* Required Plugin parts */

/* we don't want to try and load old plugins
   so this is a version symbol for screem to spot */
int screem_plugin_version = 3;

#define NAME "Link Wizard"
#define AUTHOR "David A Knight (david@screem.org)"
#define PLUGIN_VERSION "2.0"
#define TAG "a"

G_MODULE_EXPORT const gchar* g_module_check_init( GModule *module );
G_MODULE_EXPORT void g_module_unload( GModule *module );
G_MODULE_EXPORT void init( ScreemPlugin *plugin );
G_MODULE_EXPORT void add_ui( GtkWidget *window, GtkWidget *editor,
			     GtkWidget *preview, GtkWidget *link_view );
G_MODULE_EXPORT void remove_ui( GtkWidget *window, GtkWidget *editor,
				GtkWidget *preview, GtkWidget *link_view );
G_MODULE_EXPORT void popup( ScreemWindow *window,
			    guint start, guint len, GSList *attrs );

void link_wizard_display( EggAction *action, gpointer user_data );
void link_wizard_clicked( GtkWidget *widget, gint button, gpointer data );
void link_wizard_edit_popup( GtkWidget *widget );

static void create_dialog( LinkWizard *link_wizard );


G_MODULE_EXPORT const gchar* g_module_check_init( GModule *module )
{
	return NULL;
}

G_MODULE_EXPORT void g_module_unload( GModule *module )
{
}

G_MODULE_EXPORT void init( ScreemPlugin *plugin )
{
	plugin->name = NAME;
	plugin->author = AUTHOR;
	plugin->version = PLUGIN_VERSION;

	/* these can all be left out if not required */
	plugin->tag = TAG;

	g_module_symbol( plugin->module, "popup", 
			 (gpointer*)&plugin->popup );
	g_module_symbol( plugin->module, "add_ui", 
			 (gpointer*)&plugin->add_ui );
	g_module_symbol( plugin->module, "remove_ui", 
			 (gpointer*)&plugin->remove_ui );
}

/* not needed if nothing is being added to the UI (unusual) */
G_MODULE_EXPORT void add_ui( GtkWidget *window, GtkWidget *editor,
			     GtkWidget *preview, GtkWidget *link_view )
{
	LinkWizard *link_wizard;
	const gchar *ui = "\
<Root>\
<menu>\
<submenu name=\"Insert\" verb=\"Insert\">\
<submenu name=\"Wizards\" verb=\"Wizards\">\
<menuitem name=\"LinkWizard\" verb=\"LinkWizard\"/>\
</submenu>\
</submenu>\
</menu>\
<dockitem name=\"Wizards Toolbar\">\
<toolitem name=\"LinkWizard\" verb=\"LinkWizard\" />\
</dockitem>\
</Root>";
	static EggActionGroupEntry entries[] = {
		{ "LinkWizard", "label",
	  	"Screem_Link", NULL, "tip",
	  	G_CALLBACK( link_wizard_display ), NULL },
	};
	EggActionGroupEntry *entry;

	gchar *label;
	gchar *tip;

	link_wizard = g_new0( LinkWizard, 1 );
	link_wizard->window = SCREEM_WINDOW( window );
	link_wizard->editor = SCREEM_EDITOR( editor );

	label = g_strdup( _( "Link Wizard" ) );
	tip = g_strdup( _( "Insert a Link" ) );

	entry = entries;
	entry->label = label;
	entry->tooltip = tip;
	entry->user_data = link_wizard;
	
	egg_action_group_add_actions( EGG_ACTION_GROUP( link_wizard->window->action_group ),
					entry, 1 );
	egg_menu_merge_add_ui_from_string( EGG_MENU_MERGE( link_wizard->window->merge ),
						ui, strlen( ui ), NULL );
	
	g_free( label );
	g_free( tip );

	create_dialog( link_wizard );

	wizards = g_list_append( wizards, link_wizard );
}

/* required it add_ui is present */
G_MODULE_EXPORT void remove_ui( GtkWidget *window, GtkWidget *editor,
				GtkWidget *preview, GtkWidget *link_view )
{
	GList *list;
	LinkWizard *link_wizard;

	for( list = wizards; list; list = list->next ) {
		link_wizard = (LinkWizard*)list->data;
		if( link_wizard->window == SCREEM_WINDOW( window ) ) {
			/* got it */
			break;
		}
	}
       
	g_return_if_fail( list != NULL );

	/* link_wizard is the one to erase */
	wizards = g_list_remove( wizards, link_wizard );
	g_free( link_wizard );
}

/* for popup tag editing in the wizard */
G_MODULE_EXPORT void popup( ScreemWindow *window,
			    guint start, guint len, GSList *attrs )
{
	GList *list;
	GSList *attr;
	LinkWizard *link_wizard;
	GtkWidget *widget;
	gboolean onclick;

	for( list = wizards; list; list = list->next ) {
		link_wizard = (LinkWizard*)list->data;
		if( link_wizard->window == window ) {
			/* got it */
			break;
		}
	}
	g_return_if_fail( list != NULL );

	screem_editor_select_region( link_wizard->editor, start, len );

	widget = glade_xml_get_widget( link_wizard->xml,
				       "link_text" );
	gtk_widget_set_sensitive( widget, FALSE );

	/* fill in attr values */
	onclick = TRUE;
	for( attr = attrs; attr; attr = attr->next ) {
		const gchar *name;
		const gchar *value;
		
		value = (const gchar*)attr->data;
		attr = attr->next;
		name = (const gchar*)attr->data;

		if( ! g_strcasecmp( "href", name ) ) {
			gint i;
			gboolean rel;
			
			/* handle protocol */
			widget = glade_xml_get_widget( link_wizard->xml,
						       "protocols" );
			rel = TRUE;
			for( i = 0; protocols[ i ]; ++ i ) {
				if( ! strncmp( protocols[ i ], value,
					       strlen( protocols[ i ] ) ) ) {
					value += strlen( protocols[ i ] );
					rel = FALSE;
					break;
				}
			}
			if( ! protocols[ i ] ) {
				i --;
			}
			gtk_option_menu_set_history( GTK_OPTION_MENU(widget),
						     i );
			widget = glade_xml_get_widget( link_wizard->xml,
					"absolute" );
			gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ), !rel );
			widget = glade_xml_get_widget( link_wizard->xml,
						       "link_to" );
			gtk_entry_set_text( GTK_ENTRY( widget ), value );

		} else if( ! g_strcasecmp( "target", name ) ) {
			widget = glade_xml_get_widget( link_wizard->xml,
						       "target" );
			gtk_entry_set_text( GTK_ENTRY( widget ), value );
		} else if( ! g_strcasecmp( "onclick", name ) ) {
			onclick = FALSE;
		}
	}

	widget = glade_xml_get_widget( link_wizard->xml, "edit_popup" );
	gtk_widget_set_sensitive( widget, FALSE );

	link_wizard->popup_attrs = attrs;
	link_wizard->popup_start = start;
	link_wizard->popup_len = len;

	gtk_dialog_run( GTK_DIALOG( link_wizard->dialog ) );


	widget = glade_xml_get_widget( link_wizard->xml,
				       "link_text" );
	gtk_widget_set_sensitive( widget, TRUE );

	link_wizard->popup_attrs = NULL;
}

/* End of required section */

/* now the stuff that actually does the work */
void link_wizard_display( EggAction *action, gpointer user_data )
{
	LinkWizard *link_wizard;
	ScreemPage *page;
	GtkWidget *widget;
	
	link_wizard = (LinkWizard*)user_data;

	page = screem_window_get_document( link_wizard->window );

	if( page ) {
		widget = glade_xml_get_widget( link_wizard->pxml, "onclick" );
		gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ), FALSE );
		gtk_widget_show_all( link_wizard->dialog );
		gdk_window_raise( link_wizard->dialog->window );

	}
}

void link_wizard_clicked( GtkWidget *dialog, gint button, gpointer data )
{
	LinkWizard *link_wizard;
	const gchar *features[] = {
		"directories",
		"fullscreen",
		"location",
		"menubar",
		"resizable",
		"scrollbars",
		"status",
		"toolbars",
		NULL
	};
	gint i;

	link_wizard = (LinkWizard*)g_object_get_data( G_OBJECT( dialog ),
						      "link_wizard" );

	if( button == GTK_RESPONSE_APPLY || button == GTK_RESPONSE_OK ) { 
		GtkWidget *widget;
		const gchar *text;
		const gchar *uri;
		const gchar *target;
		const gchar *protocol;
		const gchar *relation;
		const gchar *reverse;
		GString *tag;
		gint pos;
		gboolean script;
		gboolean fullscreen;
		
		script = FALSE;
		fullscreen = FALSE;
		
		widget = glade_xml_get_widget( link_wizard->xml,
					       "link_text" );
		text = gtk_entry_get_text( GTK_ENTRY( widget ) );

		widget = glade_xml_get_widget( link_wizard->xml,
					       "link_to" );
		uri = gtk_entry_get_text( GTK_ENTRY( widget ) );

		widget = glade_xml_get_widget( link_wizard->xml,
					       "target" );
		target = gtk_entry_get_text( GTK_ENTRY( widget ) );
		if( ! GTK_WIDGET_SENSITIVE( widget ) ) {
			target = "";
		}

		widget = glade_xml_get_widget( link_wizard->xml,
					       "protocols" );
		widget = GTK_OPTION_MENU( widget )->menu_item;
		protocol = (const gchar*)g_object_get_data( G_OBJECT( widget ),
							    "protocol" );
		widget = glade_xml_get_widget( link_wizard->xml,
					       "protocols" );
		if( ! GTK_WIDGET_IS_SENSITIVE( widget ) ||
		    ! strcmp( _( "Other" ), protocol ) ) {
			protocol = "";
		}

		widget = glade_xml_get_widget( link_wizard->xml,
						"relation" );
		widget = GTK_OPTION_MENU( widget )->menu_item;
		relation = (const gchar *)g_object_get_data( G_OBJECT( widget ),
							"relation" );
		widget = glade_xml_get_widget( link_wizard->xml,
						"reverse_relation" );
		widget = GTK_OPTION_MENU( widget )->menu_item;
		reverse = (const gchar *)g_object_get_data( G_OBJECT( widget ),
							"relation" );
	
	
		
		/* FIXME: we should URI encode link_to */

		tag = g_string_new( "<a href=\"" );
		g_string_append_printf( tag, "%s%s\" ", protocol, uri );
		if( *target ) {
			g_string_append_printf( tag, "target=\"%s\"",
					target );
		}
		if( strcmp( "None", relation ) ) {
			g_string_append_printf( tag, "rel=\"%s\" ",
					relation );
		}
		if( strcmp( "None", reverse ) ) {
			g_string_append_printf( tag, "rev=\"%s\" ",
					reverse );
		}
	
		widget = glade_xml_get_widget( link_wizard->xml,
				"edit_popup" );
		if( GTK_WIDGET_IS_SENSITIVE( widget ) ) {
			/* we may have popup properties to add */

			widget = glade_xml_get_widget( link_wizard->pxml, "propbox" );
			script = ( GTK_WIDGET_IS_SENSITIVE( widget ) );
		}
		if( script ) {
			/* add onclick window.open() */
			GString *tmp;
			gboolean set;
			gint x;
			gint y;
			
			g_string_append( tag, " onclick=\"window.open( this.href, this.target" );

			set = FALSE;
			tmp = g_string_new( ", '" );
	
			widget = glade_xml_get_widget( link_wizard->pxml, "fullscreen" );
			fullscreen = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) );

			widget = glade_xml_get_widget( link_wizard->pxml, "dimensions" );
			if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) && ! fullscreen ) {
				set = TRUE;
			
				widget = glade_xml_get_widget( link_wizard->pxml, "width" );
				x = gtk_spin_button_get_value_as_int( GTK_SPIN_BUTTON( widget ) );
				widget = glade_xml_get_widget( link_wizard->pxml, "height" );
				y = gtk_spin_button_get_value_as_int( GTK_SPIN_BUTTON( widget ) );
				g_string_append_printf( tmp, "width=%i,height=%i", x, y );
			}
			widget = glade_xml_get_widget( link_wizard->pxml, "position" );
			if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) && ! fullscreen ) {
				widget = glade_xml_get_widget( link_wizard->pxml, "x" );
				x = gtk_spin_button_get_value_as_int( GTK_SPIN_BUTTON( widget ) );
				widget = glade_xml_get_widget( link_wizard->pxml, "y" );
				y = gtk_spin_button_get_value_as_int( GTK_SPIN_BUTTON( widget ) );
				if( set ) {
					g_string_append_c( tmp, ',' );
				}
				g_string_append_printf( tmp, "left=%i,top=%i", x, y );
				set = TRUE;
			}
			for( i = 0; features[ i ]; ++ i ) {
				widget = glade_xml_get_widget( link_wizard->pxml, features[ i ] );
				if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
					if( set ) {
						g_string_append_c( tmp, ',' );
					}
					g_string_append( tmp, features[ i ] );
					set = TRUE;
				}
			}
			if( set ) {
				g_string_append_c( tmp, '\'' );
				g_string_append( tag, tmp->str );
			}
			g_string_free( tmp, TRUE );
			
			g_string_append( tag, " ); return false;\" " );
		}
		
		/* if we are from the popup we need to fill in all
		   the attributes we haven't handled in the wizard */
		if( link_wizard->popup_attrs ) {
			GSList *attr;
			for( attr = link_wizard->popup_attrs; attr;
			     attr = attr->next ) {
				const gchar *name;
				const gchar *value;
				
				value = (const gchar*)attr->data;
				attr = attr->next;
				name = (const gchar*)attr->data;

				if( ( ( ! script ) && 
				      ! g_strcasecmp( "onclick", name ) ) ||
				    ( g_strcasecmp( "href", name ) &&
				      g_strcasecmp( "target", name ) ) ) {
					g_string_append_c( tag, ' ' );
					g_string_append( tag, name );
					if( value ) {
						g_string_append( tag, "=\"" );
						g_string_append( tag, value );
						g_string_append_c( tag, '"' );
					}
				}
			}
			
			g_string_append_c( tag, '>' );
			pos = link_wizard->popup_start;
			screem_editor_delete_forward( link_wizard->editor,
						      pos,
						      link_wizard->popup_len );
		} else {
			g_string_append_c( tag, '>' );
			
			g_string_append( tag, text );
			g_string_append( tag, "</a>" );
			
			pos = screem_editor_get_pos( link_wizard->editor );
		}
		screem_editor_insert( link_wizard->editor, pos, tag->str );
		screem_editor_set_pos( link_wizard->editor, pos + tag->len );
		
		g_string_free( tag, TRUE );
	}
	if( button != GTK_RESPONSE_APPLY ) {
		gtk_widget_hide( dialog );
        }
}

void link_wizard_edit_popup( GtkWidget *widget )
{
	LinkWizard *wiz;
	GladeXML *xml;

	xml = glade_get_widget_tree( widget );
	widget = glade_xml_get_widget( xml, "linkwizard" );

	wiz = (LinkWizard*)g_object_get_data( G_OBJECT( widget ),
					      "link_wizard" );

	gtk_widget_show_all( wiz->pdialog );
	gdk_window_raise( wiz->pdialog->window );
	
}

static void create_dialog( LinkWizard *link_wizard )
{
	GtkWidget *widget;
	GtkWidget *menu;
	gint i;

	GtkSizeGroup *group;
	const gchar *labels1[] = {
		"label7", "label8", "label9",
		NULL
	};
	const gchar *labels2[] = {
		"label10", "label11", "label12",
		NULL
	};
	
	
	link_wizard->xml = glade_xml_new( GLADE_PATH"/linkwizard.glade",
					  "linkwizard", NULL );
	
	link_wizard->dialog = glade_xml_get_widget( link_wizard->xml, 
						    "linkwizard" );

	group = gtk_size_group_new( GTK_SIZE_GROUP_HORIZONTAL );
	for( i = 0; labels1[ i ]; ++ i ) {
		widget = glade_xml_get_widget( link_wizard->xml,
					labels1[ i ] );
		gtk_size_group_add_widget( group, widget );
	}
	group = gtk_size_group_new( GTK_SIZE_GROUP_HORIZONTAL );
	for( i = 0; labels2[ i ]; ++ i ) {
		widget = glade_xml_get_widget( link_wizard->xml,
					labels2[ i ] );
		gtk_size_group_add_widget( group, widget );
	}

	g_object_set_data( G_OBJECT( link_wizard->dialog ),
			   "link_wizard", link_wizard );

	widget = glade_xml_get_widget( link_wizard->xml, "protocols" );
	menu = gtk_menu_new();
	for( i = 0; protocols[ i ]; ++ i ) {
		GtkWidget *item;

		item = gtk_menu_item_new_with_label( protocols[ i ] );
		g_object_set_data( G_OBJECT( item ), 
				   "protocol", (gpointer)protocols[ i ] );
		gtk_widget_show( item );
		gtk_menu_shell_append( GTK_MENU_SHELL( menu ), item );
	}
	gtk_widget_show( menu );
	gtk_option_menu_set_menu( GTK_OPTION_MENU( widget ), menu );

	widget = glade_xml_get_widget( link_wizard->xml, "relation" );
	menu = gtk_menu_new();
	for( i = 0; relations[ i ]; ++ i ) {
		GtkWidget *item;

		item = gtk_menu_item_new_with_label( relations[ i ] );
		g_object_set_data( G_OBJECT( item ), 
				   "relation", (gpointer)relations[ i ] );
		gtk_widget_show( item );
		gtk_menu_shell_append( GTK_MENU_SHELL( menu ), item );
	}
	gtk_widget_show( menu );
	gtk_option_menu_set_menu( GTK_OPTION_MENU( widget ), menu );

	widget = glade_xml_get_widget( link_wizard->xml, "reverse_relation" );
	menu = gtk_menu_new();
	for( i = 0; relations[ i ]; ++ i ) {
		GtkWidget *item;

		item = gtk_menu_item_new_with_label( relations[ i ] );
		g_object_set_data( G_OBJECT( item ), 
				   "relation", (gpointer)relations[ i ] );
		gtk_widget_show( item );
		gtk_menu_shell_append( GTK_MENU_SHELL( menu ), item );
	}
	gtk_widget_show( menu );
	gtk_option_menu_set_menu( GTK_OPTION_MENU( widget ), menu );

	
	glade_xml_signal_autoconnect( link_wizard->xml );

	link_wizard->pxml = glade_xml_new( GLADE_PATH"/linkwizard.glade",
					  "popup", NULL );
	
	link_wizard->pdialog = glade_xml_get_widget( link_wizard->pxml, 
						    "popup" );
	g_object_set_data( G_OBJECT( link_wizard->pdialog ),
			   "link_wizard", link_wizard );
	glade_xml_signal_autoconnect( link_wizard->pxml );

}
