/*  Screem:  style.c,
 *  Copyright (C) 2003  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>

#include <string.h>

#include "css.h"
#include "style.h"

Style* css_style_new( const gchar *pattern )
{
	Style *style;

	style = g_new0( Style, 1 );
	style->pattern = g_strdup( pattern );
	
	return style;
}

FontStyle* css_font_style_new()
{
	FontStyle *style;

	style = g_new0( FontStyle, 1 );

	return style;
}

BackStyle* css_back_style_new()
{
	BackStyle *style;

	style = g_new0( BackStyle, 1 );

	return style;
}

TextStyle* css_text_style_new()
{
	TextStyle *style;

	style = g_new0( TextStyle, 1 );

	return style;
}

BoxStyle* css_box_style_new()
{
	BoxStyle *style;

	style = g_new0( BoxStyle, 1 );

	return style;
}

ClassStyle* css_class_style_new()
{
	ClassStyle *style;

	style = g_new0( ClassStyle, 1 );

	return style;
}

void css_style_destroy( Style *style )
{
	if( style ) {
		css_font_style_destroy( style->font );
		css_back_style_destroy( style->back );
		css_text_style_destroy( style->text );
		css_box_style_destroy( style->box );
		css_class_style_destroy( style->class );

		g_free( style->pattern);
	
		g_free( style->unknown );

		g_free( style );
	}
}

void css_font_style_destroy( FontStyle *style )
{
	if( ! style ) {
		g_free( style->family );
		g_free( style->style );
		g_free( style->variant );
		g_free( style->weight );
		g_free( style->size );
		g_free( style->color );

		g_free( style );
	}
}

void css_back_style_destroy( BackStyle *style )
{
	if( style ) {
		g_free( style->color );
		g_free( style->image );
		g_free( style->repeat );
		g_free( style->attachment );
		g_free( style->position );

		g_free( style );
	}
}

void css_text_style_destroy( TextStyle *style )
{
	if( style ) {
		g_free( style->word_spacing );
		g_free( style->letter_spacing );
		g_free( style->decoration );
		g_free( style->vertical_align );
		g_free( style->transform );
		g_free( style->align );
		g_free( style->indent );
		g_free( style->height );

		g_free( style );
	}
}

void css_box_style_destroy( BoxStyle *style )
{
	if( style ) {
		gint i;
		
		for( i = 0; i < 4; i ++ ) {
			g_free( style->margin[ i ] );
			g_free( style->padding[ i ] );
			g_free( style->border_color[ i ] );
			g_free( style->border_width[ i ] );
			g_free( style->border_style[ i ] );
		}
		g_free( style->width );
		g_free( style->height );
		g_free( style->float_ );
		g_free( style->clear );

		g_free( style );
	}
}

void css_class_style_destroy( ClassStyle *style )
{
	if( style ) {
		g_free( style->display );
		g_free( style->white_space );
		g_free( style->list_style_type );
		g_free( style->list_style_image );
		g_free( style->list_style_position );

		g_free( style );
	}
}

gchar* css_style_output( Style *style )
{
	gchar *data;
	gchar *temp;
	gchar *data2;
	gchar *name;

	if( ! style )
		return NULL;
	
	/* add the class on */
	name = g_strdup( style->pattern );
	
	if( ! ( temp = css_font_style_output( style->font ) ) )
		temp = g_strdup( "" );

	data = temp;

	if( ! ( temp = css_back_style_output( style->back ) ) )
		temp = g_strdup( "" );
	
	data2 = g_strconcat( data, temp, NULL );
	g_free( data );
	g_free( temp );

	if( ! ( temp = css_text_style_output( style->text ) ) )
		temp = g_strdup( "" );

	data = g_strconcat( data2, temp, NULL );
	g_free( data2 );
	g_free( temp );

	if( ! ( temp = css_box_style_output( style->box ) ) )
		temp = g_strdup( "" );

	data2 = g_strconcat( data, temp, NULL );
	g_free( data );
	g_free( temp );


	if( ! ( temp = css_class_style_output( style->class ) ) )
		temp = g_strdup( "" );

	data = g_strconcat( data2, temp, NULL );
	g_free( data2 );
	g_free( temp );

	/* now append any stuff the wizard didn't understand */
	temp = style->unknown;
	data2 = g_strconcat( data, temp, NULL );
	g_free( data );

	data = g_strdup_printf( "%s {%s\n}", name, data2 );

	g_free( data2 );
	g_free( name );

	return data;
}

gchar* css_font_style_output( FontStyle *style )
{
	gchar *data;
	gchar *temp;
	gchar *temp2;

	if( ! style )
		return NULL;

	if( ( style->family ) && ( strlen( style->family ) > 0 ) )
		data = g_strdup_printf( "\n\tfont-family: %s;", 
					style->family );
	else
		data = g_strdup( "" );

	temp = data;

	if( ( style->style ) && ( strlen( style->style ) > 0 ) )
		data = g_strdup_printf( "\n\tfont-style: %s;", style->style );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	if( ( style->variant ) && ( strlen( style->variant ) > 0 ) )
		data = g_strdup_printf("\n\tfont-variant: %s;", style->variant);
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	if( ( style->weight ) && ( strlen( style->weight ) > 0 ) )
		data = g_strdup_printf( "\n\tfont-weight: %s;", style->weight );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	if( ( style->size ) && ( strlen( style->size ) > 0 ) )
		data = g_strdup_printf( "\n\tfont-size: %s;", style->size );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	if( ( style->color ) && ( strlen( style->color ) > 0 ) )
		data = g_strdup_printf( "\n\tcolor: %s;", style->color );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	return temp; 
}

gchar* css_back_style_output( BackStyle *style )
{
	gchar *data;
	gchar *temp;
	gchar *temp2;

	if( ! style )
		return NULL;

	if( ( style->color ) && ( strlen( style->color ) > 0 ) )
		data = g_strdup_printf( "\n\tbackground-color: %s;", 
					style->color );
	else
		data = g_strdup( "" );

	temp = data;

	if( ( style->image ) && ( strlen( style->image ) > 0 ) )
		data = g_strdup_printf( "\n\tbackground-image: url(%s);", 
					style->image );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	if( ( style->repeat ) && ( strlen( style->repeat ) > 0 ) )
		data = g_strdup_printf( "\n\tbackground-repeat: %s;", 
					style->repeat );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	if( ( style->attachment ) && ( strlen( style->attachment ) > 0 ) )
		data = g_strdup_printf( "\n\tbackground-attachment: %s;", 
					style->attachment );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	if( ( style->position ) && ( strlen( style->position ) > 0 ) )
		data = g_strdup_printf( "\n\tbackground-position: %s;", 
					style->position );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	return temp; 
}

gchar* css_text_style_output( TextStyle *style )
{
	gchar *data;
	gchar *temp;
	gchar *temp2;

	if( ! style )
		return NULL;

	if( ( style->word_spacing ) && ( strlen( style->word_spacing ) > 0 ) )
		data = g_strdup_printf( "\n\tword-spacing: %s;", 
					style->word_spacing );
	else
		data = g_strdup( "" );

	temp = data;

	if( ( style->letter_spacing ) && 
	    ( strlen( style->letter_spacing ) > 0 ) )
		data = g_strdup_printf( "\n\tletter-spacing: %s;", 
					style->letter_spacing );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	if( ( style->decoration ) && ( strlen( style->decoration ) > 0 ) )
		data = g_strdup_printf( "\n\ttext-decoration: %s;", 
					style->decoration );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	if( ( style->vertical_align ) && 
	    ( strlen( style->vertical_align ) > 0 ) )
		data = g_strdup_printf( "\n\tvertical-align: %s;", 
					style->vertical_align );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	if( ( style->transform ) && ( strlen( style->transform ) > 0 ) )
		data = g_strdup_printf( "\n\ttext-transform: %s;", 
					style->transform );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	if( ( style->align ) && ( strlen( style->align ) > 0 ) )
		data = g_strdup_printf( "\n\ttext-align: %s;", style->align );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	if( ( style->indent ) && ( strlen( style->indent ) > 0 ) )
		data = g_strdup_printf( "\n\ttext-indent: %s;", style->indent );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	if( ( style->height ) && ( strlen( style->height ) > 0 ) )
		data = g_strdup_printf( "\n\tline-height: %s;", style->height );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	return temp; 	
}

gchar* css_box_style_output( BoxStyle *style )
{
	gchar *data;
	gchar *temp;
	gchar *temp2;
	gint i;

	gboolean not_empty = TRUE;

	gchar *one;
	gchar *two;
	gchar *three;

	const gchar *side[] = { "top", "right", "bottom", "left" };

	if( ! style )
		return NULL;

	temp = g_strdup( "" );

	for( i = 0; i < 4; i ++ ) {
		if( ( style->margin[ i ] ) && 
		    ( strlen( style->margin[ i ] ) > 0 ) )
			data = g_strdup_printf( "\n\tmargin-%s: %s;",
						side[ i ],
						style->margin[ i ] );
		else
			data = g_strdup( "" );
		temp2 = temp;
		temp = g_strconcat( temp2, data, NULL );
		g_free( temp2 );
		g_free( data );
	}

	for( i = 0; i < 4; i ++ ) {
		if( ( style->padding[ i ] ) && 
		    ( strlen( style->padding[ i ] ) > 0 ) )
			data = g_strdup_printf( "\n\tpadding-%s: %s;",
						side[ i ],
						style->padding[ i ] );
		else
			data = g_strdup( "" );
		temp2 = temp;
		temp = g_strconcat( temp2, data, NULL );
		g_free( temp2 );
		g_free( data );
	}

	/* border-width border-style border-color */
	for( i = 0; i < 4; i ++ ) {
		not_empty = FALSE;
		if( style->border_width[ i ] && 
		    ( strlen( style->border_width[ i ] ) > 0 ) )
			not_empty = (gboolean)
				one = g_strconcat( " ", 
						   style->border_width[ i ],
						   NULL );
		else
			one = g_strdup( "" );
		if( style->border_style[ i ] && 
		    ( strlen( style->border_style[ i ] ) > 0 ) )
			not_empty = (gboolean)
				two = g_strconcat( " ",
						   style->border_style[ i ],
						   NULL );
		else
			two = g_strdup( "" );
		if( style->border_color[ i ] && 
		    ( strlen( style->border_color[ i ] ) > 0 ) ) 
			not_empty = (gboolean)
				three = g_strconcat( " ",
						     style->border_color[ i ],
						     NULL );
		else
			three = g_strdup( "" );

		if( not_empty ) {
			data = g_strdup_printf( "\n\tborder-%s: %s%s%s;",
						side[ i ],
						one, two, three );
			
			temp2 = temp;
			temp = g_strconcat( temp2, data, NULL );
			g_free( temp2 );
			g_free( data );
		}
	}

	if( ( style->width ) && ( strlen( style->width ) > 0 ) )
		data = g_strdup_printf( "\n\twidth: %s;", style->width );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	if( ( style->height ) && ( strlen( style->height ) > 0 ) )
		data = g_strdup_printf( "\n\theight: %s;", style->height );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	if( ( style->float_ ) && ( strlen( style->float_ ) > 0 ) )
		data = g_strdup_printf( "\n\tfloat: %s;", style->float_ );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	if( ( style->clear ) &&  ( strlen( style->clear ) > 0 ) )
		data = g_strdup_printf( "\n\tclear: %s;", style->clear );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	return temp;      
}

gchar* css_class_style_output( ClassStyle *style )
{
 	gchar *data;
	gchar *temp;
	gchar *temp2;

	if( ! style )
		return NULL;

	if( ( style->display ) && ( strlen( style->display ) > 0 ) )
		data = g_strdup_printf( "\n\tdisplay: %s;", style->display );
	else
		data = g_strdup( "" );

	temp = data;

	if( ( style->white_space ) && ( strlen( style->white_space ) > 0 ) )
		data = g_strdup_printf( "\n\twhite-space: %s;", 
					style->white_space );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	if( ( style->list_style_type ) && 
	    ( strlen( style->list_style_type ) > 0 ) )
		data = g_strdup_printf( "\n\tlist-style-type: %s;", 
					style->list_style_type );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	if( ( style->list_style_image ) && 
	    ( strlen( style->list_style_image ) > 0 ) )
		data = g_strdup_printf( "\n\tlist-style-image: url(%s);", 
					style->list_style_image );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	if( ( style->list_style_position ) && 
	    ( strlen( style->list_style_position ) > 0 ) )
		data = g_strdup_printf( "\n\tlist-style-position: %s;", 
					style->list_style_position );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	return temp;      
}

void css_font_style_set( Style *style, 
			 const gchar *param,
			 const gchar *value )
{
	gchar **values;
	
	if( ! style->font ) {
		style->font = css_font_style_new();
	}

	values = g_strsplit( value, " ", -1 );
	
	if( ! strcmp( param, "font-family" ) ) {
		g_free( style->font->family );
		style->font->family = g_strdup( value );
	} else if( ! strcmp( param, "font-style" ) ) {
		g_free( style->font->style );
		style->font->style = g_strdup( value );
	} else if( ! strcmp( param, "font-variant" ) ) {
		g_free( style->font->variant );
		style->font->variant = g_strdup( value );
	} else if( ! strcmp( param, "font-weight" ) ) {
		g_free( style->font->weight );
		style->font->weight = g_strdup( value );
	} else if( ! strcmp( param, "font-size" ) ) {
		g_free( style->font->size );
		style->font->size = g_strdup( value );
	} else if( ! strcmp( param, "color" ) ) {
		style->font->color = g_strdup( value );
	} else if( ! strcmp( param, "font" ) ) {
		
	}
	
	g_strfreev( values );
}

void css_back_style_set( Style *style, 
			 const gchar *param,
			 const gchar *value )
{
	gchar *temp;
	gchar **values;
	
	if( ! style->back ) {
		style->back = css_back_style_new();
	}

	values = g_strsplit( value, " ", -1);
	
	if( ! strcmp( param, "background-color" ) ) {
		g_free( style->back->color );
		style->back->color = g_strdup( value );
	} else if( ! strcmp( param, "background-image" ) ) {
		if( ! strncmp( "url(", value, strlen( "url(" ) ) ) {
			temp = g_strdup( value + strlen( "url(" ) );
			temp[ strlen( temp ) - 1 ] = '\0';
			value = temp;
		} else
			temp = NULL;
		/* convert value to a path relative to the stylesheet */
		g_free( style->back->image );
		style->back->image = css_relative_path( value );
		g_free( temp );
	} else if( ! strcmp( param, "background-repeat" ) ) {
		g_free( style->back->repeat );
		style->back->repeat = g_strdup( value );
	} else if( ! strcmp( param, "background-attachment" ) ) {
		g_free( style->back->attachment );
		style->back->attachment = g_strdup( value );
	} else if( ! strcmp( param, "background-position" ) ) {
		g_free( style->back->position );
		style->back->position = g_strdup( value );
	} else if( ! strcmp( param, "background" ) ) {
		/* all bundled into one property 
			[<'background-color'> || 
			<'background-image'> || 
			<'background-repeat'> || 
			<'background-attachment'> || 
			<'background-position'>] | inherit */
		if( ( ! values[ 1 ]  ) && ! strcmp( "inherit", values[ 0 ] ) ) {
			css_back_style_set( style, "background-color", 
					    "inherit" );
			css_back_style_set( style, "background-image", 
					    "inherit" );
			css_back_style_set( style, "background-repeat", 
					    "inherit" );
			css_back_style_set( style, "background-attachment", 
					    "inherit" );
			css_back_style_set( style, "background-position", 
					    "inherit" );
		}
	}
	
	g_strfreev( values );
}

void css_text_style_set( Style *style, const gchar *param,
			 const gchar *value )
{
	if( ! style->text ) {
		style->text = css_text_style_new();
	}

	if( ! strcmp( param, "word-spacing" ) ) {
		g_free( style->text->word_spacing );
		style->text->word_spacing = g_strdup( value );
	} else if( ! strcmp( param, "letter-spacing" ) ) {
		g_free( style->text->letter_spacing );
		style->text->letter_spacing = g_strdup( value );
	} else if( ! strcmp( param, "text-decoration" ) ) {
		g_free( style->text->decoration );
		style->text->decoration = g_strdup( value );
	} else if( ! strcmp( param, "vertical-align" ) ) {
		g_free( style->text->vertical_align );
		style->text->vertical_align = g_strdup( value );
	} else if( ! strcmp( param, "text-transform" ) ) {
		g_free( style->text->transform );
		style->text->transform = g_strdup( value );
	} else if( ! strcmp( param, "text-align" ) ) {
		g_free( style->text->align );
		style->text->align = g_strdup( value );
	} else if( ! strcmp( param, "text-indent" ) ) {
		g_free( style->text->indent );
		style->text->indent = g_strdup( value );
	} else if( ! strcmp( param, "line-height" ) ) {
		g_free( style->text->height );
		style->text->height = g_strdup( value );
	}
}

void css_box_style_set( Style *style, const gchar *param,
			const gchar *value )
{
	gchar **values;

	if( ! style->box ) {
		style->box = css_box_style_new();
	}

	/* grrrr, this can get messy now,
	   value may be a space separated group of values */
	values = g_strsplit( value, " ", 4 );

	if( ! strcmp( param, "margin" ) ) {
		if( ! values[ 1 ] ) {
			/* only 1 param given, set on all */
			css_box_style_set( style, "margin-top", value );
			css_box_style_set( style, "margin-right", value );
			css_box_style_set( style, "margin-bottom", value );
			css_box_style_set( style, "margin-left", value );
		} else if( ! values[ 2 ] ) {
			/* only 2 given */
			css_box_style_set( style, "margin-top", values[ 0 ] );
			css_box_style_set( style, "margin-right", values[ 1 ] );
			css_box_style_set( style, "margin-bottom", values[ 0 ] );
			css_box_style_set( style, "margin-left", values[ 1 ] );
		} else if( ! values[ 3 ] ) {
			css_box_style_set( style, "margin-top", values[ 0 ] );
			css_box_style_set( style, "margin-right", values[ 1 ] );
			css_box_style_set( style, "margin-bottom", values[ 2 ] );
			css_box_style_set( style, "margin-left", values[ 1 ] );
		} else {
			css_box_style_set( style, "margin-top", values[ 0 ] );
			css_box_style_set( style, "margin-right", values[ 1 ] );
			css_box_style_set( style, "margin-bottom", values[ 2 ] );
			css_box_style_set( style, "margin-left", values[ 3 ] );
		}
	} else if( ! strcmp( param, "margin-top" ) ) { 
		g_free( style->box->margin[ TOP ] );
		style->box->margin[ TOP ] = g_strdup( value );
	 } else if( ! strcmp( param, "margin-right" ) ) { 
		g_free( style->box->margin[ RIGHT ] );
		style->box->margin[ RIGHT ] = g_strdup( value );
	 } else if( ! strcmp( param, "margin-bottom" ) ) { 
		g_free( style->box->margin[ BOTTOM ] );
		style->box->margin[ BOTTOM ] = g_strdup( value );
	 } else if( ! strcmp( param, "margin-left" ) ) { 
		g_free( style->box->margin[ LEFT ] );
		style->box->margin[ LEFT ] = g_strdup( value );
	 } else if( ! strcmp( param, "padding" ) ) {
		if( ! values[ 1 ] ) {
			/* only 1 param given, set on all */
			css_box_style_set( style, "padding-top", value );
			css_box_style_set( style, "padding-right", value );
			css_box_style_set( style, "padding-bottom", value );
			css_box_style_set( style, "padding-left", value );
		} else if( ! values[ 2 ] ) {
			/* only 2 given */
			css_box_style_set( style, "padding-top", values[ 0 ] );
			css_box_style_set( style, "padding-right", values[ 1 ] );
			css_box_style_set( style, "padding-bottom", values[ 0 ] );
			css_box_style_set( style, "padding-left", values[ 1 ] );
		} else if( ! values[ 3 ] ) {
			css_box_style_set( style, "padding-top", values[ 0 ] );
			css_box_style_set( style, "padding-right", values[ 1 ] );
			css_box_style_set( style, "padding-bottom", values[ 2 ] );
			css_box_style_set( style, "padding-left", values[ 1 ] );
		} else {
			css_box_style_set( style, "padding-top", values[ 0 ] );
			css_box_style_set( style, "padding-right", values[ 1 ] );
			css_box_style_set( style, "padding-bottom", values[ 2 ] );
			css_box_style_set( style, "padding-left", values[ 3 ] );
		}
	 } else if( ! strcmp( param, "padding-top" ) ) { 
		g_free( style->box->padding[ TOP ] );
		style->box->padding[ TOP ] = g_strdup( value );
	 } else if( ! strcmp( param, "padding-right" ) ) { 
		g_free( style->box->padding[ RIGHT ] );
		style->box->padding[ RIGHT ] = g_strdup( value );
	 } else if( ! strcmp( param, "padding-bottom" ) ) { 
		g_free( style->box->padding[ BOTTOM ] );
		style->box->padding[ BOTTOM ] = g_strdup( value );
	 } else if( ! strcmp( param, "padding-left" ) ) { 
		g_free( style->box->padding[ LEFT ] );
		style->box->padding[ LEFT ] = g_strdup( value );
	 } else if( ! strcmp( param, "border-top-color" ) ) { 
		g_free( style->box->border_color[ TOP ] );
		style->box->border_color[ TOP ] = g_strdup( value );
         } else if( ! strcmp( param, "border-right-color" ) ) { 
		g_free( style->box->border_color[ RIGHT ] );
		style->box->border_color[ RIGHT ] = g_strdup( value );
         } else if( ! strcmp( param, "border-bottom-color" ) ) { 
		g_free( style->box->border_color[ BOTTOM ] );
		style->box->border_color[ BOTTOM ] = g_strdup( value );
         } else if( ! strcmp( param, "border-left-color" ) ) { 
		g_free( style->box->border_color[ LEFT ] );
		style->box->border_color[ LEFT ] = g_strdup( value );
         } else if( ! strcmp( param, "border-top-width" ) ) {
 		g_free( style->box->border_width[ TOP ] );
		style->box->border_width[ TOP ] = g_strdup( value );
         } else if( ! strcmp( param, "border-right-width" ) ) {
  		g_free( style->box->border_width[ RIGHT ] );
		style->box->border_width[ RIGHT ] = g_strdup( value );
         } else if( ! strcmp( param, "border-bottom-width" ) ) { 
 		g_free( style->box->border_width[ BOTTOM ] );
		style->box->border_width[ BOTTOM ] = g_strdup( value );
         } else if( ! strcmp( param, "border-left-width" ) ) { 
 		g_free( style->box->border_width[ LEFT ] );
		style->box->border_width[ LEFT ] = g_strdup( value );
         } else if( ! strcmp( param, "border-top-style" ) ) { 
 		g_free( style->box->border_style[ TOP ] );
		style->box->border_style[ TOP ] = g_strdup( value );
	 } else if( ! strcmp( param, "border-right-style" ) ) { 
 		g_free( style->box->border_style[ RIGHT ] );
		style->box->border_style[ RIGHT ] = g_strdup( value );
	 } else if( ! strcmp( param, "border-bottom-style" ) ) { 
 		g_free( style->box->border_style[ BOTTOM ] );
		style->box->border_style[ BOTTOM ] = g_strdup( value );
	 } else if( ! strcmp( param, "border-left-style" ) ) { 
 		g_free( style->box->border_style[ LEFT ] );
		style->box->border_style[ LEFT ] = g_strdup( value );
	 } else if( ! strcmp( param, "width" ) ) {
		 g_free( style->box->width );
		 style->box->width = g_strdup( value );
	 } else if( ! strcmp( param, "height" ) ) {
		 g_free( style->box->height );
		 style->box->height = g_strdup( value );
	 } else if( ! strcmp( param, "float" ) ) {
		 g_free( style->box->float_ );
		 style->box->float_ = g_strdup( value );
	 } else if( ! strcmp( param, "clear" ) ) {
		 g_free( style->box->clear );
		 style->box->clear = g_strdup( value );
	 } 
	/* deal with border-* */
	else if( ! strcmp( param, "border" ) ) {
		/* sets all borders */
		css_box_style_set( style, "border-top", value );
		css_box_style_set( style, "border-left", value );
		css_box_style_set( style, "border-right", value );
		css_box_style_set( style, "border-bottom", value );
	}
	else if( ! strcmp( param, "border-top" ) ) {
		if( values[ 0 ] ) {
			g_free( style->box->border_width[ TOP ] ); 
			style->box->border_width[ TOP ] =
				g_strdup( values[ 0 ] );
		
			if( values[ 1 ] ) {
				g_free( style->box->border_style[ TOP ] ); 
				style->box->border_style[ TOP ] =
					g_strdup( values[ 1 ] );
				if( values[ 2 ] ) {
					g_free(style->box->border_color[TOP]); 
					style->box->border_color[ TOP ] =
						g_strdup( values[ 2 ] );
				}
			}
		}
	} else if( ! strcmp( param, "border-left" ) ) {
		if( values[ 0 ] ) {
			g_free( style->box->border_width[ LEFT ] ); 
			style->box->border_width[ LEFT ] =
				g_strdup( values[ 0 ] );
	       		if( values[ 1 ] ) {
				g_free( style->box->border_style[ LEFT ] ); 
				style->box->border_style[ LEFT ] =
					g_strdup( values[ 1 ] );
				if( values[ 2 ] ) {
					g_free(style->box->border_color[LEFT]);
					style->box->border_color[ LEFT ] =
						g_strdup( values[ 2 ] );
				}
			}
		}
	} else if( ! strcmp( param, "border-bottom" ) ) {
		if( values[ 0 ] ) {
			g_free( style->box->border_width[ BOTTOM ] ); 
			style->box->border_width[ BOTTOM ] =
				g_strdup( values[ 0 ] );
			if( values[ 1 ] ) {
				g_free( style->box->border_style[ BOTTOM ] ); 
				style->box->border_style[ BOTTOM ] =
					g_strdup( values[ 1 ] );
				if( values[ 2 ] ) {
					g_free(style->box->border_color[BOTTOM]); 
					style->box->border_color[ BOTTOM ] =
						g_strdup( values[ 2 ] );
				}
			}
		}
	} else if( ! strcmp( param, "border-right" ) ) {
		if( values[ 0 ] ) {
			g_free( style->box->border_width[ RIGHT ] ); 
			style->box->border_width[ RIGHT ] =
				g_strdup( values[ 0 ] );
			if( values[ 1 ] ) {
				g_free( style->box->border_style[ RIGHT ] ); 
				style->box->border_style[ RIGHT ] =
					g_strdup( values[ 1 ] );
				if( values[ 2 ] ) {
					g_free(style->box->border_color[RIGHT]); 
					style->box->border_color[ RIGHT ] =
						g_strdup( values[ 2 ] );
				}
			}
		}
	}
	g_strfreev( values );
}

void css_class_style_set( Style *style, const gchar *param, 
			  const gchar *value )
{
	gchar *temp;

	if( ! style->class )
		style->class = css_class_style_new();
	
	if( ! strcmp( param, "display" ) ) {
		g_free( style->class->display );
		style->class->display = g_strdup( value );
	} else if( ! strcmp( param, "white-space" ) ) { 
		g_free( style->class->white_space );
		style->class->white_space = g_strdup( value );
	 } else if( ! strcmp( param, "list-style-type" ) ) { 
		g_free( style->class->list_style_type );
		style->class->list_style_type = g_strdup( value );
	} else if( ! strcmp( param, "list-style-image" ) ) {
		if( ! strncmp( "url(", value, strlen( "url(" ) ) ) {
			temp = g_strdup( value + strlen( "url(" ) );
			temp[ strlen( temp ) - 1 ] = '\0';
			value = temp;
		} else {
			temp = NULL;
		}
		/* convert value to a path relative to the stylesheet */
		g_free( style->class->list_style_image );
		style->class->list_style_image = css_relative_path( value );
		g_free( temp );
	} else if( ! strcmp( param, "list-style-position" ) ) { 
		g_free( style->class->list_style_position );
		style->class->list_style_position = g_strdup( value );
	}
}

void input_style( const gchar* styleName, const gchar *styleData, 
		  Style *style )
{
	gchar *temp;

	if( ( ! styleName ) || ( ! styleData ) || ( ! style ) ) {
		return;
	}

	/* Font */
	if( ! strcmp( styleName, "font" ) ||
	    ! strcmp( styleName, "font-family" ) ||
	    ! strcmp( styleName, "font-style" ) ||
	    ! strcmp( styleName, "font-variant" ) ||
	    ! strcmp( styleName, "font-weight" ) ||
	    ! strcmp( styleName, "font-size" ) ||
	    ! strcmp( styleName, "color" ) ) {
		css_font_style_set( style, styleName, styleData );
	    }
	/* Background */
	else if( ! strcmp( styleName, "background" ) ||
		 ! strcmp( styleName, "background-color" ) ||
		 ! strcmp( styleName, "background-image" ) ||
		 ! strcmp( styleName, "background-repeat" ) ||
		 ! strcmp( styleName, "background-attachment" ) ||
		 ! strcmp( styleName, "background-position" ) ) {
		css_back_style_set( style, styleName, styleData );
		 }
       	/* Text */
	else if( ! strcmp( styleName, "word-spacing" ) ||
		 ! strcmp( styleName, "letter-spacing" ) ||
		 ! strcmp( styleName, "text-decoration" ) ||
		 ! strcmp( styleName, "vertical-align" ) ||
		 ! strcmp( styleName, "text-transform" ) ||
		 ! strcmp( styleName, "text-align" ) ||
		 ! strcmp( styleName, "text-indent" ) ||
		 ! strcmp( styleName, "line-height" ) ) {
		css_text_style_set( style, styleName, styleData );
		 }
	/* box, border, size place */
	else if( ! strcmp( styleName, "border-top-color" ) ||
		 ! strcmp( styleName, "border-right-color" ) ||
		 ! strcmp( styleName, "border-bottom-color" ) ||
		 ! strcmp( styleName, "border-left-color" ) ||
		 ! strcmp( styleName, "border-top-width" ) ||
		 ! strcmp( styleName, "border-right-width" ) ||
		 ! strcmp( styleName, "border-bottom-width" ) ||
		 ! strcmp( styleName, "border-left-width" ) ||
		 ! strcmp( styleName, "border-top-style" ) ||
		 ! strcmp( styleName, "border-right-style" ) ||
		 ! strcmp( styleName, "border-bottom-style" ) ||
		 ! strcmp( styleName, "border-left-style" ) ||
		 ! strcmp( styleName, "margin-top" ) ||
		 ! strcmp( styleName, "margin-right" ) ||
		 ! strcmp( styleName, "margin-bottom" ) ||
		 ! strcmp( styleName, "margin-left" ) ||
		 ! strcmp( styleName, "padding-top" ) ||
		 ! strcmp( styleName, "padding-right" ) ||
		 ! strcmp( styleName, "padding-bottom" ) ||
		 ! strcmp( styleName, "padding-left" ) ||
		 ! strcmp( styleName, "width" ) ||
		 ! strcmp( styleName, "height" ) ||
		 ! strcmp( styleName, "float" ) ||
		 ! strcmp( styleName, "clear" ) ||
		 ! strncmp( styleName, "margin", strlen( "margin" ) ) ||
		 ! strncmp( styleName, "padding", strlen( "padding" ) ) ||
		 ! strncmp( styleName, "border", strlen( "border" ) ) ) {
		 css_box_style_set( style, styleName, styleData );
		 }
	/* Classification */
	else if( ! strcmp( styleName, "display" ) ||
		 ! strcmp( styleName, "white-space" ) ||
		 ! strcmp( styleName, "list-style-type" ) ||
		 ! strcmp( styleName, "list-style-image" ) ||
		 ! strcmp( styleName, "list-style-position" ) ) {
		css_class_style_set( style, styleName, styleData );
		 }
	else {
		/* we don't know what it is! */
		temp = style->unknown;
		if( ! temp ) {
			temp = g_strdup( "" );
		}
		
		style->unknown = g_strconcat( temp, "\n\t",
					      styleName, ": ",
					      styleData, ";", NULL );
		g_free( temp );
	}
}

void parse_style_data( Style *style, const gchar *data )
{
	GString *cssBuffer;
	gchar *styleName;

	cssBuffer = g_string_new( "" );
	styleName = NULL;
	while( *data != '\0' ) {
		if( *data == '/' ) {
			const gchar *needle;
			if( *( data + 1 ) == '*' ) {
				needle = "*/";
			} else if( *( data + 1 ) == '/' ) {
				needle = "\n";
			} else {
				needle = NULL;
			}
			if( needle ) {
				/* comment, jump to end */
				gchar *end;
				end = strstr( data + 1, needle );
				if( ! end ) {
					break;
				}
				data = end + strlen( needle );
			}
		}
		switch( *data ) {
		case ':':
			styleName = g_strdup( cssBuffer->str );
			g_string_assign( cssBuffer, "" );
			break;
		case ';':
			input_style( styleName, cssBuffer->str, style );
			g_string_assign( cssBuffer, "" );
			styleName = NULL;
			g_free( styleName );
			break;
		case ' ':
			/* depends if we are handling style data or not,
			   if we are we want spaces */
			if( styleName && *cssBuffer->str != '\0' ) {
				g_string_append_c( cssBuffer, ' ' );
			}
			
			break;
		case '\t':
		case '\n':
		case '\r':
			break;
		default:
			g_string_append_c( cssBuffer,*data );
			break;
		}
		data ++;
	}
	
	g_string_free( cssBuffer, TRUE );
}
