/*  Screem:  cssWizard.c,
 *  A css1 wizard
 *
 *  Copyright (C) 1999, 2002  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>

#include <gmodule.h>

#include <string.h>

#include <bonobo/bonobo-file-selector-util.h>

#include <libgnome/gnome-i18n.h>

#include <libgnomeui/gnome-popup-menu.h>
#include <libgnomeui/gnome-file-entry.h>
#include <libgnomeui/gnome-entry.h>

#include <gtk/gtkentry.h>
#include <gtk/gtktreeview.h>
#include <gtk/gtktreemodel.h>
#include <gtk/gtktreestore.h>
#include <gtk/gtktreeselection.h>
#include <gtk/gtkdialog.h>

#include <glade/glade.h>


#include "screem-plugin.h"
#include "screem-window.h"
#include "screem-editor.h"

#include "support.h"
#include "fileops.h"

#include "css.h"

#include "css-window.h"

#include "libegg/menu/egg-menu.h"

/* keep track of how many css wizards we have */
static GList *wizards = NULL;

/* Required Plugin parts */

/* we don't want to try and load old plugins
   so this is a version symbol for screem to spot */
int screem_plugin_version = 2;

#define NAME "CSS Wizard"
#define AUTHOR "David A Knight (david@screem.org)"
#define PLUGIN_VERSION "2.0"
#define TAG NULL

G_MODULE_EXPORT const gchar* g_module_check_init( GModule *module );
G_MODULE_EXPORT void g_module_unload( GModule *module );
G_MODULE_EXPORT void init( ScreemPlugin *plugin );
G_MODULE_EXPORT void add_ui( GtkWidget *window, GtkWidget *editor,
			     GtkWidget *preview, GtkWidget *link_view );
G_MODULE_EXPORT void remove_ui( GtkWidget *window, GtkWidget *editor,
				GtkWidget *preview, GtkWidget *link_view );
void css_wizard_display( EggAction *action, gpointer user_data );
void css_wizard_clicked( GtkWidget *widget, gint button, gpointer data );


G_MODULE_EXPORT const gchar* g_module_check_init( GModule *module )
{
	return NULL;
}

G_MODULE_EXPORT void g_module_unload( GModule *module )
{
}

G_MODULE_EXPORT void init( ScreemPlugin *plugin )
{
	plugin->name = NAME;
	plugin->author = AUTHOR;
	plugin->version = PLUGIN_VERSION;

	/* these can all be left out if not required */
	plugin->tag = TAG;

	g_module_symbol( plugin->module, "popup", 
			 (gpointer*)&plugin->popup );
	g_module_symbol( plugin->module, "add_ui", 
			 (gpointer*)&plugin->add_ui );
	g_module_symbol( plugin->module, "remove_ui", 
			 (gpointer*)&plugin->remove_ui );
}

/* not needed if nothing is being added to the UI (unusual) */
G_MODULE_EXPORT void add_ui( GtkWidget *window, GtkWidget *editor,
			     GtkWidget *preview, GtkWidget *link_view )
{
	CSSWizard *css_wizard;
	ScreemWindow *swindow;
	const gchar *ui = "\
<Root>\
<menu>\
<submenu name=\"Insert\" verb=\"Insert\">\
<submenu name=\"Wizards\" verb=\"Wizards\">\
<menuitem name=\"CSSWizard\" verb=\"CSSWizard\"/>\
</submenu>\
</submenu>\
</menu>\
<dockitem name=\"Wizards Toolbar\">\
<toolitem name=\"CSSWizard\" verb=\"CSSWizard\" />\
</dockitem>\
</Root>";
	static EggActionGroupEntry entries[] = {
		{ "CSSWizard", "label",
	  	GTK_STOCK_PROPERTIES, NULL, "tip",
	  	G_CALLBACK( css_wizard_display ), NULL },
	};
	EggActionGroupEntry *entry;

	gchar *label;
	gchar *tip;

	css_wizard = g_new0( CSSWizard, 1 );
	swindow = SCREEM_WINDOW( window );
		
	label = g_strdup( _( "CSS Wizard" ) );
	tip = g_strdup( _( "Cascading Style Sheet Editor" ) );
	
	entry = entries;
	entry->label = label;
	entry->tooltip = tip;
	entry->user_data = css_wizard;
	
	egg_action_group_add_actions( EGG_ACTION_GROUP( swindow->action_group ),
					entry, 1 );
	egg_menu_merge_add_ui_from_string( EGG_MENU_MERGE( swindow->merge ),
						ui, strlen( ui ), NULL );
	g_free( label );
	g_free( tip );

	/* create the ScreemCSSWindow */
	css_wizard->csswindow = screem_css_window_new();
	screem_css_window_constructed( css_wizard->csswindow,
				       SCREEM_WINDOW( window ),
				       SCREEM_EDITOR( editor ) );


	g_object_set_data( G_OBJECT( css_wizard->csswindow ),
			   "css_wizard", css_wizard );
	
	wizards = g_list_append( wizards, css_wizard );
}

/* required it add_ui is present */
G_MODULE_EXPORT void remove_ui( GtkWidget *window, GtkWidget *editor,
				GtkWidget *preview, GtkWidget *link_view )
{
	GList *list;
	CSSWizard *css_wizard;

	for( list = wizards; list; list = list->next ) {
		css_wizard = (CSSWizard*)list->data;
		if( css_wizard->csswindow->window == SCREEM_WINDOW(window) ) {
			/* got it */
			break;
		}
	}
       
	g_return_if_fail( list != NULL );

	/* css_wizard is the one to erase */
	wizards = g_list_remove( wizards, css_wizard );
	g_free( css_wizard );
}

/* End of required section */

static gchar* create_css( ScreemCSSWindow *window, GtkTreeIter *n );

static void store_current( ScreemCSSWindow *window );
static void set_current( ScreemCSSWindow *window );


void css_wizard_display( EggAction *action, gpointer user_data )
{
	CSSWizard *wizard = (CSSWizard*)user_data;

	if( ! wizard->csswindow->filename ) {
		gtk_window_set_title( GTK_WINDOW( wizard->csswindow ), 
					_( "Untitled - Screem CSS Wizard" ) );
	} else {
		gchar *title;

		title = g_strconcat( wizard->csswindow->filename,
					" - Screem CSS Wizard", NULL );
		gtk_window_set_title( GTK_WINDOW( wizard->csswindow ), title );
		g_free( title );
	}
	
	gtk_widget_show( GTK_WIDGET( wizard->csswindow ) );
}


void screem_css_dump_stylesheet( ScreemCSSWindow *window )
{
	GtkWidget *tree;
	GtkTreeIter n;

	tree = glade_xml_get_widget( window->xml, "styles" );

	if( gtk_tree_model_get_iter_first( window->model, &n ) ) {
		gchar *data;

		store_current( window );
		
		/* recursivley create css data */
		data = create_css( window, &n );
		
		screem_editor_insert( window->editor, -1, data );
		
		g_free( data );
	}
}

void screem_css_dump_style( ScreemCSSWindow *window )
{
	GtkWidget *tree;
	GtkTreeIter it;
	GtkTreeSelection *selection;
	GtkTreeModel *model;
	gchar *data;
	Style *style;
	GValue value = { 0 };
	
	tree = glade_xml_get_widget( window->xml, "styles" );

	tree = glade_xml_get_widget( window->xml, "styles" );
	selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( tree ) );
	
	if( ! gtk_tree_selection_get_selected( selection, &model, &it ) ) {
		return;
	}
	
	store_current( window );

	gtk_tree_model_get_value( model, &it, COL_STYLE, &value );
	style = (Style*)g_value_get_pointer( &value );

	data = css_style_output( style );
	screem_editor_insert( window->editor, -1, data );
	g_free( data );
}

void screem_css_style_selected( GtkTreeView *view, gboolean active )
{
	ScreemCSSWindow *window;
	GtkWidget *notebook;

	window = SCREEM_CSS_WINDOW( g_object_get_data( G_OBJECT( view ),
						       "window" ) );

	/* store details of currently selected node */
	store_current( window );
	
	/* enter details of clicked node */
	set_current( window );

	notebook = glade_xml_get_widget( window->xml, "properties" );
	gtk_widget_set_sensitive( notebook, active );
}

gboolean screem_css_style_clicked( GtkWidget *widget, GdkEventButton *event )
{
	if( event->button == 3 ) {
		ScreemCSSWindow *window;
		GtkWidget *menu;
		GtkTreeSelection *selection;
		GtkTreePath *path;
		GtkTreeViewColumn *column;
		guint x;
		guint y;

		g_signal_stop_emission_by_name( G_OBJECT( widget ),
						"button_press_event" );

		window = SCREEM_CSS_WINDOW( g_object_get_data( G_OBJECT(widget),
					    "window" ) );

		selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(widget));
			
		if( gtk_tree_view_get_path_at_pos( GTK_TREE_VIEW( widget ),
					       	event->x, event->y,
					       	&path, &column, &x, &y ) ) {
			gtk_tree_selection_select_path( selection, path );
			gtk_tree_path_free( path );

			/* popup menu */
			menu = gtk_menu_new();
			bonobo_window_add_popup( BONOBO_WINDOW( window ),
					 	GTK_MENU( menu ),
					 	"/popups/stylemenu" );
			gnome_popup_menu_do_popup_modal( menu, 0, 0, event, 0, 
						 	widget );
		}
	}

	return ( event->button == 3 );
}

void screem_css_display_style( Style *style, ScreemCSSWindow *window )
{
	GtkWidget *tree;
	GtkTreeIter it;
	GtkTreeSelection *selection;

	tree = glade_xml_get_widget( window->xml, "styles" );
	selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( tree ) );

	gtk_tree_store_append( GTK_TREE_STORE( window->model ), &it, NULL );
	gtk_tree_store_set( GTK_TREE_STORE( window->model ), &it,
			    COL_PATTERN, style->pattern,
			    COL_STYLE, style,
			    -1 );

	/* store details of currently selected node */
	store_current( window );

	gtk_tree_selection_select_iter( selection, &it );
}

void screem_css_remove_style( GtkTreeIter *it, ScreemCSSWindow *window )
{
	Style *style;
	GtkWidget *tree;
	GValue value = { 0 };

	tree = glade_xml_get_widget( window->xml, "styles" );

	gtk_tree_model_get_value( window->model, it,
				  COL_STYLE, &value );
	style = g_value_get_pointer( &value );
	g_value_unset( &value );

	css_style_destroy( style );	
}

gboolean screem_css_page_switch( GtkNotebook *notebook, GtkNotebookPage *page, 
				 guint page_num )
{
	GladeXML *xml;
	GtkWidget *widget;
	ScreemCSSWindow *window;

	xml = glade_get_widget_tree( GTK_WIDGET( notebook ) );
	if( xml ) {
		widget = glade_xml_get_widget( xml, "styles" );
		
		window = SCREEM_CSS_WINDOW( g_object_get_data( G_OBJECT( widget ),
							       "window" ) );
		
		if( ! window->closing ) {
			store_current( window );
			
			window->current_page = page_num;
			set_current( window );
		}
	}

	return TRUE;
}

void screem_css_color_set( GtkWidget *entry, guint r, guint g, guint b, guint a,
			    gpointer data )
{
	gchar *colour;
	
	colour = g_strdup_printf( "#%.2x%.2x%.2x", r >> 8, g >> 8, b >> 8 );

	gtk_entry_set_text( GTK_ENTRY( entry ), colour );

	g_free( colour );
}

void screem_css_new( ScreemCSSWindow *window, const gchar *pathname )
{
	GtkWidget *tree;
	gchar *filename;
	gchar *currentfile;
	GtkTreeIter it;

	if( ! pathname || *pathname == '\0' ) {
		/* we want a filename for the new stylesheet */
		filename = bonobo_file_selector_open( GTK_WINDOW( window ),
						      TRUE,
						      _( "Select filename for the new stylesheet" ),
						      NULL, NULL );
		
		if( ! filename )
			return;
		else if(  ! overwrite( filename ) ) {
			g_free( filename );
			return;
		}
	} else {
		filename = g_strdup( pathname );
	}

	tree = glade_xml_get_widget( window->xml, "styles" );
	
	currentfile = g_object_get_data( G_OBJECT( tree ), "filename" );
	if( currentfile ) {
		GladeXML *xml;
		GtkWidget *dialog;
		
		xml = glade_xml_new( GLADE_PATH"/css.glade",
				     "savemessage", NULL );
		dialog = glade_xml_get_widget( xml, "savemessage" );
		gtk_window_set_wmclass( GTK_WINDOW( dialog ), "Screem",
					"csssavemessage" );
		
		switch( gtk_dialog_run( GTK_DIALOG( dialog ) ) ) {
		case 2:
			g_free( filename );
			return;
			break;
		case 0:
			 screem_css_save( window );
			 break;
		default:
			break;
		}
		g_free( currentfile );
	}
	
	if( gtk_tree_model_get_iter_first( window->model, &it ) ) {
		screem_css_remove_style( &it, window );
	}

	if( window->filename )
		g_free( window->filename );
	window->filename = filename;
}

void screem_css_load( ScreemCSSWindow *window )
{
	gchar *filename;
	GString *entireFile;

	/* Open File Selection Window */
	filename = bonobo_file_selector_open( GTK_WINDOW( window ), TRUE,
					      _("Select CSS file to Open"),
					      NULL, NULL );

	entireFile = NULL;

        if( filename ) {
		gedit_recent_add( window->recent_sheets, filename );
            	entireFile = load_file( filename );


		if( window->filename ) {
			g_free( window->filename );
		}
		window->filename = NULL;

		if( entireFile ) {
			gchar *title;

			css_parse( window, entireFile );
			g_string_free( entireFile, TRUE );
			title = g_strconcat( filename,
						" - Screem CSS Wizard", NULL );
			gtk_window_set_title( GTK_WINDOW( window ), title );
			g_free( title );
		}
		window->filename = filename;
	}
}

void screem_css_save( ScreemCSSWindow *window )
{
	GtkWidget *tree;
	gchar *title;

	tree = glade_xml_get_widget( window->xml, "styles" );

	/* if no filename is set then do a save as */
	if( ! window->filename ) {
		screem_css_save_as( window );
		/* css_save_as () calls css_save so
		   we are done */
	} else {
		gchar *data;
		GtkTreeIter it;

		store_current( window );
		
		gtk_tree_model_get_iter_first( window->model, &it );

		/* recursivley create css file */
		data = create_css( window, &it );
		
		if( ! save_file( window->filename, data,
				 GNOME_VFS_PERM_USER_READ | 
				 GNOME_VFS_PERM_USER_WRITE |
				 GNOME_VFS_PERM_GROUP_READ |
				 GNOME_VFS_PERM_GROUP_WRITE |
				 GNOME_VFS_PERM_OTHER_READ ) ) {
			/* FIXME: alert user */
		}

		g_free( data );
	}

	title = g_strconcat( window->filename,
				" - Screem CSS Wizard", NULL );
	gtk_window_set_title( GTK_WINDOW( window ), title );
	g_free( title );
}

void screem_css_save_as( ScreemCSSWindow *window )
{
	GtkWidget *tree;
	gchar *filename;

	tree = glade_xml_get_widget( window->xml, "styles" );

	filename = bonobo_file_selector_save( GTK_WINDOW( window ),
					      TRUE,
					      _("Select filename for CSS file"),
					      NULL, NULL, NULL );

        if( ! filename )
                return;

	if( window->filename )
		g_free( window->filename );
	window->filename = filename;

	screem_css_save( window );
}

static gchar* create_css( ScreemCSSWindow *window, GtkTreeIter *n )
{
	gchar *data;
	gchar *temp;
       	gchar *retval;
	Style *style;

	GValue value = { 0 };

	if( ! n )
		return NULL;

	/* create the style text for this node */
	gtk_tree_model_get_value( window->model, n, COL_STYLE, &value );
	style = g_value_get_pointer( &value );
	g_value_unset( &value );

	data = css_style_output( style );
		
	retval = g_strdup( data );

	g_free( data );

	if( gtk_tree_model_iter_next( window->model, n ) ) {
		temp = create_css( window, n );
	
		if( temp ) {
			data = retval;
			retval = g_strconcat( data, "\n", temp, NULL );
			g_free( temp );
			g_free( data );
		}
	}

	return retval;
}

static void store_current( ScreemCSSWindow *window )
{
	Style *style;
	GtkWidget *widget;
	gint i;

	const gchar *temp;

	const gchar *font[] = {
		"font-family",
		"font-style",
		"font-variant",
		"font-weight",
		"font-size",
		"color",
		NULL
	};
	const gchar *background[] = {
		"background-color",
		"background-image",
		"background-repeat",
		"background-attachment",
		"background-position",
		NULL
	};
	const gchar *text[] = {
		"word-spacing",
		"letter-spacing",
		"text-decoration",
		"vertical-align",
		"text-transform",
		"text-align",
		"text-indent",
		"line-height",
		NULL
	};
	const gchar *box[] = {
		"margin-top",
		"margin-right",
		"margin-bottom",
		"margin-left",
		"padding-top",
		"padding-right",
		"padding-bottom",
		"padding-left",
		NULL
	};
	const gchar *border[] = {
		"border-top-color",
		"border-right-color",
		"border-bottom-color",
		"border-left-color",
		"border-top-width",
		"border-right-width",
		"border-bottom-width",
		"border-left-width",
		"border-top-style",
		"border-right-style",
		"border-bottom-style",
		"border-left-style",
		NULL
	};
	const gchar *sizeplace[] = {
		"width",
		"height",
		"float",
		"clear",
		NULL
	};
	const gchar *classification[] = {
		"display",
		"white-space",
		"list-style-type",
		"list-style-image",
		"list-style-position",
		NULL
	};

	style = window->current;
	
	if( ! style ) {
		return;
	}

	/* store details from previous page */
     	switch( window->current_page ) {
	case CSS_FONT:
		for( i = 0; font[ i ]; i ++ ) {
			widget = glade_xml_get_widget( window->xml, 
						       font[ i ] );
			if( GNOME_IS_ENTRY( widget ) )
				widget = gnome_entry_gtk_entry( GNOME_ENTRY( widget ) );
			else
				widget = GTK_COMBO( widget )->entry;

			temp = gtk_entry_get_text( GTK_ENTRY(widget) );
			input_style( font[ i ], temp, style );
		}
		break;
	case CSS_BACKGROUND:
		for( i = 0; background[ i ]; i ++ ) {
			widget = glade_xml_get_widget( window->xml,
						       background[ i ] );
			if( GNOME_IS_ENTRY( widget ) )
				widget = gnome_entry_gtk_entry( GNOME_ENTRY( widget ) );
			else if( GNOME_IS_FILE_ENTRY( widget ) )
				widget = gnome_file_entry_gtk_entry( GNOME_FILE_ENTRY( widget ) );
			else
				widget = GTK_COMBO( widget )->entry;

			temp = gtk_entry_get_text( GTK_ENTRY(widget) );
			input_style( background[ i ], temp, style );
		}
		break;
	case CSS_TEXT:
		for( i = 0; text[ i ]; i ++ ) {
			widget = glade_xml_get_widget( window->xml,
						       text[ i ] );
			widget = GTK_COMBO( widget )->entry;
			temp = gtk_entry_get_text( GTK_ENTRY(widget) );
			input_style( text[ i ], temp, style );
		}
		break;
	case CSS_BOX: 
		for( i = 0; box[ i ]; i ++ ) {
			widget = glade_xml_get_widget( window->xml, box[ i ] );
			widget = GTK_COMBO( widget )->entry;
			temp = gtk_entry_get_text( GTK_ENTRY(widget) );
			input_style( box[ i ], temp, style );
		}
		break;
	case CSS_BORDER: 
		for( i = 0; border[ i ]; i ++ ) {
			widget = glade_xml_get_widget( window->xml, border[ i ] );
			if( GNOME_IS_ENTRY( widget ) )
				widget = gnome_entry_gtk_entry( GNOME_ENTRY( widget ) );
			else
				widget = GTK_COMBO( widget )->entry;
			temp = gtk_entry_get_text( GTK_ENTRY(widget) );
			input_style( border[ i ], temp, style );
		}
		break;
	case CSS_SIZEPLACE:
		for( i = 0; sizeplace[ i ]; i ++ ) {
			widget = glade_xml_get_widget( window->xml, sizeplace[ i ] );
			widget = GTK_COMBO( widget )->entry;
			temp = gtk_entry_get_text( GTK_ENTRY(widget) );
			input_style( sizeplace[ i ], temp, style );
		}
		break;
	case CSS_CLASSIFICATION:
		for( i = 0; classification[ i ]; i ++ ) {
			widget = glade_xml_get_widget( window->xml,
						       classification[ i ] );
			if( GNOME_IS_FILE_ENTRY( widget ) )
				widget = gnome_file_entry_gtk_entry( GNOME_FILE_ENTRY( widget ) );
			else
				widget = GTK_COMBO( widget )->entry;
			temp = gtk_entry_get_text( GTK_ENTRY(widget) );
			input_style( classification[ i ], temp, style );
		}
		break;
	}
}

static void set_current( ScreemCSSWindow *window )
{
	Style *style;
	GtkWidget *tree;
	GtkWidget *widget;
	GtkTreeSelection *selection;
	GtkTreeModel *model;
	GtkTreeIter it;
	GValue value = { 0 };
	
	tree = glade_xml_get_widget( window->xml, "styles" );
	selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( tree ) );
	
	if( ! gtk_tree_selection_get_selected( selection, &model, &it ) ) {
		return;
	}
	
	gtk_tree_model_get_value( model, &it, COL_STYLE, &value );
	style = g_value_get_pointer( &value );
	g_value_unset( &value );
	
	window->current = style;
	
	/* insert details for current_page */
	switch( window->current_page ) {
	case CSS_FONT:
		widget = glade_xml_get_widget( window->xml, "font-family" );
		widget = gnome_entry_gtk_entry( GNOME_ENTRY( widget ) );
		if( style->font && style->font->family ) {
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->font->family );
		} else {
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		}
 
		widget = glade_xml_get_widget( window->xml, "font-style" );
		widget = GTK_COMBO( widget )->entry;
		if( style->font && style->font->style ) {
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->font->style );
		} else {
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		}

		widget = glade_xml_get_widget( window->xml, "font-variant" );
		widget = GTK_COMBO( widget )->entry;
		if( style->font && style->font->variant ) {
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->font->variant );
		} else {
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		}
		widget = glade_xml_get_widget( window->xml, "font-weight" );
		widget = GTK_COMBO( widget )->entry;
		if( style->font && style->font->weight ) {
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->font->weight );
		} else	{
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		}

		widget = glade_xml_get_widget( window->xml, "font-size" );
		widget = GTK_COMBO( widget )->entry;
		if( style->font && style->font->size ) {
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->font->size );
		} else {
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		}
	
		widget = glade_xml_get_widget( window->xml, "color" );
		widget = gnome_entry_gtk_entry( GNOME_ENTRY( widget ) );
		if( style->font && style->font->color ) {
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->font->color );
		} else {
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		}
		break;
	case CSS_BACKGROUND:
		widget = glade_xml_get_widget( window->xml,
					       "background-color" );
		widget = gnome_entry_gtk_entry( GNOME_ENTRY( widget ) );
		if( style->back && style->back->color ) {
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->back->color );
		} else {
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		}
		
		widget = glade_xml_get_widget( window->xml,
					       "background-image" );
		widget = gnome_file_entry_gtk_entry(GNOME_FILE_ENTRY(widget));
		if( style->back && style->back->image ) {
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->back->image );
		} else {
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		}
		
		widget = glade_xml_get_widget( window->xml, 
					       "background-repeat" );
		widget = GTK_COMBO( widget )->entry;
		if( style->back && style->back->repeat ) {
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->back->repeat );
		} else {
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		}
		
		widget = glade_xml_get_widget( window->xml,
					       "background-attachment" );
		widget = GTK_COMBO( widget )->entry;
		if( style->back && style->back->attachment ) {
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->back->attachment );
		} else {
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		}
		widget = glade_xml_get_widget( window->xml,
					       "background-position" );
		widget = GTK_COMBO( widget )->entry;
		if( style->back && style->back->position ) {
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->back->position );
		} else {
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		}
		break;
	case CSS_TEXT:
		widget = glade_xml_get_widget( window->xml, "word-spacing" );
		widget = GTK_COMBO( widget )->entry;
		if( style->text && style->text->word_spacing ) {
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->text->word_spacing );
		} else {
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		}
		widget = glade_xml_get_widget( window->xml, "letter-spacing" );
		widget = GTK_COMBO( widget )->entry;
		if( style->text && style->text->letter_spacing ) {
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->text->letter_spacing );
		} else {
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		}
		widget = glade_xml_get_widget( window->xml, 
					       "text-decoration" );
		widget = GTK_COMBO( widget )->entry;
		if( style->text && style->text->decoration ) {
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->text->decoration );
		} else {
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		}
		widget = glade_xml_get_widget( window->xml, "vertical-align" );
		widget = GTK_COMBO( widget )->entry;
		if( style->text && style->text->vertical_align ) {
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->text->vertical_align );
		} else {
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		}
		widget = glade_xml_get_widget( window->xml, "text-transform" );
		widget = GTK_COMBO( widget )->entry;
		if( style->text && style->text->transform ) {
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->text->transform );
		} else {
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		}
		widget = glade_xml_get_widget( window->xml, "text-align" );
		widget = GTK_COMBO( widget )->entry;
		if( style->text && style->text->align ) {
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->text->align );
		} else {
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		}
		widget = glade_xml_get_widget( window->xml, "text-indent" );
		widget = GTK_COMBO( widget )->entry;
		if( style->text && style->text->indent ) {
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->text->indent );
		} else {
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		}
		widget = glade_xml_get_widget( window->xml, "line-height" );
		widget = GTK_COMBO( widget )->entry;
		if( style->text && style->text->height ) {
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->text->height );
		} else {
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		}
		break;
	case CSS_BOX: 
		widget = glade_xml_get_widget( window->xml, "margin-top" );
		widget = GTK_COMBO( widget )->entry;
		if( style->box && style->box->margin[ TOP ] ) {
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->box->margin[ TOP ] );
		} else {
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		}
		
		widget = glade_xml_get_widget( window->xml, "margin-right" );
		widget = GTK_COMBO( widget )->entry;
		if( style->box && style->box->margin[ RIGHT ] ) {
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->box->margin[ RIGHT ] );
		} else {
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		}
		widget = glade_xml_get_widget( window->xml, "margin-bottom" );
		widget = GTK_COMBO( widget )->entry;
		if( style->box && style->box->margin[ BOTTOM ] ) {
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->box->margin[ BOTTOM ] );
		} else {
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		}
		widget = glade_xml_get_widget( window->xml, "margin-left" );
		widget = GTK_COMBO( widget )->entry;
		if( style->box && style->box->margin[ LEFT ] ) {
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->box->margin[ LEFT ] );
		} else {
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		}
		widget = glade_xml_get_widget( window->xml, "padding-top" );
		widget = GTK_COMBO( widget )->entry;
		if( style->box && style->box->padding[ TOP ] ) {
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->box->padding[ TOP ] );
		} else {
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		}
		widget = glade_xml_get_widget( window->xml, "padding-right" );
		widget = GTK_COMBO( widget )->entry;
		if( style->box && style->box->padding[ RIGHT ] ) {
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->box->padding[ RIGHT ] );
		} else {
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		}
		
		widget = glade_xml_get_widget( window->xml, "padding-bottom" );
		widget = GTK_COMBO( widget )->entry;
		if( style->box && style->box->padding[ BOTTOM ] ) {
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->box->padding[ BOTTOM ] );
		} else {
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		}
		widget = glade_xml_get_widget( window->xml, "padding-left" );
		widget = GTK_COMBO( widget )->entry;
		if( style->box && style->box->padding[ LEFT ] ) {
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->box->padding[ LEFT ] );
		} else {
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		}
		break;
	case CSS_BORDER: 
		widget = glade_xml_get_widget( window->xml, "border-top-color" );
		widget = gnome_entry_gtk_entry( GNOME_ENTRY( widget ) );
		if( style->box && style->box->border_color[ TOP ] ) {
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->box->border_color[ TOP ] );
		} else {
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		}
		
		widget = glade_xml_get_widget( window->xml, "border-right-color" );
		widget = gnome_entry_gtk_entry( GNOME_ENTRY( widget ) );
		if( style->box && style->box->border_color[ RIGHT ] ) {
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->box->border_color[ RIGHT ]);
		} else {
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		}
		
		widget = glade_xml_get_widget( window->xml, "border-bottom-color" );
		widget = gnome_entry_gtk_entry( GNOME_ENTRY( widget ) );
		if( style->box && style->box->border_color[ BOTTOM ] ) {
			gtk_entry_set_text(GTK_ENTRY( widget ),
					   style->box->border_color[ BOTTOM ]);
		} else {
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		}
		
		widget = glade_xml_get_widget( window->xml, "border-left-color" );
		widget = gnome_entry_gtk_entry( GNOME_ENTRY( widget ) );
		if( style->box && style->box->border_color[ LEFT ] ) {
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->box->border_color[ LEFT ] );
		} else {
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		}
		
		widget = glade_xml_get_widget( window->xml, "border-top-width" );
		widget = GTK_COMBO( widget )->entry;
		if( style->box && style->box->border_width[ TOP ] ) {
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->box->border_width[ TOP ] );
		} else {
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		}
		
		widget = glade_xml_get_widget( window->xml, "border-right-width" );
		widget = GTK_COMBO( widget )->entry;
		if( style->box && style->box->border_width[ RIGHT ] ) {
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->box->border_width[ RIGHT ]);
		} else {
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		}
		
		widget = glade_xml_get_widget( window->xml, "border-bottom-width" );
		widget = GTK_COMBO( widget )->entry;
		if( style->box && style->box->border_width[ BOTTOM ] ) {
			gtk_entry_set_text(GTK_ENTRY( widget ),
					   style->box->border_width[ BOTTOM ]);
		} else {
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		}
		widget = glade_xml_get_widget( window->xml, "border-left-width" );
		widget = GTK_COMBO( widget )->entry;
		if( style->box && style->box->border_width[ LEFT ] ) {
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->box->border_width[ LEFT ] );
		} else {
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		}
		widget = glade_xml_get_widget( window->xml, "border-top-style" );
		widget = GTK_COMBO( widget )->entry;
		if( style->box && style->box->border_style[ TOP ] ) {
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->box->border_style[ TOP ] );
		} else {
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		}
		widget = glade_xml_get_widget( window->xml, "border-right-style" );
		widget = GTK_COMBO( widget )->entry;
		if( style->box && style->box->border_style[ RIGHT ] ) {
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->box->border_style[ RIGHT ]);
		} else {
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		}
		widget = glade_xml_get_widget( window->xml, "border-bottom-style" );
		widget = GTK_COMBO( widget )->entry;
		if( style->box && style->box->border_style[ BOTTOM ] ) {
			gtk_entry_set_text(GTK_ENTRY( widget ),
					   style->box->border_style[ BOTTOM ]);
		} else {
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		}
		widget = glade_xml_get_widget( window->xml, "border-left-style" );
		widget = GTK_COMBO( widget )->entry;
		if( style->box && style->box->border_style[ LEFT ] ) {
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->box->border_style[ LEFT ] );
		} else {
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		}
		break;
	case CSS_SIZEPLACE:
		widget = glade_xml_get_widget( window->xml, "width" );
		widget = GTK_COMBO( widget )->entry;
		if( style->box && style->box->width ) {
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->box->width );
		} else {
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );	
		}
		widget = glade_xml_get_widget( window->xml, "height" );
		widget = GTK_COMBO( widget )->entry;
		if( style->box && style->box->height ) {
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->box->height );
		} else {
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		}
		widget = glade_xml_get_widget( window->xml, "float" );
		widget = GTK_COMBO( widget )->entry;
		if( style->box && style->box->float_ ) {
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->box->float_ );
		} else {
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		}
		widget = glade_xml_get_widget( window->xml, "clear" );
		widget = GTK_COMBO( widget )->entry;
		if( style->box && style->box->clear ) {
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->box->clear );
		} else {
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		}
		break;
	case CSS_CLASSIFICATION:
		widget = glade_xml_get_widget( window->xml, "display" );
		widget = GTK_COMBO( widget )->entry;
		if( style->class && style->class->display ) {
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->class->display );
		} else {
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		}
		widget = glade_xml_get_widget( window->xml, "white-space" );
		widget = GTK_COMBO( widget )->entry;
		if( style->class && style->class->white_space ) {
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->class->white_space );
		} else {
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		}
		widget = glade_xml_get_widget( window->xml, 
					       "list-style-type" );
		widget = GTK_COMBO( widget )->entry;
		if( style->class && style->class->list_style_type ) {
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->class->list_style_type );
		} else {
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		}
		widget = glade_xml_get_widget( window->xml,
					       "list-style-image" );
		widget = gnome_file_entry_gtk_entry(GNOME_FILE_ENTRY(widget));
		if( style->class && style->class->list_style_image ) {
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->class->list_style_image );
		} else {
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		}
		widget = glade_xml_get_widget( window->xml,
					       "list-style-position" );
		widget = GTK_COMBO( widget )->entry;
		if( style->class && style->class->list_style_position ) {
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->class->list_style_position);
		} else {
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		}
		break;
	}
}

gchar* css_relative_path( const gchar *path )
{
	/* FIXME: this is called from *style_set(), but
	   they shouldn't take a ScreemCSSWindow, but thats
	   the only way we can get the path of the css file */
	return g_strdup( path );
#if 0
	ScreemCSSWindow *window;
	GtkWidget *tree;
	gchar *ret;
	gchar *dir;

       	if( ! g_path_is_absolute( path ) )
		return g_strdup( path );

	window = wizard->csswindow;

	tree = glade_xml_get_widget( window->xml, "styles" );

	dir = g_dirname( window->filename );
	ret = relative_path( path, dir );

	g_free( dir );

	return ret;
#endif
}

void css_parse( ScreemCSSWindow *window, GString *entireFile ) 
{
	const gchar *currentChar;
	GString	*cssBuffer; 
	Style *style;
	gchar *temp;

#if 0	
	{
		CssStylesheet *sheet;
		GSList *statements;
		
		html_atom_list = html_atom_list_new ();
        	html_global_atoms_initialize (html_atom_list);
		
		sheet = css_parser_parse_stylesheet( entireFile->str,
						     entireFile->len );
		for( statements = sheet->stat; statements; statements = statements->next ) {
			CssStatement *statement;
			gchar *output;
			
			statement = (CssStatement*)statements->data;
			
			output = css_debug_print_statement( statement );
			g_print( "OUTPUT: %s\n", output );

		}
		css_stylesheet_destroy( sheet );
	}
#endif
	currentChar = entireFile->str;
	
	cssBuffer = g_string_new( "" );
	while( currentChar && *currentChar != '\0' ) {
		switch (*currentChar){
		case '/':
			currentChar++;
			if( *currentChar == '*' ) {
				currentChar = strstr( currentChar, "*/" );
				if( ! currentChar ) {
					/* comment wasn't closed! */
					break;
				}
				currentChar += strlen( "*/" );
			} else if( *currentChar == '/' ) {
				currentChar = strstr( currentChar, "\n" );
				if( ! currentChar ) {
					break;
				}
				currentChar ++;
			}
			break;
		case '{':
			/* cssBuffer should hold a pattern */
			currentChar ++;
			temp = strchr( currentChar, '}' );
			if( temp ) {
				gchar *data;
				style = css_style_new( cssBuffer->str );
				data = g_strndup( currentChar, temp - currentChar - 1 );
				currentChar = temp + 1;
				parse_style_data( style, data );
				screem_css_display_style( style, window );
				g_free( data );
			} else {
				currentChar = NULL;
			}				
			g_string_assign( cssBuffer, "" );
			break;
		case '\n':
		case '\r':
		case '\t':
			if( cssBuffer->len > 0 ) {
				g_string_append_c( cssBuffer, ' ' );
			}
			currentChar++;
			break;
		default:
			g_string_append_c(cssBuffer,*currentChar);
			currentChar++;
			break;
		}
	}
	/* destroy GStrings */
	g_string_free( cssBuffer, TRUE );
}


#if 0

/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
    Copyright (C) 2000 CodeFactory AB
    Copyright (C) 2000 Jonas Borgstr\366m <jonas@codefactory.se>
    Copyright (C) 2000 Anders Carlsson <andersca@codefactory.se>
  
    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.
  
    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.
  
    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
    Boston, MA 02111-1307, USA.
 */

#include <libgtkhtml/css/cssvalue.h>
#include <libgtkhtml/css/cssparser.h>

static const gchar *css_dimensions[] = {
	NULL,
	"",
	"%",
	"em",
	"ex",
	"px",
	"cm",
	"mm",
	"in",
	"pt",
	"pc",
	"deg",
	"rad",
	"grad",
	"ms",
	"s",
	"Hz",
	"kHz"
};

gchar *css_debug_print_value( CssValue * val )
{
	CssValueEntry *entry;
	GString *ret;
	gchar *subret;
	
	if( ! val ) {
		return g_strdup( "" );
	}
	
	ret = g_string_new( "" );
	
	switch( val->value_type ) {
	case CSS_NUMBER:
	case CSS_CM:
	case CSS_EMS:
	case CSS_PX:
	case CSS_PT:
	case CSS_DEG:
	case CSS_PERCENTAGE:
	case CSS_EXS:
	case CSS_MM:
	case CSS_PC:
	case CSS_IN:
		g_string_append_printf( ret, "%g%s", val->v.d, css_dimensions[ val->value_type ] );
		break;
	case CSS_IDENT:
		g_string_append_printf( ret, "%s",
			 html_atom_list_get_string( html_atom_list,
						    val->v.atom ) );
		break;
	case CSS_STRING:
		g_string_append_printf( ret, "\"%s\"", val->v.s );
		break;
	case CSS_FUNCTION:
		g_string_append_printf( ret, "%s(",
			 html_atom_list_get_string( html_atom_list,
						    val->v.function->name ) );
		subret = css_debug_print_value( val->v.function->args );
		g_string_append( ret, subret );
		g_free( subret );
		g_string_append( ret, ")" );
		break;
	case CSS_VALUE_LIST:
		entry = val->v.entry;

		while( entry ) {
			subret = css_debug_print_value( entry->value );
			g_string_append( ret, subret );
			g_free( subret );
			
			if( entry->next ) {
				g_string_append_c( ret, entry->list_sep );
			}
			entry = entry->next;
		}
		break;
	default:
		g_warning ("css_debug_get_value: Unknown value type %d",
			   val->value_type);
	}
	
	subret = ret->str;
	g_string_free( ret, FALSE );
	
	return subret;
}

gchar *css_debug_print_simple_selector( CssSimpleSelector * ss )
{
	gint i;
	GString *ret;
	gchar *subret;
	
	ret = g_string_new( "" );
	
	if( ss->is_star ) {
		if( ss->n_tail == 0 ) {
			g_string_append_c( ret, '*' );
		}
	} else {
		g_string_append( ret,
			 html_atom_list_get_string( html_atom_list,
						    ss->element_name ) );
	}
	
	for( i = 0; i < ss->n_tail; ++ i ) {
		CssTail *tail = &ss->tail[ i ];

		if( tail->type == CSS_TAIL_ATTR_SEL ) {
			g_string_append_printf( ret, "[%s",
				 html_atom_list_get_string( html_atom_list,
							    tail->t.attr_sel.
							    att ) );

			if( tail->t.attr_sel.match != CSS_MATCH_EMPTY ) {
				if( tail->t.attr_sel.match == CSS_MATCH_EQ ) {
					g_string_append_c( ret, '=');
				} else if( tail->t.attr_sel.match ==
					   CSS_MATCH_INCLUDES ) {
					g_string_append( ret, "~=");
				} else if( tail->t.attr_sel.match ==
					 CSS_MATCH_DASHMATCH ) {
					g_string_append( ret, "|=" );
				} else if( tail->t.attr_sel.match ==
					 CSS_MATCH_PREFIXMATCH ) {
					g_string_append( ret, "^=" );
				} else if( tail->t.attr_sel.match ==
					 CSS_MATCH_SUFFIXMATCH ) {
					g_string_append( ret, "$=" );
				} else if( tail->t.attr_sel.match ==
					 CSS_MATCH_SUBSTRINGMATCH ) {
					g_string_append( ret, "*=" );
				} else {
					g_string_append( ret, " ??? " );
				}
				if( tail->t.attr_sel.val.type ==
				    CSS_ATTR_VAL_IDENT ) {
					g_string_append( ret,
						 html_atom_list_get_string( 
					    	  html_atom_list,
						  tail->t.attr_sel.val.a.id ) );
				} else if( tail->t.attr_sel.val.type ==
					 CSS_ATTR_VAL_STRING ) {

					/* FIXME: escape string contents (see also
					 * css_value_to_string). */
					g_string_append_printf( ret, "\"%s\"",
						 tail->t.attr_sel.val.a.str );
				}
			}
			g_string_append( ret, "]" );
		} else if( tail->type == CSS_TAIL_ID_SEL ) {
			g_string_append_printf( ret, "#%s",
				 html_atom_list_get_string( html_atom_list,
							    tail->t.id_sel.
							    id ) );
		} else if( tail->type == CSS_TAIL_CLASS_SEL ) {
			g_string_append_printf( ret, ".%s",
				 html_atom_list_get_string( html_atom_list,
							    tail->t.id_sel.
							    id ) );
		} else if( tail->type == CSS_TAIL_PSEUDO_SEL ) {
			g_string_append( ret, ":" );
			g_string_append( ret,
				 html_atom_list_get_string( html_atom_list,
							    tail->t.
							    pseudo_sel.name ) );

#if 0
			/* FIXME: Handle multiple arguments in functions */
			g_string_append( ret, ":" );
			if( tail->t.pseudo.is_function ) {
				g_string_append_printf( ret, "%s(",
					 html_atom_list_get_string( 
					  html_atom_list,
					  tail->t.pseudo.func_id ) );
			}
			g_string_append( ret,
				 html_atom_list_get_string( html_atom_list,
							    tail->t.pseudo.
							    id ) );

			if( tail->t.pseudo.is_function ) {
				g_string_append( ret, ")" );
			}
#endif
		} else {
			g_string_append( ret, "< ??? >" );
		}
	}
	
	subret = ret->str;
	g_string_free( ret, FALSE );
	
	return subret;
}

gchar *css_debug_print_selector( CssSelector * s )
{
	gint i;
	GString *ret;
	gchar *subret;

	ret = g_string_new( "" );
	
	for( i = 0; i < s->n_simple; ++ i ) {
		subret = css_debug_print_simple_selector( s->simple[ i ] );
		if( subret ) {
			g_string_append( ret, subret );
			g_free( subret );
		}
		
		if( i != s->n_simple - 1 ) {
			switch( s->comb[ i ] ) {
			case CSS_COMBINATOR_PLUS:
				g_string_append( ret, " + " );
				break;
			case CSS_COMBINATOR_GT:
				g_string_append( ret, " > " );
				break;
			case CSS_COMBINATOR_EMPTY:
				g_string_append( ret, " " );
				break;
			case CSS_COMBINATOR_TILDE:
				g_string_append( ret, " ~ " );
				break;
			}
		}
	}
	
	subret = ret->str;
	g_string_free( ret, FALSE );
	
	return subret;
}

gchar *css_debug_print_declaration( CssDeclaration * decl )
{
	GString *ret;
	gchar *subret;
	
	ret = g_string_new( "" );
	
	g_string_append_printf( ret, "%s: ",
		 html_atom_list_get_string( html_atom_list, decl->property ) );

	subret = css_debug_print_value( decl->expr );
	g_string_append( ret, subret );
	g_free( subret );

	if( decl->important ) {
		g_string_append( ret, " !important" );
	}
	
	subret = ret->str;
	g_string_free( ret, FALSE );
	
	return subret;
}

gchar *css_debug_print_ruleset( CssRuleset * rs )
{
	gint i;
	GString *ret;
	gchar *subret;
	
	ret = g_string_new( "" );
	
	for( i = 0; i < rs->n_sel; ++ i ) {
		subret = css_debug_print_selector( rs->sel[ i ] );
		if( subret ) {
			g_string_append( ret, subret );
			g_free( subret );
		}
		if( i != rs->n_sel - 1 ) {
			g_string_append( ret, ", " );
		}
	}

	g_string_append( ret, " {\n" );

	for( i = 0; i < rs->n_decl; ++ i ) {
		g_string_append( ret, "  " );
		subret = css_debug_print_declaration( rs->decl[ i ] );
		g_string_append( ret, subret );
		g_free( subret );
		g_string_append( ret, ";\n" );
	}

	g_string_append( ret, "}\n\n" );
	subret = ret->str;
	g_string_free( ret, FALSE );
	
	return subret;
}

gchar *css_debug_print_statement( CssStatement *stat )
{
	GString *ret;
	gchar *subret;
	gint i;
		
	ret = g_string_new( "" );
	
	switch( stat->type ) {
	case CSS_MEDIA_RULE:
		g_string_append( ret, "@media " );
		subret = css_debug_print_value( stat->s.media_rule.media_list );
		g_string_append( ret, subret );
		g_free( subret );
		g_string_append( ret, " {\n" );
		for( i = 0; i < stat->s.media_rule.n_rs; ++ i ) {
			g_string_append( ret, "  " );
			subret = css_debug_print_ruleset( stat->s.media_rule.rs[ i ] );
			g_string_append( ret, subret );
			g_free( subret );
		}
		g_string_append( ret, "}\n" );
		break;
	case CSS_PAGE_RULE:
		g_string_append( ret, "@page " );
		if( stat->s.page_rule.name != -1 ) {
			g_string_append_printf( ret,"%s ",
				 html_atom_list_get_string( html_atom_list,
							    stat->s.page_rule.
							    name ) );
		}
		if( stat->s.page_rule.pseudo != -1 ) {
			g_string_append_printf( ret, ":%s ",
				 html_atom_list_get_string( html_atom_list,
							    stat->s.page_rule.
							    pseudo ) );
		}
		g_string_append( ret, "{\n  ");
		for( i = 0; i < stat->s.page_rule.n_decl; ++ i ) {
			subret = css_debug_print_declaration( stat->s.page_rule.decl[ i ] );
			g_string_append( ret, subret );
			g_free( subret );
			if( i != stat->s.page_rule.n_decl - 1 ) {
				g_string_append( ret, "; " );
			}
		}
		g_string_append( ret, "\n}\n" );
		break;
	case CSS_FONT_FACE_RULE:
		g_string_append( ret, "@font-face {\n  " );
		for( i = 0; i < stat->s.font_face_rule.n_decl; ++ i ) {
			subret = css_debug_print_declaration( stat->s.font_face_rule.decl[ i ] );
			g_string_append( ret, subret );
			g_free( subret );
			if( i != stat->s.font_face_rule.n_decl - 1 ) {
				g_string_append( ret, "; " );
			}
		}
		g_string_append( ret, "\n}\n");
		break;
	case CSS_RULESET:
		subret = css_debug_print_ruleset( stat->s.ruleset );
		g_string_append( ret, subret );
		g_free( subret );
		break;
	default:
		g_error( "eek" );
	}
	
	subret = ret->str;
	g_string_free( ret, FALSE );
	
	return subret;
}


#endif
