/* -*- Mode: c; c-basic-offset: 2 -*-
 *
 * raptor_serialize.c - Serializers
 *
 * $Id: raptor_serialize.c,v 1.12 2004/10/28 21:43:41 cmdjb Exp $
 *
 * Copyright (C) 2004, David Beckett http://purl.org/net/dajobe/
 * Institute for Learning and Research Technology http://www.ilrt.bristol.ac.uk/
 * University of Bristol, UK http://www.bristol.ac.uk/
 * 
 * This package is Free Software and part of Redland http://librdf.org/
 * 
 * It is licensed under the following three licenses as alternatives:
 *   1. GNU Lesser General Public License (LGPL) V2.1 or any newer version
 *   2. GNU General Public License (GPL) V2 or any newer version
 *   3. Apache License, V2.0 or any newer version
 * 
 * You may not use this file except in compliance with at least one of
 * the above three licenses.
 * 
 * See LICENSE.html or LICENSE.txt at the top of this package for the
 * complete terms and further detail along with the license texts for
 * the licenses in COPYING.LIB, COPYING and LICENSE-2.0.txt respectively.
 * 
 */


#ifdef HAVE_CONFIG_H
#include <raptor_config.h>
#endif

#ifdef WIN32
#include <win32_raptor_config.h>
#endif


#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdarg.h>
#ifdef HAVE_ERRNO_H
#include <errno.h>
#endif
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

/* Raptor includes */
#include "raptor.h"
#include "raptor_internal.h"


/* prototypes for helper functions */
static raptor_serializer_factory* raptor_get_serializer_factory(const char *name);


/* statics */

/* list of serializer factories */
static raptor_serializer_factory* serializers=NULL;


/* helper functions */


/*
 * raptor_delete_serializer_factories - helper function to delete all the registered serializer factories
 */
void
raptor_delete_serializer_factories(void)
{
  raptor_serializer_factory *factory, *next;
  
  for(factory=serializers; factory; factory=next) {
    next=factory->next;

    if(factory->finish_factory)
      factory->finish_factory(factory);

    RAPTOR_FREE(raptor_serializer_factory, factory->name);
    RAPTOR_FREE(raptor_serializer_factory, factory->label);
    if(factory->alias)
      RAPTOR_FREE(raptor_serializer_factory, factory->alias);
    if(factory->mime_type)
      RAPTOR_FREE(raptor_serializer_factory, factory->mime_type);
    if(factory->uri_string)
      RAPTOR_FREE(raptor_serializer_factory, factory->uri_string);

    RAPTOR_FREE(raptor_serializer_factory, factory);
  }
  serializers=NULL;
}


/* class methods */

/*
 * raptor_serializer_register_factory - Register a syntax that can be generated by a serializer factory
 * @name: the short syntax name
 * @label: readable label for syntax
 * @mime_type: MIME type of the syntax generated by the serializer (or NULL)
 * @uri_string: URI string of the syntax (or NULL)
 * @factory: pointer to function to call to register the factory
 * 
 * INTERNAL
 *
 **/
void
raptor_serializer_register_factory(const char *name, const char *label,
                                   const char *mime_type,
                                   const char *alias,
                                   const unsigned char *uri_string,
                                   void (*factory) (raptor_serializer_factory*)) 
{
  raptor_serializer_factory *serializer, *h;
  char *name_copy, *label_copy, *mime_type_copy, *alias_copy;
  unsigned char *uri_string_copy;
  
#if defined(RAPTOR_DEBUG) && RAPTOR_DEBUG > 1
  RAPTOR_DEBUG4("Received registration for syntax serializer %s '%s' with alias '%s'\n", 
                name, label, (alias ? alias : "none"));
  RAPTOR_DEBUG4(raptor_serializer_register_factory,
                "MIME type %s, URI %s\n", 
                (mime_type ? mime_type : "none"),
                (uri_string ? uri_string : "none"));
#endif
  
  serializer=(raptor_serializer_factory*)RAPTOR_CALLOC(raptor_serializer_factory, 1,
                                               sizeof(raptor_serializer_factory));
  if(!serializer)
    RAPTOR_FATAL1("Out of memory\n");

  for(h = serializers; h; h = h->next ) {
    if(!strcmp(h->name, name) ||
       (alias && !strcmp(h->name, alias))) {
      RAPTOR_FATAL2("serializer %s already registered\n", h->name);
    }
  }
  
  name_copy=(char*)RAPTOR_CALLOC(cstring, strlen(name)+1, 1);
  if(!name_copy) {
    RAPTOR_FREE(raptor_serializer, serializer);
    RAPTOR_FATAL1("Out of memory\n");
  }
  strcpy(name_copy, name);
  serializer->name=name_copy;
        
  label_copy=(char*)RAPTOR_CALLOC(cstring, strlen(label)+1, 1);
  if(!label_copy) {
    RAPTOR_FREE(raptor_serializer, serializer);
    RAPTOR_FATAL1("Out of memory\n");
  }
  strcpy(label_copy, label);
  serializer->label=label_copy;

  if(mime_type) {
    mime_type_copy=(char*)RAPTOR_CALLOC(cstring, strlen(mime_type)+1, 1);
    if(!mime_type_copy) {
      RAPTOR_FREE(raptor_serializer, serializer);
      RAPTOR_FATAL1("Out of memory\n");
    }
    strcpy(mime_type_copy, mime_type);
    serializer->mime_type=mime_type_copy;
  }

  if(uri_string) {
    uri_string_copy=(unsigned char*)RAPTOR_CALLOC(cstring, strlen((const char*)uri_string)+1, 1);
    if(!uri_string_copy) {
    RAPTOR_FREE(raptor_serializer, serializer);
    RAPTOR_FATAL1("Out of memory\n");
    }
    strcpy((char*)uri_string_copy, (const char*)uri_string);
    serializer->uri_string=uri_string_copy;
  }
        
  if(alias) {
    alias_copy=(char*)RAPTOR_CALLOC(cstring, strlen(alias)+1, 1);
    if(!alias_copy) {
      RAPTOR_FREE(raptor_serializer, serializer);
      RAPTOR_FATAL1("Out of memory\n");
    }
    strcpy(alias_copy, alias);
    serializer->alias=alias_copy;
  }

  /* Call the serializer registration function on the new object */
  (*factory)(serializer);
  
#if defined(RAPTOR_DEBUG) && RAPTOR_DEBUG > 1
  RAPTOR_DEBUG3("%s has context size %d\n", name, serializer->context_length);
#endif
  
  serializer->next = serializers;
  serializers = serializer;
}


/**
 * raptor_get_serializer_factory - Get a serializer factory by name
 * @name: the factory name or NULL for the default factory
 * 
 * Return value: the factory object or NULL if there is no such factory
 **/
static raptor_serializer_factory*
raptor_get_serializer_factory (const char *name) 
{
  raptor_serializer_factory *factory;

  /* return 1st serializer if no particular one wanted - why? */
  if(!name) {
    factory=serializers;
    if(!factory) {
      RAPTOR_DEBUG1("No (default) serializers registered\n");
      return NULL;
    }
  } else {
    for(factory=serializers; factory; factory=factory->next) {
      if(!strcmp(factory->name, name) ||
         (factory->alias && !strcmp(factory->alias, name)))
        break;
    }
    /* else FACTORY name not found */
    if(!factory) {
      RAPTOR_DEBUG2("No serializer with name %s found\n", name);
      return NULL;
    }
  }
        
  return factory;
}


/**
 * raptor_serializers_enumerate - Get information on syntax serializers
 * @counter: index into the list of syntaxes
 * @name: pointer to store the name of the syntax (or NULL)
 * @label: pointer to store syntax readable label (or NULL)
 * @mime_type: pointer to store syntax MIME Type (or NULL)
 * @uri_string: pointer to store syntax URI string (or NULL)
 * 
 * Return value: non 0 on failure of if counter is out of range
 **/
int
raptor_serializers_enumerate(const unsigned int counter,
                             const char **name, const char **label,
                             const char **mime_type,
                             const unsigned char **uri_string)
{
  unsigned int i;
  raptor_serializer_factory *factory=serializers;

  if(!factory || counter < 0)
    return 1;

  for(i=0; factory && i<=counter ; i++, factory=factory->next) {
    if(i == counter) {
      if(name)
        *name=factory->name;
      if(label)
        *label=factory->label;
      if(mime_type)
        *mime_type=factory->mime_type;
      if(uri_string)
        *uri_string=factory->uri_string;
      return 0;
    }
  }
        
  return 1;
}

/**
 * raptor_serializer_syntax_name_check -  Check name of a serializer
 * @name: the syntax name
 *
 * Return value: non 0 if name is a known syntax name
 */
int
raptor_serializer_syntax_name_check(const char *name) {
  return (raptor_get_serializer_factory(name) != NULL);
}


/**
 * raptor_new_serializer - Constructor - create a new raptor_serializer object
 * @name: the serializer name
 *
 * Return value: a new &raptor_serializer object or NULL on failure
 */
raptor_serializer*
raptor_new_serializer(const char *name) {
  raptor_serializer_factory* factory;
  raptor_serializer* rdf_serializer;

  factory=raptor_get_serializer_factory(name);
  if(!factory)
    return NULL;

  rdf_serializer=(raptor_serializer*)RAPTOR_CALLOC(raptor_serializer, 1,
                                           sizeof(raptor_serializer));
  if(!rdf_serializer)
    return NULL;
  
  rdf_serializer->context=(char*)RAPTOR_CALLOC(raptor_serializer_context, 1,
                                               factory->context_length);
  if(!rdf_serializer->context) {
    raptor_free_serializer(rdf_serializer);
    return NULL;
  }
  
  rdf_serializer->factory=factory;

  if(factory->init(rdf_serializer, name)) {
    raptor_free_serializer(rdf_serializer);
    return NULL;
  }
  
  return rdf_serializer;
}


/**
 * raptor_serialize_start: Start serialization with given base URI
 * @rdf_serializer:  the &raptor_serializer
 * @uri: base URI or NULL if no base URI is required
 * @iostream: &raptor_iostream to write serialization to
 * 
 * Return value: non-0 on failure.
 **/
int
raptor_serialize_start(raptor_serializer *rdf_serializer, raptor_uri *uri,
                       raptor_iostream *iostream) 
{
  if(rdf_serializer->base_uri)
    raptor_free_uri(rdf_serializer->base_uri);

  if(!iostream)
    return 1;
  
  if(uri)
    uri=raptor_uri_copy(uri);
  
  rdf_serializer->base_uri=uri;
  rdf_serializer->locator.uri=uri;
  rdf_serializer->locator.line=rdf_serializer->locator.column = 0;

  rdf_serializer->iostream=iostream;

  if(rdf_serializer->factory->serialize_start)
    return rdf_serializer->factory->serialize_start(rdf_serializer);
  return 0;
}


/**
 * raptor_serialize_start_to_filename: Start serializing to a filename
 * @rdf_serializer:  the &raptor_serializer
 * @filename:  filename to serialize to
 * 
 * Return value: non-0 on failure.
 **/
int
raptor_serialize_start_to_filename(raptor_serializer *rdf_serializer,
                                   const char *filename)
{
  unsigned char *uri_string=raptor_uri_filename_to_uri_string(filename);
  if(!uri_string)
    return 1;

  if(rdf_serializer->base_uri)
    raptor_free_uri(rdf_serializer->base_uri);

  rdf_serializer->base_uri=raptor_new_uri(uri_string);
  rdf_serializer->locator.uri=rdf_serializer->base_uri;
  rdf_serializer->locator.line=rdf_serializer->locator.column = 0;

  RAPTOR_FREE(cstring, uri_string);

  rdf_serializer->iostream=raptor_new_iostream_to_filename(filename);
  if(!rdf_serializer->iostream)
    return 1;

  if(rdf_serializer->factory->serialize_start)
    return rdf_serializer->factory->serialize_start(rdf_serializer);
  return 0;
}



/**
 * raptor_serialize_start_to_string: Start serializing to a string
 * @rdf_serializer:  the &raptor_serializer
 * @uri: base URI or NULL if no base URI is required
 * @string_p: pointer to location to hold string
 * @length_p: pointer to location to hold length of string (or NULL)
 * 
 * Return value: non-0 on failure.
 **/
int
raptor_serialize_start_to_string(raptor_serializer *rdf_serializer,
                                 raptor_uri *uri,
                                 void **string_p, size_t *length_p) 
{
  if(rdf_serializer->base_uri)
    raptor_free_uri(rdf_serializer->base_uri);

  if(uri)
    rdf_serializer->base_uri=raptor_uri_copy(uri);
  else
    rdf_serializer->base_uri=NULL;
  rdf_serializer->locator.uri=rdf_serializer->base_uri;
  rdf_serializer->locator.line=rdf_serializer->locator.column = 0;


  rdf_serializer->iostream=raptor_new_iostream_to_string(string_p, length_p, 
                                                         NULL);
  if(!rdf_serializer->iostream)
    return 1;

  if(rdf_serializer->factory->serialize_start)
    return rdf_serializer->factory->serialize_start(rdf_serializer);
  return 0;
}


/**
 * raptor_serialize_start_to_file_handle: Start serializing to a FILE*
 * @rdf_serializer:  the &raptor_serializer
 * @uri: base URI or NULL if no base URI is required
 * @fh:  FILE* to serialize to
 * 
 * NOTE: This does not fclose the handle when it is finished.
8
 * Return value: non-0 on failure.
 **/
int
raptor_serialize_start_to_file_handle(raptor_serializer *rdf_serializer,
                                      raptor_uri *uri, FILE *fh) 
{
  if(rdf_serializer->base_uri)
    raptor_free_uri(rdf_serializer->base_uri);

  if(uri)
    rdf_serializer->base_uri=raptor_uri_copy(uri);
  else
    rdf_serializer->base_uri=NULL;
  rdf_serializer->locator.uri=rdf_serializer->base_uri;
  rdf_serializer->locator.line=rdf_serializer->locator.column = 0;

  rdf_serializer->iostream=raptor_new_iostream_to_file_handle(fh);
  if(!rdf_serializer->iostream)
    return 1;

  if(rdf_serializer->factory->serialize_start)
    return rdf_serializer->factory->serialize_start(rdf_serializer);
  return 0;
}


/**
 * raptor_start_serialize: Serialize a statement
 * @rdf_serializer: the &raptor_serializer
 * @statement: &raptor_statement to serialize to a syntax
 * 
 * Return value: non-0 on failure.
 **/
int
raptor_serialize_statement(raptor_serializer* rdf_serializer,
                           const raptor_statement *statement) {
  if(!rdf_serializer->iostream)
    return 1;
  return rdf_serializer->factory->serialize_statement(rdf_serializer, statement);
}


/**
 * raptor_serialize_end: End a serialization
 * @rdf_serializer:  the &raptor_serializer
 * 
 * Return value: non-0 on failure.
 **/
int
raptor_serialize_end(raptor_serializer *rdf_serializer) 
{
  int rc;
  
  if(!rdf_serializer->iostream)
    return 1;

  if(rdf_serializer->factory->serialize_end)
    rc=rdf_serializer->factory->serialize_end(rdf_serializer);
  else
    rc=0;

  if(rdf_serializer->iostream) {
    raptor_free_iostream(rdf_serializer->iostream);
    rdf_serializer->iostream=NULL;
  }
  return rc;
}



/**
 * raptor_free_serializer - Destructor - destroy a raptor_serializer object
 * @serializer: &raptor_serializer object
 * 
 **/
void
raptor_free_serializer(raptor_serializer* rdf_serializer) 
{
  if(rdf_serializer->factory)
    rdf_serializer->factory->terminate(rdf_serializer);

  if(rdf_serializer->context)
    RAPTOR_FREE(raptor_serializer_context, rdf_serializer->context);

  if(rdf_serializer->base_uri)
    raptor_free_uri(rdf_serializer->base_uri);

  RAPTOR_FREE(raptor_serializer, rdf_serializer);
}


/**
 * raptor_serializer_get_iostream - Get the current serializer iostream
 * @serializer: &raptor_serializer object
 *
 * Return value: the serializer's current iostream or NULL if 
 **/
raptor_iostream*
raptor_serializer_get_iostream(raptor_serializer *serializer)
{
  return serializer->iostream;
}



/*
 * raptor_serializer_error - Error from a serializer - Internal
 */
void
raptor_serializer_error(raptor_serializer* serializer, const char *message, ...)
{
  va_list arguments;

  va_start(arguments, message);

  raptor_serializer_error_varargs(serializer, message, arguments);
  
  va_end(arguments);
}


/*
 * raptor_serializer_simple_error - Error from a serializer - Internal
 *
 * Matches the raptor_simple_message_handler API but same as
 * raptor_serializer_error 
 */
void
raptor_serializer_simple_error(void* serializer, const char *message, ...)
{
  va_list arguments;

  va_start(arguments, message);

  raptor_serializer_error_varargs((raptor_serializer*)serializer, message, arguments);
  
  va_end(arguments);
}


/*
 * raptor_serializer_error_varargs - Error from a serializer - Internal
 */
void
raptor_serializer_error_varargs(raptor_serializer* serializer, 
                                const char *message, 
                                va_list arguments)
{
  if(serializer->error_handler) {
    char *buffer=raptor_vsnprintf(message, arguments);
    size_t length;
    if(!buffer) {
      fprintf(stderr, "raptor_serializer_error_varargs: Out of memory\n");
      return;
    }
    length=strlen(buffer);
    if(buffer[length-1]=='\n')
      buffer[length-1]='\0';
    serializer->error_handler(serializer->error_user_data, 
                              &serializer->locator, buffer);
    RAPTOR_FREE(cstring, buffer);
    return;
  }

  raptor_print_locator(stderr, &serializer->locator);
  fprintf(stderr, " raptor error - ");
  vfprintf(stderr, message, arguments);
  fputc('\n', stderr);
}


/*
 * raptor_serializer_warning - Warning from a serializer - Internal
 */
void
raptor_serializer_warning(raptor_serializer* serializer, const char *message, ...)
{
  va_list arguments;

  va_start(arguments, message);

  raptor_serializer_warning_varargs(serializer, message, arguments);

  va_end(arguments);
}


/*
 * raptor_serializer_warning - Warning from a serializer - Internal
 */
void
raptor_serializer_warning_varargs(raptor_serializer* serializer, const char *message, 
                                  va_list arguments)
{

  if(serializer->warning_handler) {
    char *buffer=raptor_vsnprintf(message, arguments);
    size_t length;
    if(!buffer) {
      fprintf(stderr, "raptor_serializer_warning_varargs: Out of memory\n");
      return;
    }
    length=strlen(buffer);
    if(buffer[length-1]=='\n')
      buffer[length-1]='\0';
    serializer->warning_handler(serializer->warning_user_data,
                                &serializer->locator, buffer);
    RAPTOR_FREE(cstring, buffer);
    return;
  }

  raptor_print_locator(stderr, &serializer->locator);
  fprintf(stderr, " raptor warning - ");
  vfprintf(stderr, message, arguments);
  fputc('\n', stderr);
}


/**
 * raptor_serializer_set_error_handler - Set the serializer error handling function
 * @serializer: the serializer
 * @user_data: user data to pass to function
 * @handler: pointer to the function
 * 
 * The function will receive callbacks when the serializer fails.
 * 
 **/
void
raptor_serializer_set_error_handler(raptor_serializer* serializer, 
                                    void *user_data,
                                    raptor_message_handler handler)
{
  serializer->error_user_data=user_data;
  serializer->error_handler=handler;
}


/**
 * raptor_serializer_set_warning_handler - Set the serializer warning handling function
 * @serializer: the serializer
 * @user_data: user data to pass to function
 * @handler: pointer to the function
 * 
 * The function will receive callbacks when the serializer fails.
 * 
 **/
void
raptor_serializer_set_warning_handler(raptor_serializer* serializer, 
                                      void *user_data,
                                      raptor_message_handler handler)
{
  serializer->warning_user_data=user_data;
  serializer->warning_handler=handler;
}


/**
 * raptor_serializer_get_locator: Get the serializer raptor locator object
 * @rdf_serializer: raptor serializer
 * 
 * Return value: raptor locator
 **/
raptor_locator*
raptor_serializer_get_locator(raptor_serializer *rdf_serializer)
{
  return &rdf_serializer->locator;
}





/*
 * Raptor RDF/XML serializer object
 */
typedef struct {
  int depth;
  raptor_namespace_stack *nstack;
  raptor_namespace *rdf_ns;
} raptor_rdfxml_serializer_context;


/* local prototypes */
static int raptor_rdfxml_serialize_ok_xml_name(unsigned char *name);

/* create a new serializer */
static int
raptor_rdfxml_serialize_init(raptor_serializer* serializer, const char *name)
{
  raptor_rdfxml_serializer_context* context=(raptor_rdfxml_serializer_context*)serializer->context;
  raptor_uri_handler *uri_handler;
  void *uri_context;
  
  raptor_uri_get_handler(&uri_handler, &uri_context);
  context->nstack=raptor_new_namespaces(uri_handler, uri_context,
                                        raptor_serializer_simple_error,
                                        serializer,
                                        1);
  context->depth=0;
  context->rdf_ns=raptor_new_namespace(context->nstack,
                                       (const unsigned char*)"rdf",
                                       (const unsigned char*)raptor_rdf_namespace_uri,
                                       context->depth);
  return 0;
}
  

/* destroy a serializer */
static void
raptor_rdfxml_serialize_terminate(raptor_serializer* serializer)
{
  raptor_rdfxml_serializer_context* context=(raptor_rdfxml_serializer_context*)serializer->context;

  /* Frees all namespaces on the stack, including context->rdf_ns */  
  if(context->nstack)
    raptor_free_namespaces(context->nstack);
}
  

/* add a namespace */
static int
raptor_rdfxml_serialize_declare_namespace(raptor_serializer* serializer, 
                                          const unsigned char *prefix, 
                                          raptor_uri *uri)
{
  /* raptor_rdfxml_serializer_context* context=(raptor_rdfxml_serializer_context*)serializer->context; */
  return 0;
}


/* start a serialize */
static int
raptor_rdfxml_serialize_start(raptor_serializer* serializer)
{
  raptor_rdfxml_serializer_context* context=(raptor_rdfxml_serializer_context*)serializer->context;
  raptor_iostream *iostr=serializer->iostream;
  unsigned char *buffer;
  
  raptor_iostream_write_string(iostr,
                               "<?xml version=\"1.0\" encoding=\"utf-8\"?>\n");
  context->depth++;
  raptor_namespaces_start_namespace(context->nstack, context->rdf_ns);

  raptor_iostream_write_counted_string(iostr, "<rdf:RDF ", 9);

  buffer=raptor_namespaces_format(context->rdf_ns, NULL);
  raptor_iostream_write_string(iostr, (const char*)buffer);
  raptor_free_memory(buffer);

  raptor_iostream_write_counted_string(iostr, ">\n", 2);

  return 0;
}


/**
 * raptor_rdfxml_serialize_ok_xml_name - check name is OK XML Name
 * @name: XML name to check
 * 
 * An XML name starts with alpha or _, continues with alnum or _ - .
 * 
 * Return value: non-zero if is a legal XML name
 **/
static int
raptor_rdfxml_serialize_ok_xml_name(unsigned char *name) 
{
  if(!isalpha(*name) && *name != '_')
    return 0;
  name++;
  while(*name) {
    if(!isalnum(*name)  && *name != '_' && *name != '-'  && *name != '.')
      return 0;
    name++;
  }
  return 1;
}


/**
 * raptor_rdfxml_serialize_write_xml_attribute - Write the attribute/value as an XML attribute, XML escaped to an iostream
 * @world: &librdf_world world
 * @attr: attribute name
 * @value: attribute value
 * @iostr: &raptor_iostream to print to
 * 
 * Return value: non-0 on failure
 **/
static int
raptor_rdfxml_serialize_write_xml_attribute(raptor_serializer *serializer,
                                            unsigned char *attr,
                                            unsigned char *value,
                                            raptor_iostream *iostr) 
{
  size_t attr_len;
  size_t len;
  int escaped_len=0;
  unsigned char *buffer;
  unsigned char *p;
  
  attr_len=strlen((const char*)attr);
  len=strlen((const char*)value);

  if(len) {
    escaped_len=raptor_xml_escape_string(value, len,
                                         NULL, 0, '"', 
                                         NULL, NULL);
    if(escaped_len < 0) {
      raptor_serializer_error(serializer, "Bad UTF-8 encoding found while XML escaping attribute '%s' value '%s'", attr, value);
      return 1;
    }
  }

  buffer=(unsigned char*)RAPTOR_MALLOC(cstring, 1 + attr_len + 2 + escaped_len + 1 +1);
  if(!buffer)
    return 1;

  p=buffer;
  *p++=' ';
  strncpy((char*)p, (const char*)attr, attr_len);
  p+= attr_len;
  *p++='=';
  *p++='"';
  if(escaped_len) {
    raptor_xml_escape_string(value, len, 
                             p, escaped_len, '"', 
                             NULL, NULL);
    p+= escaped_len;
  }
  *p++='"';
  *p='\0';
  
  raptor_iostream_write_counted_string(iostr, buffer, p-buffer);
  RAPTOR_FREE(cstring, buffer);

  return 0;
}


/* serialize a statement */
static int
raptor_rdfxml_serialize_statement(raptor_serializer* serializer, 
                                  const raptor_statement *statement)
{
  /* raptor_rdfxml_serializer_context* context=(raptor_rdfxml_serializer_context*)serializer->context; */
  raptor_iostream* iostr=serializer->iostream;
  unsigned char* uri_string; /* predicate URI */
  unsigned char* name=NULL;  /* where to split predicate name */
  int name_is_rdf_ns=0;
  char* nsprefix="ns0";
  int rc;
  size_t len;

  if(statement->predicate_type != RAPTOR_IDENTIFIER_TYPE_ORDINAL) {
    unsigned char *p;
    size_t uri_len;

    uri_string=raptor_uri_as_counted_string((raptor_uri*)statement->predicate,
                                            &uri_len);

    if(!strncmp((const char*)uri_string, 
                raptor_rdf_namespace_uri, raptor_rdf_namespace_uri_len)) {
      name=uri_string + raptor_rdf_namespace_uri_len;
      name_is_rdf_ns=1;
      nsprefix="rdf";
    } else {
      /* FIXME: this does too much work, it should end on the first
       * illegal XML name character - requires a raptor check */
      p= uri_string + uri_len-1;
      while(p >= uri_string) {
        if(raptor_rdfxml_serialize_ok_xml_name(p))
          name=p;
        else if(name && p>uri_string && !raptor_rdfxml_serialize_ok_xml_name(p-1))
          /* if next char would make name invalid, stop */
          break;
        p--;
      }
      
      if(!name) {
        raptor_serializer_error(serializer, "Cannot split predicate URI %s into an XML qname - skipping statement", uri_string);
        return 1;
      }
    }
  }


  raptor_iostream_write_string(iostr, "  <rdf:Description");


  /* subject */
  rc=0;
  switch(statement->subject_type) {
    case RAPTOR_IDENTIFIER_TYPE_ANONYMOUS:
      rc=raptor_rdfxml_serialize_write_xml_attribute(serializer, 
                                                     (unsigned char*)"rdf:nodeID", 
                                                     (unsigned char*)statement->subject,
                                                     iostr);
      break;

    case RAPTOR_IDENTIFIER_TYPE_ORDINAL:
      /* FIXME */
      break;

    case RAPTOR_IDENTIFIER_TYPE_RESOURCE:
      rc=raptor_rdfxml_serialize_write_xml_attribute(serializer,
                                                     (unsigned char*)"rdf:about", 
                                                     (unsigned char*)raptor_uri_as_string((raptor_uri*)statement->subject),
                                                     iostr);
      break;
      
    case RAPTOR_IDENTIFIER_TYPE_PREDICATE:
    case RAPTOR_IDENTIFIER_TYPE_LITERAL:
    case RAPTOR_IDENTIFIER_TYPE_XML_LITERAL:
    default:
      raptor_serializer_error(serializer, "Do not know how to serialize node type %d\n", statement->subject_type);
  }

  if(rc) {
    raptor_iostream_write_counted_string(iostr, "/>\n", 3);
    return 1;
  }

  raptor_iostream_write_counted_string(iostr, ">\n", 2);


  /* predicate */
  if(statement->predicate_type == RAPTOR_IDENTIFIER_TYPE_ORDINAL) {
    raptor_iostream_write_counted_string(iostr, "    <rdf:_", 10);
    raptor_iostream_write_decimal(iostr, *((int*)statement->predicate));
  } else {
    raptor_iostream_write_counted_string(iostr, "    <", 5);
    raptor_iostream_write_string(iostr, nsprefix);
    raptor_iostream_write_byte(iostr, ':');
    raptor_iostream_write_string(iostr, (const char*)name);

    if(!name_is_rdf_ns) {
      int escaped_len=0;
      unsigned char *buffer;
      unsigned char *p;

      len=name-uri_string;

      raptor_iostream_write_counted_string(iostr, " xmlns:", 7);
      raptor_iostream_write_string(iostr, nsprefix);
      raptor_iostream_write_byte(iostr, '=');

      if(len) {
        escaped_len=raptor_xml_escape_string(uri_string, len,
                                             NULL, 0, '"',
                                             NULL, NULL);
        if(escaped_len < 0) {
          raptor_serializer_error(serializer, 
                                  "Bad UTF-8 encoding found while XML escaping namespace URI '%s'", uri_string);
          return 1;
        }
      }

      /* " + string + " + \0 */
      buffer=(unsigned char*)RAPTOR_MALLOC(cstring, 1 + escaped_len + 1 + 1);
      if(!buffer)
        return 1;

      p=buffer;
      *p++='"';
      if(escaped_len) {
        raptor_xml_escape_string(uri_string, len,
                                 p, escaped_len, '"',
                                 NULL, NULL);
        p+= escaped_len;
      }
      *p++='"';
      *p='\0';

      raptor_iostream_write_counted_string(iostr, (const char*)buffer, p-buffer);
      RAPTOR_FREE(cstring, buffer);
    }
  }

  /* object */
  switch(statement->object_type) {
    case RAPTOR_IDENTIFIER_TYPE_LITERAL:
    case RAPTOR_IDENTIFIER_TYPE_XML_LITERAL:

      if(statement->object_literal_language) {
        if(raptor_rdfxml_serialize_write_xml_attribute(serializer,
                                                       (unsigned char*)"xml:lang",
                                                       (unsigned char*)statement->object_literal_language,
                                                       iostr))
          return 1;
      }

      len=strlen((const char*)statement->object);
      
      if(statement->object_type == RAPTOR_IDENTIFIER_TYPE_XML_LITERAL) {
        raptor_iostream_write_string(iostr, " rdf:parseType=\"Literal\">");
        /* Print without escaping XML */
        if(len)
          raptor_iostream_write_counted_string(iostr, (const char*)statement->object, len);
      } else {
        int xml_string_len;

        if(statement->object_literal_datatype) {
          if(raptor_rdfxml_serialize_write_xml_attribute(serializer, 
                                                         (unsigned char*)"rdf:datatype",
                                                         (unsigned char*)raptor_uri_as_string((raptor_uri*)statement->object_literal_datatype),
                                                         iostr))
            return 1;
        }

        raptor_iostream_write_byte(iostr, '>');
        
        if(len) {
          xml_string_len=raptor_xml_escape_string((const unsigned char*)statement->object, len,
                                                  NULL, 0, 0, 
                                                  NULL, NULL);
          if(xml_string_len < 0) {
            raptor_serializer_error(serializer,
                                    "Bad UTF-8 encoding found while XML escaping element content '%s'", statement->object);
            return 1;
          }

          if(xml_string_len == (int)len)
            raptor_iostream_write_string(iostr, statement->object);
          else {
            unsigned char *xml_string;

            xml_string=(unsigned char*)RAPTOR_MALLOC(cstring, xml_string_len+1);
            xml_string_len=raptor_xml_escape_string((const unsigned char*)statement->object, len,
                                                    xml_string, xml_string_len, 0,
                                                    NULL, NULL);
            raptor_iostream_write_counted_string(iostr, xml_string, xml_string_len);
            RAPTOR_FREE(cstring, xml_string);
          }
        }

      }

      raptor_iostream_write_counted_string(iostr, "</", 2);
      raptor_iostream_write_string(iostr, nsprefix);
      raptor_iostream_write_byte(iostr, ':');
      if(statement->predicate_type == RAPTOR_IDENTIFIER_TYPE_ORDINAL) {
        raptor_iostream_write_byte(iostr, '_');
        raptor_iostream_write_decimal(iostr, *((int*)statement->predicate));
      } else
        raptor_iostream_write_string(iostr, (const char*)name);
      raptor_iostream_write_byte(iostr, '>');
      break;

    case RAPTOR_IDENTIFIER_TYPE_ANONYMOUS:
      if(raptor_rdfxml_serialize_write_xml_attribute(serializer, 
                                                     (unsigned char*)"rdf:nodeID",
                                                     (unsigned char*)statement->object,
                                                     iostr))
        return 1;
      raptor_iostream_write_string(iostr, "/>");
      break;

    case RAPTOR_IDENTIFIER_TYPE_RESOURCE:
      /* must be URI */
      if(raptor_rdfxml_serialize_write_xml_attribute(serializer,
                                                     (unsigned char*)"rdf:resource",
                                                     (unsigned char*)raptor_uri_as_string((raptor_uri*)statement->object),
                                                     iostr))
        return 1;
      raptor_iostream_write_string(iostr, "/>");
      break;

    case RAPTOR_IDENTIFIER_TYPE_ORDINAL:
      /* FIXME */
      break;

    case RAPTOR_IDENTIFIER_TYPE_PREDICATE:
    default:
      raptor_serializer_error(serializer, "Do not know how to serialize node type %d\n", statement->object_type);
  }

  raptor_iostream_write_string(iostr, "\n  </rdf:Description>\n");

  return 0;
}


/* end a serialize */
static int
raptor_rdfxml_serialize_end(raptor_serializer* serializer)
{
  raptor_rdfxml_serializer_context* context=(raptor_rdfxml_serializer_context*)serializer->context;
  raptor_iostream *iostr=serializer->iostream;

  raptor_iostream_write_counted_string(iostr, "</rdf:RDF>\n", 11);

  raptor_namespaces_end_for_depth(context->nstack, context->depth);
  context->depth--;

  return 0;
}


/* finish the serializer factory */
static void
raptor_rdfxml_serialize_finish_factory(raptor_serializer_factory* factory)
{

}

static void
raptor_rdfxml_serializer_register_factory(raptor_serializer_factory *factory)
{
  factory->context_length     = sizeof(raptor_rdfxml_serializer_context);
  
  factory->init                = raptor_rdfxml_serialize_init;
  factory->terminate           = raptor_rdfxml_serialize_terminate;
  factory->declare_namespace   = raptor_rdfxml_serialize_declare_namespace;
  factory->serialize_start     = raptor_rdfxml_serialize_start;
  factory->serialize_statement = raptor_rdfxml_serialize_statement;
  factory->serialize_end       = raptor_rdfxml_serialize_end;
  factory->finish_factory      = raptor_rdfxml_serialize_finish_factory;
}



void
raptor_init_serializer_rdfxml(void) {
  raptor_serializer_register_factory("rdfxml", "RDF/XML", 
                                     "application/rdf+xml",
                                     NULL,
                                     (const unsigned char*)"http://www.w3.org/TR/rdf-syntax-grammar",
                                     &raptor_rdfxml_serializer_register_factory);
}


/*
 * Raptor N-Triples serializer object
 */
typedef struct {
  int dummy;
} raptor_ntriples_serializer_context;



/* create a new serializer */
static int
raptor_ntriples_serialize_init(raptor_serializer* serializer, const char *name)
{
  return 0;
}
  

/* destroy a serializer */
static void
raptor_ntriples_serialize_terminate(raptor_serializer* serializer)
{

}
  

/* add a namespace */
static int
raptor_ntriples_serialize_declare_namespace(raptor_serializer* serializer, 
                                            const unsigned char *prefix, 
                                            raptor_uri *uri)
{
  /* NOP */
  return 0;
}


#if 0
/* start a serialize */
static int
raptor_ntriples_serialize_start(raptor_serializer* serializer)
{
  return 0;
}
#endif



/**
 * raptor_serialize_ntriples_print_string - Print an UTF-8 string using N-Triples escapes
 * @stream: FILE* stream to print to
 * @string: UTF-8 string to print
 * @delim: Delimiter character for string (such as ") or \0 for no delim
 * escaping.
 * 
 * Return value: non-0 on failure such as bad UTF-8 encoding.
 **/
static int
raptor_serialize_ntriples_print_string(raptor_iostream *iostr,
                                       const unsigned char *string,
                                       const char delim) 
{
  unsigned char c;
  size_t len=strlen((const char*)string);
  int unichar_len;
  unsigned long unichar;
  
  for(; (c=*string); string++, len--) {
    if((delim && c == delim) || c == '\\') {
      raptor_iostream_write_byte(iostr, '\\');
      raptor_iostream_write_byte(iostr, c);
      continue;
    }
    
    /* Note: NTriples is ASCII */
    if(c == 0x09) {
      raptor_iostream_write_counted_string(iostr, "\\t", 2);
      continue;
    } else if(c == 0x0a) {
      raptor_iostream_write_counted_string(iostr, "\\n", 2);
      continue;
    } else if(c == 0x0d) {
      raptor_iostream_write_counted_string(iostr, "\\r", 2);
      continue;
    } else if(c < 0x20|| c == 0x7f) {
      raptor_iostream_write_counted_string(iostr, "\\u", 2);
      raptor_iostream_format_hexadecimal(iostr, c, 4);
      continue;
    } else if(c < 0x80) {
      raptor_iostream_write_byte(iostr, c);
      continue;
    }
    
    /* It is unicode */
    
    unichar_len=raptor_utf8_to_unicode_char(NULL, string, len);
    if(unichar_len < 0 || unichar_len > (int)len)
      /* UTF-8 encoding had an error or ended in the middle of a string */
      return 1;

    unichar_len=raptor_utf8_to_unicode_char(&unichar, string, len);
    
    if(unichar < 0x10000) {
      raptor_iostream_write_counted_string(iostr, "\\u", 2);
      raptor_iostream_format_hexadecimal(iostr, unichar, 4);
    } else {
      raptor_iostream_write_counted_string(iostr, "\\U", 2);
      raptor_iostream_format_hexadecimal(iostr, unichar, 8);
    }
    
    unichar_len--; /* since loop does len-- */
    string += unichar_len; len -= unichar_len;

  }

  return 0;
}


static void
raptor_serialize_ntriples_print_statement_part(raptor_iostream* iostr,
                                               const void *term, 
                                               raptor_identifier_type type,
                                               raptor_uri* literal_datatype,
                                               const unsigned char *literal_language) 
{
  switch(type) {
    case RAPTOR_IDENTIFIER_TYPE_LITERAL:
    case RAPTOR_IDENTIFIER_TYPE_XML_LITERAL:
      raptor_iostream_write_byte(iostr, '"');
      raptor_serialize_ntriples_print_string(iostr, (const unsigned char*)term, '"');
      raptor_iostream_write_byte(iostr, '"');
      if(literal_language && type == RAPTOR_IDENTIFIER_TYPE_LITERAL) {
        raptor_iostream_write_byte(iostr, '@');
        raptor_iostream_write_string(iostr, literal_language);
      }
      if(type == RAPTOR_IDENTIFIER_TYPE_XML_LITERAL) {
        raptor_iostream_write_counted_string(iostr, "^^<", 3);
        raptor_iostream_write_string(iostr, raptor_xml_literal_datatype_uri_string);
        raptor_iostream_write_byte(iostr, '>');
      } else if(literal_datatype) {
        raptor_iostream_write_counted_string(iostr, "^^<", 3);
        raptor_iostream_write_string(iostr, raptor_uri_as_string((raptor_uri*)literal_datatype));
        raptor_iostream_write_byte(iostr, '>');
      }

      break;
      
    case RAPTOR_IDENTIFIER_TYPE_ANONYMOUS:
      raptor_iostream_write_counted_string(iostr, "_:", 2);
      raptor_iostream_write_string(iostr, term);
      break;
      
    case RAPTOR_IDENTIFIER_TYPE_ORDINAL:
      raptor_iostream_write_counted_string(iostr, "<http://www.w3.org/1999/02/22-rdf-syntax-ns#_", 45);
      raptor_iostream_write_decimal(iostr, *((int*)term));
      raptor_iostream_write_byte(iostr, '>');
      break;
  
    case RAPTOR_IDENTIFIER_TYPE_RESOURCE:
    case RAPTOR_IDENTIFIER_TYPE_PREDICATE:
      raptor_iostream_write_byte(iostr, '<');
      raptor_serialize_ntriples_print_string(iostr, raptor_uri_as_string((raptor_uri*)term), '\0');
      raptor_iostream_write_byte(iostr, '>');
      break;
      
    default:
      RAPTOR_FATAL2("Unknown type %d", type);
  }
}




/* serialize a statement */
static int
raptor_ntriples_serialize_statement(raptor_serializer* serializer, 
                                    const raptor_statement *statement)
{
  raptor_iostream *iostr=serializer->iostream;

  /* from raptor_print_statement_as_ntriples(statement, stdout); */
  raptor_serialize_ntriples_print_statement_part(iostr,
                                                 statement->subject,
                                                 statement->subject_type,
                                                 NULL, NULL);
  raptor_iostream_write_byte(iostr, ' ');
  raptor_serialize_ntriples_print_statement_part(iostr,
                                                 statement->predicate,
                                                 statement->predicate_type,
                                                 NULL, NULL);
  raptor_iostream_write_byte(iostr, ' ');
  raptor_serialize_ntriples_print_statement_part(iostr,
                                                 statement->object,
                                                 statement->object_type,
                                                 statement->object_literal_datatype,
                                                 statement->object_literal_language);
  raptor_iostream_write_counted_string(iostr, " .\n", 3);

  return 0;
}


#if 0
/* end a serialize */
static int
raptor_ntriples_serialize_end(raptor_serializer* serializer)
{
  return 0;
}
#endif
  
/* finish the serializer factory */
static void
raptor_ntriples_serialize_finish_factory(raptor_serializer_factory* factory)
{

}


static void
raptor_ntriples_serializer_register_factory(raptor_serializer_factory *factory)
{
  factory->context_length     = sizeof(raptor_ntriples_serializer_context);
  
  factory->init                = raptor_ntriples_serialize_init;
  factory->terminate           = raptor_ntriples_serialize_terminate;
  factory->declare_namespace   = raptor_ntriples_serialize_declare_namespace;
  factory->serialize_start     = NULL;
  factory->serialize_statement = raptor_ntriples_serialize_statement;
  factory->serialize_end       = NULL;
  factory->finish_factory      = raptor_ntriples_serialize_finish_factory;
}


void
raptor_init_serializer_ntriples (void) {
  raptor_serializer_register_factory("ntriples",  "N-Triples", 
                                     "text/plain",
                                     NULL,
                                     (const unsigned char*)"http://www.w3.org/TR/rdf-testcases/#ntriples",
                                     &raptor_ntriples_serializer_register_factory);
}


/*
 * Raptor 'Simple' serializer object
 */
typedef struct {
  int dummy;
} raptor_simple_serializer_context;



/* create a new serializer */
static int
raptor_simple_serialize_init(raptor_serializer* serializer, const char *name)
{
  return 0;
}
  

/* destroy a serializer */
static void
raptor_simple_serialize_terminate(raptor_serializer* serializer)
{

}
  

/* serialize a statement */
static int
raptor_simple_serialize_statement(raptor_serializer* serializer, 
                                  const raptor_statement *statement)
{
  raptor_iostream *iostr=serializer->iostream;
  
  /* was: fprintf(stdout, "%s: Statement: ", program); */
  raptor_iostream_write_string(iostr, "Statement: ");

  /* from raptor_print_statement */
  raptor_iostream_write_byte(iostr, '[');

  if(statement->subject_type == RAPTOR_IDENTIFIER_TYPE_ANONYMOUS) {
    raptor_iostream_write_string(iostr, statement->subject);
  } else {
#ifdef RAPTOR_DEBUG
    if(!statement->subject)
      RAPTOR_FATAL1("Statement has NULL subject URI\n");
#endif
    raptor_iostream_write_string(iostr, raptor_uri_as_string((raptor_uri*)statement->subject));
  }

  raptor_iostream_write_counted_string(iostr, ", ", 2);

  if(statement->predicate_type == RAPTOR_IDENTIFIER_TYPE_ORDINAL) {
    raptor_iostream_write_counted_string(iostr, "[rdf:_", 6);
    raptor_iostream_write_decimal(iostr, *((int*)statement->predicate));
    raptor_iostream_write_byte(iostr, ']');
  } else {
#ifdef RAPTOR_DEBUG
    if(!statement->predicate)
      RAPTOR_FATAL1("Statement has NULL predicate URI\n");
#endif
    raptor_iostream_write_string(iostr, raptor_uri_as_string((raptor_uri*)statement->predicate));
  }

  raptor_iostream_write_counted_string(iostr, ", ", 2);

  if(statement->object_type == RAPTOR_IDENTIFIER_TYPE_LITERAL || 
     statement->object_type == RAPTOR_IDENTIFIER_TYPE_XML_LITERAL) {
    if(statement->object_type == RAPTOR_IDENTIFIER_TYPE_XML_LITERAL) {
      raptor_iostream_write_byte(iostr, '<');
      raptor_iostream_write_string(iostr, raptor_xml_literal_datatype_uri_string);
      raptor_iostream_write_byte(iostr, '>');
    } else if(statement->object_literal_datatype) {
      raptor_iostream_write_byte(iostr, '<');
      raptor_iostream_write_string(iostr, raptor_uri_as_string((raptor_uri*)statement->object_literal_datatype));
      raptor_iostream_write_byte(iostr, '>');
    }
    raptor_iostream_write_byte(iostr, '"');
    raptor_iostream_write_string(iostr, statement->object);
    raptor_iostream_write_byte(iostr, '"');
  } else if(statement->object_type == RAPTOR_IDENTIFIER_TYPE_ANONYMOUS)
    raptor_iostream_write_string(iostr, statement->object);
  else if(statement->object_type == RAPTOR_IDENTIFIER_TYPE_ORDINAL) {
    raptor_iostream_write_counted_string(iostr, "[rdf:_", 6);
    raptor_iostream_write_decimal(iostr, *((int*)statement->object));
    raptor_iostream_write_byte(iostr, ']');
  } else {
#ifdef RAPTOR_DEBUG
    if(!statement->object)
      RAPTOR_FATAL1("Statement has NULL object URI\n");
#endif
    raptor_iostream_write_string(iostr, raptor_uri_as_string((raptor_uri*)statement->object));
  }

  raptor_iostream_write_counted_string(iostr, "]\n", 2);

  return 0;
}


  
/* finish the serializer factory */
static void
raptor_simple_serialize_finish_factory(raptor_serializer_factory* factory)
{

}


static void
raptor_simple_serializer_register_factory(raptor_serializer_factory *factory)
{
  factory->context_length     = sizeof(raptor_simple_serializer_context);
  
  factory->init                = raptor_simple_serialize_init;
  factory->terminate           = raptor_simple_serialize_terminate;
  factory->declare_namespace   = NULL;
  factory->serialize_start     = NULL;
  factory->serialize_statement = raptor_simple_serialize_statement;
  factory->serialize_end       = NULL;
  factory->finish_factory      = raptor_simple_serialize_finish_factory;
}


void
raptor_init_serializer_simple (void) {
  raptor_serializer_register_factory("simple",  "A simple format", 
                                     NULL,
                                     NULL,
                                     NULL,
                                     &raptor_simple_serializer_register_factory);
}
