/*
 * Copyright (C) 2008 Instituto Nokia de Tecnologia. All rights reserved.
 *
 * This file is part of QZion.
 *
 * QZion is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * QZion is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with QZion.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef __QZIONRECTANGLE_H__
#define __QZIONRECTANGLE_H__

#include "qzionobject.h"

class QZionRectanglePrivate;

/*!
    \brief A solid rectangle.

    A QZionRectangle is a rectangle that can be put in the canvas.
*/
class QZionRectangle : public QZionObject
{
    Q_OBJECT

public:

    /*!
      Constructs a QZionRectangle with default parameters.
    */
    QZionRectangle(QZionAbstractCanvas *canvas = NULL);

    /*!
      Constructs a QZionRectangle with specified parameters.
    */
    QZionRectangle(QZionAbstractCanvas *canvas,
                   const QColor &color, const QSize &size);

    virtual ~QZionRectangle();

    /*!
      Gets the rectangle color.
    */
    virtual QColor color() const;

    /*!
      Sets the rectangle color.
    */
    virtual void setColor(const QColor &color);

    /*!
      Gets the rectangle size.
    */
    virtual QSize size() const;

    /*!
      Sets the rectangle size.
    */
    virtual void setSize(const QSize &size);

    /*!
      Paint rectangle on canvas.
    */
    virtual void paint(QPainter *p);

    /*!
      Returns rectangle geometry.
    */
    virtual QRect rect() const;

    virtual QColor effectiveColor() const;

    virtual QRect effectiveRect() const;

    virtual void changed();

    virtual bool isClipper() const { return (_clipReferences > 0); }

protected:
    QZionRectanglePrivate *_QZionRectangle_data;

private:
    int _clipReferences;

    void addClippee(QZionObject* clipee);
    void removeClippee(QZionObject* clipee);

    void incClipperRef();
    void decClipperRef();

    /* XXXX */
    friend class QZionObject;
    friend class QZionRectanglePrivate;
};

/*!
    \internal

    Add reference to this rectangle to be used as a clipper. If at least one
    object is using a rectangle as a clipper, it won't be visible.

    \sa decClipperRef()
*/
inline void QZionRectangle::incClipperRef()
{
    _clipReferences++;
}

/*!
    \internal

    Remove reference to this rectangle to be used as a clipper.

    \sa incClipperRef()
*/
inline void QZionRectangle::decClipperRef()
{
    _clipReferences--;
}

#endif
