/*
 * Copyright (C) 2008 Instituto Nokia de Tecnologia. All rights reserved.
 *
 * This file is part of QZion.
 *
 * QZion is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * QZion is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with QZion.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "qzionimage.h"
#include "qzionrectangle.h"

#include <QDebug>
#include <QPixmapCache>
#include <QPainter>
#include <QTransform>

#include "qzionimage_p.h"
#include "qzionobject_p.h"

static int MAX_CACHE_SIZE = 4096;  // 4 mb


QZionImage::QZionImage(QZionAbstractCanvas *canvas)
    : QZionObject(canvas)
{
    QZINIT(QZionImage, canvas);

    QPixmapCache::setCacheLimit(MAX_CACHE_SIZE);
}


QZionImage::QZionImage(QZionAbstractCanvas *canvas, const QString &filepath)
    : QZionObject(canvas), _filepath(filepath)
{
    QZINIT(QZionImage, canvas);

    QPixmapCache::setCacheLimit(MAX_CACHE_SIZE);
    load(filepath);
}

QZionImage::~QZionImage()
{
    QZDELETE(QZionImage);
}

bool QZionImage::load(const QString &filename)
{
    QPixmap result;

    if (QPixmapCache::find(filename, result)) {
        setPixmap(result);
        return true;
    }

    if (!result.load(filename))
        return false;

    _filepath = filename;
    QPixmapCache::insert(filename, result);
    setPixmap(result);

    return true;
}

bool QZionImage::reload()
{
    QZD(QZionImage);

    if (d->pixmap().isNull())
        return true;

    QPixmap result;
    if (!result.load(_filepath)) {
        clear();
        return false;
    }

    QPixmapCache::insert(_filepath, result);
    setPixmap(result);

    return true;
}

void QZionImage::clear()
{
    setPixmap(QPixmap());
}


QPixmap QZionImage::pixmap() const
{
    QZD(QZionImage);
    return d->pixmap();
}


void QZionImage::setPixmap(const QPixmap &p)
{
    QZD(QZionImage);
    d->setPixmap(p);
    if (d->size.isEmpty())
        d->size = pixmap().size();
}


QSize QZionImage::size() const
{
    QZD(QZionImage);
    return d->size;
}


void QZionImage::setSize(const QSize &size)
{
    QSize fp = pixmap().size();
    if (fp.width() <= 0 || fp.height() <= 0)
        return;

    QZD(QZionImage);
    d->size = size;
    double dx = (double)size.width() / fp.width();
    double dy = (double)size.height() / fp.height();
    d->setTransform(QTransform().scale(dx, dy));

    QZionObject::setSize(size);
}


QRect QZionImage::rect() const
{
    return QRect(pos(), size());
}


QColor QZionImage::color() const
{
    return Qt::white;
}


void QZionImage::setColor(const QColor &color)
{
    setOpacity(color.alpha());
}


void QZionImage::paint(QPainter *painter)
{
    Q_UNUSED(painter);
}

void QZionImage::changed()
{
    QZD(QZionImage);

    d->updateEffectiveColor();
    QZionObject::changed();
}
