// This module implements the QextScintillaLexerProperties class.
//
// Copyright (c) 2006
// 	Riverbank Computing Limited <info@riverbankcomputing.co.uk>
// 
// This file is part of QScintilla.
// 
// This copy of QScintilla is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option) any
// later version.
// 
// QScintilla is supplied in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
// details.
// 
// You should have received a copy of the GNU General Public License along with
// QScintilla; see the file LICENSE.  If not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.


#include <qcolor.h>
#include <qfont.h>
#include <qsettings.h>

#include "qextscintillalexerproperties.h"


// The ctor.
QextScintillaLexerProperties::QextScintillaLexerProperties(QObject *parent,const char *name)
	: QextScintillaLexer(parent,name), fold_compact(TRUE)
{
}


// The dtor.
QextScintillaLexerProperties::~QextScintillaLexerProperties()
{
}


// Returns the language name.
const char *QextScintillaLexerProperties::language() const
{
	return "Properties";
}


// Returns the lexer name.
const char *QextScintillaLexerProperties::lexer() const
{
	return "props";
}


// Return the string of characters that comprise a word.
const char *QextScintillaLexerProperties::wordCharacters() const
{
	return "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ-";
}


// Returns the foreground colour of the text for a style.
QColor QextScintillaLexerProperties::color(int style) const
{
	switch (style)
	{
	case Comment:
		return QColor(0x00,0x7f,0x7f);

	case Section:
		return QColor(0x7f,0x00,0x7f);

	case Assignment:
		return QColor(0xb0,0x60,0x00);

	case DefaultValue:
		return QColor(0x7f,0x7f,0x00);
	}

	return QextScintillaLexer::color(style);
}


// Returns the end-of-line fill for a style.
bool QextScintillaLexerProperties::eolFill(int style) const
{
	return (style == Section);
}


// Returns the font of the text for a style.
QFont QextScintillaLexerProperties::font(int style) const
{
	QFont f;

	if (style == Comment)
#if defined(Q_OS_WIN)
		f = QFont("Comic Sans MS",9);
#else
		f = QFont("Bitstream Vera Serif",9);
#endif
	else
		f = QextScintillaLexer::font(style);

	return f;
}


// Returns the user name of a style.
QString QextScintillaLexerProperties::description(int style) const
{
	switch (style)
	{
	case Default:
		return tr("Default");

	case Comment:
		return tr("Comment");

	case Section:
		return tr("Section");

	case Assignment:
		return tr("Assignment");

	case DefaultValue:
		return tr("Default value");
	}

	return QString::null;
}


// Returns the background colour of the text for a style.
QColor QextScintillaLexerProperties::paper(int style) const
{
	if (style == Section)
		return QColor(0xe0,0xf0,0xf0);

	return QextScintillaLexer::paper(style);
}


// Refresh all properties.
void QextScintillaLexerProperties::refreshProperties()
{
	setCompactProp();
}


// Read properties from the settings.
bool QextScintillaLexerProperties::readProperties(QSettings &qs,const QString &prefix)
{
	int rc = TRUE;
	bool ok, flag;

	// Read the fold compact flag.
	flag = qs.readBoolEntry(prefix + "foldcompact",TRUE,&ok);

	if (ok)
		fold_compact = flag;
	else
		rc = FALSE;

	return rc;
}


// Write properties to the settings.
bool QextScintillaLexerProperties::writeProperties(QSettings &qs,const QString &prefix) const
{
	int rc = TRUE;

	// Write the fold compact flag.
	if (!qs.writeEntry(prefix + "foldcompact",fold_compact))
		rc = FALSE;

	return rc;
}


// Return TRUE if folds are compact.
bool QextScintillaLexerProperties::foldCompact() const
{
	return fold_compact;
}


// Set if folds are compact
void QextScintillaLexerProperties::setFoldCompact(bool fold)
{
	fold_compact = fold;

	setCompactProp();
}


// Set the "fold.compact" property.
void QextScintillaLexerProperties::setCompactProp()
{
	emit propertyChanged("fold.compact",(fold_compact ? "1" : "0"));
}
