/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: cmdevt.hxx,v $
 *
 *  $Revision: 1.14 $
 *
 *  last change: $Author: rt $ $Date: 2005/09/09 10:50:17 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

#ifndef _VCL_CMDEVT_HXX
#define _VCL_CMDEVT_HXX

#ifndef _GEN_HXX
#include <tools/gen.hxx>
#endif
#ifndef _SV_H
#include <sv.h>
#endif

#ifndef _VCL_DLLAPI_H
#include "dllapi.h"
#endif

#ifndef _VCL_KEYCOD_HXX
#include <keycod.hxx>
#endif
#ifndef _VCL_FONT_HXX
#include <font.hxx>
#endif

// ---------------------------
// - CommandExtTextInputData -
// ---------------------------

#define EXTTEXTINPUT_ATTR_GRAYWAVELINE          ((USHORT)0x0100)
#define EXTTEXTINPUT_ATTR_UNDERLINE             ((USHORT)0x0200)
#define EXTTEXTINPUT_ATTR_BOLDUNDERLINE         ((USHORT)0x0400)
#define EXTTEXTINPUT_ATTR_DOTTEDUNDERLINE       ((USHORT)0x0800)
#define EXTTEXTINPUT_ATTR_DASHDOTUNDERLINE      ((USHORT)0x1000)
#define EXTTEXTINPUT_ATTR_HIGHLIGHT             ((USHORT)0x2000)
#define EXTTEXTINPUT_ATTR_REDTEXT               ((USHORT)0x4000)
#define EXTTEXTINPUT_ATTR_HALFTONETEXT          ((USHORT)0x8000)

#define EXTTEXTINPUT_CURSOR_INVISIBLE           ((USHORT)0x0001)
#define EXTTEXTINPUT_CURSOR_OVERWRITE           ((USHORT)0x0002)

class VCL_DLLPUBLIC CommandExtTextInputData
{
private:
    XubString           maText;
    USHORT*             mpTextAttr;
    xub_StrLen          mnCursorPos;
    xub_StrLen          mnDeltaStart;
    xub_StrLen          mnOldTextLen;
    USHORT              mnCursorFlags;
    BOOL                mbOnlyCursor;

public:
                        CommandExtTextInputData();
                        CommandExtTextInputData( const XubString& rText,
                                                 const USHORT* pTextAttr,
                                                 xub_StrLen nCursorPos,
                                                 USHORT nCursorFlags,
                                                 xub_StrLen nDeltaStart,
                                                 xub_StrLen nOldTextLen,
                                                 BOOL bOnlyCursor );
                        CommandExtTextInputData( const CommandExtTextInputData& rData );
                        ~CommandExtTextInputData();

    const XubString&    GetText() const { return maText; }
    const USHORT*       GetTextAttr() const { return mpTextAttr; }
    USHORT              GetCharTextAttr( USHORT nIndex ) const;
    xub_StrLen          GetCursorPos() const { return mnCursorPos; }
    BOOL                IsCursorVisible() const { return (mnCursorFlags & EXTTEXTINPUT_CURSOR_INVISIBLE) == 0; }
    BOOL                IsCursorOverwrite() const { return (mnCursorFlags & EXTTEXTINPUT_CURSOR_OVERWRITE) != 0; }
    USHORT              GetCursorFlags() const { return mnCursorFlags; }
    xub_StrLen          GetDeltaStart() const { return mnDeltaStart; }
    xub_StrLen          GetOldTextLen() const { return mnOldTextLen; }
    BOOL                IsOnlyCursorChanged() const { return mbOnlyCursor; }
};

inline USHORT CommandExtTextInputData::GetCharTextAttr( USHORT nIndex ) const
{
    if ( mpTextAttr && (nIndex < maText.Len()) )
        return mpTextAttr[nIndex];
    else
        return 0;
}

// ---------------------------
// - CommandInputContextData -
// ---------------------------

class VCL_DLLPUBLIC CommandInputContextData
{
private:
    LanguageType    meLanguage;

public:
                    CommandInputContextData();
                    CommandInputContextData( LanguageType eLang );

    LanguageType    GetLanguage() const { return meLanguage; }
};

inline CommandInputContextData::CommandInputContextData()
{
    meLanguage = LANGUAGE_DONTKNOW;
}

inline CommandInputContextData::CommandInputContextData( LanguageType eLang )
{
    meLanguage = eLang;
}

// --------------------
// - CommandWheelData -
// --------------------

#define COMMAND_WHEEL_SCROLL            ((USHORT)0x0001)
#define COMMAND_WHEEL_ZOOM              ((USHORT)0x0002)
#define COMMAND_WHEEL_DATAZOOM          ((USHORT)0x0004)

#define COMMAND_WHEEL_PAGESCROLL        ((ULONG)0xFFFFFFFF)

class VCL_DLLPUBLIC CommandWheelData
{
private:
    long            mnDelta;
    long            mnNotchDelta;
    ULONG           mnLines;
    USHORT          mnMode;
    USHORT          mnCode;
    BOOL            mbHorz;

public:
                    CommandWheelData();
                    CommandWheelData( long nWheelDelta, long nWheelNotchDelta,
                                      ULONG nScrollLines,
                                      USHORT nWheelMode, USHORT nKeyModifier,
                                      BOOL mbHorz = FALSE );

    long            GetDelta() const { return mnDelta; }
    long            GetNotchDelta() const { return mnNotchDelta; }
    ULONG           GetScrollLines() const { return mnLines; }
    BOOL            IsHorz() const { return mbHorz; }

    USHORT          GetMode() const { return mnMode; }

    USHORT          GetModifier() const
                        { return (mnCode & (KEY_SHIFT | KEY_MOD1 | KEY_MOD2)); }
    BOOL            IsShift() const
                        { return ((mnCode & KEY_SHIFT) != 0); }
    BOOL            IsMod1() const
                        { return ((mnCode & KEY_MOD1) != 0); }
    BOOL            IsMod2() const
                        { return ((mnCode & KEY_MOD2) != 0); }
};

inline CommandWheelData::CommandWheelData()
{
    mnDelta         = 0;
    mnNotchDelta    = 0;
    mnLines         = 0;
    mnMode          = 0;
    mnCode          = 0;
    mbHorz          = FALSE;
}

inline CommandWheelData::CommandWheelData( long nWheelDelta, long nWheelNotchDelta,
                                           ULONG nScrollLines,
                                           USHORT nWheelMode, USHORT nKeyModifier,
                                           BOOL bHorz )
{
    mnDelta         = nWheelDelta;
    mnNotchDelta    = nWheelNotchDelta;
    mnLines         = nScrollLines;
    mnMode          = nWheelMode;
    mnCode          = nKeyModifier;
    mbHorz          = bHorz;
}

// ---------------------
// - CommandScrollData -
// ---------------------

class VCL_DLLPUBLIC CommandScrollData
{
private:
    long            mnDeltaX;
    long            mnDeltaY;

public:
                    CommandScrollData();
                    CommandScrollData( long nDeltaX, long nDeltaY );

    long            GetDeltaX() const { return mnDeltaX; }
    long            GetDeltaY() const { return mnDeltaY; }
};

inline CommandScrollData::CommandScrollData()
{
    mnDeltaX    = 0;
    mnDeltaY    = 0;
}

inline CommandScrollData::CommandScrollData( long nDeltaX, long nDeltaY )
{
    mnDeltaX    = nDeltaX;
    mnDeltaY    = nDeltaY;
}

// --------------------
// - CommandVoiceData -
// --------------------

enum DictationCommandType
{
    VOICECOMMANDTYPE_DICTATION,
    VOICECOMMANDTYPE_CONTROL
};

class VCL_DLLPUBLIC CommandVoiceData
{
private:
    XubString               maText;
    USHORT                  mnCommand;
    DictationCommandType    meType;

public:
                            CommandVoiceData();
                            CommandVoiceData( const XubString& rStr,
                                              DictationCommandType eType,
                                              USHORT nCommand );

    const XubString&        GetText() const { return maText; }
    USHORT                  GetCommand() const { return mnCommand; }
    DictationCommandType    GetType() const { return meType; }
};

inline CommandVoiceData::CommandVoiceData()
{
    meType = VOICECOMMANDTYPE_DICTATION;
    mnCommand = 0;
}

inline CommandVoiceData::CommandVoiceData( const XubString& rStr,
                                           DictationCommandType eType,
                                           USHORT nCommand ) :
    maText( rStr ),
    mnCommand( nCommand ),
    meType( eType )
{
}

// ---------------------
// - CommandModKeyData -
// ---------------------

class VCL_DLLPUBLIC CommandModKeyData
{
private:
    USHORT          mnCode;

public:
                    CommandModKeyData();
                    CommandModKeyData( USHORT nCode );

    BOOL            IsShift()   const { return mnCode & MODKEY_SHIFT; }
    BOOL            IsMod1()    const { return mnCode & MODKEY_MOD1; }
    BOOL            IsMod2()    const { return mnCode & MODKEY_MOD2; }

    BOOL            IsLeftShift() const { return mnCode & MODKEY_LSHIFT; }
    BOOL            IsLeftMod1()  const { return mnCode & MODKEY_LMOD1; }
    BOOL            IsLeftMod2()  const { return mnCode & MODKEY_LMOD2; }

    BOOL            IsRightShift() const { return mnCode & MODKEY_RSHIFT; }
    BOOL            IsRightMod1()  const { return mnCode & MODKEY_RMOD1; }
    BOOL            IsRightMod2()  const { return mnCode & MODKEY_RMOD2; }
};

inline CommandModKeyData::CommandModKeyData()
{
    mnCode = 0L;
}

inline CommandModKeyData::CommandModKeyData( USHORT nCode )
{
    mnCode = nCode;
}


// ----------------
// - CommandEvent -
// ----------------

#define COMMAND_CONTEXTMENU             ((USHORT)1)
#define COMMAND_STARTDRAG               ((USHORT)2)
#define COMMAND_WHEEL                   ((USHORT)3)
#define COMMAND_STARTAUTOSCROLL         ((USHORT)4)
#define COMMAND_AUTOSCROLL              ((USHORT)5)
#define COMMAND_VOICE                   ((USHORT)6)
#define COMMAND_STARTEXTTEXTINPUT       ((USHORT)7)
#define COMMAND_EXTTEXTINPUT            ((USHORT)8)
#define COMMAND_ENDEXTTEXTINPUT         ((USHORT)9)
#define COMMAND_INPUTCONTEXTCHANGE      ((USHORT)10)
#define COMMAND_CURSORPOS               ((USHORT)11)
#define COMMAND_PASTESELECTION          ((USHORT)12)
#define COMMAND_MODKEYCHANGE            ((USHORT)13)
#define COMMAND_HANGUL_HANJA_CONVERSION ((USHORT)14)

#define COMMAND_USER                    ((USHORT)4096)

class VCL_DLLPUBLIC CommandEvent
{
private:
    Point                               maPos;
    void*                               mpData;
    USHORT                              mnCommand;
    BOOL                                mbMouseEvent;

public:
                                        CommandEvent();
                                        CommandEvent( const Point& rMousePos, USHORT nCmd,
                                                      BOOL bMEvt = FALSE, const void* pCmdData = NULL );

    USHORT                              GetCommand() const { return mnCommand; }
    const Point&                        GetMousePosPixel() const { return maPos; }
    BOOL                                IsMouseEvent() const { return mbMouseEvent; }
    void*                               GetData() const { return mpData; }

    const CommandExtTextInputData*      GetExtTextInputData() const;
    const CommandInputContextData*      GetInputContextChangeData() const;
    const CommandWheelData*             GetWheelData() const;
    const CommandScrollData*            GetAutoScrollData() const;
    const CommandVoiceData*             GetVoiceData() const;
    const CommandModKeyData*			GetModKeyData() const;
};

inline CommandEvent::CommandEvent()
{
    mpData          = NULL;
    mnCommand       = 0;
    mbMouseEvent    = FALSE;
}

inline CommandEvent::CommandEvent( const Point& rMousePos,
                                   USHORT nCmd, BOOL bMEvt, const void* pCmdData ) :
            maPos( rMousePos )
{
    mpData          = (void*)pCmdData;
    mnCommand       = nCmd;
    mbMouseEvent    = bMEvt;
}

inline const CommandExtTextInputData* CommandEvent::GetExtTextInputData() const
{
    if ( mnCommand == COMMAND_EXTTEXTINPUT )
        return (const CommandExtTextInputData*)mpData;
    else
        return NULL;
}

inline const CommandInputContextData* CommandEvent::GetInputContextChangeData() const
{
    if ( mnCommand == COMMAND_INPUTCONTEXTCHANGE )
        return (const CommandInputContextData*)mpData;
    else
        return NULL;
}

inline const CommandWheelData* CommandEvent::GetWheelData() const
{
    if ( mnCommand == COMMAND_WHEEL )
        return (const CommandWheelData*)mpData;
    else
        return NULL;
}

inline const CommandScrollData* CommandEvent::GetAutoScrollData() const
{
    if ( mnCommand == COMMAND_AUTOSCROLL )
        return (const CommandScrollData*)mpData;
    else
        return NULL;
}

inline const CommandVoiceData* CommandEvent::GetVoiceData() const
{
    if ( mnCommand == COMMAND_VOICE )
        return (const CommandVoiceData*)mpData;
    else
        return NULL;
}

inline const CommandModKeyData* CommandEvent::GetModKeyData() const
{
    if( mnCommand == COMMAND_MODKEYCHANGE )
        return (const CommandModKeyData*)mpData;
    else
        return NULL;
}

#endif // _VCL_CMDEVT_HXX
