/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: ObjectBarManager.cxx,v $
 *
 *  $Revision: 1.16 $
 *
 *  last change: $Author: rt $ $Date: 2005/09/09 06:59:55 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

#include "ObjectBarManager.hxx"

#ifndef SD_VIEW_SHELL_HXX
#include "ViewShell.hxx"
#endif
#ifndef SD_VIEW_SHELL_BASE_HXX
#include "ViewShellBase.hxx"
#endif
#ifndef SD_VIEW_SHELL_MANAGER_HXX
#include "ViewShellManager.hxx"
#endif
#ifndef SD_DRAW_VIEW_SHELL_HXX
#include "DrawViewShell.hxx"
#endif

#include <rtl/ustring.hxx>
#ifndef _COM_SUN_STAR_BEANS_XPROPERTYSET_HPP_
#include <com/sun/star/beans/XPropertySet.hpp>
#endif
#ifndef _COM_SUN_STAR_FRAME_XLAYOUTMANAGER_HPP_
#include <com/sun/star/frame/XLayoutManager.hpp>
#endif

#include <hash_map>
#include <list>
#include <svx/extrusionbar.hxx>
#include <svx/fontworkbar.hxx>
#include <svx/dialogs.hrc>
#include <sfx2/viewfrm.hxx>


using namespace ::rtl;
using namespace ::com::sun::star;
using namespace ::com::sun::star::uno;

namespace sd {

namespace {

enum StackPosition { SP_TOP, SP_BOTTOM, SP_BELOW_SHELL };

class ActiveShellDescriptor 
{
public:
    ActiveShellDescriptor (SfxShell* pShell, ShellId nId, 
        StackPosition ePosition)
        : mpShell(pShell), mnId(nId), mePosition(ePosition),
          mbIsOnStack(false) {}
    SfxShell* mpShell;
    ShellId mnId;
    StackPosition mePosition;
    bool mbIsOnStack;
};

// Define some predicates that are used for searching in STL containers.
class IsShell
{
public:
    IsShell (const ::SfxShell* pShell) : mpShell(pShell) {}
    bool operator() (const ActiveShellDescriptor& rDescriptor)
    {
        return rDescriptor.mpShell == mpShell;
    }
    const SfxShell* mpShell;
};
class IsId
{
public:
    IsId (ShellId nId) : mnId(nId) {}
    bool operator() (const ActiveShellDescriptor& rDescriptor)
    {
        return rDescriptor.mnId == mnId;
    }
    ShellId mnId;
};


} // end of anonymous namespace.


//===== ObjectBarManager::Implementation declaration ==========================

/** Implementation class of ObjectBarManager.  For documentation of methods
    see ObjectBarManager.hxx.
*/
class ObjectBarManager::Implementation
{
public:
    Implementation (ViewShell& rViewShell);
    ~Implementation (void);

    // Define the list types used by the ObjectBarManager class.  They are
    // forward-declared in the header file.
    typedef ::std::hash_map<ShellId, ObjectBarFactory*>  
        SpecializedFactoryList;
    typedef ::std::list<ActiveShellDescriptor> ActiveObjectBarList;
    typedef ::std::hash_map<ShellId, StackPosition>
        PreferredStackPositions;

    ShellId mnDefaultObjectBarId;

    void EnableObjectBarSwitching (bool bEnable = true);

    void RegisterDefaultFactory (
        ::std::auto_ptr<ObjectBarFactory> rFactory);
    void RegisterFactory (
        ShellId nId, 
        ::std::auto_ptr<ObjectBarFactory> pFactory);
    void Clear (void);
    SfxShell* ActivateObjectBar (ShellId nId);
    void DeactivateObjectBar (SfxShell* pObjectBar);
    void SwitchObjectBar (ShellId nId);
    SfxShell* GetObjectBar (ShellId nId) const;
    ShellId GetObjectBarId (const SfxShell* pObjectBar) const;
    ShellId GetTopObjectBarId (void) const;
    SfxShell* GetTopObjectBar (void) const;
    void InvalidateAllObjectBars (void) const;
    void GetLowerShellList (::std::vector<SfxShell*>& rShellList) const;
    void GetUpperShellList (::std::vector<SfxShell*>& rShellList) const;

    void SetPreferredStackPosition (ShellId nId, StackPosition ePosition);

    void SelectionHasChanged (SdrView* pView);

    /** Return the layout manager of the frame that can be used to e.g. show
        and hide toolbars.
    */
    Reference< ::com::sun::star::frame::XLayoutManager >
        GetLayoutManager (void);

    /** Return the full tool bar name that can be used to identify a tool
        bar resource.  This is done by prefixing the given name by something
        like private:resource/toolbar.
    */
    OUString GetToolBarResourceName (const OUString& rToolBarName);

private:
    ::osl::Mutex maMutex;
    ViewShell& mrViewShell;
    /** The manager used for stacking object bars onto a view shell.
    */
    ViewShellManager& mrViewShellManager;

    /** Flag that determines whether switching the active object bars is
        allowed.
    */
    bool mbIsObjectBarSwitchingEnabled;

    ::std::auto_ptr<ObjectBarFactory> mpDefaultFactory;
    SpecializedFactoryList maSpecializedFactories;

    /** The currently active (visible) object bars.  This list can be
        interpreted as a stack with the top most object bar at its front.
    */
    ActiveObjectBarList maActiveObjectBars;

    /** The preferred stack positions both for the active and for the
        inactive object bars.
    */
    PreferredStackPositions maPreferredStackPositions;

    /** Return <TRUE/> when the object bar with the given id is active.
    */
    bool IsActive (ShellId nId);

    /** Return the preferred stack position for the shell identified by the
        given id.
        @returns
            When no preferred position was set for the specified shell then
            the given default position is returned.
    */
    StackPosition GetPreferredStackPosition (ShellId nId, StackPosition eDefaultPosition);

    /** Update the stack of active object bars so that it looks like the
        given stack.
    */
    void UpdateStack (ActiveObjectBarList& rRequestedStack);

    void TraceStack (ActiveObjectBarList& rStack);
};




//===== ObjectBarManager ======================================================

ObjectBarManager::ObjectBarManager (ViewShell& rViewShell)
    : mpImpl(::std::auto_ptr<Implementation>(new Implementation(rViewShell)))
{
}




ObjectBarManager::~ObjectBarManager (void)
{
    EnableObjectBarSwitching();
    Clear();
}




void ObjectBarManager::RegisterDefaultFactory (
    ::std::auto_ptr<ObjectBarFactory> pFactory)
{
    mpImpl->RegisterDefaultFactory(pFactory);
}




void ObjectBarManager::RegisterFactory (
        ShellId nId, 
        ::std::auto_ptr<ObjectBarFactory> pFactory)
{
    mpImpl->RegisterFactory (nId, pFactory);
}




void ObjectBarManager::EnableObjectBarSwitching (bool bEnable)
{
    mpImpl->EnableObjectBarSwitching (bEnable);
}




void ObjectBarManager::DisableObjectBarSwitching (void)
{
    mpImpl->EnableObjectBarSwitching (false);
}




void ObjectBarManager::Clear (void)
{
    mpImpl->Clear();
}




SfxShell* ObjectBarManager::ActivateObjectBar (ShellId nId)
{
    return mpImpl->ActivateObjectBar(nId);
}




void ObjectBarManager::DeactivateObjectBar (SfxShell* pObjectBar)
{
    mpImpl->DeactivateObjectBar(pObjectBar);
}




void ObjectBarManager::SwitchObjectBar (ShellId nId)
{
    mpImpl->SwitchObjectBar (nId);
}




SfxShell* ObjectBarManager::GetObjectBar (ShellId nId) const
{
    return mpImpl->GetObjectBar (nId);
}




ShellId ObjectBarManager::GetObjectBarId (const SfxShell* pObjectBar) const
{
    return mpImpl->GetObjectBarId (pObjectBar);
}




ShellId ObjectBarManager::GetTopObjectBarId (void) const
{
    return mpImpl->GetTopObjectBarId ();
}




SfxShell* ObjectBarManager::GetTopObjectBar (void) const
{
    return mpImpl->GetTopObjectBar();
}




void ObjectBarManager::InvalidateAllObjectBars (void) const
{
    mpImpl->InvalidateAllObjectBars();
}




void ObjectBarManager::GetLowerShellList (
    ::std::vector<SfxShell*>& rShellList) const
{
    mpImpl->GetLowerShellList (rShellList);
}




void ObjectBarManager::GetUpperShellList (
    ::std::vector<SfxShell*>& rShellList) const
{
    mpImpl->GetUpperShellList (rShellList);
}




void ObjectBarManager::MoveToTop (USHORT nId)
{
    mpImpl->SetPreferredStackPosition (nId, SP_TOP);
}




void ObjectBarManager::MoveToBottom (USHORT nId)
{
    mpImpl->SetPreferredStackPosition (nId, SP_BOTTOM);
}




void ObjectBarManager::MoveBelowShell (USHORT nId)
{
    mpImpl->SetPreferredStackPosition (nId, SP_BELOW_SHELL);
}




void ObjectBarManager::SetDefaultObjectBarId (ShellId nId)
{
    mpImpl->mnDefaultObjectBarId = nId;
}




ShellId ObjectBarManager::GetDefaultObjectBarId (void)
{
    return mpImpl->mnDefaultObjectBarId;
}




void ObjectBarManager::SelectionHasChanged (SdrView* pView)
{
    mpImpl->SelectionHasChanged (pView);
}




void ObjectBarManager::ShowToolBar (const ::rtl::OUString& rName)
{
    Reference< ::com::sun::star::frame::XLayoutManager > xLayouter (
        mpImpl->GetLayoutManager());
    if (xLayouter.is())
    {
        OUString sFullName (mpImpl->GetToolBarResourceName(rName));
        xLayouter->createElement (sFullName);
        xLayouter->requestElement (sFullName);
    }
} 




void ObjectBarManager::HideToolBar (const ::rtl::OUString& rName)
{
    Reference< ::com::sun::star::frame::XLayoutManager > xLayouter (
        mpImpl->GetLayoutManager());
    if (xLayouter.is())
    {
        OUString sFullName (mpImpl->GetToolBarResourceName(rName));
        xLayouter->destroyElement (sFullName);
    }
}




//===== ObjectBarManager::Implementation implementation =======================

ObjectBarManager::Implementation::Implementation (ViewShell& rViewShell)
    : mrViewShell(rViewShell),
      mrViewShellManager(mrViewShell.GetViewShellBase().GetViewShellManager()),
      mbIsObjectBarSwitchingEnabled(false),
      mpDefaultFactory (NULL),
      maSpecializedFactories(),
      maActiveObjectBars (),
      maPreferredStackPositions(),
      mnDefaultObjectBarId (snInvalidShellId)
{
}




ObjectBarManager::Implementation::~Implementation (void)
{
    // We have the ownership of the factories and because they can not be
    // auto_ptrs in an stl list we have to delete them now.
    for (SpecializedFactoryList::iterator aI (maSpecializedFactories.begin());
         aI!=maSpecializedFactories.end();
         aI++)
        delete aI->second;
    ObjectBarFactory* pFactory = mpDefaultFactory.release();
    delete pFactory;
}




void ObjectBarManager::Implementation::EnableObjectBarSwitching (bool bEnable)
{
    mbIsObjectBarSwitchingEnabled = bEnable;
}




void ObjectBarManager::Implementation::RegisterDefaultFactory (
    ::std::auto_ptr<ObjectBarFactory> pFactory)
{
    mpDefaultFactory = pFactory;
}




void ObjectBarManager::Implementation::RegisterFactory (
        ShellId nId, 
        ::std::auto_ptr<ObjectBarFactory> pFactory)
{
    maSpecializedFactories[nId] = pFactory.get();
    pFactory.release();
}




void ObjectBarManager::Implementation::Clear (void)
{
    if (mbIsObjectBarSwitchingEnabled)
    {
        ViewShellManager::UpdateLock aLock (mrViewShellManager);

        while ( ! maActiveObjectBars.empty())
            DeactivateObjectBar (maActiveObjectBars.back().mpShell);
    }
    else
    {
        maActiveObjectBars.clear();
    }
}




StackPosition ObjectBarManager::Implementation::GetPreferredStackPosition (
    ShellId nId, 
    StackPosition eDefaultPosition)
{
    StackPosition ePosition = eDefaultPosition;
    PreferredStackPositions::iterator aIterator (maPreferredStackPositions.find(nId));
    if (aIterator != maPreferredStackPositions.end())
        ePosition = aIterator->second;
    return ePosition;
}




SfxShell* ObjectBarManager::Implementation::ActivateObjectBar (ShellId nId)
{
    ::osl::MutexGuard aGuard (maMutex);

    SfxShell* pObjectBar = NULL;

    do
    {
        if ( ! mbIsObjectBarSwitchingEnabled)
            break;

        // Do not activate an object bar that is already active.  Requesting
        // this is not exactly an error but may be an indication of one.
        if (IsActive (nId))
            break;

        // Use a specialized factory when that exists or the default factory
        // otherwise.
        ObjectBarFactory* pFactory = mpDefaultFactory.get();
        SpecializedFactoryList::iterator aI (maSpecializedFactories.find(nId));
        if (aI != maSpecializedFactories.end())
            pFactory = aI->second;
        if (pFactory == NULL)
            break;

        // Create view shell with the factory.
        pObjectBar = pFactory->CreateShell (nId, NULL);

        // Put shell on the stack of the active view shells.
        if (pObjectBar == NULL)
            break;

        ViewShellManager::UpdateLock aLock (mrViewShellManager);

        StackPosition ePosition = GetPreferredStackPosition(nId,SP_TOP);
        ActiveShellDescriptor aDescriptor (pObjectBar, nId, ePosition);
        switch (ePosition)
        {
            case SP_TOP: 
                maActiveObjectBars.push_back (aDescriptor);
                break;

            case SP_BOTTOM: 
            case SP_BELOW_SHELL: 
                maActiveObjectBars.push_front (aDescriptor);
                break;
        }
    }
    while (false);
    
    return pObjectBar;
}




void ObjectBarManager::Implementation::DeactivateObjectBar (
    SfxShell* pObjectBar)
{
    if (mbIsObjectBarSwitchingEnabled)
    {
        // First try to find the object bar in the list of active object bars.
        ActiveObjectBarList::iterator aI (::std::find_if (
            maActiveObjectBars.begin(),
            maActiveObjectBars.end(),
            IsShell(pObjectBar)));
        if (aI != maActiveObjectBars.end())
        {
            ViewShellManager::UpdateLock aLock (mrViewShellManager);

            mrViewShellManager.InvalidateShellStack(pObjectBar);

            // Look up the factory that created the shell and tell it that
            // the shell is no longer in use.
            ObjectBarFactory* pFactory = mpDefaultFactory.get();
            SpecializedFactoryList::iterator iFactory (
                maSpecializedFactories.find(aI->mnId));
            if (iFactory != maSpecializedFactories.end())
                pFactory = iFactory->second;
            if (pFactory != NULL)
                pFactory->ReleaseShell (pObjectBar);

            maActiveObjectBars.erase (aI);
        }
    }
}




// Switching the (main) object bar is done in two steps:
// 1. First we create a local stack of tool bar shells as we would like to
// have it.
// 2. We then compare it to the current stack of tool bars and change that
// so that it looks like the local stack.  We do that with the minimal
// amount of stack changes.
void ObjectBarManager::Implementation::SwitchObjectBar (ShellId nId)
{
    // Switch the object bar only when a) that is generally allowed and 
    // b) the requested object bar is not already the active top-most object
    // bar.
    if (mbIsObjectBarSwitchingEnabled && GetTopObjectBarId()!=nId)
    {
        // 1. Create the requested shell stack.
        ActiveObjectBarList aRequestedStack;

        // Determine the preferred stack position of the form tool box.
        StackPosition ePosition (GetPreferredStackPosition(RID_FORMLAYER_TOOLBOX, SP_BOTTOM));

        // Maybe put the form shell below the view shell.
        if (ePosition != SP_TOP)
            aRequestedStack.push_back(ActiveShellDescriptor(NULL,RID_FORMLAYER_TOOLBOX,ePosition));

        // Push the main tool bar on the stack.
        aRequestedStack.push_back(
            ActiveShellDescriptor(NULL,nId,SP_TOP));

        // Maybe put the form shell above all other shells.
        if (mrViewShell.ISA(DrawViewShell) && ePosition==SP_TOP)
            aRequestedStack.push_back(
                ActiveShellDescriptor(NULL,RID_FORMLAYER_TOOLBOX,SP_TOP));


        // 2. Update the shell stack.
        UpdateStack(aRequestedStack);
    }
}


void ObjectBarManager::Implementation::UpdateStack (ActiveObjectBarList& rRequestedStack)
{
    ViewShellManager::UpdateLock aLock (mrViewShellManager);

    // Determine the lowest shell in which the two stacks differ.
    ActiveObjectBarList::const_iterator iRequestedShell (rRequestedStack.begin());
    ActiveObjectBarList::const_iterator iActiveShell (maActiveObjectBars.begin());

    OSL_TRACE("requested stack");
    TraceStack(rRequestedStack);
    OSL_TRACE("active stack");
    TraceStack(maActiveObjectBars);

    while (iRequestedShell!=rRequestedStack.end()
        && iActiveShell!=maActiveObjectBars.end()
        && iRequestedShell->mnId==iActiveShell->mnId)
    {
        ++iRequestedShell;
        ++iActiveShell;
    }

    // Deactivate the shells above and including the differing one on the
    // active stack.
    ::std::vector<SfxShell*> aShellsToDeactivate;
    while (iActiveShell != maActiveObjectBars.end())
    {
        aShellsToDeactivate.push_back(iActiveShell->mpShell);
        ++iActiveShell;
    }
    while ( ! aShellsToDeactivate.empty())
    {
        OSL_TRACE("deactivating object bar %p",
            aShellsToDeactivate.back());
        DeactivateObjectBar(aShellsToDeactivate.back());
        aShellsToDeactivate.pop_back();
    }

    // Activate the shells above and including the differing one on the
    // requested stack.
    while (iRequestedShell != rRequestedStack.end())
    {
        OSL_TRACE("activating object bar %p %d",
            iRequestedShell->mpShell,iRequestedShell->mnId);
        ActivateObjectBar(iRequestedShell->mnId);
        ++iRequestedShell;
    }
    OSL_TRACE("active stack at exit");
    TraceStack(maActiveObjectBars);
}




SfxShell* ObjectBarManager::Implementation::GetObjectBar (ShellId nId) const
{
    SfxShell* pShell = NULL;

    ActiveObjectBarList::const_iterator aI (::std::find_if (
        maActiveObjectBars.begin(),
        maActiveObjectBars.end(),
        IsId(nId)));
    if (aI != maActiveObjectBars.end())
        pShell = aI->mpShell;

    return pShell;
}




ShellId ObjectBarManager::Implementation::GetObjectBarId (
    const SfxShell* pObjectBar) const
{
    ShellId nId = snInvalidShellId;

    ActiveObjectBarList::const_iterator aI (::std::find_if (
        maActiveObjectBars.begin(),
        maActiveObjectBars.end(),
        IsShell(pObjectBar)));
    if (aI != maActiveObjectBars.end())
        nId = aI->mnId;

    return nId;
}




ShellId ObjectBarManager::Implementation::GetTopObjectBarId (void) const
{
    ShellId nId;

    if ( ! maActiveObjectBars.empty())
        nId = maActiveObjectBars.back().mnId;
    else
        nId = snInvalidShellId;

    return nId;
}




SfxShell* ObjectBarManager::Implementation::GetTopObjectBar (void) const
{
    SfxShell* pShell;

    if ( ! maActiveObjectBars.empty())
        pShell = maActiveObjectBars.back().mpShell;
    else
        pShell = NULL;

    return pShell;
}




void ObjectBarManager::Implementation::InvalidateAllObjectBars (void) const
{
    ActiveObjectBarList::const_iterator aI;
    for (aI=maActiveObjectBars.begin();
         aI!=maActiveObjectBars.end();
         aI++)
    {
        aI->mpShell->Invalidate();
    }
}




void ObjectBarManager::Implementation::GetLowerShellList (
    ::std::vector<SfxShell*>& rShellList) const
{
    for (ActiveObjectBarList::const_iterator aI (maActiveObjectBars.begin());
         aI!=maActiveObjectBars.end();
         ++aI)
    {
        switch (aI->mePosition)
        {
            case SP_BOTTOM:
            case SP_TOP:
                // Handled in GetUpperShellList.
                break;

            case SP_BELOW_SHELL:
                rShellList.push_back (aI->mpShell);
                break;
        }
    }
}




void ObjectBarManager::Implementation::GetUpperShellList (
    ::std::vector<SfxShell*>& rShellList) const
{
    for (ActiveObjectBarList::const_iterator aI (maActiveObjectBars.begin());
         aI!=maActiveObjectBars.end();
         ++aI)
    {
        switch (aI->mePosition)
        {
            case SP_BOTTOM:
            case SP_TOP:
                rShellList.push_back (aI->mpShell);
                break;

            case SP_BELOW_SHELL:
                // Handled in GetLowerShellList.
                break;
        }
    }
}




void ObjectBarManager::Implementation::SetPreferredStackPosition (
    ShellId nId, 
    StackPosition ePosition)
{
    if (maPreferredStackPositions[nId] != ePosition)
    {
        // Remember the new position.
        maPreferredStackPositions[nId] = ePosition;

        // Move to object bar to its requested position.
        if (mbIsObjectBarSwitchingEnabled)
        {
            ViewShellManager::UpdateLock aLock (mrViewShellManager);
            ActiveObjectBarList::iterator aI (::std::find_if (
                maActiveObjectBars.begin(),
                maActiveObjectBars.end(),
                IsId(nId)));
            if (aI != maActiveObjectBars.end())
            {
                ActiveShellDescriptor aDescriptor (aI->mpShell,nId,ePosition);

                // Remove the object bar.
                maActiveObjectBars.erase (aI);
            
                // Insert it at the requested position.
                switch (ePosition)
                {
                    case SP_TOP:
                        maActiveObjectBars.push_back (aDescriptor);
                        break;

                    case SP_BOTTOM: 
                    case SP_BELOW_SHELL: 
                        maActiveObjectBars.push_front (aDescriptor);
                        break;
                }

                // Tell the view shell manager that the stack above and
                // including the first of the active object bars is
                // invalid.  This can be optimized to be the successor of
                // at the original position of the specified object bar.
                mrViewShellManager.InvalidateShellStack(maActiveObjectBars.back().mpShell);
            }
        }
    }
}




bool ObjectBarManager::Implementation::IsActive (ShellId nId)
{
    return (GetObjectBar(nId) != NULL);
}




void ObjectBarManager::Implementation::SelectionHasChanged (SdrView* pView)
{
    ShellId nObjectBarId;
	switch (pView->GetContext())
	{
		case SDRCONTEXT_GRAPHIC: 
            nObjectBarId = RID_DRAW_GRAF_TOOLBOX; 
            break;

		case SDRCONTEXT_MEDIA:
            nObjectBarId = RID_DRAW_MEDIA_TOOLBOX; 
            break;
		
		case SDRCONTEXT_TEXTEDIT:
            nObjectBarId = RID_DRAW_TEXT_TOOLBOX; 
            break;
		
		case SDRCONTEXT_STANDARD:
            // GetContext() returns ...STANDARD when either a) there is a
            // selection but the selected shape does not belong to any of
            // the above items or b) there is no selected shape.  We have to
            // distinguish between these two cases.
            if (pView->GetMarkedObjectCount() == 0)
                nObjectBarId = mnDefaultObjectBarId;
            else
                nObjectBarId = RID_DRAW_OBJ_TOOLBOX; 
            break;
		
		default:
            nObjectBarId = mnDefaultObjectBarId;
	}
	SwitchObjectBar (nObjectBarId);

	// [shell stack has been cleared by SwitchObjectBar()]
    // activate context-sensitive toolbars: extrusion, fontwork
    if (svx::checkForSelectedCustomShapes( pView, true /* bOnlyExtruded */ ))
        ActivateObjectBar(RID_SVX_EXTRUSION_BAR);
    sal_uInt32 nCheckStatus = 0;
    if (svx::checkForSelectedFontWork( pView, nCheckStatus ))
        ActivateObjectBar(RID_SVX_FONTWORK_BAR);
    // eventually nObjectBarId ought to be as main toolbar, not the latter two

	//switch on additional context-sensitive toolbars
	if( pView->GetContext() == SDRCONTEXT_POINTEDIT )
		ActivateObjectBar(RID_BEZIER_TOOLBOX);
}





Reference< ::com::sun::star::frame::XLayoutManager >
    ObjectBarManager::Implementation::GetLayoutManager (void)
{
    Reference< ::com::sun::star::frame::XLayoutManager > xLayouter;

    Reference<beans::XPropertySet> xFrameProperties (
        mrViewShell.GetViewFrame()->GetFrame()->GetFrameInterface(),
        UNO_QUERY);
    if (xFrameProperties.is())
    {
        try
        {
            Any aValue (xFrameProperties->getPropertyValue(
                OUString::createFromAscii("LayoutManager")));
            aValue >>= xLayouter;
        }
        catch (beans::UnknownPropertyException aException)
        {
        }
    }

    return xLayouter;
}




OUString ObjectBarManager::Implementation::GetToolBarResourceName (
    const OUString& rToolBarName)
{
    OUString sFullName(OUString::createFromAscii("private:resource/toolbar/"));
    sFullName += rToolBarName;
    return sFullName;
}




void ObjectBarManager::Implementation::TraceStack (
    ObjectBarManager::Implementation::ActiveObjectBarList& rList)
{
    ActiveObjectBarList::const_iterator iShell;
    for (iShell=rList.begin(); iShell!=rList.end(); ++iShell)
    {
        OSL_TRACE("    %p %d %d", iShell->mpShell, iShell->mnId, iShell->mePosition);
    }
}


} // end of namespace sd
