/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: SlsGenericRequestQueue.hxx,v $
 *
 *  $Revision: 1.6 $
 *
 *  last change: $Author: rt $ $Date: 2005/09/09 06:10:27 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

#ifndef SD_SLIDESORTER_GENERIC_REQUEST_QUEUE_HXX
#define SD_SLIDESORTER_GENERIC_REQUEST_QUEUE_HXX

#ifndef _OSL_MUTEX_HXX_
#include "osl/mutex.hxx"
#endif

#ifndef INCLUDED_SET
#include <set>
#define INCLUDED_SET
#endif

#ifndef INCLUDED_ALGORITHM
#include <algorithm>
#define INCLUDED_ALGORITHM
#endif

namespace sd { namespace slidesorter { namespace cache {

/** This class extends the actual request data with additional information
    that is used by the priority queues.
*/
template<class RequestData>
class Request 
{
public:
    Request (
        RequestData* pData, sal_Int32 nPriority, sal_Int32 nClass)
        : mpData(pData), mnPriorityInClass(nPriority), mnClass(nClass) 
    {}
    RequestData* mpData;
    sal_Int32 mnPriorityInClass;
    sal_Int32 mnClass;
};




/** Request data is compared arbitrarily by their addresses in memory.  This
    just establishes an order so that the STL containers are happy.  The
    order is not semantically interpreted.
*/
template<class RequestData>
class RequestDataComparator
{
public:
    RequestDataComparator (RequestData&rData):mrData(rData){}
    bool operator() (const Request<RequestData>& rRequest)
    { return &mrData == rRequest.mpData; }

private:
    RequestData& mrData;
};




/** The request comparator is a function object class that is designed to be
    used in an STL set of Request objects.  It sorts requests so that the
    one with the highest priority is at the beginning of the set.
*/
template<class RequestData>
    class RequestComparator 
    {
    public:
        bool operator() (
            const Request<RequestData>& rRequest1, 
            const Request<RequestData>& rRequest2)
        {
            if (rRequest1.mnClass == rRequest2.mnClass)
                return (rRequest1.mnPriorityInClass 
                    > rRequest2.mnPriorityInClass);
            else
                return (rRequest1.mnClass < rRequest2.mnClass);
        }
    };



/** The request queue stores requests that are described by the RequestData
    class in one or more priority queues.  The number of queues is defined
    by the ClassCount template parameter.  This allows to have different
    queues for the generation of previews for visible page objects and for
    currently not visible page objects.  The first are then handled with
    highest priority while the later are only processed when the system is
    otherwise idle.
*/
template<class RequestData, int ClassCount>
class GenericRequestQueue
{
public:
    GenericRequestQueue (void);

    /** Insert a request to the queue that has highest priority in the
        specified request class.
        @param nRequestClass
            This class is designed to manage only a limited and consequtive
            number of different classes.  Typically there are only two: 0
            and 1.
    */
    void InsertFrontRequest (
        RequestData& rRequestData, 
        int nRequestClass);

    /** Add a request to the queue that has lowest priority in the
        specified request class.  When the request is already in the queue
        it is touched only when its request class does not match the
        specified one.
    */
    void AddRequest (
        RequestData& rRequestData, 
        int nRequestClass);

    /** Change the request class of the specified request.
    */
    void ChangeClass (RequestData& rRequestData, int nNewRequestClass);

    /** Move the specified request to a lower request class.  If it is
        already at the lowest class then remove the request from the queue.
        @param nClassOffset
            The class offset, typically +1 or -1, is added to the current
            priority class to get the new class.  The resulting value is
            clipped to lie inside the valid range [0,ClassCount).
    */
    void ChangePriorityClass (RequestData& rRequestData, int nClassOffset);

    /** Remove the specified request from any queue.
        @param rRequestData
            It is OK when the specified request is not a member of any
            queue.
    */
    void RemoveRequest (RequestData& rRequestData);

    /** Get request with highest priority.
    */
    RequestData& GetFront (void);

    // For debugging.
    sal_Int32 GetFrontPriorityClass (void);

    /** Really a synonym for RemoveRequest(GetFront());
    */
    void PopFront (void);

    /** Returns <TRUE/> when there is no element in any of the queues.
    */
    bool IsEmpty (void);

    /** Remove all requests from all queues.  This resets the minimum and
        maximum priorities to their start values.
    */
    void Clear (void);

    /** Return the mutex that guards the access to the priority queues.
    */
    ::osl::Mutex& GetMutex (void);

private:
    ::osl::Mutex maMutex;
    typedef ::std::set<Request<RequestData>, RequestComparator<RequestData> >
        RequestQueueContainer;
    RequestQueueContainer maRequestQueue;
    /** A lower bound of the lowest priority of all elements in the queues.
        The start value is 0.  It is assigned and then decreased every time
        when an element is inserted or marked as the request with lowest
        priority.
    */
    int mnMinimumPriority;
    /** An upper bound of the highest priority of all elements in the queues.
        The start value is 1.  It is assigned and then increased every time
        when an element is inserted or marked as the request with highest
        priority.
    */
    int mnMaximumPriority;
};


//=============================================================================
// Implementation


//=====  GenericRequestQueue  =================================================

template<class RequestData, int ClassCount>
GenericRequestQueue<RequestData,ClassCount>::GenericRequestQueue (void)
    : mnMinimumPriority(0),
      mnMaximumPriority(1)
{
}




template<class RequestData, int ClassCount>
    void GenericRequestQueue<RequestData,ClassCount>::InsertFrontRequest (
        RequestData& rRequestData, 
        int nRequestClass)
{
    ::osl::MutexGuard aGuard (maMutex);

    DBG_ASSERT(nRequestClass<ClassCount, 
        "GenericRequestQeue<>::InsertFrontRequest called with invalid request class");
    OSL_TRACE ("GenericRequestQeue<>::InsertFrontRequest called for page %d and class %d", rRequestData.GetPage()->GetPageNum(), nRequestClass);

    Request<RequestData> aRequest (
        &rRequestData, mnMaximumPriority++, nRequestClass);

    // If the request is already a member of the queue then remove it so
    // that the following insertion will (a) use the given request class and
    // (b) it is inserted with highest priority in that class.
    RemoveRequest (rRequestData);
    
    // Insert the request at the head of the queue where it will be
    // processed with highest priority in its class (until the next request
    // is inserted).
    maRequestQueue.insert (aRequest);
}




template<class RequestData, int ClassCount>
    void GenericRequestQueue<RequestData,ClassCount>::AddRequest (
        RequestData& rRequestData, 
        int nRequestClass)
{
    ::osl::MutexGuard aGuard (maMutex);

    OSL_TRACE ("GenericRequestQueue<>::AddRequest(): added request for page %d with priority class %d",
        rRequestData.GetPage()->GetPageNum(), nRequestClass);
    DBG_ASSERT(nRequestClass<ClassCount, 
        "GenericRequestQeue<>::AddRequest called with invalid request class");

    Request<RequestData> aRequest (
        &rRequestData, mnMinimumPriority--, nRequestClass);

    // If the request is already a member of the queue then remove it so
    // that the following insertion will (a) use the given request class and
    // (b) it is inserted with highest priority in that class.
    RemoveRequest (rRequestData);
    
    // Insert the request at the head of the queue where it will be
    // processed with highest priority in its class (until the next request
    // is inserted).
    maRequestQueue.insert (aRequest);
}




/** Change the request class of the specified request.
*/
template<class RequestData, int ClassCount>
    void GenericRequestQueue<RequestData,ClassCount>::ChangeClass (
        RequestData& rRequestData, int nNewRequestClass)
{
    if (nNewRequestClass>=0 && nNewRequestClass<ClassCount)
    {
        ::osl::MutexGuard aGuard (maMutex);

        typename RequestQueueContainer::iterator aRequestIterator (::std::find (
            maRequestQueue.begin(), maRequestQueue.end(), rRequestData));
        if (aRequestIterator != maRequestQueue.end())
        {
            Request<RequestData> aRequest (*aRequestIterator);
            maRequestQueue.erase (aRequestIterator);
        
            aRequest.mnClass = nNewRequestClass;
            aRequest.mnPriority = mnMaximumPriority++;
            maRequestQueue.insert (aRequestIterator);
        }
    }
}




template<class RequestData, int ClassCount>
    void GenericRequestQueue<RequestData,ClassCount>::RemoveRequest (
        RequestData& rRequest)
{
    ::osl::MutexGuard aGuard (maMutex);

    typename RequestQueueContainer::iterator aRequestIterator;
    while(1)
    {
        aRequestIterator = ::std::find_if (
            maRequestQueue.begin(), 
            maRequestQueue.end(),
            RequestDataComparator<RequestData>(rRequest));
        if (aRequestIterator != maRequestQueue.end())
        {
            if (aRequestIterator->mnPriorityInClass == mnMinimumPriority+1)
                mnMinimumPriority++;
            else if (aRequestIterator->mnPriorityInClass== mnMaximumPriority-1)
                mnMaximumPriority--;
            maRequestQueue.erase (aRequestIterator);
        }
        else 
            break;
    }
}




template<class RequestData, int ClassCount>
    RequestData& GenericRequestQueue<RequestData,ClassCount>::GetFront (void)
{
    ::osl::MutexGuard aGuard (maMutex);
    return *maRequestQueue.begin()->mpData;
}




template<class RequestData, int ClassCount>
    sal_Int32 GenericRequestQueue<RequestData,ClassCount>
    ::GetFrontPriorityClass (void)
{
    ::osl::MutexGuard aGuard (maMutex);
    return maRequestQueue.empty() ? mnMinimumPriority : maRequestQueue.begin()->mnClass;
}




template<class RequestData, int ClassCount>
    void GenericRequestQueue<RequestData,ClassCount>::PopFront (void)
{
    ::osl::MutexGuard aGuard (maMutex);

    maRequestQueue.erase (maRequestQueue.begin());
    // Reset the priority counter if possible.
    if (maRequestQueue.empty())
    {
        mnMinimumPriority = 0;
        mnMaximumPriority = 1;
    }
}




template<class RequestData, int ClassCount>
    void GenericRequestQueue<RequestData,ClassCount>::ChangePriorityClass (
        RequestData& rRequestData,
        int nClassOffset)
{
    ::osl::MutexGuard aGuard (maMutex);

    typename RequestQueueContainer::iterator aRequestIterator (
        ::std::find_if (maRequestQueue.begin(), maRequestQueue.end(), 
            RequestDataComparator<RequestData>(rRequestData)));
    if (aRequestIterator != maRequestQueue.end())
    {
        Request<RequestData> aRequest (*aRequestIterator);
        sal_Int32 nPriorityClass = aRequest.mnClass;
        aRequest.mnClass += nClassOffset;
        if (aRequest.mnClass < 0)
            aRequest.mnClass = 0;
        else if (aRequest.mnClass >= ClassCount)
            aRequest.mnClass = ClassCount-1;
        if (aRequest.mnClass != nPriorityClass)
        {
            maRequestQueue.erase (aRequestIterator);
        
            aRequest.mnPriorityInClass = mnMaximumPriority++;
            maRequestQueue.insert (aRequest);
        }
    }
}




template<class RequestData, int ClassCount>
    bool GenericRequestQueue<RequestData,ClassCount>::IsEmpty (void)
{
    ::osl::MutexGuard aGuard (maMutex);
    return maRequestQueue.empty();
}




template<class RequestData, int ClassCount>
    void GenericRequestQueue<RequestData,ClassCount>::Clear (void)
{
    ::osl::MutexGuard aGuard (maMutex);
    OSL_TRACE ("GenericRequestQueue: Clearing request queue");
    maRequestQueue.clear();
    mnMinimumPriority = 0;
    mnMaximumPriority = 1;
}




template<class RequestData, int ClassCount>
    ::osl::Mutex& GenericRequestQueue<RequestData,ClassCount>::GetMutex (void)
{
    return maMutex;
}

} } } // end of namespace ::sd::slidesorter::cache

#endif
