/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: dx_device.cxx,v $
 *
 *  $Revision: 1.5 $
 *
 *  last change: $Author: rt $ $Date: 2005/09/07 23:27:04 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

#include <canvas/debug.hxx>
#include <canvas/verbosetrace.hxx>

// for labs:
#include <stdlib.h>

#ifndef _RTL_LOGFILE_HXX_
#include <rtl/logfile.hxx>
#endif
#ifndef _RTL_MEMORY_H_
#include <rtl/memory.h>
#endif
#ifndef _RTL_STRING_HXX_
#include <rtl/string.hxx>
#endif
#ifndef _RTL_TEXTENC_H
#include <rtl/textenc.h>
#endif

#ifndef INCLUDED_RTL_MATH_HXX
#include <rtl/math.hxx>
#endif

#ifndef _BGFX_MATRIX_B2DHOMMATRIX_HXX
#include <basegfx/matrix/b2dhommatrix.hxx>
#endif
#ifndef _BGFX_NUMERIC_FTOOLS_HXX
#include <basegfx/numeric/ftools.hxx>
#endif
#ifndef _BGFX_POLYGON_B2DPOLYPOLYGONRASTERCONVERTER_HXX
#include <basegfx/polygon/b2dpolypolygonrasterconverter.hxx>
#endif

#include <canvas/elapsedtime.hxx>

#include <dx_impltools.hxx>
#include <dx_device.hxx>
#include <dx_polypolygonrasterizer.hxx>


/* Implementation of Device class */

namespace dxcanvas
{
    namespace
    {
        IDirectDraw2* createDirectDraw()
        {
            IDirectDraw* pDirectDraw;

            if( FAILED( DirectDrawCreate( NULL, &pDirectDraw, NULL ) ) )
                return NULL;

            IDirectDraw2* pDirectDraw2;

            if( FAILED( pDirectDraw->QueryInterface( IID_IDirectDraw2, (LPVOID*)&pDirectDraw2 ) ) )
                return NULL;

            return pDirectDraw2;
        }

        HRESULT WINAPI EnumTextureFormatsCallback( LPDDSURFACEDESC 	pSurfaceDesc,
                                                   LPVOID			pContext		)
        {
            // dirty cast of given context back to result ModeSelectContext
            DDPIXELFORMAT* pResult = (DDPIXELFORMAT*)pContext;

            if( pResult == NULL || pSurfaceDesc == NULL )
                return DDENUMRET_CANCEL;

            VERBOSE_TRACE( "EnumTextureFormatsCallback: advertised texture format has dwRGBBitCount %d, dwRBitMask %x, "
                           "dwGBitMask %x, dwBBitMask %x and dwRGBAlphaBitMask %x. The format uses %s alpha.",
                           pSurfaceDesc->ddpfPixelFormat.dwRGBBitCount,
                           pSurfaceDesc->ddpfPixelFormat.dwRBitMask,
                           pSurfaceDesc->ddpfPixelFormat.dwGBitMask,
                           pSurfaceDesc->ddpfPixelFormat.dwBBitMask,
                           pSurfaceDesc->ddpfPixelFormat.dwRGBAlphaBitMask,
                           pSurfaceDesc->ddpfPixelFormat.dwFlags & DDPF_ALPHAPREMULT ? "premultiplied" : "non-premultiplied" );

            // Only accept RGB surfaces with alpha channel
            if( (DDPF_ALPHAPIXELS | DDPF_RGB) ==
                (pSurfaceDesc->ddpfPixelFormat.dwFlags & (DDPF_ALPHAPIXELS | DDPF_RGB)) )
            {
                // take widest alpha channel available
                if( pSurfaceDesc->ddpfPixelFormat.dwAlphaBitDepth > pResult->dwAlphaBitDepth )
                {
                    // take new format
                    rtl_copyMemory( pResult, &pSurfaceDesc->ddpfPixelFormat, sizeof(DDPIXELFORMAT) );
                }
                else if( pSurfaceDesc->ddpfPixelFormat.dwAlphaBitDepth == pResult->dwAlphaBitDepth )
                {
                    // tie-breaking: take highest bitcount
                    if( pSurfaceDesc->ddpfPixelFormat.dwRGBBitCount > pResult->dwRGBBitCount )
                    {
                        // take new format
                        rtl_copyMemory( pResult, &pSurfaceDesc->ddpfPixelFormat, sizeof(DDPIXELFORMAT) );
                    }
                }
            }

            return DDENUMRET_OK;
        }
    }

    bool Device::setup3DDevice()
    {
        // create and setup 3D device
        // ==========================
        LPDIRECT3D2	pDirect3D;
        if( FAILED( mpDirectDraw->QueryInterface( IID_IDirect3D2, (LPVOID*)&pDirect3D ) ) )
        {
            // go defunct, and exit
            VERBOSE_TRACE( "Device::setup3DDevice(): QueryInterface() for Direct3D failed" );
            return false;
        }

        mpDirect3D = COMReference< IDirect3D2 >(pDirect3D);

        LPDIRECT3DDEVICE2 pDirect3DDevice;
        // try HW-accelerated device first
        if( FAILED(mpDirect3D->CreateDevice( IID_IDirect3DHALDevice, mpBackBufferSurface.get(), &pDirect3DDevice )) )
        {
            // TODO(P3): explicitely request system memory for the
            // backbuffer. Otherwise, software rendering might get
            // incredibly slow. This also applies to all other
            // buffers, including textures!

            // next, try MMX accelerated software renderer
            if( FAILED(mpDirect3D->CreateDevice( IID_IDirect3DMMXDevice, mpBackBufferSurface.get(), &pDirect3DDevice )) )
            {
                // last fallback: software renderer, plain
                if( FAILED(mpDirect3D->CreateDevice( IID_IDirect3DRGBDevice, mpBackBufferSurface.get(), &pDirect3DDevice )) )
                {
                    // go defunct, and exit
                    VERBOSE_TRACE( "Device::setup3DDevice(): CreateDevice() for Direct3D finally failed" );
                    mpDirect3D.reset();
                    return false;
                }
            }
        }

        mpDirect3DDevice = COMReference< IDirect3DDevice2 >(pDirect3DDevice);

        // select appropriate texture format (_need_ alpha channel here)
        rtl_fillMemory( &maTextureFormat,
                        sizeof(DDPIXELFORMAT), 0 );
        maTextureFormat.dwSize = sizeof(DDPIXELFORMAT);
        if( SUCCEEDED(mpDirect3DDevice->EnumTextureFormats( EnumTextureFormatsCallback, &maTextureFormat )) )
        {
            VERBOSE_TRACE( "Device::setup3DDevice(): chose texture format dwRGBBitCount %d, dwRBitMask %x, "
                           "dwGBitMask %x, dwBBitMask %x and dwRGBAlphaBitMask %x. The texture uses %s alpha.",
                           maTextureFormat.dwRGBBitCount,
                           maTextureFormat.dwRBitMask,
                           maTextureFormat.dwGBitMask,
                           maTextureFormat.dwBBitMask,
                           maTextureFormat.dwRGBAlphaBitMask,
                           maTextureFormat.dwFlags & DDPF_ALPHAPREMULT ? "premultiplied" : "non-premultiplied" );

            // setup the device (with as much as we can possibly do here)
            // ==========================================================

            LPDIRECT3DVIEWPORT2	pViewport;

            if( SUCCEEDED(mpDirect3D->CreateViewport( &pViewport, NULL )) )
            {
                if( SUCCEEDED(mpDirect3DDevice->AddViewport( pViewport )) )
                {
                    // setup viewport (to whole backbuffer)
                    D3DVIEWPORT2 aViewport;

                    aViewport.dwSize = sizeof(D3DVIEWPORT2);
                    aViewport.dwX = 0;
                    aViewport.dwY = 0;
                    aViewport.dwWidth = maSelectedFullscreenMode.selectedDesc.dwWidth;
                    aViewport.dwHeight = maSelectedFullscreenMode.selectedDesc.dwHeight;
                    aViewport.dvClipX = -1.0;
                    aViewport.dvClipY =  -1.0;
                    aViewport.dvClipWidth  = 2.0;
                    aViewport.dvClipHeight = 2.0;
                    aViewport.dvMinZ = 0.0;
                    aViewport.dvMaxZ = 1.0;

                    if( SUCCEEDED(pViewport->SetViewport2( &aViewport )) )
                    {
                        if( SUCCEEDED(mpDirect3DDevice->SetCurrentViewport( pViewport )) )
                        {
                            // Viewport was handed over to 3DDevice, thus we can release now
                            pViewport->Release();

                            // currently, no need for any
                            // matrix or light source
                            // setup, since we only render
                            // transformed&lighted
                            // vertices

                            // done; successfully
                            return true;
                        }
                        else
                        {
                            VERBOSE_TRACE( "Device::setup3DDevice(): SetCurrentViewport failed" );
                        }
                    }
                    else
                    {
                        VERBOSE_TRACE( "Device::setup3DDevice(): SetViewport2 failed" );
                    }
                }
                else
                {
                    VERBOSE_TRACE( "Device::setup3DDevice(): AddViewport failed" );
                }

                pViewport->Release();
            }
            else
            {
                VERBOSE_TRACE( "Device::setup3DDevice(): CreateViewport failed" );
            }
        }
        else
        {
            VERBOSE_TRACE( "Device::setup3DDevice(): EnumTextureFormats failed" );
        }

        // go defunct, and exit
        mpDirect3DDevice.reset();
        mpDirect3D.reset();

        return false;
    }

    Device::Device( HWND renderHwnd ) :
        mhWnd( renderHwnd ),
        maSelectedFullscreenMode(),
        maTextureFormat(),
        mpDirectDraw( createDirectDraw() ),
        mpPrimarySurface(),
        mpBackBufferSurface(),
        mpDirect3D(),
        mpDirect3DDevice(),
        maLastUpdate(),
        mpCachedPrimarySurface(),
        mbPageFlipping( false )
    {
        if( FAILED( mpDirectDraw->SetCooperativeLevel( renderHwnd,
                                                       DDSCL_NORMAL ) ) )
        {
            // go defunct, and exit
            VERBOSE_TRACE( "Device::Device(): SetCooperativeLevel failed" );
            mpDirectDraw.reset();
            return;
        }

        // setup query struct
        rtl_fillMemory( &maSelectedFullscreenMode.selectedDesc,
                        sizeof(DDSURFACEDESC), 0 );
        maSelectedFullscreenMode.selectedDesc.dwSize = sizeof(DDSURFACEDESC);

        // read current display mode, e.g. for screen dimension
        if( FAILED( mpDirectDraw->GetDisplayMode( &maSelectedFullscreenMode.selectedDesc )) )
        {
            // go defunct, and exit
            VERBOSE_TRACE( "Device::Device(): GetDisplayMode failed" );
            mpDirectDraw.reset();
            return;
        }

        // create primary surface reference
        DDSURFACEDESC 		aSurfaceDesc;
        IDirectDrawSurface* pPrimarySurface;

        rtl_fillMemory( &aSurfaceDesc,
                        sizeof(DDSURFACEDESC), 0 );
        aSurfaceDesc.dwSize = sizeof(aSurfaceDesc);
        aSurfaceDesc.dwFlags = DDSD_CAPS;
        aSurfaceDesc.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE | DDSCAPS_3DDEVICE;

        if( FAILED(mpDirectDraw->CreateSurface(&aSurfaceDesc, &pPrimarySurface, NULL)) )
        {
            // go defunct, and exit
            VERBOSE_TRACE( "Device::Device(): CreateSurface failed" );
            mpDirectDraw.reset();
            return;
        }

        mpPrimarySurface = COMReference< IDirectDrawSurface >(pPrimarySurface);

        // create a Clipper and associate it with the primary surface
        // and the render window
        LPDIRECTDRAWCLIPPER pClipper;
        if( FAILED(mpDirectDraw->CreateClipper( 0, &pClipper, NULL )) )
        {
            // go defunct, and exit
            VERBOSE_TRACE( "Device::Device(): CreateClipper failed" );
            mpPrimarySurface.reset();
            mpDirectDraw.reset();
            return;
        }
        if( FAILED(pClipper->SetHWnd(0, renderHwnd)) )
        {
            // go defunct, and exit
            VERBOSE_TRACE( "Device::Device(): Clipper->SetHWnd failed" );
            pClipper->Release();
            mpPrimarySurface.reset();
            mpDirectDraw.reset();
            return;
        }
        if( FAILED(mpPrimarySurface->SetClipper( pClipper )) )
        {
            // go defunct, and exit
            VERBOSE_TRACE( "Device::Device(): SetClipper failed" );
            pClipper->Release();
            mpPrimarySurface.reset();
            mpDirectDraw.reset();
            return;
        }

        // clipper is now owned by mpPrimarySurface, release our reference
        pClipper->Release();

        // TODO(F3): Check whether palette needs any setup here

        // get us a backbuffer for simulated flipping
        IDirectDrawSurface* pSurface;

        // TODO(P2): Strictly speaking, we don't need a full screen worth of 
        // backbuffer here. We could also scale dynamically with the current
        // window size, but this will make it necessary to temporarily have two
        // buffers while copying from the old to the new one. YMMV.
        const ::basegfx::B2ISize aSize( getSize() );

        rtl_fillMemory( &aSurfaceDesc,
                        sizeof(DDSURFACEDESC), 0 );
        aSurfaceDesc.dwSize = sizeof(DDSURFACEDESC);
        aSurfaceDesc.dwFlags = DDSD_CAPS | DDSD_HEIGHT | DDSD_WIDTH;
        aSurfaceDesc.dwHeight= aSize.getY();
        aSurfaceDesc.dwWidth = aSize.getX();

        aSurfaceDesc.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN | DDSCAPS_3DDEVICE | DDSCAPS_VIDEOMEMORY | DDSCAPS_LOCALVIDMEM;

        HRESULT nRes = mpDirectDraw->CreateSurface(&aSurfaceDesc, &pSurface, NULL);

        if( FAILED( nRes ) )
        {
            if( nRes == DDERR_OUTOFVIDEOMEMORY )
            {
                // local vid mem failed. Maybe AGP mem works?
                aSurfaceDesc.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN | DDSCAPS_3DDEVICE | DDSCAPS_VIDEOMEMORY | DDSCAPS_NONLOCALVIDMEM;
                if( FAILED(mpDirectDraw->CreateSurface(&aSurfaceDesc, &pSurface, NULL)) )
                {
                    // no chance, go defunct, and exit
                    VERBOSE_TRACE( "Device::Device(): CreateSurface for backbuffer failed" );
                    mpPrimarySurface.reset();
                    mpDirectDraw.reset();
                    return;
                }

                VERBOSE_TRACE( "Device::Device(): CreateSurface for backbuffer reverted to non-local video mem" );
            }
            else
            {
                // no chance, go defunct, and exit
                VERBOSE_TRACE( "Device::Device(): CreateSurface for backbuffer failed" );
                mpPrimarySurface.reset();
                mpDirectDraw.reset();
                return;
            }
        }

        VERBOSE_TRACE( "Device::Device(): created backbuffer of size %d times %d pixel",
                   aSurfaceDesc.dwWidth,
                   aSurfaceDesc.dwHeight );

        mpBackBufferSurface = COMReference< IDirectDrawSurface >(pSurface);

        if( !setup3DDevice() )
        {
            // go defunct, and exit
            VERBOSE_TRACE( "Device::Device(): setup3DDevice failed" );
            mpBackBufferSurface.reset();
            mpPrimarySurface.reset();
            mpDirectDraw.reset();
            return;
        }
    }

    namespace
    {

        HRESULT WINAPI EnumDisplayModesCallback( LPDDSURFACEDESC 	pSurfaceDesc,
                                                 LPVOID				pContext		)
        {
            // dirty cast of given context back to result ModeSelectContext
            Device::ModeSelectContext* pResult = (Device::ModeSelectContext*)pContext;

            if( pResult == NULL || pSurfaceDesc == NULL )
                return DDENUMRET_CANCEL;

            // mode selection heuristic: closest size fit, more colors
            // override less colors for a tie, higher refresh rate
            // override lesser for a tie.
            const int dx_last( labs( pResult->selectedDesc.dwWidth - pResult->requestedSize.getX()) );
            const int dy_last( labs( pResult->selectedDesc.dwHeight - pResult->requestedSize.getY()) );

            const int dx( labs( pSurfaceDesc->dwWidth - pResult->requestedSize.getX() ) );
            const int dy( labs( pSurfaceDesc->dwHeight - pResult->requestedSize.getY() ) );

            if( dx_last + dy_last > dx + dy )
            {
                // take new mode
                rtl_copyMemory( &pResult->selectedDesc, pSurfaceDesc, sizeof(DDSURFACEDESC) );
            }
            else if( dx_last + dy_last == dx + dy )
            {
                // tie breaking: color depth
                if( pSurfaceDesc->ddpfPixelFormat.dwRGBBitCount > pResult->selectedDesc.ddpfPixelFormat.dwRGBBitCount )
                {
                    // take new mode
                    rtl_copyMemory( &pResult->selectedDesc, pSurfaceDesc, sizeof(DDSURFACEDESC) );
                }
                else if( pSurfaceDesc->ddpfPixelFormat.dwRGBBitCount == pResult->selectedDesc.ddpfPixelFormat.dwRGBBitCount )
                {
                    // tie breaking: refresh rate
                    if( pSurfaceDesc->dwRefreshRate > pResult->selectedDesc.dwRefreshRate &&
                        pSurfaceDesc->dwRefreshRate != 0 &&
                        pResult->selectedDesc.dwRefreshRate <= 80 ) // limit refresh rate to 80Hz, no need to kill a
                        											// monitor/eat too much VRAM bandwidth
                    {
                        // take new mode
                        rtl_copyMemory( &pResult->selectedDesc, pSurfaceDesc, sizeof(DDSURFACEDESC) );
                    }
                }
            }

            return DDENUMRET_OK;
        }
    }

    Device::Device( HWND applHwnd, const ::basegfx::B2ISize& rFullscreenSize ) :
        mhWnd( applHwnd ),
        maSelectedFullscreenMode(),
        maTextureFormat(),
        mpDirectDraw( createDirectDraw() ),
        mpPrimarySurface(),
        mpBackBufferSurface(),
        mpDirect3D(),
        mpDirect3DDevice(),
        maLastUpdate(),
        mpCachedPrimarySurface(),
        mbPageFlipping( true )
    {
        if( FAILED( mpDirectDraw->SetCooperativeLevel( applHwnd,
                                                       DDSCL_EXCLUSIVE|DDSCL_FULLSCREEN ) ) )
        {
            // go defunct, and exit
            VERBOSE_TRACE( "Device::Device(): SetCooperativeLevel failed" );
            mpDirectDraw.reset();
            return;
        }

        // setup query struct
        rtl_fillMemory( &maSelectedFullscreenMode.selectedDesc,
                        sizeof(DDSURFACEDESC), 0 );
        maSelectedFullscreenMode.selectedDesc.dwSize = sizeof(DDSURFACEDESC);

        maSelectedFullscreenMode.requestedSize = rFullscreenSize;

        if( rFullscreenSize.getX() == 0.0 ||
            rFullscreenSize.getY() == 0.0 )
        {
            // keep current display mode, read in data
            if( FAILED( mpDirectDraw->GetDisplayMode( &maSelectedFullscreenMode.selectedDesc )) )
            {
                // go defunct, and exit
                VERBOSE_TRACE( "Device::Device(): GetDisplayMode failed" );
                mpDirectDraw.reset();
                return;
            }
        }
        else
        {
            // select display mode from requested fullscreen size
            if( FAILED( mpDirectDraw->EnumDisplayModes( 0, NULL, &maSelectedFullscreenMode, EnumDisplayModesCallback ) ) )
            {
                // go defunct, and exit
                VERBOSE_TRACE( "Device::Device(): EnumDisplayModes failed" );
                mpDirectDraw.reset();
                return;
            }

            // now try to set the selected mode
            setMode();
        }

        // create primary surface reference
        DDSURFACEDESC 		aSurfaceDesc;
        IDirectDrawSurface* pPrimarySurface;

        rtl_fillMemory( &aSurfaceDesc,
                        sizeof(DDSURFACEDESC), 0 );
        aSurfaceDesc.dwSize = sizeof(aSurfaceDesc);
        aSurfaceDesc.dwFlags = DDSD_CAPS | DDSD_BACKBUFFERCOUNT;
        aSurfaceDesc.ddsCaps.dwCaps =
            DDSCAPS_PRIMARYSURFACE |
            DDSCAPS_3DDEVICE |
            DDSCAPS_FLIP |
            DDSCAPS_COMPLEX |
            DDSCAPS_VIDEOMEMORY;
        aSurfaceDesc.dwBackBufferCount = 1;

        if( FAILED(mpDirectDraw->CreateSurface(&aSurfaceDesc, &pPrimarySurface, NULL)) )
        {
            // go defunct, and exit
            VERBOSE_TRACE( "Device::Device(): CreateSurface failed" );
            mpDirectDraw.reset();
            return;
        }

        mpPrimarySurface = COMReference< IDirectDrawSurface >(pPrimarySurface);

        // retrieve pointer to backbuffer
        IDirectDrawSurface* pBackBufferSurface;
        aSurfaceDesc.ddsCaps.dwCaps = DDSCAPS_BACKBUFFER | DDSCAPS_3DDEVICE; // reuse part of the structure
        if( FAILED(mpPrimarySurface->GetAttachedSurface(&aSurfaceDesc.ddsCaps, &pBackBufferSurface)) )
        {
            // go defunct, and exit
            VERBOSE_TRACE( "Device::Device(): GetAttachedSurface failed" );
            mpPrimarySurface.reset();
            mpDirectDraw.reset();
            return;
        }

        mpBackBufferSurface = COMReference< IDirectDrawSurface >(pBackBufferSurface);

        if( !setup3DDevice() )
        {
            // go defunct, and exit
            VERBOSE_TRACE( "Device::Device(): setup3DDevice failed" );
            mpBackBufferSurface.reset();
            mpPrimarySurface.reset();
            mpDirectDraw.reset();
            return;
        }
    }

    Device::~Device()
    {
    }

    ::basegfx::B2ISize Device::getSize() const
    {
        if( mpDirectDraw.is() )
            return ::basegfx::B2ISize( maSelectedFullscreenMode.selectedDesc.dwWidth,
                                       maSelectedFullscreenMode.selectedDesc.dwHeight );
        else
            return ::basegfx::B2ISize();
    }

    bool Device::flip()
    {
        const ::basegfx::B2ISize aScreenSize( getSize() );
        
        return flip( ::basegfx::B2IRectangle(0,0,aScreenSize.getX(),aScreenSize.getY() ),
                     ::basegfx::B2IPoint() );
    }

    bool Device::flip( const ::basegfx::B2IRectangle& rUpdateArea,
                       const ::basegfx::B2IPoint&	  rOffset )
    {
        if( mpDirectDraw.is() &&
            mpPrimarySurface.is() &&
            mpBackBufferSurface.is() )
        {
            // ignore area and offset for page flipping device
            if( mbPageFlipping )
            {
#if defined(VERBOSE) && defined(DBG_UTIL)
                renderFPSCounter();
                renderMemAvailable();
#endif
                
                // use true page flipping
                if( SUCCEEDED(mpPrimarySurface->Flip( NULL, DDFLIP_WAIT )) )
                    return true;
            }
            else
            {
                const ::basegfx::B2ISize aDestSize( getSize() );

                // determine actual window position
                POINT aPoint = { 0, 0 };
                ClientToScreen( mhWnd, &aPoint );

                // add offset to window position
                aPoint.x += rOffset.getX();
                aPoint.y += rOffset.getY();

                // clamp source rectangle to permissible surface area
                ::basegfx::B2IRectangle aSourceRect( ::std::max<sal_Int32>( rUpdateArea.getMinX(), 0 ),
                                                     ::std::max<sal_Int32>( rUpdateArea.getMinY(), 0 ),
                                                     ::std::min( rUpdateArea.getMaxX(), aDestSize.getX() ),
                                                     ::std::min( rUpdateArea.getMaxY(), aDestSize.getY() ) );
                
                // clamp dest rectangle to permissible surface area (after moving
                // it to the actual window output position)
                ::basegfx::B2IRectangle aDestRect( ::std::max<sal_Int32>( aSourceRect.getMinX() + aPoint.x, 0 ),
                                                   ::std::max<sal_Int32>( aSourceRect.getMinY() + aPoint.y, 0 ),
                                                   ::std::min( aSourceRect.getMaxX() + aPoint.x, aDestSize.getX() ),
                                                   ::std::min( aSourceRect.getMaxY() + aPoint.y, aDestSize.getX() ) );
                
                // now, dest rect might be smaller than source rect (which would induce a scaling, 
                // which in turn might trigger HEL blts instead of hw blits). Correct source rect, by
                // moving back the rectangle by the window rect.
                aSourceRect = ::basegfx::B2IRectangle( ::std::max<sal_Int32>( aDestRect.getMinX() - aPoint.x, 0 ),
                                                       ::std::max<sal_Int32>( aDestRect.getMinY() - aPoint.y, 0 ),
                                                       ::std::min( aDestRect.getMaxX() - aPoint.x, aDestSize.getX() ),
                                                       ::std::min( aDestRect.getMaxY() - aPoint.y, aDestSize.getX() ) );
                
                RECT aWinSourceRect =
                    {
                        aSourceRect.getMinX(),
                        aSourceRect.getMinY(),
                        aSourceRect.getMaxX(),
                        aSourceRect.getMaxY()
                    };
                RECT aWinDestRect =
                    {
                        aDestRect.getMinX(),
                        aDestRect.getMinY(),
                        aDestRect.getMaxX(),
                        aDestRect.getMaxY()
                    };

                while( true )
                {
                    // wait for start of the VSYNC, and hope the blit is
                    // fast enough to commence before t_vblank + t_screenrefresh
                    if( FAILED(mpDirectDraw->WaitForVerticalBlank(DDWAITVB_BLOCKBEGIN,
                                                                  NULL)) )
                        return false;

                    // make sure the blit will start _immediately_
                    // after the Blt call. If this is not warranted,
                    // wait for the next vsync.
                    if( SUCCEEDED(mpPrimarySurface->GetBltStatus(DDGBS_CANBLT)) &&
                        SUCCEEDED(mpBackBufferSurface->GetBltStatus(DDGBS_CANBLT)) )
                    {
                        // emulate page flipping by blitting the backbuffer to
                        // screen
                        if( SUCCEEDED(mpPrimarySurface->Blt( &aWinDestRect,
                                                             mpBackBufferSurface.get(),
                                                             &aWinSourceRect,
                                                             DDBLT_WAIT,
                                                             NULL )) )
                        {
#if defined(VERBOSE) && defined(DBG_UTIL)
                            renderFPSCounter();
                            renderMemAvailable();
#endif

                            return true;
                        }
                        else
                        {
                            return false;
                        }
                    }

                    VERBOSE_TRACE("Device::flip(): Waiting for next vblank");
                }
            }
        }

        return false;
    }

    SurfaceSharedPtr Device::getPrimarySurface() const
    {
        if( !mpDirectDraw.is() ||
            !mpPrimarySurface.is() ||
            !mpBackBufferSurface.is() )
        {
            return SurfaceSharedPtr();
        }

        if( mpCachedPrimarySurface.get() == NULL )
            mpCachedPrimarySurface = PlainSurfaceSharedPtr(
                new PlainSurface(mpBackBufferSurface,
                                 getSize()) );
        
        return mpCachedPrimarySurface;
    }

    bool Device::render( const PlainSurfaceSharedPtr& 	rSurface,
                         const ::basegfx::B2IPoint& 	rOutPos )
    {
        ENSURE_AND_THROW(rSurface.get() != NULL && rSurface->mpSurface.is(), "Device::render(): invalid surface");
        ENSURE_AND_THROW(mpBackBufferSurface.is(), "Device::render(): invalid target surface");

        const ::basegfx::B2ISize aDestSize( getSize() );

        if( rSurface.get() == NULL || !rSurface->mpSurface.is() || !mpBackBufferSurface.is() ||
            rOutPos.getX() >= aDestSize.getX() || rOutPos.getY() >= aDestSize.getY() )
        {
            return false;
        }

        // blit surface to backbuffer
        RECT aOutRect =
            {
                rOutPos.getX(),
                rOutPos.getY(),
                ::std::min( rOutPos.getX() + rSurface->maSize.getX(),
                            aDestSize.getX() ),
                ::std::min( rOutPos.getY() + rSurface->maSize.getY(),
                            aDestSize.getY() )
            };

        // need to account for negative outpos?
        if( rOutPos.getX() < 0 || rOutPos.getY() < 0 )
        {
            const LONG x( rOutPos.getX() < 0 ? -rOutPos.getX() : 0 );
            const LONG y( rOutPos.getY() < 0 ? -rOutPos.getY() : 0 );

            // take only the visible part of the source rect
            RECT aSourceRect =
                {
                    x,
                    y,
                    rSurface->maSize.getX() - x,
                    rSurface->maSize.getY() - y
                };

            if( FAILED(mpBackBufferSurface->Blt( &aOutRect,
                                                 rSurface->mpSurface.get(),
                                                 &aSourceRect,
                                                 DDBLT_WAIT,
                                                 NULL )) )
            {
                return false;
            }
        }
        else
        {
            // can blit complete source surface
            if( FAILED(mpBackBufferSurface->Blt( &aOutRect,
                                                 rSurface->mpSurface.get(),
                                                 NULL,
                                                 DDBLT_WAIT,
                                                 NULL )) )
            {
                return false;
            }
        }

        return true;
    }

    bool Device::render( const PlainSurfaceSharedPtr& 	rSurface,
                         const ::basegfx::B2IPoint& 	rOutPos,
                         const ::basegfx::B2IRectangle& rSourceRect )
    {
        ENSURE_AND_THROW(rSurface.get() != NULL && rSurface->mpSurface.is(), "Device::render(): invalid surface");
        ENSURE_AND_THROW(mpBackBufferSurface.is(), "Device::render(): invalid target surface");

        const ::basegfx::B2ISize aDestSize( getSize() );

        if( rSurface.get() == NULL || !rSurface->mpSurface.is() || !mpBackBufferSurface.is() ||
            rOutPos.getX() >= aDestSize.getX() || rOutPos.getY() >= aDestSize.getY() )
        {
            return false;
        }

        ::basegfx::B2IPoint aOutPos( rOutPos );
        ::basegfx::B2IRectangle aSourceRect( rSourceRect.getMinX(),
                                             rSourceRect.getMinY(),
                                             // clamp width to destination width
                                             ::std::min(rSourceRect.getMaxX(), aDestSize.getX()),
                                             // clamp height to destination height
                                             ::std::min(rSourceRect.getMaxY(), aDestSize.getY()) );

        // need to account for negative outpos?
        if( aOutPos.getX() < 0 || aOutPos.getY() < 0 )
        {
            // yes. clamp outpos to zero, and appropriately reduce the
            // source rectangle
            const sal_Int32 x( rOutPos.getX() < 0 ? 0 : rOutPos.getX() );
            const sal_Int32 y( rOutPos.getY() < 0 ? 0 : rOutPos.getY() );

            aOutPos = ::basegfx::B2IPoint( x, y );
            aSourceRect = ::basegfx::B2IRectangle( aSourceRect.getMinX() + x - rOutPos.getX(),
                                                   aSourceRect.getMinY() + y - rOutPos.getY(),
                                                   aSourceRect.getMaxX(),
                                                   aSourceRect.getMaxY() );
        }

        // blit surface to backbuffer
        RECT aOutRect =
            {
                aOutPos.getX(),
                aOutPos.getY(),
                static_cast<LONG>( ::std::min<sal_Int64>( aOutPos.getX() + aSourceRect.getWidth(),
                                                          aDestSize.getX() ) ),
                static_cast<LONG>( ::std::min<sal_Int64>( aOutPos.getY() + aSourceRect.getHeight(),
                                                          aDestSize.getY() ) )
            };

        RECT aSrcRect( tools::gdiRectFromB2IRect(aSourceRect) );

        if( FAILED(mpBackBufferSurface->Blt( &aOutRect,
                                             rSurface->mpSurface.get(),
                                             &aSrcRect,
                                             DDBLT_WAIT,
                                             NULL )) )
        {
            return false;
        }

        return true;
    }

    bool Device::render( const TextureSharedPtr& 		rTexture,
                         const ::basegfx::B2DPoint& 	rOutPos,
                         double 						fAlpha )
    {
        ENSURE_AND_THROW(mpBackBufferSurface.is(), 
                         "Device::render(): invalid target surface");
        ENSURE_AND_THROW(rTexture.get() != NULL && rTexture->mpSurface.is() && rTexture->mpTexture.is(), 
                         "Device::render(): invalid texture");

        if( rTexture.get() == NULL || !rTexture->mpSurface.is() || !rTexture->mpTexture.is() || !mpBackBufferSurface.is() )
            return false;

        D3DTEXTUREHANDLE aTextureHandle;
        if( FAILED(rTexture->mpTexture->GetHandle( mpDirect3DDevice.get(),
                                                   &aTextureHandle )) )
        {
            return false;
        }

        // TODO(P2): Better move Begin/EndScene out of this method, such
        // that we can render all sprites in batch mode, within a
        // single scene.
        if( FAILED(mpDirect3DDevice->BeginScene()) )
            return false;

		// enable texture alpha blending
        if( FAILED(mpDirect3DDevice->SetRenderState(D3DRENDERSTATE_ALPHABLENDENABLE,
                                                    TRUE)) )
            return false;

		// enable texture alpha modulation, for honoring fAlpha
        if( FAILED(mpDirect3DDevice->SetRenderState(D3DRENDERSTATE_TEXTUREMAPBLEND,
                                                    D3DTBLEND_MODULATEALPHA)) )
            return false;

		// enable texture magnification filtering (don't care if this
		// fails, it's just visually more pleasant)
        mpDirect3DDevice->SetRenderState(D3DRENDERSTATE_TEXTUREMAG,
                                         D3DFILTER_LINEAR);

		// enable texture minification filtering (don't care if this
		// fails, it's just visually more pleasant)
        mpDirect3DDevice->SetRenderState(D3DRENDERSTATE_TEXTUREMIN,
                                         D3DFILTER_LINEAR);

        // enable subpixel texture output (don't care if this
        // fails, it's just visually more pleasant)
        mpDirect3DDevice->SetRenderState(D3DRENDERSTATE_SUBPIXEL,
                                         TRUE);

        // normal combination of object...
        if( FAILED(mpDirect3DDevice->SetRenderState(D3DRENDERSTATE_SRCBLEND,
                                                    D3DBLEND_SRCALPHA)) )
            return false;

        // ..and background color
        if( FAILED(mpDirect3DDevice->SetRenderState(D3DRENDERSTATE_DESTBLEND,
                                                    D3DBLEND_INVSRCALPHA)) )
            return false;

        // disable backface culling; this enables us to mirror sprites
        // by simply reverting the triangles, which, with enabled
        // culling, would be invisible otherwise
        if( FAILED(mpDirect3DDevice->SetRenderState(D3DRENDERSTATE_CULLMODE,
                                                    D3DCULL_NONE)) )
            return false;

        // select texture for next primitive
        if( FAILED(mpDirect3DDevice->SetRenderState(D3DRENDERSTATE_TEXTUREHANDLE,
                                                    aTextureHandle)) )
            return false;

        // calc range of u and v, to include only the wanted texture
        // data
        const ::basegfx::B1DRange aURange( rTexture->calcURange() );
        const ::basegfx::B1DRange aVRange( rTexture->calcVRange() );

        // log output pos in device pixel
        VERBOSE_TRACE( "Device::render(): output pos is (%f, %f)", 
                       rOutPos.getX(),
                       rOutPos.getY() );

        const ::basegfx::B2DRectangle aOutRect( rOutPos.getX(),
                                                rOutPos.getY(),
                                                rOutPos.getX() + rTexture->maTextureScreenSize.getX(),
                                                rOutPos.getY() + rTexture->maTextureScreenSize.getY() );

        // Setup a readily positioned triangle strip, to render a texture mapped square
        D3DTLVERTEX vertices[4];
        vertices[0] = D3DTLVERTEX(
            D3DVECTOR(static_cast<D3DVALUE>(aOutRect.getMinX()),
                      static_cast<D3DVALUE>(aOutRect.getMaxY()), 0),
            1,
            D3DRGBA(1,1,1,fAlpha),
            D3DRGBA(0,0,0,0),
            static_cast<float>(aURange.getMinimum()),
            static_cast<float>(aVRange.getMaximum()));
        vertices[1] = D3DTLVERTEX(
            D3DVECTOR(static_cast<D3DVALUE>(aOutRect.getMinX()), 
                      static_cast<D3DVALUE>(aOutRect.getMinY()), 0), 
            1, 
            D3DRGBA(1,1,1,fAlpha),
            D3DRGBA(0,0,0,0), 
            static_cast<float>(aURange.getMinimum()), 
            static_cast<float>(aVRange.getMinimum()));
        vertices[2] = D3DTLVERTEX(
            D3DVECTOR(static_cast<D3DVALUE>(aOutRect.getMaxX()), 
                      static_cast<D3DVALUE>(aOutRect.getMaxY()), 0), 
            1, 
            D3DRGBA(1,1,1,fAlpha),
            D3DRGBA(0,0,0,0), 
            static_cast<float>(aURange.getMaximum()), 
            static_cast<float>(aVRange.getMaximum()));
        vertices[3] = D3DTLVERTEX(
            D3DVECTOR(static_cast<D3DVALUE>(aOutRect.getMaxX()), 
                      static_cast<D3DVALUE>(aOutRect.getMinY()), 0), 
            1, 
            D3DRGBA(1,1,1,fAlpha),
            D3DRGBA(0,0,0,0), 
            static_cast<float>(aURange.getMaximum()), 
            static_cast<float>(aVRange.getMinimum()));

        if( FAILED(mpDirect3DDevice->DrawPrimitive(D3DPT_TRIANGLESTRIP, 
                                                   D3DVT_TLVERTEX,
                                                   (LPVOID)vertices,
                                                   4,
                                                   D3DDP_WAIT)) )
        {
            mpDirect3DDevice->EndScene();
            return false;
        }

        if( FAILED(mpDirect3DDevice->EndScene()) )
            return false;
        else
            return true;
    }

    bool Device::copyBits( const TextureSharedPtr&		rTexture,
                           const ::basegfx::B2ISize&	rSize,
                           sal_uInt8*					pSrc )
    {
        RTL_LOGFILE_CONTEXT( aLog, "::dxcanvas::Device::copyBits()" );

        if( !mpDirectDraw.is() ||
            !mpPrimarySurface.is() ||
            !mpBackBufferSurface.is() )
        {
            return false;
        }

        DDSURFACEDESC aSurfaceDesc;
        
        rtl_fillMemory( &aSurfaceDesc, 
                        sizeof(DDSURFACEDESC), 0 );
        aSurfaceDesc.dwSize = sizeof(DDSURFACEDESC);

        if( SUCCEEDED( rTexture->mpSurface->Lock( NULL, 
                                                  &aSurfaceDesc, 
                                                  DDLOCK_NOSYSLOCK|DDLOCK_SURFACEMEMORYPTR|DDLOCK_WAIT|DDLOCK_WRITEONLY,
                                                  NULL )) )
        {
            // TODO(F3): Perform format conversions here!
            const sal_Int32 nSrcWidth( rSize.getX() );
            const sal_Int32 nSrcHeight( rSize.getY() );
            const sal_Int32 nDstWidth( (sal_Int32)rTexture->maTextureRect.getWidth() );
            const sal_Int32 nDstHeight( (sal_Int32)rTexture->maTextureRect.getHeight() );

            const sal_Int32 nSrcScanlineSize( 4*nSrcWidth );
            const sal_Int32 nDstScanlineSize( 4*nDstWidth );
            sal_uInt8* pDest;

            if( nDstWidth == nSrcWidth &&
                nDstHeight == nSrcHeight )
            {
                pDest = (sal_uInt8*)aSurfaceDesc.lpSurface;

                // No scaling necessary. copy bits, scanline by scanline for now
                int y;
                for( y=0; y<nSrcHeight; ++y )
                {
                    rtl_copyMemory(pDest, pSrc, nSrcScanlineSize );
                    pDest += aSurfaceDesc.lPitch;
                    pSrc += nSrcScanlineSize;
                }
            }
            else
            {
                // Need downsampling.

                // TODO(P3): use recursive filter here, at least for huge shrinks (or 
                // a somewhat more optimal box filter implementation)

                const int nXFilterSize( (nSrcWidth + nDstWidth - 1) / nDstWidth );
                const int nYFilterSize( (nSrcHeight + nDstHeight - 1) / nDstHeight);

                int x, y, i;
                unsigned int r, g, b, a;
                sal_uInt8* pScanline;
                sal_uInt8* pSrcPixel;
                sal_uInt8* pDstPixel;

                // First filter horizontally
                sal_uInt8* pTempBuf = (sal_uInt8*)rtl_allocateMemory( ::std::max(nDstScanlineSize, 4*nSrcHeight) );

                pScanline = pSrc;
                for( y=0; y<nSrcHeight; ++y )
                {
                    pDstPixel = pTempBuf;
                    for( x=0; x<nDstWidth; ++x )
                    {
                        pSrcPixel = pScanline + x*nSrcWidth/nDstWidth * 4;

                        // box-filter source pixels into one destination pixel
                        r = g = b = a = 0;
                        for( i=0; i<nXFilterSize; ++i )
                        {
                            b += *pSrcPixel++;
                            g += *pSrcPixel++;
                            r += *pSrcPixel++;
                            a += *pSrcPixel++;   
                        }

                        *pDstPixel++ = b/nXFilterSize;
                        *pDstPixel++ = g/nXFilterSize;
                        *pDstPixel++ = r/nXFilterSize;
                        *pDstPixel++ = a/nXFilterSize;
                    }

                    rtl_copyMemory( pScanline, pTempBuf, nDstScanlineSize );

                    pScanline += nSrcScanlineSize;
                }

                // Now filter vertically
                const int nSrcYRowSkip( nSrcScanlineSize - 4 );
                const sal_uInt32* pSrcWord;
                sal_uInt32* pDstWord;
                const int nDstWordRowSkip( aSurfaceDesc.lPitch >> 2 );
                int nColumnOffset;
                for( x=0; x<nDstWidth; ++x )
                {
                    pDstPixel = pTempBuf;
                    nColumnOffset = 4*x;
                    for( y=0; y<nDstHeight; ++y )
                    {
                        pSrcPixel = pSrc + y*nSrcHeight/nDstHeight * nSrcScanlineSize + nColumnOffset;

                        r = g = b = a = 0;
                        for( i=0; i<nYFilterSize; ++i )
                        {
                            b += *pSrcPixel++;
                            g += *pSrcPixel++;
                            r += *pSrcPixel++;
                            a += *pSrcPixel++;
                            
                            pSrcPixel += nSrcYRowSkip;
                        }

                        *pDstPixel++ = b/nYFilterSize;
                        *pDstPixel++ = g/nYFilterSize;
                        *pDstPixel++ = r/nYFilterSize;
                        *pDstPixel++ = a/nYFilterSize;
                    }

                    // copy from TempBuf back to current column
                    pSrcWord = (sal_uInt32*)pTempBuf;
                    pDstWord = (sal_uInt32*)((sal_uInt8*)aSurfaceDesc.lpSurface + nColumnOffset);
                    for( i=0; i<nDstHeight; ++i )
                    {
                        *pDstWord = *pSrcWord++;
                        pDstWord += nDstWordRowSkip;
                    }
                }

                rtl_freeMemory( pTempBuf );
            }

            if( SUCCEEDED(rTexture->mpSurface->Unlock( NULL )) )
                return true;
        }

        return false;
    }

    bool Device::applyClipMask( const TextureSharedPtr&				rTexture,
                                const ::basegfx::B2ISize&			rIntendedSize,
                                const ::basegfx::B2DPolyPolygon&	rClipPoly )
    {
        RTL_LOGFILE_CONTEXT( aLog, "::dxcanvas::Device::applyClipMask()" );

        if( !mpDirectDraw.is() ||
            !mpPrimarySurface.is() ||
            !mpBackBufferSurface.is() )
        {
            return false;
        }

        DDSURFACEDESC aSurfaceDesc;
        
        rtl_fillMemory( &aSurfaceDesc, 
                        sizeof(DDSURFACEDESC), 0 );
        aSurfaceDesc.dwSize = sizeof(DDSURFACEDESC);

        if( SUCCEEDED( rTexture->mpSurface->Lock( NULL, 
                                                  &aSurfaceDesc, 
                                                  DDLOCK_NOSYSLOCK|DDLOCK_SURFACEMEMORYPTR|DDLOCK_WAIT|DDLOCK_WRITEONLY,
                                                  NULL )) )
        {
            // TODO(F3): Perform format conversions here!
            const sal_Int32 nSrcWidth( rIntendedSize.getX() );
            const sal_Int32 nSrcHeight( rIntendedSize.getY() );
            const sal_Int32 nDstWidth( (sal_Int32)rTexture->maTextureRect.getWidth() );
            const sal_Int32 nDstHeight( (sal_Int32)rTexture->maTextureRect.getHeight() );

            ::basegfx::B2DPolyPolygon aLocalPoly( rClipPoly );
            if( nDstWidth != nSrcWidth ||
                nDstHeight != nSrcHeight )
            {
                // Need to scale polygon
                ::basegfx::B2DHomMatrix aMatrix;
                aMatrix.scale( (double) nDstWidth / nSrcWidth,
                               (double) nDstHeight/ nSrcHeight );
                aLocalPoly.transform( aMatrix );
            }

            // Fill texture alpha channel with 0 for areas outside the
            // polygon, and with 255 for inside areas
			PolyPolygonRasterizer aRasterizer( aLocalPoly,
                                               PolyPolygonRasterizer::BOTH_SPANS,
                    						   nDstWidth,
                    						   nDstHeight,
                                               aSurfaceDesc.lPitch,
                                               aSurfaceDesc.lpSurface );
            aRasterizer.rasterConvert( ::basegfx::FillRule_EVEN_ODD );

            if( SUCCEEDED(rTexture->mpSurface->Unlock( NULL )) )
                return true;
        }

        return false;
    }

    bool Device::clearSurface( const PlainSurfaceSharedPtr& rSurface ) const
    {        
        RTL_LOGFILE_CONTEXT( aLog, "::dxcanvas::Device::clearSurface(surface)" );

        if( !mpDirectDraw.is() ||
            !mpPrimarySurface.is() ||
            !mpBackBufferSurface.is() )
        {
            return false;
        }

        DDBLTFX aBltFx;

        rtl_fillMemory( &aBltFx, 
                        sizeof(DDBLTFX), 0 );
        aBltFx.dwSize = sizeof(DDBLTFX);
        aBltFx.dwFillColor =                           
            maSelectedFullscreenMode.selectedDesc.ddpfPixelFormat.dwRBitMask |
            maSelectedFullscreenMode.selectedDesc.ddpfPixelFormat.dwGBitMask |
            maSelectedFullscreenMode.selectedDesc.ddpfPixelFormat.dwBBitMask |
            maSelectedFullscreenMode.selectedDesc.ddpfPixelFormat.dwRGBAlphaBitMask; // set surface to white

		if( FAILED( rSurface->mpSurface->Blt( NULL, 
                                              NULL, 
                                              NULL, 
                                              DDBLT_COLORFILL | DDBLT_WAIT, 
                                              &aBltFx )) )
        {
            return false;
        }
        
        return true;
    }

    bool Device::clearSurface( const TextureSharedPtr& rSurface ) const
    {        
        RTL_LOGFILE_CONTEXT( aLog, "::dxcanvas::Device::clearSurface(texture)" );

        if( !mpDirectDraw.is() ||
            !mpPrimarySurface.is() ||
            !mpBackBufferSurface.is() )
        {
            return false;
        }

        DDBLTFX aBltFx;

        rtl_fillMemory( &aBltFx, 
                        sizeof(DDBLTFX), 0 );
        aBltFx.dwSize = sizeof(DDBLTFX);
        aBltFx.dwFillColor = 0; // clear all, also alpha channel

		if( FAILED( rSurface->mpSurface->Blt( NULL, 
                                              NULL, 
                                              NULL, 
                                              DDBLT_COLORFILL | DDBLT_WAIT, 
                                              &aBltFx )) )
        {
            return false;
        }
        
        return true;
    }

    void Device::setMode()
    {
        // ignore result; if we cannot switch mode, work on as is.
        mpDirectDraw->SetDisplayMode( maSelectedFullscreenMode.selectedDesc.dwWidth,
                                      maSelectedFullscreenMode.selectedDesc.dwHeight,
                                      maSelectedFullscreenMode.selectedDesc.ddpfPixelFormat.dwRGBBitCount,
                                      maSelectedFullscreenMode.selectedDesc.dwRefreshRate,
                                      0 );
    }

    void Device::renderInfoText( const ::rtl::OUString& rStr,
                                 const Gdiplus::PointF& rPos ) const
    {
        // render text directly to primary surface
        SurfaceGraphicsSharedPtr pGraphics;

        if( mbPageFlipping )
            pGraphics = SurfaceGraphicsSharedPtr( new SurfaceGraphics( mpBackBufferSurface ) ); // render on top of backbuffer. We have 
        																						// page flipping, anyway, thus this will
        																						// cost us nothing.
        else
            pGraphics = SurfaceGraphicsSharedPtr( new SurfaceGraphics( mpPrimarySurface ) ); // render FPS directly to front buffer. 
        																					 // That saves us another explicit blit,
        																					 // and for me, the FPS counter can blink, 
        																					 // if it likes to...

        if( !mbPageFlipping )
        {
            // clear background. We might be doing optimized redraws,
            // and the background under the FPS count will then not be
            // cleared.
            Gdiplus::SolidBrush aBrush( 
                Gdiplus::Color( 255, 255, 255 ) );

            pGraphics->get()->FillRectangle( &aBrush,
                                             rPos.X, rPos.Y, 80.0, 20.0 );
        }

        Gdiplus::SolidBrush aBrush( 
            Gdiplus::Color( 255, 0, 255 ) );
        Gdiplus::Font aFont( NULL, 
                             16,
                             Gdiplus::FontStyleRegular,
                             Gdiplus::UnitWorld,
                             NULL );
        pGraphics->get()->DrawString( rStr.getStr(), 
                                      rStr.getLength(),
                                      &aFont,
                                      rPos,
                                      &aBrush );
    }

    void Device::renderMemAvailable() const
    {
        const double nSurfaceMem( getAvailableSurfaceMem()/1024 );
                        
        ::rtl::OUString text( ::rtl::math::doubleToUString( nSurfaceMem,
                                                            rtl_math_StringFormat_F,
                                                            2,'.',NULL,' ') );

        // pad with leading space
        while( text.getLength() < 6 )
            text = ::rtl::OUString(RTL_CONSTASCII_USTRINGPARAM (" ")) + text;

        text = ::rtl::OUString(RTL_CONSTASCII_USTRINGPARAM ("S: ")) + text;

        renderInfoText( text,
                        Gdiplus::PointF( 0.0, 20) );


        const double nTexMem( getAvailableTextureMem()/1024 );

        text = ::rtl::math::doubleToUString( nTexMem,
                                             rtl_math_StringFormat_F,
                                             2,'.',NULL,' ');
        // pad with leading space
        while( text.getLength() < 6 )
            text = ::rtl::OUString(RTL_CONSTASCII_USTRINGPARAM (" ")) + text;

        text = ::rtl::OUString(RTL_CONSTASCII_USTRINGPARAM ("T: ")) + text;

        renderInfoText( text,
                        Gdiplus::PointF( 0.0, 40) );
    }

    void Device::renderFPSCounter() const
    {
        const double denominator( maLastUpdate.getElapsedTime() );
        maLastUpdate.reset();
                        
        ::rtl::OUString text( ::rtl::math::doubleToUString( denominator == 0.0 ? 100.0 : 1.0/denominator,
                                                            rtl_math_StringFormat_F,
                                                            2,'.',NULL,' ') );

        // pad with leading space
        while( text.getLength() < 6 )
            text = ::rtl::OUString(RTL_CONSTASCII_USTRINGPARAM (" ")) + text;

        text += ::rtl::OUString(RTL_CONSTASCII_USTRINGPARAM (" fps"));

        renderInfoText( text,
                        Gdiplus::PointF() ); 
    }

    ::std::size_t Device::getAvailableSurfaceMem() const
    {
        if( !mpDirectDraw.is() )
            return 0;

        ::std::size_t nRes( 0 );

        DDSCAPS aSurfaceCaps;
        DWORD	nTotal, nFree;

        // real VRAM (const_cast, since GetAvailableVidMem is non-const)
        aSurfaceCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN | DDSCAPS_VIDEOMEMORY | DDSCAPS_LOCALVIDMEM;
        if( FAILED(const_cast<IDirectDraw2&>(*mpDirectDraw).GetAvailableVidMem( &aSurfaceCaps, &nTotal, &nFree )) )
            return 0;

        nRes += nFree;

        // AGP RAM (const_cast, since GetAvailableVidMem is non-const)
        aSurfaceCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN | DDSCAPS_VIDEOMEMORY | DDSCAPS_NONLOCALVIDMEM;
        if( FAILED(const_cast<IDirectDraw2&>(*mpDirectDraw).GetAvailableVidMem( &aSurfaceCaps, &nTotal, &nFree )) )
            return 0;

        nRes += nFree;

        return nRes;
    }

    ::std::size_t Device::getAvailableTextureMem() const
    {
        if( !mpDirectDraw.is() )
            return 0;

        ::std::size_t nRes( 0 );

        DDSCAPS aSurfaceCaps;
        DWORD	nTotal, nFree;

        // TODO(F1): Check if flags are applicable

        // real VRAM (const_cast, since GetAvailableVidMem is non-const)
        aSurfaceCaps.dwCaps = DDSCAPS_TEXTURE | DDSCAPS_VIDEOMEMORY | DDSCAPS_LOCALVIDMEM;
        if( FAILED(const_cast<IDirectDraw2&>(*mpDirectDraw).GetAvailableVidMem( &aSurfaceCaps, &nTotal, &nFree )) )
            return 0;

        nRes += nFree;

        // AGP RAM (const_cast, since GetAvailableVidMem is non-const)
        aSurfaceCaps.dwCaps = DDSCAPS_TEXTURE | DDSCAPS_VIDEOMEMORY | DDSCAPS_NONLOCALVIDMEM;
        if( FAILED(const_cast<IDirectDraw2&>(*mpDirectDraw).GetAvailableVidMem( &aSurfaceCaps, &nTotal, &nFree )) )
            return 0;

        nRes += nFree;

        // TODO(F1): Add pool mem

        return nRes;
    }

}
