/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: Search.java,v $
 *
 *  $Revision: 1.2 $
 *
 *  last change: $Author: rt $ $Date: 2005/09/09 17:02:16 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

package com.sun.xmlsearch.xml.qe;

import java.io.*;
import com.sun.xmlsearch.util.*;
import com.sun.xmlsearch.xml.XmlIndex;

final class Search {
    private static final int InitNConcepts = 128;
  
    private XmlIndex _env;
    private int _max;
    private int _nConcepts;
    private int _nQueries;
    private int _nQueriesSize = 8;
    private ConceptGroupGenerator _firstGenerator = new ConceptGroupGenerator();
    private int[] _concepts = new int[DocumentCompressor.NConceptsInGroup];
    private int _free2;
    private int _size2;
    private int _startingIndex = 0;
    private int _limit = 0;
    private Query[] _queries;
    private ConceptData[] _conceptData;
    private GeneratorHeap _genHeap = new GeneratorHeap();
    private int _document;
    private byte[] _data = null;
    private int _base = 0;	// index into _data
    private NextDocGeneratorHeap _nextDocGenHeap = new NextDocGeneratorHeap();
    private IntegerArray _kTable = new IntegerArray();
    private IntegerArray _offsets = new IntegerArray();
    private IntegerArray _maxConcepts = new IntegerArray();

    private IntegerArray _docConcepts = new IntegerArray();
    private IntegerArray _queryMasks = new IntegerArray();
    private int _maxHitsToShow = 25;
    private QueryFactoryImpl _queryFactory = new QueryFactoryImpl();

    public Search(XmlIndex se) {
	_env = se;
	_nQueries = 0;
	_queries = new Query[_nQueriesSize];
  
	_size2 = InitNConcepts;
	_free2 = 0;
	_conceptData = new ConceptData[_size2];
    }

    private void addTerm(int col, int concept, double score, int query) {
	if (_env.occursInText(concept))
	    addConceptData(_queries[query].makeConceptData(query, col,
							   concept, score));
    }

    public void addConceptData(ConceptData cd) {
	if (_free2 == _size2) {
	    ConceptData[] newArray = new ConceptData[_size2 *= 2];
	    System.arraycopy(_conceptData, 0, newArray, 0, _free2);
	    _conceptData = newArray;
	}
	_conceptData[_free2++] = cd;
    }

    public Query addQuery(String context,
			  int nValidTerms, int nMissingTerms, int nHits,
			  int[] primary, IntegerArray[] columns,
			  double variantPenalty) {
	final Query query = _queryFactory.makeQuery(_env, context,
						    nValidTerms, nHits);
	query.missingTerms(nMissingTerms);
	final int queryNo = _nQueries++;
	if (queryNo == _nQueriesSize) {
	    Query[] newArray = new Query[_nQueriesSize *= 2];
	    System.arraycopy(_queries, 0, newArray, 0, queryNo);
	    _queries = newArray;
	}
	_queries[queryNo] = query;
	for (int i = 0; i < nValidTerms; i++) {
	    if (primary[i] > 0) {
		addTerm(i, primary[i], 0.0, queryNo);
	    }
	    for (int j = 0; j < columns[i].cardinality(); j++) {
		addTerm(i, columns[i].at(j), variantPenalty, queryNo);
	    }
	}
	// start stop
	query.addControlConceptData(this, queryNo);
	return query;
    }
  
    public void startSearch() {
	int i, j;
	// set up ConceptData lists
	// order search terms
	quicksort(0, _free2 - 1);
	// remove duplicates
	for (i = 0; i < _free2 - 1; i = j) {
	    for (j = i + 1; j < _free2; j++) {
		if (_conceptData[i].crqEquals(_conceptData[j])) {
		    _conceptData[j] = null;
		}
		else {
		    i = j;
		}
	    }
	}
	// create lists
	for (i = 0; i < _free2 - 1; i = j) {
	    for (j = i + 1; j < _free2; j++) {
		if (_conceptData[j] != null) {
		    if (_conceptData[i].cEquals(_conceptData[j])) {
			_conceptData[i].addLast(_conceptData[j]);
			_conceptData[j] = null;
		    }
		    else {
			i = j;
		    }
		}
	    }
	}
	// densify
	for (i = 0; i < _free2 - 1; i++)
	    if (_conceptData[i] == null)
		for (j = i + 1; j < _free2; j++)
		    if (_conceptData[j] != null) {
			_conceptData[i] = _conceptData[j];
			_conceptData[j] = null;
			break;
		    }
	// set up new document generators
	_nextDocGenHeap.reset();
	for (i = 0; i < _free2 && _conceptData[i] != null; i++) {
	    final NextDocGenerator gen = new NextDocGenerator(_conceptData[i], _env);
	    try {
		gen.first();
		if (gen.getDocument() != NonnegativeIntegerGenerator.END) {
		    /* !!! ??? is concept length used any more in any way
		       _conceptData[i].
		       setConceptLength(_env.
		       getConceptLength(_conceptData[i].getConcept()));
		       */
		    _nextDocGenHeap.addGenerator(gen);
		}
	    }
	    catch (Exception e) {
		e.printStackTrace();
	    }
	}
	_nextDocGenHeap.start(); 
	_env.reset();
	_env.resetContextSearch();
	searchDocument();
    }

    private void searchDocument() {
	final RoleFiller[] start = new RoleFiller[_nQueries];
	do {
	    try {
		switch (nextDocument(start)) {
		case 0:		// multi group
		    _genHeap.start(start);
		    while (_genHeap.next(start))
			;
		    break;
		    
		case 1:		// single group
		    if (_firstGenerator.next()) {
			_firstGenerator.generateFillers(start);
			while (_firstGenerator.next()) {
			    _firstGenerator.generateFillers(start);
			}
		    }
		    break;
	    
		case 2:		// reached the end
		    return;
		}
	    }
	    catch (Exception e) {
		e.printStackTrace(System.err);
		continue;
	    }
      
	    for (int i = 0; i < _nQueries; i++) {
		final RoleFiller next;
		if ((next = start[i]) != null && next != RoleFiller.STOP) {
		    next.scoreList(_queries[i], _document);
		}
		else if (_queries[i].zoned()) {
		    final RoleFiller rfs = _queries[i].getRoleFillers();
		    if (rfs != null && rfs != RoleFiller.STOP)
			rfs.scoreList(_queries[i], _document);
		}
	    }
	    _genHeap.reset();
	}
	while (_nextDocGenHeap.isNonEmpty());
    }
  
    // will be called for increasing values of c
    // searches index interval [_startingIndex, _nConcepts]
    // saves state in _startingIndex
    private int indexOf(int concept) throws Exception {
	int i = _startingIndex, j = _nConcepts, k;
	while (i <= j) {
	    if (_concepts[k = (i + j)/2] < concept) {
		i = k + 1;
	    }
	    else if (concept < _concepts[k]) {
		j = k - 1;
	    }
	    else {
		_startingIndex = k + 1;
		return k;
	    }
	}
	throw new Exception("indexOf " + concept + " not found");
    }

    public void results() {
	for (int q = 0; q < _nQueries; q++) {
	    System.out.println("query " + q);
	    if (_queries[q] != null)
		synchronized (_queries[q]) {
		    _queries[q].notify();
		}
	}
    }

    /*
      public void printResults(int nHits) {
      for (int q = 0; q < _nQueries; q++) {
      System.out.println("query " + q);
      if (_queries[q] != null)
      _queries[q].printHits(System.out, nHits);
      }
      }
      */

    private ConceptGroupGenerator makeGenerator(final int group) throws Exception {
	final int shift, index;
	if (group > 0) {
	    index = _base + _offsets.at(group - 1);
	    shift = _maxConcepts.at(group - 1);
	} else {
	    index = _base;
	    shift = 0;
	}
  
	// initialize generator
	final ConceptGroupGenerator gen =
	    new ConceptGroupGenerator(_data, index, _kTable.at(2*group + 1));
	// decode concept table
	_nConcepts = gen.decodeConcepts(_kTable.at(2*group), shift, _concepts);
	if (group < _limit) {
	    _max = _concepts[_nConcepts] = _maxConcepts.at(group);
	}
	else {
	    _max = _concepts[_nConcepts - 1];
	}
	_genHeap.addGenerator(gen);
	_startingIndex = 0;		// in _concepts; lower search index
	return gen;
    }

    // returns true if multigroup
    private boolean openDocumentIndex(int docNo) throws Exception {
	_data = _env.getPositions(docNo);
	_base = _env.getDocumentIndex(docNo);
    
	_startingIndex = 0;
	final int kk = _data[_base] & 0xFF, k2;
	switch (kk >> 6) {		// get type
	case 0:			// single group, no extents
	    k2 = _data[_base + 1];
	    _firstGenerator.init(_data, _base += 2, k2);
	    // decode concept table
	    _nConcepts = _firstGenerator.decodeConcepts(kk & 0x3F, 0,
							_concepts);
	    return false;
      
	case 2:			// multi group, no extents
	    _kTable.clear();
	    _offsets.clear();
	    _maxConcepts.clear();
	    ByteArrayDecompressor compr =
		new ByteArrayDecompressor(_data, _base + 1);
	    compr.decode(kk & 0x3F, _kTable);
	    compr.ascDecode(_kTable.popLast(), _offsets);
	    compr.ascDecode(_kTable.popLast(), _maxConcepts);
	    _base += 1 + compr.bytesRead();
	    _limit = _maxConcepts.cardinality();
	    return true;
      
	case 1:			// single group, extents
	case 3:			// multi group, extents
	    throw new Exception("extents not yet implemented\n");
	}
	return false;
    }

    private int nextDocument(RoleFiller[] start) throws Exception {
	while (_nextDocGenHeap.isNonEmpty()) { // still something to do
	    for (int i = 0; i < _nQueries; i++) {
		if (_queries[i] != null) {
		    _queries[i].resetForNextDocument();
		}
	    }
      
	    // gather all concepts this document has
	    // and store associated conceptData
	    int index = 0;
	    _document = _nextDocGenHeap.getDocument();
	    _docConcepts.clear();
	    _queryMasks.clear();
	    do {
		_docConcepts.add(_nextDocGenHeap.getConcept());
		_queryMasks.add(_nextDocGenHeap.getQueryMask());
		(_conceptData[index++] = _nextDocGenHeap.getTerms()).runBy(_queries);
		_nextDocGenHeap.step();
	    }
	    while (_nextDocGenHeap.atDocument(_document));

	    // if there is no saturation model, some query will always vote YES
	    // and so every document will be opened
	    // even if this case, however, savings can be achieved by not generating fillers
	    // for some queries (not scoring, etc)
	    // and, with more care, creation of some GroupGenerators can be avoided
	    // saturating queries with lots of good hits will lead to best results
	    /*
	      System.out.println("_document = " + _document);
	      System.out.println("docName = " + _env.documentName(_document));
	      */
	    int voteMask = 0;
	    for (int i = 0; i < _nQueries; i++) {
		final Query query = _queries[i];
		if (query != null) {
		    query.saveRoleFillers(null);
		    if (query.vote()) {
			// normal reset
			start[i] = query.zoned() ? RoleFiller.STOP : null;
			voteMask |= 1 << i;
		    }
		    else {
			start[i] = RoleFiller.STOP;	// prohibit setting
		    }
		}
	    }
      
	    // we may eliminate some ConceptGroupGenerators
	    // those which would be used only by Queries which voted NO
	    if (voteMask != 0) {		// need to open up document
		ConceptGroupGenerator gen;
		// !!! don't gather Fillers for disinterested Queries
		if (openDocumentIndex(_document)) {// multi group
		    // set up all needed generators
		    int i = 0;
		    while ((_queryMasks.at(i) & voteMask) == 0)
			++i;
		    //		assert(i < index);
		    int c = _docConcepts.at(i);
		    int group = 0;
		    // find first group
		    while (c > _maxConcepts.at(group) && ++group < _limit)
			;
		    gen = makeGenerator(group);
		    gen.addTerms(indexOf(c), _conceptData[i]);
	      
		    for (++i; i < index; i++) {
			if ((_queryMasks.at(i) & voteMask) > 0) {
			    c = _docConcepts.at(i);
			    if (c > _max) {	// need to find another group
				//		assert(group < _limit);
				while (c > _maxConcepts.at(group) &&
				       ++group < _limit)
				    ;
				gen = makeGenerator(group);
			    }
			    gen.addTerms(indexOf(c), _conceptData[i]);
			}
		    }
		    return 0;
		}
		else {			// single group
		    for (int i = 0; i < index; i++) {
			if ((_queryMasks.at(i) & voteMask) != 0) {
			    _firstGenerator
				.addTerms(indexOf(_docConcepts.at(i)),
					  _conceptData[i]);
			}
		    }
		    return 1;
		}
	    }
	}
	return 2;
    }

    public void listPositions() throws Exception {
	for (int d = 0; d < _env.nDocuments(); d++) {
	    if (openDocumentIndex(d)) {  // multi group
		System.out.println("open document " + _env.documentName(d));
		for (int i = 0; i <= _limit; i++) {
		    final ConceptGroupGenerator gen = makeGenerator(i);
		    System.out.println("generator " + i);
		    while (gen.step()) {
			int code = gen.getConceptCode();
			int concept = _concepts[code];
			String name = _env.fetch(concept);
			System.out.println("\t" + name
					   + " @ " + gen.position());
		    }
		}
	    }
	    else {		// firstGenerator
		System.out.println("open document " + _env.documentName(d));
		System.out.println("only generator");
		while (_firstGenerator.step()) {
		    int code = _firstGenerator.getConceptCode();
		    int concept = _concepts[code];
		    String name = _env.fetch(concept);
		    System.out.println("\t" + name
				       + " @ " + _firstGenerator.position());
		}
	    }
	}
    }

    public void listConcepts(int d, int start, int stop) throws Exception {
	if (openDocumentIndex(d)) {  // multi group
	    System.out.println("open document " + _env.documentName(d));
	    final GeneratorHeap heap = new GeneratorHeap();
	    for (int i = 0; i <= _limit; i++) {
		final ConceptGroupGenerator gen = makeGenerator(i);
		heap.addGenerator(gen);
		int[] concepts = new int[DocumentCompressor.NConceptsInGroup];
		System.arraycopy(_concepts, 0, concepts, 0, concepts.length);
		gen.setConceptTable(concepts);
	    }
	    if (heap.start()) {
		do {
		    final int position = heap.position();
		    final int concept = heap.getConcept();
		    if (position > start && position < stop) {
			final String name = _env.fetch(concept);
			System.out.println("\t" + name + " @ " + position);
		    }
		}
		while (heap.next());
	    }
	}
	else {		// firstGenerator
	    System.out.println("open document " + _env.documentName(d));
	    System.out.println("only generator");
	    while (_firstGenerator.step()) {
		int code = _firstGenerator.getConceptCode();
		int concept = _concepts[code];
		String name = _env.fetch(concept);
		System.out.println("\t" + name
				   + " @ " + _firstGenerator.position());
	    }
	}
    }

    // part of quicksearch
    private int partition(int p, int r) {
	final ConceptData x = _conceptData[(p + r) >>> 1];
	int i = p - 1, j = r + 1;
	while (true) {
	    while (x.compareWith(_conceptData[--j]))
		;
	    while (_conceptData[++i].compareWith(x))
		;
	    if (i < j) {
		final ConceptData t = _conceptData[i];
		_conceptData[i] = _conceptData[j];
		_conceptData[j] = t;
	    }
	    else {
		return j;
	    }
	}
    }

    private void quicksort(int p, int r) {
	while (p < r) {
	    final int q = partition(p, r);
	    quicksort(p, q);
	    p = q + 1;
	}
    }
}
