/*
 * Copyright (c) 2013, 2016, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package test.java.time.format;

import java.util.HashMap;
import java.util.Locale;
import java.util.Map;

class ZoneName {

    public static String toZid(String zid, Locale locale) {
        String mzone = zidToMzone.get(zid);
        if (mzone == null && aliases.containsKey(zid)) {
            zid = aliases.get(zid);
            mzone = zidToMzone.get(zid);
        }
        if (mzone != null) {
            Map<String, String> map = mzoneToZidL.get(mzone);
            if (map != null && map.containsKey(locale.getCountry())) {
                zid = map.get(locale.getCountry());
            } else {
                zid = mzoneToZid.get(mzone);
            }
        }
        return toZid(zid);
    }

    public static String toZid(String zid) {
        if (aliases.containsKey(zid)) {
            return aliases.get(zid);
        }
        return zid;
    }

    private static final String[] zidMap = new String[] {
        "Asia/Bangkok", "Indochina", "Asia/Saigon",
        "Pacific/Pago_Pago", "Samoa", "Pacific/Apia",
        "Africa/Blantyre", "Africa_Central", "Africa/Maputo",
        "America/Argentina/San_Juan", "Argentina", "America/Buenos_Aires",
        "America/Cancun", "America_Central", "America/Chicago",
        "Pacific/Nauru", "Nauru", "Pacific/Nauru",
        "America/Atikokan", "America_Eastern", "America/New_York",
        "Africa/Asmara", "Africa_Eastern", "Africa/Nairobi",
        "Europe/Berlin", "Europe_Central", "Europe/Paris",
        "Asia/Kolkata", "India", "Asia/Calcutta",
        "Australia/Darwin", "Australia_Central", "Australia/Adelaide",
        "America/Guayaquil", "Ecuador", "America/Guayaquil",
        "Europe/Vienna", "Europe_Central", "Europe/Paris",
        "Atlantic/St_Helena", "GMT", "Atlantic/Reykjavik",
        "Europe/London", "GMT", "Atlantic/Reykjavik",
        "Europe/Moscow", "Moscow", "Europe/Moscow",
        "America/St_Vincent", "Atlantic", "America/Halifax",
        "America/Bogota", "Colombia", "America/Bogota",
        "America/Marigot", "Atlantic", "America/Halifax",
        "Europe/Sarajevo", "Europe_Central", "Europe/Paris",
        "America/Hermosillo", "America_Mountain", "America/Denver",
        "America/Winnipeg", "America_Central", "America/Chicago",
        "America/Rainy_River", "America_Central", "America/Chicago",
        "Indian/Mahe", "Seychelles", "Indian/Mahe",
        "Africa/Freetown", "GMT", "Atlantic/Reykjavik",
        "America/Grand_Turk", "America_Eastern", "America/New_York",
        "America/Argentina/Ushuaia", "Argentina", "America/Buenos_Aires",
        "Atlantic/Azores", "Azores", "Atlantic/Azores",
        "Asia/Harbin", "China", "Asia/Shanghai",
        "America/Cuiaba", "Amazon", "America/Manaus",
        "Asia/Bahrain", "Arabian", "Asia/Riyadh",
        "Asia/Katmandu", "Nepal", "Asia/Katmandu",
        "Pacific/Galapagos", "Galapagos", "Pacific/Galapagos",
        "Asia/Brunei", "Brunei", "Asia/Brunei",
        "Africa/Kigali", "Africa_Central", "Africa/Maputo",
        "Asia/Makassar", "Indonesia_Central", "Asia/Makassar",
        "Africa/Maputo", "Africa_Central", "Africa/Maputo",
        "Asia/Kamchatka", "Magadan", "Asia/Magadan",
        "Atlantic/Faroe", "Europe_Western", "Atlantic/Canary",
        "America/El_Salvador", "America_Central", "America/Chicago",
        "Asia/Saigon", "Indochina", "Asia/Saigon",
        "Africa/Kinshasa", "Africa_Western", "Africa/Lagos",
        "Europe/Oslo", "Europe_Central", "Europe/Paris",
        "Asia/Hong_Kong", "Hong_Kong", "Asia/Hong_Kong",
        "Pacific/Midway", "Samoa", "Pacific/Apia",
        "Africa/Douala", "Africa_Western", "Africa/Lagos",
        "Europe/San_Marino", "Europe_Central", "Europe/Paris",
        "Pacific/Chuuk", "Truk", "Pacific/Truk",
        "Africa/Gaborone", "Africa_Central", "Africa/Maputo",
        "Africa/Tunis", "Europe_Central", "Europe/Paris",
        "Africa/Khartoum", "Africa_Eastern", "Africa/Nairobi",
        "Europe/Isle_of_Man", "GMT", "Atlantic/Reykjavik",
        "Europe/Skopje", "Europe_Central", "Europe/Paris",
        "America/Merida", "America_Central", "America/Chicago",
        "Antarctica/DumontDUrville", "DumontDUrville", "Antarctica/DumontDUrville",
        "Atlantic/Reykjavik", "GMT", "Atlantic/Reykjavik",
        "Indian/Mauritius", "Mauritius", "Indian/Mauritius",
        "Africa/Malabo", "Africa_Western", "Africa/Lagos",
        "Africa/Juba", "Africa_Eastern", "Africa/Nairobi",
        "America/Resolute", "America_Central", "America/Chicago",
        "Africa/Abidjan", "GMT", "Atlantic/Reykjavik",
        "Antarctica/McMurdo", "New_Zealand", "Pacific/Auckland",
        "Asia/Thimphu", "Bhutan", "Asia/Thimphu",
        "Europe/Zaporozhye", "Europe_Eastern", "Europe/Bucharest",
        "Antarctica/Davis", "Davis", "Antarctica/Davis",
        "Indian/Antananarivo", "Africa_Eastern", "Africa/Nairobi",
        "Africa/Harare", "Africa_Central", "Africa/Maputo",
        "Pacific/Marquesas", "Marquesas", "Pacific/Marquesas",
        "Africa/Tripoli", "Europe_Eastern", "Europe/Bucharest",
        "America/North_Dakota/Beulah", "America_Central", "America/Chicago",
        "America/Buenos_Aires", "Argentina", "America/Buenos_Aires",
        "America/Tortola", "Atlantic", "America/Halifax",
        "Asia/Kuwait", "Arabian", "Asia/Riyadh",
        "Europe/Rome", "Europe_Central", "Europe/Paris",
        "America/Eirunepe", "Amazon", "America/Manaus",
        "Australia/Hobart", "Australia_Eastern", "Australia/Sydney",
        "America/Thule", "Atlantic", "America/Halifax",
        "Asia/Beirut", "Europe_Eastern", "Europe/Bucharest",
        "America/Bahia_Banderas", "America_Central", "America/Chicago",
        "Africa/Dar_es_Salaam", "Africa_Eastern", "Africa/Nairobi",
        "America/Argentina/Tucuman", "Argentina", "America/Buenos_Aires",
        "America/Paramaribo", "Suriname", "America/Paramaribo",
        "Africa/Kampala", "Africa_Eastern", "Africa/Nairobi",
        "Pacific/Port_Moresby", "Papua_New_Guinea", "Pacific/Port_Moresby",
        "America/Mendoza", "Argentina", "America/Buenos_Aires",
        "Asia/Dushanbe", "Tajikistan", "Asia/Dushanbe",
        "Asia/Qyzylorda", "Kazakhstan_Eastern", "Asia/Almaty",
        "Antarctica/Vostok", "Vostok", "Antarctica/Vostok",
        "Pacific/Majuro", "Marshall_Islands", "Pacific/Majuro",
        "Asia/Tehran", "Iran", "Asia/Tehran",
        "Asia/Hovd", "Hovd", "Asia/Hovd",
        "Antarctica/Rothera", "Rothera", "Antarctica/Rothera",
        "Africa/Brazzaville", "Africa_Western", "Africa/Lagos",
        "Europe/Tirane", "Europe_Central", "Europe/Paris",
        "Asia/Urumqi", "China", "Asia/Shanghai",
        "Asia/Krasnoyarsk", "Krasnoyarsk", "Asia/Krasnoyarsk",
        "America/Tegucigalpa", "America_Central", "America/Chicago",
        "Asia/Vientiane", "Indochina", "Asia/Saigon",
        "Asia/Pontianak", "Indonesia_Western", "Asia/Jakarta",
        "America/Bahia", "Brasilia", "America/Sao_Paulo",
        "Asia/Choibalsan", "Choibalsan", "Asia/Choibalsan",
        "America/Regina", "America_Central", "America/Chicago",
        "Africa/Cairo", "Europe_Eastern", "Europe/Bucharest",
        "Asia/Irkutsk", "Irkutsk", "Asia/Irkutsk",
        "Europe/Luxembourg", "Europe_Central", "Europe/Paris",
        "America/St_Kitts", "Atlantic", "America/Halifax",
        "America/Manaus", "Amazon", "America/Manaus",
        "America/Noronha", "Noronha", "America/Noronha",
        "Pacific/Gambier", "Gambier", "Pacific/Gambier",
        "America/Edmonton", "America_Mountain", "America/Denver",
        "Pacific/Palau", "Palau", "Pacific/Palau",
        "America/Lower_Princes", "Atlantic", "America/Halifax",
        "Africa/Ouagadougou", "GMT", "Atlantic/Reykjavik",
        "Asia/Yerevan", "Armenia", "Asia/Yerevan",
        "America/Montevideo", "Uruguay", "America/Montevideo",
        "Europe/Minsk", "Europe_Eastern", "Europe/Bucharest",
        "Europe/Amsterdam", "Europe_Central", "Europe/Paris",
        "Pacific/Efate", "Vanuatu", "Pacific/Efate",
        "Asia/Manila", "Philippines", "Asia/Manila",
        "America/Dawson", "America_Pacific", "America/Los_Angeles",
        "America/Argentina/Cordoba", "Argentina", "America/Buenos_Aires",
        "Australia/Melbourne", "Australia_Eastern", "Australia/Sydney",
        "Asia/Rangoon", "Myanmar", "Asia/Rangoon",
        "America/Los_Angeles", "America_Pacific", "America/Los_Angeles",
        "Africa/Casablanca", "Europe_Western", "Atlantic/Canary",
        "Africa/Porto-Novo", "Africa_Western", "Africa/Lagos",
        "Asia/Macau", "China", "Asia/Shanghai",
        "America/Boa_Vista", "Amazon", "America/Manaus",
        "Europe/Guernsey", "GMT", "Atlantic/Reykjavik",
        "Africa/Monrovia", "GMT", "Atlantic/Reykjavik",
        "America/Godthab", "Greenland_Western", "America/Godthab",
        "Africa/Ceuta", "Europe_Central", "Europe/Paris",
        "Asia/Oral", "Kazakhstan_Western", "Asia/Aqtobe",
        "America/Yakutat", "Alaska", "America/Juneau",
        "Indian/Mayotte", "Africa_Eastern", "Africa/Nairobi",
        "America/Denver", "America_Mountain", "America/Denver",
        "America/New_York", "America_Eastern", "America/New_York",
        "Pacific/Rarotonga", "Cook", "Pacific/Rarotonga",
        "America/Louisville", "America_Eastern", "America/New_York",
        "Africa/El_Aaiun", "Europe_Western", "Atlantic/Canary",
        "Africa/Sao_Tome", "GMT", "Atlantic/Reykjavik",
        "Pacific/Fiji", "Fiji", "Pacific/Fiji",
        "Asia/Damascus", "Europe_Eastern", "Europe/Bucharest",
        "Asia/Ulaanbaatar", "Mongolia", "Asia/Ulaanbaatar",
        "America/Cayman", "America_Eastern", "America/New_York",
        "America/Tijuana", "America_Pacific", "America/Los_Angeles",
        "Atlantic/Bermuda", "Atlantic", "America/Halifax",
        "Australia/Sydney", "Australia_Eastern", "Australia/Sydney",
        "Asia/Aden", "Arabian", "Asia/Riyadh",
        "Australia/Eucla", "Australia_CentralWestern", "Australia/Eucla",
        "America/Indiana/Petersburg", "America_Eastern", "America/New_York",
        "America/Panama", "America_Eastern", "America/New_York",
        "Europe/Istanbul", "Europe_Eastern", "Europe/Bucharest",
        "America/Kralendijk", "Atlantic", "America/Halifax",
        "America/Catamarca", "Argentina", "America/Buenos_Aires",
        "America/Nassau", "America_Eastern", "America/New_York",
        "Europe/Paris", "Europe_Central", "Europe/Paris",
        "Asia/Jakarta", "Indonesia_Western", "Asia/Jakarta",
        "Australia/Lindeman", "Australia_Eastern", "Australia/Sydney",
        "America/Sao_Paulo", "Brasilia", "America/Sao_Paulo",
        "America/Juneau", "Alaska", "America/Juneau",
        "America/Grenada", "Atlantic", "America/Halifax",
        "America/Cayenne", "French_Guiana", "America/Cayenne",
        "Antarctica/Casey", "Australia_Western", "Australia/Perth",
        "Africa/Algiers", "Europe_Central", "Europe/Paris",
        "America/Miquelon", "Pierre_Miquelon", "America/Miquelon",
        "Asia/Tokyo", "Japan", "Asia/Tokyo",
        "Africa/Windhoek", "Africa_Western", "Africa/Lagos",
        "Africa/Bujumbura", "Africa_Central", "Africa/Maputo",
        "America/Guatemala", "America_Central", "America/Chicago",
        "Africa/Dakar", "GMT", "Atlantic/Reykjavik",
        "Asia/Bishkek", "Kyrgystan", "Asia/Bishkek",
        "America/Guadeloupe", "Atlantic", "America/Halifax",
        "Africa/Ndjamena", "Africa_Western", "Africa/Lagos",
        "Europe/Simferopol", "Europe_Eastern", "Europe/Bucharest",
        "America/Santa_Isabel", "America_Pacific", "America/Los_Angeles",
        "Asia/Dubai", "Gulf", "Asia/Dubai",
        "America/Maceio", "Brasilia", "America/Sao_Paulo",
        "America/Anchorage", "Alaska", "America/Juneau",
        "Australia/Currie", "Australia_Eastern", "Australia/Sydney",
        "Africa/Djibouti", "Africa_Eastern", "Africa/Nairobi",
        "Europe/Budapest", "Europe_Central", "Europe/Paris",
        "America/Argentina/Salta", "Argentina", "America/Buenos_Aires",
        "Asia/Calcutta", "India", "Asia/Calcutta",
        "America/Indiana/Winamac", "America_Eastern", "America/New_York",
        "Asia/Yekaterinburg", "Yekaterinburg", "Asia/Yekaterinburg",
        "America/Santiago", "Chile", "America/Santiago",
        "Asia/Aqtobe", "Kazakhstan_Western", "Asia/Aqtobe",
        "Asia/Dili", "East_Timor", "Asia/Dili",
        "America/Detroit", "America_Eastern", "America/New_York",
        "Africa/Libreville", "Africa_Western", "Africa/Lagos",
        "Pacific/Ponape", "Ponape", "Pacific/Ponape",
        "Pacific/Wallis", "Wallis", "Pacific/Wallis",
        "Asia/Vladivostok", "Vladivostok", "Asia/Vladivostok",
        "Africa/Lubumbashi", "Africa_Central", "Africa/Maputo",
        "Africa/Asmera", "Africa_Eastern", "Africa/Nairobi",
        "Pacific/Guam", "Chamorro", "Pacific/Saipan",
        "America/Chicago", "America_Central", "America/Chicago",
        "America/Swift_Current", "America_Central", "America/Chicago",
        "America/Coral_Harbour", "America_Eastern", "America/New_York",
        "America/Cambridge_Bay", "America_Mountain", "America/Denver",
        "America/Costa_Rica", "America_Central", "America/Chicago",
        "America/Curacao", "Atlantic", "America/Halifax",
        "America/Recife", "Brasilia", "America/Sao_Paulo",
        "Africa/Bangui", "Africa_Western", "Africa/Lagos",
        "America/Cordoba", "Argentina", "America/Buenos_Aires",
        "Asia/Baghdad", "Arabian", "Asia/Riyadh",
        "America/Shiprock", "America_Mountain", "America/Denver",
        "America/Glace_Bay", "Atlantic", "America/Halifax",
        "America/North_Dakota/Center", "America_Central", "America/Chicago",
        "Europe/Stockholm", "Europe_Central", "Europe/Paris",
        "America/Halifax", "Atlantic", "America/Halifax",
        "Atlantic/Canary", "Europe_Western", "Atlantic/Canary",
        "Europe/Volgograd", "Volgograd", "Europe/Volgograd",
        "America/Moncton", "Atlantic", "America/Halifax",
        "Pacific/Tongatapu", "Tonga", "Pacific/Tongatapu",
        "America/Argentina/Buenos_Aires", "Argentina", "America/Buenos_Aires",
        "Asia/Samarkand", "Uzbekistan", "Asia/Tashkent",
        "Pacific/Apia", "Samoa", "Pacific/Apia",
        "America/Sitka", "Alaska", "America/Juneau",
        "Europe/Warsaw", "Europe_Central", "Europe/Paris",
        "Africa/Accra", "GMT", "Atlantic/Reykjavik",
        "Europe/Bratislava", "Europe_Central", "Europe/Paris",
        "Europe/Zurich", "Europe_Central", "Europe/Paris",
        "Indian/Reunion", "Reunion", "Indian/Reunion",
        "America/Mazatlan", "America_Mountain", "America/Denver",
        "Pacific/Tarawa", "Gilbert_Islands", "Pacific/Tarawa",
        "America/Indiana/Knox", "America_Central", "America/Chicago",
        "Asia/Tbilisi", "Georgia", "Asia/Tbilisi",
        "Asia/Novosibirsk", "Novosibirsk", "Asia/Novosibirsk",
        "Atlantic/Faeroe", "Europe_Western", "Atlantic/Canary",
        "Africa/Bissau", "GMT", "Atlantic/Reykjavik",
        "Asia/Amman", "Europe_Eastern", "Europe/Bucharest",
        "Africa/Lagos", "Africa_Western", "Africa/Lagos",
        "Africa/Banjul", "GMT", "Atlantic/Reykjavik",
        "America/Araguaina", "Brasilia", "America/Sao_Paulo",
        "America/Nipigon", "America_Eastern", "America/New_York",
        "Europe/Vilnius", "Europe_Eastern", "Europe/Bucharest",
        "America/Montserrat", "Atlantic", "America/Halifax",
        "Asia/Baku", "Azerbaijan", "Asia/Baku",
        "Africa/Lusaka", "Africa_Central", "Africa/Maputo",
        "Europe/Uzhgorod", "Europe_Eastern", "Europe/Bucharest",
        "America/Argentina/Rio_Gallegos", "Argentina", "America/Buenos_Aires",
        "America/Blanc-Sablon", "Atlantic", "America/Halifax",
        "Asia/Kabul", "Afghanistan", "Asia/Kabul",
        "America/Jamaica", "America_Eastern", "America/New_York",
        "Europe/Vatican", "Europe_Central", "Europe/Paris",
        "Africa/Nouakchott", "GMT", "Atlantic/Reykjavik",
        "Africa/Addis_Ababa", "Africa_Eastern", "Africa/Nairobi",
        "Europe/Athens", "Europe_Eastern", "Europe/Bucharest",
        "Atlantic/Madeira", "Europe_Western", "Atlantic/Canary",
        "America/Thunder_Bay", "America_Eastern", "America/New_York",
        "Europe/Brussels", "Europe_Central", "Europe/Paris",
        "Africa/Luanda", "Africa_Western", "Africa/Lagos",
        "Africa/Mogadishu", "Africa_Eastern", "Africa/Nairobi",
        "America/Matamoros", "America_Central", "America/Chicago",
        "Pacific/Norfolk", "Norfolk", "Pacific/Norfolk",
        "America/Scoresbysund", "Greenland_Eastern", "America/Scoresbysund",
        "America/Indianapolis", "America_Eastern", "America/New_York",
        "Pacific/Pitcairn", "Pitcairn", "Pacific/Pitcairn",
        "Asia/Singapore", "Singapore", "Asia/Singapore",
        "America/Port-au-Prince", "America_Eastern", "America/New_York",
        "Pacific/Honolulu", "Hawaii_Aleutian", "Pacific/Honolulu",
        "Antarctica/Syowa", "Syowa", "Antarctica/Syowa",
        "Atlantic/Cape_Verde", "Cape_Verde", "Atlantic/Cape_Verde",
        "America/Asuncion", "Paraguay", "America/Asuncion",
        "America/Martinique", "Atlantic", "America/Halifax",
        "Europe/Gibraltar", "Europe_Central", "Europe/Paris",
        "Africa/Lome", "GMT", "Atlantic/Reykjavik",
        "Australia/Lord_Howe", "Lord_Howe", "Australia/Lord_Howe",
        "America/Argentina/La_Rioja", "Argentina", "America/Buenos_Aires",
        "Europe/Jersey", "GMT", "Atlantic/Reykjavik",
        "America/Kentucky/Louisville", "America_Eastern", "America/New_York",
        "America/Monterrey", "America_Central", "America/Chicago",
        "Europe/Belgrade", "Europe_Central", "Europe/Paris",
        "Asia/Gaza", "Europe_Eastern", "Europe/Bucharest",
        "Asia/Ho_Chi_Minh", "Indochina", "Asia/Saigon",
        "Europe/Prague", "Europe_Central", "Europe/Paris",
        "Indian/Christmas", "Christmas", "Indian/Christmas",
        "Pacific/Fakaofo", "Tokelau", "Pacific/Fakaofo",
        "America/Dominica", "Atlantic", "America/Halifax",
        "America/Ojinaga", "America_Mountain", "America/Denver",
        "Asia/Colombo", "India", "Asia/Calcutta",
        "Asia/Nicosia", "Europe_Eastern", "Europe/Bucharest",
        "Europe/Copenhagen", "Europe_Central", "Europe/Paris",
        "America/Creston", "America_Mountain", "America/Denver",
        "Asia/Ashgabat", "Turkmenistan", "Asia/Ashgabat",
        "Asia/Shanghai", "China", "Asia/Shanghai",
        "Pacific/Easter", "Easter", "Pacific/Easter",
        "Africa/Maseru", "Africa_Southern", "Africa/Johannesburg",
        "America/La_Paz", "Bolivia", "America/La_Paz",
        "Pacific/Truk", "Truk", "Pacific/Truk",
        "America/Inuvik", "America_Mountain", "America/Denver",
        "America/Belem", "Brasilia", "America/Sao_Paulo",
        "Asia/Hebron", "Europe_Eastern", "Europe/Bucharest",
        "Asia/Jerusalem", "Israel", "Asia/Jerusalem",
        "America/Belize", "America_Central", "America/Chicago",
        "America/Rio_Branco", "Amazon", "America/Manaus",
        "America/Dawson_Creek", "America_Mountain", "America/Denver",
        "America/Anguilla", "Atlantic", "America/Halifax",
        "America/Port_of_Spain", "Atlantic", "America/Halifax",
        "America/St_Barthelemy", "Atlantic", "America/Halifax",
        "America/Indiana/Marengo", "America_Eastern", "America/New_York",
        "America/St_Johns", "Newfoundland", "America/St_Johns",
        "Asia/Jayapura", "Indonesia_Eastern", "Asia/Jayapura",
        "Europe/Riga", "Europe_Eastern", "Europe/Bucharest",
        "America/Phoenix", "America_Mountain", "America/Denver",
        "America/Boise", "America_Mountain", "America/Denver",
        "Pacific/Kiritimati", "Line_Islands", "Pacific/Kiritimati",
        "Africa/Johannesburg", "Africa_Southern", "Africa/Johannesburg",
        "America/Pangnirtung", "America_Eastern", "America/New_York",
        "America/Toronto", "America_Eastern", "America/New_York",
        "Australia/Brisbane", "Australia_Eastern", "Australia/Sydney",
        "Asia/Aqtau", "Kazakhstan_Western", "Asia/Aqtobe",
        "America/Vancouver", "America_Pacific", "America/Los_Angeles",
        "Africa/Mbabane", "Africa_Southern", "Africa/Johannesburg",
        "Europe/Vaduz", "Europe_Central", "Europe/Paris",
        "Asia/Karachi", "Pakistan", "Asia/Karachi",
        "Asia/Riyadh", "Arabian", "Asia/Riyadh",
        "Indian/Maldives", "Maldives", "Indian/Maldives",
        "Asia/Anadyr", "Magadan", "Asia/Magadan",
        "Europe/Helsinki", "Europe_Eastern", "Europe/Bucharest",
        "America/Nome", "Alaska", "America/Juneau",
        "Asia/Yakutsk", "Yakutsk", "Asia/Yakutsk",
        "Asia/Yangon", "Myanmar", "Asia/Rangoon",
        "Africa/Conakry", "GMT", "Atlantic/Reykjavik",
        "Asia/Seoul", "Korea", "Asia/Seoul",
        "America/Antigua", "Atlantic", "America/Halifax",
        "Asia/Almaty", "Kazakhstan_Eastern", "Asia/Almaty",
        "America/Fortaleza", "Brasilia", "America/Sao_Paulo",
        "Pacific/Tahiti", "Tahiti", "Pacific/Tahiti",
        "Asia/Kashgar", "China", "Asia/Shanghai",
        "America/Whitehorse", "America_Pacific", "America/Los_Angeles",
        "Europe/Kaliningrad", "Europe_Eastern", "Europe/Bucharest",
        "Pacific/Enderbury", "Phoenix_Islands", "Pacific/Enderbury",
        "America/St_Lucia", "Atlantic", "America/Halifax",
        "Atlantic/Stanley", "Falkland", "Atlantic/Stanley",
        "Asia/Omsk", "Omsk", "Asia/Omsk",
        "America/Menominee", "America_Central", "America/Chicago",
        "Asia/Novokuznetsk", "Novosibirsk", "Asia/Novosibirsk",
        "Asia/Sakhalin", "Sakhalin", "Asia/Sakhalin",
        "Asia/Muscat", "Gulf", "Asia/Dubai",
        "Pacific/Noumea", "New_Caledonia", "Pacific/Noumea",
        "Asia/Phnom_Penh", "Indochina", "Asia/Saigon",
        "Antarctica/Mawson", "Mawson", "Antarctica/Mawson",
        "Indian/Cocos", "Cocos", "Indian/Cocos",
        "Europe/Tallinn", "Europe_Eastern", "Europe/Bucharest",
        "Africa/Nairobi", "Africa_Eastern", "Africa/Nairobi",
        "Europe/Ljubljana", "Europe_Central", "Europe/Paris",
        "America/Montreal", "America_Eastern", "America/New_York",
        "Asia/Kuala_Lumpur", "Malaysia", "Asia/Kuching",
        "Asia/Magadan", "Magadan", "Asia/Magadan",
        "Africa/Bamako", "GMT", "Atlantic/Reykjavik",
        "Australia/Broken_Hill", "Australia_Central", "Australia/Adelaide",
        "America/Indiana/Indianapolis", "America_Eastern", "America/New_York",
        "Asia/Taipei", "Taipei", "Asia/Taipei",
        "Europe/Samara", "Moscow", "Europe/Moscow",
        "America/Indiana/Vevay", "America_Eastern", "America/New_York",
        "Atlantic/South_Georgia", "South_Georgia", "Atlantic/South_Georgia",
        "Pacific/Wake", "Wake", "Pacific/Wake",
        "Asia/Tashkent", "Uzbekistan", "Asia/Tashkent",
        "America/St_Thomas", "Atlantic", "America/Halifax",
        "America/Argentina/San_Luis", "Argentina_Western", "America/Argentina/San_Luis",
        "Arctic/Longyearbyen", "Europe_Central", "Europe/Paris",
        "Asia/Chongqing", "China", "Asia/Shanghai",
        "Europe/Monaco", "Europe_Central", "Europe/Paris",
        "Asia/Qatar", "Arabian", "Asia/Riyadh",
        "America/Chihuahua", "America_Mountain", "America/Denver",
        "America/Havana", "Cuba", "America/Havana",
        "Pacific/Auckland", "New_Zealand", "Pacific/Auckland",
        "America/Jujuy", "Argentina", "America/Buenos_Aires",
        "America/Goose_Bay", "Atlantic", "America/Halifax",
        "Africa/Niamey", "Africa_Western", "Africa/Lagos",
        "Asia/Kathmandu", "Nepal", "Asia/Katmandu",
        "America/Caracas", "Venezuela", "America/Caracas",
        "Indian/Comoro", "Africa_Eastern", "Africa/Nairobi",
        "America/Argentina/Jujuy", "Argentina", "America/Buenos_Aires",
        "America/Guyana", "Guyana", "America/Guyana",
        "America/Indiana/Tell_City", "America_Central", "America/Chicago",
        "America/Metlakatla", "America_Pacific", "America/Los_Angeles",
        "Europe/Mariehamn", "Europe_Eastern", "Europe/Bucharest",
        "Europe/Dublin", "GMT", "Atlantic/Reykjavik",
        "Europe/Lisbon", "Europe_Western", "Atlantic/Canary",
        "America/Puerto_Rico", "Atlantic", "America/Halifax",
        "Asia/Pyongyang", "Korea", "Asia/Seoul",
        "America/North_Dakota/New_Salem", "America_Central", "America/Chicago",
        "Asia/Dhaka", "Bangladesh", "Asia/Dhaka",
        "America/Rankin_Inlet", "America_Central", "America/Chicago",
        "America/Adak", "Hawaii_Aleutian", "Pacific/Honolulu",
        "America/Campo_Grande", "Amazon", "America/Manaus",
        "Europe/Chisinau", "Europe_Eastern", "Europe/Bucharest",
        "Pacific/Saipan", "Chamorro", "Pacific/Saipan",
        "Pacific/Niue", "Niue", "Pacific/Niue",
        "Europe/Madrid", "Europe_Central", "Europe/Paris",
        "Pacific/Kwajalein", "Marshall_Islands", "Pacific/Majuro",
        "America/Porto_Velho", "Amazon", "America/Manaus",
        "Indian/Kerguelen", "French_Southern", "Indian/Kerguelen",
        "America/Santarem", "Brasilia", "America/Sao_Paulo",
        "Asia/Kuching", "Malaysia", "Asia/Kuching",
        "Australia/Adelaide", "Australia_Central", "Australia/Adelaide",
        "Europe/Bucharest", "Europe_Eastern", "Europe/Bucharest",
        "Australia/Perth", "Australia_Western", "Australia/Perth",
        "Europe/Sofia", "Europe_Eastern", "Europe/Bucharest",
        "Indian/Chagos", "Indian_Ocean", "Indian/Chagos",
        "America/Yellowknife", "America_Mountain", "America/Denver",
        "America/Managua", "America_Central", "America/Chicago",
        "America/Iqaluit", "America_Eastern", "America/New_York",
        "Pacific/Kosrae", "Kosrae", "Pacific/Kosrae",
        "Pacific/Guadalcanal", "Solomon", "Pacific/Guadalcanal",
        "America/Barbados", "Atlantic", "America/Halifax",
        "America/Aruba", "Atlantic", "America/Halifax",
        "Europe/Andorra", "Europe_Central", "Europe/Paris",
        "Pacific/Chatham", "Chatham", "Pacific/Chatham",
        "America/Santo_Domingo", "Atlantic", "America/Halifax",
        "America/Indiana/Vincennes", "America_Eastern", "America/New_York",
        "Europe/Kiev", "Europe_Eastern", "Europe/Bucharest",
        "Pacific/Funafuti", "Tuvalu", "Pacific/Funafuti",
        "America/Mexico_City", "America_Central", "America/Chicago",
        "America/Kentucky/Monticello", "America_Eastern", "America/New_York",
        "America/Argentina/Catamarca", "Argentina", "America/Buenos_Aires",
        "Pacific/Johnston", "Hawaii_Aleutian", "Pacific/Honolulu",
        "Europe/Podgorica", "Europe_Central", "Europe/Paris",
        "Europe/Zagreb", "Europe_Central", "Europe/Paris",
        "Pacific/Pohnpei", "Ponape", "Pacific/Ponape",
        "Antarctica/Palmer", "Chile", "America/Santiago",
        "America/Argentina/Mendoza", "Argentina", "America/Buenos_Aires",
        "America/Lima", "Peru", "America/Lima",
        "Antarctica/Macquarie", "Macquarie", "Antarctica/Macquarie",
        "Europe/Malta", "Europe_Central", "Europe/Paris",
        "America/Danmarkshavn", "GMT", "Atlantic/Reykjavik",
    };
    private static final String[] mzoneMap = new String[] {
        "America_Eastern", "TC", "America/Grand_Turk",
        "America_Eastern", "BS", "America/Nassau",
        "America_Eastern", "CA", "America/Toronto",
        "America_Eastern", "KY", "America/Cayman",
        "America_Eastern", "PA", "America/Panama",
        "America_Eastern", "JM", "America/Jamaica",
        "America_Pacific", "CA", "America/Vancouver",
        "America_Pacific", "MX", "America/Tijuana",
        "Europe_Western", "FO", "Atlantic/Faeroe",
        "Arabian", "YE", "Asia/Aden",
        "Arabian", "BH", "Asia/Bahrain",
        "Arabian", "KW", "Asia/Kuwait",
        "Arabian", "QA", "Asia/Qatar",
        "Arabian", "IQ", "Asia/Baghdad",
        "Korea", "KP", "Asia/Pyongyang",
        "Africa_Central", "ZW", "Africa/Harare",
        "Africa_Central", "ZM", "Africa/Lusaka",
        "Africa_Central", "MW", "Africa/Blantyre",
        "Africa_Central", "BW", "Africa/Gaborone",
        "Africa_Central", "CD", "Africa/Lubumbashi",
        "Africa_Central", "BI", "Africa/Bujumbura",
        "Africa_Central", "RW", "Africa/Kigali",
        "Africa_Western", "CF", "Africa/Bangui",
        "Africa_Western", "AO", "Africa/Luanda",
        "Africa_Western", "NE", "Africa/Niamey",
        "Africa_Western", "CD", "Africa/Kinshasa",
        "Africa_Western", "CM", "Africa/Douala",
        "Africa_Western", "CG", "Africa/Brazzaville",
        "Africa_Western", "GQ", "Africa/Malabo",
        "Africa_Western", "TD", "Africa/Ndjamena",
        "Africa_Western", "GA", "Africa/Libreville",
        "Atlantic", "PR", "America/Puerto_Rico",
        "Atlantic", "AN", "America/Curacao",
        "Atlantic", "VI", "America/St_Thomas",
        "Atlantic", "GP", "America/Guadeloupe",
        "Atlantic", "TT", "America/Port_of_Spain",
        "Atlantic", "AG", "America/Antigua",
        "Atlantic", "MF", "America/Marigot",
        "Atlantic", "DM", "America/Dominica",
        "Atlantic", "VG", "America/Tortola",
        "Atlantic", "MQ", "America/Martinique",
        "Atlantic", "GL", "America/Thule",
        "Atlantic", "AI", "America/Anguilla",
        "Atlantic", "BB", "America/Barbados",
        "Atlantic", "BM", "Atlantic/Bermuda",
        "Atlantic", "BQ", "America/Kralendijk",
        "Atlantic", "LC", "America/St_Lucia",
        "Atlantic", "MS", "America/Montserrat",
        "Atlantic", "SX", "America/Lower_Princes",
        "Atlantic", "GD", "America/Grenada",
        "Atlantic", "VC", "America/St_Vincent",
        "Atlantic", "KN", "America/St_Kitts",
        "Atlantic", "AW", "America/Aruba",
        "GMT", "GM", "Africa/Banjul",
        "GMT", "LR", "Africa/Monrovia",
        "GMT", "ML", "Africa/Bamako",
        "GMT", "SH", "Atlantic/St_Helena",
        "GMT", "TG", "Africa/Lome",
        "GMT", "GB", "Europe/London",
        "GMT", "MR", "Africa/Nouakchott",
        "GMT", "GN", "Africa/Conakry",
        "GMT", "SL", "Africa/Freetown",
        "GMT", "BF", "Africa/Ouagadougou",
        "GMT", "ST", "Africa/Sao_Tome",
        "GMT", "SN", "Africa/Dakar",
        "GMT", "CI", "Africa/Abidjan",
        "GMT", "IE", "Europe/Dublin",
        "GMT", "GH", "Africa/Accra",
        "Chile", "AQ", "Antarctica/Palmer",
        "America_Central", "CR", "America/Costa_Rica",
        "America_Central", "HN", "America/Tegucigalpa",
        "America_Central", "CA", "America/Winnipeg",
        "America_Central", "SV", "America/El_Salvador",
        "America_Central", "MX", "America/Mexico_City",
        "America_Central", "BZ", "America/Belize",
        "America_Central", "GT", "America/Guatemala",
        "America_Mountain", "CA", "America/Edmonton",
        "America_Mountain", "MX", "America/Hermosillo",
        "New_Zealand", "AQ", "Antarctica/McMurdo",
        "India", "LK", "Asia/Colombo",
        "Gulf", "OM", "Asia/Muscat",
        "China", "MO", "Asia/Macau",
        "Africa_Eastern", "ER", "Africa/Asmera",
        "Africa_Eastern", "TZ", "Africa/Dar_es_Salaam",
        "Africa_Eastern", "SO", "Africa/Mogadishu",
        "Africa_Eastern", "DJ", "Africa/Djibouti",
        "Africa_Eastern", "MG", "Indian/Antananarivo",
        "Africa_Eastern", "KM", "Indian/Comoro",
        "Africa_Eastern", "UG", "Africa/Kampala",
        "Africa_Eastern", "YT", "Indian/Mayotte",
        "Africa_Eastern", "ET", "Africa/Addis_Ababa",
        "Chamorro", "GU", "Pacific/Guam",
        "Africa_Southern", "LS", "Africa/Maseru",
        "Africa_Southern", "SZ", "Africa/Mbabane",
        "Indochina", "KH", "Asia/Phnom_Penh",
        "Indochina", "TH", "Asia/Bangkok",
        "Indochina", "LA", "Asia/Vientiane",
        "Europe_Central", "AT", "Europe/Vienna",
        "Europe_Central", "SK", "Europe/Bratislava",
        "Europe_Central", "BA", "Europe/Sarajevo",
        "Europe_Central", "CZ", "Europe/Prague",
        "Europe_Central", "BE", "Europe/Brussels",
        "Europe_Central", "RS", "Europe/Belgrade",
        "Europe_Central", "SE", "Europe/Stockholm",
        "Europe_Central", "MT", "Europe/Malta",
        "Europe_Central", "IT", "Europe/Rome",
        "Europe_Central", "LU", "Europe/Luxembourg",
        "Europe_Central", "HU", "Europe/Budapest",
        "Europe_Central", "NO", "Europe/Oslo",
        "Europe_Central", "ME", "Europe/Podgorica",
        "Europe_Central", "MK", "Europe/Skopje",
        "Europe_Central", "NL", "Europe/Amsterdam",
        "Europe_Central", "LI", "Europe/Vaduz",
        "Europe_Central", "PL", "Europe/Warsaw",
        "Europe_Central", "ES", "Europe/Madrid",
        "Europe_Central", "TN", "Africa/Tunis",
        "Europe_Central", "SI", "Europe/Ljubljana",
        "Europe_Central", "DE", "Europe/Berlin",
        "Europe_Central", "GI", "Europe/Gibraltar",
        "Europe_Central", "CH", "Europe/Zurich",
        "Europe_Central", "MC", "Europe/Monaco",
        "Europe_Central", "VA", "Europe/Vatican",
        "Europe_Central", "HR", "Europe/Zagreb",
        "Europe_Central", "AL", "Europe/Tirane",
        "Europe_Central", "AD", "Europe/Andorra",
        "Europe_Central", "DK", "Europe/Copenhagen",
        "Europe_Central", "SM", "Europe/San_Marino",
        "Europe_Eastern", "SY", "Asia/Damascus",
        "Europe_Eastern", "FI", "Europe/Helsinki",
        "Europe_Eastern", "AX", "Europe/Mariehamn",
        "Europe_Eastern", "BG", "Europe/Sofia",
        "Europe_Eastern", "EG", "Africa/Cairo",
        "Europe_Eastern", "LB", "Asia/Beirut",
        "Europe_Eastern", "GR", "Europe/Athens",
        "Europe_Eastern", "JO", "Asia/Amman",
        "Europe_Eastern", "CY", "Asia/Nicosia",
    };
    private static final String[] aliasMap = new String[] {
        "Mexico/BajaNorte", "America/Tijuana",
        "Antarctica/South_Pole", "Antarctica/McMurdo",
        "US/Michigan", "America/Detroit",
        "America/Porto_Acre", "America/Rio_Branco",
        "US/Alaska", "America/Anchorage",
        "Asia/Ujung_Pandang", "Asia/Makassar",
        "Canada/Atlantic", "America/Halifax",
        "W-SU", "Europe/Moscow",
        "Kwajalein", "Pacific/Kwajalein",
        "Europe/Bratislava", "Europe/Prague",
        "Canada/Central", "America/Winnipeg",
        "Canada/Mountain", "America/Edmonton",
        "Iceland", "Atlantic/Reykjavik",
        "Asia/Ulan_Bator", "Asia/Ulaanbaatar",
        "UTC", "Etc/UTC",
        "Europe/Guernsey", "Europe/London",
        "Singapore", "Asia/Singapore",
        "Atlantic/Faeroe", "Atlantic/Faroe",
        "Greenwich", "Etc/GMT",
        "America/Fort_Wayne", "America/Indiana/Indianapolis",
        "Etc/Universal", "Etc/UTC",
        "Chile/EasterIsland", "Pacific/Easter",
        "Pacific/Samoa", "Pacific/Pago_Pago",
        "Europe/Nicosia", "Asia/Nicosia",
        "Etc/Zulu", "Etc/UTC",
        "Asia/Ashkhabad", "Asia/Ashgabat",
        "America/Louisville", "America/Kentucky/Louisville",
        "Australia/North", "Australia/Darwin",
        "America/Atka", "America/Adak",
        "America/Marigot", "America/Guadeloupe",
        "Brazil/DeNoronha", "America/Noronha",
        "Turkey", "Europe/Istanbul",
        "Zulu", "Etc/UTC",
        "Europe/Vatican", "Europe/Rome",
        "Israel", "Asia/Jerusalem",
        "America/Rosario", "America/Argentina/Cordoba",
        "Jamaica", "America/Jamaica",
        "Asia/Katmandu", "Asia/Kathmandu",
        "Canada/East-Saskatchewan", "America/Regina",
        "ROK", "Asia/Seoul",
        "Asia/Macao", "Asia/Macau",
        "Australia/South", "Australia/Adelaide",
        "US/Arizona", "America/Phoenix",
        "Australia/Yancowinna", "Australia/Broken_Hill",
        "Canada/Pacific", "America/Vancouver",
        "Libya", "Africa/Tripoli",
        "Japan", "Asia/Tokyo",
        "Arctic/Longyearbyen", "Europe/Oslo",
        "Africa/Timbuktu", "Africa/Bamako",
        "America/Indianapolis", "America/Indiana/Indianapolis",
        "Etc/Greenwich", "Etc/GMT",
        "Australia/ACT", "Australia/Sydney",
        "GMT", "Etc/GMT",
        "Mexico/BajaSur", "America/Mazatlan",
        "Cuba", "America/Havana",
        "Brazil/West", "America/Manaus",
        "Asia/Saigon", "Asia/Ho_Chi_Minh",
        "America/Jujuy", "America/Argentina/Jujuy",
        "Australia/Victoria", "Australia/Melbourne",
        "America/Catamarca", "America/Argentina/Catamarca",
        "America/Ensenada", "America/Tijuana",
        "Europe/San_Marino", "Europe/Rome",
        "Europe/Isle_of_Man", "Europe/London",
        "Mexico/General", "America/Mexico_City",
        "US/Hawaii", "Pacific/Honolulu",
        "Europe/Mariehamn", "Europe/Helsinki",
        "US/Indiana-Starke", "America/Indiana/Knox",
        "Australia/NSW", "Australia/Sydney",
        "Australia/West", "Australia/Perth",
        "Brazil/Acre", "America/Rio_Branco",
        "Australia/Tasmania", "Australia/Hobart",
        "Atlantic/Jan_Mayen", "Europe/Oslo",
        "America/Buenos_Aires", "America/Argentina/Buenos_Aires",
        "Europe/Jersey", "Europe/London",
        "Brazil/East", "America/Sao_Paulo",
        "America/Virgin", "America/St_Thomas",
        "Navajo", "America/Denver",
        "GB", "Europe/London",
        "Poland", "Europe/Warsaw",
        "Pacific/Yap", "Pacific/Chuuk",
        "America/Argentina/ComodRivadavia", "America/Argentina/Catamarca",
        "Asia/Calcutta", "Asia/Kolkata",
        "America/Mendoza", "America/Argentina/Mendoza",
        "Universal", "Etc/UTC",
        "Australia/Queensland", "Australia/Brisbane",
        "Asia/Dacca", "Asia/Dhaka",
        "US/Pacific", "America/Los_Angeles",
        "Asia/Chungking", "Asia/Chongqing",
        "Pacific/Truk", "Pacific/Chuuk",
        "ROC", "Asia/Taipei",
        "US/Aleutian", "America/Adak",
        "Pacific/Ponape", "Pacific/Pohnpei",
        "Canada/Yukon", "America/Whitehorse",
        "PRC", "Asia/Shanghai",
        "Africa/Asmera", "Africa/Asmara",
        "GB-Eire", "Europe/London",
        "America/St_Barthelemy", "America/Guadeloupe",
        "US/Central", "America/Chicago",
        "Egypt", "Africa/Cairo",
        "Chile/Continental", "America/Santiago",
        "Portugal", "Europe/Lisbon",
        "Europe/Tiraspol", "Europe/Chisinau",
        "America/Coral_Harbour", "America/Atikokan",
        "Europe/Belfast", "Europe/London",
        "America/Cordoba", "America/Argentina/Cordoba",
        "America/Shiprock", "America/Denver",
        "NZ-CHAT", "Pacific/Chatham",
        "Eire", "Europe/Dublin",
        "US/East-Indiana", "America/Indiana/Indianapolis",
        "Australia/Canberra", "Australia/Sydney",
        "Canada/Newfoundland", "America/St_Johns",
        "UCT", "Etc/UCT",
        "Australia/LHI", "Australia/Lord_Howe",
        "Iran", "Asia/Tehran",
        "US/Eastern", "America/New_York",
        "Canada/Eastern", "America/Toronto",
        "US/Samoa", "Pacific/Pago_Pago",
        "America/Knox_IN", "America/Indiana/Knox",
        "Canada/Saskatchewan", "America/Regina",
        "Asia/Thimbu", "Asia/Thimphu",
        "US/Mountain", "America/Denver",
        "NZ", "Pacific/Auckland",
        "Asia/Tel_Aviv", "Asia/Jerusalem",
        "Hongkong", "Asia/Hong_Kong",
        "Asia/Rangoon", "Asia/Yangon",
    };

    private static final Map<String, String> zidToMzone = new HashMap<>();
    private static final Map<String, String> mzoneToZid = new HashMap<>();
    private static final Map<String, Map<String, String>> mzoneToZidL = new HashMap<>();
    private static final Map<String, String> aliases = new HashMap<>();

    static {
        for (int i = 0; i < zidMap.length; i += 3) {
            zidToMzone.put(zidMap[i], zidMap[i + 1]);
            mzoneToZid.put(zidMap[i + 1], zidMap[i + 2]);
        }

        for (int i = 0; i < mzoneMap.length; i += 3) {
            String mzone = mzoneMap[i];
            Map<String, String> map = mzoneToZidL.get(mzone);
            if (map == null) {
                map = new HashMap<>();
                mzoneToZidL.put(mzone, map);
            }
            map.put(mzoneMap[i + 1], mzoneMap[i + 2]);
        }

        for (int i = 0; i < aliasMap.length; i += 2) {
            aliases.put(aliasMap[i], aliasMap[i + 1]);
        }
    }
}
