/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * version 2 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "Parser_p.h"

#include "Debug.h"

#include "Compiler_p.h"
#include "Lexer_p.h"

#include <GTLCore/AST/Statement.h>
#include <GTLCore/AST/Tree.h>
#include <GTLCore/TypeManager.h>

namespace AST = GTLCore::AST;
using namespace OpenShiva;

struct Parser::Private {
  GTLCore::String kernelName;
  GTLCore::AST::Tree* tree;
};

Parser::Parser( Compiler* _compiler , Lexer* _lexer) : ParserBase(_compiler, _lexer), d(new Private)
{
  d->tree = 0;
}

Parser::~Parser()
{
  delete d;
}

GTLCore::AST::Tree* Parser::parse()
{
  SHIVA_DEBUG("Parsing with ParserNG");
  SHIVA_ASSERT( not d->tree );
  d->tree = new GTLCore::AST::Tree();
  startContext();
  getNextToken();
  if(isOfType( currentToken(), GTLCore::Token::KERNEL ) )
  {
    getNextToken();
    if( isOfType( currentToken(), GTLCore::Token::IDENTIFIER ) )
    {
      d->kernelName = currentToken().string;
      getNextToken();
      if( isOfType( currentToken(), GTLCore::Token::STARTBRACE ) )
      {
        getNextToken();
        parseKernelBody();
        isOfType( currentToken(), GTLCore::Token::ENDBRACE );
        getNextToken();
      }
    }
  }
  
  return d->tree;
}

void Parser::parseKernelBody()
{
  while(true)
  {
    switch(currentToken().type)
    {
      case GTLCore::Token::END_OF_FILE:
      case GTLCore::Token::ENDBRACE:
        return;
      case GTLCore::Token::STRUCT:
        parseStructDefinition();
        break;
      default:
      {
        if( currentToken().isFunctionType() )
        {
          parseFunction();
        } else {
          SHIVA_DEBUG("unexpected " << GTLCore::Token::typeToString( currentToken().type ) );
          reportUnexpected( currentToken() );
          getNextToken();
        }
      }
    }
  }
}

GTLCore::AST::Tree* Parser::tree()
{
  return d->tree;
}

AST::Statement* Parser::parseStatement()
{
  if( isType(currentToken()) )
  {
    return parseVariableDeclaration();
  }
  switch( currentToken().type )
  {
    case GTLCore::Token::CONST:
      return parseVariableDeclaration();
    case GTLCore::Token::STRUCT:
    {
      parseStructDefinition();
      return new AST::DummyStatement();
    }
    case GTLCore::Token::STARTBRACKET:
    case GTLCore::Token::IDENTIFIER:
      return parseExpressionStatement();
    case GTLCore::Token::IF:
      return parseIfStatement();
    case GTLCore::Token::WHILE:
      return parseWhileStatement();
    case GTLCore::Token::FOR:
      return parseForStatement();
    case GTLCore::Token::RETURN:
      return parseReturnStatement();
    case GTLCore::Token::STARTBRACE:
    {
      startContext();
      AST::Statement* statement = parseStatementList();
      endContext();
      return statement;
    }
    case GTLCore::Token::PRINT:
    {
      return parsePrintStatement();
    }
    default:
      if( currentToken().isUnaryOperator() or currentToken().isConstant() )
      {
        return parseExpressionStatement();
      }
      SHIVA_DEBUG("unexpected");
      reportUnexpected( currentToken() );
      getNextToken();
      return 0;
  }
}

const GTLCore::Type* Parser::parseType()
{
  switch( currentToken().type )
  {
    case GTLCore::Token::BOOL2:
      getNextToken();
      return typeManager()->getVector( GTLCore::Type::Boolean, 2 );
    case GTLCore::Token::BOOL3:
    case GTLCore::Token::BOOL4:
      getNextToken();
      return typeManager()->getVector( GTLCore::Type::Boolean, 4 );
    case GTLCore::Token::INT2:
      getNextToken();
      return typeManager()->getVector( GTLCore::Type::Integer32, 2 );
    case GTLCore::Token::INT3:
    case GTLCore::Token::INT4:
      getNextToken();
      return typeManager()->getVector( GTLCore::Type::Integer32, 4 );
    case GTLCore::Token::FLOAT2:
      getNextToken();
      return typeManager()->getVector( GTLCore::Type::Float, 2 );
    case GTLCore::Token::FLOAT3:
    case GTLCore::Token::FLOAT4:
      getNextToken();
      return typeManager()->getVector( GTLCore::Type::Float, 4 );
    default:
      return GTLCore::ParserBase::parseType();
  }
}
