/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * version 2 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#ifndef _OPENSHIVA_KERNEL_H_
#define _OPENSHIVA_KERNEL_H_

#include <list>

namespace GTLCore {
  class ErrorMessage;
  class Region;
  class AbstractImage;
}

#include <GTLCore/String.h>

namespace OpenShiva {
  /**
   * This is the main class in OpenShiva, it is used to compile the kernel, and to apply it on
   * a set of images.
   * 
   * @ingroup OpenShiva
   */
  class Kernel {
    public:
      Kernel(const GTLCore::String& _name, int _channelsNb = 4 );
      ~Kernel();
    private:
      void cleanup();
    public:
      /**
       * @return the name of the kernel
       */
      const GTLCore::String& name() const;
      /**
       * Set the code source of the module.
       */
      void setSource(const GTLCore::String& source, bool _enableHydraCompatibility = false);
      /**
       * Load the module from the given file name.
       */
      void loadFromFile(const GTLCore::String& fileName);
      /**
       * Start the compilation of the module.
       */
      void compile();
      /**
       * @return true if the module was successfully compiled.
       */
      bool isCompiled() const;
      /**
       * @return a string with the content of the compilation error.
       */
      const std::list<GTLCore::ErrorMessage>& compilationErrors() const;
      /**
       * @return a string with the content of the compilation error.
       */
      GTLCore::String compilationErrorsMessage() const;
      /**
       * @return the assembly source code, it's mostly usefull for testing purpose
       */
      GTLCore::String asmSourceCode() const;
    public: // Function call
      void evaluatePixeles( const GTLCore::Region& _region, const std::list< GTLCore::AbstractImage* >& _inputImages, GTLCore::AbstractImage* _outputImage) const;
      /**
       * Run the function called "int runTest()" in the Kernel.
       */
      int runTest() const;
    private:
      struct Private;
      Private * const d;
  };
}

#endif
