/*
 *  Copyright (c) 2007 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * version 2 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "Type.h"
#include "Type_p.h"

#include "Debug.h"

#include <iostream>

#include <llvm/DerivedTypes.h>

using namespace GTLCore;

//-------- Type::StructDataMember -------//

struct Type::StructDataMember::Private
{
  GTLCore::String name;
  const Type* type;
  std::list<int> initialSizes;
};

Type::StructDataMember::StructDataMember(const GTLCore::String& _name, const Type* _type) : d(new Private)
{
  d->name = _name;
  d->type = _type;
  GTL_ASSERT( _type->dataType() != Type::ARRAY);
}

Type::StructDataMember::StructDataMember(const GTLCore::String& _name, const Type* _type, std::list<int> _initialSizes) : d(new Private)
{
  d->name = _name;
  d->type = _type;
  d->initialSizes = _initialSizes;
  GTL_ASSERT( d->initialSizes.empty() or ( not d->initialSizes.empty() and _type->dataType() == Type::ARRAY) );
}


Type::StructDataMember::StructDataMember(const StructDataMember& _rhs) : d(new Private(*_rhs.d))
{
}

Type::StructDataMember& Type::StructDataMember::operator=(const StructDataMember& _rhs)
{
  *d = *_rhs.d;
  return *this;
}

Type::StructDataMember::~StructDataMember()
{
  delete d;
}

const GTLCore::String& Type::StructDataMember::name() const
{
  return d->name;
}

const Type* Type::StructDataMember::type() const
{
  return d->type;
}

const std::list<int>& Type::StructDataMember::initialSizes() const
{
  return d->initialSizes;
}

//-------- Type -------//

const Type* Type::Undefined = new Type(Type::UNDEFINED);
const Type* Type::Boolean = new Type(Type::BOOLEAN);
const Type* Type::Integer8 = new Type(Type::INTEGER8);
const Type* Type::UnsignedInteger8 = new Type(Type::UNSIGNED_INTEGER8);
const Type* Type::Integer16 = new Type(Type::INTEGER16);
const Type* Type::UnsignedInteger16 = new Type(Type::UNSIGNED_INTEGER16);
const Type* Type::Integer32 = new Type(Type::INTEGER32);
const Type* Type::UnsignedInteger32 = new Type(Type::UNSIGNED_INTEGER32);
const Type* Type::Double = new Type(Type::DOUBLE);
const Type* Type::Float = new Type(Type::FLOAT);
const Type* Type::Half = new Type(Type::HALF);
const Type* Type::Void = new Type(Type::VOID);
const Type* Type::Pointer = new Type(Type::POINTER);

Type::Type(DataType _dataType) : d(new Private(_dataType))
{
  GTL_DEBUG("Create type for " << _dataType);
  GTL_ASSERT(_dataType != STRUCTURE and _dataType != ARRAY and _dataType != VECTOR );
  if(_dataType == STRUCTURE or _dataType == ARRAY)
  {
    d->dataType = UNDEFINED;
  }
  init(_dataType);
}

Type::Type(const GTLCore::String& _structName, const std::vector<StructDataMember>& _members) : d(new Private)
{
  init(STRUCTURE);
  d->structName = _structName;
  d->structDataMembers = new std::vector<StructDataMember>( _members );
  std::vector< const llvm::Type * > types;
  for(std::vector<StructDataMember>::iterator it = d->structDataMembers->begin();
      it != d->structDataMembers->end(); ++it)
  {
    types.push_back(it->type()->d->type());
  }
  d->setType(llvm::StructType::get(types));
}

Type::Type(const Type* _arrayType) : d(new Private)
{
  init(ARRAY);
  d->arrayType = _arrayType;
  
  std::vector< const llvm::Type * > types;
  types.push_back(llvm::Type::Int32Ty);
  types.push_back( llvm::PointerType::get( _arrayType->d->type(), 0));
  d->setType( llvm::StructType::get(types));
}

Type::Type(int _size, const Type* _vectorType) : d(new Private)
{
  init(VECTOR);
  d->arrayType = _vectorType;
  d->vectorSize = _size;
  d->setType( llvm::VectorType::get( _vectorType->d->type(), d->vectorSize ) );
}

void Type::init(DataType _dataType)
{
  d->dataType = _dataType;
  d->arrayType = 0;
  d->structDataMembers = 0;
  d->structFunctionMembers = 0;
  d->structPrivateFunctionMembers = 0;
}

Type::Type(const Type& _rhs) : d(new Private(*_rhs.d))
{
  GTL_ABORT("Can't copy type.");
}

Type& Type::operator=(const Type& _rhs)
{
  GTL_ABORT("Can affect a type to an other type.");
}

Type::~Type()
{
  delete d;
}

Type::DataType Type::dataType() const
{
  return d->dataType;
}

const Type* Type::embeddedType() const
{
  return d->arrayType;
}

const Type* Type::arrayType() const
{
  return d->arrayType;
}

const GTLCore::String& Type::structName() const
{
  return d->structName;
}

const std::vector<Type::StructDataMember>* Type::structDataMembers() const
{
  return d->structDataMembers;
}

const std::vector<Type::StructFunctionMember>* Type::structFunctionMembers() const
{
  return d->structFunctionMembers;
}

int Type::bitsSize() const
{
  switch( dataType() )
  {
    case Type::BOOLEAN:
      return 1;
    case Type::INTEGER8:
    case Type::UNSIGNED_INTEGER8:
      return 8;
    case Type::INTEGER16:
    case Type::UNSIGNED_INTEGER16:
      return 16;
    case Type::INTEGER32:
    case Type::UNSIGNED_INTEGER32:
      return 32;
    case Type::HALF:
      return 16;
    case Type::FLOAT:
      return 32;
    case Type::VOID:
      return 0;
    case Type::VECTOR:
      return d->vectorSize * embeddedType()->bitsSize();
    default:
      return -1;
      break;
  }

}

bool Type::isSigned() const
{
  return not isUnsigned();
}

bool Type::isUnsigned() const
{
  switch( dataType() )
  {
    case Type::UNSIGNED_INTEGER8:
    case Type::UNSIGNED_INTEGER16:
    case Type::UNSIGNED_INTEGER32:
      return true;
    case Type::VECTOR:
      return embeddedType()->isUnsigned();
    default:
      return false;
  }
}

int Type::vectorSize() const
{
  GTL_ASSERT( dataType() == Type::VECTOR );
  return d->vectorSize;
}

bool Type::isNumber() const
{
  return d->dataType == BOOLEAN or d->dataType == INTEGER8 or d->dataType == UNSIGNED_INTEGER8 or d->dataType == INTEGER16 or d->dataType == UNSIGNED_INTEGER16 or d->dataType == INTEGER32 or d->dataType == UNSIGNED_INTEGER32 or d->dataType == DOUBLE or d->dataType == FLOAT or d->dataType == HALF;
}
