/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * version 2 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#ifndef _OPENCTL_TOKEN_H_
#define _OPENCTL_TOKEN_H_

#include <GTLCore/String.h>
#include <sstream>

namespace GTLCore {
  /**
    * This structure holds information concerning a token, like position (line / column),
    * type or value (for string, integer, float or identifier).
    */
  struct Token {
    /**
      * List of possible token type
      */
    enum Type {
    // Not really token
      UNFINISHED_STRING = -4,
      UNFINISHED_COMMENT = -3,
      END_OF_FILE = -2,
      UNKNOWN = - 1,
    // Special characters
      SEMI = 0, ///< ;
      COLONCOLON, ///< :
      COMA, ///< ,
      DOT, ///< .
      STARTBRACE, ///< {
      ENDBRACE, ///< }
      STARTBRACKET, ///< (
      ENDBRACKET, ///< )
      STARTBOXBRACKET, ///< [
      ENDBOXBRACKET, ///< ]
      EQUAL, ///< =
      EQUALEQUAL, ///< ==
      DIFFERENT, ///< !=
      AND, ///< and &&
      OR, ///< or ||
      BITAND, ///< &
      BITXOR, ///< ^
      BITOR, ///< |
      INFERIOR, ///< <
      INFERIOREQUAL, ///< <=
      SUPPERIOR, ///< >
      SUPPERIOREQUAL, ///< >=
      LEFTSHIFT, ///< <<
      RIGHTSHIFT, ///< >>
      PLUS, ///< +
      PLUSPLUS, ///< ++
      MINUS, ///< -
      MINUSMINUS, ///< --
      MULTIPLY, ///< *
      DIVIDE, ///< /
      MODULO, ///< %
      TILDE, ///< ~
      NOT, ///< not !
      // Constants
      FLOAT_CONSTANT,
      INTEGER_CONSTANT,
      STRING_CONSTANT,
      IDENTIFIER,
      // Keywords,
      BOOL,
      BOOL2,
      BOOL3,
      BOOL4,
      CONST,
      FLOAT,
      FLOAT2,
      FLOAT3,
      FLOAT4,
      ELSE,
      FOR,
      IF,
      IMPORT,
      INT,
      INT2,
      INT3,
      INT4,
      KERNEL,
      LONG,
      RETURN,
      SHORT,
      SIGNED,
      SIZE,
      STRUCT,
      UNSIGNED,
      VOID,
      WHILE,
      NAMESPACE,
      CTLVERSION,
      HALF,
      INPUT,
      OUTPUT,
      PRINT,
      UNIFORM,
      VARYING,
      TTRUE,
      TFALSE,
    };
    /// type of the token
    Type type;
    /// line of the token
    int line;
    /// Column of the token
    int column;
    /// String or identifier name
    GTLCore::String string;
    /// Integer value
    int i;
    /// Floating point value
    float f;
    Token();
    /**
      * Creates a token of the given type
      */
    Token(Type _type, int _line, int _column);
    /**
      * Creates an identifier or a string constant
      */
    Token(Type _type, const GTLCore::String& _string, int _line, int _column);
    /**
      * Creates an integer constant
      */
    Token(int _i, int _line, int _column);
    /**
      * Creates a floating point constant
      */
    Token(float _f, int _line, int _column);
    bool isExpressionTerminal();
    bool isNativeType() const;
    bool isFunctionType() const;
    bool isConstant() const;
    bool isBinaryOperator() const;
    int binaryOperationPriority() const;
    bool isUnaryOperator() const;
    bool isOperator() const;
    bool isPrimary() const;
    static GTLCore::String typeToString(Type );
  };

};


#endif
