/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * version 2 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#ifndef _GTLCORE_STRING_H_
#define _GTLCORE_STRING_H_

#include <string>
#include <list>

namespace GTLCore {
  /**
   * String class which provides convenient function on top of \ref std::string .
   * @ingroup GTLCore
   */
  class String : public std::string {
    public:
      /**
       * Construct an empty string.
       */
      String();
      /**
       * Construct a \ref String from a single character.
       */
      String(char );
      /**
       * Construct a \ref String from a C string.
       */
      String(const char* );
      /**
       * Construct a \ref String from a C++ string.
       */
      String(const std::string& );
      /**
       * Construct a \ref String from an other \ref String .
       */
      String(const String& );
    public:
      /**
       * Append a C string to the current \ref String, and return it the
       * result.
       */
      String& append(const char* );
      /**
       * Append a C++ string to the current \ref String, and return the
       * result.
       */
      String& append(const std::string& );
      /**
       * Append a \ref String to the current \ref String, and return the
       * result.
       */
      String& append(const String& );
    public:
      /**
       * Attempt to convert the string to an integer.
       */
      int toInt() const;
      /**
       * Attempt to convert the string to a float.
       */
      float toFloat() const;
      /**
       * @return the n first characters
       */
      String head(int n) const;
      /**
       * @return the n last characters
       */
      String tail(int n) const;
      /**
       * @return a lower case version of the string.
       */
      String toLower() const;
      /**
       * @return a upper case version of the string.
       */
      String toUpper() const;
      /**
       * Split the \ref String along the given separators
       * @param _separators list of separators given in a string (one character for one separator)
       * @return a list of \ref String
       */
      std::list<String> split( const String& _separators, bool _allowEmpty = false) const;
      /**
       * Split the \ref String along the given separators
       * @param _separators the list of separators
       * @return a list of \ref String
       */
      std::list<String> split( const std::list<String>& _separators, bool _allowEmpty = false) const;
      /**
       * @return true if this \ref String starts with \a _sw .
       */
      bool startWith(const GTLCore::String& _sw) const;
    public:
      String& operator=(char c);
    public:
      /**
       * @return a \ref String containing the number
       */
      static String number(int );
      /**
       * @return a \ref String containing the number
       */
      static String number(float );
  };
  
}

#endif
