/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * version 2 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "PixelDescription.h"

#include "Debug.h"
#include "Type.h"

using namespace GTLCore;


struct PixelDescription::Private {
  std::vector<const Type*> channelsType;
  int bitsSize;
};

PixelDescription::PixelDescription( const Type* channelType_, int channels_) : d(new Private)
{
  GTL_ASSERT( channelType_->bitsSize() > 0 );
  d->bitsSize = channels_ * channelType_->bitsSize();
  d->channelsType.reserve(channels_);
  for(int i = 0; i< channels_; ++i)
  {
    d->channelsType.push_back( channelType_ );
  }
}

PixelDescription::PixelDescription( const std::vector<const Type* >& channelsType_) : d(new Private)
{
  d->channelsType = channelsType_;
  d->bitsSize = 0;
  for( std::vector<const Type*>::iterator it = d->channelsType.begin();
       it != d->channelsType.end(); ++it)
  {
    GTL_ASSERT( (*it)->bitsSize() > 0 );
    d->bitsSize += (*it)->bitsSize();
  }
}

PixelDescription::PixelDescription( const PixelDescription& rhs) : d(new Private(*rhs.d))
{
}

PixelDescription& PixelDescription::operator=(const PixelDescription& rhs)
{
  *d = *rhs.d;
  return *this;
}

PixelDescription::~PixelDescription()
{
  delete d;
}

const std::vector<const Type*>& PixelDescription::channelTypes() const
{
  return d->channelsType;
}

int PixelDescription::channels() const
{
  return d->channelsType.size();
}

bool PixelDescription::sameTypeChannels() const
{
  const Type* _type = *d->channelsType.begin();
  for( std::vector<const Type*>::const_iterator it = d->channelsType.begin();
       it != d->channelsType.end(); ++it)
  {
    if( _type != *it )
      return false;
  }
  return true;
}

int PixelDescription::bitsSize() const
{
  return d->bitsSize;
}

bool PixelDescription::operator<( const PixelDescription& _rhs ) const
{
  if( d->bitsSize == _rhs.d->bitsSize )
  {
    if( d->channelsType.size() == _rhs.d->channelsType.size() )
    {
      for(unsigned int i = 0; i < d->channelsType.size(); ++i)
      {
        if( d->channelsType[i] != _rhs.d->channelsType[i] )
        {
          return d->channelsType[i] < _rhs.d->channelsType[i];
        }
      }
      return false;
    } else {
      return (d->channelsType.size() < _rhs.d->channelsType.size());
    }
    return false;
  } else
  {
    return (d->bitsSize < _rhs.d->bitsSize );
  }
}

bool PixelDescription::operator==( const PixelDescription& _rhs ) const
{
  if( d->channelsType.size() != _rhs.d->channelsType.size()
      and d->bitsSize != _rhs.d->bitsSize )
  {
    return false;
  } else {
    for(unsigned int i = 0; i < d->channelsType.size(); ++i)
    {
      if( d->channelsType[i] != _rhs.d->channelsType[i] )
      {
        return false;
      }
    }
    return true;
  }
}

bool PixelDescription::operator!=( const PixelDescription& _rhs ) const
{
  return not ( *this == _rhs );
}
