/*
 *  Copyright (c) 2007-2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * version 2 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "CompilerBase_p.h"

#include <map>

#include "ConvertCenter_p.h"
#include "Debug.h"
#include "ErrorMessage.h"
#include "ScopedName.h"
#include "Function.h"

using namespace GTLCore;

struct CompilerBase::Private {
  GTLCore::String moduleName;
  std::list<ErrorMessage> errorMessages;
  std::map< GTLCore::ScopedName, GTLCore::Function* > functions;
  ConvertCenter* convertCenter;
};

CompilerBase::CompilerBase() : d(new Private)
{
  d->convertCenter = new ConvertCenter;
}
CompilerBase::~CompilerBase()
{
  delete d->convertCenter;
  delete d;
}

void CompilerBase::appendError( const GTLCore::ErrorMessage& _msg)
{
  GTLCore::ErrorMessage msg = _msg;
  if( msg.fileName() == "")
  {
    msg.setFileName( d->moduleName );
  }
  GTL_DEBUG( msg.fileName() << " at " << msg.line() << ": " << msg.errorMessage() );
  d->errorMessages.push_back(msg);
}

void CompilerBase::appendError( GTLCore::ErrorMessage* msg)
{
  GTL_ASSERT(msg);
  if( msg->fileName() == "")
  {
    GTL_ASSERT(d);
    msg->setFileName( d->moduleName );
  }
  GTL_DEBUG( msg->fileName() << " at " << msg->line() << ": " << msg->errorMessage() );
  d->errorMessages.push_back(*msg);
  delete msg;
}

void CompilerBase::appendErrors( const std::list<GTLCore::ErrorMessage>& _msgs)
{
  d->errorMessages.insert( d->errorMessages.begin(), _msgs.begin(), _msgs.end() );
}

const std::list<GTLCore::ErrorMessage>& CompilerBase::errorMessages() const
{
  return d->errorMessages;
}

void CompilerBase::setModuleName( const GTLCore::String& _moduleName)
{
  d->moduleName = _moduleName;
}
const GTLCore::String& CompilerBase::moduleName() const
{
  return d->moduleName;
}

GTLCore::Function* CompilerBase::function( const GTLCore::ScopedName&  name)
{
  std::map< GTLCore::ScopedName, GTLCore::Function* >::iterator it = d->functions.find( name );
  if( it == d->functions.end() )
  {
    return 0;
  } else {
    return it->second;
  }
}

bool CompilerBase::declareFunction( const GTLCore::ScopedName& name, GTLCore::Function* func)
{
  std::map< GTLCore::ScopedName, GTLCore::Function* >::iterator it = d->functions.find( name );
  if( it == d->functions.end() )
  {
    d->functions[name] = func;
    return true;
  } else {
    return false;
  }
}

ConvertCenter* CompilerBase::convertCenter()
{
  return d->convertCenter;
}

