/*
 *  Copyright (c) 2007-2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * version 2 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "CodeGenerator_p.h"

#include <llvm/Constants.h>
#include <llvm/Function.h>
#include <llvm/Instructions.h>
#include <llvm/Module.h>
#include <llvm/Value.h>

#include "Debug.h"
#include "ExpressionResult_p.h"
#include "Function_p.h"
#include "GenerationContext_p.h"
#include "Parameter.h"
#include "Type.h"
#include "Type_p.h"
#include "Value.h"
#include "VariableNG_p.h"
#include "ModuleData_p.h"

#include "AST/AccessorExpression.h"

#define UNIFORMIZE_TYPES( _v1_, _v2_) \
  GTL_ASSERT( _v1_.value() ); \
  GTL_ASSERT( _v2_.value() ); \
  GTL_CHECK_EQUAL( lhsType, rhsType); \
  GTLCore::ExpressionResult v1 = _v1_; \
  GTLCore::ExpressionResult v2 = _v2_; \
  GTL_ASSERT( v1.value() ); \
  GTL_ASSERT( v2.value() );
  
#define UNIFORMIZE_VALUE_TYPES( _v1_, _v2_ ) \
  GTL_CHECK_EQUAL( lhsType, rhsType ); \
  llvm::Value* v1 = _v1_; \
  llvm::Value* v2 = _v2_; \
  GTL_ASSERT( v1 ); \
  GTL_ASSERT( v2 );

#define UNIFORMIZE_CONSTANT_TYPES( _v1_, _v2_ ) \
  GTL_CHECK_EQUAL( lhsType, rhsType ); \
  llvm::Constant* v1 = _v1_; \
  llvm::Constant* v2 = _v2_; \
  GTL_ASSERT( v1 ); \
  GTL_ASSERT( v2 );

#define MAKE_VALUE_BOOLEAN( _v1_, _v2_ ) \
  llvm::Value* v1 = convertValueTo( currentBlock, _v1_, llvm::IntegerType::get(1)); \
  llvm::Value* v2 = convertValueTo( currentBlock, _v2_, llvm::IntegerType::get(1)); \
  GTL_ASSERT( v1 ); \
  GTL_ASSERT( v2 );

#define MAKE_CONSTANT_BOOLEAN( _v1_, _v2_ ) \
  llvm::Constant* v1 = convertConstantTo( _v1_, llvm::IntegerType::get(1)); \
  llvm::Constant* v2 = convertConstantTo( _v2_, llvm::IntegerType::get(1)); \
  GTL_ASSERT( v1 ); \
  GTL_ASSERT( v2 );

using namespace GTLCore;

struct CodeGenerator::Private {
  ModuleData* module;
};

CodeGenerator::CodeGenerator(ModuleData* module) : d(new Private)
{
  d->module = module;
}

CodeGenerator::~CodeGenerator()
{
  delete d;
}

llvm::Constant* CodeGenerator::integerToConstant(int v)
{
  return llvm::ConstantInt::get(llvm::Type::Int32Ty, v);
}

llvm::Constant* CodeGenerator::boolToConstant(bool v)
{
  return llvm::ConstantInt::get(llvm::IntegerType::get(1), v);
}

llvm::Constant* CodeGenerator::floatToConstant(float v)
{
  return llvm::ConstantFP::get(llvm::APFloat(v));
}


llvm::Constant* CodeGenerator::valueToConstant( const GTLCore::Value& v)
{
  if( v.type() == GTLCore::Type::Integer32 or v.type() == GTLCore::Type::UnsignedInteger32 )
  {
    return integerToConstant( v.asInt32() );
  } else if( v.type() == GTLCore::Type::Float ) {
    return floatToConstant( v.asFloat() );
  } else if( v.type() == GTLCore::Type::Boolean ) {
    return floatToConstant( v.asBoolean() );
  }
  return 0;
}

llvm::Value* CodeGenerator::convertPointerToCharP(llvm::BasicBlock* _currentBlock, llvm::Value* _value)
{
  return convertPointerTo( _currentBlock, _value, llvm::IntegerType::get(8) );
}

llvm::Value* CodeGenerator::convertPointerTo(llvm::BasicBlock* _currentBlock, llvm::Value* _value, const llvm::Type* _type)
{
  return new llvm::BitCastInst( _value, llvm::PointerType::get( _type, 0 ), "", _currentBlock );
}

llvm::Value* CodeGenerator::convertValueTo(llvm::BasicBlock* _currentBlock, llvm::Value* _value, const Type* _valueType, const Type* _targetType)
{
  GTL_DEBUG("Convert value " << *_value << " from " << *_valueType << " to " << *_targetType);
  llvm::Constant* _constant = dynamic_cast<llvm::Constant*>(_value);
  if(_constant)
  {
    return convertConstantTo( _constant, _valueType, _targetType);
  }
  GTL_ASSERT(_value);
  GTL_ASSERT(_valueType);
  GTL_ASSERT(_targetType);
  if(_value->getType() == _targetType->d->type() )
  { // Don't cast the value as it has allready the correct type
    return _value;
  }
  if( _targetType->dataType() == Type::VECTOR and _valueType->dataType() != Type::VECTOR )
  {
    GTL_DEBUG("Convert value to a vector");
    // Create a vector
    llvm::Value* resultLoad = llvm::ConstantVector::get( static_cast<const llvm::VectorType*>(_targetType->d->type()), std::vector<llvm::Constant*>() );
    llvm::Value* vecElt = convertValueTo( _currentBlock, _value, _valueType, _targetType->embeddedType() );
    // Affect the same value to each element of the vector
    for(int i = 0; i < _targetType->vectorSize(); ++i)
    {
      resultLoad = llvm::InsertElementInst::Create( resultLoad, vecElt, i, "", _currentBlock );
    }
    GTL_DEBUG("Done");
    return resultLoad;
  }
  GTL_DEBUG("Create cast instruction");
  GTL_DEBUG( *_value << " to "  << *_targetType );
  GTL_ASSERT(_targetType->d->type()->isFirstClassType());
  GTL_ASSERT(_value->getType()->isFirstClassType());
  return llvm::CastInst::create(
                      llvm::CastInst::getCastOpcode(_value, _valueType->isSigned(), _targetType->d->type(), _targetType->isSigned()), _value, _targetType->d->type(), "", _currentBlock );
}

llvm::Constant* CodeGenerator::convertConstantTo(llvm::Constant* _constant, const Type* _constantType, const Type* _targetType)
{
  GTL_DEBUG( *_constant->getType() << " " << *_targetType->d->type() );
  if(_constant->getType() == _targetType->d->type() )
  { // Don't cast the value as it has allready the correct type
    return _constant;
  }
  if( _targetType->dataType() == Type::VECTOR and _constantType->dataType() != Type::VECTOR )
  {
    llvm::Constant* vecElt = convertConstantTo( _constant, _constantType, _targetType->embeddedType() );
    // Affect the same value to each element of the vector
    std::vector<llvm::Constant*> constants;
    for(int i = 0; i < _targetType->vectorSize(); ++i)
    {
      constants.push_back( vecElt );
    }
    // store back
    return llvm::ConstantVector::get( llvm::cast<const llvm::VectorType>(_targetType->d->type()), constants );
  }
  GTL_ASSERT( _targetType->d->type()->isFirstClassType() );
  GTL_ASSERT( _constant->getType()->isFirstClassType() );
  return llvm::ConstantExpr::getCast(
                      (unsigned)llvm::CastInst::getCastOpcode(_constant, _constantType->isSigned(), _targetType->d->type(), _targetType->isSigned()), _constant, _targetType->d->type() );
}



llvm::Value* CodeGenerator::convertToHalf( GenerationContext& _generationContext, llvm::BasicBlock* currentBlock, llvm::Value* value, const GTLCore::Type* _valueType )
{
  std::vector<const llvm::Type*> llvmArguments;
  llvmArguments.push_back( llvm::Type::FloatTy );
  llvm::Function* func = dynamic_cast<llvm::Function*>( _generationContext.llvmModule()->getOrInsertFunction( "_Z10float2halff",
      llvm::FunctionType::get( llvm::Type::Int16Ty , llvmArguments, false ) ) );
  std::vector<llvm::Value*> convertedParams;
  convertedParams.push_back( convertValueTo( currentBlock, value, _valueType, GTLCore::Type::Float ) );
  return llvm::CallInst::Create( func, convertedParams.begin(), convertedParams.end(), "", currentBlock );
}

llvm::Value* CodeGenerator::convertFromHalf( GenerationContext& _generationContext, llvm::BasicBlock* currentBlock, llvm::Value* value)
{
  std::vector<const llvm::Type*> llvmArguments;
  llvmArguments.push_back( llvm::Type::Int16Ty );
  llvm::Function* func = dynamic_cast<llvm::Function*>( _generationContext.llvmModule()->getOrInsertFunction( "_Z10half2floatt",
      llvm::FunctionType::get( llvm::Type::FloatTy , llvmArguments, false ) ) );
  std::vector<llvm::Value*> convertedParams;
  convertedParams.push_back( convertValueTo( currentBlock, value, GTLCore::Type::Half, GTLCore::Type::Integer16 ) );
  return llvm::CallInst::Create( func, convertedParams.begin(), convertedParams.end(), "", currentBlock );
  
}

llvm::Value* CodeGenerator::vectorValueAt( llvm::BasicBlock* _currentBlock, llvm::Value* _vector, llvm::Value* _index)
{
  return new llvm::ExtractElementInst( _vector, _index, "", _currentBlock );
}

llvm::Value* CodeGenerator::createRound( llvm::BasicBlock* _currentBlock, llvm::Value* _val )
{
  _val = GTLCore::CodeGenerator::createAdditionExpression( _currentBlock, _val, GTLCore::Type::Float,
                          GTLCore::CodeGenerator::floatToConstant( 0.5 ), GTLCore::Type::Float );
  return GTLCore::CodeGenerator::convertValueTo( _currentBlock, _val,
        GTLCore::Type::Float, GTLCore::Type::Integer32 );
}


void CodeGenerator::createIfStatement( llvm::BasicBlock* before, llvm::Value* test, const Type* testType, llvm::BasicBlock* firstAction, llvm::BasicBlock* lastAction, llvm::BasicBlock* after)
{
  if( not lastAction->getTerminator() )
  {
    llvm::BranchInst::Create( after, lastAction );
  }
  llvm::BranchInst::Create( firstAction, after, convertValueTo( before, test, testType, GTLCore::Type::Boolean ), before);
}

void CodeGenerator::createIfElseStatement( llvm::BasicBlock* before, llvm::Value* test, const Type* testType, llvm::BasicBlock* firstAction, llvm::BasicBlock* lastAction, llvm::BasicBlock* firstElseAction, llvm::BasicBlock* lastElseAction, llvm::BasicBlock* after)
{
  if( not lastAction->getTerminator() )
  {
      llvm::BranchInst::Create( after, lastAction );
  }
  llvm::BranchInst::Create( firstAction, firstElseAction, convertValueTo( before, test, testType, GTLCore::Type::Boolean ), before);
  if( not lastElseAction->getTerminator() )
  {
      llvm::BranchInst::Create( after, lastElseAction );
  }
}

void CodeGenerator::createForStatement(llvm::BasicBlock* before, llvm::BasicBlock* test, llvm::Value* testResult, const Type* testType, llvm::BasicBlock* update, llvm::BasicBlock* firstAction, llvm::BasicBlock* lastAction, llvm::BasicBlock* after)
{
  llvm::BranchInst::Create( test, before );
  llvm::BranchInst::Create( firstAction, after, convertValueTo( before, testResult, testType, GTLCore::Type::Boolean ), test );
  llvm::BranchInst::Create( update, lastAction);
  llvm::BranchInst::Create( test, update);
}

llvm::BasicBlock* CodeGenerator::createIterationForStatement( GenerationContext& generationContext, llvm::BasicBlock* before, GTLCore::VariableNG* variable, llvm::Value* maxValue, const Type* maxValueType, llvm::BasicBlock* firstAction, llvm::BasicBlock* lastAction)
{
  // i < size
  llvm::BasicBlock* forTestBlock = llvm::BasicBlock::Create("forTestBlock");
  generationContext.llvmFunction()->getBasicBlockList().push_back( forTestBlock);
  llvm::Value* forTest = createStrictInferiorExpression(forTestBlock, variable->get( generationContext, forTestBlock ), variable->type(), maxValue, maxValueType);
  GTL_ASSERT(forTest);
  // i += pixelSize
  llvm::BasicBlock* updateBlock = llvm::BasicBlock::Create("updateBlock");
  generationContext.llvmFunction()->getBasicBlockList().push_back( updateBlock);
  createIncrementExpression( generationContext, updateBlock, variable );
  // Exit block
  llvm::BasicBlock* finBlock = llvm::BasicBlock::Create("finBlock");
  generationContext.llvmFunction()->getBasicBlockList().push_back( finBlock);
  // Create for statement
  createForStatement(before, forTestBlock, forTest, Type::Boolean, updateBlock, firstAction, lastAction, finBlock);
  return finBlock;
}

void CodeGenerator::createWhileStatement( llvm::BasicBlock* before, llvm::BasicBlock* test, llvm::Value* testResult, const Type* testType, llvm::BasicBlock* firstAction, llvm::BasicBlock* lastAction,  llvm::BasicBlock* after)
{
  llvm::BranchInst::Create( test, before);
  llvm::BranchInst::Create( firstAction, after, convertValueTo( test, testResult, testType, GTLCore::Type::Boolean ), test);
  llvm::BranchInst::Create( test, lastAction );

}


llvm::Function* CodeGenerator::createFunction( llvm::FunctionType* type, const GTLCore::String& name)
{
  return llvm::Function::Create( type, llvm::GlobalValue::LinkOnceLinkage,
                  name, d->module->llvmModule() );
}

void CodeGenerator::createIncrementExpression( GenerationContext& gc, llvm::BasicBlock* currentBlock, VariableNG* var)
{
  var->set( gc, currentBlock, createAdditionExpression( currentBlock, var->get(gc, currentBlock), var->type(), integerToConstant(1), Type::Integer32), var->type());
}

void CodeGenerator::createIncrementExpression( llvm::BasicBlock* _currentBlock, llvm::Value* _pointer)
{
  new llvm::StoreInst(
      createAdditionExpression( _currentBlock,
                                new llvm::LoadInst( _pointer, "", _currentBlock ),
                                Type::Integer32, integerToConstant(1), Type::Integer32 ),
                      _pointer, "", _currentBlock );
}

void CodeGenerator::createDecrementExpression( llvm::BasicBlock* currentBlock, VariableNG* var)
{
  GenerationContext gc(this, 0, 0, 0);
  var->set(gc, currentBlock, createSubstractionExpression( currentBlock, var->get(gc, currentBlock), var->type(), integerToConstant(1), Type::Integer32), var->type());
}

void CodeGenerator::createDecrementExpression( llvm::BasicBlock* _currentBlock, llvm::Value* _pointer)
{
  new llvm::StoreInst(
      createSubstractionExpression( _currentBlock,
                                new llvm::LoadInst( _pointer, "", _currentBlock ),
                                Type::Integer32, integerToConstant(1), Type::Integer32 ),
                                _pointer, "", _currentBlock );
}

#define CREATE_BINARY_OPERATOR( _creator_, _v1_, _v2_) \
  if( _v1_.isConstant() and _v2_.isConstant() ) \
  { \
    return GTLCore::ExpressionResult( _creator_(_v1_.constant(), lhsType, _v2_.constant(), rhsType ), lhsType); \
  } else { \
    return GTLCore::ExpressionResult( _creator_( currentBlock, _v1_.value(), lhsType, _v2_.value(), rhsType ), lhsType); \
  }

#define CREATE_BINARY_EXPRESSION(_GTL_Function_Name_, _LLVM_Constant_Expr_, _LLVM_Binary_Operator ) \
  llvm::Value* CodeGenerator::_GTL_Function_Name_(llvm::BasicBlock* currentBlock, llvm::Value* lhs, const Type* lhsType, llvm::Value* rhs, const Type* rhsType) \
  { \
    UNIFORMIZE_VALUE_TYPES( lhs, rhs ) \
    return llvm::BinaryOperator::_LLVM_Binary_Operator( v1, v2, "", currentBlock ); \
  } \
  llvm::Constant* CodeGenerator::_GTL_Function_Name_( llvm::Constant* lhs, const Type* lhsType, llvm::Constant* rhs, const Type* rhsType) \
  { \
    UNIFORMIZE_CONSTANT_TYPES( lhs, rhs ) \
    return llvm::ConstantExpr::_LLVM_Constant_Expr_( v1, v2 ); \
  } \
  ExpressionResult CodeGenerator::_GTL_Function_Name_( llvm::BasicBlock* currentBlock, ExpressionResult lhs, ExpressionResult rhs) \
  { \
    const Type* lhsType = lhs.type(); \
    const Type* rhsType = rhs.type(); \
    GTL_DEBUG( "lhs = " << lhs << " lhsType = " << *lhsType << " rhs = " << rhs << " rhsType = " << *rhsType ); \
    UNIFORMIZE_TYPES(lhs, rhs); \
    CREATE_BINARY_OPERATOR(  _GTL_Function_Name_, v1, v2); \
  }

CREATE_BINARY_EXPRESSION(createAdditionExpression, getAdd, createAdd)
CREATE_BINARY_EXPRESSION(createSubstractionExpression, getSub, createSub)
CREATE_BINARY_EXPRESSION(createMultiplicationExpression, getMul, createMul)
CREATE_BINARY_EXPRESSION(createRightShiftExpression, getAShr, createAShr);
CREATE_BINARY_EXPRESSION(createLeftShiftExpression, getShl, createShl);


llvm::Value* CodeGenerator::createDivisionExpression(llvm::BasicBlock* currentBlock, llvm::Value* lhs, const Type* lhsType, llvm::Value* rhs, const Type* rhsType)
{
  UNIFORMIZE_VALUE_TYPES( lhs, rhs );
  if( v1->getType()->isFloatingPoint()
      or ( lhsType->dataType() == GTLCore::Type::VECTOR
           and lhsType->embeddedType()->dataType() == GTLCore::Type::FLOAT ) )
  {
    return llvm::BinaryOperator::createFDiv(v1, v2, "", currentBlock );
  } else {
    return llvm::BinaryOperator::createSDiv(v1, v2, "", currentBlock );
  }
}

llvm::Constant* CodeGenerator::createDivisionExpression( llvm::Constant* lhs, const Type* lhsType, llvm::Constant* rhs, const Type* rhsType)
{
  UNIFORMIZE_CONSTANT_TYPES( lhs, rhs );
  if( v1->getType()->isFloatingPoint() )
  {
    return llvm::ConstantExpr::getFDiv( v1, v2);
  } else {
    return llvm::ConstantExpr::getSDiv( v1, v2);
  }
}

ExpressionResult CodeGenerator::createDivisionExpression( llvm::BasicBlock* currentBlock, ExpressionResult lhs, ExpressionResult rhs)
{
  const Type* lhsType = lhs.type();
  const Type* rhsType = rhs.type();
  UNIFORMIZE_TYPES(lhs, rhs);
  CREATE_BINARY_OPERATOR(  createDivisionExpression, v1, v2);
}

llvm::Value* CodeGenerator::createModuloExpression(llvm::BasicBlock* currentBlock, llvm::Value* lhs, const Type* lhsType, llvm::Value* rhs, const Type* rhsType)
{
  UNIFORMIZE_VALUE_TYPES( lhs, rhs );
  return llvm::BinaryOperator::createSRem( v1, v2, "", currentBlock );
}

llvm::Constant* CodeGenerator::createModuloExpression( llvm::Constant* lhs, const Type* lhsType, llvm::Constant* rhs, const Type* rhsType)
{
  UNIFORMIZE_CONSTANT_TYPES( lhs, rhs );
  return llvm::ConstantExpr::getSRem( v1, v2);
}

ExpressionResult CodeGenerator::createModuloExpression( llvm::BasicBlock* currentBlock, ExpressionResult lhs, const Type* lhsType, ExpressionResult rhs, const Type* rhsType)
{
  UNIFORMIZE_TYPES(lhs, rhs);
  CREATE_BINARY_OPERATOR(  createModuloExpression, v1, v2);
}


#define CREATE_BINARY_BOOLEAN_EXPRESSION(_GTL_Function_Name_, _LLVM_Constant_Expr_, _LLVM_Binary_Operator ) \
  llvm::Value* CodeGenerator::_GTL_Function_Name_(llvm::BasicBlock* currentBlock, llvm::Value* lhs, const Type* lhsType, llvm::Value* rhs, const Type* rhsType) \
  { \
    MAKE_BOOLEAN_VALUE( lhs, rhs ) \
    return llvm::BinaryOperator::_LLVM_Binary_Operator( v1, v2, "", currentBlock ); \
  } \
  llvm::Constant* CodeGenerator::_GTL_Function_Name_( llvm::Constant* lhs, const Type* lhsType, llvm::Constant* rhs, const Type* rhsType) \
  { \
    MAKE_BOOLEAN_CONSTANT( lhs, rhs ) \
    return llvm::ConstantExpr::_LLVM_Constant_Expr_( v1, v2 ); \
  } \
  ExpressionResult CodeGenerator::_GTL_Function_Name_( llvm::BasicBlock* currentBlock, ExpressionResult lhs, const Type* lhsType, ExpressionResult rhs, const Type* rhsType) \
  { \
    UNIFORMIZE_TYPES(lhs, rhs); \
    CREATE_BINARY_OPERATOR(  _GTL_Function_Name_, v1, v2); \
  }

CREATE_BINARY_EXPRESSION(createOrExpression, getOr, createOr)
CREATE_BINARY_EXPRESSION(createAndExpression, getAnd, createAnd)

#define CREATE_BINARY_INTEGER_EXPRESSION(_GTL_Function_Name_, _LLVM_Constant_Expr_, _LLVM_Binary_Operator ) \
  llvm::Value* CodeGenerator::_GTL_Function_Name_(llvm::BasicBlock* currentBlock, llvm::Value* lhs, const Type* lhsType, llvm::Value* rhs, const Type* rhsType) \
  { \
    UNIFORMIZE_VALUE_TYPES( lhs, rhs ) \
    GTL_ASSERT( v1->getType()->isInteger() ); \
    GTL_ASSERT( v2->getType()->isInteger() ); \
    return llvm::BinaryOperator::_LLVM_Binary_Operator( v1, v2, "", currentBlock ); \
  } \
  llvm::Constant* CodeGenerator::_GTL_Function_Name_( llvm::Constant* lhs, const Type* lhsType, llvm::Constant* rhs, const Type* rhsType) \
  { \
    UNIFORMIZE_CONSTANT_TYPES( lhs, rhs ) \
    GTL_ASSERT( v1->getType()->isInteger() ); \
    GTL_ASSERT( v2->getType()->isInteger() ); \
    return llvm::ConstantExpr::_LLVM_Constant_Expr_( v1, v2 ); \
  } \
  ExpressionResult CodeGenerator::_GTL_Function_Name_( llvm::BasicBlock* currentBlock, ExpressionResult lhs, const Type* lhsType, ExpressionResult rhs, const Type* rhsType) \
  { \
    UNIFORMIZE_TYPES(lhs, rhs); \
    CREATE_BINARY_OPERATOR(  _GTL_Function_Name_, v1, v2); \
  }

CREATE_BINARY_INTEGER_EXPRESSION(createBitXorExpression, getXor, createXor )
CREATE_BINARY_INTEGER_EXPRESSION(createBitOrExpression, getOr, createOr )
CREATE_BINARY_INTEGER_EXPRESSION(createBitAndExpression, getAnd, createAnd )


#define CREATE_COMPARISON_EXPRESSION(_GTL_Function_Name_, _LLVM_UInt_Expr_, _LLVM_SInt_Expr_, _LLVM_Float_Expr_ ) \
  llvm::Value* CodeGenerator::_GTL_Function_Name_(llvm::BasicBlock* currentBlock, llvm::Value* lhs, const Type* lhsType, llvm::Value* rhs, const Type* rhsType) \
  { \
    return createComparisonExpression( currentBlock, lhs, lhsType, rhs, rhsType, llvm::ICmpInst::_LLVM_UInt_Expr_, llvm::ICmpInst::_LLVM_SInt_Expr_, llvm::FCmpInst::_LLVM_Float_Expr_); \
  } \
  llvm::Constant* CodeGenerator::_GTL_Function_Name_( llvm::Constant* lhs, const Type* lhsType, llvm::Constant* rhs, const Type* rhsType) \
  { \
    return createComparisonExpression( lhs, lhsType, rhs, rhsType,  llvm::ICmpInst::_LLVM_UInt_Expr_, llvm::ICmpInst::_LLVM_SInt_Expr_, llvm::FCmpInst::_LLVM_Float_Expr_); \
  } \
  ExpressionResult CodeGenerator::_GTL_Function_Name_( llvm::BasicBlock* currentBlock, ExpressionResult lhs, const Type* lhsType, ExpressionResult rhs, const Type* rhsType) \
  { \
    return createComparisonExpression( currentBlock, lhs, lhsType, rhs, rhsType,  llvm::ICmpInst::_LLVM_UInt_Expr_, llvm::ICmpInst::_LLVM_SInt_Expr_, llvm::FCmpInst::_LLVM_Float_Expr_); \
  }

CREATE_COMPARISON_EXPRESSION(createEqualExpression, ICMP_EQ, ICMP_EQ, FCMP_OEQ);
CREATE_COMPARISON_EXPRESSION(createDifferentExpression, ICMP_NE, ICMP_NE, FCMP_ONE);
CREATE_COMPARISON_EXPRESSION(createStrictInferiorExpression, ICMP_ULT, ICMP_SLT, FCMP_OLT);
CREATE_COMPARISON_EXPRESSION(createInferiorOrEqualExpression, ICMP_ULE, ICMP_SLE, FCMP_OLE);
CREATE_COMPARISON_EXPRESSION(createStrictSupperiorExpression, ICMP_UGT, ICMP_SGT, FCMP_OGT);
CREATE_COMPARISON_EXPRESSION(createSupperiorOrEqualExpression, ICMP_UGE, ICMP_SGE, FCMP_OGE);

llvm::Value* CodeGenerator::createComparisonExpression(llvm::BasicBlock* currentBlock, llvm::Value* lhs, const Type* lhsType, llvm::Value* rhs, const Type* rhsType, unsigned int unsignedIntegerPred, unsigned int signedIntegerPred, unsigned int floatPred)
{
  UNIFORMIZE_VALUE_TYPES( lhs, rhs );
  if( v1->getType()->isFloatingPoint() )
  {
    return new llvm::FCmpInst( (llvm::FCmpInst::Predicate)floatPred, v1, v2, "", currentBlock );
  } else if( v2->getType()->isInteger() ) {
    if( rhsType->isSigned() )
    {
      return new llvm::ICmpInst( (llvm::ICmpInst::Predicate)signedIntegerPred, v1, v2, "", currentBlock );
    } else {
      return new llvm::ICmpInst( (llvm::ICmpInst::Predicate)unsignedIntegerPred, v1, v2, "", currentBlock );
    }
  } else {
    GTL_ABORT("Invalid comparison");
    return 0;
  }
}

llvm::Constant* CodeGenerator::createComparisonExpression( llvm::Constant* lhs, const Type* lhsType, llvm::Constant* rhs, const Type* rhsType, unsigned int unsignedIntegerPred, unsigned int signedIntegerPred, unsigned int floatPred)
{
  UNIFORMIZE_CONSTANT_TYPES( lhs, rhs );
  unsigned op = v1->getType()->isFloatingPoint() ? floatPred : 
      ( rhsType->isSigned() ? signedIntegerPred : unsignedIntegerPred);
  return (llvm::Constant*)llvm::ConstantExpr::getCompare( op, v1, v2);
}

ExpressionResult CodeGenerator::createComparisonExpression( llvm::BasicBlock* currentBlock, ExpressionResult lhs, const Type* lhsType, ExpressionResult rhs, const Type* rhsType, unsigned int unsignedIntegerPred, unsigned int signedIntegerPred, unsigned int floatPred)
{
  UNIFORMIZE_TYPES(lhs, rhs);
  if( v1.isConstant() and v2.isConstant() )
  {
    return GTLCore::ExpressionResult(createComparisonExpression( v1.constant(), lhsType, v2.constant(), rhsType, unsignedIntegerPred, signedIntegerPred, floatPred), Type::Boolean);
  } else {
    return GTLCore::ExpressionResult(createComparisonExpression( currentBlock, v1.value(), lhsType, v2.value(), rhsType, unsignedIntegerPred, signedIntegerPred, floatPred), Type::Boolean);
  }
}

#define CREATE_UNARY_OPERATOR( _creator_, _v1_) \
  if( _v1_.isConstant() ) \
  { \
    return GTLCore::ExpressionResult( _creator_(_v1_.constant(), rhsType), rhsType); \
  } else { \
    return GTLCore::ExpressionResult( _creator_( currentBlock, _v1_.value(), rhsType ), rhsType ); \
  }

ExpressionResult CodeGenerator::createMinusExpression( llvm::BasicBlock* currentBlock, ExpressionResult rhs, const Type* rhsType)
{
  CREATE_UNARY_OPERATOR( createMinusExpression, rhs);
}

llvm::Value* CodeGenerator::createMinusExpression(llvm::BasicBlock* currentBlock, llvm::Value* rhs, const Type* rhsType)
{
  return llvm::BinaryOperator::createNeg( rhs, "", currentBlock );
}

llvm::Constant* CodeGenerator::createMinusExpression( llvm::Constant* rhs, const Type* rhsType)
{
  return llvm::ConstantExpr::getNeg( rhs);
}

ExpressionResult CodeGenerator::createNotExpression( llvm::BasicBlock* currentBlock, ExpressionResult rhs, const Type* rhsType)
{
  CREATE_UNARY_OPERATOR( createNotExpression, rhs);
}

llvm::Value* CodeGenerator::createNotExpression(llvm::BasicBlock* currentBlock, llvm::Value* rhs, const Type* rhsType)
{
  return llvm::BinaryOperator::createNot( rhs, "", currentBlock );
}

llvm::Constant* CodeGenerator::createNotExpression( llvm::Constant* rhs, const Type* rhsType)
{
  return llvm::ConstantExpr::getNot( rhs);
}

ExpressionResult CodeGenerator::createTildeExpression( llvm::BasicBlock* currentBlock, ExpressionResult rhs, const Type* rhsType)
{
  CREATE_UNARY_OPERATOR( createTildeExpression, rhs);
}

llvm::Value* CodeGenerator::createTildeExpression(llvm::BasicBlock* currentBlock, llvm::Value* rhs, const Type* rhsType)
{
  return llvm::BinaryOperator::createXor( rhs, integerToConstant(0xFFFFFFFF ), "", currentBlock );
}

llvm::Constant* CodeGenerator::createTildeExpression( llvm::Constant* rhs, const Type* rhsType)
{
  return llvm::ConstantExpr::getXor(rhs, integerToConstant(0xFFFFFFFF ));
}

llvm::Value* CodeGenerator::accessArrayValue( llvm::BasicBlock* _currentBlock, llvm::Value* _pointer, llvm::Value* _index )
{
  GTL_DEBUG( *_pointer << " " << *_index );
  std::vector<llvm::Value*> indexes;
  indexes.push_back( llvm::ConstantInt::get(llvm::Type::Int32Ty, 0)); // Access the structure of the array
  indexes.push_back( llvm::ConstantInt::get(llvm::Type::Int32Ty, 1)); // Access the size of the array
  
  // Select the pointer
  llvm::Value* iptr = llvm::GetElementPtrInst::Create( _pointer, indexes.begin(), indexes.end(), "", _currentBlock);
  // pointer on the values
  llvm::Value* iptr_val = new llvm::LoadInst( iptr, "", _currentBlock);
  return llvm::GetElementPtrInst::Create( iptr_val, _index, "", _currentBlock);
}

GTLCore::ExpressionResult CodeGenerator::callFunction( GenerationContext& _gc, llvm::BasicBlock* bb, const GTLCore::Function* _function, const std::list<AST::Expression*>& _arguments )
{
  // Function Type
  GTL_ASSERT( _function );
  llvm::Function* llvmFunction = _function->d->data->function( _arguments.size() );
#ifndef NDEBUG
    const llvm::FunctionType *FTy = llvm::cast<llvm::FunctionType>(llvm::cast<llvm::PointerType>( llvmFunction->getType())->getElementType());
    GTL_DEBUG( *FTy );
#endif
  GTL_ASSERT( llvmFunction );
  std::vector< Parameter >::const_iterator oparam_it = _function->parameters().begin();
  std::vector<llvm::Value*> convertedParams;
  for( std::list< AST::Expression* >::const_iterator it = _arguments.begin();
       it != _arguments.end(); ++it, ++oparam_it)
  {
    if(oparam_it->output())
    {
      AST::AccessorExpression* aexp = dynamic_cast<AST::AccessorExpression*>( *it );
      GTL_ASSERT( aexp );
      llvm::Value* pointer = aexp->pointer( _gc, bb );
      GTL_ASSERT( pointer->getType() == llvm::PointerType::get( oparam_it->type()->d->type(), 0) );
      convertedParams.push_back( pointer );
      GTL_DEBUG(*pointer->getType() << " " << *oparam_it->type()->d->type() );
    } else {
      llvm::Value* value = 0;
      if( (*it)->type()->dataType() == GTLCore::Type::ARRAY or (*it)->type()->dataType() == GTLCore::Type::STRUCTURE )
      {
        if( AST::AccessorExpression* aexp = dynamic_cast<AST::AccessorExpression*>( *it ) )
        {
          value = aexp->pointer( _gc, bb );
        } else if( dynamic_cast<AST::FunctionCallExpression*>( *it )
                   or dynamic_cast<AST::GlobalDataExpression*>( *it ) ) {
            value = (*it)->generateValue(_gc, bb).value();
        } else {
          GTL_ABORT("Can't pass an array or a structure");
        }
      } else {
        value = (*it)->generateValue(_gc, bb).value();
      }
      GTL_ASSERT( value );
      GTL_DEBUG( "param's type = " << *oparam_it->type()->d->type() << " value = " << *value );
      if( oparam_it->type()->d->type()->isFirstClassType() )
      {
        value = _gc.codeGenerator()-> convertValueTo( bb, value, (*it)->type(), oparam_it->type() );
      }
      convertedParams.push_back( value );
    }
    
  }
  
#ifndef NDEBUG
  {
    const llvm::FunctionType *FTy =
      llvm::cast<llvm::FunctionType>(llvm::cast<llvm::PointerType>(llvmFunction->getType())->getElementType());
    
    GTL_ASSERT( convertedParams.size() == FTy->getNumParams() or
            (FTy->isVarArg() and convertedParams.size() > FTy->getNumParams()) );
    
    for (unsigned i = 0; i < convertedParams.size(); ++i) {
      if( i < FTy->getNumParams() and (FTy->getParamType(i) != convertedParams[i]->getType()) )
      {
        GTL_DEBUG( "Wrong parameter " << i << " : " << FTy->getParamType(i) << " => " << *FTy->getParamType(i) << " but got " << convertedParams[i] << " => " << *convertedParams[i]->getType() );
      } else {
        GTL_DEBUG( "Parameter " << i << " : " << FTy->getParamType(i) << " => " << *FTy->getParamType(i) << " but got " << convertedParams[i]->getType() << " => " << *convertedParams[i]->getType() );
      }
    }
  }
#endif
  
  llvm::CallInst *CallFunc = llvm::CallInst::Create(llvmFunction, convertedParams.begin(), convertedParams.end(), "", bb);
  CallFunc->setTailCall();
  return GTLCore::ExpressionResult( CallFunc, _function->returnType(), true );
}
