/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * version 2 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "AbstractImage.h"

#include <cstring>

#include "Debug.h"
#include "PixelDescription.h"
#include "Region.h"

using namespace GTLCore;

struct AbstractImage::Private {
  Private( const GTLCore::PixelDescription& _pixelDescription) : pixelDescription(_pixelDescription)
  {
  }
  GTLCore::PixelDescription pixelDescription;
  int pixelSize;
};

AbstractImage::AbstractImage( const GTLCore::PixelDescription& _pixelDescription ) : d(new Private(_pixelDescription))
{
  GTL_ASSERT(d->pixelDescription.bitsSize() % 8 == 0);
  d->pixelSize = d->pixelDescription.bitsSize() / 8;
}

AbstractImage::~AbstractImage()
{
  delete d;
}

const GTLCore::PixelDescription& AbstractImage::pixelDescription() const
{
  return d->pixelDescription;
}

int AbstractImage::pixelSize() const
{
  return d->pixelSize;
}

int AbstractImage::compare( const AbstractImage* _image, const GTLCore::Region& _region) const
{
  // Test pixelDescription
  if( pixelDescription() != _image->pixelDescription() )
  {
    std::cout << "Different pixel description : " << pixelDescription() << " != " << _image->pixelDescription() << std::endl;
    return EXIT_FAILURE;
  }
  // Test pixel data
  int _pixelSize = pixelDescription().bitsSize();
  if( _pixelSize % 8 != 0)
  {
    GTL_ABORT( "Can't compare image of pixel size : " << _pixelSize << "bits.");
  }
  _pixelSize /= 8;
  int errorCount = 0;
  for(int y = 0; y < _region.height(); ++y)
  {
    for(int x = 0; x < _region.width(); ++x)
    {
      if( memcmp( data( x, y ), _image->data( x, y ), _pixelSize ) != 0)
      {
        if( errorCount < 100 )
        {
          GTL_ERROR( "Pixel (" << x << ", " << y << ") is different." );
        } else if( errorCount == 100 )
        {
          GTL_ERROR( "and more...");
        }
        ++errorCount;
      }
    }
  }
  return errorCount;
}
