/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "Token_p.h"

#include "Debug.h"

using namespace GTLCore;


GTLCore::String Token::typeToString(Type type)
{
  switch(type)
  {
// Not really token
    case END_OF_FILE:
      return "end of file";
    case UNKNOWN:
      return "unknown token";
// Special characters
    case Token::SEMI:
      return ";";
    case Token::COLON:
      return ":";
    case Token::COLONCOLON:
      return "::";
    case Token::COMA:
      return ",";
    case Token::DOT:
      return ".";
    case Token::STARTBRACE:
      return "{";
    case Token::ENDBRACE:
      return "}";
    case Token::STARTBRACKET:
      return "(";
    case Token::ENDBRACKET:
      return ")";
    case Token::STARTBOXBRACKET:
      return "[";
    case Token::ENDBOXBRACKET:
      return "]";
    case Token::EQUAL:
      return "=";
    case Token::EQUALEQUAL:
      return "==";
    case Token::PLUSEQUAL:
      return "+=";
    case Token::MINUSEQUAL:
      return "-=";
    case Token::MULTIPLYEQUAL:
      return "*=";
    case Token::DIVIDEEQUAL:
      return "/=";
    case Token::DIFFERENT:
      return "!=";
    case Token::AND:
      return "and";
    case Token::OR:
      return "or";
    case Token::BITAND:
      return "&";
    case Token::BITXOR:
      return "^";
    case Token::BITOR:
      return "|";
    case Token::INFERIOR:
      return "<";
    case Token::INFERIOREQUAL:
      return "<=";
    case Token::SUPPERIOR:
      return ">";
    case Token::SUPPERIOREQUAL:
      return ">=";
    case Token::LEFTSHIFT:
      return "<<";
    case Token::RIGHTSHIFT:
      return ">>";
    case Token::PLUS:
      return "+";
    case Token::PLUSPLUS:
      return "++";
    case Token::MINUS:
      return "-";
    case Token::MINUSMINUS:
      return "--";
    case Token::MULTIPLY:
      return "*";
    case Token::DIVIDE:
      return "/";
    case Token::MODULO:
      return "%";
    case Token::TILDE:
      return "~";
    case Token::NOT:
      return "not";
// Constants
    case FLOAT_CONSTANT:
      return "float constant";
    case INTEGER_CONSTANT:
      return "integer constant";
    case STRING_CONSTANT:
      return "string constant";
    case IDENTIFIER:
      return "identifier";
    case BOOL:
      return "bool";
    case BOOL2:
      return "bool2";
    case BOOL3:
      return "bool3";
    case BOOL4:
      return "bool4";
    case BOOLN:
      return "booln";
    case CONST:
      return "const";
    case FLOAT:
      return "float";
    case FLOAT2:
      return "float2";
    case FLOAT3:
      return "float3";
    case FLOAT4:
      return "float4";
    case FLOATN:
      return "floatn";
    case ELSE:
      return "else";
    case FOR:
      return "for";
    case IF:
      return "if";
    case IMPORT:
      return "import";
    case INT:
      return "int";
    case INT2:
      return "int2";
    case INT3:
      return "int3";
    case INT4:
      return "int4";
    case INTN:
      return "intn";
    case KERNEL:
      return "kernel";
    case LIBRARY:
      return "library";
    case LONG:
      return "long";
    case RETURN:
      return "return";
    case SHORT:
      return "short";
    case SIGNED:
      return "signed";
    case SIZE:
      return "size";
    case STRUCT:
      return "struct";
    case UNSIGNED:
      return "unsigned";
    case VOID:
      return "void";
    case WHILE:
      return "while";
    case NAMESPACE:
      return "namespace";
    case CTLVERSION:
      return "ctlversion";
    case HALF:
      return "half";
    case INPUT:
      return "input";
    case OUTPUT:
      return "output";
    case PRINT:
      return "print";
    case UNIFORM:
      return "uniform";
    case VARYING:
      return "varying";
    case TTRUE:
      return "true";
    case TFALSE:
      return "false";
    case DEPENDENT:
      return "dependent";
    case ALLCHANNELS:
      return "@allchannels";
    case COLORCHANNELS:
      return "@colorchannels";
    case ALPHACHANNEL:
      return "@alphachannel";
    case ALPHA:
      return "@alpha";
    case TYPE:
      return "@type";
    case MIN:
      return "@min";
    case MAX:
      return "@max";
    case OPERATION:
      return "@operation";
    case UNIT:
      return "@unit";
    case VAR:
      return "@var";
    default:
      return "[TODO] " + GTLCore::String::number( type );
  }
}

Token::Token() : type(UNKNOWN), line(-1), column(-1)
{
}

Token::Token(Type _type, int _line, int _column) : type(_type), line(_line), column(_column)
{
}

Token::Token(Type _type, const GTLCore::String& _string, int _line, int _column) : type(_type), line(_line), column(_column), string(_string)
{
  GTL_ASSERT( _type == STRING_CONSTANT or _type == IDENTIFIER );
}

Token::Token(int _i, int _line, int _column) : type(INTEGER_CONSTANT), line(_line), column(_column), i(_i)
{
}

Token::Token(float _f, int _line, int _column) : type(FLOAT_CONSTANT), line(_line), column(_column), f(_f)
{
}

bool Token::isNativeType() const
{
  return type == Token::INT or type == Token::INT2 or type == Token::INT3 or type == Token::INT4 or type == Token::INTN or type == Token::FLOAT or type == Token::FLOAT2 or type == Token::FLOAT3 or type == Token::FLOAT4 or type == Token::FLOATN or type == Token::HALF or type == Token::BOOL or type == Token::BOOL2 or type == Token::BOOL3 or type == Token::BOOL4 or type == Token::BOOLN or type == Token::UNSIGNED or type == Token::SIGNED;
}

bool Token::isFunctionType() const
{
  return isNativeType() or type == Token::VOID or type == Token::IDENTIFIER;
}

bool Token::isConstant() const
{
  return type == Token::INTEGER_CONSTANT or type == Token::FLOAT_CONSTANT or type == Token::TTRUE or type == Token::TFALSE;
}

bool Token::isPrimary() const
{
  return isConstant() or type == Token::IDENTIFIER;
}

bool Token::isBinaryOperator() const
{
  return binaryOperationPriority() != -1;
}

enum Priority {
  ASSIGNEMENT_PRIORITY,
  OR_PRIORITY,
  AND_PRIORITY,
  BITOR_PRIORITY,
  BITXOR_PRIORITY,
  BITAND_PRIORITY,
  EQUALITY_PRIORITY,
  RELATIONAL_PRIORITY,
  SHIFT_PRIORITY,
  ADDITIVE_PRIORITY,
  MULTIPLICATIVE_PRIORITY
};

int Token::binaryOperationPriority() const
{
  switch( type )
  {
    case Token::EQUAL:
    case Token::PLUSEQUAL:
    case Token::MINUSEQUAL:
    case Token::MULTIPLYEQUAL:
    case Token::DIVIDEEQUAL:
      return ASSIGNEMENT_PRIORITY;
    case Token::OR:
      return OR_PRIORITY;
    case Token::AND:
      return AND_PRIORITY;
    case Token::BITOR:
      return BITOR_PRIORITY;
    case Token::BITXOR:
      return BITXOR_PRIORITY;
    case Token::BITAND:
      return BITAND_PRIORITY;
    case Token::EQUALEQUAL:
    case Token::DIFFERENT:
      return EQUALITY_PRIORITY;
    case Token::INFERIOREQUAL:
    case Token::INFERIOR:
    case Token::SUPPERIOREQUAL:
    case Token::SUPPERIOR:
      return RELATIONAL_PRIORITY;
    case Token::RIGHTSHIFT:
    case Token::LEFTSHIFT:
      return SHIFT_PRIORITY;
    case Token::PLUS:
    case Token::MINUS:
      return ADDITIVE_PRIORITY;
    case Token::MULTIPLY:
    case Token::DIVIDE:
    case Token::MODULO:
      return MULTIPLICATIVE_PRIORITY;
    default:
      return -1;
  }
}

bool Token::isUnaryOperator() const
{
  return type == Token::MINUS or type == Token::TILDE or type == Token::NOT or type == Token::PLUSPLUS or type == Token::MINUSMINUS;
}

bool Token::isOperator() const
{
  return isUnaryOperator() or isBinaryOperator();
}

bool Token::isExpressionTerminal()
{
  return type == Token::SEMI or type == Token::ENDBRACKET or type == Token::COMA or type == Token::ENDBOXBRACKET or type == Token::ENDBRACE;
}
